/*
 * Copyright (c) 1991 by Sozobon, Limited.  Author: Johann Ruegg
 *
 * Permission is granted to anyone to use this software for any purpose
 * on any computer system, and to redistribute it freely, with the
 * following restrictions:
 * 1) No charge may be made other than reasonable charges for reproduction.
 * 2) Modified versions must be clearly marked as such.
 * 3) The authors are not responsible for any harmful consequences
 *    of using this software, even if they result from defects in it.
 */

#ifndef UNIXHOST
#include "ar.h"
#define SWAPW(a,b)
#define SWAPL(a,b)
#else
#include "crossar.h"
#define SWAPW(a,b)  swapw(a,b)
#define SWAPL(a,b)  swapl(a,b)
#endif

#include <stdio.h>
#include <io.h>
#include <string.h>
#include <malloc.h>
#include <fcntl.h>
#include "syms.h"
#include "structs.h"
#include "global.h"

/*
 * Root pointers for file names chain, object file headers chain and 
 * library files chain
 */

FILENAME  *pFileNames;
OFILEINFO *ohead;
LIBINFO   *ahead;

extern int tb;
extern int db;
extern int trelb;
extern int drelb;
extern int vflag;


static FILENAME *pFileName;
static int       pFileHndl;


static void doobj(void);
static void afree(void);
static void ascan(void);
static void process(void);
static int  ahdsyms(LIBINFO *pLibEntry);
static void doarch(void);


/****************************************************************************
 *                                                                          *
 * Open input file and pass on to lower levels                              *
 *                                                                          *
 ****************************************************************************/

void pass1(char *sFile)

{
/*
 * Allocate storage for file name and then add it to the list of
 * file names.
 */

pFileName = (FILENAME *)mmalloc(strlen(sFile) + sizeof(FILENAME));
strcpy(pFileName->sName, sFile);
listadd((GENERIC **)&pFileNames, (GENERIC *)pFileName);

pFileHndl = open(sFile, O_RDONLY|O_BINARY);                 /* Open the file at top level   */

if(vflag)                                       /* Show progress if needed  */
    printf("Processing file %s:\n", pFileName->sName);

process();                                      /* Do what we do with file  */

close(pFileHndl);                               /* And finally close it     */
}


/****************************************************************************
 *                                                                          *
 * Decide if we have a library file or an object file                       *
 *                                                                          *
 ****************************************************************************/

static void process(void)

{
unsigned short magic;

magic = 0;                                  /* Fetch magic number from file */
rread(pFileHndl, (char *)&magic, 2);

SWAPW((char *)&magic, 1);                   /* Get byte order right         */
if(magic == OMAGIC)                         /* Is it an object file?        */
    doobj();
else if(magic == AMAGIC)                    /* Is it a library file?        */
    doarch();
else
    fatals("Unrecognised magic number", pFileName->sName); /* Neither, panic */
}


/****************************************************************************
 *                                                                          *
 * Add contents of library file to list of library objects                  *
 *                                                                          *
 ****************************************************************************/

static void doarch()

{
LIBHEAD  LibHead;
long     lSize;
long     lObjStart;
LIBINFO *pLibTemp;
int      iReuse;

lObjStart = 2;                          /* Allow for Library magic number   */
iReuse    = 0;                          /* 1st time round we must allocate  */

/*
 * read library headers in until we have no more or we get one with a blank
 * name. This is because CP/M 68K pads out the files to 128 bytes in length.
 */

while(read(pFileHndl, &LibHead, sizeof(LibHead)) == sizeof(LibHead) && LibHead.ar_name[0])
    {
    #ifndef UNIXHOST
    lSize = LibHead.ar_size;            /* Get length of current obj file   */
    #else
    lSize = crossl((char *)&LibHead.ar_size);
    #endif

    if(vflag > 2)
        printf("\n%.14s size %lx\n", LibHead.ar_name, lSize);

    /*
     * allocate storage for a new library object file header
     * and add it to the list.
     */

    if(iReuse == 0)
        {
        pLibTemp = (LIBINFO *)mmalloc(sizeof(LIBINFO));
        listadd((GENERIC **)&ahead, (GENERIC *)pLibTemp);
        }

    pLibTemp->aoffs = lObjStart + sizeof(LibHead); /* Location in lib file  */
    pLibTemp->used  = 0;                           /* Obj not required yet  */
    pLibTemp->pSymbols = NULL;                        /* No symbols loaded  */
    memcpy(pLibTemp->sFileName, LibHead.ar_name, 14); /* Get obj file name  */

    iReuse = ahdsyms(pLibTemp);             /* Try to read symbol table in  */
    lObjStart += lSize + sizeof(LibHead);   /* Step over file to next one   */
    llseek(pFileHndl, lObjStart, 0);
    }

ascan();
afree();
}


/****************************************************************************
 *                                                                          *
 * Grab the symbol table from the current library object file               *
 *                                                                          *
 ****************************************************************************/

static int ahdsyms(LIBINFO *pLibEntry)

{
char  sTemp[HDRFILL];
short iMagic;
long  lBodyLen;

rread(pFileHndl, (char *)&iMagic, 2);   /* Fetch object file magic number   */
SWAPW((char *)&iMagic, 1);              /* Make sure it is right way around */
if(iMagic != OMAGIC)                    /* Drop out early if not legit      */
    return(1);                          /* Say it wasn't really an object   */

/*
 * read in segment and symbol table lengths and skip over
 * remainder of object file header.
 */

rread(pFileHndl, (char *)&pLibEntry->ah, sizeof(LENINFO));
SWAPL((char *)&pLibEntry->ah, sizeof(LENINFO) / 4);
rread(pFileHndl, sTemp, HDRFILL);

/*
 * Skip over segment data and read in the symbol table. Also check if
 * the object file contains any extended length symbols.
 */

lBodyLen = pLibEntry->ah.tsize + pLibEntry->ah.dsize;
llseek(pFileHndl, lBodyLen, 1);
getsyms(pFileHndl, pLibEntry->ah.syms, &pLibEntry->pSymbols);

pLibEntry->havex = chkx(pLibEntry->pSymbols);
return(0);
}


/****************************************************************************
 *                                                                          *
 * Make multiple passes through a library file and extract files which are  *
 * required to build the target.                                            *
 *                                                                          *
 ****************************************************************************/

static void ascan()

{
LIBINFO   *pLibEntry;
int       iGotSome;
OFILEINFO *pObjFile;

do
    {
    iGotSome = 0;       /* Assume there is nothing needed from this library */

    /*
     * walk the list of library entries and examine each one
     */

    for(pLibEntry = ahead; pLibEntry != 0; pLibEntry = pLibEntry->pNext)
        {
        /*
         * Ignore object file if we have already decided that it is
         * required or if it does not contain any of the unaccounted
         * for symbols
         */

        if(pLibEntry->used)
            continue;

        if(needed(pLibEntry->pSymbols, pLibEntry->havex) == 0)
            continue;

        iGotSome = 1; /* Say we added at least one new obj this time around */

        /*
         * Create an object file entry and add it to the list and then copy
         * the required information over
         */

        pObjFile = (OFILEINFO *)mmalloc(sizeof(OFILEINFO));
        listadd((GENERIC **)&ohead, (GENERIC *)pObjFile);
        memcpy(pObjFile->sFileName, pLibEntry->sFileName, 14); /* Obj name  */
        pObjFile->finfo    = pFileName;                        /* Lib name  */
        pObjFile->oh       = pLibEntry->ah;              /* Segment lengths */
        pObjFile->pSymbols = pLibEntry->pSymbols;        /* Symbol table    */
        pObjFile->havex    = pLibEntry->havex;      /* Extended symbol flag */

        if(vflag > 1)               /* Print out some details if requested  */
            {
            printf("%s(%.14s): ", pFileName->sName, pLibEntry->sFileName);
            printf("Code %lx Data %lx BSS %lx Symbols %lx\n",
                   pObjFile->oh.tsize, pObjFile->oh.dsize, pObjFile->oh.bsize,
                   pObjFile->oh.syms);
            }


        llseek(pFileHndl, pLibEntry->aoffs+HDRSIZE, 0); /* Goto code start  */
        fbcopy(pFileHndl, tb, pObjFile->oh.tsize);      /* Suck in code     */
        fbcopy(pFileHndl, db, pObjFile->oh.dsize);      /* Suck in data     */
        llseek(pFileHndl, pObjFile->oh.syms, 1);        /* Skip symbol tab  */
        fbcopy(pFileHndl, trelb, pObjFile->oh.tsize);   /* Get code reloc   */
        fbcopy(pFileHndl, drelb, pObjFile->oh.dsize);   /* get data reloc   */


        pLibEntry->used = 1;                         /* Mark obj as taken   */

        p1syms(pObjFile->pSymbols, pObjFile->havex, pObjFile);
        }
    } while(iGotSome);  /* Scan lib file until we stop needing object files */
}


/****************************************************************************
 *                                                                          *
 * Clear out the library file chain in preparation for the next lib file    *
 *                                                                          *
 ****************************************************************************/

static void afree(void)

{
LIBINFO *pLibEntry;
LIBINFO *pNext;

/*
 * scan the list from start to finish releasing the blocks as we go
 */

for(pLibEntry = ahead; pLibEntry != 0; pLibEntry = pNext)
    {
    /*
     * If an object file in the library was used then we don't free up
     * the symbol table because it has been passed on to the object
     * file entry created when the object file was added to the target.
     */

    if(pLibEntry->used == 0)
        {
        if(pLibEntry->pSymbols)
            symfree(pLibEntry->pSymbols);
        }

    pNext = pLibEntry->pNext;
    free(pLibEntry);
    }

ahead = NULL;                       /* Mark library chain as being empty    */
}


/****************************************************************************
 *                                                                          *
 * Add an ordinary object file to the target                                *
 *                                                                          *
 ****************************************************************************/

static void doobj()

{
OFILEINFO *pObjFile;
char       sTemp[HDRFILL];

/*
 * Create an object file entry for this object and add it to the list
 */

pObjFile = (OFILEINFO *)mmalloc(sizeof(OFILEINFO));
listadd((GENERIC **)&ohead, (GENERIC *)pObjFile);

/*
 * Get segment length information and ensure it is in right order
 */

rread(pFileHndl, (char *)&pObjFile->oh, sizeof(LENINFO));
SWAPL((char *)&pObjFile->oh, sizeof(LENINFO) / 4);

rread(pFileHndl, sTemp, HDRFILL);           /* Skip over rest of header     */
pObjFile->sFileName[0] = 0;                 /* Set up file name information */
pObjFile->finfo        = pFileName;

if(vflag > 1)                               /* Display details if requested */
    {
    printf("%s: ", pFileName->sName);
    printf("Code %lx Data %lx BSS %lx SYmbols %lx\n",
    pObjFile->oh.tsize, pObjFile->oh.dsize, pObjFile->oh.bsize,
    pObjFile->oh.syms);
    }

/*
 * Read in all object file information
 */

fbcopy(pFileHndl,  tb, pObjFile->oh.tsize);
fbcopy(pFileHndl,  db, pObjFile->oh.dsize);
getsyms(pFileHndl, pObjFile->oh.syms, &pObjFile->pSymbols);
fbcopy(pFileHndl,  trelb, pObjFile->oh.tsize);
fbcopy(pFileHndl,  drelb, pObjFile->oh.dsize);

pObjFile->havex = chkx(pObjFile->pSymbols); /* Check for extended symbols   */

p1syms(pObjFile->pSymbols, pObjFile->havex, pObjFile);
}
