/* Driver for LPR LANCE Ethernet card */

/* HT 30.01.91: ISR-nderungen fr TC 2.0 */

#include <stdio.h>
#include <tos.h>
#include "global.h"
#include "mbuf.h"
#include "enet.h"
#include "timer.h"
#include "iproute.h"
#include "internet.h"
#include "arp.h"
#include "trace.h"
#include "eth.h"

#define R_BAN1 31
#define T_BAN1  7

IOPB iopb;
DRE  rring_[R_BANZ+1], tring_[T_BANZ+1]; /* +1 wgen quadword */
volatile DRE  *rring, *tring;
char rbuf[R_BANZ][BGR], tbuf[T_BANZ][BGR];
volatile int  rint_buff[R_BANZ];
volatile struct statistics statistics = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
volatile struct ecb ecb;	/* HT: volatile */

extern void eth_start( void ); 


int eth_init(struct interface *interface)
{
        int     i, timeout;

        RAP = CSR0;
        RDP = STOP;
        

        rring = ring_init('r', rring_);
        tring = ring_init('t', tring_);


        iopb.mode = 0;
        iopb.pad[0] = (interface->hwaddr[0] & 0x00ff) |
                ((interface->hwaddr[1] << 8 ) & 0xff00); 
        iopb.pad[1] = (interface->hwaddr[2] & 0x00ff) | 
                ((interface->hwaddr[3] << 8 ) & 0xff00); 
        iopb.pad[2] = (interface->hwaddr[4] & 0x00ff) | 
                ((interface->hwaddr[5] << 8 ) & 0xff00); 
        for(i=0; i<4; i++)               
                iopb.lad[i] = 0xffff;
        iopb.rr_lw = (int)rring;
        iopb.rr_len = R_LEN << 5;
        iopb.rr_hb = (char)( ((long)rring >> 16) & 0xff );
        iopb.tr_lw = (int)tring;
        iopb.tr_len = T_LEN << 5;
        iopb.tr_hb = (char)( ((long)tring >> 16) & 0xff );

        Setexc(29, eth_start);
		
        statistics.fflags = 0;
        ecb.ind_r=ecb.ind_w=ecb.tb_r=ecb.tb_w=ecb.rb_r = 0;
        for( i = 0; i < R_BANZ; i++)
                rint_buff[i] = -1;

        RAP = CSR3;
        RDP = BSWP | BCON;

        RAP = CSR1;
        RDP = (int)(&iopb);
        RAP = CSR2;
        RDP = (int)( ((long)(&iopb) >> 16) & 0xff );

        RAP = CSR0;
        RDP = INIT;

        for (timeout = 30000; (RDP & IDON) == 0 && (timeout > 0); timeout--)
                ;
        if(timeout == 0)
        {
                printf("LANCE initialisation timeout\n");
                return -1;
        }
        
        RDP = STRT | IDON | INEA;

        return 0;
}


int eth_send (struct mbuf *bp,				/* Buffer to send */
    	  struct interface *interface,  /* Pointer to interface control block */
          int32 gateway,				/* IP address of next hop */
          char precedence,
          char delay,
          char throughput,
          char reliability)
{
        char *eth_dest;

        eth_dest = res_arp(interface, ARP_ETHER, gateway, bp);
        
        if (eth_dest != NULLCHAR)
                return ((*interface->output)(interface, eth_dest, interface->hwaddr, IP_TYPE, bp));
        if (interface->trace != 0)
                printf("Ethernetadresse muss bestimmt werden!\n");
        return 0;
}


int eth_output(	struct interface *interface, char dest[], 
                char source[], int16 type, struct mbuf *bp)
{
        struct ether ep;
        struct mbuf *hdr;

        memcpy(ep.dest,dest,EADDR_LEN);
        memcpy(ep.source,source,EADDR_LEN);
        ep.type = type;
        hdr = htonether(&ep);
        hdr->next = bp;
        (*interface->raw)(interface,hdr);
        return 0;
}


int eth_raw(struct interface *interface, struct mbuf *bp)
{
        int timeout;
        char *fill_ptr;
        int count;

        for (timeout=5000; tring[ecb.tb_w].status != 0; timeout--)
        {
                if (timeout == 0)
                {
                        printf("LANCE: no free transmit buffer\n");
                        return (-1);
                }
        }

        dump (interface, IF_TRACE_OUT, TRACE_ETHER, bp);

        fill_ptr = tbuf[ecb.tb_w];
        for (count=0; bp != NULLBUF; bp = free_mbuf(bp))
        {
                memcpy(fill_ptr, bp->data, bp->cnt);
                count += bp->cnt;
                fill_ptr += bp->cnt;
        }

        if(count < RUNT)
                count = (RUNT +1) & ~1;

        tring[ecb.tb_w].bcnt = (-count) | ONES;
        tring[ecb.tb_w].ertdmc = 0;
        tring[ecb.tb_w].status = (OWN | STP | ENP);
        RDP = TDMD | INEA;
        
        ecb.tb_w = ++ecb.tb_w & T_BAN1; /*weiter(ecb.tb_w, T_BANZ);*/
        return (0);
}


void eth_isr( int sig_nr )
{
        int status, rmd1, tmd1, tmd3;

        status = RDP;
        RDP = status & ~INEA;

        if( status & SERR )
        {
                if(status & BABL)
                {
                        statistics.babl++;
                        statistics.fflags |= F1;
                }
                if(status & CERR)
                {
                        statistics.cerr++;
                        statistics.fflags |= F2;
                }
                if(status & MISS)
                {
                        statistics.miss++;
                        statistics.fflags |= F3;
                }
                if(status & MERR)
                {
                        statistics.merr++;
                        statistics.fflags |= F4;
                }
        }
        if(status & TINT)
        {
                while( ((tmd1=tring[ecb.tb_r].status) & (OWN|STP|ENP)) == (STP|ENP) )
                {
                        statistics.t_count++;
                        if(tmd1 & DERR)
                        {
                                tmd3 = tring[ecb.tb_r].ertdmc;
                                if(tmd3 & TBUFF){
                                        statistics.tbuff++;
                                        statistics.fflags |= F12;
                                }
                                if(tmd3 & UFLO)
                                {
                                        statistics.uflo++;
                                        statistics.fflags |= F13;
                                }
                                if(tmd3 & LCOL)
                                {
                                        statistics.lcol++;
                                        statistics.fflags |= F14;
                                }
                                if(tmd3 & LCAR)
                                {
                                        statistics.lcar++;
                                        statistics.fflags |= F15;
                                }
                                if(tmd3 & RTRY)
                                {
                                        statistics.rtry++;
                                        statistics.fflags |= F16;
                                }
                        }
                        if(tmd1 & MORE)
                        {
                                statistics.more++;
                                statistics.fflags |= F9;
                        }
                        if(tmd1 & ONE)
                        {
                                statistics.one++;
                                statistics.fflags |= F10;
                        }
                        if(tmd1 & DEF)
                        {
                                statistics.def++;
                                statistics.fflags |= F11;
                        }
                        tring[ecb.tb_r].status = 0;
                        ecb.tb_r = ++ecb.tb_r & T_BAN1;/*weiter(ecb.tb_r, T_BANZ);*/
                }
        }
        if(status & RINT)
        {
                while( ((rmd1=rring[ecb.rb_r].status) & OWN) == 0
                        && rint_buff[ecb.ind_w] == -1)
                {
                        if(rmd1 & FRAM)
                        {
                                statistics.fram++;
                                statistics.fflags |= F5;
                        }
                        if((rmd1 & (OFLO|ENP))==(OFLO|ENP))
                        {
                                statistics.oflo++;
                                statistics.fflags |= F6;
                        }
                        if(rmd1 & CRC)
                        {
                                statistics.crc++;
                                statistics.fflags |= F7;
                        }
                        if(rmd1 & RBUFF)
                        {
                                statistics.rbuff++;
                                statistics.fflags |= F8;
                        }
                        rint_buff[ecb.ind_w] = ecb.rb_r;
                        ecb.ind_w = ++ecb.ind_w & R_BAN1; /*weiter(ecb.ind_w, R_BANZ);*/
                        ecb.rb_r  = ++ecb.rb_r  & R_BAN1;/*weiter(ecb.rb_r, R_BANZ);*/
                }
        }

        RDP = INEA;
}


int eth_recv(struct interface *interface)
{
        register int index;

        while( (index = rint_buff[ecb.ind_r]) != -1 && Cconis()==0 )
        {
                rint (index, interface);
                rint_buff[ecb.ind_r] = -1;
                ecb.ind_r = ++ecb.ind_r & R_BAN1; /*weiter (ecb.ind_r, R_BANZ);*/
        }
        if (statistics.fflags)
                fehler();
        
        return 0;
}


int eth_stop(int16 dev)
{
        int d;

        RAP = CSR0;
        RDP = STOP;

        /* kurze Verzoegerung */
        for (d=0;d<100;d++)
                ; 

        return 0;
}


/************************************************************/
/* Attach a LPR LANCE Ethernet controller to the system     */
/* argv[0]: hardware type, must be "LANCE"					*/
/* argv[1]: maximum transmission unit, bytes, e.g., "1500"  */
/* argv[2]: Ethernetaddress -- in HEX / ASCII --            */

int eth_attach(int argc, char *argv[])
{
        register struct interface *interface;
        extern struct interface *ifaces;

        if ( strcmp(argv[0], "LANCE") )
        {
                printf("%s: unknown hardware\n", argv[0]);
                return -1;
        }

        arp_init (ARP_ETHER, EADDR_LEN, IP_TYPE, ARP_TYPE, ether_bdcst, pether, gaether);

        if ( (interface = (struct interface *) calloc (1, sizeof(struct interface)) ) == NULLIF
                || (interface->name = malloc((unsigned)strlen(argv[0])+1)) == NULLCHAR)
        {
                printf("eth_attach: no memory!\n");
                return -1;
        }
        
        strcpy(interface->name,argv[0]);
        interface->mtu = atoi(argv[1]);
        interface->send = eth_send;
        interface->output = eth_output;
        interface->raw = eth_raw;
        interface->recv = eth_recv;
        interface->stop = eth_stop;
        interface->init = eth_init;
        interface->dev = 0;
        interface->hwaddr = malloc(EADDR_LEN);

        /* Synthax prfen */ 
        gether (interface->hwaddr, argv[2]);  

        /* EAP auslesen */
        if ( ((int *)interface->hwaddr)[2] == 0 )
             ((int *)interface->hwaddr)[2] = EAP;  

		/* Initialize device */
        if ((*interface->init)(interface) != 0)
        {
                free(interface->name);
                free(interface->hwaddr);
                free((char *)interface);
                return -1;
        }
        interface->next = ifaces;
        ifaces = interface;

        return 0;
}



/****************************************************************************/
DRE*
ring_init(char art, DRE deskr[])
{
        int i;

        deskr = (DRE*)( ((long)deskr & 0xfffffff8L) + 0x8L);

        if(art == 't'){
                for(i=0; i<T_BANZ; i++){
                        deskr[i].b_lw = (int)(tbuf[i]);
                        deskr[i].b_hb = (char) ( ((long)(tbuf[i]) >> 16) & 0xff);
                        deskr[i].status = 0x0;
                        deskr[i].ertdmc = 0x0;
                }
        }

        if(art == 'r'){
                for(i=0; i<R_BANZ; i++){
                        deskr[i].b_lw = (int)(rbuf[i]);
                        deskr[i].b_hb = (char) ( ((long)(rbuf[i]) >> 16) & 0xff);
                        deskr[i].status = 0x80;
                        deskr[i].bcnt = (-BGR) | ONES;
                        deskr[i].ertdmc = 0x0;
                }
        }

        return deskr;
}

/****************************************************************************/
/*
int
weiter(int zeiger, int maximum)
{
        if(zeiger < maximum - 1)
                zeiger++;
        else
                zeiger = 0;

        return zeiger;
}
*/



/****************************************************************************/
int
rint(int index, struct interface *interface)
{
        int status;
        int mcnt;
        struct mbuf *bp;
        struct ether header;
        char multicast;
        int rval = 0;

        status = rring[index].status;
        mcnt = rring[index].ertdmc;

        if((status & OWN) != 0 || (status & (STP | ENP)) == 0)
                 rval = -1;
        else
        {
                if(status & DERR)
                        rval = -1;
                else
                {
                        bp = alloc_mbuf(mcnt);
                        if(bp == NULLBUF){
                                printf("LANCE: no buffer for received packet\n");
                                rval = -1;
                        }
                        else
                        {
                                memcpy(bp->data, rbuf[index], mcnt);
                                bp->cnt = mcnt;

                                dump(interface, IF_TRACE_IN, TRACE_ETHER, bp);

                                pullup(&bp, header.dest, EADDR_LEN);
                                pullup(&bp, header.source, EADDR_LEN); 
                                header.type = pull16(&bp);

                                if(header.dest[0] & 1){
                                        multicast = 1;
                                        statistics.bdcasts++;
                                }
                                else
                                        multicast = 0;

                                switch(header.type)
                                {
                                        case ARP_TYPE :
                                                arp_input(interface, bp);
                                                break;
                                        case IP_TYPE  :
                                                ip_route(bp, multicast); 
                                                break;
                                        default       :
                                                if(interface->trace != 0)
                                                        printf("LANCE: unknown protocol\n");
                                                free_p(bp);
                                                break;

                                }
                        }
                }
        }
        statistics.r_count++;
        rring[index].status = OWN;
        return (rval);
}



/****************************************************************************/
int
fehler( void )
{
        int flags;

        flags = statistics.fflags ;
        if(flags & (F1 | F4 | F6 | F8 | F12 | F13)){
                eth_stop(0);	/* parameter not used */
                if(flags & F1)
                        printf("Error: BABL\n");
                if(flags & F4)
                        printf("Error: MERR\n");
                if(flags & F6)
                        printf("Error: OFLO\n");
                if(flags & F8)
                        printf("Error: RBUFF\n");
                if(flags & F12)
                        printf("Error: TBUFF\n");
                if(flags & F13)
                        printf("Error: UFLO\n");
                printf("LANCE stopped!\n");
        }
        else
                if( flags & (F2 | F15) )
                {
                        /*
                        eth_stop();
                        */
                        if(flags & F2)
                                printf("Error: CERR\n");
                        if(flags & F15)
                                printf("Error: LCAR\n");
                        printf("Check ethernet drop-cable for possible errors ?\n");
                }

        statistics.fflags = 0;
        return 0;
}

/****************************************************************************/
int
doetherstat(int argc, char** argv)
{
        printf("########### STATISTICS: ####################\n");
        printf("        t_count  %ld\n", statistics.t_count);
        printf("        r_count  %ld\n", statistics.r_count);
        printf("        bdcasts  %ld\n", statistics.bdcasts);
        printf("        babl     %d\n", statistics.babl);
        printf("        cerr     %d\n", statistics.cerr);
        printf("        miss     %d\n", statistics.miss);
        printf("        merr     %d\n", statistics.merr);
        printf("        fram     %d\n", statistics.fram);
        printf("        oflo     %d\n", statistics.oflo);
        printf("        crc      %d\n", statistics.crc);
        printf("        rbuff    %d\n", statistics.rbuff);
        printf("        more     %d\n", statistics.more);
        printf("        one      %d\n", statistics.one);
        printf("        def      %d\n", statistics.def);
        printf("        tbuff    %d\n", statistics.tbuff);
        printf("        uflo     %d\n", statistics.uflo);
        printf("        lcol     %d\n", statistics.lcol);
        printf("        lcar     %d\n", statistics.lcar);
        printf("        rtry     %d\n", statistics.rtry);
        printf("###########################################\n");
        return 0;
}
