
/********************************************************
 * File:	bsc_ascii.c
 *
 * Description:
 *	Contains routines for translating binary data
 *	to ascii and vice versa.
 *
 * Routines:
 *	bsc_toascii
 *	bsc_fromascii	
 *
 * Author:
 *	Jonathan Masel
 ********************************************************/


#include "modules.h"
#include "msg.h"
#include "bss.h"


#define	bin2ascii(x)	((x>=0) && (x<=9))? x+='0': x+=('a'-0xa)
#define	ascii2bin(x)	((x>='0') && (x<='9'))? x-='0': x-=('a'-0xa)


/********************************************************
 * routine:	bsc_toascii
 *
 * description:
 *	A T_FRAME of binary data is destined for
 *	transmission. Allocate a new T_FRAME and
 *	fill it with ascii values of the data.
 *	This means that the data is now double the
 *	length of the original.
 *	Used when operating in non-transparent mode.
 *	Note that in the target T_FRAME (t1), only
 *	the data buffer is used: this must be large enoug
 *	to accomodate 2 * (hlen + blen) of the original.
 *
 * arguments:
 *	t		the T_FRAME with binary data
 *	child		the channel's child structure
 *
 * return code:
 *	t1		the T_FRAME with ascii data
 *	0		on error
 *
 * side effects:
 *
 ********************************************************/
T_FRAME *
bsc_toascii(t, child)
T_FRAME *t;
BISYNC_LINK *child;
{
	register int i;
	T_FRAME *t1;
	register unsigned char *h, *b, *a;
	register int hlen, blen;
	LAYER_INFO *lt, *lt1;
	unsigned char nyb1, nyb2;
	extern T_FRAME *gett();

	t1 = gett(T_BUF, LAYER(BISYNC), child->key, 0);
	if( !t1 )
		return(0);

	lt = &t->layer[LAYER(BISYNC)];
	lt1 = &t1->layer[LAYER(BISYNC)];
	h = (unsigned char *)&t->hbuf[lt->hoff];
	b = &t->buf[lt->boff];
	hlen = t->hsize - lt->hoff;
	blen = t->bsize - lt->boff;
	if( t1->bsize < (2 * (hlen+blen)) ){
		relm(t1);
		return(0);
	}

	/* initialize destination data pointer */
	lt1->boff -= (2 * (hlen+blen));
	a = t1->buf + lt1->boff;

	/* copy header buffer */
	i = hlen;
	while( i-- ){
		nyb1 = ((*h) & 0xf0) >> 4;
		nyb2 = (*h) & 0x0f;
		bin2ascii(nyb1);
		bin2ascii(nyb2);
		*a++ = nyb1;
		*a++ = nyb2;
		h++;
	}

	/* copy data buffer */
	i = blen;
	while( i-- ){
		nyb1 = ((*b) & 0xf0) >> 4;
		nyb2 = (*b) & 0x0f;
		bin2ascii(nyb1);
		bin2ascii(nyb2);
		*a++ = nyb1;
		*a++ = nyb2;
		b++;
	}

	return(t1);
}


/********************************************************
 * routine:	bsc_fromascii
 *
 * description:
 *	An R_FRAME with ascii data has been received.
 *	The data is translated to binary and the
 *	original buffer returned to the receive buffer pool.
 *	Used when operating in non-transparent mode.
 *
 * arguments:
 *	r		the R_FRAME with ascii data
 *	child		the channel's child structure
 *
 * return code:
 *	r1		the R_FRAME with binary data
 *
 * side effects:
 *
 ********************************************************/
R_FRAME *
bsc_fromascii(r, child)
R_FRAME *r;
BISYNC_LINK *child;
{
	register R_FRAME *r1, *r2, *r3;
	register unsigned char nyb1, nyb2;
	register unsigned char *rb;
	register int i, blen;
	char split;
	R_FRAME *rtmp;
	extern R_FRAME *getr();

	r1 = r3 = 0;
	rtmp = r;
	split = 0;
	while( r ){
		r2 = getr(R_BUF, child->key);
		if( !r2 ){
			if( r1 ) relm(r1);
			relm(rtmp);
			return(0);
		}
		r2->flags = 0;	/* not RX_POOL */
		if( !r1 )
			r1 = r2;
		else
			r3->nextb = r2;
		r3 = r2;
		rb = r2->buf + r2->boff;
		i = 0;
		blen = 0;
		while( i < r->blen ){
			/*
			 * read next two data bytes.
			 */
			if( split )
				split = 0;
			else {
				nyb1 = r->buf[i+(r->boff)];
				nyb1 &= ~0x80;
				i++;
				if( i >= r->blen ){
					split = 1;
					break;
				}
			}
			nyb2 = r->buf[i+(r->boff)];
			nyb2 &= ~0x80;
			i++;
			ascii2bin(nyb1);
			ascii2bin(nyb2);
			*rb++ = ((nyb1 << 4) | nyb2);
			blen++;
		}
		r2->blen = blen;
		r1->flen += blen;
		r = r->nextb;
	}

	/*
	 * update receive error bits
	 */
	r1->hdr.status = rtmp->hdr.status;
	r1->hdr.err_info = rtmp->hdr.err_info;

	relm(rtmp);
	return(r1);
}
