
/********************************************************
 * File:	bisync_rx
 *
 * Description:
 *	This file handles reception of BISYNC messages.
 *
 * Routines:
 *
 * Author:
 *	Jonathan Masel
 ********************************************************/


#include "gct.h"
#include "msg.h"
#include "modules.h"
#include "types.h"
#include "status.h"
#include "bss.h"
#include "bsc.h"


/********************************************************
 * routine:	bsc_rxind
 *
 * description:
 *	This is the primtive called to handle received frames.
 *
 * arguments:
 *	r		points to the R_FRAME received
 *	child		the channel's child structure
 *
 * return code:
 *
 * side effects:
 *
 ********************************************************/
bsc_rxind(r, child)
register R_FRAME *r;
BISYNC_LINK *child;
{
	int mtype;
	T_FRAME *t;
	GCT *gct;

	/* only handle valid frames */
	if( r->hdr.status ){
		bsc_sendnak(child);
		relm(r);
		return;
	}

	GETGCT(gct);
	mtype = bsc_what(r, child);
	switch( mtype ){
	case SYN:
	case SOH:
	case WACK:
	case RVI:
	case TTD:
	case ITB:
	case ETB:
	case EOT:
	case ETX:
	case -1:
		/*
		 * all above characters are either invalid
		 * at the start of a message, or define
		 * a message type not supported
		 */
		break;
	case STX:
		bsc_rx(r, child);
		return;
	case ENQ:
		if( child->vr == 0 )
			bsc_send(ACK0, 2, child);
		else if( child->vr == 1 )
			bsc_send(ACK1, 2, child);
		break;
	case NAK:
		bsc_retx(child);
		break;
	case ACK0:
		if( child->va == 0 )
			goto ack;
		if( child->flags & TIMER_RECOVERY )
			bsc_retx(child);
		break;
	case ACK1:
		if( child->va == 1 )
			goto ack;
		if( child->flags & TIMER_RECOVERY )
			bsc_retx(child);
		break;
ack:
		t = child->i_conf;
		if( t ){
			child->i_conf = t->layer[LAYER(BISYNC)].nextf;
			conf_frm(t, child->upper);
			if( child->outst > 0 )
				--child->outst;
		}
		if( !child->i_conf )
			child->i_tail = 0;
		child->va = 1 - child->va;
		/*
		 * if acknowledgements are still outstanding,
		 * restart the acknowledge timer.
		 * otherwise, stop it
		 */
		if( child->va != child->vs )
			start_timer(gct->bisync->timer_table,
				r->hdr.id, BISYNC_TIMER, child->Tack);
		else
			stop_timer(gct->bisync->timer_table,
				r->hdr.id);

		bsc_tx(child);
		break;
	}

	/* this is reached for all but data messages */
	relm(r);
}


/********************************************************
 * routine:	bsc_what
 *
 * description:
 *	Decide what sort of a frame has just been received.
 *
 * arguments:
 *	r		the R_FRAME received
 *	child		the channel's child structure
 *
 * return code:
 *	type of message received
 *	-1 on error
 *
 * side effects:
 *
 ********************************************************/
bsc_what(r, child)
R_FRAME *r;
BISYNC_LINK *child;
{
	register unsigned char *c;
	register int i;

	c = r->buf + r->boff;
	i = 0;
swtch:
	i++;
	if( i > 2) return(-1);
	switch( *c ){
	case STX:
		if( (r->blen > 1) && (*(c+1) == ENQ) )
			return(TTD);
	case SYN:
	case SOH:
	case ITB:
	case ETB:
	case ETX:
	case EOT:
	case ENQ:
	case NAK:
		return(*c);
	case DLE:
		if( i > 1 ) return(-1);
		if( i > r->blen ) return(-1);
		switch( *++c ){
		case '0':
			return(ACK0);
		case '1':
			return(ACK1);
		case ';':
			return(WACK);
		case '<':
			return(RVI);
		}
		goto swtch;
	}
}


/********************************************************
 * routine:	bsc_rx
 *
 * description:
 *	Handle a received text message.
 *
 * arguments:
 *	r		the message received
 *	child		the channel's child structure
 *
 * return code:
 *
 * side effects:
 *
 ********************************************************/
bsc_rx(r, child)
register R_FRAME *r;
register BISYNC_LINK *child;
{
	register int i, tr;
	R_FRAME *r1;
	GCT *gct;
	extern R_FRAME *bsc_fromascii();

	GETGCT(gct);
	tr = child->flags & TRANSPARENT;
	/* remove [DLE] STX at head of message */
	i = tr? 2: 1;
	r->boff += i;
	r->blen -= i;
	r->flen -= i;

	/* remove ETX at tail
	(use last R_FRAME in message) */
	r1 = r;
	while( r1->nextb )
		r1 = r1->nextb;
	r1->blen--;
	r->flen--;

	if( !tr ){
		r = bsc_fromascii(r, child);
		if( !r )
			return(-1);
	}

	bsc_trace(r);

	/* send message to upper layer */
	(*gct->send)(child->upper, r);

	/* acknowledge message */
	if( child->vr == 0 )
		bsc_send(ACK0, 2, child);
	else if( child->vr == 1 )
		bsc_send(ACK1, 2, child);
	child->vr = 1 - child->vr;
}
