
/********************************************************
 * File:	bsc_tx.c
 *
 * Description:
 *	This file contains routine for handling
 *	transmission of all types of supported BISYNC messages.
 *
 * Routines:
 *	bsc_txreq
 *	bsc_timeout
 *	bsc_retx
 *	bsc_sendack0
 *	bsc_sendack1
 *	bsc_sendnak
 *	bsc_sendenq
 *	bsc_send
 *
 * Author:
 *	Jonathan Masel
 ********************************************************/


#include "gct.h"
#include "msg.h"
#include "modules.h"
#include "types.h"
#include "status.h"
#include "bss.h"
#include "bsc.h"
#include "fatal.h"


/********************************************************
 * routine:	bsc_txreq
 *
 * description:
 *	This is the primitive called from an upper layer
 *	to request transmission of data.
 *
 * arguments:
 *	t		points to the T_FRAME received
 *	child		the child structure of the channel
 *
 * return code:
 *
 * side effects:
 *
 ********************************************************/
bsc_txreq(t, child)
register T_FRAME *t;
BISYNC_LINK *child;
{
	if( child->i_tail ){
		child->i_tail->layer[LAYER(BISYNC)].nextf = t;
		child->i_tail = t;
	} else {
		child->i_head = child->i_tail =
		child->i_conf = child->i_ack = t;
	}
	if( !child->i_head )
		child->i_head = t;

	bsc_tx(child);
}


/********************************************************
 * routine:	bsc_tx
 *
 * description:
 *	Flush the queue of T_FRAMES awaiting transmission.
 *	The transmit window size is two.
 *
 * arguments:
 *	child		the child structure of the channel concerned
 *
 * return code:
 *	number of frames sent to driver for transmission
 *
 * side effects:
 *
 ********************************************************/
bsc_tx(child)
BISYNC_LINK *child;
{
	GCT *gct;
	register T_FRAME *t, *t1;
	register LAYER_INFO *l, *lower;
	register int i, tr;
	register int count = 0;
	extern T_FRAME *bsc_toascii();

	GETGCT(gct);
	t = child->i_head;

	while( child->outst < child->wsize ){
		if( !t )
			break;

		tr = child->flags & TRANSPARENT;
		if( !tr ){
			t1 = bsc_toascii(t, child);
			if( !t1 ){
				break;
			}
			/* clear bit that the driver would clear */
			t->hdr.rsp_req &= ~BISYNC_RSP;
		} else {
			t->hdr.rsp_req |= BISYNC_RSP;
			t1 = t;
		}


		/* prepare lower layer's info */
		lower = &t1->layer[LAYER(child->lower)];
		lower->flags = 0;
		lower->boff = t1->layer[LAYER(BISYNC)].boff;
		lower->hoff = t1->layer[LAYER(BISYNC)].hoff;
		lower->nextf = 0;

		/*
		 * driver will add STX at start of frame
		 * and ETX at end
		 */
		lower->flags = BISYNC_STX;
		if( tr )
			lower->flags |= BISYNC_TRANSP;

		/* advance vs variable */
		child->vs = 1 - child->vs;

		/* advance queue pointers */
		child->i_head = t->layer[LAYER(BISYNC)].nextf;
		child->outst++;

		bsc_trace(t);

		/* send to lower layer */
		(*gct->send)(child->lower, t1);

		count++;
		t = child->i_head;
	}
	if( count ){
		start_if_stopped(gct->bisync->timer_table,
			child->key, BISYNC_TIMER, child->Tack);
	}
	return(count);
}


/********************************************************
 * routine:	bsc_timeout
 *
 * description:
 *	This is the primitive routine called on
 *	expiration of the acknowledge timer.
 *	Send an ENQ.
 *
 * arguments:
 *	h		points to the message received
 *	child		the child structure of the channel
 *
 * return code:
 *
 * side effects:
 *
 ********************************************************/
bsc_timeout(h, child)
HDR *h;
BISYNC_LINK *child;
{
	GCT *gct;

	bsc_trace(h);
	relm(h);
	if( child->flags & TIMER_RECOVERY ){
		bsc_retx(child);
	} else {
		GETGCT(gct);
		child->flags |= TIMER_RECOVERY;
		bsc_sendenq(child);
		start_timer(gct->bisync->timer_table,
			child->key, BISYNC_TIMER, child->Tack);
	}
}

/********************************************************
 * routine:	bsc_retx
 *
 * description:
 *	This routine is called to handle frame re-transmission.
 *	It is called on timeouts and on reception of
 *	negative acknowledgements.
 *	It assumes a) that infinite re-tries are to be done
 *	and b) that the entire transmission window is to
 *	be re-transmitted (this is reasonable, since its size is 1).
 *
 * arguments:
 *	the link for which re-tranmission is to be done
 *
 * return code:
 *
 * side effects:
 *
 ********************************************************/
bsc_retx(child)
BISYNC_LINK *child;
{
	/*
	 * leave the timer recovery state
	 * (this implies infinite re-tries)
	 */
	child->flags &= ~TIMER_RECOVERY;

	/*
	 * reset queue pointers and parameters.
	 * since the entire window is to be re-txed,
	 * outstanding frame count is set to zero
	 */
	child->i_head = child->i_conf;
	child->vs = child->va;
	child->outst = 0;

	/* transmit window again */
	bsc_tx(child);
}

bsc_sendack0(child)
BISYNC_LINK *child;
{
	bsc_send(ACK0, 2, child);
}

bsc_sendack1(child)
BISYNC_LINK *child;
{
	bsc_send(ACK1, 2, child);
}

bsc_sendnak(child)
BISYNC_LINK *child;
{
	bsc_send(NAK, 1, child);
}

bsc_sendenq(child)
BISYNC_LINK *child;
{
	bsc_send(ENQ, 1, child);
}


/********************************************************
 * routine:	bsc_send
 *
 * description:
 *	Send a BSC control character sequence
 *
 * arguments:
 *	ctrl	the value of the sequence (up to 4 bytes)
 *	len	the length of the sequence
 *	child	the channel's child structure
 *
 * return code:
 *
 * side effects:
 *
 ********************************************************/
bsc_send(ctrl, len, child)
unsigned int ctrl;
int len;
BISYNC_LINK *child;
{
	register T_FRAME *t;
	extern T_FRAME *gett();
	GCT *gct;
	LAYER_INFO *l;
	unsigned char c;
	int hoff;

	if( (len > 4) || (len < 1) )
		return(0);

	GETGCT(gct);
	t = gett(T_HDR, LAYER(BISYNC), child->key, 0);
	if( !t ){
		return(0);
	}

	hoff = t->layer[LAYER(BISYNC)].hoff;
	while( len-- ){
		c = ctrl & 0xff;
		t->hbuf[--hoff] = c;
		ctrl = ctrl >> 8;
	}
	t->layer[LAYER(BISYNC)].hoff = hoff;

	/* prepare rest of lower layers info */
	l = &t->layer[LAYER(child->lower)];
	l->hoff = hoff;
	l->flags = 0;
	l->boff = t->layer[LAYER(BISYNC)].boff;
	l->nextf = 0;

	(*gct->send)(child->lower, t);
	return(1);
}
