/*
 * psend.c
 *	download file (executable) to board
 *	via parallel port (and PIB)
 *
 * Compile time constants are used for two categories:
 *	a) the target host computer
 *	b) the binary format of the file being down-loaded
 * The following compile time flags may be used:
 *	target host is one out of the following:
 *		-Dsun, -Dibm_pc, -Dmacintosh, -DsystemV
 *	binary file format is
 *		-DBSD -DSUN	SUN linker format
 *		-DCOFF		Common Object File Format (system V)
 *		-DSREC		Motorola S records
 *
 * Routines have been called from the C libraries
 * for the following purposes:
 * 1. file handling
 *          These routines all follow standard C (Unix) interfaces.
 * 2. interactive i/o
 *          printf and scanf (sscanf, etc.) are used. These should
 *          also be standard amongst C compilers.
 * 3. port i/o
 *    In the IBM/PC and MAC-II versions, the routines inp and outp
 *    are called for reading and writing a byte to the ADI board.
 *    The MAC-II version is supplied (simple memory based accesses).
 *    The IBM/PC version was compiled with MicroSoft C: inp and
 *    outp belong to the MicroSoft C library routines.
 *    Their purpose is to transfer bytes over a BIOS i/o port.
 *    If a different C compiler is used, there should be routines
 *    that perform these functions in the C library, but their names
 *    may differ. In this case, it will be necessary to either alter
 *    the calls to inp and outp (or write routines inp and outp
 *    that call the library routines).
 */

#include <fcntl.h>
#include <stdio.h>
#include <ctype.h>

#ifdef sun
#include <string.h>
#include <signal.h>
#include <sys/mman.h>
#endif

#include "aout.h"


#ifdef ibm_pc
#include <sys/types.h>
#include <sys/stat.h>
#endif /* ibm_pc */

#define	MAXIMUM	0x2000	/* must precede include of psend.h */
#include "psend.h"


char		*cboard;
int		ads_board;
unsigned long	address;		/* address reached in load */
unsigned int	adi_board;

#ifdef sun
int		VMEfd;
#endif

#if defined(sun) || defined(systemV)
int		DeviceTarget;
#ifdef PIB
int		PibBoard;
#endif
#endif

#ifdef BSD
#define TXT_SIZE(x)	((x).a_magic==ZMAGIC? (x).a_text - sizeof(struct exec): (x).a_text)
#endif /* BSD */

#ifdef SREC
unsigned long	entry_point;
#endif

int		RelocAdd;
unsigned long	BaseReloc;

PSEND		sect;

main(argc, argv)
int argc;
char *argv[];
{
	int i, n;
	int filearg;
	FILE *f;		/* input file descriptor */
	int go;
	int format;
	char line[32];
	int options;		/* number of options given */
#ifdef BSD
	struct exec ahdr;
#endif /* BSD */
#ifdef COFF
	static struct filehdr fhdr;
	static struct aouthdr ahdr;
	static struct scnhdr shdr;
	int inquire;
	int section;
	FILE *random;
#endif /* COFF */
#ifdef SREC
	static struct srec srec;
#endif /* SREC */

	if( argc < 3 ){
#ifdef COFF
		printf("Usage: %s [-g] [-i] file board\n", argv[0]);
#else
		printf("Usage: %s [-g] file board\n", argv[0]);
#endif /* COFF */
		exit(1);
	}

#ifdef sun
	VMEfd = -1;
	DeviceTarget = 0;
#ifdef PIB
	PibBoard = 0;
#endif
#endif

	/*
	 * target board may be given as either a device file name
	 * (for UNIX systems) or the OR of ADI and ADS addresses	
	 */
	if( allHex(argv[argc-1]) ){
		i = sscanf(argv[argc-1], "%x", &adi_board);
		if( i < 1 )
			BadTarget(argv[argc-1]);
#ifdef PIB
		if( adi_board > 0xffff )
			PibBoard = 1;
#endif

		/*
		 * parameter is given as bit-OR of ADI address
		 * with ADS index (last 3 bits)
		 */
		ads_board = adi_board & 0x7;
		adi_board &= ~7;
		i = parallel_init(adi_board, ads_board);
		if( i == -1 ) exit(1);
	} else {
#if defined(ibm_pc) || defined(macintosh)
		BadTarget(argv[argc-1]);
#else
		DeviceTarget = 1;
		cboard = argv[argc-1];
		ads_board = open(cboard, O_WRONLY);
		if( ads_board == -1 ){
			perror(cboard);
			exit(1);
		}
#endif
	}

	BaseReloc = 0;
	RelocAdd = 0;

	/*
	 * check for go option
	 */
	if( !strcmp(argv[1], "-g") )
		go = 1;
	else
		go = 0;
	options = go;

#ifdef COFF
	inquire = 0;
	if( !strcmp(argv[1+options], "-i") ){
		inquire = 1;
		options++;
	}
#endif /* COFF */

/*********************************************************/
/***          MAIN LOOP FOR EACH FILE                  ***/
/*********************************************************/

	for(filearg = options+1; filearg < (argc-1); filearg++){

	/*
	 * check for reloc option
	 */
	if( !strcmp(argv[filearg], "-add") ){
		sscanf(argv[filearg+1], "%lx", &BaseReloc);
		RelocAdd = 1;
		options += 2;
		filearg++;
		continue;
	}
	if( !strcmp(argv[filearg], "-sub") ){
		sscanf(argv[filearg+1], "%lx", &BaseReloc);
		RelocAdd = 0;
		options += 2;
		filearg++;
		continue;
	}

#ifdef COFF
	random = fopen(argv[filearg], "r");
	if( !random ){
		perror(argv[filearg]);
		goto next_file;
	}
#endif
	/*
	 * f is used to read file header
	 */
#ifdef ibm_pc
	f = fopen(argv[filearg], "rb");
#else
	f = fopen(argv[filearg], "r");
#endif
	if( !f ){
		perror(argv[filearg]);
		goto next_file;
	}


/*************************************************
	start BSD section
*************************************************/
#ifdef BSD
	/*
	 * read headers, extracting relevant information
	 */
	i = fread(&ahdr, sizeof(struct exec), 1, f);
#ifdef DEBUG
	printf("read ahdr returned %d\n", i);
#endif
#ifdef ibm_pc
	swapaout(&ahdr);

#ifdef DEBUG
	printf("aout.a_magic = %x\n", ahdr.a_magic);
	printf("aout.a_text = %lx\n", ahdr.a_text);
	printf("aout.a_data = %lx\n", ahdr.a_data);
	printf("aout.a_entry = %lx\n", ahdr.a_entry);
#endif
#endif /* ibm_pc */

	if( N_BADMAG(ahdr) ){
		printf("bad magic (0%o)\n", ahdr.a_magic);
		goto next_file;
	}

	/*
	 * send text and data sections
	 * (bss is never sent)
	 */
	address = ahdr.a_entry;
	sect.hdr.address = address;
	sect.hdr.flags = 0;
	sect.hdr.length = 0;
#ifdef DEBUG
#ifdef ibm_pc
printf("sending to address %lx\n", sect.hdr.address);
#else
printf("sending to address %x\n", sect.hdr.address);
#endif /* ibm_pc */
#endif /* DEBUG */

	/*
	 * send data section
	 */
	send(f, TXT_SIZE(ahdr) + ahdr.a_data);

#ifdef DEBUG
	printf("\tsent");
#endif /* DEBUG */

	printf("\n");

#endif /* BSD */

/*************************************************
	end BSD section
*************************************************/


/*************************************************
	start COFF section
*************************************************/
#ifdef COFF
	fread(&fhdr, sizeof(struct filehdr), 1, f);
	fread(&ahdr, sizeof(struct aouthdr), 1, f);
#ifdef ibm_pc
	swapfhdr(&fhdr);
	swapahdr(&ahdr);
#endif /* ibm_pc */

	if( (fhdr.f_magic != MC68MAGIC) &&
		(fhdr.f_magic != MC68TVMAGIC) &&
		(fhdr.f_magic != M68MAGIC) &&
		(fhdr.f_magic != M68TVMAGIC) ){
		fprintf(stderr, "Bad magic in file header (%x)\n",
			fhdr.f_magic);
		goto next_file;
	}

	for(section = 0; section < fhdr.f_nscns; section++){
		fread(&shdr, sizeof(struct scnhdr), 1, f);
#ifdef ibm_pc
		swapscn(&shdr);
#endif /* ibm_pc */
		if( !shdr.s_scnptr )
			continue;
		if( fseek(random, shdr.s_scnptr, 0) == -1 ){
			perror("lseek");
			goto next_file;
		}
		address = shdr.s_paddr;
		sect.hdr.address = address;
		sect.hdr.flags = 0;
		sect.hdr.length = 0;

		printf("%s to %lx (%lx bytes)  ",
			shdr.s_name, shdr.s_paddr, shdr.s_size);
		fflush(stdout);
		if( inquire ){
			printf("(y/n? ");
			fflush(stdout);
			gets(line);
			if( (line[0] != 'y') && (line[0] != 'Y') )
				goto next_file;
		}

		send(random, shdr.s_size);
		printf("\tsent\n");
	}

#endif /* COFF */

/*************************************************
	end COFF section
*************************************************/


/*************************************************
	start S-record section
*************************************************/

#ifdef SREC

	sect.hdr.flags = 0;
	for(;;){
		i = Sparse(f, &srec, sect.data);

		if( i <= 0 ){
			if( i == 0 )
				continue;
			else
				break;
		}
		if( RelocAdd )
			srec.address += BaseReloc;
		else
			srec.address -= BaseReloc;

#ifdef DEBUG
#ifdef ibm_pc
		printf("parsed: adr = %lx, length = %x\n",
			srec.address, srec.length);
#endif
#endif /* DEBUG */
		sect.hdr.address = srec.address;
		n = sect.hdr.length = srec.length;
#ifdef ibm_pc
		swapphdr(&sect);
#endif
#if defined(sun) || defined(systemV)
		if( DeviceTarget )
			i = write(ads_board, &sect, n+sizeof(PHDR));
		else
#endif
		i = parallel_write(adi_board, &sect, n+sizeof(PHDR));
#ifdef ibm_pc
		swapphdr(&sect);
#endif
		if( i < n ){
			perror("write");
			goto next_file;
		}
	}

#endif /* SREC */

/*************************************************
	end S-record section
*************************************************/

	/*
	 * send last packet with entry point only
	 */
	sect.hdr.flags = ENTRY_POINT | LAST_PKT;
	if( go )
		sect.hdr.flags |= GO;
#ifdef BSD
	sect.hdr.address = ahdr.a_entry;
#ifdef DEBUG
	printf("entry point is %x\n", ahdr.a_entry);
#endif /* DEBUG */
#endif
#ifdef COFF
	sect.hdr.address = ahdr.entry_pt;
#ifdef DEBUG
	printf("entry point is %x\n", ahdr.entry_pt);
#endif /* DEBUG */
#endif /* COFF */
#ifdef SREC
	sect.hdr.address = entry_point;
#ifdef DEBUG
	printf("entry point is %x\n", entry_point);
#endif /* DEBUG */
#endif /* SREC */

	if( RelocAdd )
		sect.hdr.address += BaseReloc;
	else
		sect.hdr.address -= BaseReloc;
	sect.hdr.length = 0;

#ifdef ibm_pc
	swapphdr(&sect);
#endif
#if defined(sun) || defined(systemV)
	if( DeviceTarget )
		i = write(ads_board, &sect, sizeof(PHDR));
	else
#endif
	i = parallel_write(adi_board, &sect, sizeof(PHDR));
#ifdef ibm_pc
	swapphdr(&sect);
#endif

	if( i < 1 ){
		perror("write1");
		goto next_file;
	}
next_file:
#ifdef COFF
	fclose(random);
#endif /* COFF */
	fclose(f);
	}
/*********************************************************/
/***        EACH OF MAIN LOOP FOR EACH FILE            ***/
/*********************************************************/

#if defined(sun) || defined(systemV)
	if( DeviceTarget )
		close(ads_board);
	else
		if( VMEfd != -1 )
			close(VMEfd);
#endif
	exit(0);	/* clean exit */
}

send(f, size)
FILE *f;			/* a file descriptor */
unsigned long size;		/* the number of bytes to send */
{
	unsigned long i, n;		/* number of bytes to read */
	long j;
	long req;

	i = 0;
	if( RelocAdd )
		sect.hdr.address += BaseReloc;
	else
		sect.hdr.address -= BaseReloc;
	sect.hdr.flags = 0;
	while(i < size){
		/*
		 * read at most MAXIMUM bytes
		 */
		n = size - i;
		if( n > MAXIMUM )
			n = MAXIMUM;
		req = n;
		n = fread(sect.data, 1, n, f);
#ifdef DEBUG
		printf("read %ld bytes from file \n", n);
#endif
		if( n == 0 ){
			if( feof(f) )
				return;
			if( ferror(f) )
				perror("read");
		}

		sect.hdr.length = n;

#ifdef ibm_pc
		swapphdr(&sect);
#endif
#if defined(sun) || defined(systemV)
		if( DeviceTarget )
			j = write(ads_board, &sect, n+sizeof(PHDR));
		else
#endif
		j = parallel_write(adi_board, &sect, n+sizeof(PHDR));
#ifdef ibm_pc
		swapphdr(&sect);
#endif

#ifdef DEBUG
		printf("send-write returns %d\n", j);
		printf("flags=%lx, addr=%lx length=%lx\n",
			sect.hdr.flags, sect.hdr.address,
			sect.hdr.length);
#endif /* DEBUG */

		if( j == -1 ){
			perror("write2");
			exit(3);
		}
		printf(".");
		fflush(stdout);
		sect.hdr.address += n;
		i += n;
	}

	if( RelocAdd )
		sect.hdr.address += BaseReloc;
	else
		sect.hdr.address -= BaseReloc;
}


BadTarget(s)
char *s;
{
	fprintf(stderr, "Bad target address: %s\n", s);
	exit(1);
}

/*
 * checks if an address is given.
 */
allHex(s)
char *s;
{
	while( *s ){
		if( !isxdigit(*s) )
			return(0);
		s++;
	}
	return(1);
}

#ifdef SREC

hex(c)
unsigned char c;
{
	if( isdigit(c) )
		return( c - '0' );
	if( isupper(c) )
		return( 10 + c - 'A' );
	if( islower(c) )
		return( 10 + c - 'a' );
	return(-1);	
}

Sparse(f, s, data)
FILE *f;
struct srec *s;
unsigned char *data;
{
	register int i, j;
	static unsigned char line[512];
	int alen = 0;	/* address length */
	int nyb1, nyb2;
	static char warning[] = "Warning: %s is not an S-record (discarded)\n";

	if( !fgets(line, sizeof(line), f) ){
		if( ferror(f) )
			perror("read");
		return(-1);
	}
	if( (line[0] != 'S') && (line[0] != 's') ){
		fprintf(stderr, warning, line);
		return(0);
	}
	s->type = hex(line[1]);
	s->address = 0;
	s->length = 0;
	switch( line[1] ){
	case '0':
	case '5':
		return(0);
	case '1':
		if( !readadr(line, &s->address, 2) )
			return(1);
		alen = 2;
		break;
	case '2':
		if( !readadr(line, &s->address, 3) )
			return(1);
		alen = 3;
		break;
	case '3':
		if( !readadr(line, &s->address, 4) )
			return(1);
		alen = 4;
		break;
	case '4':
	case '6':
		fprintf(stderr, warning, line);
		return(0);
	case '7':
	case '8':
	case '9':
		readadr(line, &entry_point, 0);
		return(0);
	default:
		fprintf(stderr, warning, line);
		return(0);
	}

	/*
	 * we should only reach here for S1, S2 and S3 records
	 */
	nyb1 = hex(line[2]);
	nyb2 = hex(line[3]);
	s->length = (nyb1 << 4) | nyb2;
	s->length -= alen;	/* discard address bytes */
	s->length--;		/* ignore checksum at end */
	i = 4 + (2*alen);	/* index of first data nybble */
	j = s->length;
	while( j-- ){
		nyb1 = hex(line[i]);
		nyb2 = hex(line[i+1]);
		*data++ = (nyb1 << 4) | nyb2;
		i += 2;
	}
	return(1);
}

readadr(line, l, n)
unsigned char line[];
unsigned long *l;
int n;
{
	register int i;
	unsigned char data;
	long res;
	int nyb1, nyb2;

	res = 0;
	i = 4;
	while( n-- ){
		nyb1 = hex(line[i]);
		nyb2 = hex(line[i+1]);
		data = (nyb1 << 4) | nyb2;
		res = res << 8;
		res |= data;
		i += 2;
	}
	*l = res;
	return(1);
}

#endif /* SREC */

/* write (control) handshake bits */
#define ADS_BRK		0x0001	/* ads break */
#define HOST_REQ	0x0002 	/* request flag */
#define HOST_ACK	0x0004	/* acknowledge flag */
#define ADS_ALL		0x0008	/* address all ads */
#define	ADS_SELECT	0x0070	/* ADS select address */
#define ADS_RESET	0x0080	/* device reset */

/* read (status) hadshake bits */
#define ADS_INT		0x0001 	/* ads service request */
#define ADS_REQ		0x0002	/* ads request */
#define ADS_ACK		0x0004	/* ads acknowledge */
#define HOST_BRK	0x0008	/* host service request */


#if defined(ibm_pc) || defined(macintosh)
#define	ADI_REG	unsigned char
#endif
#if defined(sun) || defined(systemV)
#define	ADI_REG	unsigned short
#endif

#ifdef PIB
struct pib_regs {
	unsigned char fill1;
	unsigned char data;
	unsigned char fill2;
	unsigned char handshake;
};

#define	INIT_PIB(p)	(p)->handshake = 0x0f
#define	RESET_BOARD(p)	(p)->handshake = 0x0b
#define	ABORT_BOARD(p)	(p)->handshake = 0x07
#define ASSERT_HREQ(p)	(p)->handshake = 0x0d
#define	NEGATE_HREQ(p)	(p)->handshake = 0xf
#define	ASSERT_HACK(p)	(p)->handshake = 0x0e
#define	NEGATE_HACK(p)	(p)->handshake = 0xf
#define	IS_REQ(p)	(!((p)->handshake & 1))
#define	IS_ACK(p)	(!((p)->handshake & 2))
#endif

#define	HANDSHAKE(a)	(a)
#define	DATA(a)		(a + sizeof(ADI_REG))

ADI_REG			control;
ADI_REG			status;
extern ADI_REG		inp();


parallel_init(a, ads_board)
unsigned int a;
int ads_board;
{
#ifdef sun
	int i, j, mod, off;
	int size;
	char *VmeSpace;

	DeviceTarget = 0;
	size = getpagesize();
	mod = a % size;
	off = a - mod;
	VmeSpace = "/dev/vme16d16";
#ifdef PIB
	if( PibBoard )
		VmeSpace = "/dev/vme24d16";
#endif
	if( VMEfd == -1 )
		VMEfd = open(VmeSpace, O_RDWR);
	if( VMEfd == -1 ){
		perror(VmeSpace);
		return(-1);
	}
/*
SUN OS/3
	adr = valloc(size);
	if( !adr ){
		perror("valloc");
		return(-1);
	}
	j = PROT_READ | PROT_WRITE;
	i = (int)mmap(adr, size, j, MAP_SHARED, VMEfd, off);
*/
/* SUN OS/4 */
	j = PROT_READ | PROT_WRITE;
	i = (int)mmap(0, size, j, MAP_SHARED, VMEfd, off);
	if( i == -1 ){
		perror("mmap");
		return(-1);
	}
	adi_board = i + mod;
#ifdef PIB
	if( !pibpoke(adi_board) ){
#else
	if( !vmepokew(adi_board) ){
#endif
		fprintf(stderr, "Can't find %x on VMEbus\n", adi_board);
		return(-1);
	}
	a = adi_board;
#endif

#ifdef systemV
	return(-1);
#endif

	init_board(adi_board, ads_board);
	return(0);
}

#ifdef sun
static int SegErr;

SegViolation()
{
	SegErr = 1;
}

vmepokew(adr)
unsigned short *adr;
{
	SegErr = 0;
	signal(SIGSEGV, SegViolation);
	*adr = 0;
	signal(SIGSEGV, SIG_DFL);
	return(SegErr == 0);
}
#ifdef PIB
pibpoke(pib)
struct pib_regs *pib;
{
	SegErr = 0;
	signal(SIGSEGV, SegViolation);
	INIT_PIB(pib);
	signal(SIGSEGV, SIG_DFL);
	return(SegErr == 0);
}
#endif
#endif

parallel_write(a, buf, count)
unsigned int a;
register int count;
unsigned char buf[];
{
	register int i;

#ifdef systemV
	return(-1):
#endif
	for(i = 0; i < count; i++)
		parallel_putc(a, buf[i]);
	i = count;

	return(i);
}

parallel_putc(a, c)
unsigned int a;
register unsigned char c;
{
	register int i;

	write_data(a, c);

	/* select this board */
	select_board(a);

	assert_req(a);

	i = 0;
	while( !ack_asserted(a) ){
		if( i++ > 0x4000 ){
			fprintf(stderr, "Warning: Communication failure");
			fprintf(stderr, "ACK stuck low\n");
			i = 0;
		}
	}

	negate_req(a);

	i = 0;
	while( ack_asserted(a) ){
		if( i++ > 0x4000 ){
			fprintf(stderr, "Warning: Communication failure");
			fprintf(stderr, "ACK stuck high\n");
			i = 0;
		}
	}
}

init_board(adi, ads)
unsigned int adi, ads;
{
#ifdef PIB
	if( PibBoard ){
		INIT_PIB((struct pib_regs *)adi);
		return;
	}
#endif
	/*
	 * zero handshake bits
	 * and set address select bits
	 */
	control = ads << 4;
	outp(HANDSHAKE(adi), control);

}

write_data(a, c)
{
#ifdef PIB
	if( PibBoard ){
		((struct pib_regs *)a)->data = c;
		return;
	}
#endif
	outp(DATA(a), c);
}

select_board(a)
{
	int i;

#ifdef PIB
	if( PibBoard ) return;
#endif
	outp(HANDSHAKE(a), control);
	for(i = 0; i < 10; i++);
}

assert_req(a)
{
	int i;
#ifdef PIB
	if( PibBoard ){
		ASSERT_HREQ((struct pib_regs *)a);
		return;
	}
#endif
	control |= HOST_REQ;
	outp(HANDSHAKE(a), control);
	for(i = 0; i < 10; i++);
}

negate_req(a)
{
	int i;
#ifdef PIB
	if( PibBoard ){
		NEGATE_HREQ((struct pib_regs *)a);
		return;
	}
#endif
	control &= ~HOST_REQ;
	outp(HANDSHAKE(a), control);
	for(i = 0; i < 10; i++);
}

ack_asserted(a)
{
#ifdef PIB
	if( PibBoard )
		return( IS_ACK((struct pib_regs *)a) );
#endif
	status = inp(HANDSHAKE(a));
	return( status & ADS_ACK );
}


#ifndef ibm_pc
ADI_REG
inp(c)
ADI_REG *c;
{
	return(*c);
}

outp(c, x)
ADI_REG *c;
ADI_REG x;
{
	*c = x;
}
#endif

#ifdef macintosh
perror(str)
{
	extern short MacOSErr;
	extern int errno;

	fprintf(stderr, "Error in %s. errno = %d, MacOSErr = %d\n",
			str, errno, MacOSErr);
}
#endif

#ifdef ibm_pc
unsigned swapshort(adr)
short *adr;
{
	unsigned short a, b;

	a = (*adr & 0xff00) >> 8;
	b = (*adr & 0x00ff);
	*adr = (b << 8) | a; 
}

swaplong(adr)
unsigned long *adr;
{
	unsigned long a, b, c, d;

	a = (*adr & 0xff000000) >> 24;
	b = (*adr & 0x00ff0000) >> 16;
	c = (*adr & 0x0000ff00) >> 8;
	d = (*adr & 0x000000ff);
	*adr = (d << 24) | (c << 16) | (b << 8) | a;
}

#ifdef BSD
swapaout(aout)
register struct exec *aout;
{
	swapshort(&aout->a_machtype);
	swapshort(&aout->a_magic);
	swaplong(&aout->a_text);
	swaplong(&aout->a_data);
	swaplong(&aout->a_bss);
	swaplong(&aout->a_syms);
	swaplong(&aout->a_entry);
	swaplong(&aout->a_trsize);
	swaplong(&aout->a_drsize);
}
#endif /* BSD */

#ifdef COFF
swapfhdr(f)
struct filehdr *f;
{
	swapshort(&f->f_magic);
	swapshort(&f->f_nscns);
	swaplong(&f->f_timdat);
	swaplong(&f->f_symptr);
	swaplong(&f->f_nsyms);
	swapshort(&f->f_opthdr);
	swapshort(&f->f_flags);
}

swapahdr(a)
struct aouthdr *a;
{
	swapshort(&a->magic);
	swapshort(&a->vstamp);
	swaplong(&a->tsize);
	swaplong(&a->dsize);
	swaplong(&a->bsize);
	swaplong(&a->entry_pt);
	swaplong(&a->text_start);
	swaplong(&a->data_start);
}

swapscn(s)
struct scnhdr *s;
{
	swaplong(&s->s_paddr);
	swaplong(&s->s_vaddr);
	swaplong(&s->s_size);
	swaplong(&s->s_scnptr);
	swaplong(&s->s_relptr);
	swaplong(&s->s_lnnoptr);
	swapshort(&s->s_nreloc);
	swapshort(&s->s_nlnno);
	swaplong(&s->s_flags);
}
#endif /* COFF */

swapphdr(phdr)
register PHDR *phdr;
{
	swaplong(&phdr->flags);
	swaplong(&phdr->address);
	swaplong(&phdr->length);
}
#endif /* ibm_pc */
