
/********************************************************
 * File:	ddc.c
 *
 * Description:
 *	The DDCMP module is a partial implementation
 *	of DEC'c DDCMP protocol.
 *	This file contains that modules main loop and set-up code.
 *
 * Routines:
 *	ddc
 *	ddc_trace
 *
 * Author:
 *	Jonathan Masel
 ********************************************************/

#include "gct.h"
#include "fatal.h"
#include "modules.h"
#include "msg.h"
#include "types.h"
#include "timer.h"
#include "status.h"
#include "mng.h"
#include "../driver/mtypes.h"
#include "bss.h"



extern HDR	*receive();
extern HDR	*grab();



/********************************************************
 *
 * Routine:	ddc
 *
 * Description:
 *	This is the main loop of the DDCMP module.
 *	Messages are "grabbed" from the module's
 *	input queue until the queue is exhausted.
 *	Then the module suspends until a message
 *	is received.
 *
 * Arguments:
 *
 * Return code:
 *
 * Side effects:
 *
 ********************************************************/

ddc()
{
	GCT *gct;
	register BSS		*bss;
	register DDCMP_LINK	*child;
	register HDR		*h;
	unsigned		i, j;
	int			(* *protocol)();
	extern int		ddc_txreq(), ddc_rxind(),
				ddc_timeout(), ddc_undef();
	extern int		ddc_undef();

	GETGCT(gct);
	bss = gct->ddcmp;

	bss->rom = 0;
	bss->max = 8;

	/*
	 * set up primitives table
	 */
	i = (unsigned)ddc_undef;
	for(j = 0; j < TABLE; j++)
		gct->ddcmp->protocol[j] = (int (*)())i;

	protocol = bss->protocol;
	i = (unsigned)ddc_txreq + bss->rom;
	protocol[SUB(TX_REQ)] = (int (*)())i;
	i = (unsigned)ddc_rxind + bss->rom;
	protocol[SUB(RX_IND)] = (int (*)())i;
	i = (unsigned)ddc_timeout + bss->rom;
	protocol[SUB(TM_EXP)] = (int (*)())i;

	for(i = 0; i < bss->max; i++){
		bss->child[i].key = i;
		bss->child[i].flags = 0;
		bss->child[i].smac = protocol;
	}

	/*
	 * Initialize timer table
	 */
	bss->timer_table = (TIMER_TABLE *)
			(bss->child + bss->max);
	if (i = service_timer(DDCMP, bss->max, bss->timer_table, 
		DDCMP_TICK, DDCMP_RSP))
		FATAL("DDCMP: cannot init timer table", 0);

	/*
	 * DDCMP main loop
	 */
	for (;;) {
		if (!(h = receive(DDCMP, 0))) {
			continue;
		}
		/*
		 * Until the queue is exhausted:
		 * Handle the received message, then
		 * "grab" next message from queue.
		 */
		if (h->type & PRMTV) {
			/*
			 * Primitive from upper layer or
		 	 * internally generated by the
			 * DDCMP module.
			 * Call the appropriate routine
			 * from the state table.
			 */
			if (h->id >= bss->max) {
				FATAL("DDCMP: bad id %x", h->id);
				relm(h);
				continue;
			}

			child = (DDCMP_LINK *)&(bss->child[h->id]);
			if( !(child->flags & INITIALIZED) ){
				FATAL("DDCMP: %d not initialized\n", h->id);
				continue;
			}
			(*child->smac[h->type & SUBTYPE])(h,child);
		} /* Primitive */
		else {
			child = (DDCMP_LINK *)&(bss->child[h->id]);
			i = ddc_req(h, child);
		} /* Non-primitive */

	} /* for */
} /* ddc */

ddc_undef(h, child)
HDR *h;
DDCMP_LINK *child;
{
}


/********************************************************
 * routine:	ddc_trace
 *
 * description:
 *	Trace a ddcmp primitive.
 *
 * arguments:
 *	h		points to the primtive message
 *
 * return code:
 *
 * side effects:
 *
 ********************************************************/
#define	DDCMP_MASK(ev)		(((unsigned short) 1) << \
				((ev) & SUBTYPE))

#define	min(i, j)		(i < j? i: j)


ddc_trace(h)
HDR *h;
{
	MSG *m;
	GCT *gct;
	EVENT_INFO *ev;
	DDCMP_LINK *child;
	T_FRAME *t;
	R_FRAME *r;
	LAYER_INFO *l;
	int i, j, k;
	extern MSG *getm();

	GETGCT(gct);
	child = (DDCMP_LINK *)&gct->ddcmp->child[h->id];
	if( !(child->trace_mask & DDCMP_MASK(h->type)) )
		return;
	m = getm(TRACE_EV, 0, 0, sizeof(EVENT_INFO)-EV_DATA_SIZE);
 	if (!m)
		return;

	ev = (EVENT_INFO *)m->param;
	ev->src = DDCMP;
	/* ev->dst = dst; */
	ev->id = h->id;
	ev->type = h->type;
	/* ev->status = h->status; */
	ev->time = gct->time;
	ev->par = (unsigned int)h;
	ev->data_length = 0;

	/*
	 * pass some data for R_FRAMEs and T_FRAMEs
	 */
	if( h->type == RX_IND ){
		r = (R_FRAME *)h;
		i = min(r->blen, EV_DATA_SIZE);
		m->len += i;
		ev->data_length = i;
		for(k = 0; k < i; k++)
			ev->data[k] = r->buf[k+r->boff];
	} else if( h->type == TX_REQ ){
		t = (T_FRAME *)h;
		l = &t->layer[LAYER(DDCMP)];
		i = t->hsize - l->hoff;
		j = t->bsize - l->boff;
		if( i > EV_DATA_SIZE )
			i -= EV_DATA_SIZE;
		if( (i+j) > EV_DATA_SIZE )
			j -= ((i+j)-EV_DATA_SIZE);
		m->len += i + j;
		ev->data_length = i + j;
		for(k = 0; k < i; k++)
			ev->data[k] = t->hbuf[l->hoff+k];
		for(k = 0; k < j; k++)
			ev->data[k+i] = t->buf[l->boff+k];
	}

	if( (*gct->send)(SYS_MNG, m) )
		relm(m);
}
