
/********************************************************
 * File:	ddc_tx.c
 *
 * Description:
 *	This file contains routine for handling
 *	transmission of all types of supported DDCMP messages.
 *
 * Routines:
 *	ddc_txreq
 *	ddc_header
 *	ddc_confirm
 *	ddc_timeout
 *	ddc_sendstrt
 *	ddc_sendstrtack
 *	ddc_sendack
 *	ddc_sendnak
 *	ddc_sendrep
 *	ddc_send
 *
 * Author:
 *	Jonathan Masel
 ********************************************************/


#include "gct.h"
#include "msg.h"
#include "modules.h"
#include "types.h"
#include "status.h"
#include "bss.h"
#include "ddc.h"
#include "../driver/imp.h"


/********************************************************
 * routine:	ddc_txreq
 *
 * description:
 *	This is the primitive called from an upper layer
 *	to request transmission of data.
 *
 * arguments:
 *	t		points to the T_FRAME received
 *	child		the child structure of the channel
 *
 * return code:
 *
 * side effects:
 *
 ********************************************************/
ddc_txreq(t, child)
register T_FRAME *t;
DDCMP_LINK *child;
{
	if( child->i_tail ){
		child->i_tail->layer[LAYER(DDCMP)].nextf = t;
		child->i_tail = t;
	} else {
		child->i_head = child->i_tail =
		child->i_conf = child->i_ack = t;
	}
	if( !child->i_head )
		child->i_head = t;

	ddc_tx(child);
}


/********************************************************
 * routine:	ddc_tx
 *
 * description:
 *	Flush the queue of T_FRAMES awaiting transmission.
 *
 * arguments:
 *	child		the child structure of the channel concerned
 *
 * return code:
 *	number of frames sent to driver for transmission
 *
 * side effects:
 *
 ********************************************************/
ddc_tx(child)
DDCMP_LINK *child;
{
	GCT *gct;
	register T_FRAME *t;
	register LAYER_INFO *l, *lower;
	register int i;
	register int count = 0;

	GETGCT(gct);
	t = child->i_head;

	while( child->outst < child->wsize ){
		if( !t )
			break;

		/* build the DDCMP header */
		i = ddc_header(t, SOH, child);
		if( i ){
			t->hdr.status = i;
			/* lower bit normally lowered by lower */
			t->hdr.rsp_req &= ~RSP(LAYER(child->lower));
			child->i_head = t->layer[LAYER(DDCMP)].nextf;
			t = child->i_head;
			continue;
		}

		/* prepare lower layer's info */
		lower = &t->layer[LAYER(child->lower)];
		lower->flags = 0;
		lower->boff = t->layer[LAYER(DDCMP)].boff;
		lower->hoff = t->layer[LAYER(DDCMP)].hoff;
		lower->nextf = 0;

		/* advance vs variable */
		child->vs++;

		/* advance queue pointers */
		child->i_head = t->layer[LAYER(DDCMP)].nextf;
		child->outst++;
		count++;

		/* send to lower layer */
		t->hdr.id = child->lower_id;
		t->hdr.rsp_req |= DDCMP_RSP;
		ddc_trace(t);
		(*gct->send)(child->lower, t);

		t = child->i_head;
	}
	if( count ){
		start_if_stopped(gct->ddcmp->timer_table,
			child->key, DDCMP_TIMER, child->Tack);
	}
	return(count);
}



/********************************************************
 * routine:	ddc_header
 *
 * description:
 *	Build a DDCMP header for a data or maintenance message.
 *
 * arguments:
 *	t		the T_FRAME for which the header is to be added
 *	type		the message type
 *	child		the channel's child structure
 *
 * return code:
 *	0		on success
 *	error code	otherwise (usually no memory available)
 *
 * side effects:
 *
 ********************************************************/
ddc_header(t, type, child)
register T_FRAME *t;
DDCMP_LINK *child;
{
	register LAYER_INFO *l;
	register DDCMP_DHDR *dhdr;
	int length = 0;
	int class, size;
	GCT *gct;

	l = &t->layer[LAYER(DDCMP)];

	/*
	 * DDCMP header must reside in a seperate buffer
	 * (so that its CRC can be sent by the IMP)
	 * If one has not been allocated for this T_FRAME
	 * do it now.
	 */
	if( !t->hbuf ){
		GETGCT(gct);
		class = TX | HEADER | child->lower_id;
		t->hbuf = (unsigned char *)(*gct->getmem)(class, &size);
		if( !t->hbuf )
			return(TX_NO_HDR);
		t->hdr.class |= HEADER;
		t->hsize = size;
		l->hoff = t->hsize;
	}
	if( !t->buf ){
		GETGCT(gct);
		class = TX | BUFFER | child->lower_id;
		t->buf = (unsigned char *)(*gct->getmem)(class, &size);
		if( !t->buf )
			return(TX_NO_HDR);
		t->hdr.class |= BUFFER;
		t->bsize = size;
		l->boff = t->bsize;
		/* copy header buffer to buffer */
		{ int i;
		for(i = t->hsize; i > l->hoff;){
			i--;
			l->boff--;
			t->buf[l->boff] = t->hbuf[i];
		}
		l->hoff = t->hsize;
		}
	}

	if( l->hoff < sizeof(DDCMP_DHDR) )
		return(TX_NO_HDR);

	length = t->hsize - l->hoff;
	if( t->buf )
		length += (t->bsize - l->boff);

	l->hoff -= sizeof(DDCMP_DHDR);
	dhdr = (DDCMP_DHDR *)(t->hbuf + l->hoff);
	dhdr->type = type;

	/*
	 * write the count in 14 bits (Intel style)
	 */
	dhdr->count1 = (length & 0xff);	/* 8 lower bits */
	dhdr->count2 = (length & 0x3f00) >> 8;	/* 6 upper bits */
	dhdr->flags = 0;
	dhdr->rep = child->vr;
	dhdr->num = child->vs;
	dhdr->addr = child->addr;
	return(0);
}


/********************************************************
 * routine:	ddc_timeout
 *
 * description:
 *	This is the primitive routine called on
 *	expiration of the acknowledge timer.
 *	Send an ENQ.
 *
 * arguments:
 *	h		points to the message received
 *	child		the child structure of the channel
 *
 * return code:
 *
 * side effects:
 *
 ********************************************************/
ddc_timeout(h, child)
HDR *h;
DDCMP_LINK *child;
{
	int cmd;
	GCT *gct;

	ddc_trace(h);
	relm(h);
	/*
	 * enter hunt mode and restart timer
	 */
	GETGCT(gct);
	cmd = ENTER_HUNT_MODE | (child->key << 1);
	issue_cmd(cmd);
	if( child->flags & TIMER_RECOVERY ){
		child->i_head = child->i_conf;
		ddc_tx(child);
	} else {
		child->flags |= TIMER_RECOVERY;
		ddc_sendrep(child);
	}
	start_timer(gct->ddcmp->timer_table,
		child->key, DDCMP_TIMER, child->Tack);
}


ddc_sendack(child)
DDCMP_LINK *child;
{
	child->vr++;
	ddc_send(ACK, child, 0, child->vr, 0);
}

ddc_sendnak(child)
DDCMP_LINK *child;
{
	ddc_send(NAK, child, 0, child->vr, 0);
}

ddc_sendrep(child)
DDCMP_LINK *child;
{
	ddc_send(REP, child, 0, 0, child->vs);
}

ddc_sendstrt(child)
DDCMP_LINK *child;
{
	ddc_send(STRT, child, 0, 0, 0);
}

ddc_sendstack(child)
DDCMP_LINK *child;
{
	ddc_send(STACK, child, 0, 0, 0);
}

/********************************************************
 * routine:	ddc_send
 *
 * description:
 *	Send a DDCMP control character sequence
 *
 * arguments:
 *	ctrl	the type of control message
 *	child	the channel's child structure
 *	subt	the NAK error subtype
 *	rep	the response number (for ACK/NAK msgs)
 *	num	the transmit number (for REP msgs)
 *
 * return code:
 *
 * side effects:
 *
 ********************************************************/
ddc_send(ctrl, child, subt, rep, num)
unsigned int ctrl;
DDCMP_LINK *child;
{
	register T_FRAME *t;
	extern T_FRAME *gett();
	GCT *gct;
	register LAYER_INFO *l;
	register DDCMP_CHDR *dhdr;
	int length;

	GETGCT(gct);
	t = gett(T_HDR, LAYER(DDCMP), child->key, 0);
	if( !t )
		return(0);

	l = &t->layer[LAYER(DDCMP)];
	if( l->hoff < sizeof(DDCMP_DHDR) )
		return(TX_NO_HDR);
	length = (int)(t->hsize - l->hoff);
	l->hoff -= sizeof(DDCMP_DHDR);
	dhdr = (DDCMP_CHDR *)(t->hbuf + l->hoff);
	dhdr->type = ENQ;
	dhdr->ctype = ctrl;
	dhdr->subtype = subt;
	dhdr->flags = 0;
	dhdr->rep = rep;
	dhdr->num = num;
	dhdr->addr = child->addr;

	/* prepare rest of lower layers info */
	l = &t->layer[LAYER(child->lower)];
	l->flags = 0;
	l->boff = t->layer[LAYER(DDCMP)].boff;
	l->hoff = t->layer[LAYER(DDCMP)].hoff;
	l->nextf = 0;

	/* send to lower layer */
	t->hdr.id = child->lower_id;
	ddc_trace(t);
	(*gct->send)(child->lower, t);
	return(1);
}


/********************************************************
 * routine:	ddc_confirm
 *
 * description:
 *	Confirm transmissions of all acknowledge messages.
 *
 * arguments:
 *	child		points to the channel's child structure
 *	nr		the N(R) variable in the received frame
 *
 * return code:
 *	The number of frames confirmed;
 *
 * side effects:
 *
 ********************************************************/
ddc_confirm(child, nr)
DDCMP_LINK *child;
int nr;
{
	register T_FRAME *t;
	register int count = 0;
	GCT *gct;

	GETGCT(gct);
	t = child->i_conf;
	if( !t )
		return(0);
	while( child->va != nr ){
		child->i_conf = t->layer[LAYER(DDCMP)].nextf;
		conf_frm(t, child->upper);
		if( child->outst > 0 )
			child->outst--;
		else
			break;
		t = child->i_conf;
		child->va++;
		if( !t )
			break;
	}

	/*
	 * if acknowledgements are still outstanding,
	 * restart the acknowledge timer.
	 * otherwise, stop it
	 */
	if( child->va != child->vs )
		start_timer(gct->ddcmp->timer_table,
			child->key, DDCMP_TIMER, child->Tack);
/*
this is also the rx-ers inactivity timer!!! don't stop!!
	else
		stop_timer(gct->ddcmp->timer_table, child->key);
*/

	if( !child->i_conf )
		child->i_tail = 0;

	ddc_tx(child);
}
