/********************************************************
 *
 * File:	bss.h
 *
 * Description:
 *	This file contains definitions for all the
 *	BSS resident data structures of the LAPB module.
 *
 * Author:
 *	Dorit Ordentlich
 *
 ********************************************************/

/*
 * This macro converts the relocated load-time address
 * into an address relative to the start address in
 * ROM of the LAPB.
 * This macro can be used instead of calling lapb_getadr()
 * in places where bss pointer exists.
 */
#define	LAPB_GETADR(adr)	(((unsigned long) (adr)) + bss->rom)


/*
 * module BSS (global) flags:
 */
#define	LAPB_INIT		0x8000	/* LAPB module initialized */
#define	PCHAN_BUSY(pchan)	(((unsigned short) 1) << (pchan))
				/* Local busy on physical channel */
				/* pchan */

/*
 * Child (per link) flags:
 * note that 0x8000 is reserved for loopback mode.
 */
#define	FLIP_MODE	0x4000	/* Flip addresses A & B on Tx	*/
#define	LINK_INIT	0x2000	/* Link initialized		*/
#define	DCE		0x1000	/* DCE (1) or DTE (0)		*/
#define	EXTENDED	0x0800	/* Extended (modulo 128) mode	*/
#define	L3_EST		0x0400	/* Layer 3 initiated establish	*/
#define	PEND_REL	0x0200	/* Pending release		*/
#define	REJ_REC		0x0100	/* REJ recovery			*/
#define	PEND_RNR	0x0080	/* Pending RNR			*/
#define	LCL_BUSY	0x0040	/* Local busy			*/
#define	RMT_BUSY	0x0020	/* Remote busy			*/
#define	PEND_ACK	0x0010	/* Pending acknowledgement 	*/
#define	T1_RUN		0x0008	/* T1 is running		*/
#define	PCHAN		0x000f	/* Physical channel number	*/
#define	MAX_PCHAN	12	/* Max # of physical channels */
#define	PCHAN_CLASS	0x0003	/* Memory class to use with a */
				/* physical channel: use 2 LSBits. */



/*
 * LAPB global work area structure.
 * This structure resides in the work area
 * in LAPB bss.
 */
typedef struct {
	struct t_frame	*tx_pool_head;	/* Tx pool head pointer	*/
	struct t_frame	*tx_pool_tail;	/* Tx pool head pointer	*/
	struct lapb_l	*active_head;	/* Head of active links queue */
	struct lapb_l	*active_tail;	/* Tail of active links queue */
	unsigned short	tick;		/* Module timers resolution  */

	/*
	 * Error counters
	 */
	unsigned long	too_short;	/* Frame too short	*/
	unsigned long	undef_ctl;	/* Undefined control	*/
	unsigned long	incorrect_len;	/* Incorrect length	*/
	unsigned long	invalid_adr;	/* Invalid address	*/
	unsigned short	ev_mask;	/* Module's event mask	*/
} LAPB_GWA;

/*
 * Link structure.
 * This structure consists of CHILD structure fields,
 * followed by LAPB specific fields which are located
 * in the CHILD's work area.
 */
typedef struct lapb_l {
	/*
	 * Fields common to all CHILD structures
	 */
	unsigned short	key;	/* Link ID		*/
	unsigned short	flags;
	unsigned char	upper;	/* Upper layer module	*/
	unsigned char	lower;	/* Lower layer module	*/
	unsigned char	layer;	/* Protocl layer	*/
	unsigned char	state;	/* Protocol state	*/
	int	(* *smac)();	/* State machine	*/

	/*
	 * LAPB specific fields (work area)
	 */
	unsigned short	pchan;	/* Physical channel ID */
	unsigned short	pad1;	/* Reserved			*/
	unsigned short	cadr;	/* Address for transmitted commands */
	unsigned short	radr;	/* Address for transmitted responses */
	unsigned short	T1;	/* Timer T1			*/
	unsigned short	N1;	/* Max bytes in I frame		*/
	unsigned char	N2;	/* Max number of retransmission	*/
	unsigned char	k;	/* Max outstanding I frames	*/
	unsigned short	pad2;	/* Reserved			*/
	unsigned char	vs;	/* Send state variable		*/
	unsigned char	va;	/* Acknowledge state variable	*/
	unsigned char	vr;	/* Receive state variable	*/
	unsigned char	rc;	/* Retry count			*/
	struct t_frame	*i_conf;	/* I queue conf pointer	*/
	struct t_frame	*i_ack;		/* I queue ack pointer	*/
	struct t_frame	*i_head;	/* I queue head pointer	*/
	struct t_frame	*i_tail;	/* I queue tail pointer	*/
	struct t_frame	*i_garbage;	/* I "throw away" queue	*/
	char		frmr[8];	/* Saved FRMR information */
	struct lapb_l	*next;		/* Next active link	*/
	unsigned long	pad3;		/* Reserved		*/

	/*
	 * Statistics counters
	 */
	unsigned long	tx_i;		/* Tx I count		*/
	unsigned long	tx_rr;		/* Tx RR count		*/
	unsigned long	tx_rnr;		/* Tx RNR count		*/
	unsigned long	tx_rej;		/* Tx REJ count		*/
	unsigned long	tx_frmr;	/* Tx FRMR count	*/
	unsigned long	rx_i;		/* Rx I count		*/
	unsigned long	rx_rr;		/* Rx RR count		*/
	unsigned long	rx_rnr;		/* Rx RNR count		*/
	unsigned long	rx_rej;		/* Rx REJ count		*/
	unsigned long	rx_frmr;	/* Rx FRMR count	*/
	unsigned long	ns_err;		/* N(S) errors count	*/
} LAPB_LINK;
