/********************************************************
 *
 * File:	protocol.h
 *
 * Description:
 *	LAPB definitions
 *
 * Note:
 *	Wherever possible the same definitions are
 *	used both for modulo-8 and for modulo-128
 *	operation. Where different definitions 
 *	are needed for modulo-8 operation, the macro's
 *	assume the existence of a boolean variable named
 *	'extended', and test this variable to determine
 *	the operational mode.
 *
 * Author:
 *	Dorit Ordentlich
 *
 ********************************************************/

#define	MOD8		8
#define	MOD128		128
#define	MODULUS		(extended ? MOD128 : MOD8)

/*
 * Frame header structure.
 * Note: it is assumed that the tx and rx data buffers
 * are at least 3 bytes long, so the ctl field may be
 * accessed as a word, even in frames with a 
 * single-octet control field.
 */
typedef struct {
	unsigned char	adr;
	unsigned char	ctl[1];	/* 2 bytes must be allocated	*/
				/* for S/I formats in extended	*/
				/* mode.			*/
} LAPB_HDR;

#define	ADR_LEN		1
#define	LONG_CTL	(extended  ? 2 : 1)

#define	SHORT_CTL	1
#define	LONG_HDR	( ADR_LEN + LONG_CTL )
#define	SHORT_HDR	( ADR_LEN + SHORT_CTL )

/*
 * The following macro is used to access the control field
 * as a word.
 */
#define	CTL_WRD(h)	(*(((unsigned short *) (h)->ctl)))


/*
 * Minimum frame length (not including FCS)
 * Note that S or I frames have a single-byte control field in
 * modulo-8 mode and two-bytes control field in extended
 * (modulo 128) mode.
 */
#define	MIN_U_LEN	2	/* U frames: 1 (adr) + 1 (ctl)	    */
#define	MIN_SI_LEN	(extended ? \
			3 	/* S or I frames: 1 (adr) + 2 (ctl) */ \
			: 2)	/* S or I frames: 1 (adr) + 1 (ctl) */
#define	FRMR_INFO_LEN	(extended ? 5 : 3)
#define	FRMR_LEN	(MIN_U_LEN + FRMR_INFO_LEN)
	
/*
 * Address field definitions and macros.
 */
#define	LAPB_A			0x03
#define	LAPB_B			0x01

#define	INVALID_ADR(adr)	(((adr) != LAPB_A) && ((adr) != LAPB_B))

/*
 * The FLIP bit (flips A to B and B to A):
 */
#define	FLIP_BIT		0x02

/*
 * The following macros determine whether a
 * received address field indicates a command or
 * a response. 
 */
#define	IS_CMD(adr, link)	((adr) == (link)->radr)
#define	IS_RSP(adr, link)	((adr) == (link)->cadr)


/*
 * Control field
 */
#define	I_FMT			0x0100
#define	SU_FMT			0x0300
#define	S_FMT			0x0100
#define	U_FMT			0x0300

/*
 * S format definitions
 */
#define	S_TYPE			(extended ? 0xff00 : 0x0f00)
#define	RR			0x0100
#define	RNR			0x0500
#define	REJ			0x0900

/*
 * S/I formats definitions and macros
 */
#define	SI_PF			(extended ? 0x0001 : 0x1000)

#define	NS			(extended ? 0xfe00 : 0x0e00)
#define	NS_SHIFT		9
#define	NS_STEP			(1 << NS_SHIFT)
#define RX_NS(ctl)		(((ctl) & NS) >> NS_SHIFT)
#define	TX_NS(vs)		((vs) << NS_SHIFT)

#define	NR			(extended ? 0x00fe : 0xe000)
#define	NR_SHIFT		(extended ? 1 : 13)
#define	RX_NR(ctl)		(((ctl)	& NR) >> NR_SHIFT)
#define	TX_NR(vr)		((vr) << NR_SHIFT)

#define	VALID_NR(nr, va, vs)	(((va) <= (vs)) ? \
				((((nr) < (va)) || ((nr) > (vs))) ? \
				0 : 1) : \
				((((nr) > (vs)) && ((nr) < (va))) ? \
				0 : 1))

/*
 * U format definitions and macros.
 * The following are for use when the control field
 * is accessed (read or write) as a short word.
 */
#define	U_TYPE	0xef00
#define	SABME	0x6f00
#define	SABM	0x2f00
#define	DM	0x0f00
#define	DISC	0x4300
#define	UA	0x6300
#define	FRMR	0x8700

#define	U_PF	0x1000

/*
 * The following are for use when the control field
 * is accessed (read or write) as a byte.
 */
#define	UA_B	(UA >> 8)

/*
 * FRMR information field structure:
 */
typedef struct {
	unsigned char	ctl;	/* Rejected frame control field	*/
	unsigned char	vr_vs;	/* Current V(R) and V(S) value	*/
	unsigned char	cause;	/* The cause of the FRMR condition */
} FRMR_INFO8; /* Modulo 8 */

typedef struct {
	unsigned short	ctl;	/* Rejected frame control field	*/
	unsigned char	vs;	/* Current V(S) value		*/
	unsigned char	vr;	/* Current V(R) value and CMD bit*/
	unsigned char	cause;	/* The cause of the FRMR condition */
} FRMR_INFO;

#define	FRMR_CMD	0x01	/* The rejected frame was a command */

/*
 * FRMR cause bit definitions:
 */
#define	FRMR_Z	0x08		/* N(R) error			*/
#define	FRMR_Y	0x04		/* N1 error			*/
#define	FRMR_X	0x02		/* Incorrect length		*/
#define	FRMR_W	0x01		/* Undefined control		*/

/*
 * Default parameter values
 */
#define	LAPB_DF_T1	TIMER_VAL(10)	/* One second	*/
#define	LAPB_DF_N2	3
#define	LAPB_DF_N1	260
#define	LAPB_DF_k	3



/*
 * The folowing bits are used to classify
 * received frames. They are written in 
 * the err_info field of the R_FRAME header
 * when the frame is analyzed, and tested
 * by the specfic routine that handles the 
 * frame.
 */
#define	PF		0x1001	/* Poll/Final for all frame types */
#define	CMD		0x8000	/* Command frame		*/

#define	UNDEF_CTL	0x4000	/* Undefined control field	*/
#define	N1_ERR		0x2000	/* N1 error			*/
#define	NS_ERR		0x0800	/* N(S) error			*/
#define	NR_ERR		0x0400	/* N(R) error			*/
#define	INCORRECT_LEN	0x0200	/* Incorrect length 		*/



/*
 * MDL_ERROR indication codes.
 * These codes are contained in the status field
 * of MDL_ERROR indication messages sent by the
 * LAPB module to layer 2 management module.
 * Note: the codes (A - O) are as defined for LAPD.
 */
#define	UNSOLICITED_S1	0x01	/* Unsolicited S, F=1     (code A) */
#define	UNSOLICITED_DM1	0x02	/* Unsolicited DM, F=1    (code B) */
#define	UNSOLICITED_UA1	0x03	/* Unsolicited UA, F=1    (code C) */
#define	UNSOLICITED_UA0	0x04	/* Unsolicited UA, F=0    (code D) */
#define	UNSOLICITED_DM0	0x05	/* Unsolicited DM, F=0    (code E) */
#define	PEER_RE_CONN	0x06	/* Peer initiated re-connect (code F) */
#define	SABME_RL	0x07	/* SABME retry limit 	  (code G) */
#define	DISC_RL		0x08	/* DISC retry limit 	  (code H) */
#define	STATUS_EN_RL	0x09	/* Status enq retry limit (code I) */
#define	NR_ERROR	0x0a	/* N(R) error 		  (code J) */
#define	RX_FRMR		0x0b	/* FRMR received 	  (code K) */
#define	RX_UNIMPL	0x0c	/* Unimlemented frame 	  (code L) */
#define	RX_I_NPERM	0x0d	/* I field not permitted  (code M) */
#define	RX_BAD_SIZE	0x0e	/* Frame with wrong size  (code N) */
#define	N1_ERROR	0x0f	/* N1 error 		  (code O) */

#define	NS_ERROR	0x10	/* N(S) error			*/
#define	CTL_NO_MEM	0x11	/* Cannot tx control: no memory	*/
#define	CTL_TX_FAIL	0x12	/* Cannot pass control to lower	*/
#define	I_TX_FAIL	0x13	/* Cannot pass I to lower	*/
#define	NO_CONF		0x14	/* No confirmation from lower	*/
#define	GARBAGE_IND	0x15	/* Frame(s) in garbage queue	*/
#define	I_RX_FAIL	0x16	/* Cannot pass I to upper	*/
#define	UNEXP_PF	0x17	/* Unexpected P/F bit in UI	*/
#define	RX_NO_HDR	0x18	/* Rx frame discarded (header	*/
				/* not contained in the first	*/
				/* buffer.			*/
#define	FRMR_RL		0x19	/* FRMR retry limit		*/
