/********************************************************
 *
 * File:	bss.h
 *
 * Description:
 *	This file contains definitions for all the
 *	BSS resident data structures of the LAPD module.
 *
 * Author:
 *	Dorit Ordentlich
 *
 ********************************************************/

/*
 * This macro converts the relocated load-time address
 * into an address relative to the start address in
 * ROM of the LAPD.
 * This macro can be used instead of calling lapd_getadr()
 * in places where bss pointer exists.
 */
#define	LAPD_GETADR(adr)	(((unsigned long) (adr)) + bss->rom)


/*
 * module BSS (global) flags:
 */
#define	LAPD_INIT		0x8000	/* LAPD module initialized */
#define	PCHAN_BUSY(pchan)	(((unsigned short) 1) << (pchan))
				/* Local busy on physical channel */
				/* pchan */

/*
 * Child (per link) flags:
 * note that 0x8000 is reserved for loopback mode.
 */
#define	FLIP_MODE	0x4000	/* Flip TEI's LSbit on transmit	*/
#define	LINK_INIT	0x2000	/* Link initialized		*/
#define	MNG_LINK	0x1000	/* A management link		*/
#define	B_LINK		0x0800	/* A broadcast link		*/
#define	L3_EST		0x0400	/* Layer 3 initiated establish	*/
#define	PEND_REL	0x0200	/* Pending release		*/
#define	REJ_REC		0x0100	/* REJ recovery			*/
#define	PEND_RNR	0x0080	/* Pending RNR			*/
#define	LCL_BUSY	0x0040	/* Local busy			*/
#define	RMT_BUSY	0x0020	/* Remote busy			*/
#define	PEND_ACK	0x0010	/* Pending acknowledgement 	*/
#define	T200_RUN	0x0008	/* T200 is running		*/
#define	T203_RUN	0x0004	/* T203 is running		*/
#define	PCHAN		0x000f	/* Physical channel number	*/
#define	MAX_PCHAN	12	/* Max # of physical channels */
#define	PCHAN_CLASS	0x0003	/* Memory class to use with a */
				/* physical channel: use 2 LSBits. */
#define	MAX_FAST_PCHAN	3	/* Max # of fast physical */
				/* channels (for which a TEI */
				/* table is used */
#define	IS_FAST(pchan)	((pchan) < MAX_FAST_PCHAN)



/*
 * LAPD global work area structure.
 * This structure resides in the work area
 * in LAPD bss.
 */
typedef struct {
	struct t_frame	*tx_pool_head;	/* Tx pool head pointer	*/
	struct t_frame	*tx_pool_tail;	/* Tx pool head pointer	*/
	struct lapd_l	*active_head;	/* Head of active links queue */
	struct lapd_l	*active_tail;	/* Tail of active links queue */
	unsigned short	tick;		/* Module timers resolution  */

	/*
	 * Error counters
	 */
	unsigned long	too_short;	/* Frame too short	*/
	unsigned long	undef_ctl;	/* Undefined control	*/
	unsigned long	incorrect_len;	/* Incorrect length	*/
	unsigned long	invalid_adr;	/* Invalid address	*/
	unsigned long	inactive_dlci;	/* Inactive DLCI	*/

	/*
	 * LAPD TEI's tables:
	 * For each of physical channels 0, 1 and 2,
	 * a TEI's table is maintained:
	 * These tables contain a 4-byte entry for each
	 * possible TEI value (0x00 to 0x7f) for
	 * the corresponding physical channel.
	 * Each entry is a pointer to a linked list of
	 * LAPD links, which includes all links assigned with
	 * the corresponding TEI on the corresponding physical channel.
	 * the linked list is formed using the next_sapi
	 * pointer in the LAPD_LINK structure.
	 * For example, tei_tab[1][0x7f] is the entry for broadcast
	 * TEI on physical channel # 1.
	 */
	struct lapd_l	**tei_tab[MAX_FAST_PCHAN];
	unsigned short	ev_mask;	/* Module's event mask	*/
} LAPD_GWA;

/*
 * Link structure.
 * This structure consists of CHILD structure fields,
 * followed by LAPD specific fields which are located
 * in the CHILD's work area.
 */
typedef struct lapd_l {
	/*
	 * Fields common to all CHILD structures
	 */
	unsigned short	key;	/* Link ID		*/
	unsigned short	flags;
	unsigned char	upper;	/* Upper layer module	*/
	unsigned char	lower;	/* Lower layer module	*/
	unsigned char	layer;	/* Protocl layer	*/
	unsigned char	state;	/* Protocol state	*/
	int	(* *smac)();	/* State machine	*/

	/*
	 * LAPD specific fields (work area)
	 */
	unsigned short	pchan;	/* Physical channel ID */
	unsigned short	dlci;	/* Link address			*/
	unsigned short	cadr;	/* Address for transmitted commands */
	unsigned short	radr;	/* Address for transmitted responses */
	unsigned short	T200;	/* Timer T200			*/
	unsigned short	N201;	/* Max bytes in I field		*/
	unsigned char	N200;	/* Max number of retransmission	*/
	unsigned char	k;	/* Max outstanding I frames	*/
	unsigned short	T203;	/* Timer T203			*/
	unsigned char	vs;	/* Send state variable		*/
	unsigned char	va;	/* Acknowledge state variable	*/
	unsigned char	vr;	/* Receive state variable	*/
	unsigned char	rc;	/* Retry count			*/
	struct t_frame	*i_conf;	/* I queue conf pointer	*/
	struct t_frame	*i_ack;		/* I queue ack pointer	*/
	struct t_frame	*i_head;	/* I queue head pointer	*/
	struct t_frame	*i_tail;	/* I queue tail pointer	*/
	struct t_frame	*i_garbage;	/* I "throw away" queue	*/
	struct t_frame	*ui_head;	/* UI queue head pointer*/
	struct t_frame	*ui_tail;	/* UI queue head pointer*/
	struct lapd_l	*next;		/* Next active link	*/
	struct lapd_l	*next_sapi;	/* Next link with same TEI */

	/*
	 * Statistics counters
	 */
	unsigned long	tx_i;		/* Tx I count		*/
	unsigned long	tx_rr;		/* Tx RR count		*/
	unsigned long	tx_rnr;		/* Tx RNR count		*/
	unsigned long	tx_rej;		/* Tx REJ count		*/
	unsigned long	tx_frmr;	/* Tx FRMR count	*/
	unsigned long	rx_i;		/* Rx I count		*/
	unsigned long	rx_rr;		/* Rx RR count		*/
	unsigned long	rx_rnr;		/* Rx RNR count		*/
	unsigned long	rx_rej;		/* Rx REJ count		*/
	unsigned long	rx_frmr;	/* Rx FRMR count	*/
	unsigned long	ns_err;		/* N(S) errors count	*/
	unsigned long	tx_ui;		/* Tx UI count		*/
	unsigned long	rx_ui;		/* Rx UI count		*/
} LAPD_LINK;
