/********************************************************
 *
 * File:	protocol.h
 *
 * Description:
 *	Q.921 definitions
 *
 * Author:
 *	Dorit Ordentlich
 *
 ********************************************************/

#define	MODULUS		128

/*
 * Frame header structure.
 * Note: it is assumed that the tx and rx data buffers
 * are at least 4 bytes long, so the ctl field may be
 * accessed as a word, even in frames with a 
 * single-octet control field.
 */
typedef struct {
	unsigned short	adr;
	unsigned char	ctl[2];
} LAPD_HDR;

#define	ADR_LEN		2
#define	LONG_CTL	2

#define	SHORT_CTL	1
#define	LONG_HDR	( ADR_LEN + LONG_CTL )
#define	SHORT_HDR	( ADR_LEN + SHORT_CTL )

/*
 * The following macro is used to access the control field
 * as a word.
 */
#define	CTL_WRD(h)	(*(((unsigned short *) (h)->ctl)))

/*
 * Minimum frame length (not including FCS)
 */
#define	MIN_U_LEN	3	/* U frames: 2 (adr) + 1 (ctl)	    */
#define	MIN_SI_LEN	4	/* S or I frames: 2 (adr) + 2 (ctl) */
#define	FRMR_INFO_LEN	5
#define	FRMR_LEN	(MIN_U_LEN + FRMR_INFO_LEN)
	
/*
 * Address field definitions and macros.
 */
#define	CR			0x0200
#define	EA0			0x0100
#define	EA1			0x0001

#define	EA_MASK			(EA0 | EA1)
#define	EA_VAL			(EA1)
#define	INVALID_ADR(adr)	(((adr) & EA_MASK) != EA_VAL)

#define	SAPI_MASK		0xfc00
#define	SAPI_SHIFT		10
#define	TEI_MASK		0x00fe
#define	TEI_SHIFT		1
#define	DLCI_MASK		(SAPI_MASK | TEI_MASK)

/*
 * The FLIP bit:
 */
#define	FLIP_BIT		0x0002

/*
 * The following macros determine whether a
 * received address field indicates a command or
 * a response. 
 */
#define	IS_CMD(adr, link)	((adr) == (link)->radr)
#define	IS_RSP(adr, link)	((adr) == (link)->cadr)

/*
 * The following macros generate the address
 * field for transmitted commmands or responses.
 * Parameters:
 *	adr		DLCI
 *	user		0x0200 for user side,
 *			0x0000 for network side.
 */
#define	LAPD_CMD_ADR(adr, user)	(((adr) | (~(user) & CR)) | EA1)
#define	LAPD_RSP_ADR(adr, user)	(((adr) | (user)) | EA1)


/*
 * The following macros derive the SAPI
 * and TEI values from a given DLCI.
 */
#define	SAPI(dlci)		(((dlci) & SAPI_MASK) >> SAPI_SHIFT)
#define	TEI(dlci)		(((dlci) & TEI_MASK) >> TEI_SHIFT)

/*
 * The following macro creates a DLCI
 * from a given SAPI and TEI.
 */
#define	DLCI(sapi, tei)		(((sapi) << 10) | ((tei) << 1))

/*
 * Control field
 */
#define	I_FMT			0x0100
#define	SU_FMT			0x0300
#define	S_FMT			0x0100
#define	U_FMT			0x0300

/*
 * S format definitions
 */
#define	S_TYPE			0xff00
#define	RR			0x0100
#define	RNR			0x0500
#define	REJ			0x0900

/*
 * S/I formats definitions and macros
 */
#define	SI_PF			0x0001

#define	NS			0xfe00
#define	NS_SHIFT		9
#define	NS_STEP			(1 << NS_SHIFT)
#define RX_NS(ctl)		(((ctl) & NS) >> NS_SHIFT)
#define	TX_NS(vs)		((vs) << NS_SHIFT)

#define	NR			0x00fe
#define	NR_SHIFT		1
#define	RX_NR(ctl)		(((ctl)	& NR) >> NR_SHIFT)
#define	TX_NR(vr)		((vr) << NR_SHIFT)

#define	VALID_NR(nr, va, vs)	(((va) <= (vs)) ? \
				((((nr) < (va)) || ((nr) > (vs))) ? \
				0 : 1) : \
				((((nr) > (vs)) && ((nr) < (va))) ? \
				0 : 1))

/*
 * U format definitions and macros.
 * The following are for use when the control field
 * is accessed (read or write) as a short word.
 */
#define	U_TYPE	0xef00
#define	SABME	0x6f00
#define	DM	0x0f00
#define	UI	0x0300
#define	DISC	0x4300
#define	UA	0x6300
#define	FRMR	0x8700
#define	XID	0xaf00

#define	U_PF	0x1000

/*
 * The following are for use when the control field
 * is accessed (read or write) as a byte.
 */
#define	UA_B	(UA >> 8)
#define	UI_B	(UI >> 8)
#define	XID_B	(XID >> 8)

/*
 * FRMR information field structure:
 */
typedef struct {
	unsigned short	ctl;	/* Rejected frame control field	*/
	unsigned char	vs;	/* Current V(S) value		*/
	unsigned char	vr;	/* Current V(R) value and CMD bit*/
	unsigned char	cause;	/* The cause of the FRMR condition */
} FRMR_INFO;

#define	FRMR_CMD	0x01	/* The rejected frame was a command */

/*
 * FRMR cause bit definitions:
 */
#define	FRMR_Z	0x08		/* N(R) error			*/
#define	FRMR_Y	0x04		/* N201 error			*/
#define	FRMR_X	0x02		/* Incorrect length		*/
#define	FRMR_W	0x01		/* Undefined control		*/

/*
 * Default parameter values
 */
#define	LAPD_DF_T200	TIMER_VAL(10)	/* One second	*/
#define	LAPD_DF_T203	TIMER_VAL(100)	/* Ten seconds	*/
#define	LAPD_DF_N200	3
#define	LAPD_DF_N201	260
#define	LAPD_DF_k	3



/*
 * The folowing bits are used to classify
 * received frames. They are written in 
 * the err_info field of the R_FRAME header
 * when the frame is analyzed, and tested
 * by the specfic routine that handles the 
 * frame.
 */
#define	PF		0x1001	/* Poll/Final for all frame types */
#define	CMD		0x8000	/* Command frame		*/

#define	UNDEF_CTL	0x4000	/* Undefined control field	*/
#define	N201_ERR	0x2000	/* N201 error			*/
#define	NS_ERR		0x0800	/* N(S) error			*/
#define	NR_ERR		0x0400	/* N(R) error			*/
#define	INCORRECT_LEN	0x0200	/* Incorrect length 		*/



/*
 * MDL_ERROR indication codes.
 * These codes are contained in the status field
 * of MDL_ERROR indication messages sent by the
 * LAPD module to layer 2 management module.
 */
#define	UNSOLICITED_S1	0x01	/* Unsolicited S, F=1     (code A) */
#define	UNSOLICITED_DM1	0x02	/* Unsolicited DM, F=1    (code B) */
#define	UNSOLICITED_UA1	0x03	/* Unsolicited UA, F=1    (code C) */
#define	UNSOLICITED_UA0	0x04	/* Unsolicited UA, F=0    (code D) */
#define	UNSOLICITED_DM0	0x05	/* Unsolicited DM, F=0    (code E) */
#define	PEER_RE_EST	0x06	/* Peer initiated re-est  (code F) */
#define	SABME_RL	0x07	/* SABME retry limit 	  (code G) */
#define	DISC_RL		0x08	/* DISC retry limit 	  (code H) */
#define	STATUS_EN_RL	0x09	/* Status enq retry limit (code I) */
#define	NR_ERROR	0x0a	/* N(R) error 		  (code J) */
#define	RX_FRMR		0x0b	/* FRMR received 	  (code K) */
#define	RX_UNIMPL	0x0c	/* Unimlemented frame 	  (code L) */
#define	RX_I_NPERM	0x0d	/* I field not permitted  (code M) */
#define	RX_BAD_SIZE	0x0e	/* Frame with wrong size  (code N) */
#define	N201_ERROR	0x0f	/* N201 error 		  (code O) */

/*
 * Extended MDL_ERROR indication 
 * (non standard) codes.
 * These codes are contained in the status field
 * of EXT_MDL_ERR_IND indication messages sent by the
 * LAPD module to layer 2 management module.
 */
#define	NS_ERROR	0x10	/* N(S) error			*/
#define	CTL_NO_MEM	0x11	/* Cannot tx control: no memory	*/
#define	CTL_TX_FAIL	0x12	/* Cannot pass control to lower	*/
#define	I_TX_FAIL	0x13	/* Cannot pass I to lower	*/
#define	NO_CONF		0x14	/* No confirmation from lower	*/
#define	GARBAGE_IND	0x15	/* Frame(s) in garbage queue	*/
#define	I_RX_FAIL	0x16	/* Cannot pass I to upper	*/
#define	UNEXP_PF	0x17	/* Unexpected P/F bit in UI/XID	*/
#define	RX_NO_HDR	0x18	/* Rx frame discarded (header	*/
				/* not contained in the first	*/
				/* buffer.			*/
#define	UI_TX_FAIL	0x19	/* Cannot pass UI to lower	*/
#define	UI_RX_FAIL	0x1a	/* Cannot pass UI to upper/mng	*/
#define	XID_TX_FAIL	0x1b	/* Cannot pass XID to lower	*/
#define	XID_RX_FAIL	0x1c	/* Cannot pass XID to mng	*/
