{  DAQ3000:  SCSI Data Acquisition Subsystem for the Apple Macintosh  			}
{																							}
{  SCSIMover:  Program for digital sample acquisition and reproduction.			}
{																							}
{																							}
{  (C) John Eng    January 1989															}


PROGRAM SCSIMover;

	USES
		SCSIManager;

	CONST
		targetID = 4;			{  SCSI target ID number  }
		timeout = 120;		{  Protocol timeout (2 sec)  }
		memresv = 50000;	{  Amount of memory reserved outside sample area  }

	VAR
		cdb : PACKED ARRAY[0..11] OF Byte;			{  Command Data Block for Toolbox SCSI routines  }
		buffer : Ptr;										{  Main digitized sound buffer  }
		stat, message : integer;							{  Status and message values returned by the Toolbox  }
		npoints : longint;									{  Number of data points (12 bit values)  }
		nbytes : longint;									{  Number of bytes currently in data buffer  }
		maxpoints : longint;								{  Maximum number of data points for available memory  }
		maxbytes : longint;								{  Size of data buffer  }
		n : longint;										{  User input variable  }
		answer : STRING;								{  User input variable  }
		ch : char;											{  User input variable  }
		frame : Rect;										{  Represents screen window  }
		err : OSErr;										{  Toolbox system error code  }
		errword : integer;								{  SCSI status word returned by SCSIFast routines  }



{  External Function Declarations														}
{																							}
{  SCSIRFast():	p = Pointer to buffer for receiving digitized data					}
{					nbytes = Number of bytes to be received							}
{					laststat = Pointer to storage for the last SCSI status word		}
{																							}
{  SCSIWFast():	p = Pointer to buffer of data to be transmitted					}
{					nbytes = Number of bytes to be transmitted						}
{					laststat = Pointer to storage for the last SCSI status word		}


	FUNCTION SCSIRFast (p : Ptr;
									nbytes : longint;
									laststat : Ptr) : OSErr;
	external;


	FUNCTION SCSIWFast (p : Ptr;
									nbytes : longint;
									laststat : Ptr) : OSErr;
	external;



{  Utility Routines																		}
{																							}
{  userprompt()	Pauses program and prompts user to continue						}
{  errmsg()		Handles error messages												}
{  hex2dec	()		Converts a hexadecimal string into an integer value				}
{  initmem()		Sets up the memory buffer and the screen window				}
{  showmenu()	Displays the main command menu									}


	PROCEDURE userprompt;
		VAR
			answer : STRING;
	BEGIN
		write('Press <Return> to continue --> ');
		readln(answer);
	END;


	PROCEDURE errmsg (n : integer);
		VAR
			s : STRING;
	BEGIN
		writeln('Error ', integer(err) : 0, ' at point ', n : 0, '.');
		writeln('SCSI status word according to SCSIFast:  ', longint(errword) : 0);
		writeln('SCSI status word according to SCSI Manager:  ', longint(SCSIStat) : 0);
		writeln('Suggest resetting the DAQ3000 and performing a SCSI bus reset.');
		writeln;
		userprompt;
	END;


	FUNCTION hex2dec (x : STRING) : integer;
		CONST
			hexdig = '0123456789ABCDEF';
		VAR
			i : integer;
	BEGIN
		i := 0;
		WHILE length(x) > 0 DO
			BEGIN
				i := 16 * i;
				i := i + (pos(copy(x, 1, 1), hexdig) - 1);
				delete(x, 1, 1);
			END;
		hex2dec := i;
	END;


	PROCEDURE initmem;
		VAR
			grow : Size;
	BEGIN
		maxbytes := MaxMem(grow) - memresv;
		buffer := NewPtr(maxbytes);
		IF buffer = NIL THEN
			BEGIN
				writeln('WARNING!  Unable to allocate a data buffer.');
				userprompt;
			END;
		maxpoints := maxbytes DIV 2;
		npoints := 0;
		nbytes := 0;
		SetRect(frame, 10, 40, 480, 330);
		SetTextRect(frame);
		ShowText;
	END;


	PROCEDURE showmenu;
	BEGIN
		writeln(chr(12), 'SCSIMover for the DAQ3000  (John Eng)');
		writeln;
		writeln;
		writeln('Commands:');
		writeln;
		writeln('1  Receive SCSI Data from DAQ3000');
		writeln('2  Send SCSI Data to DAQ3000');
		writeln('3  Download 6502 Program');
		writeln('4  Reset SCSI Bus');
		writeln('5  Load SCSI Data from Disk');
		writeln('6  Save SCSI Data to Disk');
		writeln('7  Memory Information');
		writeln('0  Quit');
		writeln;
	END;



{  Main SCSI Data Transfer Routine														}
{																							}
{  SCSITransfer():	trfunc = 1 for SCSI read, 2 for SCSI write					}
{						p = Pointer to data buffer										}
{						nbytes = Number of bytes in the buffer							}
{																							}
{  SCSITransfer() sets up the SCSI data transfer with calls to the SCSI Manager	}
{  routines of the Macintosh Toolbox.  The actual data transfer is performed by	}
{  the external assembly language routines SCSIRFast() and SCSIWFast().  Error	}
{  messages are generated for error occurrences along the way.					}


	PROCEDURE SCSITransfer (trfunc : integer;
									p : Ptr;
									nbytes : longint);
	BEGIN
		err := SCSIGet;
		IF err <> noErr THEN
			errmsg(1)
		ELSE
			BEGIN
				err := SCSISelect(targetID);
				IF err <> noErr THEN
					errmsg(2)
				ELSE
					BEGIN
						err := SCSICmd(@cdb, 6);
						IF err <> noErr THEN
							errmsg(3)
						ELSE
							BEGIN
								CASE trfunc OF
									1 : 
										err := SCSIRFast(p, nbytes, @errword);
									2 : 
										err := SCSIWFast(p, nbytes, @errword);
									OTHERWISE
										;
								END;
								IF err <> noErr THEN
									errmsg(4)
								ELSE
									BEGIN
										err := SCSIComplete(stat, message, timeout);
										IF (err <> noErr) OR (stat <> 0) OR (message <> 0) THEN
											BEGIN
												writeln('Status:', stat);
												writeln('Message:', message);
												errmsg(5);
											END;
									END;
							END;
					END;
			END;
	END;



{  Main Command Routines																}
{																							}
{  receive()		Causes data to be received in a buffer by setting up parameters	}
{					for a call to SCSITransfer.											}
{																							}
{  send()			Causes data to be sent from a buffer by setting up parameters	}
{					for a call to SCSITransfer.											}
{																							}
{  download()		Downloads machine language code to the DAQ3000.  The code	}
{					is read from a text file which contains a list of hexadecimal		}
{					values.																}
{																							}
{  load()			Loads previously saved sample data from a disk file.  The		}
{					main memory buffer is altered.  Toolbox routines are			}
{					used for file I/O because they are much faster than standard	}
{					Pascal I/O procedures.  A Toolbox routine is also used to get	}
{					the filename from the user.											}
{																							}
{  save()			Saves sample data to a disk file.  The main memory buffer is	}
{					unaltered.  As in load(), Toolbox routines are used for file I/O	}
{					and to get the desired filename.										}
{																							}
{  showmemory()	Display the current sample size,  the total sample buffer size,	}
{					and the total system heap memory remaining.						}


	PROCEDURE receive (p : Ptr;
									nbytes : longint);
		VAR
			npoints : longint;
	BEGIN
		npoints := nbytes DIV 2;
		cdb[0] := 8;
		cdb[1] := npoints DIV 65536;
		cdb[2] := (npoints MOD 65536) DIV 256;
		cdb[3] := (npoints MOD 65536) MOD 256;
		cdb[4] := 0;
		cdb[5] := 0;
		SCSITransfer(1, p, nbytes);
	END;


	PROCEDURE send (p : Ptr;
									nbytes : longint);
		VAR
			npoints : longint;
	BEGIN
		npoints := nbytes DIV 2;
		cdb[0] := 10;
		cdb[1] := npoints DIV 65536;
		cdb[2] := (npoints MOD 65536) DIV 256;
		cdb[3] := (npoints MOD 65536) MOD 256;
		cdb[4] := 0;
		cdb[5] := 0;
		SCSITransfer(2, p, nbytes);
	END;


	PROCEDURE download;
		VAR
			where : Point;
			dummy : SFTypeList;
			reply : SFReply;
			pbuff : PACKED ARRAY[0..2047] OF Byte;
			i, nbytes : integer;
			infile : text;
			s : STRING;
	BEGIN
		where.v := 50;
		where.h := 50;
		SFGetFile(where, 'File to download:', NIL, -1, dummy, NIL, reply);
		IF reply.good THEN
			BEGIN
				nbytes := 0;
				FOR i := 0 TO 2047 DO
					pbuff[i] := 0;
				reset(infile, reply.fName);
				WHILE NOT (eof(infile)) DO
					BEGIN
						readln(infile, s);
						pbuff[nbytes] := hex2dec(s);
						nbytes := nbytes + 1;
					END;
				close(infile);
				writeln('Program length:  ', nbytes : 0, ' bytes.');
				nbytes := 256 * ((nbytes + 255) DIV 256);
				cdb[0] := 10;
				cdb[1] := 0;
				cdb[2] := 0;
				cdb[3] := 0;
				cdb[4] := nbytes DIV 256;
				cdb[5] := 0;
				SCSITransfer(2, @pbuff, nbytes);
				userprompt;
			END;
	END;


	PROCEDURE load (p : Ptr;
									VAR nbytes : longint);
		VAR
			where : Point;
			dummy : SFTypeList;
			reply : SFReply;
			err : OSErr;
			refNum : integer;
			n : longint;
	BEGIN
		where.v := 50;
		where.h := 50;
		SFGetFile(where, 'Load sample from:', NIL, -1, dummy, NIL, reply);
		IF reply.good THEN
			BEGIN
				err := FSOpen(reply.fName, reply.vRefNum, refNum);
				IF err <> noErr THEN
					BEGIN
						writeln('Unable to open the file.');
						userprompt;
					END
				ELSE
					BEGIN
						err := GetEOF(refNum, n);
						IF n <= maxbytes THEN
							BEGIN
								nbytes := n;
								err := FSRead(refNum, nbytes, p);
							END
						ELSE
							BEGIN
								writeln('Not enough memory in data buffer.');
								userprompt;
							END;
						err := FSClose(refNum);
						err := FlushVol(NIL, reply.vRefNum);
					END;
			END;
	END;


	PROCEDURE save (p : Ptr;
									nbytes : longint);
		VAR
			where : Point;
			reply : SFReply;
			err : OSErr;
			refNum : integer;
	BEGIN
		where.v := 50;
		where.h := 50;
		SFPutFile(where, 'Save sample as:', '', NIL, reply);
		IF reply.good THEN
			BEGIN
				err := FSDelete(reply.fName, reply.vRefNum);
				err := Create(reply.fName, reply.vRefNum, '????', '????');
				IF err <> noErr THEN
					BEGIN
						writeln('Unable to create the file.');
						userprompt;
					END
				ELSE
					BEGIN
						err := FSOpen(reply.fName, reply.vRefNum, refNum);
						IF err <> noErr THEN
							BEGIN
								writeln('Unable to open the file.');
								userprompt;
							END
						ELSE
							err := FSWrite(refNum, nbytes, p);
						err := FSClose(refNum);
						err := FlushVol(NIL, reply.vRefNum);
					END;
			END;
	END;


	PROCEDURE showmemory;
		VAR
			n, grow : Size;
	BEGIN
		n := MaxMem(grow);
		writeln('Size of current sample     = ', nbytes : 0);
		writeln('Size of sample data buffer = ', maxbytes : 0);
		writeln;
		writeln('Bytes in heap zone = ', FreeMem : 0);
		writeln('Bytes contiguous   = ', n : 0);
		writeln;
		userprompt;
	END;



{  Main Program																			}
{																							}
{  The data buffer is initialized and maintained from here.  A simple command		}
{  loop gets input from the user and calls the appropriate command routine.		}


BEGIN
	MaxApplZone;
	initmem;
	err := SCSIReset;
	REPEAT
		showmenu;
		write('Selection?  ');
		readln(answer);
		writeln;
		writeln;
		writeln;
		ch := answer[1];
		CASE ch OF
			'1' : 
				BEGIN
					write('Receive how many 12-bit data values?  ');
					readln(n);
					IF n <= maxpoints THEN
						BEGIN
							npoints := n;
							nbytes := npoints * 2;
							receive(buffer, nbytes);
						END
					ELSE
						BEGIN
							writeln('Not enough memory in data buffer.');
							userprompt;
						END;
				END;
			'2' : 
				send(buffer, nbytes);
			'3' : 
				download;
			'4' : 
				err := SCSIReset;
			'5' : 
				BEGIN
					load(buffer, nbytes);
					npoints := nbytes DIV 2;
				END;
			'6' : 
				save(buffer, nbytes);
			'7' : 
				showmemory;
			OTHERWISE
				BEGIN
				END;
		END;
	UNTIL (ch = '0');
	DisposPtr(buffer);
END.