/********************************************************
 *
 * File:        driver.c
 *
 * Description:
 *      This is a task that interfaces with the driver.
 *      It illustrates how messages are sent and received
 *      to and from the driver module.
 *      The data is transmitted as a telegram (the user-interface
 *      module treats it as a telegram, since the first byte
 *      of data is 0xaa).
 *      The example task is incorporated into the system by
 *      replacing LAPB. Hence, it assumes the module ID of
 *      LAPB.
 *
 * Environment
 *      You should run this session from a directory containing
 *      the example g-files as distributed.
 *      You must first make sure that the g-files are edited
 *      as follows:
 *
 *      imp.g
 *          SCC-1 should be set to internal loopback mode
 *          by setting its configuration register to 007e
 *          and its mode register to 001c.
 *      driver.g
 *          The upper module for SCC-1 should be set to 01 (LAPB).
 *      alloc.g
 *          The last line (board's highest address) should
 *          be set to 7a000
 *      modules.g
 *          add the line "lapb   7a000" to this file
 *
 * Compiler
 *      All examples in this manual were compiled and tested
 *      with a compiler that allocates 32-bit values for integer
 *      and pointer variables. Most ANSI checking is done, but
 *      not all, so warnings may be generated if you use a fully
 *      ANSI-compliant C compiler.
 *
 ********************************************************/


#include "include/gct.h"     /* defines global config table structure */
#include "include/msg.h"     /* defines MSG, T_FRAME, and R_FRAME */
#include "include/modules.h" /* defines module ids */
#include "include/types.h"   /* defines "type" field in HDR structure */
#include "include/timer.h"   /* defines timer table structure */

extern T_FRAME *gett();
struct t_table t_table;

my_task()
{
    GCT *gct;    /* pointer to the global config table defined
                    in gct.h */
    HDR *h;      /* pointer to the header structure of messages
                    regardless of the type: MSG, T_FRAME or
                    R_FRAME */

   /*
    * macro that returns the value of the
    * base of the global config table in gct
    */
    GETGCT(gct);

   /*
    * The LAPB module requests 1 timer table entry
    * at location &t_table.  The timer table will
    * be scanned every 1 tick.  This defines the
    * resolution of the timer table -- the period
    * upon which the table is scanned for a
    * timeout, and a message sent.
    * No (0) response is required to the keep_time request.
    */
    service_timer(LAPB, 1, &t_table, 1, 0);
    for(;;){     /* do forever */
        h = (HDR *)(*gct->receive)();   /* wait for a message
                                           from the receive queue */
        if( !h ) continue;              /* leave loop if error */

    /*
     * if the message is a RESET,
     * start a timer 0 (the only timer) of type 0 and a
     * timeout of every 10 times the timer table is scanned.
     * Confirm the RESET message (release memory it used).
     */
        if( h->type == RESET ){
            start_timer(&t_table, 0, 0, 10);
            conf_msg(h, DBM);
        } else

    /*
     * If the message is a timer expiration,
     * confirm that message,
     * call my routine that formats a telegram and sends it
     * to the DRIVER below me to be sent out over the SCC.
     * A telegram is any frame received by the user-interface
     * module (DBM), with the first byte equal to 0xaa.
     * The user-interface prints the data (assumed to be ASCII
     * characters) on the screen.
     */
        if( h->type == TM_EXP ){
            conf_msg(h, DBM);
            send_telegram();

    /* If the message is a data frame from the DRIVER,
     * call my own routine to handle this.
     */
        } else if( h->type == RX_IND )
            handle_rframe(h);
        else
            relm(h);
    }
}

send_telegram()
{
    GCT *gct;    /* define pointer to the GCT */
    T_FRAME *t;  /* define pointer to a T_FRAME */
    int i;
    unsigned char *c;

   /*
    * definition of GCT values provided in a macro.
    * t is a pointer a newly allocated T_FRAME that will
    * have one data BUFFER.  It is acquired by layer 1, the
    * child ID is not used (0), and no response (0) is required
    * to the message that acquires the T_FRAME.
    */
    GETGCT(gct);
    t = gett(TX|BUFFER, 1, 0, 0);

   /* Decrement the layer 1 buffer offset by 13 -- the size
    * of the aa character plus "hello world" plus trailing 0.
    */
    t->layer[1].boff -= 13;
    i = t->layer[1].boff;     /* offset in buffer of start of data */
    c = &t->buf[i];           /* points to start of data in buffer */

    /* fill the data buffer with 0xaa and the hello world string */
    /* (0xaa is a magic number for telegrams) */
    *c++ = 0xaa;
    strcpy(c, "hello world");

   /* Initialize the lower layer T_FRAME fields (required by
    * the software
    * The id field is set to 0 so that the T_FRAME will
    * be handled by driver child 0 (and transmitted on SCC-1)
    */
    t->hdr.id = 0;          /* actually, already set by gett */
    t->layer[0].flags = 0;
    t->layer[0].hoff = t->layer[1].hoff;
    t->layer[0].boff = t->layer[1].boff;
    t->layer[0].nextf = 0;

   /*
    * send the T_FRAME to the DRIVER which will then transmit
    * it over the SCC.
    */
    (*gct->send)(DRIVER, t);
}

handle_rframe(r)
R_FRAME *r;
{
    GCT *gct;

    GETGCT(gct);

   /*
    * Any R_FRAME message received will be that from the DRIVER.
    * The idea is to pass it along up to the user interface
    * so that it will be printed on the screen.
    * If the R_FRAME is not destined for me, call relm to return
    * it to the driver's receive buffer pool.
    */
    if( r->hdr.dst != LAPB )      /* "can't" happen */
            relm(r);
    else
            (*gct->send)(DBM, r);
}
