/********************************************************
 *
 * File:        lapd_example.c
 *
 * Description:
 *      This is a task that interfaces with the LAPD.
 *      It illustrates how messages are sent and received
 *      to and from the LAPD module.
 *      Both upper layer (layer 3) module and layer 2
 *      management module interaction with LAPD is
 *      illustrated. The RESET, CONFIG_LINK and MDL_ASN_REQ
 *      messages would normally be sent to LAPD by a
 *      management module. The DL_EST_REQ and DL_DATA_REQ
 *      would be issued by a layer 3 module.
 *
 *      First it sends a RESET message to configure
 *      the LAPD module.
 *
 *      Then it configures and assigns two LAPD links.
 *      IMP SCC 1 and 2 should be physically connected
 *      to enable these two links to communicate with
 *      each other.
 *
 *      Then a DL_ESTABLISH request is issued to one of
 *      the links, and the task waits for the DL_ESTABLISH
 *      confirm.
 *
 *      Now an I-frame is transmitted on one link;
 *      as it is received on the other link it is passed
 *      to the user interface task which in turn displays
 *      the frame on the screen.
 *
 * The following routines demonstrate the use of the
 * corresponding primitives and action requests:
 *
 *      reset_lapd               -    RESET action request.
 *      wait_lapd_rsp            -    Wait for a response message
 *                                    from the LAPD module.
 *      configure_lapd_link      _    CONFIG_LINK action request.
 *      assign_lapd_link         -    MDL_ASSIGN request primitive.
 *      establish_lapd_link      -    DL_ESTABLISH request primitive,
 *                                    DL_ESTABLISH confirm primitive,
 *                                    DL_RELEASE indication primitive.
 *      transmit_Iframe          -    DL_DATA request primitive.
 *      receive_Iframe           -    DL_DATA indication primitive.
 *
 * Environment
 *      You should run this session from a directory containing
 *      the example g-files as distributed.
 *      You must first make sure that the g-files are edited
 *      as follows:
 *
 *      imp.g
 *          SCC-1 ans SCC-2 should be set to external loopback
 *          mode by setting their configuration registers to 1410
 *          and their mode registesr to 000c.
 *      driver.g
 *          The upper module for SCC-1 and SCC-2 should be set
 *          to 11 (LAPD).
 *
 *      Since we are executing a module from RAM, it will be necessary
 *      to edit the alloc.g and modules.g files as follows:
 *      alloc.g
 *          The last line (board's highest address) should
 *          be set to 6c000.
 *      modules.g
 *          add the line "x25   6c000" to this file
 *          This demo module will replace the X.25 module.
 *
 * Compiler
 *      All examples in this manual were compiled and tested
 *      with a compiler that allocates 32-bit values for integer
 *      and pointer variables. Most ANSI checking is done, but
 *      not all, so warnings may be generated if you use a fully
 *      ANSI-compliant C compiler.
 *
 ********************************************************/

#include "include/gct.h"
#include "include/fatal.h"
#include "include/modules.h"
#include "include/msg.h"
#include "include/types.h"
#include "driver/mtypes.h"
#include "include/l2types.h"
#include "lapd/bss.h"
#include "lapd/config.h"
#include "lapd/protocol.h"
#include "lapd/mtypes.h"

/*
 * LAPD module's start address (used in the
 * RESET message sent to LAPD).
 */
#define LAPD_START_ADR  0

/*
 * Module ID for the example task.
 * This is a layer-3 module.
 */
#define MY_MODULE    (0x00 | L3)

/*
 * DLCI for use in this example
 */
#define EXAMPLE_SAPI    16
#define EXAMPLE_TEI     1
#define EXAMPLE_DLCI    DLCI(EXAMPLE_SAPI, EXAMPLE_TEI)

/*
 * LLIDs for the LAPD links
 */
#define LINK0_ID        0
#define LINK1_ID        1

/*
 * Response required field in messages generated
 * by the example module:
 */
#define MY_RSP          RSP(LAYER(MY_MODULE)) /* Request a response */
#define NO_RSP          0               /* Don't request a response */

extern MSG      *getm();
extern T_FRAME  *gett();

/*
 * Allocate space for LAPD TEI tables.
 * The tables are allocated for SCC0 and SCC 1
 * (SCC 2 not used).
 */
#define LAPD_TEI_TAB_SIZE       0x200   /* 0x80 TEIs * 4       */
char            pchan0_tei_tab[LAPD_TEI_TAB_SIZE];
char            pchan1_tei_tab[LAPD_TEI_TAB_SIZE];

/********************************************************
 *
 * Routine:     lapd_example
 *
 * Description:
 *      This is the LAPD example module.
 *
 * Arguments:
 *
 * Return code:
 *
 * Side effects:
 *
 ********************************************************/

lapd_example()
{
        register GCT    *gct;
        register HDR    *h;

        GETGCT(gct);

        /*
         * Before the example modules starts wait
         * for a RESET action request. All other requests
         * are ignored.
         */
        wait_reset();

        /*
         * First RESET the LAPD module
         */
        if (reset_lapd())
                goto end;

        /*
         * Configure two LAPD links
         */
        if (configure_lapd_link(LINK0_ID, 0, 1))
                goto end;
        if (configure_lapd_link(LINK1_ID, 1, 0))
                goto end;

        /*
         * Assign the two links with the same DLCI
         * and with default protocol parameters.
         */
        if (assign_lapd_link(LINK0_ID, EXAMPLE_DLCI))
                goto end;
        if (assign_lapd_link(LINK1_ID, EXAMPLE_DLCI))
                goto end;

        /*
         * Establish the data link
         */
        if (establish_lapd_link(LINK0_ID))
                goto end;

        /*
         * Now I-frames can be transmitted between the two links.
         * Transmit a frame on link 0.
         */
        if (transmit_Iframe(LINK0_ID))
                goto end;

        /*
         * The frame should be received on link 1.
         * The received frame is passed to the user interface
         * module which displays it on the screen.
         */
        receive_Iframe(LINK1_ID);

end:
        /*
         * End of example task. from now on just
         * empty the input queue and discard all
         * messages.
         */
        for (;;) {
                h = (HDR *)(*gct->receive)(MY_MODULE, 0);
                if (!h)
                        continue;
                relm(h);
        }
} /* lapd_example */

/********************************************************
 *
 * Routine:     wait_reset
 *
 * Description:
 *      This routine is called by the example
 *      module to wait for a RESET message. Only
 *      after the RESET message is received the module
 *      can start running.
 *      No information within the RESET message is
 *      relevant to the example module.
 *
 * Arguments:
 *
 * Return code:
 *
 * Side effects:
 *
 ********************************************************/

wait_reset()
{
        register GCT    *gct;
        register HDR    *h;

        GETGCT(gct);

        for(;;) {
                /*
                 * Get next message from the task's input queue
                 */
                if (!(h = (HDR *)(*gct->receive)(MY_MODULE, 0)))
                        continue;

                if (h->type != RESET)
                        relm(h);
                else
                        break;
        } /* for */
        conf_msg(h, DBM);
} /* wait_reset */

/********************************************************
 *
 * Routine:     reset_lapd
 *
 * Description:
 *      This routine is called by the example
 *      module to reset the LAPD module.
 *      A RESET action request is built and sent to
 *      the LAPD module, and then the example module
 *      waits for the RESET response from LAPD.
 *
 *      LAPD is configured with the following parameters:
 *              Max number of links to support: 2
 *              Tx pool length:                 16 T_FRAMEs
 *              Timers resolution:              2 ticks
 *              Event mask:                     0 (no trace).
 *
 * Arguments:
 *
 * Return code:
 *      0  -  LAPD RESET completed
 *      -1 -  failed
 *
 * Side effects:
 *
 ********************************************************/

reset_lapd()
{
        register GCT            *gct;
        register MSG            *m;
        register LAPD_CONF      *config;

        GETGCT(gct);

        m = getm(RESET, 0, MY_RSP, sizeof(LAPD_CONF));
        if (!m)
                return(-1);

        m->hdr.src = MY_MODULE;
        m->hdr.dst = LAPD;
        config = (LAPD_CONF *) m->param;
        config->rom = LAPD_START_ADR;
        config->max = 2;
        config->tx_pool_len = 16;
        config->tick = 2;
        config->pchan0_tei_tab = pchan0_tei_tab;
        config->pchan1_tei_tab = pchan1_tei_tab;
        config->pchan2_tei_tab = 0;
        config->ev_mask = 0xffff;

        if (gct->send(LAPD, m)) {
                relm(m);
                return(-1);
        }

        /*
         * Wait for LAPD's response.
         */
        if (wait_lapd_rsp(RESET))
                return(-1);

        return(0);
} /* reset_lapd */

/********************************************************
 *
 * Routine:     wait_lapd_rsp
 *
 * Description:
 *      This routine is called by the example
 *      module to wait for a response from the LAPD
 *      module.
 *      It hangs until a response of the specified
 *      type is received.
 *      The returned status is checked.
 *
 * Arguments:
 *      type - Expected response type
 *
 * Return code:
 *      0 -    The expected response was received with
 *             a zero status.
 *      -1 -   The expected response was received with
 *             a non-zero status.
 *
 * Side effects:
 *
 ********************************************************/

wait_lapd_rsp(type)
unsigned int            type;
{
        register GCT            *gct;
        register HDR            *h;
        register int            status;

        GETGCT(gct);

        for (;;) {
                /*
                 * Get next message from the task's input queue
                 */
                h = (HDR *)(*gct->receive)(MY_MODULE, 0);
                if (!h)
                        continue;
                if (h->type == (type & ~REQUEST))
                        break;
                else
                        relm(h);
        } /* for */

        status = h->status;
        relm(h);
        if (status)
                return(-1);
        else
                return(0);
} /* wait_lapd_rsp */

/********************************************************
 *
 * Routine:     configure_lapd_link
 *
 * Description:
 *      This routine is called by the example
 *      module to configure a LAPD link.
 *      The link is configured as a user or network link
 *      over SCC 0 or 1 as indicated in the given
 *      parameters.
 *      A CONFIG_LINK action request is built and sent
 *      to the LAPD module.
 *
 * Arguments:
 *      id -  link ID (LLID)
 *      scc - SCC number
 *      un - user/network (1/0)
 *
 * Return code:
 *      0  -  link configured
 *      -1 -  failed
 *
 * Side effects:
 *
 ********************************************************/

configure_lapd_link(id, scc, un)
unsigned int            id;
unsigned int            scc;
unsigned int            un;
{
        register GCT                     *gct;
        register MSG                     *m;
        register LAPD_LINK_CONFIG        *link_conf;

        GETGCT(gct);

        /*
         * Prepare a CONFIG_LINK message according to the
         * specified configuration parameters.
         */
        m = getm(CONFIG_LINK, id, NO_RSP, sizeof(LAPD_LINK_CONFIG));
        if (!m)
                return(-1);

        m->hdr.src = MY_MODULE;
        m->hdr.dst = LAPD;
        link_conf = (LAPD_LINK_CONFIG *) m->param;
        link_conf->upper = MY_MODULE;
        link_conf->lower = DRIVER;
        link_conf->flags = scc;
        if (un)
                link_conf->flags |= CONFIG_USER;

        if (gct->send(LAPD, m)) {
                relm(m);
                return(-1);
        }

        return(0);
} /* configure_lapd_link */

/********************************************************
 *
 * Routine:     assign_lapd_link
 *
 * Description:
 *      This routine is called by the example
 *      module to assign a LAPD link.
 *      The link is assigned with the specified DLCI
 *      and with default parameters.
 *      A MDL_ASSIGN request primitive is built and sent
 *      to the LAPD module.
 *
 *      The link is assigned with the following parameters:
 *              N201: 260 octets
 *              N200: 3
 *              k:    3
 *              T200: 5
 *              T203: 50
 *
 *      With a 0.1 second tick and a timer resolution of
 *      2 ticks (see reset_lapd above), this gives a T200
 *      value of 1 second and a T203 value of 10 seconds.
 *
 * Arguments:
 *      id -   link ID (LLID)
 *      dlci - DLCI
 *
 * Return code:
 *      0  -  link configured
 *      -1 -  failed
 *
 * Side effects:
 *
 ********************************************************/

assign_lapd_link(id, dlci)
unsigned int            id;
unsigned short          dlci;
{
        register GCT                     *gct;
        register MSG                     *m;
        register LAPD_ASSIGN_PAR         *asn_par;;

        GETGCT(gct);

        /*
         * Prepare a MDL_ASN_REQ message.
         */
        m = getm(MDL_ASN_REQ, id, NO_RSP, sizeof(LAPD_ASSIGN_PAR));
        if (!m)
                return(-1);

        m->hdr.src = MY_MODULE;
        m->hdr.dst = LAPD;
        asn_par = (LAPD_ASSIGN_PAR *) m->param;
        asn_par->dlci = dlci;
        asn_par->T200 = 5;
        asn_par->N201 = 260;
        asn_par->N200 = 3;
        asn_par->k = 3;
        asn_par->T203 = 50;

        if (gct->send(LAPD, m)) {
                relm(m);
                return(-1);
        }

        return(0);
} /* assign_lapd_link */

/********************************************************
 *
 * Routine:     establish_lapd_link
 *
 * Description:
 *      This routine is called by the example
 *      module to establish a LAPD link.
 *      A DL_ESTABLISH request is sent to the LAPD module.
 *      Then wait for a DL_ESTABLISH confirm or a DL_RELEASE
 *      indication from the LAPD module. The first indicates
 *      that link establishment succeeded, the second
 *      indicates failure.
 *      The routine hangs until either of these is received,
 *      discarding any other messages that arrive.
 *
 * Arguments:
 *      id -   link ID (LLID)
 *
 * Return code:
 *      0  -  link established
 *      -1 -  link not established
 *
 * Side effects:
 *
 ********************************************************/

establish_lapd_link(id)
unsigned int            id;
{
        register GCT                     *gct;
        register MSG                     *m;
        register HDR                     *h;

        GETGCT(gct);

        /*
         * Prepare a DL_EST_REQ message.
         */
        m = getm(DL_EST_REQ, id, NO_RSP, 0);
        if (!m)
                return(-1);

        m->hdr.src = MY_MODULE;
        m->hdr.dst = LAPD;

        if (gct->send(LAPD, m)) {
                relm(m);
                return(-1);
        }

        /*
         * Wait for DL_EST_CNF or DL_REL_IND from
         * this link.
         */
        for(;;) {
                /*
                 * Get next message from the task's input queue
                 */
                if (!(h = (HDR *)(*gct->receive)(MY_MODULE, 0)))
                        continue;

                if ((h->src == LAPD) && (h->id == id)) {
                        if (h->type == DL_EST_CNF) {
                                relm(h);
                                return(0);
                        }
                        else if (h->type == DL_REL_IND) {
                                relm(h);
                                return(-1);
                        }
                }

                /*
                 * Any other message is discarded.
                 */
                 relm(h);
        } /* for */
} /* establish_lapd_link */

/********************************************************
 *
 * Routine:     transmit_Iframe
 *
 * Description:
 *      This routine is called by the example
 *      module to transmit an I-frame.
 *
 *      A DL_DATA request is built with a data buffer
 *      containing the string "data buf" and a header
 *      buffer containing the string "hdr buf".
 *      All the per-layer information fields are prepared
 *      for the LAPD module, and the request is
 *      sent to LAPD.
 *
 * Arguments:
 *      id -   link ID (LLID)
 *
 * Return code:
 *      0  -  DL_DATA request issued to LAPD.
 *      -1 -  failed.
 *
 * Side effects:
 *
 ********************************************************/

transmit_Iframe(id)
unsigned int            id;
{
        register GCT            *gct;
        register T_FRAME        *t;
        register unsigned char  *c;
        register int            i;
        register int            hlen, len;
        char                    *hdr_buf = "hdr buf ";
        char                    *data_buf = "data buf\n";

        GETGCT(gct);

        /*
         * Allocate a T_FRAME with a header buffer and a
         * data buffer.
         */
        t = gett(TX | HEADER | BUFFER, LAYER(MY_MODULE), id,
                NO_RSP);
        if (!t)
                return(-1);

        /*
         * Fill the header buffer and data buffer.
         * Note: it is assumed that enough space is
         * allocated in the header buffer and the
         * data buffer to contain the strings (in
         * a "real" implementation this should be checked).
         */
        hlen = strlen(hdr_buf);
        len = strlen(data_buf) + 1;
        t->layer[LAYER(MY_MODULE)].hoff = t->hsize - hlen;
        t->layer[LAYER(MY_MODULE)].boff -= len;

        c = t->hbuf + t->layer[LAYER(MY_MODULE)].hoff;
        for (i = 0; i < hlen; i++)
                c[i] = hdr_buf[i];

        c = t->buf + t->layer[LAYER(MY_MODULE)].boff;
        for (i = 0; i < len; i++)
                c[i] = data_buf[i];

        /*
         * Prepare T_FRAME fields for the LAPD module.
         * Note that only the LAPD layer information is
         * required.
         */
        t->hdr.src = MY_MODULE;
        t->hdr.dst = LAPD;
        t->layer[LAYER(LAPD)].hoff = t->layer[LAYER(MY_MODULE)].hoff;
        t->layer[LAYER(LAPD)].boff = t->layer[LAYER(MY_MODULE)].boff;
        t->layer[LAYER(LAPD)].flags = 0;
        t->layer[LAYER(LAPD)].nextf = 0;

        /*
         * Send the frame to LAPD module
         */
        if (gct->send(LAPD, t)) {
                relm(t);
                return(-1);
        }

        /*
         * Since no response was requested the LAPD
         * module will release the T_FRAME after an
         * acknowledgement is received for this frame.
         */
        return(0);
} /* transmit_Iframe */

/********************************************************
 *
 * Routine:     receive_Iframe
 *
 * Description:
 *      This routine is called by the example
 *      module to handle an I-frame received from
 *      the LAPD module.
 *      It hangs until a DL_DATA indication is received
 *      from LAPD for the specified link. Then the
 *      R_FRAME is passed to the user interface module.
 *      Any other message received is discarded.
 *
 * Arguments:
 *      id -   link ID (LLID)
 *
 * Return code:
 *
 * Side effects:
 *
 ********************************************************/

receive_Iframe(id)
unsigned int            id;
{
        register GCT            *gct;
        register HDR            *h;
        register R_FRAME        *r;

        GETGCT(gct);

        for (;;) {
                /*
                 * Get next message from the task's input queue
                 */
                h = (HDR *)(*gct->receive)(MY_MODULE, 0);
                if (!h)
                        continue;

                if ((h->src == LAPD) && (h->type == DL_DATA_IND) &&
                        (h->id == id))
                        break;
                else
                        relm(h);
        } /* for */

        /*
         * h->id is the ID of the link
         * the R_FRAME is directed to.
         */

        /*
         * Now the frame is going to be sent to the UI module
         * so it can be displayed on the screen.
         * In order to display an R_FRAME on the screen the
         * UI module expects the first character to be 0xaa.
         * An octet 0xaa will be added to the beginning of the
         * frame.
         */
        r = (R_FRAME *)h;
        r->boff -= 1;
        r->flen += 1;
        r->blen += 1;
        r->buf[r->boff] = 0xaa;

        /*
         * Now send the frame to the user interface module
         * DBM (debug nodule)
         */
        if (gct->send(DBM, r)) {
                relm(r);
                return(-1);
        }

        return(0);
} /* receive_iframe */
