/*
 * Example system initialization.
 *
 * This example code illustrates system initialization,
 * allocation of all system data structures, initialization
 * of the GCT and configuration of the DRIVER module.
 * 
 * Also included are all user required routines: the system
 * interface routines for message passing and memory handling
 * as well as a sample interrupt routine (which illustrates
 * the calling of the driver's interrupt handler via the GCT).
 *
 * The operating system used is EDX. The only task created
 * is the task that executes the DRIVER module code.
 * Two partitions are used, P0 and P1. Memory is allocated from
 * P0 for "short" structures (T_FRAME's, R_FRAME's and header
 * buffers) and P1 is used for "long" structures (data buffers
 * and MSG's).
 *
 * Compiler
 *      All examples in this manual were compiled and tested
 *      with a compiler that allocates 32-bit values for integer
 *      and pointer variables. Most ANSI checking is done, but
 *      not all, so warnings may be generated if you use a fully
 *      ANSI-compliant C compiler.
 *
 */


#include "include/gct.h"
#include "include/modules.h"
#include "include/msg.h"
#include "include/types.h"
#include "driver/config.h"

#include "edx/edx.h"

#ifndef VECTOR_LOC
#define VECTOR_LOC(x)          ((x)*4)
#endif


/* system defines */
#define DRIVER_TID    0        /* driver task ID */

#define P0_PID        0        /* message partition id */
#define P1_PID        1        /* T_FRAME partition id */

#define P0_N          100      /* number of blocks in P0 */
#define P1_N          100      /* number of blocks in P1 */

#define P0_S          32       /* block size in partition 0 */
#define P1_S          500      /* block size in partition 1 */

/*
 * EDX_RAM_SIZE is given by the formula:
 *      (64 + (TOTAL_TASKS*32) + USER_STACKS
 *          + (TOTAL_EVENTS*8) + (TOTAL_PARTITIONS*16)
 *          + (TOTAL_BLOCKS*8))
 */
#define EDX_RAM_SIZE  (64 + (1*32) + 0x200 + (100*8) + (1*16) +\
                         ((P0_N+P1_N)*8))


#define DR_MAX        3        /* max number of driver children */


/*
 * space allocations
 * NOTE: The allocation of driver includes space
 * for the child structures (included within the BSS).
 * The definition of BSS depicts one child only, so
 * all other (DR_MAX-1) are allocated here specifically.
 */
GCT        myGCT;
char       edx_ram[EDX_RAM_SIZE];
char       driverBSS[sizeof(BSS) + (sizeof(CHILD)*(DR_MAX-1))];
char       p0_p[P0_S * P0_N];
char       p1_p[P1_S * P1_N];

/*
 * These are the 68302 interrupt routines.
 * The only one included in this example is scc0_isr.
 */
extern int gpip3_isr(), gpip2_isr(), scc0_isr(), dmae_isr();
extern int dma_isr(), scc1_isr(), tim0_isr(), scc2_isr();
extern int gpip1_isr(), tim1_isr(), scp_isr(), tim2_isr();
extern int smi0_isr(), smi1_isr(), gpip0_isr(), error_isr();

int (* imp_isr[])() = {
        error_isr, gpip0_isr, smi1_isr, smi0_isr,
        tim2_isr, scp_isr, tim1_isr, gpip1_isr,
        scc2_isr, tim0_isr, scc1_isr, dma_isr,
        dmae_isr, scc0_isr, gpip2_isr, gpip3_isr
};


/*
 * define program entry point:
 * initialize stack pointer and jump to main
 */
asm("        xdef     _start");
asm("        xref     _main");
asm("_start:");
asm("        move.l   #$4000,sp");
asm("        jmp      _main");

/* 
 * example interrupt routine.
 * the address of this routine is linked to the EVT.
 */
asm("        xdef     scc0_isr");
asm("scc0_isr:");
asm("        movem.l  d0/d1/a0/a1,-(sp)"); /* save registers spoiled by C */
asm("        move.l   #$ec,a0");           /* address of GCT to a0 */
asm("        movea.l  88(a0),a0");         /* get gct->IMPscc0 */
asm("        move.l   a0,d0");             /* update condition codes */
asm("        beq      out");               /* error check */
/*
### Rev 0.5 change
### pass argument to interrupt handler routine
*/
asm("        move.l   #0,-(sp)");          /* use child #0 */
asm("        jsr      (a0)");              /* call driver routine */
/*
### Rev 0.5 change
### remove argument from stack
*/
asm("        add.l    #4,sp");             /* remove argument */
asm("out:    movem.l  (sp)+,d0/d1/a0/a1"); /* restore registers */
asm("        rte");

/*
 * main
 */
main()
{
        E_CONF conf;        /* EDX configuration structure.
                                defined in edx.h */
        int i;
        GCT *gct;
        extern int edxpoll();
        extern send(), receive(), grab();
        extern getmem(), relmem();
        extern int driver();                /* driver task */

        gct = &myGCT;

        /*
         * Initialize EVT
         */
        *(char **)(0x80) = (char *)0x21100c;   /* EDX entry point */
        *(char **)(0xec) = (char *)gct;        /* GCT pointer */
        *(char **)(0x100) = edx_ram;           /* EDX RAM pointer */

        /*
         * Initialize the GCT
         */
        gct->flags = 0;
        gct->time = 0;
        gct->send = send;
        gct->receive = receive;
        gct->grab = grab;
        gct->getmem = getmem;
        gct->relmem = relmem;
        gct->cput = 0;            /* no output needed */

        gct->rxchar = 0;
        gct->tick = 0;
        gct->IMPgpip3 = 0;
        gct->IMPgpip2 = 0;
        gct->IMPscc0 = 0;
        gct->IMPdmae = 0;
        gct->IMPdma = 0;
        gct->IMPscc1 = 0;
        gct->IMPtim0 = 0;
        gct->IMPscc2 = 0;
        gct->IMPgpip1 = 0;
        gct->IMPtim1 = 0;
        gct->IMPscp = 0;
        gct->IMPtim2 = 0;
        gct->IMPsmi0 = 0;
        gct->IMPsmi1 = 0;
        gct->IMPgpip0 = 0;
/*
### Rev 0.5 bug fixed
### following line was missing
*/
        gct->IMPerror = 0;

/*******************************************
        system configuration
*******************************************/

        /*
         * allocate per-module RAM
         */
        gct->driver = (struct module_bss *)driverBSS;

        conf.max_tasks = 1;
        conf.max_events = 100;
        conf.max_partitions = 2;
        conf.max_blocks = P0_N + P1_N;
        conf.ssize = 0x400;
        conf.polling = edxpoll;

        sc_init(&conf);

        /* create partitions */
        /* we should do error checking on all partitions */
        i = sc_pcreate(P0_PID, p0_p, P0_S, P0_N);
        if( i )
                printf("error: pid %d", P0_PID);
        i = sc_pcreate(P1_PID, p1_p, P1_S, P1_N);
        if( i )
                printf("error: pid %d", P1_PID);

        /* create tasks */
        /* driver entry point is "driver",
                user stack size = 0x200 */
        i = sc_tcreate(DRIVER_TID, driver, 0x200);
        if( i ){
                printf("Cannot create task %x", DRIVER_TID);
                exit();
        }

        configure_system();

/*
### Rev 0.5 bug fix
### The following line was missing
*/
        sc_multi();         /* start the multi-tasking environment */

} /* main */


/*
 * configure_system
 * description:
 *        Set up global parameters, configuration.
 *        Send the RESET message to the DBM.
 */

configure_system()
{
        MSG *m;
        int i;
        GCT *gct;
        struct driver_config *driver;
/*
### Rev 0.5 bug fix
### driver configuration structure allocated in static memory
### (m->param contains a pointer to this structure)
*/
        static struct driver_config DriverConf;
        extern MSG *getm();

        GETGCT(gct);

        /*
         * set configuration structure
         */
        m = getm(RESET, 0, 0, sizeof(struct driver_config));
        if (!m)
                printf("config: no mem", 0);

/*
### Rev 0.5 bug fix
### following three lines fix bug described above
*/
        driver = &DriverConf;
        m->param[0] = (long)driver;
        m->len = sizeof(long);


        /*
         * fill in driver configuration information
         */

/*
### Rev 0.5 change
### Added following line
*/
        driver->index = 0;      /* the first (and only) IMP in system */
        driver->rom = 0;        /* when running a module from EPROM,
                                   always set rom = 0 */
        driver->max = DR_MAX;   /* three children: one for each SCC */
        driver->num_timers = 8;
        driver->trace_mask = 0;
        driver->intr_trace = 0;

        /*
         * set IMP register values
         */
        driver->bar = 0x700;     /* internal RAM = 0x700000 */
        driver->scr = 0x800;
        driver->internal_ram = (struct imp *)0x700000;

        /* general purpose dma */
        driver->dma_mode = 0;
        driver->dma_fc = 0;

        /* interrupt controller */
        driver->intr_mode = 0xa0;
        for(i = 0; i < 16; i++)
                *((int *)VECTOR_LOC(0xa0 | i)) = (int)imp_isr[i];
        driver->intr_mask = 0xffff;

        /* parallel port A */
        driver->pa_control = 0;
        driver->pa_direction = 0;
        driver->pb_control = 0;
        driver->pb_direction = 0;

        /* chip selects */
        driver->cs0_base = 0;
        driver->cs0_option = 0;
        driver->cs1_base = 0;
        driver->cs1_option = 0;
        driver->cs2_base = 0;
        driver->cs2_option = 0;
        driver->cs3_base = 0;
        driver->cs3_option = 0;

        /* timer */
        driver->t0_mode = 0;
        driver->t0_ref = 0;
        driver->t1_mode = 0;
        driver->t1_ref = 0;
        driver->wd_ref = 0;

/*
### rev 0.5 changes
### All SCC fields are now set here. This is only a
### cosmetic change.
### The first two lines in the "for" loop below were added
### to initialize the two new fields.
*/
        /* configure each SCC */
        for(i = 0; i < 3; i++){
                /* set driver's variables */
                driver->scc[i].imp = driver->internal_ram;
                driver->scc[i].child_id = i;
                driver->scc[i].upper = 0x1f;
                driver->scc[i].loop_mode = 0xffff;
                driver->scc[i].rxerr_mask = 0;
                driver->scc[i].psize = 10;
                driver->scc[i].minpool = 2;
                driver->scc[i].intr_trace = 0xff;

                /* SCC registers */
                driver->scc[i].conf = 0x7e;
                driver->scc[i].mode = 0x1c;
                driver->scc[i].mask = 0x1e;
                driver->scc[i].sync = 0x7e7e;

                /* SCC parameter RAM */
                driver->scc[i].tfc = 0;
                driver->scc[i].tsize = 8;
                driver->scc[i].rfc = 0;
                driver->scc[i].rsize = 8;
                driver->scc[i].rlen = 260;

                /* HDLC parameter RAM */
                driver->scc[i].pram.h.r_max = 260;
                driver->scc[i].pram.h.r_mask = 0;
                driver->scc[i].pram.h.r_addr1 = 0;
                driver->scc[i].pram.h.r_addr2 = 0;
                driver->scc[i].pram.h.r_addr3 = 0;
                driver->scc[i].pram.h.r_addr4 = 0;
                driver->scc[i].pram.h.r_cmaskl = 0xf0b8;
                driver->scc[i].pram.h.r_cmaskh = 0;
        }

        /* SCP/SMI and serial interface */
        driver->scp_smi_mode = 0;
        driver->si_mask = 0;
        driver->si_mode = 0;


        /*
         * write system configuration registers in supervisor mode.
         * this is done here because the DRIVER executes in
         * user mode and hence cannot write these registers.
         */
        *(short *)0xf2 = driver->bar;
        *(long *)0xf4 = driver->scr;

        /*
         *        Call all other startup functions 
         * (initialize any peripherals, etc.)
         */

        /*
         * send the DRIVER a RESET message with
         * configuration parameters
         */
        if ((*(gct->send))(DRIVER, m))
                printf("cannot send RESET to DRIVER", 0);
}

edxpoll()
{
        /* here goes the content for the EDX polling routine
        (called every task switch) */
}

/*
 * EDX system interface routines
 */
send(mod, m)
int mod;
long m;
{
         int i;

        /*
         * now we convert module ID to task ID
         *
         * NOTE: the module ID passed to the send routine
         * is not necessarily identical to the task ID.
         * Module ID's are predefined (see include/modules.h)
         * whereas the task ID's are set by the operating system.
         * In this example the ID's are equal, however
         * we include the conversion of module ID to task ID
         * for completeness. (In EDX, the user does not have
         * complete control over the ID's of his tasks). 
         * If, in your system, you can guarantee that the
         * task ID's will always equal the module ID's,
         * the conversion is redundant.
         */
         if( mod == DRIVER )
                 i = DRIVER_TID;
         else
                 return( -1 );

         /* system call */
         i = sc_send(i, m);
         return(i);
}

receive()
/* arguments can be ignored */
{
        return( sc_receive() );
}

grab()
/* arguments can be ignored */
{
        return( sc_grab() );
}

getmem(class, size)
int class, *size;
{
        char *c;
        int i, pid;

        /*
         * find partition ID and return size of blocks
         * for that partition.
         * in this example, we will allocate all short
         * structures from P0 and long ones from P1
         */
        class &= ~WHICH_SCC;       /* ignore the per-SCC bits */
        switch( class ){
        case CONTROL:              /* MSG structures */
        case (TX | BUFFER):        /* tx data buffers */
        case (RX | BUFFER):        /* rx data buffers */
                pid = P1_PID;
                *size = P1_S;
                break;
        case TX:                   /* T_FRAME structures */
        case RX:                   /* R_FRAME structures */
        case (TX | HEADER):        /* tx header buffers */
                pid = P0_PID;
                *size = P0_S;
                break;
        default:
                return(0);
        }

        i = sc_gblock(pid);
/*
### Rev 0.5 bug fixed
### the following line was return(h)
*/
        return(i);
}

relmem(class, blk)
int class;
char *blk;
{
        int i, pid;

        class &= ~WHICH_SCC;       /* ignore the per-SCC bits */
        switch( class ){
        case CONTROL:
        case (TX | BUFFER):
        case (RX | BUFFER):
                pid = P1_PID;
                break;
        case TX:
        case (TX | HEADER):
        case RX:
                pid = P0_PID;
                break;
        default:
                return(-1);
        }

        i = sc_rblock(pid, blk);
        return(i);
}
