/********************************************************
 *
 * File:        x25intr.c
 *
 * Description:
 *        This is a task that interfaces with the X.25.
 *        It sends and receives messages to and from the
 *        X.25 module.
 *
 *        First it will send two configure link messages to
 *        the lapb module in order configure two links of lapb.
 *        These two links will be physically configured (on
 *        the board) to communicate with each other via a
 *        physical link.
 *
 *        From here all the interface is done only with the
 *        X.25 module.
 *        After the two lapb links are configured the task will
 *        send two configure interface Action requests to X.25
 *        module in order to configure two DTE/DCE interfaces
 *        on each lapb physical link.
 *        One interface will be configured as a DTE and the other
 *        as a DCE.
 *
 *        When the interfaces are configured it will send four
 *        configure lchan action requests for four lchans, two
 *        lchans on each interface.
 *        On each interface one lchan will be configured as a
 *        Virtual call, and the other as a Permanent Virtual Circuit.
 *
 *        After the lchans are configured, a restart interface action
 *        request will be sent to one of the interfaces.
 *        This will cause the interface to send the lower module
 *        a DL connect request.
 *        When the DL connect confirm arrives the X.25 module will
 *        send an X.25 restart packet on this interface.
 *
 *        On receiving the RESTART packet, the X.25 module will
 *        generate an N_CLR_IND primitive for each lchan that is
 *        configured on the interface, and request transmission of
 *        a RESTART CONFIRM packet.
 *        Reception of the peer entitle of the RESTART CONFIRM
 *        packet will result in the generation of an N_CLR_IND
 *        primitive for each lchan that is configured on the interface.
 *
 *        At this stage, the PVCs are ready to transmit and
 *        receive data, the VCs may now be issued a CALL_REQUEST
 *        primitive.
 *
 *        After sending the CALL_REQUEST primitive on one of the lchans
 *        This module will get two primitives messages from the X.25
 *        task:
 *        A call confirm primitive with the id (id field in the HDR
 *        structure of the msg) set to the number of the lchan that
 *        sent the call request primitive.
 *        A call indication primitive with the id of the lchan that
 *        was configured on the other DTE/DCE interface with the
 *        same lcgn and lcn of the lchan that requested the call.
 *
 * Routines:
 *        x25
 *        x25_wait_rst
 *
 * Environment
 *      You should run this session from a directory containing
 *      the example g-files as distributed.
 *      You must first make sure that the g-files are edited
 *      as follows:
 *
 *      imp.g
 *          SCC-1 ans SCC-2 should be set to external loopback
 *          mode by setting their configuration registers to 1410
 *          and their mode registesr to 000c.
 *      driver.g
 *          The upper module for SCC-1 and SCC-2 should be set
 *          to 01 (LAPB).
 *
 *      Since we are executing a module from RAM, it will be necessary
 *      to edit the alloc.g and modules.g files as follows:
 *      alloc.g
 *          The last line (board's highest address) should
 *          be set to 6a800.
 *      modules.g
 *          The following lines should be in modules.g:
 *          "lapb    7a000"
 *          "x25     6c000"
 *          "ft      6a800"
 *      This demo module will replace the FT (File Transfer) module.
 *
 * Compiler
 *      All examples in this manual were compiled and tested
 *      with a compiler that allocates 32-bit values for integer
 *      and pointer variables. Most ANSI checking is done, but
 *      not all, so warnings may be generated if you use a fully
 *      ANSI-compliant C compiler.
 *
 ********************************************************/

#include "include/gct.h"
#include "include/fatal.h"
#include "include/modules.h"
#include "include/msg.h"
#include "include/types.h"
#include "x25/bss.h"
#include "lapb/bss.h"
#include "x25/protocol.h"
#include "include/l2types.h"
#include "x25/mtypes.h"
#include "lapb/mtypes.h"

#define X25INTR       6
#define READY         0x01
#define DATA_PHASE    0x02

extern MSG        *getm();
extern T_FRAME    *gett();
extern MSG        *rsvp();

x25intr()
{
        register GCT      *gct;
        int               module;
        MSG               *m;
        HDR               *h;
        X25_INTR_CONFIG   *intr_conf;

        /*
         * Flags used by the application to keep track of each
         * X.25 circuits state.
         */
        unsigned char     channel_flags[4];
        int               i;
        X25_CALL_FACILS   *call_req_facils;
        T_FRAME           *t;
        unsigned char     *c;
        char              *hdr_buf = "hdr buf ";
        char              *data_buf = "data buf\n";
        int               hlen, len;
        R_FRAME           *r;

        GETGCT(gct);

        for (i = 0; i < 4; i++)
                channel_flags[i] = 0;
        module = X25INTR;

        /*
         * Before the module's main loop starts wait
         * for a RESET action request. All other requests
         * are ignored.
         */
        x25intr_wait_rst(gct);

        /*
         * First configure two lapb links
         */
        if (configure_lapb_link(0, DCE, module))
                goto end;
        if (configure_lapb_link(1, 0, module))
                goto end;
        /*
         * Configure two DTE/DCE interface that will communicate with
         * each one via a physical connection on the board.
         * The first will be configured as a DTE and the second
         * as a DCE.
         */
        if (configure_interface(0, I_DCE_M, module))
                goto end;
        if (configure_interface(1, 0, module))
                goto end;

        /*
         * Configure two Virtual Call lchans one on each
         * interface.
         * these two lchans should have the same logical
         * channel group number (lcgn) and logical channel
         * number (lcn) in order to communicate with each
         * other.
         */
        if (configure_lchan(0, 0, 0, module, 0, 7))
                goto end;
        if (configure_lchan(1, 1, 0, module, 0, 7))
                goto end;

        /*
         * Configure two Permanent Virtual Circuit lchans
         * one on each interface.
         * these two lchans should have the same logical
         * channel group number (lcgn) and logical channel
         * number (lcn) in order to communicate with each other.
         * The combination of lcgn and lcn of a lchans should
         * be deferment from the combination of other lchans
         * already configured on the same interface, other wise
         * the configuration will fail.
         */
        if (configure_lchan(2, 0, PVC, module, 1, 7))
                goto end;
        if (configure_lchan(3, 1, PVC, module, 1, 7))
                goto end;

        /*
         * Now send a P_RESTART_REQ - restart action request
         * to one of the interfaces in order to restart them.
         * The message will be sent with id = 0 that is to
         * interface #0.
         */
        m = getm(INTR_EST_RSTRT, 0, 0, 0);
        if (!m)
                goto end;

        if (gct->send(X25, m)) {
                relm(m);
                goto end;
        }

        do {
                h = (HDR *)(*gct->receive)(module, 0);
                if (!h)
                        continue;
                if (h->type != N_CLR_IND) {
                        relm(h);
                        continue;
                }
                switch ((int)(h->id)) {
                case 0:
                        channel_flags[0] |= READY;
                        break;
                case 1:
                        channel_flags[1] |= READY;
                        break;
                case 2:
                        channel_flags[2] |= DATA_PHASE;
                        break;
                case 3:
                        channel_flags[3] |= DATA_PHASE;
                        break;
                default:
                        relm(h);
                        continue;
                        break;
                }
                relm(h);
        } while (!(channel_flags[0] & READY) ||
                !(channel_flags[1] & READY) ||
                !(channel_flags[2] & DATA_PHASE) ||
                !(channel_flags[3] & DATA_PHASE));
        /*
         * In this stage lchans #2 and #3 are in the data
         * transfer state (p4), and lchans #0 and #1 are
         * in the ready state (p1).
         * call the call_request procedure in order to put
         * lchans #0 and #1 in the data phase state.
         */

        /*
         * Now send a N_CALL_REQ primitive to the X.25
         * module.
         * The id will be for lchan #0.
         */
        m = getm(N_CALL_REQ, 0, 0, sizeof(X25_CALL_FACILS));
        if (!m)
                goto end;

        /*
         * The parameter field of the message includes facilities
         * values.
         * If a certain facility is requested a flag in
         * the facilities field will be set.
         * The definition of the X25_CALL_FACILS structure
         * and the flags for each facility are defined in
         * the file x25/protocol.h
         * In this explanation all the fields are set to zero.
         */
        call_req_facils = (X25_CALL_FACILS *)m->param;
        call_req_facils->facilities = 0;
        call_req_facils->called_throput = 0;
        call_req_facils->calling_throput = 0;
        call_req_facils->cug_sel = 0;
        call_req_facils->cug_outg_acc = 0;
        call_req_facils->bilateral_cug = 0;
        call_req_facils->user_data_len = 0;
        call_req_facils->fast_rest = 0;
        call_req_facils->nui_len = 0;
        call_req_facils->rpoa_len = 0;
        for (i = 0; i < MAX_NUI_LEN; i++)
                call_req_facils->nui[i] = 0;
        for (i = 0; i < MAX_RPOA_LEN; i++)
                call_req_facils->rpoa[i] = 0;

        call_req_facils->t_delay = 0;
        for (i = 0; i < MAX_ADDR_LEN; i++) {
                call_req_facils->called_addr[i] = 0;
                call_req_facils->calling_addr[i] = 0;
        }
        call_req_facils->others_len = 0;

        if (gct->send(X25, m)) {
                relm(m);
                goto end;
        }

        /*
         * Now, if the call was established, an N_CALL_CNF
         * should arrive for the lchan that requested the call
         * and an N_CALL_IND for the lchan that is connected
         * the other interface and has the same lcgn and lcn.
         * If the call was not established for some reason
         * an N_CLR_IND primitive will be send to both lchans.
         */
        do {
                h = (HDR *)(*gct->receive)(module, 0);
                if (!h)
                        continue;
                if (h->type == N_CLR_IND) {
                        relm(h);
                        goto end;
                }
                if ((h->type != N_CALL_IND) &&
                                        (h->type != N_CALL_CNF)) {
                        relm(h);
                        continue;
                }
                if (h->type == N_CALL_CNF)
                        channel_flags[0] |= DATA_PHASE;
                else if (h->type == N_CALL_IND)
                        channel_flags[1] |= DATA_PHASE;
                else {
                        relm(h);
                        continue;
                }
                relm(h);
        } while (!(channel_flags[0] & DATA_PHASE) ||
                                !(channel_flags[1] & DATA_PHASE));
        /*
         * Now data can be transferred from logical channel #0 to
         * logical channel #1, from #1 to #0 from #2 to #3 and
         * from #3 to #2.
         */

        /*
         * A TX_FRAME will be send from logical channel #0,
         * this frame will include a data buffer and
         * a header buffer.
         * the header buffer will be filled with the string
         * "header buffer", the data buffer will be filled
         * with the string "data buffer"
         * It is assumed that the length of the data and
         * header buffer configured is enough to include
         * these strings.
         * The definitions for the T_FRAME structure are
         * in the ads programmers manual.
         */
        hlen = strlen(hdr_buf);
        len = strlen(data_buf) + 1;
        t = gett(TX | HEADER | BUFFER, LAYER(module), 0, 0);
        if( !t )
                goto end;
        t->layer[LAYER(module)].hoff = t->hsize - hlen;
        t->layer[LAYER(module)].boff -= len;
        t->layer[LAYER(module)].nextf = 0;

        t->layer[LAYER(X25)].hoff = t->layer[LAYER(module)].hoff;
        t->layer[LAYER(X25)].boff = t->layer[LAYER(module)].boff;
        t->layer[LAYER(X25)].nextf = 0;
        t->layer[LAYER(X25)].flags = 0;

        c = t->hbuf + t->layer[LAYER(module)].hoff;
        for (i = 0; i < hlen; i++)
                c[i] = hdr_buf[i];

        c = t->buf + t->layer[LAYER(module)].boff;
        for (i = 0; i < len; i++)
                c[i] = data_buf[i];
        /*
         * Send the frame to X.25 module
         */
        if (gct->send(X25, t)) {
                relm(t);
                goto end;
        }
        /*
         * Now an R_FRAME is expected to arrive with the id
         * #1 because the T_frame was sent with the id #0 and
         * #0 and #1 are connected through the DTE/DCE interfaces
         * that are connected through the lower module.
         */
        for (;;) {
                h = (HDR *)(*gct->receive)(module, 0);
                if (!h)
                        continue;
                if ((h->type == N_DATA_IND) && (h->id == 1))
                        break;
                relm(h);
        }
        /*
         * h->id is the number of logical channel that the
         * the R_FRAME is directed to.
         */

        /*
         * Now the frame is going to be send to the UI module
         * so it can be displayed on the screen.
         * In order to display an R_FRAME on the screen the
         * UI module expects the first character to be 0xaa.
         * An octet 0xaa will be added to the beginning of the
         * frame.
         */
        r = (R_FRAME *)h;
        r->boff -= 1;
        r->flen += 1;
        r->blen += 1;
        r->buf[r->boff] = 0xaa;
        /*
         * Now send the frame to the user interface module
         * DBM (debug nodule)
         */
        if (gct->send(DBM, r)) {
                relm(r);
                goto end;
        }

end:
        for (;;) {
                h = (HDR *)(*gct->receive)(module, 0);
                if (!h)
                        continue;
                relm(h);
        }
} /* x25 */

x25intr_wait_rst(gct)
GCT        *gct;
{
        register HDR  *h;
        int           module;


        module = X25INTR;
        for(;;) {
                if (!(h = (HDR *)(*gct->receive)(module, 0)))
                        continue;

                if (h->dst != module) {
                        relm(h);
                        continue;
                }

                if (h->type != RESET)
                        conf_msg(h, DBM);
                else
                        break;
        } /* for */
        conf_msg(h, DBM);
} /* x25intr_wait_rst */

/********************************************************
 *
 * Routine:        configure_interface
 *
 * Description:
 *        This routine will send CONFIG_INTR action requests to
 *        the X.25 module.
 *
 * Arguments:
 *        id        interface id
 *        flags        configuration flags
 *        id        module id
 *
 * Side effects:
 *
 ********************************************************/

configure_interface(id, flags, module)
unsigned int    id;
unsigned short  flags;
int             module;
{
        MSG             *m;
        X25_INTR_CONFIG *intr_conf;
        register GCT    *gct;

        GETGCT(gct);
        /*
         * Prepare a CONFIG_INTR message according to the
         * specified configuration parameters.
         * The message dose not ask for a response, normally
         * there will be a response required and the status will
         * be check on the response.
         */
        m = getm(CONFIG_INTR, id, 0, sizeof(X25_INTR_CONFIG));
        if (!m)
                return(-1);

        m->hdr.src = module;
        intr_conf = (X25_INTR_CONFIG *) m->param;
        intr_conf->flags = flags;
        intr_conf->upper = module;
        intr_conf->lower = LAPB;
        /*
         * The lower child id will be the same as the interface id
         */
        intr_conf->lower_child = id;
        /*
         * default value for the timer.
         * See timer.c in the lib for explanation on how the timer
         * value is calculated.
         */
        intr_conf->T10 = 18;
        intr_conf->N10 = 1;
        intr_conf->T28 = 18;
        intr_conf->tx_def_max_ud = 80;
        intr_conf->rx_def_max_ud = 80;
        intr_conf->def_w_sz = 2;

        if (gct->send(X25, m)) {
                relm(m);
                return(-1);
        }
        return(0);
}

/********************************************************
 *
 * Routine:        configure_lchan
 *
 * Description:
 *        This routine will send CONFIG_LINK action requests to
 *        the X.25 module.
 *
 * Arguments:
 *        id            lchan (logical channel) id.
 *        intr_id       dTE/DCE interface id.
 *        flags         lchan configuration flag.
 *        module        module id.
 *        lcgn          logical channel group number.
 *        lcn           logical channel number.
 *
 * Side effects:
 *
 ********************************************************/

configure_lchan(id, intr_id, flags, module, lcgn, lcn)
unsigned int    id;
unsigned short  intr_id;
unsigned short  flags;
int             module;
unsigned char   lcgn;
unsigned char   lcn;
{
        MSG               *m;
        X25_LCHAN_CONFIG  *lchan_conf;
        register GCT      *gct;

        GETGCT(gct);
        /*
         * Prepare a CONFIG_LINK message according to the
         * specified configuration parameters.
         */
        m = getm(X25_CONFIG_LCHAN, id, 0, sizeof(X25_LCHAN_CONFIG));
        if (!m)
                return(-1);

        m->hdr.src = module;
        lchan_conf = (X25_LCHAN_CONFIG *) m->param;
        lchan_conf->intr_id = intr_id;
        lchan_conf->upper = module;
        lchan_conf->flags = flags;
        lchan_conf->lcgn = lcgn;
        lchan_conf->lcn = lcn;
        lchan_conf->w = 2;
        lchan_conf->T11 = 20;
        lchan_conf->T12 = 18;
        lchan_conf->T13 = 18;
        lchan_conf->N12 = 1;
        lchan_conf->N13 = 1;
        lchan_conf->tx_max_ud_field = 128;
        lchan_conf->rx_max_ud_field = 128;

        if (gct->send(X25, m)) {
                relm(m);
                return(-1);
        }
        return(0);
}

/*
 * Default parameter values
 */
#define TICK           1        /* 0.1 second */
#define TIMER_RES      2        /* Two ticks (0.2 second) */
#define TIMER_VAL(t)   ((t) / (TIMER_RES * TICK))
#define LAPB_DF_T1     TIMER_VAL(10)        /* One second */
#define LAPB_DF_N2     3
#define LAPB_DF_N1     260
#define LAPB_DF_k      3

/********************************************************
 *
 * Routine:        configure_lapb_link
 *
 * Description:
 *        This routine will send CONFIG_LINK action requests to
 *        the LAPB module.
 *
 * Arguments:
 *        id            link id.
 *        flags         link configuration flags.
 *        module        module id.
 *
 * Side effects:
 *
 ********************************************************/

configure_lapb_link(id, flags, module)
unsigned int   id;
unsigned short flags;
int            module;
{
        GCT    *gct;
        MSG    *m;
        LAPB_LINK_CONFIG  *link_conf;

        GETGCT(gct);

        /*
         * Prepare a CONFIG_LINK message according to the
         * specified configuration parameters.
         */
        m = getm(CONFIG_LINK, id, 0, sizeof(LAPB_LINK_CONFIG));
        if (!m)
                return(-1);

        m->hdr.src = module;
        link_conf = (LAPB_LINK_CONFIG *) m->param;
        link_conf->upper = X25;
        link_conf->lower = DRIVER;
        link_conf->T1 = LAPB_DF_T1;
        link_conf->N1 = LAPB_DF_N1;
        link_conf->N2 = LAPB_DF_N2;
        link_conf->k = LAPB_DF_k;
        link_conf->flags = id & WHICH_SCC;
        link_conf->flags |= flags;

        if (gct->send(LAPB, m)) {
                relm(m);
                return(-1);
        }

        return(0);
}
