/********************************************************
 *
 * File:	protocol.h
 *
 * Description:
 *	x25 definitions
 *
 * Author:
 *	Tamar Mishory
 *
 ********************************************************/

/*
 * Packed header structure.
 *
 * The first char contains two fields:
 * Bits 7-4 : general format identifier GFI. 
 * Bits 3-0 : logical channel group number which in some packets is 0.
 *
 * The second char is the logical channel number which in some
 * packets is 0.
 *
 * The third char contains the packet type identifier.
 * Bit 0 distinguishes data packets from all other packets
 * The type id is not always 8 bits in some packets it is less
 * and the bits left are used for other fields.
 */

typedef struct {
	unsigned char	gfi_lcgn;
	unsigned char	lcn;
	unsigned char	typeid;
	unsigned char	info[2];
} X25_HDR;

#define HDR_LEN			3

/*
 * field definitions and macros.
 */
#define GFI			0xf0
#define GFI_SHIFT		4
#define Q_BIT			0x80
#define A_BIT			0x80
#define D_BIT			0x40
#define P_MODULU		0x30
#define P_MOD8			0x10
#define P_MOD128		0x20
#define MBIT_8			0x10
#define MBIT_128		0x01
#define IS_P_MOD8(h)		(((((h)->gfi_lcgn) & GFI) & P_MODULU) \
					== P_MOD8)
#define IS_P_MOD128(h)		(((((h)->gfi_lcgn) & GFI) & P_MODULU) \
					== P_MOD128)
#define LCGN			0x0f
#define NO_DATA_PACKET		0x01
#define	DATA_PACKET(h)		(!(((h)->typeid) & NO_DATA_PACKET))
#define PR8			0xe0
#define PR8_SHIFT		5
#define PR128			0xfe
#define PR128_SHIFT		1
#define PR(h, mod128)		(mod128 ? \
				((((h)->info[0]) & PR128) >> \
						PR128_SHIFT) : \
				((((h)->typeid) & PR8) >> PR8_SHIFT))
#define PR_SHIFT(mod128)	(mod128 ? PR128_SHIFT : PR8_SHIFT)
					

#define PS8			0x0e
#define PS_SHIFT		1
#define PS128			0xfe
#define PS(h, mod128)		(mod128 ? \
				((((h)->typeid) & PS128) >> \
							PS_SHIFT) : \
				((((h)->typeid) & PS8) >> PS_SHIFT))

#define	TX_PS(ps)		((ps) << PS_SHIFT)

#define	VALID_PR(pr, pa, ps)	(((pa) <= (ps)) ? \
				((((pr) < (pa)) || ((pr) > (ps))) ? \
				0 : 1) : \
				((((pr) > (ps)) && ((pr) < (pa))) ? \
				0 : 1))

#define	DATA_HDR_LEN(mod8)	(mod8 ? 3 : 4)
#define	INTRPT_HDR_LEN		3
#define	STNDRD_USR_DATA_LEN	128
#define	STNDRD_WINDOW_SZ	2
#define	MIN_RESTART_LEN		4
#define	MAX_RESTART_LEN		5
#define	MIN_RESET_LEN		4
#define	MAX_RESET_LEN		5


/*
 * Per interface states
 */
#define INTR_NOT_CONFIG		0
#define INTR_NOT_EST		1
#define INTR_NOT_READY		2
#define RESTART_REQ_PEER	3	/* Restart request peer */
#define RESTART_REQ_UPPER	4	/* Restart request upper */
#define INTR_READY		5	/* ssc ready */

/*
 * Packet type identifiers
 */
#define CALL_REQ_IND		0x0b	/* call request packet	*/
#define CALL_ACC_CONN		0x0f	/* call accept packet	*/
#define CLR_REQ_IND		0x13	/* clear req/indication	*/
#define CLR_CONFRM		0x17	/* clear confirmed	*/
#define INTERRPUT_PCT		0x23	/* interrupt packet	*/
#define INTERRPUT_CONFRM	0x27	/* interrupt confirmation*/
#define FLOW_CTRL_MASK		0x1f	/*flow control mask	*/
#define RR_PCT			0x01	/* rr packet		*/
#define RNR_PCT			0x05	/* rnr packet		*/
#define RESET_REQ_IND		0x1b	/* reset req indication	*/
#define RESET_CONFRM		0x1f	/* reset confirmation	*/
#define RESTART_PACKET		0xfb	/* restart packet 	*/
#define RESTART_CONFRM		0xff	/* restart confirmation	*/
#define DIAGNOSTIC		0xf1	/* diagnostic packet	*/
#define REGISTRATION_REQ	0xf3	/* registration request	*/
#define REGISTRATION_CONF	0xf7	/* registration confirm	*/



#define CALL_SETUP_PACKET(h)	((((h)->typeid) == CALL_REQ_IND) || \
				(((h)->typeid) == CALL_ACC_CONN))
#define CLEARING_PACKET(h)	((((h)->typeid) == CLR_REQ_IND) || \
				(((h)->typeid) == CLR_CONFRM))
#define IS_RR_PCT(p)	((p) & FLOW_CTRL_MASK) == RR_PCT
#define IS_RNR_PCT(p)	((p) & FLOW_CTRL_MASK) == RNR_PCT
#define FLOW_CTL_PACKET(h)	((IS_RR_PCT((h)->typeid)) || \
				(IS_RNR_PCT((h)->typeid)))

/*
 * MN_ERROR indication codes.
 * These codes are contained in the status field
 * of MN_ERROR indication messages sent by the
 * X25 module to layer 2 management module.
 */
#define	N_NO_HDR	0x01	/* Rx frame discarded (header	      */
#define	RX_INACTV_INT	0x02	/* Rx frame discarded inactiv intr    */
#define	RX_INTR_NOT_EST	0x03	/* Rx frame discarded i not establish */
#define	RX_PKT_TO_SHORT	0x04	/* Rx frame discarded pkt to short    */
#define	RX_BAD_GFI	0x05	/* Rx frame discarded bad gfi         */
#define	RX_UNASIGNED_LC	0x06	/* Rx frame discarded unassigend lcn  */
#define	BAD_I_STATE	0x07	/* Bad interface state  	      */
#define	RX_I_NOT_READY	0x08	/* Bad interface state  	      */
#define	BAD_L_STATE	0x09	/* Bad lchan state  	      	      */
#define	RCV_DL_DISC_IND	0x0a	/* Recived a DL_DICS_IND from l2      */
#define	RCV_DL_DISC_CNF	0x0b	/* Recived a DL_DICS_CNF from l2      */
#define	RESTART_T_OUT	0x0c	/* timeout when executin restart proc */
#define	N_TX_FAIL	0x0d	/* Transmit t_frame failed */
#define	N_RX_FAIL	0x0e	/* Recived t_frame failed */
#define	RESET_T_OUT	0x0f	/* timeout when executin reset proc */
#define	RX_UNDEF_PCKT	0x10	/* Undefined packet type */
#define	UNCONST_QBIT	0x11	/* unconsistent qbit in full data seq */
#define	BAD_RX_SEC	0x12	/* dbit = 0, mbit = 1, and not full */
#define	CAN_NOT_SPLIT	0x13	/* can not split a t_frame no mem */
#define	RCV_DIAG	0x14	/* Recived a DIAGNOSTIC packet      */
#define	RX_PCT_TO_LONG	0x15	/* packet to long      		*/
#define	NO_MSG_FOR_COPY	0x16	/* no msg for packet copying	*/
#define	RX_DL_CONN_CNF	0x17	/* Received DL_CONN_CNF without asking*/
#define	NO_MSG_FOR_IND	0x18	/* no msg for indication to upper */
#define	NET_CONG_OP	0x19	/* Network congestion/operational */
#define	RX_REGCNF_NOSND	0x1a	/* Received reg cnf without send req */
#define	CLR_T_OUT	0x1b	/* Received a restart pakcet. */
#define	REGREQ_T_OUT	0x1c	/* Registration request timeout. */

#define PTYPE_CORRECT(id, flags)	(((id) == CALL_REQ_IND) || \
				((id) == CALL_ACC_CONN) || \
				((id) == CLR_REQ_IND) || \
				((id) == CLR_CONFRM) || \
				(!((id) & NO_DATA_PACKET)) || \
				((id) == INTERRPUT_PCT) || \
				((id) == INTERRPUT_CONFRM) || \
				(((id) & FLOW_CTRL_MASK) == RR_PCT) || \
				(((id) & FLOW_CTRL_MASK) == RNR_PCT)|| \
				((id) == RESET_REQ_IND) || \
				((id) == RESET_CONFRM) || \
				((id) == RESTART_PACKET) || \
				((id) == RESTART_CONFRM) || \
				((!dce) && (flags & DIAG_SUPPORT) && \
				((id) == DIAGNOSTIC)) || \
				(dce && (flags & REG_SUPPORT) && \
				((id) == REGISTRATION_REQ)) || \
				((!dce) &&  (flags & REG_SUPPORT) && \
				((id) == REGISTRATION_CONF)))

#define MAX_INTRPT_USR_DATA		32

/*
 * Coding of restarting cause field in 
 * restart indication packets.
 */
#define LOCAL_P_ERR			0x01
#define NW_CONGESTION			0x03
#define NW_OPERATIONAL			0x07
#define REG_CANS_CONF			0x7f

/*
 * Coding of clearing cause field in
 * clear indication packets
 */
#define NUMBER_BUSY			0x01
#define OUT_OF_ORDER			0x09
#define REMOTE_P_ERR			0x11
#define RCA_NOT_SUB			0x19
#define INCOMP_DEST			0x21
#define FSA_NOT_SUB			0x29
#define SHIP_ABSENT			0x39
#define INVALID_F_REQ			0x03
#define ACCESS_BARED			0x0b
#define VC_LOCAL_P_ERR			0x13
#define VC_NW_CONGESTION		0x05
#define NOT_OBTAINABLE			0x0d
#define RPOA_O_O_ORDER			0x15

/*
 * Coding of reseting cause field in
 * reset indication packets
 */
#define PVC_OUT_OF_ORDER		0x01
#define PVC_REMOTE_P_ERR		0x03
#define PVC_LOCAL_P_ERR			0x05
#define PVC_NW_CONGESTION		0x07
#define REM_DTE_OPRATIOANEL		0x09
#define PVC_NW_OPERATIONAL		0x0f
#define PVC_INCOMP_DEST			0x11
#define NW_OUT_OF_ORDER			0x1d

/*
 * Coding of cause field in registration confirmation packets
 */
#define RP_REG_CANS_CONF		0x7f
#define RP_INVAL_REQ			0x03
#define RP_LOCAL_P_ERR			0x13
#define RP_NW_CONGESTION		0x05

#define BAD_RP_CAUSE(c)		(((c) != RP_REG_CANS_CONF) && \
					((c) != RP_INVAL_REQ) && \
					((c) != RP_LOCAL_P_ERR) && \
					((c) != RP_NW_CONGESTION))


/*
 * Coding of diagnostic fields in clear, reset 
 * and restart indication, registration confirmation
 * and diagnostic packets.
 */
#define NO_ADD_INFORM			0
#define INVALID_PS			1
#define INVALID_PR			2
#define UNDEFF_PCKT			33
#define INVALID_PCKT_ON_PVC		35
#define UNASSIGNED_LOGICAL_CHANNEL	36
#define PCKT_TOO_SHORT			38
#define PCKT_TOO_LONG			39
#define INVALID_GFI			40
#define NONZERO_OCT114_OCT218		41
#define UNAUTH_INTRPT_CONFRM		43
#define UNAUTH_INTRPT			44
#define TMEXP_FOR_CALL			49
#define TMEXP_FOR_CLR			50
#define TMEXP_FOR_RESET			51
#define TMEXP_FOR_RESTART		52
#define INVAL_FACIL_REQ			65
#define INVAL_FACIL_VAL			66
#define INVAL_CALLED_ADDR		67
#define INVAL_CALLING_ADDR		68
#define INVAL_FACIL_LEN			69
#define DUP_FACIL_REQ			73
#define NON0_ADR_LEN			74
#define IMPROPER_CAUSE			81
#define UNCONST_QBIT_SET		83


#define MAX_RPOA_LEN			4
#define MAX_NUI_LEN			18
#define MAX_ADDR_LEN			18
#define MAX_OTHER_FACILS		30
#define X25_MAX_USR_DATA		128
#define MAX_INFO_LEN			24

typedef struct {
	unsigned short	facilities;	/* Bit for each facility */
	unsigned short	cug_sel;	/* closed user group slct */
	unsigned short	cug_outg_acc;	/* CUG with outgoing access */
	unsigned short	rpoa[MAX_RPOA_LEN];
	unsigned short	t_delay;		/* Transmit delay */
	unsigned short	bilateral_cug;		/* bilateral CUG */
	unsigned short	user_data_len;
	unsigned char	calling_pckt_sz; 	/* Calling pckt size */
	unsigned char	called_pckt_sz; 	/* called pckt size */
	unsigned char	calling_w_sz; 		/* calling window sz */
	unsigned char	called_w_sz; 		/* called window size */
	unsigned char	calling_throput;
	unsigned char	called_throput;
	unsigned char	fast_rest;		/* Restriction */
	unsigned char	nui_len;		/* NUI field len */
	unsigned char	nui[MAX_NUI_LEN]; 
	unsigned char	rpoa_len;		/* RPOA length */
	unsigned char	call_redef_res; /*call redirect deflect reason*/
	unsigned char	alternate_addr[MAX_ADDR_LEN];
	unsigned char	cline_add_mod_not; /*called line addr modified*/
	/*
	 * The calling and called address of the call
	 */
	unsigned char	called_addr[MAX_ADDR_LEN];
	unsigned char	calling_addr[MAX_ADDR_LEN];
	/*
	 * Non X25 facilities starting with a marker 0x00
	 * or extended fcilities startin with code 0xff
	 */
	unsigned char	others_len;
	unsigned char	others[MAX_OTHER_FACILS];
	unsigned char	user_data[X25_MAX_USR_DATA];
} X25_CALL_FACILS;

typedef struct {
	unsigned short	facilities;	/* Bit for each facility */
	unsigned short	user_data_len;
	unsigned char	charging_info_rcv;	/* receive info flags */
	unsigned char	mn_info_len;		/* Monetary unit ilen */
	unsigned char	sc_info_len;		/* Segment count ilen */
	unsigned char	cd_info_len;		/* Call duration ilen */
	unsigned char	nm_info[MAX_INFO_LEN];	/* Monetary unit info */
	unsigned char	sc_info[MAX_INFO_LEN];	/* Segment count info */
	unsigned char	cd_info[MAX_INFO_LEN];	/* Call duration info */
	unsigned char	call_def_res;	/* call deflaction reason */
	unsigned char	alternate_addr[MAX_ADDR_LEN];
	unsigned char	cline_add_mod_not; /*called line addr modified*/
	/*
	 * The calling and called address of the call
	 */
	unsigned char	called_addr[MAX_ADDR_LEN];
	unsigned char	calling_addr[MAX_ADDR_LEN];
	/*
	 * Non X25 facilities starting with a marker 0x00
	 * or extended fcilities startin with code 0xff
	 */
	unsigned char	others_len;
	unsigned char	others[MAX_OTHER_FACILS];
	unsigned char	user_data[X25_MAX_USR_DATA];
	unsigned char	cause;
	unsigned char	diagnostic;
} X25_CLR_FACILS;

#define MAX_SETUP_PLEN (MAX_PARAM * 4)
/*
 *	Facilities coding
 */
#define	P_SIZE_NEG	0x8000	/* pckt size negotiation */
#define	W_SIZE_NEG	0x4000	/* window size negotiation */
#define	TROUGHPUT_NEG	0x2000	/* Troughput class negotiation */
#define	CUG_SEL		0x1000	/* Closed usr group selection */
#define	CUG_OUTG_ACC	0x0800	/* CUG with outgoing access */
#define	BILAT_CUG	0x0400	/* Bilatural CUG */
#define	REVERSE_CHARGE	0x0200	/* Reverse charging */
#define	FAST_SLCT_FACIL	0x0100	/* Fast select */
#define	NUI_SLCT	0x0080	/* NUI selection */
#define	CHARGING_INFO	0x0040	/* charging information req */
#define	RPOA_SLCT	0x0020	/* RPOA selection */
#define	CALL_DEF_SEL	0x0010	/*Call deflection selection */
#define	CALL_RD_NOT	0x0010	/*Call redirection/deflection notify*/
#define	CALLED_LAMN	0x0008	/* Called line addr modified */
#define	T_DELAY		0x0004	/* Transit delay */

/*
 * Chrging information receive
 */
#define	MONETARY_UNIT	0x80	/* Monetary unit */
#define	SEGMENT_CNT	0x40	/* Segment count */
#define	CALL_DUR	0x20	/* Call duration */

/*
 *	Coding of the facility code filed
 */
#define	P_SIZE_NEG_C	0x42	/* Pckt size neg code */
#define	W_SIZE_NEG_C	0x43	/* Window size neg code */
#define	TROUGHPUT_CODE	0x02	/* Troughput class negitiation */
#define	CUG_SEL_B	0x03	/* Closed user group basic */
#define	CUG_SEL_E	0x47	/* Closed user group extended */
#define	CUG_OUTG_B	0x09	/* CUG outgoing basic */
#define	CUG_OUTG_E	0x48	/* CUG outgoing extended */
#define	BILAT_CODE	0x41	/* Bilatural CUG 		*/
#define	REVERSE_FAST	0x01	/* Reverse charging + fast slct */
#define	NUI_SLCT_CODE	0xc6	/* NUI selection		*/
#define	INFO_CODE	0x04	/* charging information		*/
#define	RCV_INFO_MU	0xc5	/* Receive info momentary unit*/
#define	RCV_INFO_SC	0xc2	/* Receive info segment count */
#define	RCV_INFO_CD	0xc1	/* Receive info call duration */
#define	RPOA_B		0x44	/* RPOA selection basic		*/
#define	RPOA_E		0xc4	/* RPOA selection extended	*/
#define	CALL_DEFLCT_C	0xd1	/* Call deflection selection*/
#define	CALL_RD_NOTE_C	0xc3	/* Call red/def notification */
#define	CALLED_LAMN_C	0x08	/* Called line addr modified */
#define	T_DELAY_CODE	0x49	/* TRransmit delay		*/
#define	FACIL_MARKER	0x00	/* Marker			*/
#define	FACIL_RSVD	0xff	/* Reserved for extention	*/

/*
 * Coding of throughput classes
 */
#define	BS75		0x3	/* 75 bits per second		*/
#define	BS150		0x4	/* 150 bits per second		*/
#define	BS300		0x5	/* 300 bits per second		*/
#define	BS600		0x6	/* 600 bits per second		*/
#define	BS1200		0x7	/* 1200 bits per second		*/
#define	BS2400		0x8	/* 2400 bits per second		*/
#define	BS4800		0x9	/* 4800 bits per second		*/
#define	BS9600		0xa	/* 9600 bits per second		*/
#define	BS19200		0xb	/* 19200 bits per second	*/
#define	BS48000		0xc	/* 48000 bits per second	*/
#define	BS64000		0xd	/* 64000 bits per second	*/

/*
 * Coding of packet sizes
 */
#define	PCKT_SZ16	4	/* 16 bit packet size		*/
#define	PCKT_SZ32	5	/* 32 bit packet size		*/
#define	PCKT_SZ64	6	/* 64 bit packet size		*/
#define	PCKT_SZ128	7	/* 128 bit packet size		*/
#define	PCKT_SZ256	8	/* 256 bit packet size		*/
#define	PCKT_SZ512	9	/* 512 bit packet size		*/
#define	PCKT_SZ1024	10	/* 1024 bit packet size		*/
#define	PCKT_SZ2048	11	/* 2048 bit packet size		*/
#define	PCKT_SZ4096	12	/* 4096 bit packet size		*/
