; DAQ3000:  SCSI Data Acquisition Subsystem for the Apple Macintosh
;
; SCSIFast.Asm:  Fast SCSI read and write routines for pseudo DMA data
; transfers in real time.
;
;
; (C) John Eng    January 1989


scsiRd          EQU     $580000                 ; Hardware-specific equates for the
scsiWr          EQU     $580001                 ; NCR 5380 SCSI controller chip.
                                                ;
dackRd          EQU     $200                    ; These equates are for the Macintosh Plus
dackWr          EQU     $200                    ; and are not guaranteed to work on other models.

sCDR            EQU     $00                     ; Current Data Register
sODR            EQU     $00                     ; Output Data Register
sICR            EQU     $10                     ; Initiator Command Register
sMR             EQU     $20                     ; Mode Register
sTCR            EQU     $30                     ; Target Command Register
sCSR            EQU     $40                     ; Current Status Register
sSER            EQU     $40                     ; Select Enable Register
sBSR            EQU     $50                     ; Bus and Status Register
sDMAtx          EQU     $50                     ; Start DMA Send
sIDR            EQU     $60                     ; Input Data Register
sTDMArx         EQU     $60                     ; Start DMA Target Receive
sRESET          EQU     $70                     ; Reset Parity/Interrupt
sIDMArx         EQU     $70                     ; Start DMA Initiator Receive

RDMA            EQU     scsiRd+sIDR+dackRd      ; Symbols used here for the 5380 registers
SDMA            EQU     scsiWr+sODR+dackWr
RSCSI1          EQU     scsiRd+sICR
RSCSI4          EQU     scsiRd+sCSR
RSCSI5          EQU     scsiRd+sBSR
WSCSI1          EQU     scsiWr+sICR
WSCSI2          EQU     scsiWr+sMR
WSCSI3          EQU     scsiWr+sTCR
WSCSI5          EQU     scsiWr+sDMAtx
WSCSI7          EQU     scsiWr+sIDMArx

RETVAL          EQU     20                      ; Offset to Pascal return value
PARM1           EQU     16                      ; Offset to first Pascal parameter
PARM2           EQU     12                      ; Offset to second Pascal parameter
PARM3           EQU     8                       ; Offset to third Pascal parameter



; FUNCTION SCSIRFast(p: Ptr; nbytes: longint; laststat: Ptr): OSErr;
;
; This function reads from SCSI the number of bytes given by "nbytes"
; into a memory buffer pointed by "p".  Upon exiting, status byte and
; bus state information is returned in the word pointed by "laststat".

                XDEF    SCSIRFast

SCSIRFast       LINK    A6,#0                   ; Set up stack frame
                MOVE.W  SR,D0                   ; Get status register
                AND.L   #$00002000,D0           ; Look for supervisor mode
                BEQ     PRIVERR                 ; Branch to error if not supervisor
                MOVE.W  SR,-(SP)                ; Save status register on stack
                MOVE.W  #$2700,SR               ; Disable all nonmaskable interrupts

                MOVE.L  PARM1(A6),D0            ; Get first parameter
                AND.L   #$00FFFFFF,D0           ; Strip top 8 bits (24-bit address)
                MOVE.L  D0,A0                   ; Set up A0 as the pointer
                MOVE.L  PARM2(A6),A1            ; Get second parameter
                ADD.L   A0,A1                   ; Calculate target address

                MOVE.B  #%00000001,WSCSI3       ; Specify DATA IN bus phase
                MOVE.B  #%00000010,WSCSI2       ; Set DMA mode bit
                MOVE.B  D0,WSCSI7               ; Start initiator DMA receive
                MOVE.B  RSCSI1,D0               ; Get current initiator command register
                AND.B   #%10011110,D0           ; Mask out appropriate bits
                MOVE.B  D0,WSCSI1               ; Deassert data bus

RLOOP           MOVE.B  RSCSI5,D0               ; Get bus and status register
                BTST    #3,D0                   ; Test for phase mismatch
                BEQ     PHASERR                 ; Branch to error if mismatch exists
                BTST    #6,D0                   ; Test for DRQ
                BEQ     RLOOP                   ; Loop until DRQ
                MOVE.B  RDMA,(A0)               ; Receive and store data byte
                ADDQ.L  #1,A0                   ; Increment address pointer
                CMP.L   A1,A0                   ; Test address pointer
                BLT     RLOOP                   ; Loop until done

                MOVEQ   #0,D2                   ; No error
                BRA     DONE                    ; Go cleanup
PHASERR         MOVEQ   #5,D2                   ; Phase mismatch error
DONE            MOVE.B  RSCSI4,D1               ; Get current bus state
                MOVE.B  #%00000000,WSCSI2       ; Clear DMA mode
                MOVE.L  PARM3(A6),A0            ; Get pointer to status return
                MOVE.B  D0,(A0)+                ; Store last status register
                MOVE.B  D1,(A0)                 ; Store current bus state
                MOVE.W  D2,RETVAL(A6)           ; Store return value

                MOVE.W  (SP)+,SR                ; Restore status register
DONE2           UNLK    A6                      ; Restore A6
                MOVE.L  (SP)+,A0                ; Get return address
                ADD.L   #12,SP                  ; Release parameter space
                JMP     (A0)                    ; Return to calling routine

PRIVERR         MOVE.W  #7,RETVAL(A6)           ; Return error for privilege violation
                BRA     DONE2                   ; Go clean up



; FUNCTION SCSIWFast(p: Ptr; nbytes: longint; laststat: Ptr): OSErr;
;
; This function writes to SCSI the number of bytes given by "nbytes"
; into a memory buffer pointed by "p".  Upon exiting, status byte and
; bus state information is returned in the word pointed by "laststat".

                XDEF    SCSIWFast

SCSIWFast       LINK    A6,#0                   ; Set up stack frame
                MOVE.W  SR,D0                   ; Get status register
                AND.L   #$00002000,D0           ; Look for supervisor mode
                BEQ     PRIVERR                 ; Branch to error if not supervisor
                MOVE.W  SR,-(SP)                ; Save status register on stack
                MOVE.W  #$2700,SR               ; Disable all nonmaskable interrupts

                MOVE.L  PARM1(A6),D0            ; Get first parameter
                AND.L   #$00FFFFFF,D0           ; Strip top 8 bits (24-bit address)
                MOVE.L  D0,A0                   ; Set up A0 as the pointer
                MOVE.L  PARM2(A6),A1            ; Get second parameter
                ADD.L   A0,A1                   ; Calculate target address

                MOVE.B  #%00000000,WSCSI3       ; Specify DATA OUT bus phase
                MOVE.B  #%00000010,WSCSI2       ; Set DMA mode bit
                MOVE.B  RSCSI1,D0               ; Get current initiator command register
                AND.B   #%10011110,D0           ; Mask out appropriate bits
                OR.B    #%00000001,D0           ; Mask in data bus assertion bit
                MOVE.B  D0,WSCSI1               ; Assert data bus
                MOVE.B  D0,WSCSI5               ; Start DMA send

WLOOP           MOVE.B  RSCSI5,D0               ; Get bus and status register
                BTST    #3,D0                   ; Test for phase mismatch
                BEQ     PHASERR                 ; Branch to error if mismatch exists
                BTST    #6,D0                   ; Test for DRQ
                BEQ     WLOOP                   ; Loop until DRQ
                MOVE.B  (A0),SDMA               ; Get and send data byte
                ADDQ.L  #1,A0                   ; Increment address pointer
                CMP.L   A1,A0                   ; Test address pointer
                BLT     WLOOP                   ; Loop until done

                MOVEQ   #0,D2                   ; No error
                BRA     DONE                    ; Go cleanup

                END
