******************************************************************************
*  This code was designed to demonstrate how the 24C01 could                 *
*  be interfaced to the 68HC05 microcontroller.  The interface uses 2 lines  *
*  from Port A (PA0 and PA1) to communicate.                                 *
*                                                                            *
*  The code shown demonstrates a 'random read' and 'byte write'.  The other  *
*  modes of operation can be created by expanding upon these routines.       *
*  Acknowledge polling is used to determine when the write cycle finishes.   *
*                                                                            *
*  The mainline of this program reads the data located at address 002dH and  *
*  then writes that data back to address 0041H.  This program has been tested*
*  using the 24C01.                                                          *
******************************************************************************

SCLBIT	EQU	  1 		BIT INDICATING PORTA SCL POSITION
SDABIT  EQU	  0		BIT INDICATING PORTA SDA POSITION
SDAOUT	EQU	$03		MAKES SDA AN OUTPUT IF STORED IN DDRA
SDAIN	EQU	$02		MAKES SDA AN INPUT IF STORED IN DDRA
DMASK	EQU	$80		USED TO MASK BIT TO SEND TO DUT

PORTA	EQU	$00		PORT A MEMORY LOCATION
DDRA	EQU	$04		PORT A DIRECTION REGISTER OFFSET

ADDR	EQU	$80		LOCATION FOR 2404 ADDRESS TO ACCESS
DATA	EQU	$82		LOCATION FOR 2404 DATA TRANSFERED
COUNT	EQU	$83		COUNTER LOCATION FOR LOOPING
COUNT2	EQU	$85		COUNTER LOCATION FOR ACK POLLING
TEMP1	EQU	$86		SCRATCH PAD REGISTER
TEMP2	EQU	$87		SCRATCH PAD REGISTER

****************************
* RESET VECTOR ENTRY POINT *
****************************

	ORG	$1FFE		RESET VECTOR ADDRESS TO PROGRAM ENTRY
	FDB	$0100		JUMP TO BEGINNING OF EXECUTABLE CODE

***********************
* PROGRAM ENTRY POINT *
***********************

	ORG	$0100  		BEGINNING OF EXECUTABLE CODE

BEGIN:  LDA	#$FF		MAKE PORTA ALL OUTPUTS
	STA	DDRA

	LDA	#$FF		MAKE PORTA ALL ONES
	STA	PORTA
	LDA	#$03		MAKE SDA AND SCL OUTPUTS
	STA	DDRA
	LDA	#$2D
	STA	ADDR
	JSR	RDBYT           READ DATA FROM ADDRESS 002DH
	LDA	#$41
	STA	ADDR
	JSR	WRBYT		WRITE DATA BACK TO ADDRESS 0041H
	JSR	ACKPOL		PERFORM ACK POLLING
	BRA	*		LOOP UNTIL RESET


**********************************************************************
* READ A BYTE "RANDOM READ SEQUENCE".  THE ADDRESS TO READ IS STORED *
*  IN ADDR.  THE DATA FROM THE DUT IS STORED IN DATA.                *
**********************************************************************

RDBYT:	JSR	START            READ A BYTE FROM THE ADDRESS INDICATED
	LDA	ADDR               IN 'ADDR'
	LSLA
	ORA	#$01
	STA	DATA
	JSR	OUTBYT           SEND SLAVE ADDRESS
	JSR	INBYT            READ DATA FROM 2404
	JSR	NACK             CLOCK WITHOUT ACKNOWLEDGE BY PROCESSOR
	JSR	STOP             SEND STOP COMMAND
	RTS

***********************************************************************
* WRITE A BYTE "BYTE WRITE SEQUENCE".  THE ADDRESS TO WRITE IS STORED *
*  IN ADDR.  THE DATA TO WRITE IS STORED IN DATA.                     *
***********************************************************************

WRBYT:	LDA	DATA             WRITE TO BYTE POINTED TO BY ADDR THE
	STA	TEMP1		   VALUE IN LOCATION 'DATA'
	JSR	START            SEND START COMMAND
	LDA	ADDR
	LSLA
	STA	DATA
	JSR	OUTBYT           SEND SLAVE ADDRESS
	JSR	NACK             GET ACKNOWLEDGE
	LDA	TEMP1
	STA	DATA
	JSR	OUTBYT           SEND WRITE DATA
	JSR	NACK             GET ACKNOWLEDGE
	JSR	STOP             SEND STOP
	RTS

*****************************************************************
* READ 8 BITS FROM THE DUT.  THE RESULTS ARE RETURNED IN DATA.  *
*****************************************************************

INBYT:  LDA	#SDAIN		 MAKE SDA AN INPUT
	STA	DDRA
	JSR	CLOCK		 GET ACK
	LDA	#$08             PREPARE TO SHIFT IN 8 BITS
	STA	COUNT
LOOPI:	JSR	CLOCK            CLOCK DATA
	RORA
	ROL	DATA
	DEC	COUNT
	BNE	LOOPI            LOOP UNTIL 8 BITS ARE READ
	RTS

***********************************************************************
* WRITE 8 BITS TO THE DUT.  THE DATA TO SEND IS IN DATA.  IF THE LAST *
* BIT TO SEND IS A ONE THE SDA LINE IS MADE AN INPUT BEFORE THE EIGHTH*
* CLOCK PULSE TO AVOID BUS CONTENTION WHEN THE DUT ACKNOWLEDGES.  THE *
* ROUTINE FINISHES WITH SDA IN AN INPUT STATE.                        *
***********************************************************************

OUTBYT:	LDA	#$08             PREPARE TO SHIFT OUT 8 BITS
	STA	COUNT
	BCLR	#SDABIT,PORTA 	 MAKE SDA A 0
LOOPO:	LDA	DATA
	AND     #DMASK           IS THE DATA TO BE SHIFTED A 1 OR A 0
	BEQ	IS0              JUMP IF DATA SHOULD BE 0
	BSET	#SDABIT,PORTA 	 MAKE SDA A 1
	LDA	COUNT		 CHECK TO SEE IF LAST BIT TO SEND
	CMP	#$01
	BNE	IS1
	LDA	#SDAIN		 MAKE SDA AN INPUT IF A 1 IS THE LAST BIT
	STA	DDRA
	BRA	IS1
IS0:    BCLR	#SDABIT,PORTA 	 MAKE SDA A 0
IS1:	JSR	CLOCK            SEND CLOCK SIGNAL
	LSL	DATA
	DEC	COUNT
	BNE	LOOPO            LOOP UNTIL ALL 8 BITS HAVE BEEN SENT
	LDA	#SDAIN 		 MAKE SDA AN INPUT
	STA	DDRA
	RTS

************************************************************************
* PERFORM ACKNOWLEDGE POLLING TO DETERMINE WHEN THE WRITE CYCLE        *
* COMPLETES.  UPON RETURN FROM THIS ROUTINE THE A REGISTER INDICATES   *
* WHETHER THE DUT EVER ACKNOWLEDGED THE WRITE.  A=0 PART ACKNOWLEDGED, *
* A=1 NO ACKNOWLEDGE RECEIVED.                                         *
************************************************************************

ACKPOL:	LDA	#$07F		 MAX NUMBER OF TIMES TO CHECK THE PART
	STA	COUNT2
AKLOOP:	DEC	COUNT2           RETURN IF THE PART
	BEQ	OUTACK		  NEVER ISSUES AN ACKNOWLEDGE
	JSR	START            SEND START COMMAND
	LDA	#$00
	STA	DATA
	JSR	OUTBYT           SEND SLAVE ADDRESS
	JSR	NACK             SEE IF PART ACKNOWLEDGES
	CMP	#$00
	BNE	AKLOOP		 LOOP IF NO ACKNOWLEDGE
OUTACK: STA	TEMP2
	JSR	STOP             SEND STOP
	LDA	TEMP2
	RTS

************************
* ISSUE A STOP COMMAND *
************************

STOP:	BCLR	#SDABIT,PORTA 	  MAKE SURE SDA IS LOW
	BSET	#SCLBIT,PORTA 	  BRING SCL HIGH
	NOP		 	  PROVIDE SET-UP TIME
	NOP
	NOP
	NOP
	BSET	#SDABIT,PORTA 	  BRING SDA HIGH
	RTS

*************************
* ISSUE A START COMMAND *
*************************

START:  BSET	#SDABIT,PORTA     MAKE SURE THAT SDA IS HIGH
	BSET	#SCLBIT,PORTA 	  MAKE SURE THAT SCL IS HIGH
	BCLR	#SDABIT,PORTA     FORCE SDA LOW
	NOP			  PROVIDE SET-UP TIME
	NOP
	NOP
	NOP
	BCLR	#SCLBIT,PORTA 	  FORCE SCL LOW
	RTS

*********************************************************
* ISSUE AN ACKNOWLEDGE.  THE ACK ROUTINE DOES NOT CHECK *
*  TO SEE IF THE DUT ACTUALLY ISSUES AN ACKNOWLEDGE.    *
*********************************************************

ACK:    LDA	#SDAOUT		 MAKE SDA AN OUTPUT
	STA	DDRA
	BCLR	#SDABIT,PORTA 	 PERFORM AN 'ACKNOWLEDGE' WITH SDA LOW
	JSR	CLOCK            GENERATE A CLOCK PULSE
	RTS

**********************************************
* CLOCK IN A 1 TO THE DUT.  THIS ROUTINE IS  *
*  CALLED WHEN A READ SEQUENCE HAS FINISHED. *
**********************************************

NACK:   LDA	#SDAIN		 MAKE SDA AN INPUT (HI)
	STA	DDRA
	JSR	CLOCK            GENERATE A CLOCK PULSE
	STA	TEMP2
	LDA	#SDAOUT		 MAKE SDA AN OUTPUT
	STA	DDRA
	LDA	TEMP2
	RTS

******************************************************************
* ISSUE A CLOCK PULSE.  WHILE THE CLOCK IS HIGH THE VALUE ON THE *
*  SDA LINE IS PLACED IN THE CARRY FLAG.  WHEN A READ IS TAKING  *
*  PLACE THE CARRY FLAG WILL INDICATE THE VALUE FROM THE DUT.    *
******************************************************************

CLOCK:  BSET	#SCLBIT,PORTA 	 PROVIDE A CLOCK ON SCL, START HIGH
	NOP
	LDA	PORTA		 READ SDA WHILE SCL IS HIGH
	BCLR	#SCLBIT,PORTA
	AND	#$01             SDA VALUE IS IN LOWER BIT OF A REG
	RTS
