;*****************************************************************************
;  This code was designed to demonstrate how the 2404 family of parts could  *
;  be interface to the 8031 microcontroller.  The interface uses 2 lines     *
;  from Port 1 (P1.0 and P1.1) to communicate.  Other IIC compatible parts   *
;  can be added to the bus as long as they do not have $A as their device    *
;  identifier.  The routines RDBYT and WRBYT are tailored specifically to the*
;  2404 family.  The routines START, STOP, ACK, NACK, OUTBYT, and INBYT can  *
;  be considered generic IIC routines.                                       *
;                                                                            *
;  The code shown demonstrates a 'random read' and 'byte write'.  The other  *
;  modes of operation can be created by expanding upon these routines.       *
;  Acknowledge polling is used to determine when the write cycle is complete.*
;                                                                            *
;  This code will work with all Xicor IIC compatible EEPROMs reguardless of  *
;  their size.  As long as the address pins are configured correctly this    *
;  code will not know the difference between a bus with a single 24C16 and a *
;  bus with eight 2402s.                                                     *
;                                                                            *
;  The mainline of this program reads the data located at address 002DH and  *
;  then writes that data back to address 0041H.  This program has been tested*
;  using the 2402, 2404, 24C04, and 24C16.                                   *
;*****************************************************************************

SDA	BIT	P1.0		; THE SDA BIT IS PORT 1 BIT 0
SCL	BIT	P1.1		; THE SCL BIT IS PORT 1 BIT 1
ADDR	EQU	40H             ; LOCATION FOR 2404 ADDRESS TO ACCESS
COUNT	EQU	R0		; COUNTER VARIABLE
DATA1	EQU	R1              ; LOCATION FOR 2404 DATA TRANSFERED
TEMP	EQU	R2		; SCRATCH PAD BYTE
COUNT2	EQU	R3		; COUNTER FOR ACK POLLING

;***************************
; RESET VECTOR ENTRY POINT *
;***************************

	ORG	0000H           ; RESET VECTOR LOCATION
	LJMP    BEGIN		; JUMP TO BEGINNING OF PROGRAM

;**********************
; PROGRAM ENTRY POINT *
;**********************

	ORG	0100H		; PROGRAM CODE STARTS AT 0100H
BEGIN:	MOV	SP,#60H         ; INITIALIZE STACK POINTER
	MOV	DPTR,#002DH	; PREPARE TO READ ADDRESS #002DH
	MOV	ADDR,DPH
	MOV	ADDR+1,DPL
	LCALL	RDBYT           ; READ DATA FROM ADDRESS 002DH
	MOV	DPTR,#0041H	; PREPARE TO WRITE VALUE STORED IN
	MOV	ADDR,DPH	;   DATA1 TO DUT LOCATION #0041H
	MOV	ADDR+1,DPL
	LCALL	WRBYT		; WRITE DATA TO ADDRESS #0041H
	LCALL	ACKPOL		; USE ACKNOWLEDGE POLLING
DONE:	LJMP	DONE            ; LOOP UNTIL RESET

;*********************************************************************
; READ A BYTE "RANDOM READ SEQUENCE".  THE ADDRESS TO READ IS STORED *
;  IN ADDR.  THE DATA FROM THE DUT IS STORED IN DATA1.               *
;*********************************************************************

RDBYT:	LCALL	START           ; READ A BYTE FROM THE ADDRESS INDICATED
	MOV 	A,ADDR          ;  IN 'ADDR'
	CLR	C
	RLC	A
	ORL	A,#0A0H         ; BUILD SLAVE ADDRESS FOR WRITE
	MOV	TEMP,A
	MOV	DATA1,A
	LCALL	OUTBYT          ; SEND SLAVE ADDRESS
	LCALL	ACK             ; SEND ACKNOWLEDGE
	MOV     A,ADDR+1
	MOV	DATA1,A
	LCALL	OUTBYT          ; SEND WORD ADDRESS
	LCALL	ACK             ; SEND ACKNOWLEDGE
	LCALL	START           ; SEND START COMMAND
	MOV	A,TEMP          ; BUILD SLAVE ADDRESS FOR READ
	ORL	A,#01H          ; R/W BIT = 1
	MOV	DATA1,A
	LCALL	OUTBYT          ; SEND SLAVE ADDRESS
	LCALL	ACK             ; SEND ACKNOWLEDGE
	LCALL	INBYT           ; READ DATA FROM 2404
	LCALL	NACK            ; CLOCK WITHOUT ACKNOWLEDGE
	LCALL	STOP            ; SEND STOP COMMAND
	RET

;**********************************************************************
; WRITE A BYTE "BYTE WRITE SEQUENCE".  THE ADDRESS TO WRITE IS STORED *
;  IN ADDR.  THE DATA TO WRITE IS STORED IN DATA1.                    *
;**********************************************************************

WRBYT:	MOV	A,DATA1         ; WRITE TO BYTE POINTED TO BY ADDR THE
	MOV	TEMP,A		; VALUE IN LOCATION 'DATA'
	LCALL	START           ; SEND START COMMAND
	MOV	A,ADDR		; BUILD SLAVE ADDRESS FOR WRITE
	CLR	C
	RLC	A
	ORL	A,#0A0H
	MOV	DATA1,A
	LCALL	OUTBYT          ; SEND SLAVE ADDRESS
	LCALL	ACK             ; SEND ACKNOWLEDGE
	MOV	A,ADDR+1
	MOV	DATA1,A
	LCALL	OUTBYT          ; SEND WORD ADDRESS
	LCALL	ACK             ; SEND ACKNOWLEDGE
	MOV	A,TEMP
	MOV	DATA1,A
	LCALL	OUTBYT          ; SEND WRITE DATA
	LCALL	ACK             ; SEND ACKNOWLEDGE
	LCALL	STOP            ; SEND STOP
	RET

;***************************************************************
; READ 8 BITS FROM THE DUT. THE RESULTS ARE RETURNED IN DATA1. *
;***************************************************************

INBYT:	SETB	SDA             ; READ 8 BITS, MAKE SDA AN INPUT
	MOV	COUNT,#08H
LOOPI:	LCALL	CLOCK           ; CLOCK DATA
	MOV	A,DATA1		; BUILD BYTE USING DATA1
	RLC	A               ; ROLL IN NEXT BIT FROM DUT
	MOV	DATA1,A		; SAVE DATA
	DJNZ	COUNT,LOOPI     ; LOOP UNTIL 8 BITS ARE READ
	RET

;*********************************************************
; WRITE 8 BITS TO THE DUT. THE DATA TO SEND IS IN DATA1. *
;*********************************************************

OUTBYT:	MOV	COUNT,#08H      ; PREPARE TO SHIFT OUT 8 BITS
LOOPO:	MOV	A,DATA1		; LOAD DATA TO BE SENT TO DUT
	RLC	A		; ROTATE BIT TO SEND INTO CARRY FLAG
	MOV	SDA,C		; SEND CARRY TO SDA
	MOV	DATA1,A		; SAVE ROTATED DATA
	LCALL	CLOCK           ; SEND CLOCK SIGNAL TO DUT
	DJNZ	COUNT,LOOPO     ; LOOP UNTIL ALL 8 BITS HAVE BEEN SENT
	RET

;*********************************************************************
; PERFORM ACKNOWLEDGE POLLING TO DETERMINE WHEN THE WRITE CYCLE      *
;  COMPLETES.  UPON RETURN FROM THIS ROUTINE THE CARRY BIT INDICATES *
;  WHETHER THE DUT EVER ACKNOWLEDGED THE WRITE.  CARRY=0 PART        *
;  ACKNOWLEDGED, CARRY=1 NO ACKNOWLEDGE RECEIVED.                    *
;*********************************************************************

ACKPOL: MOV	COUNT2,#080H	; MAX NUMBER OF TIMES TO CHECK THE PART
AKLOOP: DJNZ	COUNT2,LOOK	; RETURN IF THE PART
	SJMP	OUTACK		;  NEVER ISSUES AN ACKNOWLEDGE.
LOOK:	LCALL	START           ; SET UP FOR A READ
	MOV	A,#0A0H         ; MAKE SLAVE ADDRESS FOR A READ
	MOV	DATA1,A
	LCALL	OUTBYT          ; SEND SLAVE ADDRESS
	LCALL	NACK            ; SEND ACKNOWLEDGE
	JC	AKLOOP		; LOOP IF NO ACKNOWLEDGE RECEIVED
OUTACK:	LCALL	STOP		; ISSUE A STOP BEFORE RETURNING
	RET

;***********************
; ISSUE A STOP COMMAND *
;***********************

STOP:   CLR	SDA             ; SEND STOP CONDITION TO DUT, SDA LOW
	SETB	SCL		; SCL HIGH
	NOP			; MAKE SUTE THE SET UP TIME IS VALID
	NOP
	NOP
	NOP
	SETB	SDA		; SDA HIGH
	RET

;************************
; ISSUE A START COMMAND *
;************************

START:  SETB	SDA		; SEND START CONDITION TO DUT, SDA HIGH
	SETB	SCL		; SCL HIGH
	NOP			; MAKE SURE SCL SET UP TIME IS VALID
	NOP
	NOP
	NOP
	CLR	SDA		; SDA LOW
	NOP			; MAKE SURE THE SET UP TIME IS VALID
	NOP
	NOP
	NOP
	CLR	SCL		; SCL LOW
	RET

;********************************************************
; ISSUE AN ACKNOWLEDGE.  THE ACK ROUTINE DOES NOT CHECK *
;  TO SEE IF THE DUT ACTUALLY ISSUES AN ACKNOWLEDGE.    *
;********************************************************

ACK:	CLR	SDA             ; PERFORM AN ACKNOWLEDGE, SDA LOW
	LCALL	CLOCK           ; GENERATE A CLOCK PULSE
	RET

;*********************************************
; CLOCK IN A 1 TO THE DUT.  THIS ROUTINE IS  *
;  CALLED WHEN A READ SEQUENCE HAS FINISHED. *
;*********************************************

NACK:	SETB	SDA		; CLOCK A 1 INTO THE DUT, SDA HIGH
	LCALL	CLOCK           ; GENERATE A CLOCK PULSE
	RET

;*****************************************************************
; ISSUE A CLOCK PULSE.  WHILE THE CLOCK IS HIGH THE VALUE ON THE *
;  SDA LINE IS PLACED IN THE CARRY FLAG.  WHEN A READ IS TAKING  *
;  PLACE THE CARRY FLAG WILL INDICATE THE VALUE FROM THE DUT.    *
;*****************************************************************

CLOCK:  NOP			; MAKE SURE THE DATA SET UP TIME IS VALID
	SETB	SCL             ; GENERATE A CLOCK PULSE, SCL HIGH
	NOP			; MAKE SURE CLOCK HIGH TIME IS VALID
	NOP
	NOP
	MOV	C,SDA		; READ STATE ON SDA, SAVE IN CARRY FLAG
	CLR	SCL		; SCL LOW
	RET

	END
