******************************************************************************
*  This code was designed to demonstrate how the 2444 could be interfaced to *
*  the 68HC11 microcontroller.  The interface uses 3 lines from Port 1 (PD3, *
*  PD4, and PD5) to communicate.  The DI and DO pins on the 2444 are tied    *
*  together which allows 1 less port line to be used.                        *
*                                                                            *
*  The code shown demonstrates RCL, WREN, READ, WRITE, and STORE             *
*  instructions.  The remaining instructions (WRDS and SLEEP) can be issued  *
*  using the same routine as other non-data instructions.                    *
*                                                                            *
*  The program issues a sequence of instructions to read the contents of     *
*  address 5 and stores the same value in address 9.  The sequence of        *
*  instructions is as follows :                                              *
*    1. RCL        sets the previous recall latch                            *
*    2. WREN       sets the write enable latch                               *
*    3. READ       data from address 5 is read                               *
*    4. WRITE      the data read during step 3 is written to address 9       *
*    5. STO        the ram's contents is transfered to the EEPROM            *
*                                                                            *
*  Data transfer is performed with the most significant bit first.  During   *
*  the READ and WRITE instructions the data sequence is inverted from that   *
*  shown in the data book (D15 is shifted first).                            *
******************************************************************************

SKBIT	EQU	$08		 MASK INDICATING PORTD SK POSITION
CEBIT	EQU	$10		 MASK INDICATING PORTD CE POSITION
DIOBIT	EQU	$20		 MASK INDICATING PORTD DATA POSITION
DOUT	EQU	$38		 MASK TO MAKE DI/O AN OUTPUT
DIN	EQU	$18		 MASK TO MAKE DI/O AN INPUT

WRDS	EQU	$80              RESET WRITE ENABLE LATCH
STO	EQU	$81		 TRANSFERS FROM RAM TO EEPROM
SLEEP	EQU	$82		 PLACES PART INTO POWER DOWN MODE
WRITE	EQU	$83		 RAM WRITE
WREN	EQU	$84		 SET WRITE ENABLE LATCH
RCL	EQU	$85		 TRANSFERS FROM EEPROM TO RAM, RESETS
* 				   WRITE ENABLE LATCH
READ	EQU	$86		 RAM READ

DDRD	EQU	$09              DATA DIRECTION REGISTER FOR PORT D
PORTD	EQU	$08              ADDRESS FOR PORT D
ADDR	EQU	$80              LOCATION FOR 2444 ADDRESS TO ACCESS
INST	EQU	$81              INSTRUCTION FOR PART
RWDAT	EQU	$82              LOCATION FOR 2444 DATA TRANSFERED
COUNT	EQU	$84		 COUNTER VARIABLE

*********************************************
* RESET VECTOR TO BEGINNING OF PROGRAM CODE *
*********************************************

	ORG	$FFFE            RESET VECTOR TO PROGRAM ENTRY POINT
	FDB	$E000

******************************
* START OF PROGRAM EXECUTION *
******************************

	ORG	$E000            BEGINNING OF EXECUTABLE CODE

BEGIN:	LDS	#$00FF           INITIALIZE STACK POINTER
	LDX	#$1000		 INITIALIZE PAGE OFFSET LOCATION
	LDAA	#DOUT
	STAA	DDRD,X		 MAKE CE, SK, DI/O OUTPUTS
	LDAA	#$00
	STAA	PORTD,X          INITIALIZE CE, SK, DI/O TO ZEROS

	LDAA	#RCL		 PERFORM A RECALL TO SET
	STAA	INST		   THE RECALL LATCH
	JSR	CEHIGH
	JSR     OUTBYT
	JSR	CELOW

	LDAA	#WREN		 PERFORM A WRITE ENABLE TO SET
	STAA	INST		   THE WRITE ENABLE LATCH
	JSR	CEHIGH
	JSR     OUTBYT
	JSR	CELOW

	LDAA	#$05		 READ THE CONTENTS OF ADDRESS 5
	STAA	ADDR		   THE VALUE READ WILL BE IN STORED
	JSR	RDWRD		   IN RWDATA

	LDAA	#$09             WRITE THE DATA JUST READ INTO
	STAA	ADDR		   ADDRESS 9
	JSR	WRWRD

	LDAA	#STO		 PERFORM A STORE OPERATION
	STAA	INST
	JSR 	CEHIGH
	JSR	OUTBYT
	JSR 	CELOW

	BRA	*                LOOP UNTIL RESET

******************************************************
* WRITE THE WORD SPECIFIED IN RWDAT.  THE ADDRESS TO *
*  BE WRITTEN IS SPECIFIED IN ADDR.                  *
******************************************************

WRWRD:  JSR	CEHIGH		 WRITE VALUE IN RWDATA INTO LOCATION
	LDAA	ADDR		   SPECIFIED IN ADDR
	LSLA			 JUSTIFY ADDRESS IN INSTRUCTION
	LSLA
	LSLA
	ORAA    #WRITE		 MASK IN WRITE INSTRUCTION
	STAA	INST
	JSR     OUTBYT		 SEND WRITE INSTRUCTION TO DUT
	LDAA    RWDAT
	STAA	INST
	JSR	OUTBYT           SEND IN UPPER BYTE OF DATA
	LDAA	RWDAT+1
	STAA	INST
	JSR	OUTBYT		 SEND IN LOWER BYTE OF DATA
	JSR	CELOW
	RTS

*********************************************************
* READ THE WORD AT THE LOCATION SPECIFIED IN ADDR.  THE *
*   DATA READ WILL BE PLACED IN RWDAT.                  *
*********************************************************

RDWRD:  JSR	CEHIGH		 READ THE ADDRESS SPECIFIED IN ADDR
	LDAA	ADDR
	LSLA			 JUSTIFY ADDRESS TO READ
	LSLA
	LSLA
	ORAA    #READ		 MASK IN READ INSTRUCTION
	STAA	INST
	JSR	SEND7		 SEND IN 7 BITS OF READ INSTRUCTION
	LDAA	#DIN		 MAKE DATA LINE AN INPUT
	STAA	DDRD,X
	JSR	CLOCK		 SEND EIGHTH CLOCK PULSE FOR READ INSTRUCTION
	LDAA	#$10		 PREPARE TO SHIFT IN 16 BITS
	STAA	COUNT
BITX:   CLC			 ASSUME BIT IS GOING TO BE A ZERO (CLEAR CARRY)
	LDAA	PORTD,X		 READ BIT VALUE
	ANDA	#DIOBIT		 MASK BIT OUT OF BYTE READ
	BEQ	NO1		 LEAVE CARRY FLAG ALONE IF BIT IS A 0
	SEC			 SET CARRY IF BIT IS A 1
NO1:    ROL	RWDAT+1		 ROLL CARRY FLAG INTO DATA WORD
	ROL	RWDAT
	JSR	CLOCK		 SEND A CLOCK PULSE
	DEC	COUNT		 LOOP UNTIL
	BNE	BITX		   16 BITS ARE READ
	LDAA	#DOUT		 MAKE DATA LINE AN OUTPUT
	STAA	DDRD,X
	JSR	CELOW		 BRING CE LOW
	RTS

******************************************************
* SEND DATA OUT TO THE PART.  THE DATA TO BE SENT IS *
*  LOCATED IN INST.                                  *
******************************************************

SEND7:	LDAA	#$07		 SHIFT OUT 7 BITS FOR READ INSTRUCTION
	STAA	COUNT
	BRA	LOOPO
OUTBYT:	LDAA	#$08             PREPARE TO SHIFT OUT 8 BITS
	STAA	COUNT
LOOPO:	ROL	INST
	BCC	IS0              JUMP IF DATA SHOULD BE 0
	BSET	PORTD,X DIOBIT	 SEND 1 TO DI/O
	BRA	IS1
IS0:	BCLR	PORTD,X DIOBIT	 SEND 0 TO DI/O
IS1:	JSR	CLOCK            SEND CLOCK SIGNAL
	DEC	COUNT
	BNE	LOOPO            LOOP UNTIL ALL 8 BITS HAVE BEEN SENT
	RTS

*****************
* BRING CE HIGH *
*****************

CEHIGH:	BSET	PORTD,X #CEBIT	 BRING CE HIGH
	RTS

****************
* BRING CE LOW *
****************

CELOW:  BCLR	PORTD,X DIOBIT	 BRING DATA LINE LOW
	BCLR	PORTD,X #CEBIT	 BRING CE LOW
	RTS

************************
* ISSUE A CLOCK PULSE. *
************************

CLOCK:  BSET	PORTD,X SKBIT	 BRING SK HIGH
	BCLR	PORTD,X SKBIT	 BRING SK LOW
	RTS

