;*****************************************************************************
;  This code was designed to demonstrate how the 2444 could be interfaced to *
;  the Z8 microcontroller.  The interface uses 3 lines from Port 2 (P27,     *
;  P26, and P25) to communicate.  The DI and DO pins on the 2444 are tied    *
;  together which allows 1 less port line to be used.                        *
;                                                                            *
;  The code shown demonstrates RCL, WREN, READ, WRITE, and STORE             *
;  instructions.  The remaining instructions (WRDS and SLEEP) can be issued  *
;  using the same routine as other non-data instructions.                    *
;                                                                            *
;  The program issues a sequence of instructions to read the contents of     *
;  address 5 and stores the same value in address 9.  The sequence of        *
;  instructions is as follows :                                              *
;    1. RCL        sets the previous recall latch                            *
;    2. WREN       sets the write enable latch                               *
;    3. READ       data from address 5 is read                               *
;    4. WRITE      the data read during step 3 is written to address 9       *
;    5. STO        the ram's contents is transfered to the EEPROM            *
;                                                                            *
;  Data transfer is performed with the most significant bit first.  During   *
;  the READ and WRITE instructions the data sequence is inverted from that   *
;  shown in the data book (D15 is shifted first).                            *
;*****************************************************************************

SKHI	.EQU	020H		; SK GENERATED ON BIT 5 OF PORT 1
CEHI	.EQU    040H		; CE GENERATED ON BIT 6 OF PORT 1
DIOHI	.EQU    080H		; DI AND DO GENERATED ON BIT 7 OF PORT 1
SKLO	.EQU	0DFH		; SK GENERATED ON BIT 5 OF PORT 1
CELO	.EQU    0BFH		; CE GENERATED ON BIT 6 OF PORT 1
DIOLO	.EQU    07FH		; DI AND DO GENERATED ON BIT 7 OF PORT 1

WRDS	.EQU	080H 		; RESET WRITE ENABLE LATCH
STO	.EQU	081H		; TRANSFERS FROM RAM TO EEPROM
SLEEP	.EQU	082H		; PLACES PART INTO POWER DOWN MODE
WRITE	.EQU	083H		; RAM WRITE
WREN	.EQU	084H		; SET WRITE ENABLE LATCH
RCL	.EQU	085H		; TRANSFERS FROM EEPROM TO RAM, RESETS
				;   WRITE ENABLE LATCH
READ	.EQU	086H		; RAM READ

DATAHI	.EQU	R4		; REGISTER WHERE HIGH ORDER DATA IS STORED
DATALO	.EQU	R5		; REGISTER WHERE LOW ORDER DATA IS STORED
ADDR	.EQU	R6		; REGISTER WHERE THE PART ADDRESS IS STORED
COUNT	.EQU	R7		; LOOP COUNT VARIABLE
INST	.EQU	R8		; REGISTER WHERE THE INSTRUCTION IS STORED
TEMP	.EQU	R9		; SCRATCH PAD REGISTER

;*********************************
; START OF EXECUTION AFTER RESET *
;*********************************

RESETV:	.ORG	0FF0CH		; RESET VECTOR LOCATION
	JP  	0FF0FH		; CHANGE PROGRAM COUNTER FROM 00XXH TO FFXXH

;***********************
; INITIALIZE PROCESSOR *
;***********************

INITL:	.ORG	0FF0FH		; INITIALIZE PROCESSOR
	LD	P0,#0FFH	; MAKE PORT0 (ADD 8-15) 0FFH
	LD	P01M,#96H	; MAKE PORT0 UPPER ORDER ADDRESS
	SRP	#00H		; REGISTER POINTER TO FIRST BANK
	LD	P2,#00H		; INITIALIZE PORT 2 TO 000H
	LD	P2M,#00H	; MAKE PORT 2 AN OUTPUT REGISTER
	LD	P3M,#01H	; MAKE PORT2 OUTPUTS HAVE ACTIVE HIGHS
	LD	SPL,#7FH	; SET STACK POINTER TO 007FH
	LD	SPH,#00H
	JP	BEGIN1		; JUMP TO START OF PROGRAM

;****************************
; BEGINNING OF PROGRAM CODE *
;****************************

BEGIN1:	.ORG	0E000H

	LD	INST,#RCL	; ENABLE THE RECALL LATCH
	CALL	CEHIGH          ; BRING CE HIGH
	CALL   	OUTBYT		; SEND INSTRUCTION TO DUT
	CALL	CELOW		; BRING CE BACK LOW

	LD	INST,#WREN	; SET THE WRITE ENABLE LATCH
	CALL	CEHIGH
	CALL   	OUTBYT
	CALL	CELOW

	LD	ADDR,#05H	; READ ADDRESS 5, STORE THE DATA
	CALL	RDWRD		;   IN DATALO AND DATAHI

	LD	ADDR,#09H 	; WRITE THE DATA IN DATALO AND DATAHI
	CALL	WRWRD		;   INTO ADDRESS 9

	LD	INST,#STO	; PERFORM A STORE OPERATION
	CALL	CEHIGH
	CALL	OUTBYT
	CALL	CELOW

DONE:	JP	DONE            ; LOOP UNTIL RESET

;***************************************************************
; WRITE THE WORD SPECIFIED IN DATAHI AND DATALO.  THE ADDRESS  *
; TO BE WRITTEN IS SPECIFIED IN ADDR.                          *
;***************************************************************

WRWRD:  CALL	CEHIGH		; ENABLE THE DUT
	LD	INST,ADDR	; LOAD THE ADDRESS TO WRITE
	RL	INST		; JUSTIFY ADDRESS IN INSTRUCTION
	RL	INST
	RL	INST
	OR	INST,#WRITE	; BUILD WRITE INSTRUCTION
	CALL	OUTBYT		; SEND WRITE INSTRUCTION TO DUT
	LD	INST,DATAHI
	CALL	OUTBYT          ; SEND IN UPPER BYTE OF DATA
	LD	INST,DATALO
	CALL	OUTBYT		; SEND IN LOWER BYTE OF DATA
	CALL	CELOW		; DISABLE THE DUT
	RET

;***********************************************************
; READ THE WORD ADDRESS SPECIFIED IN ADDR.  THE DATA WILL  *
; BE STORED IN DATALO AND DATAHI.                          *
;***********************************************************

RDWRD:  CALL	CEHIGH		; READ THE ADDRESS SPECIFIED IN ADDR
	LD	INST,ADDR
	RL	INST		; JUSTIFY ADDRESS TO READ
	RL	INST
	RL	INST
	OR     	INST,#READ	; MASK IN READ INSTRUCTION
	CALL	SEND7		; SEND IN 7 BITS OF READ INSTRUCTION
	LD	P2M,#DIOHI	; MAKE THE DATA LINE AN INPUT
	CALL	CLOCK		; SEND EIGHTH CLOCK PULSE FOR READ INSTRUCTION
	LD	COUNT,#10H	; PREPARE TO SHIFT IN 16 BITS
BITX:   LD	TEMP,P2		; READ VALUE ON DO
	RLC	TEMP
NO1:   	RLC	DATALO		;   AND DATAHI
	RLC	DATAHI
	CALL	CLOCK		; SEND A CLOCK PULSE
	DJNZ	COUNT,BITX	; LOOP WHILE READING DATA
	LD	P2M,#00H	; MAKE THE DATA LINE AN OUTPUT
	CALL	CELOW		; DISABLE THE PART
	RET

;*****************************************************
; SEND DATA OUT TO THE PART.  THE DATA TO BE SENT IS *
;   LOCATED IN INST.                                 *
;*****************************************************

SEND7:	LD	COUNT,#07H	; SHIFT OUT 7 BITS FOR READ INSTRUCTION
	JP	LOOPO
OUTBYT:	LD	COUNT,#08H      ; PREPARE TO SHIFT OUT 8 BITS
LOOPO:	RLC	INST		; FIND DATA TO SHIFT OUT TO DI
	JP	C,SEND1		; SEND THE DATA OUT ON DI
	AND	P2,#DIOLO
	JP	ENDBIT
SEND1:	OR	P2,#DIOHI
ENDBIT:	CALL	CLOCK           ; SEND CLOCK SIGNAL
	DJNZ	COUNT,LOOPO     ; LOOP WHILE STILL SHIFTING OUT DATA
	RET

;****************
; BRING CE HIGH *
;****************

CEHIGH: OR	P2,#CEHI	; BRING CE HIGH
	RET

;***************
; BRING CE LOW *
;***************

CELOW:  AND	P2,#CELO	; BRING CE LOW
	RET

;***********************
; ISSUE A CLOCK PULSE. *
;***********************

CLOCK:	OR	P2,#SKHI	; SEND IN CLOCK PULSE (BRING SK HIGH)
	AND	P2,#SKLO	; BRING SK LOW
	RET

.END
