/*3456789012345678901234567890123456789012345678*/
/*
 * checkfat.c vom 26.05.1998
 * 
 * Autor:
 * Thomas Binder
 * (gryf@hrzpub.tu-darmstadt.de)
 *
 * Zweck:
 * FAT-Testprogramm fr den Aufruf in einer Shell
 * oder (mit kleinem Hilfsprogramm) aus dem Auto-
 * Ordner. Findet falsche Dateilngen, illegale
 * Start- und Folgecluster, Clusterschleifen,
 * Ordnerschleifen, Files mit als defekt
 * markierten Clustern, Cluster mit mehreren
 * Vorgngern, falsche Clustereintrge, verwaiste
 * Cluster und mehrfach belegte Cluster. Ebenso
 * werden alle als defekt markierten Cluster
 * gefunden. Mittels Kommandozeilenoptionen kann
 * die Ausgabe aller Filenamen, zustzlich die
 * Ausgabe smtlicher Cluster jedes Files sowie
 * eine detailierte Ergebnis-Ausgabe eingeschaltet
 * werden.
 * Optional kann CheckFat auch Fehler in den fr
 * VFAT ntigen Zusatzeintrgen finden, die z.B.
 * leicht dann auftreten, wenn man das Medium auch
 * hufiger ohne VFAT beschreibt.
 * Wichtig: CheckFat ist nur fr 16-Bit-FATs
 * geeignet!
 *
 * Vielen Dank auch an meine Betatester (in alpha-
 * betischer Reihenfolge):
 * Alexander Clauss, Dirk Klemmt, Rainer Riedl,
 * Michael Schwingen, Uwe Seimet, Manfred Ssykor
 * und Arno Welzel.
 *
 * History:
 * Irgendwann 1993: Erstellung
 * 23.03.1995: Deutliche Verbesserung. CheckFat
 *             findet jetzt auch Clusterschleifen,
 *             illegale Folgecluster und ist auf
 *             die Steuerung der Ausgabe ber
 *             Kommandozeilenoptionen vorbereitet.
 * 24.03.1995: CheckFat findet jetzt auch
 *             Ordnerschleifen und Cluster mit
 *             mehreren Vorgngern.
 * 26.03.1995: Da neuere GEMDOS-Versionen wirklich
 *             alle Datencluster belegen knnen,
 *             werden jetzt Clusternummern von 2
 *             bis numcl + 1 akzeptiert (bisher
 *             nur bis numcl - 1). Auerdem wurden
 *             jetzt die Steuerung per
 *             Kommandozeile integriert und einige
 *             interne Optimierungen vorgenommen.
 * 27.03.1995: Beginn der Kommentierung, dabei
 *             noch kleinere Optimierungen und
 *             Fehlerbeseitigungen.
 * 29.03.1995: "Fehler" in der Meldung bei Start
 *             ohne Parameter ausgebaut (fehlendes
 *             Newline). Auerdem darf die
 *             Laufwerksangabe jetzt auch mehr als
 *             einen Buchstaben lang sein, der
 *             Rest wird einfach ignoriert (damit
 *             kann beispielsweise auch c: oder
 *             c:\ bergeben werden)
 * 31.03.1995: Anpassungen an MiNTLib
 * 24.07.1995: Neue Kommandozeilenoption -h fr
 *             Tastendruck vor Programmende.
 * 10.10.1995: CheckFat merkt sich jetzt auch alle
 *             Dateinamen (auf Wunsch auch mit
 *             Pfad, dabei aber weitaus hherer
 *             Speicherbedarf!) und gibt bei
 *             mehrfach belegten Clustern die
 *             Dateien an (besser: Ist beim
 *             Scannen der Files ein Cluster schon
 *             von einer anderen Datei belegt,
 *             gibt CheckFat deren Namen aus. Wenn
 *             ein Cluster also viermal belegt
 *             ist, wird bei drei Dateien der
 *             Hinweis erscheinen, da der Cluster
 *             bereits von der letzten belegt
 *             ist.) Diese Ausgabe erfolgt nur,
 *             wenn sie per -x (siehe unten)
 *             angefordert wurde!
 *             In diesem Zusammenhang gibt es auch
 *             drei neue Kommandozeilenoptionen:
 *             -x meldet, wie gerade beschrieben,
 *                mehrfach belegte Cluster mit
 *                dem Dateinamen.
 *             -l merkt sich die kompletten Pfade
 *                (normal werden nur die Filenamen
 *                gespeichert).
 *             -a gibt bei mehrfach belegten
 *                Clustern diesen Umstand bei
 *                jedem beteiligten Cluster aus,
 *                whrend dies sonst nur beim
 *                ersten Mal passiert (die
 *                restlichen Cluster der Datei
 *                sind ja damit zwangsweise
 *                ebenfalls bereits belegt).
 * 11.10.1995: Fehler in work_dir behoben, der zum
 *             berlauf des scl-Arrays und damit
 *             zur berschreibung der Variablen
 *             drive fhrte. Auerdem wird jetzt
 *             zustzlich geprft, ob bei der
 *             Rekursion noch gengend Platz im
 *             my_path-Array ist.
 *             Darberhinaus stimmt jetzt auch der
 *             Returncode bei Fehlerabbruch, und
 *             fr die zu speichernden Filenamen
 *             (-x/-l) wird nicht mehr immer ein
 *             Byte zu wenig angefordert.
 *             Sicherheitsberprfungen fr den
 *             BPB.
 * 12.10.1995: Bei Unterverzeichnissen wird jetzt
 *             geprft, ob sie als erstes die
 *             Ordner "." und ".." enthalten. Wenn
 *             nicht, wird das Verzeichnis nicht
 *             bearbeitet. Ebenso werden falsche
 *             Eintrge fr den Startcluster
 *             dieser beiden Pseudoverzeichnisse
 *             gemeldet.
 *             Kleinen Fehler in der BPB-Prfung
 *             entfernt (der ersten Datencluster
 *             mu hinter dem Wurzelverzeicnis
 *             beginnen, nicht hinter der 2. FAT).
 * 13.10.1995: Freitag, der 13., und trotzdem habe
 *             ich heute erfahren, da ich meine
 *             letzte Vordiplomsklausur bestanden
 *             habe :)
 *             Unabhngig davon meldet CheckFat
 *             jetzt bei der Schluanalyse der FAT
 *             auch Cluster, die auf sich selbst
 *             zeigen.
 * 19.10.1995: Compilierung mit Memdebug, um zu
 *             testen, ob alle mallocs/frees in
 *             Ordnung sind. Sind sie zum Glck :)
 * 20.10.1995: Probleme mit Thing und TOSWIN
 *             (Parameter werden nicht erkannt),
 *             daher ein wenig Debug-Output.
 * 26.10.1995: Die Probleme lagen an Thing, der
 *             Debug-Output ist also wieder weg.
 *             Es gibt eine neue Kommandozeilen-
 *             Option -d, mit der die Ausgabe von
 *             als defekt markierten Clustern
 *             eingeschaltet wird. Bisher wurden
 *             sie immer ausgegeben und als Fehler
 *             gemeldet; das hat sich aber als
 *             unbrauchbar erwiesen, weil Cluster
 *             16383 bei greren Partionen wegen
 *             eines GEMDOS-Fehlers von den
 *             meisten Partitionierungsprogrammen
 *             als defekt markiert wird.
 *             Die Versionsnummer entfernt, weil
 *             sie ohnehin recht wenig aussagt.
 * 29.10.1995: Neue Option -u, bei der CheckFat
 *             nicht mehr Dlock benutzt.
 * 12.12.1995: Bei einem defekten Cluster 16383
 *             erfolgt jetzt die Meldung, da dies
 *             normalerweise eine Schutzmanahme
 *             gegen einen GEMDOS-Fehler ist.
 * 20.12.1995: CheckFat gibt jetzt bei Option -?
 *             die Kurzanleitung aus (also so, wie
 *             wie es auch bei Aufruf komplett
 *             ohne Parameter geschieht).
 * 21.12.1995: In der Kurzanleitung fehlte noch
 *             die gestern eingefhrte Option -?.
 * 15.01.1996: Letzte Vorbereitungen fr die
 *             Verffentlichung.
 * 18.02.1996: Neue Option -s fr eingeschrnkte
 *             "Gesprchigkeit" von CheckFat.
 * 27.02.1996: CheckFat meldet jetzt auch bei als
 *             defekt markiertem Cluster 16384,
 *             da dies meistens eine Vorsichts-
 *             manahme ist.
 * 01.03.1996: CheckFat macht es jetzt wirklich
 *             richtig und meldet die Vorsichts-
 *             manahme bei den beiden Clustern,
 *             die an der 32767-Sektoren-Grenze
 *             liegen.
 * 05.03.1996: Vergleich beider FATs und Meldung
 *             eventueller Untschiede. Durch die
 *             neue Option -1 kann man CheckFat
 *             anweisen, bei Unterschieden zum
 *             Test FAT 1 zu benutzen.
 *             Auerdem werden jetzt auch die
 *             Startcluster vor der berprfung
 *             auf Korrektheit ausgegeben.
 *             Files mit leerem Cluster sowie
 *             Verweise auf unbelegte Cluster
 *             werden jetzt auch bzw. gesondert
 *             gemeldet.
 *             Neue Option -f zur Ausgabe des
 *             Fragmentierungsgrades.
 * 06.03.1996: Ausfhrlichere Ausgabe des
 *             Fragmentierungsgrades.
 * 13.03.1996: Die BPB-berprfung testet jetzt
 *             auch, ob die Sektor- oder Cluster-
 *             gre 0 ist. Das kann z.B. dann
 *             passieren, wenn man eine Partition
 *             mit mehr als 1 GB erstellt hat.
 *             Sowas sollte ein Partionierer nur
 *             nach einer Sicherheitsabfrage
 *             erlauben, weil es unter GEMDOS
 *             nicht korrekt funktioniert.
 * 17.03.1996: Korrekte berprfung der FAT-Gre
 *             im BPB-Test. Auerdem wird jetzt
 *             eine eigene BPB-Struktur benutzt,
 *             die mit vorzeichenlosen Integern
 *             arbeitet (dadurch werden jetzt
 *             Laufwerke mit 32768 Bytes groen
 *             Clustern korrekt geprft).
 *             Workaround fr Partitionen, deren
 *             Cluster 65536 Bytes gro sind (das
 *             sind im BPB 0, da nur 16 Bit-Werte
 *             in der Struktur stehen!) Das GEMDOS
 *             des Falcon kommt damit zwar klar,
 *             ratsam ist es jedoch nicht.
 * 15.09.1996: Blden Fehler in work_dir entfernt:
 *             War fr das realloc kein Speicher
 *             mehr vorhanden, wurde das Einlesen
 *             nicht richtig abgebrochen; mit sehr
 *             blen Folgen, weil der nchste
 *             Cluster dann an oder kurz hinter
 *             Adresse 0 geladen wurde. Auerdem
 *             wurde der vor dem realloc belegte
 *             Speicher nicht wieder freigegeben.
 *             Dateinamen, die '?', '*', ':' oder
 *             ein Zeichen mit ASCII-Code < 32
 *             enthalten, werden jetzt gemeldet.
 * 04.04.1997: Bei Option -x wurde fr Ordner
 *             immer ein Byte zuwenig angefordert.
 * 06.04.1997: Bei -f werden Clustersprnge jetzt
 *             gekennzeichnet, wenn zustzlich -v
 *             sowie -c oder -n aktiv sind.
 * 11.01.1998: CheckFat sollte jetzt auch mit DOS-
 *             Partitionen (> 32766 Cluster)
 *             zurechtkommen, bentigt dafr aber
 *             einen AHDI-3.0-kompatiblen Treiber,
 *             da sonst nicht auf Sektoren grer
 *             65535 zugegriffen werden kann.
 *             Neue Option -i zur ausfhrlichen
 *             Anzeige des BPB.
 * 28.01.-
 * 30.01.1998: Neue Option -t zur berprfung auf
 *             VFAT-Strukturfehler
 * 08.05.1998: Nur neue Versionsnummer, nderung
 *             ist in unicode.h
 * 26.05.1998: VFAT-Prfung findet jetzt auch
 *             Filenamen, die nicht korrekt mit
 *             0 abgeschlossen sind bzw. vorher
 *             schon Fllbytes enthalten
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <mintbind.h>
#include <portab.h>
#ifdef MEMDEBUG
#include <memdebug.h>
#endif
#include "unicode.h"

/* Erweiterter Rwabs() fr 32-Bit-Sektornummer */
#define R_wabs(a, b, c, d, e, f) \
  bios(4, (WORD)a, (void *)b, (WORD)c, (WORD)d, \
  (WORD)e, (LONG)f)

/* Bentigte Strukturen und Konstanten */

#define VERSION   "vom 26.05.1998"

#define NOT16BIT  1
#define NOMEM     2
#define READERROR 3
#define NOBPB     4
#define TOODEEP   5
#define LOCKED    6
#define NOBIOS    7
#define WRONGBPB  8
#define TMFILES   9
#define NOAHDI    10

#define WRONG       0
#define ILLEGAL     1
#define LOOPS       2
#define DIRLOOPS    3
#define DESTROYED   4
#define EMPTYCLUST  5
#define AMBIGUOUS   6
#define ILLCL       7
#define DAMAGED     8
#define ORPHAN      9
#define MULTIPLE    10
#define ILLSUBDIR   11
#define ILLPSEUDO   12
#define SELFPOINTER 13
#define EMPTYPOINT  14
#define ILLNAME     15
#define ILLVFAT     16
#define VFATDIREND  17
#define VFATNOMATCH 18
#define NO_OF_TYPES 19

#define VFAT_ILLSEQ   1
#define VFAT_OUTOFSEQ 2
#define VFAT_INCOMPL  3
#define VFAT_ILLSCL   4
#define VFAT_CHKSUM   5

/*
 * Die maximale Rekursionstiefe bestimmt sich in
 * etwa durch [*_StkSize / (61 + PATHMAX)] - 4,
 * wobei man natrlich eine recht groe
 * Sicherheitsreserve fr den Stack lassen sollte
 */
#define MAXDEPTH  17

/*
 * Bestimmt, wie lang ein Pfad inklusive Nullbyte
 * maximal sein darf. Sollte diese Grenze whrend
 * der Rekursion innerhalb von work_dir
 * berschritten werden, wird CheckFat mit der
 * Meldung "Zu tiefe Ordnerverschachtelung"
 * abgebrochen.
 */
#define PATHMAX   129

#define MAXCLUST  (bpb->numcl + 1)

typedef struct
{
  char  dir_name[11];
  BYTE  dir_attr;
  BYTE  dir_dummy[10];
  UWORD dir_time;
  UWORD dir_date;
  UWORD dir_stcl;
  ULONG dir_flen;
} DIR;

typedef struct
{
  UBYTE seq;
  UBYTE unicode0_4[10];
  UBYTE attr;
  UBYTE res;
  UBYTE chksum;
  UBYTE unicode5_10[12];
  UBYTE scl[2];
  UBYTE unicode11_12[4];
} VFAT_DIR;

typedef struct
{
  UWORD recsiz;
  UWORD clsiz;
  UWORD clsizb;
  UWORD rdlen;
  UWORD fsiz;
  UWORD fatrec;
  UWORD datrec;
  UWORD numcl;
  UWORD bflags;
} _BPB_;

typedef struct
{
  WORD  puns;
  char  pun[16];
  LONG  part_start[16];
  LONG  P_cookie;
  LONG  *P_cookptr;
  UWORD P_version;
  UWORD P_max_sector;
  LONG  reserved[16];
} PUN_INFO;

/* Prototypen */

void leave(WORD retcode);
WORD eval_args(WORD argc, char *argv[]);
void usage(void);
void err(WORD code);
void correct_dir(DIR *dir, WORD entries);
void work_dir(DIR *dir, UWORD entries, char *path,
  UWORD *scl, WORD depth, WORD big);
void swap(UWORD *value);
void lswap(ULONG *value);
PUN_INFO *getpun(void);
LONG pun_ptr(void);
void printbpb(void);
void wrongbpb(void);
void vfat_check(char *path, DIR *dir, UWORD start,
  UWORD max, UWORD *newi);
char uni2ascii(UBYTE off, UBYTE cp);

/* Globale Variablen */

char  *errtext[] = {
        "Kein Fehler!",
        "Medium hat keine 16-Bit-Fat!",
        "Kein Speicher mehr frei!",
        "Lesefehler!",
        "Bios-Parameterblock unlesbar!",
        "Zu tiefe Ordnerverschachtelung!",
        "Laufwerk gesperrt!",
        "Kein BIOS-Laufwerk!",
        "Bios-Parameterblock fehlerhaft!",
        "Zu viele Files!?",
        "AHDI-3.0-kompatibler Treiber bentigt!"},
      *reporttext[NO_OF_TYPES] = {
        "%d unterschiedliche Dateilnge(n)!\n",
        "%d File(s)/Ordner mit illegalem Start/"
          "Folgecluster!\n",
        "%d File(s)/Ordner mit "
          "Clusterschleife!\n",
        "%d Ordnerschleife(n)!\n",
        "%d File(s)/Ordner mit defektem "
          "Cluster!\n",
        "%d File(s)/Ordner mit leerem Cluster!\n",
        "%d Cluster ohne eindeutigen "
          "Vorgnger!\n",
        "%d illegale(r) Folgecluster!\n",
        "%d als defekt markierte(r) Cluster!\n",
        "%d verwaiste(r) Cluster!\n",
        "%d mehrfach belegte(r) Cluster!\n",
        "%d Verzeichnis(se) ohne '.' und/oder "
          "'..'!\n",
        "%d falsche(r) Startcluster bei '.' "
          "und/oder '..'!\n",
        "%d auf sich selbst zeigende(r) "
          "Cluster!\n",
        "%d Verweis(e) auf leere(n) Cluster!\n",
        "%d Dateiname(n) mit illegalen "
          "Zeichen!\n",
        "%d ungltige(r) VFAT-Name(n)!\n",
        "%d unvollstndige(r) VFAT-Name(n)!\n",
        "%d falsch zugeordnete(r) VFAT-Name(n)!\n"},
      *filenames[65536L];
UWORD filenr,
      fragments,
      fragmented,
      min_dist,
      max_dist,
      *fat,
      *checkfat,
      *tempcheck,
      startclusters[MAXDEPTH];
WORD  drive,
      to_report[NO_OF_TYPES],
      shownames,    /* Namen anzeigen */
      showclusters, /* Cluster anzeigen */
      verbose,      /* Ausfhrliche Meldungen */
      crosslinks,   /* Namen bei Crosslinks */
      showall,      /* Alle Crosslinks */
      longnames,    /* Pfadnamen bei Crosslinks */
      showdamaged,  /* Defekte Cluster melden */
      nolock,       /* Device nicht locken */
      silent,       /* Keine Rahmenausgaben */
      usefat1,      /* Bei Bedarf FAT1 benutzen */
      showfrag,     /* Fragmentierung zeigen */
      showbpb,      /* BPB anzeigen */
      checkvfat,    /* VFAT-Struktur prfen */
      hold,         /* Auf Tastendruck warten */
      quit,
      entries;
LONG  lock,
      clsiz_b;
ULONG distances;
_BPB_ *bpb;
DIR   *rootdir;

void main(WORD argc, char *argv[])
{
  UWORD     i, j,
            length,
            free_areas,
            free_length,
            min_free,
            max_free,
            critic1,
            critic2,
            badfat,
            *fat2;
  WORD      differ = 0,
            big = 0;
  LONG      minfat,
            k;
  PUN_INFO  *pun;

  fat = 0L;
  checkfat = tempcheck = 0L;
  rootdir = 0L;
  shownames = showclusters = verbose = quit =
    hold = showall = longnames = showdamaged =
    nolock = usefat1 = showfrag = crosslinks =
    showbpb = checkvfat = 0;
/*
 * Kommandozeile auswerten, beenden, wenn
 * fehlerhaft
 */
  if (!eval_args(argc, argv))
    leave(2);
/*
 * Versuchen, das gewnschte Bios-Device gegen
 * GEMDOS-Zugriffe zu schtzen. Ist der Aufruf
 * vorhanden und es trat dabei ein Fehler auf,
 * wird CheckFat mit einer entsprechenden Meldung
 * verlassen.
 */
  if (!nolock &&
    ((lock = Dlock(1, drive)) != -32L))
  {
    if (lock == -46L)
      err(NOBIOS);
    if (lock != 0L)
      err(LOCKED);
  }
  for (i = 0; i < NO_OF_TYPES; i++)
    to_report[i] = 0;
/* Versuchen, den Bios-Parameterblock zu lesen */
  if ((bpb = (_BPB_ *)Getbpb(drive)) == 0L)
    err(NOBPB);
/*
 * Sicherstellen, da es sich um eine 16-Bit-FAT
 * handelt
 */
  if ((bpb->bflags & 1) != 1)
    err(NOT16BIT);
/*
 * Einige Tests, ob es wirklich ein gltiger BPB
 * ist:
 * - Bytes pro Sektor durch 512 teilbar
 * - Bytes pro Sektor nicht Null
 * - Bytes pro Cluster = Sektoren pro Cluster *
 *   Bytes pro Sektor
 * - Wurzelverzeichnis mindestens 1 Sektor lang
 * - mindestens ein, maximal 65518 Datencluster
 * - FAT gro genug fr alle Datencluster
 * - Erster Datensektor hinter Ende von Directory
 */
  if (bpb->recsiz % 512)
    wrongbpb();
  if (!bpb->recsiz)
    wrongbpb();
/* Workaround fr Cluster mit 65536 Bytes */
  if (((LONG)bpb->recsiz * (LONG)bpb->clsiz) ==
    0x10000L)
  {
    clsiz_b = 0x10000L;
  }
  else
    clsiz_b = bpb->clsizb;
  if (clsiz_b != (bpb->clsiz * bpb->recsiz))
    wrongbpb();
  if (bpb->rdlen < 1)
    wrongbpb();
  if ((bpb->numcl < 1) || (bpb->numcl > 65518U))
    wrongbpb();
  minfat = ((LONG)(bpb->numcl + 2) * 2L +
    ((LONG)bpb->recsiz - 1L)) / (LONG)bpb->recsiz;
  if ((WORD)minfat > bpb->fsiz)
    wrongbpb();
  if ((bpb->fatrec + bpb->fsiz + bpb->rdlen) >
    bpb->datrec)
  {
    wrongbpb();
  }
/*
 * Wenn es mehr als 65535 Sektoren sind, dies
 * vermerken und prfen, ob ein AHDI-3.0-
 * kompatibler Treiber vorhanden und fr diese
 * Partition zustndig ist (da dann erweitertes
 * Rwabs() bentigt wird).
 */
  if (((LONG)bpb->datrec + (LONG)bpb->numcl *
    (LONG)bpb->clsiz - 1L) > 65535L)
  {
    big = 1;
    pun = getpun();
    if ((pun == NULL) || (pun->pun[drive] & 128))
      err(NOAHDI);
  }
/*
 * Speicherplatz fr Wurzelverzeichnis, FAT und
 * zwei Testtabellen anfordern. Gelingt dies
 * nicht, Programm mit Meldung verlassen.
 */
  if ((rootdir = (DIR *)malloc((LONG)bpb->rdlen
    * (LONG)bpb->recsiz)) == 0L)
  {
    err(NOMEM);
  }
  if ((fat = (UWORD *)malloc((LONG)bpb->fsiz *
    (LONG)bpb->recsiz)) == 0L)
  {
    err(NOMEM);
  }
  if ((checkfat = (UWORD *)malloc((LONG)bpb->fsiz
    * (LONG)bpb->recsiz)) == 0L)
  {
    err(NOMEM);
  }
  if ((tempcheck = (UWORD *)malloc((LONG)
    bpb->fsiz * (LONG)bpb->recsiz)) == 0L)
  {
    err(NOMEM);
  }
  fat2 = tempcheck;
/*
 * Anzahl der Eintrge im Wurzelverzeichnis
 * berechnen
 */
  entries = (WORD)((LONG)bpb->rdlen *
    bpb->recsiz / 32);
/*
 * Die beiden kritischen Cluster berechnen, die
 * bei GEMDOS-Versionen kleiner < 0.30 als defekt
 * markiert werden sollten und daher von CheckFat
 * nur mit Bemerkung gemeldet werden.
 */
  critic1 = 32767U / bpb->clsiz;
  critic2 = critic1 + 1;
/*
 * Wurzelverzeichnis und FATs einlesen. Tritt
 * dabei ein Fehler auf, Programm beenden
 */
  if (Rwabs(0, (void *)rootdir, bpb->rdlen,
    bpb->fatrec + bpb->fsiz, drive))
  {
    err(READERROR);
  }
  if (Rwabs(0, (void *)fat, bpb->fsiz,
    bpb->fatrec, drive))
  {
    err(READERROR);
  }
  if (Rwabs(0, (void *)fat2, bpb->fsiz,
    bpb->fatrec - bpb->fsiz, drive))
  {
    err(READERROR);
  }
  if (!silent)
  {
    printf("CheckFat: Prfe FAT von Laufwerk "
      "%c...\n", (char)(drive + 65));
  }
/*
 * FAT-Eintrge in Motorola-Format wandeln und
 * vergleichen
 */
  for (i = 0; i <= MAXCLUST; i++)
  {
    swap(&fat[i]);
    swap(&fat2[i]);
    if (fat[i] != fat2[i])
    {
      if (verbose)
      {
        printf("Eintrag %u in beiden FATs nicht "
          "identisch!\n", i);
      }
      else if (!differ)
      {
        printf("FATs sind ab Eintrag %u nicht "
          "identisch!\n", i);
      }
      differ = 1;
    }
  }
/*
 * Soll bei Ungleichheit FAT 1 benutzt werden,
 * einfach die betroffenen Pointer tauschen
 */
  if (differ && usefat1)
  {
    tempcheck = fat;
    fat = fat2;
  }
  memset(filenames, 0, sizeof(char *) * 65536L);
  memset(checkfat, 0, (LONG)bpb->fsiz *
    (LONG)bpb->recsiz);
  memset(tempcheck, 0, (LONG)bpb->fsiz *
    (LONG)bpb->recsiz);
  filenr = fragments = fragmented = max_dist = 0;
  min_dist = 0xffffU;
  distances = 0;
/*
 * Wurzelverzeichnis bearbeiten. Da die Funktion
 * work_dir rekursiv arbeitet, wird dadurch der
 * gesamte Dateibaum durchgetestet. Die globale
 * Variable quit wird auf einen Wert != 0 gesetzt,
 * wenn dabei ein Lesefehler oder Speichermangel
 * aufgetreten ist. In diesem Fall wird CheckFat
 * mit einer Meldung beendet.
 */
  work_dir(rootdir, entries, "\\",
    startclusters, 0, big);
  if (quit)
    err(quit);
  if (!silent)
    puts("\nCheckFat-Ergebnis:");
/* Ggf. BPB-Werte ausgeben */
  if (showbpb)
    printbpb();
/* Fragmentierungsgrad ausgeben, wenn gewnscht */
  if (showfrag)
  {
    if (filenr && fragmented)
    {
      printf("%u Prozent der Files/Ordner (%u "
        "von %u) sind fragmentiert!\n",
        (UWORD)(+((ULONG)fragmented * 100UL) /
        (ULONG)filenr), fragmented, filenr);
      if (verbose)
      {
        printf("Clustersprnge insgesamt: %u\n",
          fragments);
        printf("Mittlere Anzahl von Cluster"
          "sprngen pro fragmentierter/m Datei/"
          "Ordner: %u\n", fragments / fragmented);
        printf("Mittlere Lnge eines Sprungs: "
          "%lu Cluster (minimal: %u, maximal: "
          "%u)\n", distances / (ULONG)fragments,
          min_dist, max_dist);
      }
    }
    else
    {
      puts("Es gibt keine fragmentierten Dateien/"
        "Ordner!");
    }
    free_areas = free_length = max_free = 0;
    min_free = 0xffffU;
    for (i = 2; i <= MAXCLUST;) /* Kein Smiley */
    {
      if (fat[i])
      {
        for (j = i + 1; (j <= MAXCLUST) && fat[j];
          j++);
        i = j;
      }
      else
      {
        free_areas++;
        length = 0;
        for (j = i; (j <= MAXCLUST) && !fat[j];
          j++)
        {
          length++;
        }
        if (length < min_free)
          min_free = length;
        if (length > max_free)
          max_free = length;
        free_length += length;
        i = j;
      }
    }
    if (free_areas)
    {
      if (free_areas > 1)
      {
        printf("Der freie Platz ist in %u "
          "Bereiche zerteilt!\n", free_areas);
        if (verbose)
        {
          printf("Durchschnittliche Lnge eines "
            "Bereichs: %u Cluster (%lu Bytes)\n",
            free_length / free_areas,
            (ULONG)(free_length / free_areas) *
            (ULONG)clsiz_b);
          printf("Kleinster Bereich: %u Cluster "
            "(%lu Bytes)\n", min_free,
            (ULONG)min_free * (ULONG)clsiz_b);
          printf("Grter Bereich: %u Cluster "
            "(%lu Bytes)\n", max_free,
            (ULONG)max_free * (ULONG)clsiz_b);
        }
      }
      else
      {
        printf("%lu Bytes in %u Cluster(n) sind "
          "an einem Stck frei!\n",
          (ULONG)free_length * (ULONG)clsiz_b,
          free_length);
      }
    }
  }
/*
 * Alle (gltigen) Clustereintrge durchgehen und
 * Fehler melden bzw. merken (ersteres nur, wenn
 * ausfhrliche Ausgabe per -v gewnscht ist)
 */
  memset(tempcheck, 0, (LONG)bpb->fsiz *
    (LONG)bpb->recsiz);
  for (i = 2; i <= MAXCLUST; i++)
  {
/*
 * Prfen, ob weitere Cluster auf den Folgecluster
 * des aktuellen zeigen oder ob der Folgecluster
 * nicht belegt ist
 */
    if ((fat[i] >= 2) && (fat[i] <= MAXCLUST))
    {
      if (!fat[fat[i]])
      {
        if (verbose)
        {
          printf("Cluster %u verweist auf einen "
            "unbelegten Cluster!\n", i);
        }
        to_report[EMPTYPOINT]++;
      }
      if (tempcheck[fat[i]] > 0)
      {
        if (verbose)
        {
          printf("Folgende Cluster zeigen alle "
            "auf Cluster %u:\n", fat[i]);
          printf("%u %u ", tempcheck[fat[i]], i);
          for (j = i + 1; j <= MAXCLUST; j++)
          {
            if (fat[j] == fat[i])
              printf("%u ", j);
          }
          puts("");
        }
        to_report[AMBIGUOUS]++;
        tempcheck[fat[i]] = -1;
      }
      else
      {
        if (!tempcheck[fat[i]])
          tempcheck[fat[i]] = i;
      }
    }
/* Prfen, ob Cluster mehrfach belegt ist */
    if (checkfat[i] > 1)
    {
      if (verbose)
      {
        printf("Cluster %u ist %d-fach belegt!\n",
          i, checkfat[i]);
      }
      to_report[MULTIPLE]++;
    }
/* Prfen, ob Cluster verwaist ist */
    if ((fat[i] > 1) && ((fat[i] <= MAXCLUST) ||
      (fat[i] == 0xffffU)) && (checkfat[i] == 0))
    {
      if (verbose)
        printf("Cluster %u ist verwaist!\n", i);
      to_report[ORPHAN]++;
    }
/* Prfen, ob Cluster gltigen Folgecluster hat */
    if (fat[i] && ((fat[i] < 2) ||
      ((fat[i] < 0xfff0U) &&
      (fat[i] > MAXCLUST))))
    {
      if (verbose)
      {
        printf("Cluster %u hat illegalen "
          "Folgecluster!\n", i);
      }
      to_report[ILLCL]++;
    }
/*
 * Prfen, ob Cluster als defekt markiert ist. Die
 * Ausgabe erfolgt nur, wenn Option -d aktiv ist.
 */
    if (showdamaged && (fat[i] >= 0xfff0U) &&
      (fat[i] <= 0xfff7U))
    {
      if (verbose)
      {
        printf("Cluster %u ist als defekt "
          "markiert!\n", i);
        if ((i == critic1) || (i == critic2))
        {
          puts("Dies ist in der Regel nur ein "
            "Schutz gegen einen GEMDOS-Fehler!");
        }
      }
      to_report[DAMAGED]++;
    }
/* Prfen, ob Cluster auf sich selbst verweist */
    if (fat[i] == i)
    {
      if (verbose)
      {
        printf("Cluster %u zeigt auf sich "
          "selbst!\n", i);
      }
      to_report[SELFPOINTER]++;
    }
  }
/* Eventuell gefundene Fehler berichten */
  badfat = differ;
  if (differ)
    puts("Unterschiedliche FATs!");
  for (i = 0; i < NO_OF_TYPES; i++)
  {
    badfat += to_report[i];
    if (to_report[i])
      printf(reporttext[i], to_report[i]);
  }
  if (!badfat && !silent)
    puts("Alles OK!");
/*
 * Am Schlu den Speicher und das Laufwerk wieder
 * freigeben und 0 zurckliefern, wenn kein Fehler
 * gefunden wurde, sonst 3
 */
  for (k = 0L; k < 65536L; k++)
  {
    if (filenames[k])
      free(filenames[k]);
  }
  free((void *)tempcheck);
  free((void *)checkfat);
  free((void *)fat);
  free((void *)rootdir);
  if (!nolock && (lock == 0L))
    Dlock(0, drive);
  if (badfat)
    leave(3);
  else
    leave(0);
}

/*
 * leave
 *
 * Verlt CheckFat mit einem gegebenen Returncode
 * und wartet dabei gegebenenfalls vorher auf eine
 * Taste.
 *
 * Eingabe:
 * retcode: Zurckzuliefernder Returncode
 */
void leave(WORD retcode)
{
  if (hold)
  {
    if (!silent)
      puts("\nBitte eine Taste drcken!");
    Cnecin();
  }
  exit(retcode);
}

/*
 * eval_args
 *
 * Wertet die Kommandozeile aus, die CheckFat
 * bergeben wurde.
 *
 * Eingabe:
 * argc: Anzahl der Elemente in argv
 * argv: Stringarray mit einzelnen Optionen
 *       (argc und argv haben exakt die gleiche
 *       Bedeutung wie bei main!)
 *
 * Rckgabe:
 * 0: Parameter waren fehlerhaft
 * 1: Parameter OK
 */
WORD eval_args(WORD argc, char *argv[])
{
  WORD  i, j;

  drive = -1;
  if (argc < 2)
  {
    usage();
    return(0);
  }
  for (i = 1; i < argc; i++)
  {
    if (*argv[i] == '-')
    {
      if (strlen(argv[i]) == 1)
      {
        puts("CheckFat: Falsches Optionsformat "
          "- ignoriert");
        continue;
      }
      for (j = 1; j < strlen(argv[i]); j++)
      {
        switch (argv[i][j])
        {
          case 'v':
            verbose = 1;
            break;
          case 'n':
            shownames = 1;
            break;
          case 'c':
            shownames = showclusters = 1;
            break;
          case 'a':
            showall = 1;
            break;
          case 'x':
            crosslinks = 1;
            break;
          case 'l':
            crosslinks = longnames = 1;
            break;
          case 'd':
            showdamaged = 1;
            break;
          case 'u':
            nolock = 1;
            break;
          case 's':
            silent = 1;
            break;
          case '1':
            usefat1 = 1;
            break;
          case 'f':
            showfrag = 1;
            break;
          case 'i':
            showbpb = 1;
            break;
          case 't':
            checkvfat = 1;
            break;
          case 'h':
            hold = 1;
            break;
          case '?':
            usage();
            return(0);
          default:
            printf("CheckFat: Unbekannte Option "
              "'%c' - ignoriert\n", argv[i][j]);
        }
      }
    }
    else
    {
      drive = (*argv[i] & ~32) - 65;
      if ((drive < 0) || (drive > 31))
      {
        puts("CheckFat: Falsche "
          "Laufwerksangabe!");
        return(0);
      }
      if (i != (argc - 1))
      {
        puts("CheckFat: berflssige Parameter "
          "- ignoriert");
      }
      break;
    }
  }
  if (drive == -1)
  {
    puts("CheckFat: Laufwerksangabe fehlt!");
    return(0);
  }
  return(1);
}

/*
 * usage
 *
 * Gibt die Kurzanleitung aus.
 */
void usage(void)
{
  puts("CheckFat "VERSION);
#ifdef STCOMPUTER
  puts("(c) 1996 by MAXON Computer GmbH");
#endif
  puts("Geschrieben in Pure C von Thomas Binder");
  puts("\nAufruf: checkfat [Optionen] Laufwerk");
  puts("\nOptionen:");
  puts("\t-v: Ausfhrliche FAT-Fehlermeldungen");
  puts("\t-n: Alle Filenamen anzeigen");
  puts("\t-c: Alle Cluster zu allen Files "
    "anzeigen (schliet -n ein)");
  puts("\t-a: Alle mehrfach belegten Cluster "
    "einer Datei anzeigen");
  puts("\t-x: Bei mehrfach belegten Clustern "
    "auch die anderen Dateinamen melden");
  puts("\t-l: Komplette Pfadnamen bei -x melden "
    "(schliet -x ein, hoher Speicher-\n"
    "\t    bedarf!)");
  puts("\t-d: Bei der Ergebnisausgabe als "
    "defekt markierte Cluster melden");
  puts("\t-1: Bei Ungleichheit FAT 1 als "
    "Prfgrundlage benutzen");
  puts("\t-f: Fragmentierungsinformationen "
    "ausgeben");
  puts("\t-i: BPB des Laufwerks ausgeben");
  puts("\t-t: VFAT-Struktur berprfen");
  puts("\t-u: Laufwerk nicht mit Dlock sperren");
  puts("\t-s: Keine unntigen Ausgaben");
  puts("\t-h: Nach Programmende auf Tastendruck "
    "warten");
  puts("\t-?: Dieser Hilfetext");
}

/*
 * work_dir
 *
 * Bearbeitet alle Files eines Verzeichnisses und
 * alle Unterverzeichnisse. Gefunden werden dabei
 * Clusterschleifen, Ordnerschleifen, illegale
 * Start- und Folgecluster sowie Files mit
 * defekten Clustern. Auerdem werden alle
 * belegten Cluster vermerkt, um spter mehrfach
 * vergebene und verwaiste Cluster erkennen zu
 * knnen.
 *
 * Eingabe:
 * dir: Zeiger auf das zu bearbeitende Verzeichnis
 * entries: Maximale Anzahl an Eintrgen in dir
 * path: Pfadname von dir (mit abschlieendem \\)
 * scl: Array mit Anfangsclustern der
 *      "Elternverzeichnisse" von dir
 * depth: Bisherige Schachtelungstiefe, 0 =
 *        Wurzelverzeichnis. scl enthlt somit
 *        depth Eintrge.
 * big: Partition hat mehr als 65535 Sektoren und
 *      mu daher mit erweitertem Rwabs() gelesen
 *      werden.
 */
void work_dir(DIR *dir, UWORD entries, char *path,
  UWORD *scl, WORD depth, WORD big)
{
  UWORD i, j, k,
        frag,
        dist,
        nameok,
        readerr,
        first,
        cl,
        lastcl;
  ULONG clusts,
        must;
  DIR   *subdir,
        *newdir;
  char  my_path[PATHMAX],
        fname[13];

/*
 * Sollte maximale Schachtelungstiefe erreicht
 * sein, Funktion abbrechen und Fehler melden
 */
  if (depth == MAXDEPTH)
  {
    if (!shownames)
      printf("%s: ", path);
    puts("Pfad zu tief verschachtelt!\n");
    quit = TOODEEP;
    return;
  }
/* Eintrge des Verzeichnisses umwandeln */
  correct_dir(dir, entries);
/*
 * Prfen, ob die beiden ersten Eintrge . und ..
 * heissen und Verzeichnisse sind. Wenn nicht,
 * Verzeichnis nicht bearbeiten. Dieser Test mu
 * natrlich im Wurzelverzeichnis entfallen.
 */
  if (depth)
  {
    if (strncmp(dir[0].dir_name, ".          ",
      11) || strncmp(dir[1].dir_name,
      "..         ", 11) ||
      !(dir[0].dir_attr & 16) ||
      !(dir[1].dir_attr & 16))
    {
      if (!shownames)
        printf("%s: ", path);
      puts("Kein '.' und/oder '..'!");
      to_report[ILLSUBDIR]++;
      return;
    }
/*
 * Jetzt noch testen, ob . und .. die richtigen
 * Startcluster haben. Falls nicht, wird dies
 * gemeldet, das Verzeichnis aber trotzdem
 * bearbeitet.
 */
    if (dir[0].dir_stcl != scl[depth - 1])
    {
      if (!shownames)
        printf("%s: ", path);
      puts("Falscher Startcluster fr '.'!");
      to_report[ILLPSEUDO]++;
    }
    if (dir[1].dir_stcl != ((depth == 1) ? 0 :
      scl[depth - 2]))
    {
      if (!shownames)
        printf("%s: ", path);
      puts("Falscher Startcluster fr '..'!");
      to_report[ILLPSEUDO]++;
    }
  }
/* Alle Eintrge durchgehen */
  for (i = (depth) ? 2 : 0; i < entries; i++)
  {
/*
 * Ist das erste Zeichen des aktuellen Filenamens
 * eine Null, ist der letzte Eintrag erreicht
 */
    if (!dir[i].dir_name[0])
      break;
/* Gelschte Eintrge berspringen */
    if (dir[i].dir_name[0] == (char)0xe5)
      continue;
/* Ggf. VFAT-Eintrge prfen */
    if (checkvfat && (dir[i].dir_attr == 0x0f))
    {
      vfat_check(path, dir, i, entries, &i);
      if (i == 0xffff)
        return;
      i--;
      continue;
    }
/* Laufwerkslabel berspringen */
    if (dir[i].dir_attr & 8)
      continue;
    frag = 0;
/*
 * Startcluster ermitteln und aktuelle Filenummer
 * erhhen. Durch diese Nummer erhlt jedes File
 * eine individuelle Kennzeichnung, mit der es
 * mglich ist, Clusterschleifen zuverlssig zu
 * erkennen. Da filenr ein 16-Bit-Wert ist, darf
 * ein Laufwerk maximal 65535 Dateien/Ordner
 * beherbergen; allerdings ist diese Grenze schon
 * technisch nicht erreichbar, da eine 16-Bit-Fat
 * maximal 65518 Dateicluster haben kann. Auerdem
 * ermglicht es diese individuelle Nummer, bei
 * bereits belegten Clustern zu ermitteln, zu
 * welcher Datei sie gehren.
 */
    cl = dir[i].dir_stcl;
    if ((filenr++) == 65519U)
    {
      quit = TMFILES;
      return;
    }
/* Den aktuellen Filenamen basteln */
    for (j = 0; j < 8; j++)
    {
      if (dir[i].dir_name[j] != ' ')
        fname[j] = dir[i].dir_name[j];
      else
        break;
    }
    fname[j] = 0;
    if (dir[i].dir_name[8] != ' ')
    {
      fname[j++] = '.';
      for (k = 0; k < 3; k++)
      {
        if (dir[i].dir_name[8 + k] != ' ')
          fname[j + k] = dir[i].dir_name[8 + k];
        else
          break;
      }
      fname[j + k] = 0;
    }
    nameok = 1;
    if (!*fname)
      nameok = 0;
    else
    {
      for (j = 0; j < strlen(fname); j++)
      {
        if (((UBYTE)fname[j] < 32) ||
          (fname[j] == '?') ||
          (fname[j] == '*') ||
          (fname[j] == ':'))
        {
          if ((UBYTE)fname[j] < 32)
            fname[j] = '?';
          nameok = 0;
        }
      }
    }
    if (!nameok)
     to_report[ILLNAME]++;
    if (dir[i].dir_attr & 16)
    {
/*
 * Eintrag ist Ordner, also neuen Pfadnamen
 * zusammensetzen; dabei vorher prfen, ob my_path
 * noch gengend Platz bietet; wenn nicht, wird
 * TOODEEP gemeldet
 */
      if ((strlen(path) + 2 + strlen(fname)) >
        PATHMAX)
      {
        printf("%s%s\\ ist zu lang!\n", path,
          fname);
        quit = TOODEEP;
        return;
      }
      strcpy(my_path, path);
      strcat(my_path, fname);
      strcat(my_path, "\\");
      if (shownames)
      {
        printf("%s%s ", my_path, nameok ? "" :
          "(illegaler Name!)");
        if (showclusters)
          printf("%u ", cl);
      }
      else
      {
        if (!nameok)
        {
          printf("%s: Illegaler Name!\n",
            my_path);
        }
      }
/*
 * Startcluster berprfen und notfalls
 * beanstanden. Bei Ordnern wird auch Null als
 * Startcluster beanstandet, da Ordner nie
 * komplett leer sein drfen (sie mssen
 * mindestens . und .. enthalten).
 */
      if ((cl < 2) || (cl > MAXCLUST))
      {
        if (!shownames)
          printf("%s: ", my_path);
        puts("Illegaler Startcluster!");
        to_report[ILLEGAL]++;
        continue;
      }
/*
 * Prfen, ob dieser Startcluster im aktuellen
 * Pfad schon einmal vorhanden war. Wenn ja, zeigt
 * das neue Verzeichnis auf einen seiner
 * Vorgnger, es gibt also eine Ordnerschleife.
 */
      for (j = 0; j < depth; j++)
      {
        if (scl[j] == cl)
        {
          if (!shownames)
            printf("%s: ", my_path);
          puts("Ordnerschleife!");
          to_report[DIRLOOPS]++;
          break;
        }
      }
      if (j != depth)
        continue;
/*
 * Bei aktiver Kommandozeilenoption -x den Ordner-
 * bzw. Pfadnamen speichern. Ist dazu nicht mehr
 * genug Speicher vorhanden, wird work_dir
 * abgebrochen. Die Speicherung erfolgt erst
 * jetzt, weil vorher noch nicht sichergestellt
 * war, ob der Ordner berhaupt gltig ist, was
 * unter Umstnden Speicher spart.
 */
      if (crosslinks)
      {
        if (longnames)
        {
          filenames[filenr] =
            malloc(strlen(my_path) + 1);
        }
        else
        {
          filenames[filenr] =
            malloc(strlen(fname) + 2);
        }
        if (filenames[filenr] == 0L)
        {
          if (shownames)
            puts("");
          quit = NOMEM;
          return;
        }
        if (longnames)
          strcpy(filenames[filenr], my_path);
        else
        {
          strcpy(filenames[filenr], fname);
          strcat(filenames[filenr], "\\");
        }
      }
/*
 * Den Startcluster in der Liste vermerken und
 * Speicher fr ein neues Verzeichnis anfordern
 */
      scl[depth] = cl;
      if ((subdir = (DIR *)malloc(clsiz_b)) != 0L)
      {
/*
 * War Speicher vorhanden, alle Cluster des neuen
 * Directories ermitteln und prfen
 */
        clusts = first = 0;
        while (cl < 0xfff0U)
        {
/* Testen, ob aktueller Cluster gltig ist */
          if ((cl < 2) || ((cl > MAXCLUST) &&
            (cl < 0xfff0U)))
          {
            if (!shownames)
              printf("%s: ", my_path);
            puts("Illegaler Folgecluster!");
            to_report[ILLEGAL]++;
            break;
          }
/*
 * Testen, ob der Cluster berhaupt belegt ist;
 * falls nicht, trotzdem einlesen
 */
          if (!fat[cl])
          {
            if (!shownames)
              printf("%s: ", my_path);
            puts("Enthlt leeren Cluster!");
            to_report[EMPTYCLUST]++;
          }
/*
 * Gehrt der aktuelle Cluster bereits zu diesem
 * Ordner, liegt eine Clusterschleife vor
 */
          if (tempcheck[cl] == filenr)
          {
            if (!shownames)
              printf("%s: ", my_path);
            puts("Clusterschleife!");
            to_report[LOOPS]++;
            break;
          }
/*
 * War der Cluster bereits belegt, dies (ggf. mit
 * File- bzw. Pfadnamen der Datei, die ihn belegt)
 * ausgeben (je nach Einstellung von showall
 * nur, wenn es der erste Cluster war). Dabei wird
 * die Einleseschleife nicht abgebrochen, da das
 * Unterverzeichnis ja noch weitergeht (wenn auch
 * nicht mehr ganz korrekt).
 */
          if (tempcheck[cl])
          {
            if (showall || !first)
            {
              first = 1;
              if (!shownames)
                printf("%s: ", my_path);
              if (crosslinks)
              {
                printf("Cluster %d bereits durch "
                  "%s belegt!\n", cl,
                  filenames[tempcheck[cl]]);
              }
              else
              {
                printf("Cluster %d bereits "
                  "belegt!\n", cl);
              }
            }
          }
/*
 * War alles OK, den aktuellen Cluster einlesen.
 * Tritt dabei ein Fehler auf, die Routine
 * abbrechen.
 */
          if (big)
          {
            readerr = (UWORD)R_wabs(0,
              (void *)((LONG)subdir +
              clusts * clsiz_b), bpb->clsiz, -1,
              drive, (LONG)bpb->datrec +
              (LONG)(cl - 2) * (LONG)bpb->clsiz);
          }
          else
          {
            readerr = (UWORD)Rwabs(0,
              (void *)((LONG)subdir + clusts *
              clsiz_b), bpb->clsiz, bpb->datrec +
              (cl - 2) * bpb->clsiz, drive);
          }
          if (readerr)
          {
            free((void *)subdir);
            if (shownames)
              puts("");
            quit = READERROR;
            break;
          }
/*
 * In der FAT-Testtabelle den Belegungsgrad des
 * aktuellen Clusters erhhen und vermerken, da
 * er bereits zu diesem Ordner gehrt
 */
          checkfat[cl]++;
          tempcheck[cl] = filenr;
/*
 * Folgecluster ermitteln und ggf. neuen Speicher
 * anfordern
 */
          lastcl = cl;
          cl = fat[cl];
          clusts++;
          if (!cl)
            break;
          if (cl < 0xfff0U)
          {
/* Clustersprung merken, falls ntig */
            if (showfrag && ((lastcl + 1) != cl))
            {
              if (!frag)
              {
                fragmented++;
                if (verbose && shownames &&
                  !showclusters)
                {
                  puts("Enthlt Clustersprung!");
                }
              }
              fragments++;
              if (verbose)
              {
                dist = abs(lastcl - cl);
                distances += dist;
                if (dist < min_dist)
                  min_dist = dist;
                if (dist > max_dist)
                  max_dist = dist;
                if (showclusters)
                  printf("-/- ");
              }
              frag = 1;
            }
            if (showclusters)
              printf("%u ", cl);
            newdir = (DIR *)realloc((void *)
              subdir, (clusts + 1L) * clsiz_b);
            if (newdir == 0L)
            {
              free(subdir);
              quit = NOMEM;
              if (shownames)
                puts("");
              break;
            }
            subdir = newdir;
          }
        }
        if (!quit)
        {
/*
 * Alle Cluster sind ermittelt. War der letzte
 * Cluster ein als defekt markierter, dieses
 * vermerken.
 */
          if ((cl >= 0xfff0U) && (cl < 0xfff8U))
          {
            if (!shownames)
              printf("%s: ", my_path);
            puts("Enthlt defekten Cluster!");
            to_report[DESTROYED]++;
          }
          if (shownames)
            puts("");
/*
 * work_dir rekursiv mit dem neu eingelesenen
 * Verzeichnis aufrufen
 */
          work_dir(subdir, (UWORD)(clusts *
            clsiz_b / 32L), my_path, scl,
            depth + 1, big);
          free(subdir);
        }
      }
      else
      {
        quit = NOMEM;
        if (shownames)
          puts("");
      }
    }
    else
    {
/*
 * Aktueller Verzeichniseintrag ist eine Datei.
 * Auch hier zunchst den Startcluster prfen.
 */
      if (shownames)
      {
        printf("%s%s%s ", path, fname,
          nameok ? "" : "(illegaler Name!)");
        if (showclusters)
          printf("%u ", cl);
      }
      else
      {
        if (!nameok)
        {
          printf("%s%s: Illegaler Name!\n",
            path, fname);
        }
      }
      clusts = first = 0;
/*
 * Bei Dateien Startcluster Null nicht
 * beanstanden, es handelt sich hierbei um
 * regulre Files mit Lnge Null
 */
      if (!cl)
      {
        if (shownames)
          puts("");
        continue;
      }
/* Startcluster prfen */
      if ((cl < 2) || (cl > MAXCLUST))
      {
        if (!shownames)
          printf("%s%s: ", path, fname);
        puts("Illegaler Startcluster!");
        to_report[ILLEGAL]++;
      }
      else
      {
/*
 * War Startcluster OK, aus der Dateilnge
 * berechnen, wieviele Cluster diese Datei
 * belegen mu. Danach alle Cluster des Files
 * durchgehen. Zuvor wird noch, falls ntig (-x),
 * der Datei- bzw. Pfadname gespeichert.
 */
        if (crosslinks)
        {
          if (longnames)
          {
            filenames[filenr] = malloc(
              strlen(path) + strlen(fname) + 1);
          }
          else
          {
            filenames[filenr] =
              malloc(strlen(fname) + 1);
          }
          if (filenames[filenr] == 0L)
          {
            quit = NOMEM;
            if (shownames)
              puts("");
            return;
          }
          if (longnames)
          {
            strcpy(filenames[filenr], path);
            strcat(filenames[filenr], fname);
          }
          else
            strcpy(filenames[filenr], fname);
        }
        must = (dir[i].dir_flen / clsiz_b);
        if (dir[i].dir_flen % clsiz_b)
          must++;
        while (cl < 0xfff0U)
        {
/* Cluster auf Gltigkeit prfen */
          if ((cl < 2) || ((cl > MAXCLUST) &&
            (cl < 0xfff0U)))
          {
            if (!shownames)
              printf("%s%s: ", path, fname);
            puts("Illegaler Folgecluster!");
            to_report[ILLEGAL]++;
            break;
          }
/*
 * Prfen, ob der Cluster berhaupt belegt ist;
 * falls nicht, trotzdem mitzhlen
 */
          if (!fat[cl])
          {
            if (!shownames)
              printf("%s%s: ", path, fname);
            puts("Enthlt leeren Cluster!");
            to_report[EMPTYCLUST]++;
          }
/*
 * War der aktuelle Cluster schon durch diese
 * Datei belegt, Clusterschleife melden
 */
          if (tempcheck[cl] == filenr)
          {
            if (!shownames)
              printf("%s%s: ", path, fname);
            puts("Clusterschleife!");
            to_report[LOOPS]++;
            break;
          }
/*
 * Ist der Cluster bereits durch eine andere
 * Datei belegt, dieses melden. Die Schleife wird
 * dabei nicht abgebrochen, da die Datei dadurch
 * ja noch nicht beendet ist.
 */
          if (tempcheck[cl])
          {
            if (showall || !first)
            {
              first = 1;
              if (!shownames)
                printf("%s%s: ", path, fname);
              if (crosslinks)
              {
                printf("Cluster %d bereits durch "
                  "%s belegt!\n", cl,
                  filenames[tempcheck[cl]]);
              }
              else
              {
                printf("Cluster %d bereits "
                  "belegt!\n", cl);
              }
            }
          }
/*
 * Jetzt Belegungsgrad des aktuellen Clusters
 * erhhen und vermerken, da er bereits in diesem
 * File benutzt wurde
 */
          checkfat[cl]++;
          tempcheck[cl] = filenr;
/*
 * Folgecluster ermitteln und Anzahl der durch das
 * File tatschlich belegten Cluster um eins
 * erhhen
 */
          lastcl = cl;
          cl = fat[cl];
          clusts++;
          if (!cl)
            break;
          if (cl < 0xfff0U)
          {
/* Clustersprung merken, falls ntig */
            if (showfrag && ((lastcl + 1) != cl))
            {
              if (!frag)
              {
                fragmented++;
                if (verbose && shownames &&
                  !showclusters)
                {
                  puts("Enthlt Clustersprung!");
                }
              }
              fragments++;
              if (verbose)
              {
                dist = abs(lastcl - cl);
                distances += dist;
                if (dist < min_dist)
                  min_dist = dist;
                if (dist > max_dist)
                  max_dist = dist;
                if (showclusters)
                  printf("-/- ");
              }
              frag = 1;
            }
            if (showclusters)
              printf("%u ", cl);
          }
        }
/*
 * Sind alle Cluster geprft, testen, ob das Ende
 * der Verkettung durch einen defekten Cluster
 * markiert wurde. Wenn ja, dies vermerken.
 */
        if ((cl >= 0xfff0U) && (cl < 0xfff8U))
        {
          if (!shownames)
            printf("%s%s: ", path, fname);
          puts("Enthlt defekten Cluster!");
          to_report[DESTROYED]++;
        }
        else
        {
/*
 * Ansonsten prfen, ob die Datei die richtige
 * Anzahl von Clustern belegt
 */
          if (clusts != must)
          {
            if (!shownames)
              printf("%s%s: ", path, fname);
            puts("Abweichende Lngen!");
            to_report[WRONG]++;
            continue;
          }
        }
        if (shownames)
          puts("");
      }
    }
    if (quit)
      return;
  }
}

/*
 * correct_dir
 *
 * Wandelt die Eintrge eines eingelesenen
 * Verzeichnisses vom Intel- ins Motorola-Format
 * um. Bei VFAT-Prfung werden Eintrge mit
 * Attribut 0xf nicht behandelt.
 *
 * Eingabe:
 * dir: Zeiger auf Verzeichnis
 * entries: Anzahl zu wandelnder Eintrge in dir
 */
void correct_dir(DIR *dir, WORD entries)
{
  WORD  i;
  
  for (i = 0; i < entries; i++)
  {
    if (!checkvfat || (dir[i].dir_attr != 0xf))
    {
      swap(&dir[i].dir_time);
      swap(&dir[i].dir_date);
      swap(&dir[i].dir_stcl);
      lswap(&dir[i].dir_flen);
    }
  }
}

/*
 * swap
 *
 * 16-Bit-Wert vom Intel- ins Motorola-Format
 * umwandeln (und umgekehrt).
 *
 * Eingabe:
 * value: Zeiger auf zu wandelndes Wort
 */
void swap(UWORD *value)
{
  *value = ((*value & 255) << 8) + (*value >> 8);
}

/*
 * lswap
 *
 * Wie swap, jedoch fr 32-Bit-Werte.
 *
 * Eingabe:
 * value: Zeiger auf zu wandelnden Long
 */
void lswap(ULONG *value)
{
  UWORD high, low;
  
  low = (UWORD)(*value & 65535L);
  high = (UWORD)(*value >> 16);
  
  swap(&low);
  swap(&high);
  
  *value = ((LONG)low << 16) + (LONG)high;
}

/*
 * err
 *
 * Fehlerbehandlungsroutine. Wird angesprungen,
 * wenn whrend des Programmablaufs ein Fehler
 * aufgetreten ist. Hier wird, wenn ntig, aller
 * angeforderter Speicher freigegeben und eine
 * Meldung ausgegeben, welcher Fehler aufgetreten
 * ist. Die Funktion kehrt nicht zurck, sondern
 * beendet das Programm.
 *
 * Eingabe:
 * code: Fehlercode
 */
void err(WORD code)
{
  LONG i;

/* Wenn ntig, Speicher freigeben */
  for (i = 0L; i < 65536L; i++)
  {
    if (filenames[i])
      free(filenames[i]);
  }
  if (tempcheck != 0L)
    free((void *)tempcheck);
  if (checkfat != 0L)
    free((void *)checkfat);
  if (fat != 0L)
    free((void *)fat);
  if (rootdir != 0L)
    free((void *)rootdir);
/* Meldung ausgeben */
  printf("CheckFat: %s\n", errtext[code]);
/* Laufwerk freigeben, wenn ntig */
  if (!nolock && (lock == 0L))
    Dlock(0, drive);
/* Und tsch... */
  leave(1);
}

/*
 * getpun
 *
 * Ermittelt den Zeiger auf die PUN_INFO-Struktur
 * des aktuellen Plattentreibers, der mindestens
 * AHDI 3.0 sein mu.
 *
 * Rckgabe:
 * NULL: Kein Plattentreiber vorhanden oder nicht
 *       AHDI 3.0-kompatibel
 * sonst: Zeiger auf PUN_INFO-Struktur des
 *        Treibers
 */
PUN_INFO *getpun(void)
{
  PUN_INFO *p;

  p = (PUN_INFO *)Supexec(pun_ptr);
  if (p != NULL)
  {
    if ((p->P_cookie == 0x41484449L) && /* AHDI */
      (p->P_cookptr == &(p->P_cookie)) &&
      (p->P_version >= 0x0300U))
    {
      return(p);
    }
  }
  return(NULL);
}

/*
 * pun_ptr
 *
 * Hilfsroutine, die ber Supexec() aufgerufen
 * wird und den aktuellen Inhalt der System-
 * variablen pun_ptr (0x516) liefert.
 *
 * Rckgabe:
 * Aktueller Inhalt von Adresse 0x516 als LONG
 */
LONG pun_ptr(void)
{
  return(*(LONG *)0x516);
}

/*
 * printbpb
 *
 * Gibt den aktuellen Inhalt des BPB aus.
 */
void printbpb(void)
{
  printf("Sektorgre (recsiz) ............"
    "........ %5u (0x%04x)\n", bpb->recsiz,
    bpb->recsiz);
  printf("Sektoren pro Cluster (clsiz) ...."
    "........ %5u (0x%04x)\n", bpb->clsiz,
    bpb->clsiz);
  printf("Clustersize (clsizb) ............"
    "........ %5u (0x%04x)\n", bpb->clsizb,
    bpb->clsizb);
  printf("Sektoren im Wurzelverzeichnis "
    "(rdlen) ... %5u (0x%04x)\n", bpb->rdlen,
    bpb->rdlen);
  printf("Sektoren pro FAT (fsiz) ........."
    "........ %5u (0x%04x)\n", bpb->fsiz,
    bpb->fsiz);
  printf("Startsektor der 2. FAT (fatrec) ."
    "........ %5u (0x%04x)\n", bpb->fatrec,
    bpb->fatrec);
  printf("Startsektor des 1. Datenclusters "
    "(datrec) %5u (0x%04x)\n", bpb->datrec,
    bpb->datrec);
  printf("Gesamtzahl Cluster (numcl) ......"
    "........ %5u (0x%04x)\n", bpb->numcl,
    bpb->numcl);
  printf("Flags (bflags) .................."
    "........ %5u (0x%04x)\n", bpb->bflags,
    bpb->bflags);
}

/*
 * wrongbpb
 *
 * Meldet einen ungltigen BPB und gibt ihn ggf.
 * vorher noch aus. Die Funktion kehrt nicht
 * zurck, sondern beendet das Programm.
 */
void wrongbpb(void)
{
  if (showbpb)
    printbpb();
  err(WRONGBPB);
}

/*
 * vfat_check
 *
 * berprft in einem Verzeichnis die Gltigkeit
 * eines oder mehrerer VFAT-Eintrge.
 *
 * Eingabe:
 * path: Zeiger auf aktuellen Pfadnamen
 * dir: Zeiger auf aktuelle Verzeichnistabelle
 * start: Index des VFAT-Eintrags in dir
 * max: Anzahl der Eintrge in dir
 * newi: Zeiger auf Variable, in die der Index des
 *       nchsten Nicht-VFAT-Eintrags geschrieben
 *       wird; ggf. 0xffff, wenn es keine weiteren
 *       Eintrge mehr gibt
 */
void vfat_check(char *path, DIR *dir, UWORD start,
  UWORD max, UWORD *newi)
{
  UBYTE     chksum,
            realchksum,
            compval,
            maxseq,
            *hlp;
  VFAT_DIR  *entry;
  char		longname[833], /* 64 * 13 + 1 */
            *lptr,
            *lptrmax,
            *errtext,
            shortname[13];
  WORD      i,
            j,
            nameok;

/* Vorbereitungen treffen */
  entry = (VFAT_DIR *)&dir[start];
  chksum = entry->chksum;
  compval = 0x40;
  maxseq = (entry->seq & 0x3f) - 1;
  memset(longname, 1, 832);
  lptr = longname;
  lptrmax = &longname[13 * (maxseq + 1)];
/*
 * Das abschlieende Nullbyte setzen, weil VFAT-
 * Eintrge nicht zwingend nullterminiert sind
 * (wenn der letzte VFAT-Slot komplett gefllt
 * werden kann, sind sie es nicht)
 */
  *lptrmax = 0;
/*
 * Eine Schleife ber alle laut erstem Eintrag
 * vorhandenen Namensteile laufen lassen
 */
  for (*newi = start; *newi <= start + maxseq;
    (*newi)++)
  {
/* Prfen, ob Verzeichnisende erreicht */
    if (*newi == max)
    {
      printf("%s%s: Verzeichnis endet in VFAT-"
        "Sequenz!\n", shownames ? "" : path,
        lptr);
      to_report[VFATDIREND]++;
      *newi = 0xffff;
      return;
    }
    lptr = &longname[(start + maxseq - *newi)
      * 13];
/* Den Namen fortschreiben */
    hlp = entry->unicode0_4;
    for (i = 0; i < 5; i++, hlp += 2)
      lptr[i] = uni2ascii(hlp[0], hlp[1]);
    hlp = entry->unicode5_10;
    for (i = 5; i < 11; i++, hlp += 2)
      lptr[i] = uni2ascii(hlp[0], hlp[1]);
    hlp = entry->unicode11_12;
    for (i = 11; i < 13; i++, hlp += 2)
      lptr[i] = uni2ascii(hlp[0], hlp[1]);
    nameok = 0;
/*
 * Prfen, ob im Sequenzbyte Bit 6 und Bit 7
 * korrekt gesetzt sind (beim ersten Eintrag mu
 * nur Bit 6 gesetzt sein, danach mssen beide
 * immer gelscht sein)
 */
    if ((entry->seq & 0xc0) != compval)
      nameok = VFAT_ILLSEQ;
/* Die Sequenznummer mu korrekt sein */
    else if ((entry->seq & 0x3f) !=
      (start + maxseq - *newi + 1))
    {
      nameok = VFAT_OUTOFSEQ;
    }
/*
 * Das Attributbyte mu 0xf, der Startcluster 0
 * sein; auerdem darf sich die Checksumme nicht
 * ndern
 */
    else if (entry->attr != 0xf)
      nameok = VFAT_INCOMPL;
    else if ((entry->scl[0]) || (entry->scl[1]))
      nameok = VFAT_ILLSCL;
    else if (entry->chksum != chksum)
      nameok = VFAT_CHKSUM;
    compval = 0;
/*
 * War eine der Bedingungen nicht erfllt, den
 * bis dahin aufgebauten VFAT-Namen als illegal
 * melden. Dabei wird unterschieden, ob das
 * Attributbyte 0xf war oder nicht, denn wenn
 * nicht, gehrt der aktuelle Eintrag schon nicht
 * mehr zu einem VFAT-Namen und darf daher nicht
 * bercksichtigt werden.
 */
    if (nameok)
    {
      if (entry->attr != 0xf)
        lptr += 13;
      else
        (*newi)++;
      switch (nameok)
      {
        case VFAT_ILLSEQ:
          errtext = "ungltiges Sequenzbyte";
          break;
        case VFAT_OUTOFSEQ:
          errtext = "falsches Sequenzbyte";
          break;
        case VFAT_INCOMPL:
          errtext = "Sequenz unvollstndig";
          break;
        case VFAT_ILLSCL:
          errtext = "Startcluster nicht 0";
          break;
        case VFAT_CHKSUM:
          errtext = "Checksummenunterschied";
          break;
      }
      printf("%s%s: Illegaler VFAT-Eintrag "
        "(%s)!\n", shownames ? "" : path, lptr,
        errtext);
      to_report[ILLVFAT]++;
      return;
    }
    entry++;
  }
/*
 * Nach dem Nullbyte drfen nur noch 1-Bytes im
 * Namen folgen, vorher drfen keine Fullbytes
 * zu finden sein. Auerdem sind keine leeren
 * VFAT-Namen mglich.
 */
  lptr = strrchr(longname, 0);
  if (lptr == longname)
  {
  	printf("%s: Illegaler VFAT-Eintrag "
  	  "(leerer langer Name)!\n",
  	  shownames ? "" : path);
  	to_report[ILLVFAT]++;
  	return;
  }
  if (strrchr(lptr - 1, '\001') != NULL)
  {
    printf("%s%s: Illegaler VFAT-Eintrag "
      "(langer Name nicht korrekt "
      "beendet)!\n", shownames ? "" : path,
      longname);
    to_report[ILLVFAT]++;
    return;
  }
  for (lptr++; lptr < lptrmax; lptr++)
  {
    if (*lptr != '\001')
    {
      printf("%s%s: Illegaler VFAT-Eintrag "
        "(langer Name nicht korrekt "
        "beendet)!\n", shownames ? "" : path,
        longname);
      to_report[ILLVFAT]++;
      return;
    }
  }
/* Nochmals prfen, ob Verzeichnisende erreicht */
  if (*newi == max)
  {
    printf("%s%s: Verzeichnis endet in VFAT-"
      "Sequenz!\n", shownames ? "" : path,
      longname);
    to_report[VFATDIREND]++;
    *newi = 0xffff;
    return;
  }
/*
 * Im Anschlu an den VFAT-Namen mu eine Datei
 * oder ein Verzeichnis folgen (nicht gelscht!)
 */
  if ((entry->attr & 8) || (entry->seq == 0xe5))
  {
    printf("%s%s: Illegaler VFAT-Eintrag (es "
      "folgt weder Datei noch Verzeichnis)!\n",
      shownames ? "" : path, longname);
    to_report[ILLVFAT]++;
    return;
  }
/*
 * Jetzt prfen, ob die Checksumme zum Eintrag
 * pat
 */
  dir = (DIR *)entry;
  realchksum = 0;
  for (i = 0; i < 11; i++)
  {
    realchksum = (((realchksum & 1) << 7) |
      ((realchksum & 0xfe) >> 1)) +
      dir->dir_name[i];
  }
/*
 * Falsche Prfsummen zusammen mit dem nicht
 * passenden 8+3-Namen melden
 */
  if (realchksum != chksum)
  {
    for (i = 0; i < 8; i++)
    {
      if (dir->dir_name[i] != ' ')
        shortname[i] = dir->dir_name[i];
      else
        break;
    }
    shortname[i] = 0;
    if (dir->dir_name[8] != ' ')
    {
      shortname[i++] = '.';
      for (j = 0; j < 3; j++)
      {
        if (dir->dir_name[8 + j] != ' ')
          shortname[i + j] = dir->dir_name[8 + j];
        else
          break;
      }
      shortname[i + j] = 0;
    }
    printf("%s%s: Pat nicht zum 8+3-Namen %s!\n",
      shownames ? "" : path, longname, shortname);
    to_report[VFATNOMATCH]++;
  }
}

/*
 * uni2ascii
 *
 * Wandelt ein Unicode-Zeichen - soweit mglich -
 * in ein ATARI-ASCII-Zeichen um. Nicht umsetzbare
 * Zeichen werden durch ein Fragezeichen ersetzt.
 * Das Zeichen 65535 (Codepage 255, Offset 255),
 * das in VFAT-Eintrgen zum Auffllen dient, wird
 * auf '\001' abgebildet.
 *
 * Eingabe:
 * off: Offset innerhalb der Codepage
 * cp: Die Codepage des Zeichens
 *
 * Rckgabe:
 * Entsprechendes ATARI-ASCII-Zeichen, '\001' oder
 * '?'
 */
char uni2ascii(UBYTE off, UBYTE cp)
{
  WORD  *codepage;

  if ((off == 0xff) && (cp == 0xff))
    return('\001');
  if ((codepage = codepages[cp]) == NULL)
    return('?');
  if (codepage[off] == -1)
    return('?');
  else
    return((char)(codepage[off] & 0xff));
}

/* EOF */
