/*
 *	RIMM.C - Module principal de la Librairie Dynamique Gem "RIM Manager"
 *	par Arnaud BERCEGEAY
 *	Version SozobonX
 */

#include <atari.h>
#include <stdlib.h>
#include <string.h>
#include <xgemfast.h>
#include <ldg.h>

#include "rimm_struct.h"

/***************************************************************************
  Prototypages des fonctions
 ***************************************************************************/
 
char *  __infos( void );
int     __OpenRIMDataBase( RIMDB * rimdb, char * path );
int     __CloseRIMDataBase( RIMDB * rimdb );
int     __GetRIMInfos( RIMDB * rimdb, long no_rim, RIMINFOS ** riminfos );
int     __DoRIMConfiguration( RIMDB * rimdb, long no_rim );
int     __OpenImg( RIMDB * rimdb, long no_rim, char *filename, RIMIMG * rimimg);
int     __CloseImg( RIMIMG * rimimg);

/***************************************************************************
  Variables globales
 ***************************************************************************/

PROC Proc[]  = 
{
	"__infos",      "Alerte d'information", (void *)__infos,
	"__OpenRIMDataBase", "Chargement des RIMs", (void *)__OpenRIMDataBase,
	"__CloseRIMDataBase", "Libration des RIMs", (void *)__CloseRIMDataBase,
	"__GetRIMInfos", "Rcupre infos sur un RIM", (void *)__GetRIMInfos,
	"__DoRIMConfiguration", "Configurer le RIM", (void *)__DoRIMConfiguration,
	"__OpenImg", "Ouvrir une image", (void *)__OpenImg,
	"__CloseImg", "Ouvrir une image", (void *)__CloseImg,
};

char Info[] = "Librairie RIMM, par Arnaud BERCEGEAY - 1999.";

LDGLIB Ldg[] = {
				0x0010,	/* version de la lib, recommand */
				7,	   	/* nombre de fonction dans la lib */
				Proc, 	/* Pointeurs vers nos fonctions */
				Info,  	/* Information sur la lib */
				0,    	/* Flags de la lib, obligatoire */
				NULL,    /* pas de fonction "close" */
				NULL     /* pas d'extension */
				};

/***************************************************************************
  Fonctions de la librairie
 ***************************************************************************/

/***************************************************************************
  char * __infos( void )
   
  retourne une chaine a utiliser directement dans form_alert()
 ***************************************************************************/
 
char * __infos( void )
{
	return("[1][Librairie RIMM|par Arnaud BERCEGEAY|(c)1999 / FREEWARE !][OK]");
}

/***************************************************************************
  int __OpenRIMDataBase( RIMDB * rimdb, char * path )
  
  rimdb : structure qui sera initialise au retour de cette fonction.
  inutile d'initialiser cette structure avant d'appeler cette fonction
  path  : chemin o charger les RIMs (typiquement "C:\PARX.SYS\RIM\")
  doit se terminer par un "\".
  si path vaut NULL, c'est le chemin "C:\PARX.SYS\RIM\" qui sera utilis.
  
  cette fonction va charger les RIMs trouvs. Toutes les informations
  seront stockes dans la structure RIMDB passe en paramtre.
 ***************************************************************************/

int __OpenRIMDataBase( RIMDB * rimdb, char * path )
{
	long total_malloc_size;
	long rim_size;
	int nb_rim, no_rim;
	char * ptr_reserved1;
	char ** ptr_toc;
	char local_path[128];
	char rim_filename[128];
	DTA *old_dta;
	DTA dtabuffer;
	int retour_dta;
	long Fh;

	/* pas de paramtre : on ne fait rien */
	
	if (rimdb == NULL)
	{
		return -1;
	}
	
	/* initialisation */
	
	rimdb->nb_rim = 0;
	rimdb->reserved1 = NULL;
	if (path)
	{
		strcpy(rimdb->path,path);
		strcpy(local_path,path);
		strcat(local_path,"RIM\\*.RIM");
	}
	else
	{
		strcpy(rimdb->path,"C:\\PARX.SYS\\");
		strcpy(local_path,"C:\\PARX.SYS\\RIM\\*.RIM");
	}
	
	/* estimation de l'espace mmoire ncessaire pour stocker les RIMs */
	
	total_malloc_size = 0L;
	nb_rim = 0;
	
	old_dta = Fgetdta();
	Fsetdta(&dtabuffer);
	retour_dta = Fsfirst(local_path,0x10);
	while (retour_dta>=0)
	{
		if ( !(dtabuffer.d_attrib & (FA_HIDDEN | FA_SYSTEM | FA_LABEL | FA_DIR)) )
		{
			// c'est un fichier RIM
			total_malloc_size += (dtabuffer.d_length + 1L) & ~1L;
			nb_rim++;
		}
		retour_dta = Fsnext();
	}
	
	total_malloc_size += (nb_rim + 1) * 4 ;
	
	/* rservation de l'espace mmoire ncessaire */
	
	rimdb->reserved1 = ldg_Malloc(total_malloc_size);
	if (rimdb->reserved1 == NULL)
	{
		Fsetdta(old_dta);
		return -1;
	}

	/* chargement des RIMs */
	
	ptr_reserved1 = rimdb->reserved1 + ( (nb_rim + 1) * 4 ) ;
	ptr_toc = (char**) rimdb->reserved1;
	no_rim = 0;

	retour_dta = Fsfirst(local_path,0x10);
	while (retour_dta>=0)
	{
		if ( !(dtabuffer.d_attrib & (FA_HIDDEN | FA_SYSTEM | FA_LABEL | FA_DIR)) )
		{
			// c'est un fichier RIM
			// rim_filename contiendra le nom complet du fichier (ou du dossier)
			strcpy(rim_filename,local_path);
			strcpy(rim_filename + strlen(rim_filename) - 5,dtabuffer.d_fname);
			Fh = Fopen(rim_filename,FO_READ);
			if (Fh <= 0L)
				break;
			rim_size = Fseek(0L,Fh,SEEK_END);
			rim_size = (rim_size + 1L) & ~1L ;
			if (ptr_reserved1 + rim_size > rimdb->reserved1 + total_malloc_size)
			{
				Fclose(Fh);
				break;
			}
			Fseek(0L,Fh,SEEK_SET);
			Fread(Fh,rim_size,ptr_reserved1);
			Fclose(Fh);
			*ptr_toc++ = ptr_reserved1;
			ptr_reserved1 += rim_size;
			no_rim++;
			if (no_rim > nb_rim)
				break;
		}
		retour_dta = Fsnext();
	}
	*ptr_toc = 0L;
	Fsetdta(old_dta);
	
	rimdb->nb_rim = no_rim;
	
	return (no_rim);
}

/***************************************************************************
 int __CloseRIMDataBase( RIMDB * rimdb )
 
 libre l'espace mmoire rserve par __OpenRIMDataBase()
 ***************************************************************************/

int __CloseRIMDataBase( RIMDB * rimdb )
{
	/* pas de paramtre : erreur */
	
	if (rimdb == NULL)
		return -1;
	
	/* libration de l'espace mmoire */
	
	ldg_Free(rimdb->reserved1);
	rimdb->reserved1 = NULL;
	rimdb->nb_rim = 0;
	
	return 0;
}

/***************************************************************************
 int __GetRIMInfos( RIMDB * rimdb , int no_rim, RIMINFOS * riminfos)
 
 
 ***************************************************************************/

int __GetRIMInfos( RIMDB * rimdb, long no_rim, RIMINFOS ** riminfos )
{
	static RIMINFOS riminfos_null = {} ;
	
	/* pour viter les erreurs d'adressage chez le client : */
	
	*riminfos = &riminfos_null;
	
	/* pas de paramtre : erreur */
	
	if (rimdb == NULL)
		return -1;
	
	/* on retourne un pointeur sur la structure RIM */
	
	if ((no_rim < 1) || (no_rim > rimdb->nb_rim))
		return -2;
	
	*riminfos = ((RIMINFOS **)(rimdb->reserved1))[no_rim-1];
	
	return 0;
}

/***************************************************************************
 int __DoRIMConfiguration( RIMDB * rimdb , int no_rim)
 
 
 ***************************************************************************/

int __DoRIMConfiguration( RIMDB * rimdb, long no_rim )
{
	RIMINFOS * riminfos;
	int (*Do_Param)(int code, char * path);
	
	/* pas de paramtre : erreur */
	
	if (rimdb == NULL)
		return -3;
	
	/* on retourne un pointeur sur la structure RIM */
	
	if ((no_rim < 1) || (no_rim > rimdb->nb_rim))
		return -4;
	
	riminfos = ((RIMINFOS **)(rimdb->reserved1))[no_rim-1];
	if (riminfos->Configurable)
	{
		(long)Do_Param = (long)(riminfos) + 68L ;
		return (*Do_Param)(1,rimdb->path);
	}
	
	return -5;
}

/***************************************************************************
 int __OpenImg( RIMDB * rimdb, long no_rim, char *filename, RIMIMG * rimimg)
 
 ***************************************************************************/

int __OpenImg( RIMDB * rimdb, long force_no_rim, char *filename, RIMIMG * rimimg)
{
	long no_premier_rim ;
	long no_dernier_rim ;
	long no_rim;
	RIMINFOS * riminfos;
	long (*Test_File)(short code, char * adr_src, long size_src,
	                  long size_file, unsigned long ext, MFDB * adr_mfdb);
	char buffer[1024];
	long taille_buffer_lu;
	long taille_fichier;
	unsigned long ext;
	long code_retour;
	MFDB mfdb;
	char *s;
	
	rimimg->no_rim = 0;
	
	if (filename)
	{
		rimimg->Fh = Fopen(filename,FO_READ);
		if (rimimg->Fh <= 0)
			return -1;
		/* on va jusqu' la fin pour rcuprer la taille du fichier */
		taille_fichier = (long)Fseek(0L,rimimg->Fh,SEEK_END);
		/* et on revient au dbut... */
		Fseek(0L,rimimg->Fh,0);
		/* ... pour y charger les 1024 premiers octets */
		taille_buffer_lu = (long)Fread(rimimg->Fh,1024L,buffer);
		/* affectation de ext */
		s    = &filename[strlen(filename)-1];
		ext  = ((u_long)(*s)       ) & 0x000000FFL; s--;
		ext |= ((u_long)(*s) << 8  ) & 0x0000FF00L; s--;
		ext |= ((u_long)(*s) << 16 ) & 0x00FF0000L; s--;
		ext |= ((u_long)(*s) << 24 ) & 0xFF000000L;		
	}
	else // RIM gnrant
	{
		rimimg->Fh = 0;
		/* affectation de ext */
		ext = '_RIM';
	}
	
	if (force_no_rim > 0)
	{
		no_premier_rim = force_no_rim;
		no_dernier_rim = force_no_rim;
	}
	else
	{
		no_premier_rim = 1;
		no_dernier_rim = rimdb->nb_rim;
	}
	
	for (no_rim=no_premier_rim; no_rim<=no_dernier_rim; no_rim++)
	{
		/* on localise la fonction Test_File du RIM */
		riminfos = ((RIMINFOS **)(rimdb->reserved1))[no_rim-1];
		(long)Test_File = (long)(riminfos) + 56L ;
		
		/* on initialise les champs de la mfdb */
		mfdb.fd_addr     = NULL;
		mfdb.fd_w        = rimimg->width;
		mfdb.fd_h        = rimimg->height;
		mfdb.fd_wdwidth  = (mfdb.fd_w / 16) + ((mfdb.fd_w % 16) != 0);
		mfdb.fd_stand    = 0;
		mfdb.fd_nplanes  = rimimg->nbplans;
		mfdb.fd_r1       = 0;
		mfdb.fd_r2       = 0;
		mfdb.fd_r3       = rimimg->Fh;
		
		if (rimimg->Fh)
		{
			Fseek(0L,rimimg->Fh,0);
			code_retour = (*Test_File)(0,buffer,taille_buffer_lu,taille_fichier,ext,&mfdb);
		}
		else /* c'est donc un RIM gnrant */
		{
			code_retour = (*Test_File)(0,0L,0L,0L,ext,&mfdb);
		}
		
		if ((code_retour == 2) || (code_retour == 3))
			break;
	}
	
	if ((code_retour == 2) || (code_retour == 3))
	{
		rimimg->no_rim  = no_rim;
		rimimg->width   = mfdb.fd_w ;
		rimimg->height  = mfdb.fd_h ;
		rimimg->nbplans = mfdb.fd_nplanes ;
		return no_rim;
	}
	
	return 0;
}

/***************************************************************************
 int __CloseImg( RIMIMG * rimimg)
 
 ***************************************************************************/

int __CloseImg( RIMIMG * rimimg )
{
	if (rimimg->Fh)
		Fclose(rimimg->Fh);
	
	return 0;
}

/***************************************************************************
  Boucle principale de la librairie...
 ***************************************************************************/

int main( void) {
	ldg_init( Ldg);
	return 0;
}

/* EOF */
