/*-
 * Copyright (c) 1989 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	@(#)slcompress.c	7.7 (Berkeley) 5/7/91
 */

/*
 * Routines to compress and uncompess tcp packets (for transmission
 * over low speed serial lines.
 *
 * Van Jacobson (van@helios.ee.lbl.gov), Dec 31, 1989:
 *	- Initial distribution.
 *
 * static char rcsid[] =
 * "$Header: slcompress.c,v 1.19 89/12/31 08:52:59 van Exp $";
 */

#include "netinfo.h"
#include "kerbind.h"
#include "in.h"
#include "ip.h"
#include "tcp.h"
#include "buf.h"
#include "slcompress.h"

#ifndef SL_NO_STATS
#define INCR(counter) ++comp->counter;
#else
#define INCR(counter)
#endif

extern void	*memcpy (void *, const void *, unsigned long);
extern int	memcmp (const void *, const void *, unsigned long);
extern void	*bzero (void *, unsigned long);

#define BCMP(p1,p2,n)		memcmp((p2), (p1), (u_long)(n))
#define BCOPY(p1,p2,n)		memcpy((p2), (p1), (u_long)(n))
#define ovbcopy(p1,p2,n)	memcpy((p2), (p1), (u_long)(n))

struct slcompress *
slc_init(void)
{
	struct slcompress *comp;
	struct cstate *tstate;
	u_short i;

	comp = kmalloc (sizeof (struct slcompress));
	if (comp == 0) {
		DEBUG (("sl_compress_init: no mem"));
		return 0;
	}
	tstate = comp->tstate;
	bzero (comp, sizeof(*comp));
	for (i = MAX_STATES - 1; i > 0; --i) {
		tstate[i].cs_id = i;
		tstate[i].cs_next = &tstate[i - 1];
	}
	tstate[0].cs_next = &tstate[MAX_STATES - 1];
	tstate[0].cs_id = 0;
	comp->last_cs = &tstate[0];
	comp->last_recv = 255;
	comp->last_xmit = 255;
	return comp;
}

void
slc_free (comp)
	struct slcompress *comp;
{
	if (comp)
		kfree (comp);
}

/* ENCODE encodes a number that is known to be non-zero.  ENCODEZ
 * checks for zero (since zero has to be encoded in the long, 3 byte
 * form).
 */
#define ENCODE(n) { \
	if ((u_short)(n) >= 256) { \
		*cp++ = 0; \
		cp[1] = (n); \
		cp[0] = (n) >> 8; \
		cp += 2; \
	} else { \
		*cp++ = (n); \
	} \
}
#define ENCODEZ(n) { \
	if ((u_short)(n) >= 256 || (u_short)(n) == 0) { \
		*cp++ = 0; \
		cp[1] = (n); \
		cp[0] = (n) >> 8; \
		cp += 2; \
	} else { \
		*cp++ = (n); \
	} \
}

#define DECODEL(f) { \
	if (*cp == 0) {\
		(f) = htonl(ntohl(f) + ((cp[1] << 8) | cp[2])); \
		cp += 3; \
	} else { \
		(f) = htonl(ntohl(f) + (u_long)*cp++); \
	} \
}

#define DECODES(f) { \
	if (*cp == 0) {\
		(f) = htons(ntohs(f) + ((cp[1] << 8) | cp[2])); \
		cp += 3; \
	} else { \
		(f) = htons(ntohs(f) + (u_long)*cp++); \
	} \
}

#define DECODEU(f) { \
	if (*cp == 0) {\
		(f) = htons((cp[1] << 8) | cp[2]); \
		cp += 3; \
	} else { \
		(f) = htons((u_long)*cp++); \
	} \
}

u_char
slc_compress (b, comp, compress_cid)
	BUF *b;
	struct slcompress *comp;
	long compress_cid;
{
	struct cstate *cs = comp->last_cs->cs_next;
	struct ip_dgram *ip = (struct ip_dgram *)b->dstart;
	u_long hlen = ip->hdrlen;
	struct tcp_dgram *oth;
	struct tcp_dgram *th;
	u_long deltaS, deltaA;
	u_long changes = 0;
	u_char new_seq[16];
	u_char *cp = new_seq;

	/*
	 * Bail if this is an IP fragment or if the TCP packet isn't
	 * `compressible' (i.e., ACK isn't set or some other control bit is
	 * set).  (We assume that the caller has already made sure the
	 * packet is IP proto TCP).
	 */
	if (ip->fragoff & (IP_MF|IP_FRAGOFF) || b->dend - b->dstart < 40)
		return (TYPE_IP);

	th = (struct tcp_dgram *) IP_DATA (b);
	if ((th->flags & (TCPF_SYN|TCPF_FIN|TCPF_RST|TCPF_ACK)) != TCPF_ACK)
		return (TYPE_IP);
	/*
	 * Packet is compressible -- we're going to send either a
	 * COMPRESSED_TCP or UNCOMPRESSED_TCP packet.  Either way we need
	 * to locate (or create) the connection state.  Special case the
	 * most recently used connection since it's most likely to be used
	 * again & we don't have to do any reordering if it's used.
	 */
	INCR (sls_packets);
	if (ip->saddr != cs->cs_ip.saddr ||
	    ip->daddr != cs->cs_ip.daddr ||
	    *(long *)th != ((long *)&cs->cs_ip)[cs->cs_ip.hdrlen]) {
		/*
		 * Wasn't the first -- search for it.
		 *
		 * States are kept in a circularly linked list with
		 * last_cs pointing to the end of the list.  The
		 * list is kept in lru order by moving a state to the
		 * head of the list whenever it is referenced.  Since
		 * the list is short and, empirically, the connection
		 * we want is almost always near the front, we locate
		 * states via linear search.  If we don't find a state
		 * for the datagram, the oldest state is (re-)used.
		 */
		struct cstate *lcs;
		struct cstate *lastcs = comp->last_cs;

		do {
			lcs = cs; cs = cs->cs_next;
			INCR (sls_searches);
			if (ip->saddr == cs->cs_ip.saddr &&
			    ip->daddr == cs->cs_ip.daddr &&
			    *(long *)th == ((long *)&cs->cs_ip)[cs->cs_ip.hdrlen])
				goto found;
		} while (cs != lastcs);

		/*
		 * Didn't find it -- re-use oldest cstate.  Send an
		 * uncompressed packet that tells the other side what
		 * connection number we're using for this conversation.
		 * Note that since the state list is circular, the oldest
		 * state points to the newest and we only need to set
		 * last_cs to update the lru linkage.
		 */
		INCR (sls_misses);
		comp->last_cs = lcs;
		hlen += th->hdrlen;
		hlen <<= 2;
		goto uncompressed;

	found:
		/*
		 * Found it -- move to the front on the connection list.
		 */
		if (cs == lastcs)
			comp->last_cs = lcs;
		else {
			lcs->cs_next = cs->cs_next;
			cs->cs_next = lastcs->cs_next;
			lastcs->cs_next = cs;
		}
	}

	/*
	 * Make sure that only what we expect to change changed. The first
	 * line of the `if' checks the IP protocol version, header length &
	 * type of service.  The 2nd line checks the "Don't fragment" bit.
	 * The 3rd line checks the time-to-live and protocol (the protocol
	 * check is unnecessary but costless).  The 4th line checks the TCP
	 * header length.  The 5th line checks IP options, if any.  The 6th
	 * line checks TCP options, if any.  If any of these things are
	 * different between the previous & current datagram, we send the
	 * current datagram `uncompressed'.
	 */
	oth = (struct tcp_dgram *)&((long *)&cs->cs_ip)[hlen];
	deltaS = hlen;
	hlen += th->hdrlen;
	hlen <<= 2;

	if (((u_short *)ip)[0] != ((u_short *)&cs->cs_ip)[0] ||
	    ((u_short *)ip)[3] != ((u_short *)&cs->cs_ip)[3] ||
	    ((u_short *)ip)[4] != ((u_short *)&cs->cs_ip)[4] ||
	    th->hdrlen != oth->hdrlen ||
	    (deltaS > 5 &&
	     BCMP(ip + 1, &cs->cs_ip + 1, (deltaS - 5) << 2)) ||
	    (th->hdrlen > 5 &&
	     BCMP(th + 1, oth + 1, (th->hdrlen - 5) << 2)))
		goto uncompressed;

	/*
	 * Figure out which of the changing[IF_PRIORITIES-1])
			return 1;
		if (ppp->rsel == 0) {
			ppp->rsel = proc;
			return 0;
		}
		return 2;

	case O_WRONLY:
		if (ppp->nif->snd.qlen < ppp->nif->snd.maxqlen)
			return 1;
		if (ppp->wsel == 0) {
			ppp->wsel = proc;
			return 0;
		}
		return 2;

	case O_RDWR:
		if (ppp->xsel == 0) {
			ppp->xsel = proc;
			return 0;
		}
		return 2;
	}
	return 0;
}

static void
pppdev_unselect	(fp, proc, mode)
	FILEPTR *fp;
	long proc;
	short mode;
{
	struct ppp *ppp = &ppp_priv[fp->fc.aux];

	switch (mode) {
	case O_RDONLY:
		if (proc == ppp->rsel)
			ppp->rsel = 0;
		break;

	case O_WRONLY:
		if (proc == ppp->wsel)
			ppp->wsel = 0;
		break;

	case O_RDWR:
		if (proc == ppp->xsel)
			ppp->xsel = 0;
		break;
	}
}
                                                                                                                                                                                                                                                                                                               
Some notes on using PPP.

The PPP driver is part of slip.xif. So if you want to connect to another host
using PPP, copy slip.xif into /c/mint or /c/multitos.

To establish a PPP connection you will need the programm "pppd". Look for
the file ppp-212.tgz on you favorite Atari ftp server.
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               This file contains some notes about the PLIP driver (plip.xif), the
protocol it uses and how to build a PLIP cable.

What is PLIP ?
--------------

PLIP is a non standard protocol to transfer IP packets over bidirectional
parallel (centronics) lines. It is similar to SLIP in that it uses the
same algorithm to encode/decode outgoing/incoming data.

It is probably incompatible with all other so called protocols, because
it is my invention :)

I did not use a more `standard' protocol, because the Atari centronics
port lacks ACKNOWLEDGE, PAPEROUT, SELECT, FORMFEED (and probably other)
signals, some of which are used by the other protocols.

If you want to use PLIP you will have to build a cable connector
yourself. Someone told me about a Turbo-Laplink connecter that should
be compatible with the cable described below, but I did no see one 'til
now.

PLIP's main advantage over SLIP is that parallel lines are usually much
faster than the standard (not highspeed) serial lines.
The throughput depends very much on the speed of the computers you use,
because handshaking is done in software and can be expected to be at
least 18 kB/sec for two slow 8MHz ST's connected via PLIP.

Because it is much cheaper then ethernet cards, I could also think of
building tiny networks (3-4 hosts) using PLIP. When using more than 2
machines, some of them must have two or more parallel ports (and act
as gateways).

PLIP cable
----------

To build a PLIP cable, you need two male DB-25 connectors and some metres
of >= 11 conductor cable.

Note that the cable should not be longer than about 10 metres, because
the parallel port uses standard TTL signals.

The wiring is:

	pin (one end)	pin (other end)
	1  (STROBE)	11 (BUSY)
	2  (D0)		2  (D0)
	3  (D1)		3  (D1)
	4  (D2)		4  (D2)
	5  (D3)		5  (D3)
	6  (D4)		6  (D4)
	7  (D5)		7  (D5)
	8  (D6)		8  (D6)
	9  (D7)		9  (D7)
	11 (BUSY)	1  (STROBE)
	25 (GROUND)	25 (GROUND)

Pins 18-24 are extra grounds.

Build one yourself and have some fun.
You can buy the necessary incredients in computer stores for a few bucks.

Nonstandard PLIP protocol
-------------------------

For the ones interested in the protocol, here we go:

The data is encoded using the SLIP protocol, which defines the following
special characters:

END = 192 (decimal)
ESC = 219 (decimal)
ESC_ESC = 221 (decimal)
ESC_END = 220 (decimal)

Every character `c' of the IP packet is sent using the following algorithm:

if c == END then send ESC followed by ESC_END
else if c == ESC then send ESC followed by ESC_ESC
else send c

After all bytes have been sent this way, a single END character is sent to
indicate the end of the packet. Optionally the sender may send an END cha-
racter at the beginning of each packet. The PLIP receiver will drop the
resulting zero length packet.

Every byte is sent over the parallel line in the following way (note that
one ends STROBE is connected with the other ends BUSY):

sender:			receiver:

DATA OUT		WAIT ACK
SEND ACK		DATA IN
WAIT ACK		SEND ACK

The shortcuts have the following meaning:
DATA OUT/IN	--	write/read the byte to/from the port
SEND ACK	--	send an acknowledge (ie drop the STROBE wire for
			some micro seconds).
WAIT ACK	--	wait until an acknowledge is received (BUSY has
			dropped for some micro seconds).

Note that an interrupt is generated when the BUSY wire drops. Thus the
first byte sent will trigger the BUSY interrupt on the remote side, ie
transmission of packets is interrupt driven (no polling).

After this first interrupt the receiver disables the BUSY interrupt and
receives the whole packet polling the BUSY wire. Note that this is done
at low interrupt priority so that other interrupts can still be handled.

Waiting for acknowledgements is implemented using an adaptive timeout
mechanism.

Note that the SLIP encoding makes sure that the receiver/sender may get
in sync with the sender/receiver again after one end has timed out.

Also note that the character transmission protocoll used will let one
end time out if both ends try to send at the same time, ie collisions
can be detected.

Establishing a PLIP connection
------------------------------

This is essentially the same as establishing a SLIP connection with the
old SLIP driver (oslip.xif). Read the file README.OSLIP. The PLIP interfaces
are called pl0, pl1, ... Currently only pl0 is implemented (corresponds to
the standard Atari centronics port).


Happy PLIP'ing -- Kay.
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                            