;----------------------------------------------------------------------------
;File name:	CEN_PLEP.S			Revision date:	1999.10.19
;Creator:	Ulf Ronald Andersson		Creation date:	1997.08.18
;(c)1997 by:	Ulf Ronald Andersson		All rights reserved
;Feedback to:	dlanor@oden.se			Released as FREEWARE
;----------------------------------------------------------------------------
	include	RA_TOS.I
	include	RA_JAR.I
	include	RA_XB.I
	include	RA_RAM.I
;-------------------------------------
	include	sting\PORT.I
	include	sting\LAYER.I
	include	sting\TRANSPRT.I
;----------------------------------------------------------------------------
	output	.PRG
;----------------------------------------------------------------------------
XB_cpu_mode	set	1	;never test mode or call Super in XB-macros
USE_TIMER	=	0	;1 enables 'my_timer_func', 0 disables it.
USE_LOOPING	=	1	;1 enables multipacket sending, 0 disables it
;----------------------------------------------------------------------------
M_YEAR	=	1999
M_MONTH	=	10
M_DAY	=	19
;-------------------------------------
M_TITLE	MACRO
	dc.b	'Centr. PLEP'
	ENDM
M_VERSION	MACRO
	dc.b	'01.00'
	ENDM
M_AUTHOR	MACRO
	dc.b	'Ronald Andersson'
	ENDM
;-------------------------------------
	struct	PLEP		;queue structure for raw datagram storage
	struc_p		PLEP_next	;-> next PLEP datagram
	uint16		PLEP_length	;length of raw datagram
	d_alias		PLEP_data	;data of varying size follows
	d_end	PLEP
;-------------------------------------
;	PLEP port constants
;-------------------------------------
MAX_mtu		=	4*Kb		;mtu may not be raised above this level
MAX_buffers	=	10		;max number of buffers used in plep_tx_q
;
MAX_portwork	=	10/5		;For high level STinG ports
MAX_delay	=	100/5		;tween-interrupt delays must be less
WATCH_max	=	64		;max number of loops per interrupt
WATCH_step	=	256/WATCH_max	;watch count step per interrupt loop
;
RAM_chunk	=	2*MAX_buffers*(MAX_mtu+sizeof_PLEP+2)
;-------------------------------------
; Cen_Plip character codes
;-------------------------------------
cp_ERRL		= $00	;ix F  eq AF	;ERRor Lo, request refused
cp_ERRH		= $FF	;ix E  eq AE	;ERRor Hi, spurious error
cp_DREQ		= $FE	;ix D  eq AD	;Data Request from receiver
cp_DBEG		= $FD	;ix C  eq AC	;Data BEGin (ungated) from sender
cp_DEND		= $FC	;ix B  eq AB	;Data END from sender
cp_DOFF		= $FB	;ix A  eq AA	;Data OFFset from sender
cp_DACK		= $FA	;ix 9  eq A9	;Data ACKnowledge from receiver
cp_res0		= $F9	;ix 8  eq A8
cp_sel8		= $F8	;ix 7  eq A7	;gated select 8 (DBEG for unit 8)
cp_sel7		= $F7	;ix 6  eq A6	;gated select 7 (DBEG for unit 7)
cp_sel6		= $F6	;ix 5  eq A5	;gated select 6 (DBEG for unit 6)
cp_sel5		= $F5	;ix 4  eq A4	;gated select 5 (DBEG for unit 5)
cp_sel4		= $F4	;ix 3  eq A3	;gated select 4 (DBEG for unit 4)
cp_sel3		= $F3	;ix 2  eq A2	;gated select 3 (DBEG for unit 3)
cp_sel2		= $F2	;ix 1  eq A1	;gated select 2 (DBEG for unit 2)
cp_sel1		= $F1	;ix 0  eq A0	;gated select 1 (DBEG for unit 1)
;-------
cp_offset	= $80			;encoded char offset for cp_DOFF
cp_base		= $A0			;decoded char equivalent to cp_reso when encoded
cp_DOFF_dec	= $FF&(cp_base-cp_sel1-cp_offset)
cp_DOFF_enc	= $FF&(cp_sel1+cp_offset)
;----------------------------------------------------------------------------
;Start of:	STX program
;----------------------------------------------------------------------------
	SECTION	TEXT
;----------------------------------------------------------------------------
text_start:
basepage	=	(text_start-$100)
;----------------------------------------------------------------------------
start:
	bra	start_1
;----------------------------------------------------------------------------
;Start of:	Resident STX data
;----------------------------------------------------------------------------
my_port:
	dc.l	port_name_s	;prt_des_name
	dc.w	L_PAR_PTP	;prt_des_type
	dc.w	0		;prt_des_active		Activation flag
	dc.l	0		;prt_des_flags
	dc.l	0		;prt_des_ip_addr	IP_number
	dc.l	-1		;prt_des_sub_mask
	dc.w	MAX_mtu		;prt_des_mtu
	dc.w	MAX_mtu		;prt_des_max_mtu
	dc.l	0		;prt_des_stat_sd_data
	dc.l	0		;prt_des_send		->Tx queue
	dc.l	0		;prt_des_stat_rcv_data
	dc.l	0		;prt_des_receive	->Rx queue
	dc.w	0		;prt_des_stat_dropped
	dc.l	my_driver	;prt_des_driver		->driver struct
	dc.l	0		;prt_des_next		->next port
;------------------------------------
my_driver:
	dc.l	my_set_state	;drv_des_set_state
	dc.l	my_cntrl	;drv_des_cntrl
	dc.l	my_send		;drv_des_send
	dc.l	my_receive	;drv_des_receive
	dc.l	driver_name_s	;drv_des_name
	dc.l	version_s	;drv_des_version
	dc.w	((M_YEAR-1980)<<9)|(M_MONTH<<5)|M_DAY
	dc.l	author_s	;drv_des_author
	dc.l	0		;drv_des_next		->next driver
basepage_p:
	dc.l	basepage	;drv_des_basepage	->basepage of self
;------------------------------------
port_name_s:
driver_name_s:
	M_TITLE
	dc.b	NUL
	even
;------------------------------------
version_s:
	M_VERSION
	dc.b	NUL
	even
;------------------------------------
author_s:
	M_AUTHOR
	dc.b	NUL
	even
;------------------------------------
sting_drivers:	ds.l	1	;DRV_LIST	*sting_drivers;
tpl:		ds.l	1	;TPL		*tpl;
stx:		ds.l	1	;STX		*stx;
;------------------------------------
state_vector:	dc.l	cv_Dummy_State	;-> Interrupt state routine
;------------------------------------
state_t1:	dc.l	0	;long tick for state machine at work start
state_t2:	dc.l	0	;long tick for state machine after work
;
plep_open_f:	dc.w	0	;flags port open for traffic
;
plep_rx_q:	dc.l	0	;-> queue of raw plep datagrams received
recv_buf_p:	dc.l	0	;-> current raw plep datagram being received
recv_pos_p:	dc.l	0	;-> current receive position in datagram
recv_max_p:	dc.l	0	;-> receive position of datagram end
;
plep_tx_q_ct:	dc.w	0	;counts buffers in plep_tx_q
plep_tx_q:	dc.l	0	;-> queue of raw plep datagrams to send
send_buf_p:	dc.l	0	;-> current raw plep datagram to send
send_pos_p:	dc.l	0	;-> current send position in datagram
send_max_p:	dc.l	0	;-> send position of datagram end
;------------------------------------
plep_crc:	dc.w	0
plep_crc_ix:	dc.w	0
DOFF_char:	dc.w	0
;-------------------------------------
RX_ATT_pair:			;both bytes below as one word
RX_ATT_flag:	dc.b	$FF	;byte == 0 if RX_ATT_data valid
RX_ATT_data:	dc.b	$00	;byte == last received ATT data 
ref_perm_f:	dc.b	$00	;byte == $FF/$00 flagging permission to refuse reception (to send)
		dc.b	$00	;unused flag
		EVEN
IO_progress_f:			;both bytes below, as one word
RX_progress_f:	dc.b	0
TX_progress_f:	dc.b	0
;------------------------------------
plep_regs:	ds.l	16
;------------------------------------
byte_watch:	dc.b	0
orig_imrb:	dc.b	0
orig_ierb:	dc.b	0
orig_aer:	dc.b	0
;------------------------------------
	IFNE	USE_LOOPING
port_time:	ds.w	1
	ENDC	;USE_LOOPING
;----------------------------------------------------------------------------
;End of:	Resident STX data
;----------------------------------------------------------------------------
;Start of:	Interrupt state machine macros
;----------------------------------------------------------------------------
def_state	MACRO	state
cv_\1:
	ENDM
;-------------------------------------
set_state	MACRO	state
	move.l	#cv_\1,state_vector
	ENDM
;-------------------------------------
loop_state	MACRO
	bra	state_loop
	ENDM
;-------------------------------------
goto_state	MACRO	state
	set_state	\1
	loop_state
	ENDM
;-------------------------------------
new_state	MACRO	state
	goto_state	\1
	def_state	\1
	ENDM
;-------------------------------------
pl_timeout	MACRO	ref_tick,limit,destination
	move.l		(_hz_200).w,d0
	sub.l		\1,d0
	cmp.l		\2,d0
	bhs.\0		\3
	ENDM
;----------------------------------------------------------------------------
;End of:	Interrupt state machine macros
;----------------------------------------------------------------------------
;-------------------------------------
;	PLEP port macros
;-------------------------------------
;macro pl_interrupt is used to enable system interrupts temporarily,
;mainly to allow the system timers to advance, but affects all interrupts
;allowed by the mask in given in the 'intmask' argument.
;
pl_interrupt	MACRO	intmask
	move		\1,sr		;reenable old interrupts
	or		#$0700,sr	;disable all interrupts
	ENDM	;pl_interrupt
;-------------------------------------
pl_strobe_LO	MACRO	temp
	move.b		#14,(hw_psgsel).w
	move.b		(hw_psgrd).w,\1
	bclr		#5,\1
	move.b		\1,(hw_psgwr).w
	ENDM	;pl_strobe_LO
;-------------------------------------
pl_strobe_HI	MACRO	temp
	move.b		#14,(hw_psgsel).w
	move.b		(hw_psgrd).w,\1
	bset		#5,\1
	move.b		\1,(hw_psgwr).w
	ENDM	;pl_strobe_HI
;-------------------------------------
pl_send_ACK	MACRO	temp1,temp2
	move.b		#14,(hw_psgsel).w	;select port A
	move.b		(hw_psgrd).w,\1		;get port A data
	move.b		\1,\2
	bclr		#5,\1
	bset		#5,\2
	move.b		\1,(hw_psgwr).w		;lower strobe
	move.b		\1,(hw_psgwr).w		;lower strobe
	move.b		\2,(hw_psgwr).w		;raise strobe
	move.b		\2,(hw_psgwr).w		;raise strobe
	ENDM	;pl_send_ACK
;-------------------------------------
pl_dir_in	MACRO	temp
	move.b		#7,(hw_psgsel).w
	move.b		(hw_psgrd).w,\1
	bclr		#7,\1
	move.b		\1,(hw_psgwr).w
	ENDM	;pl_dir_in
;-------------------------------------
pl_dir_out	MACRO	temp
	move.b		#7,(hw_psgsel).w
	move.b		(hw_psgrd).w,\1
	bset		#7,\1
	move.b		\1,(hw_psgwr).w
	ENDM	;pl_dir_out
;-------------------------------------
pl_recv_byte	MACRO	dest
	move.b		#15,(hw_psgsel).w	;select port B
	move.b		(hw_psgrd).w,\1		;dest.b = data from port B
	ENDM	;pl_recv_byte
;-------------------------------------
pl_send_byte	MACRO	data
	move.b		#15,(hw_psgsel).w	;select port B
	move.b		\1,(hw_psgwr).w		;write data byte to port B
	ENDM	;pl_send_byte
;-------------------------------------
pl_test_busy	MACRO
	btst		#0,(hw_gpip).w
	ENDM	;pl_test_busy
;-------------------------------------
pl_test_ACK	MACRO
	btst		#0,(hw_iprb).w
	ENDM	;pl_test_ACK
;-------------------------------------
pl_take_ACK	MACRO
	btst		#0,(hw_iprb).w
	beq.s		.done_\@
	move.b		#$FE,(hw_iprb).w
.done_\@:
	ENDM	;pl_take_ACK
;-------------------------------------
pl_clear_ACK	MACRO
	move.b		#$FE,(hw_iprb).w
	ENDM	;pl_clear_ACK
;-------------------------------------
pl_cli	MACRO
	bclr		#0,(hw_imrb).w		;disable 'busy' interrupt in MFP
	ENDM	;pl_cli
;-------------------------------------
pl_sti	MACRO
	bset		#0,(hw_imrb).w		;enable 'busy' interrupt in MFP
	ENDM	;pl_sti
;-------------------------------------
pl_send_ATT	MACRO	temp,data
	pl_dir_out	\1
	pl_send_byte	\2
	pl_strobe_LO	\1
	ENDM	;pl_send_ATT
;-------------------------------------
pl_crc	MACRO
;-------------------------------------
	move	plep_crc(pc),d1	;4
	rol.w	#8,d1		;2
	eor.b	d0,d1		;2
	move.b	d1,d0		;2
	lsr.b	#4,d0		;2
	eor.b	d0,d1		;2
	move.b	d1,d0		;2
	lsl.w	#7,d0		;2
	eor.b	d1,d0		;2
	lsl.w	#5,d0		;2
	eor.w	d0,d1		;2
	move	d1,plep_crc	;2
	ENDM
;-------------------------------------
pl_abort_send	MACRO
	move.l		send_buf_p(pc),d0
	ble.s		.exit_abort_send
	move.l		d0,a0			;a0 -> aborted plep datagram
	move.l		plep_tx_q(pc),(a0)	;link datagram as root of queue
	move.l		a0,plep_tx_q		;and store it as queue root
	addq		#1,plep_tx_q_ct		;increment buffer count
	clr.l		send_buf_p
	addq		#1,prt_des_stat_dropped+my_port	;increment drop count
.exit_abort_send:
	pl_strobe_HI	d0			;normalize strobe
	pl_dir_in	d0			;set port for input
	sf		TX_progress_f
	set_state	Init_Recv
	ENDM	;pl_abort_send
;-------------------------------------
pl_abort_recv	MACRO
	tst.b		RX_progress_f		;are we in an early mode ?
	beq.s		.skip_dropping
	addq		#1,prt_des_stat_dropped+my_port	;increment drop count
.skip_dropping:
	pl_strobe_HI	d0			;normalize strobe
	pl_dir_in	d0			;set port for input
	sf		RX_progress_f
	set_state	Init_Recv
	ENDM	;pl_abort_recv
;----------------------------------------------------------------------------
;Start of:	Port driver functions
;----------------------------------------------------------------------------
;void	my_timer_func(void);
;
	IFNE	USE_TIMER
my_timer:
	rts
	ENDC	;USE_TIMER
;
;end of my_timer_func
;----------------------------------------------------------------------------
my_cntrl:
	link	a6,#0
	movem.l	d3-d5/a2-a5,-(sp)
	moveq	#E_PARAMETER,d0			;prep E_PARAMETER error for bad port arg
	lea	my_port(pc),a5			;a5 -> my_port
	cmpa.l	8(a6),a5			;port argument correct ?
	bne.s	.exit				;exit if port argument incorrect
	moveq	#E_FNAVAIL,d0			;prep E_FNAVAIL error code for unsupported opcodes
	move	16(a6),d2			;d2 = cntrl_port opcode
	lea	my_CTL_func_t(pc),a0		;a0 -> CTL function table
.seek_CTL_loop:				;loop start for seeking opcode in table
	move	(a0)+,d1			;d1 = CTL opcode from table
	ble.s	.exit				;exit with E_FNAVAIL if opcode unsupported
	move.l	(a0)+,a1			;a0 = CTL routine ptr from table
	cmp	d2,d1				;correct opcode now ?
	bne.s	.seek_CTL_loop			;loop back to seek correct opcode
.found_CTL:
	move.l	12(a6),d0			;d0 = long argument
	move.l	d0,a0				;a0 = long argument
	jsr	(a1)				;call a CTL subroutine
.exit:
	movem.l	(sp)+,d3-d5/a2-a5
	unlk	a6
	rts
;
def_CTL	MACRO	opcode,routine
	dc.w	\1
	dc.l	\2
	ENDM
;
my_CTL_func_t:
	def_CTL	CTL_GENERIC_SET_IP,my_SET_IP
	def_CTL	CTL_GENERIC_GET_IP,my_GET_IP
	def_CTL	CTL_GENERIC_SET_MTU,my_SET_MTU
	def_CTL	CTL_GENERIC_GET_MTU,my_GET_MTU
	def_CTL	CTL_GENERIC_GET_MMTU,my_GET_MMTU
	def_CTL	CTL_GENERIC_GET_TYPE,my_GET_TYPE
	def_CTL	CTL_GENERIC_GET_STAT,my_GET_STAT
	def_CTL	CTL_GENERIC_CLR_STAT,my_CLR_STAT
	dc.w	0
;
;end of my_cntrl
;----------------------------------------------------------------------------
;Start of:	CTL subroutines
;----------------------------------------------------------------------------
my_SET_IP:
	move.l	d0,prt_des_ip_addr(a5)
	moveq	#E_NORMAL,d0
	rts
;------------------------------------
my_GET_IP:
	move.l	prt_des_ip_addr(a5),(a0)
	moveq	#E_NORMAL,d0
	rts
;------------------------------------
my_SET_MTU:
	clr.l	d1
	move	prt_des_max_mtu(a5),d1
	cmp.l	d1,d0
	bhi.s	E_PARAMETER_rts
	cmp	#48,d0
	bhs.s	accept_set_mtu
E_PARAMETER_rts:
	moveq	#E_PARAMETER,d0
	rts
;
accept_set_mtu:
	move	d0,prt_des_mtu(a5)
	moveq	#E_NORMAL,d0
	rts
;------------------------------------
my_GET_MTU:
	move	prt_des_mtu(a5),(a0)
	moveq	#E_NORMAL,d0
	rts
;------------------------------------
my_GET_MMTU:
	move	prt_des_max_mtu(a5),(a0)
	moveq	#E_NORMAL,d0
	rts
;------------------------------------
my_GET_TYPE:
	move	prt_des_type(a5),(a0)
	moveq	#E_NORMAL,d0
	rts
;------------------------------------
my_GET_STAT:
	clr.l	d0
	move	prt_des_stat_dropped(a5),d0
	move.l	d0,(a0)+
	move.l	prt_des_stat_sd_data(a5),(a0)+
	move.l	prt_des_stat_rcv_data(a5),(a0)+
	moveq	#E_NORMAL,d0
	rts
;------------------------------------
my_CLR_STAT:
	clr	prt_des_stat_dropped(a5)
	clr.l	prt_des_stat_sd_data(a5)
	clr.l	prt_des_stat_rcv_data(a5)
	moveq	#E_NORMAL,d0
	rts
;----------------------------------------------------------------------------
;End of:	CTL subroutines
;----------------------------------------------------------------------------
;void	my_send(PORT *port);
;
my_send:
	move.l		my_port+prt_des_send(pc),d0	;anything to send
	ble.s		.exit_direct			;exit if none to send
	cmpi.l		#my_port,4(sp)			;argument == my_port ?
	bne.s		.exit_direct			;exit if argument incorrect
	movem.l		d2-d3/a2-a5,-(sp)		;protect entry regs
	lea		my_port+prt_des_send(pc),a5	;a5 ->Tx queue ptr
	move.l		d0,d3				;d3 ->Tx queue
;
	IFNE	USE_LOOPING
	move		(_hz_200+2).w,port_time		;find work start time
	ENDC	;USE_LOOPING
.send_loop:
	move.l		d3,a4				;a4 -> Tx dgram
	move.l		IPDG_next(a4),(a5)		;unlink chain
	check_dgram_ttl	(a4)				;check TTL
	tst		d0
	bpl.s		.send_it			;continue if ok
	addq		#1,prt_des_stat_dropped-prt_des_send(a5)
	bra.s		.skip_it
;
.send_it:
	bsr		pass_send_pkt
	bgt.s		.count_sent_data
	move.l		(a5),IPDG_next(a4)	;relink to chain head
	move.l		a4,(a5)			;relink as chain head again
	bra.s		.exit
;
.count_sent_data:
	add.l		d0,prt_des_stat_sd_data-prt_des_send(a5)	;sent data
	bsr		retrigger_plep			;retrigger PLEP
.discard_datagram:
	IP_discard	(a4),#1				;discard datagram
.skip_it:
	IFNE	USE_LOOPING
	move.l		(a5),d3			;d3 = Tx queue, empty ?
	ble.s		.exit			;exit when queue empty
	move		(_hz_200+2).w,d0
	sub		port_time(pc),d0	;d0 = elapsed time
	cmp		#MAX_portwork,d0	;timeout ?
	blo		.send_loop		;loop back until 20 ms
	ENDC	;USE_LOOPING
.exit:
	movem.l		(sp)+,d2-d3/a2-a5		;restore entry regs
.exit_direct:
	bra		retrigger_plep			;retrigger PLEP
;
;end of my_send
;----------------------------------------------------------------------------
;void	my_receive(PORT *port);
;
my_receive:
	cmpi.l		#my_port,4(sp)		;argument == my_port ?
	bne		.exit_direct		;exit if argument incorrect
	move.l		plep_rx_q(pc),d0	;d0 -> raw Rx Queue, empty ?
	beq		.exit_direct		;exit directly if empty
	move		sr,-(sp)		;push sr
	movem.l		d2-d5/a2-a5,-(sp)	;push entry regs
	lea		my_port(pc),a5		;a5 -> my_port
;-------
	IFNE	USE_LOOPING
	move		(_hz_200+2).w,port_time		;find work start time
.transfer_loop:			;loop start to transfer raw dgrams to internal
	ENDC	;USE_LOOPING
	or		#$0700,sr		;disable interrupts
	move.l		plep_rx_q(pc),d0	;d0 -> raw Rx Queue, empty ?
	beq.s		.exit			;we're done if empty
	move.l		d0,a4			;a4 -> raw Rx queue
	move.l		(a4),plep_rx_q		;unlink first Rx datagram
	move		8*4(sp),sr		;reenable interrupts
;-------
	move		PLEP_length(a4),d0		;d0 = raw datagram length
	lea		PLEP_data(a4),a0	;a0 -> raw datagram header
	clr.l		-(sp)			;reserve result space on stack
	lea		(sp),a1			;a1 -> result space
	bsr		make_IP_dgram		;create an internal datagram
	move.l		(sp)+,d0		;pop result to d0
	ble.s		.error			;drop packet on failure
;-------
	move.l		d0,a3			;a3 -> internal datagram
	set_dgram_ttl	(a3)			;init time-to-live data
;-------
	lea		prt_des_receive(a5),a0	;a0 = receive queue root -> first datagram
	bra.s		.store_test
;-------------------------------------
.store_loop:
	lea		IPDG_next(a1),a0	;a0 = link -> next datagram
.store_test:
	move.l		(a0),a1			;a1 -> next datagram or is NULL
	move.l		a1,d0			;test next datagram
	bne.s		.store_loop		;loop until queue end found
	move.l		a3,(a0)			;store new dgram at queue end
	clr.l		d0
	move		PLEP_length(a4),d0
	add.l		d0,prt_des_stat_rcv_data(a5)	;count received data
	bra.s		.release		;go release raw datagram
;-------------------------------------
.error:
	addq		#1,prt_des_stat_dropped(a5)	;increment drop count
.release:
	or		#$0700,sr		;disable interrupts
	R_free		(a4)			;release raw datagram buffer
	move		8*4(sp),sr		;reenable interrupts
	IFNE	USE_LOOPING
	move.l		plep_rx_q(pc),d3	;d3 -> Rx queue, empty ?
	ble.s		.exit			;exit when queue empty
	move		(_hz_200+2).w,d0
	sub		port_time(pc),d0	;d0 = elapsed time
	cmp		#MAX_portwork,d0	;timeout ?
	blo		.transfer_loop		;loop back until 20 ms
	ENDC	;USE_LOOPING
.exit:
	movem.l		(sp)+,d2-d5/a2-a5	;pull entry regs
	move		(sp)+,sr		;pull sr
.exit_direct:
	bra		retrigger_plep
;
;end of my_receive
;----------------------------------------------------------------------------
;uint16	my_set_state(PORT *port, uint16 state);
;
my_set_state:
	link		a6,#0
	movem.l		d2-d5/a2-a5,-(sp)
	lea		my_port(pc),a5		;a5 -> my_port
	clr.l		d0			;prep error flag
	cmpa.l		8(a6),a5		;argument correct ?
	bne.s		.exit			;exit if argument incorrect
	move		12(a6),d0		;d0 = new state
	beq.s		.passivate
.activate:
;
;Add port dependent activation code here
;This must include all initialization the port needs
;
	xbios		Supexec,pl_open(pc)
;
	bra.s		.done
;
.passivate:
;
;Add port dependent passivation code here
;This must include release of any KRmalloc blocks etc
;
	xbios		Supexec,pl_close(pc)
;	
.done:
	moveq		#1,d0
.exit:
	movem.l		(sp)+,d2-d5/a2-a5
	unlk		a6
	rts
;
;end of my_set_state
;----------------------------------------------------------------------------
;End of:	Port driver functions
;----------------------------------------------------------------------------
;Start of:	Resident subroutines
;----------------------------------------------------------------------------
;Aregs:	a0-a3 are free,  a4->datagram
;Dregs  d0-d3 are free
;
pass_send_pkt:
	clr.l		d0				;d0 = NULL  (error)
	cmpi		#MAX_buffers,plep_tx_q_ct	;too many buffers ?
	bhs.s		.exit_direct			;exit if past limit
;
	moveq		#sizeof_IPHD,d3
	add		IPDG_opt_length(a4),d3
	add		IPDG_pkt_length(a4),d3	;d3 = PLEP buffer size
	moveq		#sizeof_PLEP,d0		;d0 = PLEP header size
	add		d3,d0			;d0 = total size of PLEP struct
	move		sr,-(sp)		;push int_mask
	ori		#$0700,sr		;disable int_mask
	R_alloc		d0			;allocate RA RAM
	move		(sp)+,sr		;pull int_mask
	tst.l		d0			;did we get any ?
	ble.s		.exit_direct		;exit if no RAM available
;
	move.l		d0,a2			;a2 -> PLEP buffer
	clr.l		(a2)			;init link to next buffer
	move		d3,PLEP_length(a2)	;set data length of this one
;
	lea		IPDG_hdr(a4),a0		;a0 -> IP header
	lea		PLEP_data(a2),a1	;a1 -> PLEP buffer
	moveq		#sizeof_IPHD,d0		;d0 =  standard IP header size
	buf_copy_l.w	a0,a1,d0		;copy header as longs
	move.l		IPDG_options(a4),a0
	move		IPDG_opt_length(a4),d0
	buf_copy_l.w	a0,a1,d0		;copy options as longs
	move.l		IPDG_pkt_data(a4),a0
	move		IPDG_pkt_length(a4),d0
	buf_copy_b.w	a0,a1,d0		;copy packet data as bytes
;
	move		sr,d2			;save interrupt mask in d2
	ori		#$0700,sr		;disable interrupts
;
	lea		plep_tx_q(pc),a0	;a0 -> send queue root ptr
	move.l		a0,d0			;d0 -> send queue root ptr
.store_loop:
	move.l		d0,a0			;a0 -> current chain ptr
	move.l		(a0),d0			;d0 -> next datagram or is NULL
	bne		.store_loop		;loop until queue end reached
	move.l		a2,(a0)			;append new buffer to queue end
	addq		#1,plep_tx_q_ct		;increment buffer count
	move		d2,sr			;restore interrupt mask
	move.l		d3,d0			;return PLEP packet length
.exit_direct:
	rts		;error <= zero (flagged) else => PLEP length
;
;ends	pass_send_pkt
;----------------------------------------------------------------------------
retrigger_plep:
	move		sr,-(sp)		;push old SR (for interrupt mask)
	or		#$0700,sr		;disable interrupts
	move.l		state_vector(pc),d0	;d0 -> current mode routine
	cmp.l		#cv_Init_Recv,d0	;are we in the initial mode ?
	bls.s		.try_more_sending	;try sending when resting
	pl_timeout.s	state_t1(pc),#MAX_delay,.timeout_1
	bra.s		.exit
;-------
.timeout_1:
	pl_timeout.s	state_t2(pc),#MAX_delay,.timeout_2
	bra.s		.exit
;-------
.timeout_2:
	bsr		abort_transfer		;abort old traffic at timeout
.try_more_sending:
	move		plep_tx_q_ct(pc),d0	;Is there anything to send ?
	ble.s		.exit
.send_stuff:
	pl_strobe_HI	d0			;normalize strobe
	move.l		(_hz_200).w,state_t1	;memorize start time
	pl_send_ATT	d1,#cp_DBEG		;send cp_DBEG byte
	set_state	Init_Send
.exit:
	move		(sp)+,sr		;restore entry interrupt mask
.exit_direct:
	rts					;return to caller
;
;ends	retrigger_plep
;----------------------------------------------------------------------------
;void  make_IP_dgram (uint8 *buffer, int16 buff_len, IP_DGRAM **dgram);
;
make_IP_dgram:
	movem.l		d3-d5/a3-a5,-(sp)
	move.l		a0,a4			;a4 is buffer
	move		d0,d5			;d5 is buff_len
	move.l		a1,a5			;a5 is dgram
	clr.l		(a5)			;*dgram = NULL
	cmp		#sizeof_IPHD,d5
	ble		.exit			;if (buff_len < sizeof (IP_HDR)) return;
;-------
	KRmalloc	#sizeof_IPDG
	tst.l		d0
	ble		.exit			;if ((temp = KRmalloc (sizeof (IP_DGRAM))) == NULL) return;
	move.l		d0,a3			;a3 is temp
	lea		IPDG_hdr(a3),a1		;dst_arg = & temp->hdr
	move		#sizeof_IPHD,d0		;len_arg = sizeof_IPHD
	buf_copy_l.w	a4,a1,d0		;memcpy (& temp->hdr, buffer, sizeof (IP_HDR)); buffer += sizeof (IP_HDR);
;-------
	cmp		IPDG_hdr+IPHD_length(a3),d5
	blo		.free_a3		;if (temp->hdr.length > buff_len
	move.b		IPDG_hdr+IPHD_verlen_f(a3),d0
	and.l		#amask_IPHD_f_hd_len,d0
	asl		#2,d0
	cmp		d5,d0
	bhi.s		.free_a3		;|| (temp->hdr.hd_len << 2) > buff_len
	move		d0,d3			;d3 = (temp->hdr.hd_len << 2)
	sub		#sizeof_IPHD,d0
	blt.s		.free_a3		;|| temp->hdr.hd_len < 5) {KRfree (temp); return;}
	move		d0,IPDG_opt_length(a3)
	KRmalloc	d0
	move.l		d0,IPDG_options(a3)	;temp->options  = KRmalloc (temp->opt_length = (temp->hdr.hd_len << 2) - sizeof (IP_HDR));
	clr.l		d0
	move		IPDG_hdr+IPHD_length(a3),d0
	sub		d3,d0
	move		d0,IPDG_pkt_length(a3)
	KRmalloc	d0
	move.l		d0,IPDG_pkt_data(a3)	;temp->pkt_data = KRmalloc (temp->pkt_length = temp->hdr.length - (temp->hdr.hd_len << 2));
	ble.s		.discard_a3
	move.l		d0,d3			;d3 = temp->pkt_data
	move.l		IPDG_options(a3),d0	;d0 = temp->options
	ble.s		.discard_a3		;if (temp->options == NULL || temp->pkt_data == NULL ) {IP_discard (temp, TRUE); return;}
	move.l		d0,a1			;a1 = temp->options
	move		IPDG_opt_length(a3),d0
	beq.s		.have_option
	buf_copy_l.w	a4,a1,d0		;memcpy (temp->options, buffer, temp->opt_length);
.have_option:
	move.l		d3,a1			;a1 = temp->pkt_data
	move		IPDG_pkt_length(a3),d0
	beq.s		.have_data
	buf_copy_b.w	a4,a1,d0		;memcpy (temp->pkt_data, buffer + temp->opt_length, temp->pkt_length);
.have_data:
	clr.l		IPDG_next(a3)
	move.l		a3,(a5)			;*dgram = temp;
.exit:
	movem.l		(sp)+,d3-d5/a3-a5
	rts
;
.free_a3:
	KRfree		(a3)
	bra		.exit
;
.discard_a3:
	IP_discard	(a3),#1
	bra		.exit
;
;ends	make_IP_dgram
;----------------------------------------------------------------------------
;void  pl_open(void);
;
pl_open:
;-------
	tst.b		plep_open_f
	beq.s		.plep_closed
	bsr		pl_close
.plep_closed:
	bsr		release_buffers
;-------
	move.b		(hw_imrb).w,d3		;d3 = entry hw_imrb
	pl_cli					;mask reception interrupt
	move		sr,-(sp)		;push interrupt mask
	or		#$0700,sr		;disable interrupts
;-------
	pl_dir_in	d0			;set port for input
	pl_strobe_HI	d0			;normalize strobe
	pl_take_ACK				;clear pending ACK if present
;-------
	XB_remove	cent_busy_XB(pc),(iv_cenbusy).w	;unlink interrupt code
	bpl.s		.done_primary_open
	move.b		d3,orig_imrb		;save initial hw_imrb state
	move.b		(hw_ierb).w,orig_ierb	;save initial hw_ierb state
	move.b		(hw_aer).w,orig_aer	;save initial hw_aer state
.done_primary_open:
;-------
	XB_install	cent_busy_XB(pc),(iv_cenbusy).w	;link interrupt code
	bset		#0,(hw_ierb).w		;enable BUSY interrupt
	bclr		#0,(hw_aer).w		;set BUSY edge sense to falling
	pl_take_ACK				;ignore misfire at switch
	set_state	Init_Recv
	move		(sp)+,sr		;reenable interrupts
	pl_take_ACK				;clear pending ACK if present
	pl_sti					;unmask reception interrupt
	st		plep_open_f
	rts
;-------
;ends	pl_open
;----------------------------------------------------------------------------
;void  pl_close(void);
;
pl_close:
	move		sr,-(sp)		;push interrupt mask
	or		#$0700,sr		;disable interrupts
	set_state	Dummy_State
;-------
	move.b		(hw_imrb).w,d3		;d3 = entry hw_imrb
	pl_cli					;mask reception interrupt
	pl_dir_in	d0			;set port for input
	pl_strobe_HI	d0			;normalize strobe
	pl_take_ACK				;clear pending ACK if present
;
	XB_remove	cent_busy_XB(pc),(iv_cenbusy).w
	bmi.s		.done_primary_close
	move.b		orig_imrb(pc),d3	;d3 = original imrb
;
	btst		#0,orig_ierb(pc)
	beq.s		.clr_b0_ierb
.set_b0_ierb:
	bset		#0,(hw_ierb).w
	bra.s		.done_ierb
;
.clr_b0_ierb:
	bclr		#0,(hw_ierb).w
.done_ierb:
;-------
	btst		#0,orig_aer(pc)
	beq.s		.clr_b0_aer
.set_b0_aer:
	bset		#0,(hw_aer).w
	bra.s		.done_aer
;-------
.clr_b0_aer:
	bclr		#0,(hw_aer).w
.done_aer:
;-------
.done_primary_close:
	pl_take_ACK				;clear pending ACK if present
;-------
	btst		#0,d3
	beq.s		.clr_b0_imrb
.set_b0_imrb:
	bset		#0,(hw_imrb).w
	bra.s		.done_imrb
;
.clr_b0_imrb:
	bclr		#0,(hw_imrb).w
.done_imrb:
;-------
	sf		plep_open_f
	move		(sp)+,sr		;pull interrupt mask
release_buffers:
	move		sr,-(sp)		;push interrupt mask
	or		#$0700,sr		;disable interrupts
	pl_strobe_HI	d0			;normalize strobe
	pl_dir_in	d0			;set port for input
	move.l		recv_buf_p(pc),d0
	ble.s		.recv_buf_released
	R_free.i	d0
	clr.l		recv_buf_p
.recv_buf_released:
.release_rx_queue:
	pl_interrupt	(sp)			;poll interrupts
	move.l		plep_rx_q(pc),d0	;d0 -> raw datagram or is null
	beq.s		.rx_queue_released
	move.l		d0,a0			;a0 -> raw datagram
	move.l		(a0),plep_rx_q		;unlink this raw datagram
	R_free		(a0)			;release allocated buffer
	bra		.release_rx_queue
;
.rx_queue_released:
	move.l		send_buf_p(pc),d0
	ble.s		.send_buf_released
	R_free.i	send_buf_p(pc)
	clr.l		send_buf_p
.send_buf_released:
.release_tx_queue:
	pl_interrupt	(sp)			;poll interrupts
	move.l		plep_tx_q(pc),d0	;d0 -> raw datagram or is null
	beq.s		.tx_queue_released
	move.l		d0,a0			;a0 -> raw datagram
	move.l		(a0),plep_tx_q		;unlink this raw datagram
	R_free		(a0)			;release allocated buffer
	bra		.release_tx_queue
;
.tx_queue_released:
	clr		plep_tx_q_ct
	move		(sp)+,sr		;pull interrupt mask
	rts
;-------
;ends	pl_close
;----------------------------------------------------------------------------
allocate_recv_buf:
	move.l		recv_buf_p(pc),d0	;is there an old buffer ?
	bgt.s		.have_buffer		;keep old buffer if present
	R_alloc		#sizeof_PLEP+MAX_mtu+2	;allocate raw datagram buffer
	tst.l		d0			;did we get any ?
	ble.s		.exit			;exit on allocation failure
	move.l		d0,recv_buf_p
.have_buffer:
	move.l		d0,a0			;a0 -> receive buffer
	clr.l		(a0)			;clear its own queue link
	move		#MAX_mtu,d0
	move		d0,PLEP_length(a0)	;setup its max data length
	lea		PLEP_data(a0),a1
	move.l		a1,recv_pos_p		;init buffer position ptr
	lea		PLEP_data(a0,d0.w),a1	;a1 -> buffer end
	move.l		a1,recv_max_p		;init buffer end ptr
.exit:
	rts		;GT with d0 = a0 -> buf, or LE with d0 <= 0
;----------------------------------------------------------------------------
abort_transfer:
	clr		-(sp)			;push 0 as prel return value
	move		sr,-(sp)		;push SR (for interrupt mask)
	or		#$0700,sr		;disable interrupts
	move		IO_progress_f(pc),d0
	beq		.nothing_to_abort
	move.b		TX_progress_f(pc),d0
	beq.s		.done_TX_abort
	movem.l		d2/a2,-(sp)
	pl_abort_send
	movem.l		(sp)+,d2/a2
.done_TX_abort:
	move.b		RX_progress_f(pc),d0
	beq.s		.done_RX_abort
	movem.l		d2/a2,-(sp)
	pl_abort_recv
	movem.l		(sp)+,d2/a2
.done_RX_abort:
.nothing_to_abort:
	set_state	Init_Recv
	move		(sp)+,sr		;pull entry interrupt mask
	move		(sp)+,d0		;pull return value
	ext.l		d0			;make value long
	rts					;return to caller
;-------
;ends	abort_transfer
;----------------------------------------------------------------------------
;End of:	Resident subroutines
;----------------------------------------------------------------------------
;Start of:	Resident interrupt routines
;----------------------------------------------------------------------------
	XB_define	cent_busy_XB,'plep'	;XBRA header
	or		#$0700,sr		;disable all interrupts
	movem.l		d0-d2/a0-a2,plep_regs
	move		(sp),-(sp)		;repush pre-exception SR
	clr		d0
	move.b		(sp),d0
	move.b		SR_table(pc,d0),(sp)	;patch it for polling SR
	move.l		(_hz_200).w,state_t1	;register time tick before work
	pl_take_ACK				;debounce ACK jitter
	bra		state_entry
;-------------------------------------
SR_table:
	rept	8
	dc.b	$24,$24,$24,$24,$24,$25,$26,$27,$24,$24,$24,$24,$24,$25,$26,$27
	endr
	rept	8
	dc.b	$A4,$A4,$A4,$A4,$A4,$A5,$A6,$A7,$A4,$A4,$A4,$A4,$A4,$A5,$A6,$A7
	endr
;-------------------------------------
state_entry:
	pl_strobe_HI	d0			;normalize strobe
	pl_dir_in	d0			;normalize port direction
	pl_test_busy				;ACK or ATT ?
	sne		RX_ATT_flag		;flag ACK presence
	bne.s		.state_jump		;ACK carries no data
	pl_recv_byte	RX_ATT_data		;store RX_ATT_data
	pl_send_ACK	d0,d1			;ACKnowledge received ATT
.state_jump:
	move		RX_ATT_pair(pc),d0
	movea.l		state_vector(pc),a0	;a0 -> state routine
	jmp		(a0)			;jump to state routine
;-------------------------------------
state_loop:
	move.l		(_hz_200).w,state_t2	;register time tick after work
	addq.b		#WATCH_step,byte_watch	;count loops
	bcs.s		state_exit		;exit anyway when watch overflows
	pl_test_ACK				;new interrupt pending ?
	beq.s		state_exit		;else just exit
	pl_clear_ACK
	pl_take_ACK				;debounce ACK jitter
	pl_interrupt	(sp)			;poll interrupts
	bra		state_entry
;-------
state_exit:
	addq		#2,sp			;pop polling sr off stack
	movem.l		plep_regs(pc),d0-d2/a0-a2
	move.b		#$FE,(hw_isrb).w	;flag service to hardware
	rte					;return from exception
;----------------------------------------------------------------------------
	def_state	Dummy_State	;-------;This state does nothing,
	loop_state			;-------;and remains until forced.
;----------------------------------------------------------------------------
new_Refuse_Request:
	new_state	Refuse_Request	;-------
	bpl.s		new_initial_ATT		;incoming ATT => back to Init_Recv
	pl_send_ATT	d1,#cp_ERRL		;write cp_ERRL to port B
abnorm_Init_Recv:
	goto_state	Init_Recv	;-------;goto initial state
;-------------------------------------
new_initial_ATT:
	set_state	Init_Recv	;-------
	bra		initial_RX_ATT
;-------------------------------------
	def_state	Init_Recv	;-------
	bpl		initial_RX_ATT		;incoming ATT => ATT analysis
;-------
;here we received a redundant ACK, which is a chance to try sending
;-------
	move		plep_tx_q_ct(pc),d0	;anything to send ?
	beq.s		abnorm_Init_Recv	;else let's go IDLE
	pl_send_ATT	d1,#cp_DBEG		;send cp_DBEG byte
	goto_state	Init_Send	;-------
;-------------------------------------
refuse_reception:
	new_state	Force_Reply	;-------
	bpl.s		new_initial_ATT		;incoming ATT => back to Init_Recv
	sf		ref_perm_f		;don't refuse next time
	pl_send_ATT	d1,#cp_DBEG		;send cp_DBEG byte
	goto_state	Init_Send	;-------
;-------------------------------------
initial_RX_ATT:
	cmp		#cp_ERRL,d0		;incoming ATT == error report ?
	beq		abnorm_Init_Recv	;if so, ignore it, but note abnormalcy
	cmp		#cp_DBEG,d0		;incoming ATT == DBEG request ?
	bne		new_Refuse_Request	;else go tell B he is insane
;-------
;Here we received a legal DBEG request, but we may need to send something instead
;-------
	move		plep_tx_q_ct(pc),d0	;anything to send ?
	beq.s		.accept_DBEG		;else go accept DBEG
	move.b		ref_perm_f(pc),d0	;do we have permission to refuse ?
	bne.s		refuse_reception	;if so, go try sending instead
.accept_DBEG:
	bsr		allocate_recv_buf	;ensure buffer allocation
	ble		new_Refuse_Request	;if invalid, go tell B we are insane
	new_state	Send_DREQ	;-------
	bpl		new_initial_ATT		;incoming ATT => back to Init_Recv
	pl_send_ATT	d1,#cp_DREQ		;write cp_DREQ to port B
	new_state	Sent_DREQ	;-------
	bpl		new_initial_ATT		;incoming ATT => back to Init_Recv
	clr.l		plep_crc		;clr plep_crc and plep_crc_ix
	pl_send_ACK	d0,d1			;send ACK to start data stream
	new_state	Recv_First	;-------
	bmi		Recv_First_ERRH		;incoming ACKs are illegal here !
	sub.b		#cp_sel1,d0		;d0 = half_decoded data
	cmp.b		#cp_ERRL-cp_sel1,d0
	bls.s		Recv_First_cp_XXXX
	move.l		recv_pos_p(pc),a0	;a0 -> IP packet position
	cmpa.l		recv_max_p(pc),a0	;is it too high ?
	bhs.s		Recv_First_ERRH		;if too high, abort reception
	add.b		#cp_base,d0		;d0 = decoded char of packet
	move.b		d0,(a0)+		;store IP packet byte
	move.l		a0,recv_pos_p		;update position pointer
	pl_crc
	st		RX_progress_f
	goto_state	Recv_Loop	;-------
;-------------------------------------
Recv_First_cp_XXXX:
	ext.w		d0
	add		d0,d0
	move		Recv_First_cp_XXXX_t(pc,d0),d0
	jmp		Recv_First_cp_XXXX_t(pc,d0)
;-------------------------------------
Recv_First_cp_XXXX_t:
	dc.w		Recv_First_ERRH-Recv_First_cp_XXXX_t	;cp_sel1
	dc.w		Recv_First_ERRH-Recv_First_cp_XXXX_t	;cp_sel2
	dc.w		Recv_First_ERRH-Recv_First_cp_XXXX_t	;cp_sel3
	dc.w		Recv_First_ERRH-Recv_First_cp_XXXX_t	;cp_sel4
	dc.w		Recv_First_ERRH-Recv_First_cp_XXXX_t	;cp_sel5
	dc.w		Recv_First_ERRH-Recv_First_cp_XXXX_t	;cp_sel6
	dc.w		Recv_First_ERRH-Recv_First_cp_XXXX_t	;cp_sel7
	dc.w		Recv_First_ERRH-Recv_First_cp_XXXX_t	;cp_sel8
	dc.w		Recv_First_ERRH-Recv_First_cp_XXXX_t	;cp_res0
	dc.w		Recv_First_ERRH-Recv_First_cp_XXXX_t	;cp_DACK
	dc.w		Recv_Loop_DOFF-Recv_First_cp_XXXX_t	;cp_DOFF
	dc.w		Recv_Loop_DEND-Recv_First_cp_XXXX_t	;cp_DEND
	dc.w		Recv_First_ERRH-Recv_First_cp_XXXX_t	;cp_DBEG
	dc.w		Recv_First_ERRH-Recv_First_cp_XXXX_t	;cp_DREQ
	dc.w		Recv_First_ERRH-Recv_First_cp_XXXX_t	;cp_ERRH
	dc.w		Recv_First_ERRL-Recv_First_cp_XXXX_t	;cp_ERRL
;-------------------------------------
Recv_First_ERRL:
	goto_state	Init_Recv	;-------
;-------------------------------------
Recv_First_ERRH:
	goto_state	Refuse_Request
;-------------------------------------
	def_state	Recv_Loop
	bmi		Recv_Loop_ERRH		;incoming ACKs are illegal here !
	sub.b		#cp_sel1,d0		;d0 = half_decoded data
	cmp.b		#cp_ERRL-cp_sel1,d0
	bls.s		Recv_Loop_cp_XXXX
	move.l		recv_pos_p(pc),a0	;a0 -> IP packet position
	cmpa.l		recv_max_p(pc),a0	;is it too high ?
	bhs		Recv_Loop_ERRH		;if too high, abort reception
	add.b		#cp_base,d0		;d0 = decoded char of packet
	move.b		d0,(a0)+		;store IP packet byte
	move.l		a0,recv_pos_p		;update position pointer
	pl_crc
	goto_state	Recv_Loop	;-------
;-------------------------------------
Recv_Loop_cp_XXXX:
	ext.w		d0
	add		d0,d0
	move		Recv_Loop_cp_XXXX_t(pc,d0),d0
	jmp		Recv_Loop_cp_XXXX_t(pc,d0)
;-------------------------------------
Recv_Loop_cp_XXXX_t:
	dc.w		Recv_Loop_ERRH-Recv_Loop_cp_XXXX_t	;cp_sel1
	dc.w		Recv_Loop_ERRH-Recv_Loop_cp_XXXX_t	;cp_sel2
	dc.w		Recv_Loop_ERRH-Recv_Loop_cp_XXXX_t	;cp_sel3
	dc.w		Recv_Loop_ERRH-Recv_Loop_cp_XXXX_t	;cp_sel4
	dc.w		Recv_Loop_ERRH-Recv_Loop_cp_XXXX_t	;cp_sel5
	dc.w		Recv_Loop_ERRH-Recv_Loop_cp_XXXX_t	;cp_sel6
	dc.w		Recv_Loop_ERRH-Recv_Loop_cp_XXXX_t	;cp_sel7
	dc.w		Recv_Loop_ERRH-Recv_Loop_cp_XXXX_t	;cp_sel8
	dc.w		Recv_Loop_ERRH-Recv_Loop_cp_XXXX_t	;cp_res0
	dc.w		Recv_Loop_ERRH-Recv_Loop_cp_XXXX_t	;cp_DACK
	dc.w		Recv_Loop_DOFF-Recv_Loop_cp_XXXX_t	;cp_DOFF
	dc.w		Recv_Loop_DEND-Recv_Loop_cp_XXXX_t	;cp_DEND
	dc.w		Recv_Loop_ERRH-Recv_Loop_cp_XXXX_t	;cp_DBEG
	dc.w		Recv_Loop_ERRH-Recv_Loop_cp_XXXX_t	;cp_DREQ
	dc.w		Recv_Loop_ERRH-Recv_Loop_cp_XXXX_t	;cp_ERRH
	dc.w		Recv_Loop_ERRL-Recv_Loop_cp_XXXX_t	;cp_ERRL
;-------------------------------------
Recv_Loop_ERRL:
	pl_abort_recv				;abort reception
	goto_state	Init_Recv	;-------
;-------------------------------------
Recv_Loop_ERRH:
	pl_abort_recv
	goto_state	Refuse_Request
;-------------------------------------
Recv_Loop_DOFF:
	new_state	Recv_DOFF_data	;-------
	bmi.s		Recv_Loop_ERRH		;incoming ACKs are illegal here !
	add.b		#cp_DOFF_dec,d0		;d0 = decoded byte of packet
	move.l		recv_pos_p(pc),a0	;a0 -> IP packet position
	move.b		d0,(a0)+		;store IP packet byte
	move.l		a0,recv_pos_p		;update position pointer
	pl_crc
	goto_state	Recv_Loop	;-------
;-------------------------------------
Recv_Loop_DEND:
	new_state	Check_CRC	;-------
	bmi.s		.have_ACK		;we expect an ACK here
	cmp		#cp_ERRL,d0		;incoming ATT == ERRL
	beq		Recv_Loop_ERRL
	bra		Recv_Loop_ERRH
;-------
.have_ACK:
	move		plep_crc(pc),d0		;d0 = CRC
	bne		Recv_Loop_ERRH		;send refusal if CRC bad
	move.l		recv_buf_p(pc),a0	;a0 -> recv buffer struct
	move.l		recv_pos_p(pc),d0	;d0 -> end of received data
	lea		PLEP_data+2(a0),a1	;a1 -> start of received data
;NB: The '+2' above is to exclude crc from length
	sub.l		a1,d0			;d0 = received data length
	ble		Recv_First_ERRH		;abort on sick data length ?
	move		d0,PLEP_length(a0)	;store real data length
	lea		plep_rx_q(pc),a0	;a0 -> receive queue root ptr
	move.l		a0,d0			;d0 -> receive queue root ptr
.store_loop:
	move.l		d0,a0			;a0 -> current chain ptr
	move.l		(a0),d0			;d0 -> next datagram or is NULL
	bne		.store_loop		;loop until queue end reached
	move.l		recv_buf_p(pc),(a0)	;store new dgram at queue end
	clr.l		recv_buf_p		;erase buffer pointer
	sf		RX_progress_f		;flag RX not progressing
	st		ref_perm_f		;we may refuse opponent now
	pl_send_ATT	d1,#cp_DEND		;confirm good packet transfer
	new_state	Post_Recv	;-------
	pl_send_ACK	d0,d1			;give sender a chance for more
restart_Init_Recv:
	goto_state	Init_Recv	;-------
;----------------------------------------------------------------------------
	def_state	Init_Send	;-------
	bpl		new_initial_ATT		;incoming ATT => back to Init_Recv
	pl_send_ACK	d0,d1			;send ACK to request DREQ confirmation
	new_state	Test_DREQ	;-------
	bmi		new_Refuse_Request	;incoming ACKs are illegal here !
	cmp		#cp_ERRL,d0		;incoming ATT == error report ?
	beq		abnorm_Init_Recv	;if so, ignore it, but note abnormalcy
	cmp		#cp_DREQ,d0		;incoming ATT == DREQ request ?
	bne		new_Refuse_Request	;else go tell B he is insane
;Now, before sending any actual data, we must first init some stuff
	move.l		plep_tx_q(pc),d0	;d0 -> next buffer in plep_tx_q
	ble		new_Refuse_Request	;abort if no buffer left
	move.l		d0,send_buf_p		;send_buf_p -> that buffer
	move.l		d0,a0			;a0 -> the buffer found
	move.l		(a0),plep_tx_q		;unlink it from plep_tx_q
	subq		#1,plep_tx_q_ct		;decrement buffer count
	st		TX_progress_f		;note TX in progress
	move		PLEP_length(a0),d0	;d0 = data length of packet
	lea		PLEP_data(a0),a0	;a0 -> data start in packet
	move.l		a0,send_pos_p		;send_pos_p -> data start
	lea		(a0,d0.w),a0		;A0 -> data end in packet
	move.l		a0,send_max_p		;send_max_p -> data end
	clr.l		plep_crc		;clr plep_crc and plep_crc_ix
	new_state	Send_Loop	;-------
	bmi.s		send_block_data		;incoming ACK => proceed to send
break_Send_Loop:
	pl_abort_send				;incoming ATT => abort sending
	cmp		#cp_ERRL,RX_ATT_pair	;incoming ATT == error report ?
	beq		abnorm_Init_Recv	;if so, don't reply, but note abnormalcy
	goto_state	Refuse_Request	;-------;else tell B he is insane again
;-------------------------------------
send_block_data:
	move.l		send_pos_p(pc),a0	;a0 -> next character to send
	cmpa.l		send_max_p(pc),a0	;is it at or beyond packet end
	bhs		send_finish		;go handle packet end
	move.b		(a0),d0			;d0 = IP byte for crc calc
	pl_crc
	move.b		(a0)+,d0		;d0 = IP byte to send
	move.l		a0,send_pos_p		;update pointer
encode_send_data:
	sub.b		#cp_base,d0		;d0 = half encoded char
	cmp.b		#cp_ERRL-cp_sel1,d0
	bls.s		Send_cp_XXXX
	add.b		#cp_sel1,d0		;d0 = full encoded char
	pl_send_ATT	d1,d0			;send PLEP byte
	loop_state			;-------;keep state Send_Loop
;-------------------------------------
Send_cp_XXXX:
	add.b		#cp_DOFF_enc,d0		;d0 = offset encoded char
	move.b		d0,DOFF_char		;store char for later
	pl_send_ATT	d1,#cp_DOFF		;send cp_DOFF byte
	new_state	Send_DOFF_data	;-------
	bpl		break_Send_Loop		;incoming ATT => abort sending
	pl_send_ATT	d1,DOFF_char(pc)		;send cp_DOFF data byte
	goto_state	Send_Loop	;-------
;-------------------------------------
send_finish:
	move		plep_crc_ix(pc),d0
	addq		#1,plep_crc_ix
	cmp		#2,d0
	bhs.s		send_DEND		;send DEND after CRC
	lea		plep_crc(pc),a0
	move.b		(a0,d0),d0		;d0 = a CRC byte
	bra		encode_send_data
;-------------------------------------
send_DEND:
	pl_send_ATT	d1,#cp_DEND		;send cp_DOFF byte
	new_state	Sent_DEND	;-------
	bpl		break_Send_Loop		;incoming ATT => abort sending
	pl_send_ACK	d0,d1			;ask for DEND confirmation
	new_state	Test_DEND	;-------
	bmi		break_Send_Loop		;incoming ACK is illegal here
	cmp		#cp_DEND,d0		;incoming ATT == DEND ?
	bne		break_Send_Loop		;if not, abort sending
;-------
;Here the packet has been fully transferred, so we just need to clean up
;-------
	new_state	Post_send	;-------
	sf		ref_perm_f		;we may not refuse opponent now
	R_free.i	send_buf_p(pc)
	clr.l		send_buf_p
	sf		TX_progress_f		;flag TX not progressing
	move		plep_tx_q_ct(pc),d0	;any more to send ?
	beq.s		let_receiver_reply	;else go try reception
	pl_send_ATT	d1,#cp_DBEG		;send cp_DBEG byte
	goto_state	Init_Send	;-------
;-------------------------------------
let_receiver_reply:
	pl_send_ACK	d0,d1			;give opponent redundant ACK
	goto_state	Init_Recv		;and prepare for reception
;----------------------------------------------------------------------------
;End of:	cent_busy_XB
;----------------------------------------------------------------------------
;Since TOS can leave garbage in iv_cenbusy at boot, here is a dumb routine
;to reside there when CEN_PLEP is turned 'off'.  It will only 'eat' all the
;incoming 'busy' pulses without responding in any way.
;
dumb_cenbusy:
	bclr		#0,(hw_imrb).w
	move.b		#$FE,(hw_isrb).w		;flag service to hardware
	rte
;----------------------------------------------------------------------------
;End of:	Resident interrupt routines
;----------------------------------------------------------------------------
;	Resident library function code will be expanded here
;
	make		JAR_links
	_uniref		RAM_own
	make		RAM_links
;----------------------------------------------------------------------------
;all beyond this point will be reused for RAM buffers in going resident
;----------------------------------------------------------------------------
;Start of:	STX Non-resident initialization code with tests
;----------------------------------------------------------------------------
RAM_start:		;this will be RAM buffer start
start_1:
	move.l		a0,d0
	sne		d7
	bne.s		.have_basepage
	move.l		4(sp),d0
.have_basepage:
	move.l		d0,a5
	lea		mystack(pc),sp
	move.l		a5,basepage_p
	tst.b		d7
	bne		.ACC_launch
	gemdos		Mshrink,#0,(a5),#initial_size
	lea		bp_arglen(a5),a0
	lea		STinG_Load_s(pc),a1
	str_comp	a0,a1
	bne		.bad_launch
;
	gemdos		Super,0.w
	move.l		d0,d4
;
	bsr		ensure_iv_cenbusy
;
	eval_cookie	#"STiK"
	move.l		d0,d3				;d3 = d0 -> DRV_LIST structure
	gemdos		Super|_ind,d4
;
	move.l		d3,sting_drivers		;sting_drivers -> DRV_LIST structure
	ble		.STiK_not_found
	move.l		d3,a3				;a3 -> DRV_LIST structure
	lea		DRV_LIST_magic(a3),a0
	lea		STiKmagic_s(pc),a1
	moveq		#10-1,d0
.strcmp_loop:					;loop to test STiKmagic of DRV_LIST
	cmpm.b		(a0)+,(a1)+
	dbne		d0,.strcmp_loop
	bne		.STiK_not_valid
;
	move.l		DRV_LIST_get_dftab(a3),a0	;a0 -> get_dftab function
	pea		TRANSPORT_DRIVER_s		;-(sp) = "TRANSPORT_TCPIP"
	jsr		(a0)				;call get_dftab
	addq		#4,sp
	move.l		d0,tpl				;store pointer in 'tpl'
	ble		.driver_not_valid
;
	move.l		DRV_LIST_get_dftab(a3),a0	;a0 -> get_dftab function
	pea		MODULE_DRIVER_s			;-(sp) = "MODULE_LAYER"
	jsr		(a0)				;call get_dftab
	addq		#4,sp
	move.l		d0,stx				;store pointer in 'tpl'
	ble		.layer_not_valid
.install:
	link		a6,#-8
	query_chains	-8(a6),-4(a6),0.w
	movem.l		-8(a6),a3/a4			;a3->ports  a4->drivers
	unlk		a6
;
	move.l		a3,d0
	beq.s		.bad_port_list
.port_loop:
	move.l		d0,a3
	move.l		prt_des_next(a3),d0
	bne		.port_loop
;
	move.l		a4,d0
	beq.s		.bad_driver_list
.driver_loop:
	move.l		d0,a4
	move.l		drv_des_next(a4),d0
	bne		.driver_loop
;
	IFNE	USE_TIMER
	TIMER_call	my_timer_func(pc),#HNDLR_SET
	tst		d0
	beq		.timer_failure
	ENDC	;USE_TIMER
;
.final_install:
	move.l		#my_port,prt_des_next(a3)
	move.l		#my_driver,drv_des_next(a4)
;-------
	gemdos		Super,0.w
	move.l		d0,d4
	RAM_own		#1		;setup current bp as RAM owner
	RAM_set		RAM_start(pc),#RAM_chunk
	gemdos		Super|_ind,d4
;-------
	gemdos		Ptermres,#resident_size,#0
;-------------------------------------
.ACC_launch:
	lea		ACC_launch_s(pc),a0
	bsr		report_error
.loop:
	bra		.loop
;-------------------------------------
.bad_launch:
	lea		bad_launch_s(pc),a0
	bra.s		.error_exit
;-------------------------------------
.bad_port_list:
	lea		bad_port_list_s(pc),a0
	bra.s		.error_exit
;-------------------------------------
.bad_driver_list:
	lea		bad_driver_list_s(pc),a0
	bra.s		.error_exit
;-------------------------------------
	IFNE	USE_TIMER
.timer_failure:
	lea		timer_failure_s(pc),a0
	bra		.error_exit
	ENDC	;USE_TIMER
;-------------------------------------
.STiK_not_found:
	lea		STiK_not_found_s,a0
	bra.s		.error_exit
;-------------------------------------
.STiK_not_valid:
	lea		STiK_not_valid_s,a0
	bra.s		.error_exit
;-------------------------------------
.driver_not_valid:
	lea		driver_not_valid_s,a0
	bra.s		.error_exit
;-------------------------------------
.layer_not_valid:
	lea		layer_not_valid_s(pc),a0
.error_exit:
	bsr.s		report_error
	gemdos		Pterm,#E_ERROR
;-------------------------------------
ensure_iv_cenbusy:
	move		sr,d2				;d2 = entry SR
	or		#$0700,sr			;disable interrupts
	move.l		(ev_buserr).w,a1		;save ev_buserr
	move.l		(ev_adrerr).w,a2		;save ev_adrerr
	move.l		#bad_iv_cenbusy,(ev_buserr).w	;patch ev_adrerr
	move.l		#bad_iv_cenbusy,(ev_adrerr).w	;patch ev_adrerr
	move.l		(iv_cenbusy).w,a0		;a0 = suspect pointer
	move.l		sp,d1				;d1 = entry SP
	move.l		(a0),d0				;test pointer ?
good_iv_cenbusy:
	moveq		#E_OK,d0		;flag good pointer
	bra.s		know_iv_cenbusy		;go to common restore code
;
bad_iv_cenbusy:		;bus error or address error leads here !!!
	move.l		d1,SP			;SP = entry SP  (pop big frame)
	moveq		#E_ERROR,d0		;flag bad pointer
know_iv_cenbusy:
	move.l		a1,(ev_buserr).w	;restore ev_buserr
	move.l		a2,(ev_adrerr).w	;restore ev_adrerr
	move		d2,sr			;SR = entry SR
	tst.l		d0			;remember test result
	beq.s		.exit			;just exit if pointer is good
.patch_vector:
	move.l		#dumb_cenbusy,(iv_cenbusy).w	;patch iv_cenbusy
.exit:
	rts
;-------------------------------------
report_error:
	move.l		a0,-(sp)
	lea		error_title_s(pc),a0
	bsr.s		Cconws_sub
	move.l		(sp)+,a0
	bsr.s		Cconws_sub
	lea		error_tail_s(pc),a0
Cconws_sub:
	gemdos		Cconws,(a0)
	rts
;----------------------------------------------------------------------------
;End of:	STX Non-resident initialization code with tests
;----------------------------------------------------------------------------
;	Non-resident library function code will be expanded here
;
	make	JAR_links
	make	RAM_links
;----------------------------------------------------------------------------
text_limit:
text_size	= text_limit-text_start
	SECTION	DATA
data_start:
;----------------------------------------------------------------------------
STinG_Load_s:
	dc.b	10,'STinG_Load',NUL
STiKmagic_s:
	dc.b	'STiKmagic',NUL
TRANSPORT_DRIVER_s:
	dc.b	'TRANSPORT_TCPIP',NUL
MODULE_DRIVER_s:
	dc.b	'MODULE_LAYER',NUL
;
ACC_launch_s:
	dc.b	'This non-ACC, was launched as an ACC,',CR,LF
	dc.b	'so now you must reset the computer !',CR,LF
	dc.b	'I am looping forever to avoid damage',CR,LF
	dc.b	'that could occur if I try to exit !',CR,LF
	dc.b	NUL
;
bad_launch_s:	
	dc.b	'This STX should only be launched by',CR,LF
	dc.b	'STinG, or another TCP/IP stack with',CR,LF
	dc.b	'a compatible module interface !',CR,LF
	dc.b	NUL
;
bad_port_list_s:
	dc.b	'The list chain of STinG ports was',CR,LF
	dc.b	'not found...!',CR,LF
	dc.b	NUL
;
bad_driver_list_s:
	dc.b	'The list chain of STinG drivers was',CR,LF
	dc.b	'not found...!',CR,LF
	dc.b	NUL
;
	IFNE	USE_TIMER
timer_failure_s:
	dc.b	'STinG TIMER_call function failed !',CR,LF
	dc.b	NUL
	ENDC	;USE_TIMER
;
STiK_not_found_s:
	dc.b	'There is no STiK cookie in the jar !',CR,LF
	dc.b	NUL
;
STiK_not_valid_s:
	dc.b	'The STiK cookie data is corrupt !',CR,LF
	dc.b	NUL
;
driver_not_valid_s:
	dc.b	'The main STinG driver is not valid !',CR,LF
	dc.b	NUL
;
layer_not_valid_s:
	dc.b	'The STinG module layer is not valid !',CR,LF
	dc.b	NUL
;
error_title_s:
	dc.b	BEL,CR,LF
	dc.b	'------------'
	M_TITLE
	dc.b	' '
	M_VERSION
	dc.b	'------------',CR,LF
	dc.b	NUL
;
error_tail_s:
	dc.b	BEL,CR,LF,NUL
	EVEN
;----------------------------------------------------------------------------
data_limit:
data_size	=	data_limit-data_start
	SECTION	BSS
bss_start:
;----------------------------------------------------------------------------
		ds.l	200		;subroutine stack >= 100 longs
mystack:	ds.l	1		;top of subroutine stack
;----------------------------------------------------------------------------
used_bss_end:
used_bss_size	=	used_bss_end-bss_start
used_RAM_size	=	(used_bss_size+data_size+(text_limit-RAM_start))
pad_RAM_size	=	RAM_chunk-used_RAM_size+8
		ds.b	pad_RAM_size
;----------------------------------------------------------------------------
bss_limit:
bss_size	=	bss_limit-bss_start
;----------------------------------------------------------------------------
initial_size	=	text_size+data_size+bss_size+$100
resident_size	=	initial_size
;----------------------------------------------------------------------------
		END
;----------------------------------------------------------------------------
;End of file:	CEN_PLEP.S
;----------------------------------------------------------------------------
