(* WINDOWS Package Version 4.0 (C) 1989 by Daniel Singer

Description:
************

  The WINDOWS Package allows programmers in TURBO Pascal (2.0 or higher)
incorportate trouble-free text windows into their own programs.  The WINDOWS
package is completely dynamic (minus 32 bytes) and therefore takes almost no
global data space away from the application.

Technical Info:
***************

Space:

  WINDOWS 4.0 requires 32 bytes of global data space.  Each variable of type
WINDOW takes an additional 4 bytes to store.  When a window is initialized,
some heapspace is allocated to it, the magnitude of which is dependant on the
physical dimensions of the window (see WINDSIZE function). WINDOWS 4.0 requires
4 KILOBYTES of operating overhead while doing a job (display, hide, select).
This overhead space is returned to the heap when the job is done.  Some
heapspace, besides this temporary overhead space, is required to keep house
in WINDOWS 4.0. Unless there are hundreds of valid windows, the magnitude of
this space should be WELL UNDER 1k.

Methods:

  WINDOWS 4.0 does all its "scrap work" on a buffer (that's the 4k temporary
space mentioned above), and not on the screen.  When a WINDOWS process is
invoked, a copy of the screen is placed into the page buffer.  Then, all
manipulation of windows is done on the buffer.  Finally, the buffer is written
to the screen.  This accomplishes two things.  First, the user does not see
the shuffling of windows, which is distracting, when different windows are
selected.  Second, all actions of WINDOWS are clean; windows just appear or
disappear.  See SELECTWINDOW procedure for more information.

Notice to the Public:
*********************

  I worked hard on this package.  I encourage people to write and sell programs
which incorporate all or part of WINDOWS 4.0.  Please modify and enhance this
package to suit your personal needs.  I will ask only this:  give me credit
where credit is do.  If you supply someone with a source listing of a program
which uses WINDOWS 4.0, put my name in there with it.  Second, and finally,
don't sell this package as your own.  That's fraud, illegal, immoral, and
it's not nice.

If you find WINDOWS 4.0 to be useful/educational, please let
me know.  If you find this package good enough to pay for, your contribution
will be greatly appreciated (support a college student!)

  Daniel Singer
  Stripd Tiger Software
  2245 Iroquois Road
  Wilmette, IL  60091-1409


Files required to compile:
**************************

WINDOW4A.PAS -- (this file)
BLOX4A  .BIN -- machine code for optimized data transfer

User Procedures:
****************

initWindowSystem;
      must be called first and only once to set up the system.

initWindow (xL, yT, width, length, title, windowVar);
      allocates heapspace for a window with upper-lefthand corner coordinates
      (xL, yT) with dimensions width, length.  This procedure WILL display
      the window, and put the TITLE centered at the top.  If the given
      dimensions are out of range, for one reason or another, they will be
      truncated to fit the screen.  If that results in an unusable window
      (i.e., width < 3 or height < 3) then the window is not allocated, and
      windowVar is undefined.  WindowVar is SELECTED when this procedure is
      called.  You must SELECTWINDOW or HIDEWINDOW to get out of it.

selectWindow (windowVar);
      puts windowVar on top of any other windows.  If windowVar is not on the
      screen, it is put on the screen.  Its old contents and the cursor are
      restored.

hideWindow (windowVar);
      saves contents of the window and the cursor, removes window from the
      screen, and restores whatever was beneath it.

purgeWindow (windowVar);
      deallocates the heapspace for windowVar.  WindowVar is no longer a
      valid window.  This procedur DOES remove windowVar from the screen.
      The cursor is undefined upon exit.

purgeWindowSystem;
      deallocates all heapspace taken by WINDOWS 4.0.  All windows currently
      instantiated will be purged, as if called sequentially with purgeWindow.
      The cursor is undefined upon exit.

User Functions:
***************

textWidth (windowVar): integer;
      returns the number of text characters that will fit on one line of the
      window WindowVar.

textLength (windowVar): integer;
      returns the number of text lines within the window WindowVar.

windowExists (windowVar): boolean;
      returns truth of window's existance.  If windowVar has not been initial-
      ized or has been purged, windowExists is false.

Summary of Enhancements Since Version 3.1
*****************************************
1. BLOX machine code has been re-optimized in two ways

     a. multiplication by 160 (which requires many machine cycles) was
        replaced with four left-shifts (few cycles) and a multiplication by
        10 (which requires much less machine cycles than 160)

     b. block move routine was changed from a loop with external increments
        and conditionals to a one-line machine instruction

2. User-specified titles are assigned to windows

3. Out-of-range dimensions for a window no longer result in a default window
   being automatically allocated.  Instead, the window is truncated to fit
   the screen.  If that fails, then the window is NOT INSTANTIATED; i.e.,
   nothing is done.

4. Init and Purge modify the screen.  Initwindow will display the window;
   purgeWindow will remove it.

*)

Const
  maxWindowLen =  $800;   {number of words on a full text page}
  titleSize    =    70;   {maximum length of a title}
  Width        =    80;   {screen dimensions in characters}
  yLength      =    25;
  xMin         =     3;   {window must have at least these dimensions}
  yMin         =     3;
  colorSeg     = $B800;   {base segment address of screen buffer}
  bytesPerLine =   160;   {on the screen}

Type
  titleString     = string [titleSize];
  word            = integer;
  screenBufferPtr = ^screenBuffer;
  screenBuffer    = array [0..maxWindowLen] of word;
  window          = ^windowRec;
  windowRec       = record
                      xL, yT, xWidth, yLen,          {logical dimensions}
                      xCursor, yCursor,              {set internally}
                      windowSize : integer;          {in BYTES}
                      buffer     : screenBufferPtr;  {holds what's underneath}
                      active,                        {true if cursor in window}
                      onScreen   : boolean;          {true if window visible}
                      nextWindow : window            {points along the list}
                    end;
  windowStackPtr  = ^windowStackRec;
  windowStackRec  = record
                      w        : window;
                      previous : windowStackPtr
                    end;

Var
  pageBuffer,                                 {where all the fun stuff happens}
  iBuffer     : screenBufferPtr;   {internal buffer, made and killed as needed}
  firstWindow : window;                       {initialized by initWindowSystem}
  windowStack : windowStackPtr;

procedure initWindowSystem;  {initializes things}
  begin {pr init system}
    windowStack := nil;
    firstWindow := nil
  end; {pr init system}

procedure pushWindow (var ws : windowStackPtr; pushW : window);
  var
    element: windowStackPtr;

  begin {pr pushw}
    new (element);
    element^.w := pushW;
    element^.previous := ws;
    ws := element
  end; {pr pushw}

procedure popWindow (var ws : windowStackPtr; var popW : window);
  var
    victim : windowStackPtr;

  begin {pr popw}
    popW := ws^.w;   {get top of the stack}
    victim := ws;
    if not (ws = nil) then
      begin
        ws:= ws^.previous;
        dispose (victim)
      end
  end;  {pr popw}

function windSize (xWidth, yLen : integer) : integer;  {returned in BYTES}
  begin {fn windSize}
    windSize := 2 * (xWidth + 1) * (yLen + 1)
  end; {fn windSize}

procedure killBuffer (var bp : screenBufferPtr; siz : integer);  {de-allocates}
  begin {pr kill}
    freeMem (bp, siz)
  end; {pr kill}

procedure makeBuffer (var bp : screenBufferPtr; siz : integer);  {allocates}
  begin {pr make}
    getMem (bp, siz)
  end; {pr make}

procedure makePageBuffer;  {used for doing screen operations in the background}
  begin {pr make page}
    new (pageBuffer)
  end;  {pr make page}

procedure purgePageBuffer;
  begin {pr purge page}
    dispose (pageBuffer)
  end; {pr purge page}

procedure loadPageBuffer;  {from color seg}
  begin
    move (mem[colorSeg: 0], pageBuffer^, maxWindowLen * 2) {save page}
  end;

procedure showPageBuffer; {to color seg}
  begin
    move (pageBuffer^, mem[colorSeg: 0], maxWindowLen * 2)
  end;

procedure equate (var b1, b2 : screenBuffer; siz : integer);
  {copies buffer b2 into buffer b1.  since window buffers are only allocated
  to be as large as necessary, the size parameter is important towards keeping
  everything under control. }

  begin {pr eq}
    move (b2, b1, siz)
  end; {pr eq}

procedure getBox (xLeft, yTop, xWidth, yLen : integer;
               boxPointer, pageBuffer : screenBufferPtr); EXTERNAL 'a:BLOX4a.BIN';

procedure putBox (xLeft, yTop, xWidth, yLen : integer;
               boxPointer, pageBuffer : screenBufferPtr); EXTERNAL getBox[3];


function windowExists (test: window): boolean; { USER }
  {checks windowList for a match}

  var
    w: window;
    exist: boolean;

    begin {fn exist}
      exist := false;
      w     := firstWindow;
      if not (test = nil) then
        while not ((w = nil) or (exist)) do
          begin {still looking}
            exist := (w = test);
            if not exist then
              w := w^.nextWindow
          end;  {still looking}
      windowExists := exist
    end;  {fn exist}

procedure windowOFF (var thisWindow : window);

  {puts the screen under the window back in order and returns with an undefined
   cursor in the full text window;  ONSCREEN field is NOT made FALSE, nor
   should it be, because it's used in selectWindow.}

  begin {pr windowOFF}
    if windowExists (thisWindow) then
      if thisWindow^.onScreen then
        with thisWindow^ do
          begin
            if active then
              begin
                xCursor := whereX;   {save cursor position inside the window}
                yCursor := whereY;
                active  := false
              end;
            makeBuffer (iBuffer, windowSize);
            getBox (xL, yT, xWidth + 1, yLen + 1, iBuffer, pageBuffer);
            putBox (xL, yT, xWidth + 1, yLen + 1, buffer, pageBuffer);
            equate (buffer^, iBuffer^, windowSize);
            killBuffer (iBuffer, windowSize);
            window (1, 1, width, yLength)
          end
  end; {pr windowOFF}


procedure windowON (var thisWindow : window);

  {saves area beneath the window, puts thisWindow on the screen, and sets
   its cursor; ACTIVE field is NOT made TRUE, nor should it be.}

  begin {pr windowON}
    if windowExists (thisWindow) then
      with thisWindow^ do
       begin
          window (1, 1, width, yLength);
          onScreen := true;
          makeBuffer (iBuffer, windowSize);
          getBox (xL, yT, xWidth + 1, yLen + 1, iBuffer, pageBuffer);
          putBox (xL, yT, xWidth + 1, yLen + 1, buffer, pageBuffer);  {old contents}
          equate (buffer^, iBuffer^, windowSize);
          killBuffer (iBuffer, windowSize);
          window (xL + 1, yT + 1, xL + xWidth - 1, yT + yLen - 1);
          gotoXY (xCursor, yCursor)
        end
  end; {pr windowON}

Procedure selectWindow (var thisWindow : window);   {USER}

  {check to see if thisWindow exists, if it doesn't the process exits,
  otherwise, the window is put on top of all currently onScreen windows,
  and the cursor is placed at its old location.  The window's old contents
  are restored to the window.

  implementation: WINDOWS maintains a stack which is logically equivalent to
  the windows on the screen (i.e., top of stack is the most recent window
  selected/put on the screen).  When SELECTWINDOW is called, the windows on
  the stack are popped off (and off the screen) until the window in question
  is on top of the stack.  This effectively removes any windows which were
  on top of the one we wish to select.  Next, all the windows that were
  removed are replaced, in such a way that their relative order is maintained.
  Finally, the window in question is put on the screen, on top of all the
  other windows.  It is, naturally, put on the top of the stack.  The net
  effect is to pull the window we want out from under any windows which were
  on top if it.  All this window flipping is done on the page buffer, so the
  user doesn't see it.}

  var
    offScreen  : windowStackPtr;  {temp stack}
    testWindow : window;


  begin {pr selectWindow}
    if windowExists (thisWindow) then
      begin {exists}
        makePageBuffer;
        loadPageBuffer;

        offScreen  := nil;  {init temp stack}
        testWindow := nil;
        while not ((windowStack = nil) OR (testWindow = thisWindow)) do
          begin {removing windows on the screen}
              popWindow (windowStack, testWindow); {get top/stack}
                if testWindow^.onScreen then
                  windowOFF (testWindow); {pull it off the screen}
                if not (testWindow = thisWindow) then
                  pushWindow (offScreen, testWindow)  {record order}
            end;  {removing windows on the screen}

{notes: at this point, all windows which were on top of thisWindow have been
removed from the screen, including thisWindow.  Now, we replace the all the
other windows and put ours on top}

          while not (offScreen = nil) do  {replace windows}
            begin
              popWindow (offScreen, testWindow);
              if testWindow^.onScreen then
                windowON (testWindow);
              testWindow^.active := false;
              pushWindow (windowStack, testWindow)
            end;
        windowON (thisWindow); {turn our window on}
        thisWindow^.active := true;
        pushWindow (windowStack, thisWindow);  {put it on the list}
        showPageBuffer;
        purgePageBuffer
      end {exists}
  end; {pr selectWindow}

procedure initWindow (x1, y1, wide, leng : integer; title : titleString;
                      var thisWindow: window);   {USER}

  {makes window coordinates valid, allocates buffer space}

  var
    windowValid : boolean;

  procedure drawBorder (xL, yT, xWidth, yLen: integer);
    const
      blCorner = ''; {212}
      brCorner = ''; {190}
      tlCorner = ''; {213}
      trCorner = ''; {184}
      horizont = ''; {205}
      vertical = ''; {179}

    var
      x, y: integer;

    begin {pr drawBorder}
      gotoXY (xL, yT);
      write (tlCorner);
      gotoXY (xL + xWidth, yT);
      write (trCorner);
      gotoXY (xL, yT + yLen);
      write (blCorner);
      gotoXY (xL + xWidth, yT + yLen);
      write (brCorner);
      for x:= xL + 1 to xL + xWidth - 1 do
        begin
          gotoXY (x, yT);
          write (horizont);
          gotoXY (x, yT + yLen);
          write (horizont)
        end;
      for y:= yT + 1 to yT + yLen - 1 do
        begin
         gotoXY (xL, y);
           write (vertical);
          gotoxY (xL + xWidth, y);
          write (vertical)
        end
    end;  {pr drawBorder}

  procedure makeValid (var xL, yT, xWidth, yLen : integer;
                       var success : boolean);
    {window is truncated to fit the screen.  if window is invalid, SUCCESS
    is false on exit}

    begin {pr make valid}
      success := true;
        while xL + xWidth > width do
          xL := pred (xL);
        while yT + yLen > yLength do
          yT := pred (yT);
        if xL < 1 then
          success := false
        else
          if yT < 1 then
            success := false
          else
            begin {so far, so good; top coordinate on screen and in range}
              while (xL + xWidth) > width do  {truncate width}
                xWidth := pred (xWidth);
              while (yT + yLen) >= yLength do {truncate height}
                yLen := pred (yLen);
              if xWidth < xMin then
                success := false
              else
                if yLen < yMin then
                  success := false
            end {if...}
    end; {pr makeValid}

  begin {pr init}
    if windowStack^.w^.active then {must turn deactivate window below}
      with windowStack^.w^ do
        begin
          active := false;
          xCursor := whereX;
          yCursor := whereY
        end; {with/if}
    makeValid (x1, y1, wide, leng, windowValid); {fixes minor errors, if poss.}
    if windowValid then
      begin
        new (thisWindow);
        with thisWindow^ do
          begin
            active     := true;
            onScreen   := true;
            xL         := x1;
            yT         := y1;
            xWidth     := wide;
            yLen       := leng;
            windowSize := windSize (xWidth, yLen);
            makeBuffer (buffer, windowSize);
            nextWindow := firstWindow {link}
          end;
        firstWindow := thisWindow;   {link}
        makePageBuffer;
        loadPageBuffer;
        getBox (x1, y1, 1 + wide, 1 + leng, thisWindow^.buffer, pageBuffer);
        purgePageBuffer;
        window (1, 1, width, ylength);
        drawBorder (x1, y1, wide, leng);
        gotoXY (x1 + ((wide - length (title)) div 2), y1);
        write (title);
        thisWindow^.xCursor := 1;
        thisWindow^.yCursor := 1;
        pushWindow (windowStack, thisWindow);
        window (x1 + 1, y1 + 1, x1 + wide - 1, y1 + leng - 1);
        clrScr
      end {if ok window}
  end;  {pr init}

Procedure hideWindow (var thisWindow: window); { USER }

  {removes selected window from the screen, but does NOT purge
    cursor is undefined at exit}

  var
    nothing: window;  {dummy}

  begin {pr hide}
    selectWindow (thisWindow);  {put it on top}
    makePageBuffer;
    loadPageBuffer;
    windowOFF (thisWindow);
    showPageBuffer;
    purgePageBuffer;
    thisWindow^.active   := false;
    thisWindow^.onScreen := false;
    popWindow (windowStack, nothing);  {pop it from keep track stack}
    window (1, 1, width, yLength);
  end;  {pr hide}

procedure purgeWindow (var thisWindow: window);   { USER }

  {removes window from screen, de-allocates buffer space and window
   record from the heap}

  var
    w: window;

  begin {pr purge}
    if windowExists (thisWindow) then
      begin
        hideWindow (thisWindow);
        if thisWindow = firstWindow then
          firstWindow := firstWindow^.nextWindow
        else
          begin
            w := firstWindow;
            while not (w^.nextWindow = thisWindow) do
              w := w^.nextWindow;
            w^.nextWindow:= thisWindow^.nextWindow
          end;
        with thisWindow^ do
          killBuffer (buffer, windowSize);
        dispose (thisWindow);
        thisWindow := nil
      end {exists}
  end;  {pr purge}

procedure purgeWindowSystem;   {de-allocates EVERYTHING} { USER }

  var
    n, w: window;

  begin {pr purge window sys}
    w := firstWindow;
    while not (w = nil) do
      begin
        n := w;
        purgeWindow (w);
        w := n^.nextWindow
      end;
    while not (windowStack = nil) do   {dispose of stack}
      popWindow (windowStack, n)
  end;  {pr purge window sys}

function textWidth (w: window): integer;  {USER}
  begin
    textWidth := w^.xWidth - 2
  end;

function textLength (w: window): integer;  {USER}
  begin
    textLength := w^.yLen - 1
  end;


