/*******************************************************************
 * Fonctions de gestion de fichiers de paramtres (.inf)           *
 *                                                                 *
 * Fichier source des fonctions.                                   *
 *                                                                 *
 * (C) 1996 - Philippe Castella                                    *
 *                                                                 *
 *******************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <string.h>
#include "InitFile.h"

/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*
 * Dclaration des fonctions prives au module.                    *
 *=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
static int ReadLine (FILE *ficIni, char *ligne);
static void ReadFile (FILE *fileHandle, InitFile **strIni);
static InfSection *AddSection (InitFile **strIni, char *section);
static void AddVar (InitFile **strIni, char *section, char *var, char *val);
static void SaveFile (InitFile *strIni);
static void GetValeur (InitFile *strIni, char *section, char *var, char *buf);

/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*
 * Dfinition des fonctions de la librairie.                       *
 *=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/

/*-----------------------------------------------------------------*
 * InfCreate : fonction de cration/ouverture d'un fichier .inf    *
 *                                                                 *
 * Entre : nom du fichier .inf                                    *
 * Sortie : un pointeur sur la nouvelle structure cre            *
 *                                                                 *
 * Dernire modification : 11 dcembre 1996                        *
 *-----------------------------------------------------------------*/
InitFile *InfCreate (char *nomFic)
{
	InitFile *strIni;
	FILE *fileHandle;

	/* Allocation de la mmoire ncessaire  la structure */
	strIni = (InitFile *)malloc (sizeof (InitFile));
	if (strIni)
	{
		/* Ouverture du fichier */
		fileHandle = fopen (nomFic, "r");

		/* Mise jour de la structure */
		strIni->FileName = (char *)malloc (strlen(nomFic) + 1);
		if (! strIni->FileName)
		{	/* Si Pb, on libre la mmoire et on s'en va */
			free(strIni);
			return (InitFile *)0;
		}
		strcpy (strIni->FileName, nomFic);
		strIni->FirstSection = (InfSection *)0;

		/* Si fichier existe, on lit son contenu */
		if (fileHandle)
		{	/* Lecture ligne  ligne du fichier = remplir la structure */
			ReadFile (fileHandle, &strIni);

			/* Fermeture du fichier */
			fclose (fileHandle);
		}
	}
	else
		strIni = (InitFile *)0;

	return strIni;
}

/*-----------------------------------------------------------------*
 * InfWriteBool    : Ajout d'une variable de type boolen.         *
 * InfWriteInteger : Ajout d'une variable de type entier.          *
 * InfWriteString  : Ajout d'une variable de type chane de car.   *
 *                                                                 *
 * Entre : nom de la section                                      *
 *          nom de la variable                                     *
 *          valeur de la variable                                  *
 * I/O    : L'adresse du pointeur sur la structure de donnes.     *
 *                                                                 *
 * Dernire modification : 11 dcembre 1996                        *
 *-----------------------------------------------------------------*/
void InfWriteBool (InitFile **strIni, char *section, char *var, short val)
{
	if (val)
		AddVar (strIni, section, var, "1");
	else
		AddVar (strIni, section, var, "0");
}

void InfWriteInteger (InitFile **strIni, char *section, char *var, long  val)
{
	char buf[10];

	ltoa (val, buf, 10);
	AddVar (strIni, section, var, buf);
}

void InfWriteString (InitFile **strIni, char *section, char *var, char *val)
{
	AddVar (strIni, section, var, val);
}

/*-----------------------------------------------------------------*
 * InfReadBool    : Lecture d'une variable de type boolen.        *
 * InfReadInteger : Lecture d'une variable de type entier.         *
 * InfReadString  : Lecture d'une variable de type chane de car.  *
 *                                                                 *
 * Entre : Le pointeur sur la structure de donnes.               *
 *          nom de la section                                      *
 *          nom de la variable                                     *
 *          valeur par dfaut de la variable si non trouve        *
 * I/O    : valeur de la variable                                  *
 *                                                                 *
 * Dernire modification : 11 dcembre 1996                        *
 *-----------------------------------------------------------------*/
void InfReadBool (InitFile *strIni, char *section, char *var, short def, short *val)
{
	char buf[129];

	/* Rcupration de la valeur de la variable */
	GetValeur (strIni, section, var, buf);

	/* Si rien trouv -> retourner val par dfaut, sinon la valeur */
	if (! *buf)
		*val = def;
	else
		*val = (short)atoi (buf);
}

void InfReadInteger (InitFile *strIni, char *section, char *var, long  def, long  *val)
{
	char buf[129];

	/* Rcupration de la valeur de la variable */
	GetValeur (strIni, section, var, buf);

	/* Si rien trouv -> retourner val par dfaut, sinon la valeur */
	if (! *buf)
		*val = def;
	else
		*val = atol (buf);
}

void InfReadString (InitFile *strIni, char *section, char *var, char *def, char  *val)
{
	char buf[129];

	/* Rcupration de la valeur de la variable */
	GetValeur (strIni, section, var, buf);

	/* Si rien trouv -> retourner val par dfaut, sinon la valeur */
	if (! *buf)
		strcpy (val, def);
	else
		strcpy (val, buf);
}

/*-----------------------------------------------------------------*
 * InfFree : Fonction de sauvegarde du fichier .inf et de          *
 *           restitution de la mmoire alloue dynamiquement.      *
 *                                                                 *
 * Entre : Le pointeur sur la structure de donnes.               *
 *                                                                 *
 * Dernire modification : 11 dcembre 1996                        *
 *-----------------------------------------------------------------*/
void InfFree (InitFile *strIni)
{
	/* Si on ne pointe pas sur du vide */
	if (strIni != (InitFile *)0)
	{
		/* Sauvegarde du fichier */
		SaveFile (strIni);

		/* Effacement de la structure */
		free (strIni->FileName);
		free (strIni);
	}
}

/*-----------------------------------------------------------------*
 * InfDeleteVar : Fonction de suppression d'une variable.          *
 *                                                                 *
 * Entre : Le pointeur sur la structure de donnes.               *
 *          Le nom de la section                                   *
 *          Le nom de la variable                                  *
 *                                                                 *
 * Dernire modification : 25 dcembre 1996                        *
 *-----------------------------------------------------------------*/
void InfDeleteVar (InitFile **strIni, char *section, char *var)
{
	InfSection *strSection;
	InfVar *strVar, *oldVar;

	if (*strIni)
	{
		/* On recherche si la section existe bien */
		strSection = (*strIni)->FirstSection;
		while (strSection != (InfSection *)0)
		{
			if (strcmp(strSection->SectionName, section) == 0)
			{
				strVar = oldVar = strSection->FirstVar;
				while (strVar != (InfVar *)0)
				{
					if (strcmp (strVar->VarName, var) == 0)
					{
						oldVar->NextVar = strVar->NextVar;
						free (strVar->VarName);
						free (strVar->VarVal);
						free (strVar);
						return;
					}
					oldVar = strVar;
					strVar = strVar->NextVar;
				}
			}
			strSection = strSection->NextSection;
		}
	}
}

/*-----------------------------------------------------------------*
 * InfDeleteSection : Fonction de suppression d'une section.       *
 *                                                                 *
 * Entre : Le nom de la section                                   *
 *          Le nom de la variable                                  *
 * I/O    : L'adresse du pointeur sur la structure de donnes.     *
 *                                                                 *
 * Dernire modification : 25 dcembre 1996                        *
 *-----------------------------------------------------------------*/
void InfDeleteSection (InitFile **strIni, char *section)
{
	InfSection *strSection, *oldSection;

	if (*strIni)
	{
		/* On recherche si la section existe bien */
		strSection = (*strIni)->FirstSection;
		oldSection = strSection;
		while (strSection != (InfSection *)0)
		{
			if (strcmp(strSection->SectionName, section) == 0)
			{
				if (oldSection == strSection)
					(*strIni)->FirstSection = strSection->NextSection;
				else
					oldSection->NextSection = strSection->NextSection;
				free (strSection->SectionName);
				free (strSection);
				return;
			}
			oldSection = strSection;
			strSection = strSection->NextSection;
		}
	}
}

/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*
 * Dfinition des fonctions prives de la librairie                *
 *=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/

/*-----------------------------------------------------------------*
 * ReadLine : Fonction de lecture d'une ligne dans un fichier ASCII*
 *                                                                 *
 * Entre : Le descripteur du fichier ouvert avec fopen            *
 * I/O    : un buffer pour stocker les donnes lues                *
 * Sortie : Le nombre de caractres composant la ligne, -1 si EOF  *
 *                                                                 *
 * Dernire modification : 11 dcembre 1996                        *
 *-----------------------------------------------------------------*/
static int ReadLine (FILE *ficIni, char *ligne)
{
	register char car;
	register int cpt = 0;

	/* Lire car. par car. jusqu' fin de ligne ou fin de fichier   */
	while (! feof(ficIni) && ((car = fgetc(ficIni)) != '\n'))
		ligne[cpt++] = car;
	ligne[cpt] = '\0';

	/* Si fin de fichier, on le signale */
	if (feof(ficIni))
		cpt = -1;

	/* Renvoie le nombre de caractres lus */
	return cpt;
}

/*-----------------------------------------------------------------*
 * ReadFile : Fonction de lecture des donnes contenue dans le     *
 *            fichier .inf                                         *
 *                                                                 *
 * I/O    : L'adresse du pointeur sur la structure de donnes.     *
 *                                                                 *
 * Dernire modification : 11 dcembre 1996                        *
 *                                                                 *
 * Remarque : une ligne ne doit pas dpasser 256 caractres c--d  *
 *            128 car. pour la variable et autant pour la valeur.  *
 *-----------------------------------------------------------------*/
static void ReadFile (FILE *fileHandle, InitFile **strIni)
{
	register int nbCar, i, j;
	char section[65];
	char read_buf[257];
	char var[129];
	char val[129];

	/* Lecture du fichier ligne par ligne */
	while ((nbCar = ReadLine (fileHandle, read_buf)) > -1)
	{
		if (nbCar > 0)
		{	/* Pas une ligne vide */
			if (read_buf[0] == '[') /* Si dbut de Section */
			{	/* [nom de la section] */
				i = 1;
				j = 0;
				/* Lire titre de la section */
				while (read_buf[i] != ']' && read_buf[i] != '\0')
					section[j++] = read_buf[i++];
				section[j] = '\0';

				/* Cration de la section dans la structure de donnes */
				AddSection(strIni, section);
			}
			else
			{	/* var=val */
				i = 0;
				j = 0;
				/* Lire nom de la variable */
				while (read_buf[i] != '=')
					var[j++] = read_buf[i++];
				var[j] = '\0';
				i++;

				/* Lire valeur de la variable */
				j = 0;
				while (read_buf[i] != '\0')
					val[j++] = read_buf[i++];
				val[j] = '\0';

				/* Cration de la variable dans la structure de donnes*/
				AddVar(strIni, section, var, val);
			}
		}
	}
}

/*-----------------------------------------------------------------*
 * AddSection : Fonction permettant de crer une nouvelle section  *
 *              si celle-ci n'existe pas dj.                     *
 *                                                                 *
 * Entre : le nom de la section                                   *
 * I/O    : L'adresse du pointeur sur la structure de donnes.     *
 *                                                                 *
 * Dernire modification : 11 dcembre 1996                        *
 *-----------------------------------------------------------------*/
static InfSection *AddSection (InitFile **strIni, char *section)
{
	InfSection *strSection = (*strIni)->FirstSection;
	InfSection *NewSection;

	/* On recherche si la section existe dj */
	while (strSection != (InfSection *)0)
	{
		if (strcmp(strSection->SectionName, section) == 0)
			return strSection;
		strSection = strSection->NextSection;
	}

	/* Si elle n'existe pas, on la rajoute */
	NewSection = (InfSection *)malloc (sizeof(InfSection));
	if (NewSection)
	{
		/* Allocation mmoire pour le nom de la section */
		NewSection->SectionName = (char *)malloc (strlen(section) + 1);
		if (! NewSection->SectionName)
		{
			free (NewSection);
			return (InfSection *)0;
		}

		/* Mettre  jour le contenu de la structure */
		strcpy (NewSection->SectionName, section);
		NewSection->NextSection = (InfSection *)0;
		NewSection->FirstVar = (InfVar *)0;

		strSection = (*strIni)->FirstSection;
		/* Si premire section ...*/
		if (strSection == (InfSection *)0)
		{
			(*strIni)->FirstSection = NewSection;
			return (*strIni)->FirstSection;
		}
		else /* ...sinon */
		{
			while (strSection->NextSection != (InfSection *)NULL)
				strSection = strSection->NextSection;

			strSection->NextSection = NewSection;
  		return strSection->NextSection;
  	}
	}
	return (InfSection *)0;
}

/*-----------------------------------------------------------------*
 * AddVar : Fonction permettant de crer une nouvelle variable dans*
 *          une section donnes si elle n'existe pas dj.         *
 *                                                                 *
 * Entre : le nom de la section                                   *
 *          le nom de la variable                                  *
 *          la valeur de la variable                               * 
 * I/O    : L'adresse du pointeur sur la structure de donnes.     *
 *                                                                 *
 * Dernire modification : 11 dcembre 1996                        *
 *-----------------------------------------------------------------*/
static void AddVar (InitFile **strIni, char *section, char *var, char *val)
{
	InfSection *strSection;
	InfVar *strVar, *NewVar;
	short ok = FALSE;

	if (*strIni)
	{
		/* Recherche la bonne section (cration si ncessaire) */
		strSection = AddSection (strIni, section);
		if (strSection == (InfSection *)0)
			return;

		/* On recherche si la variable existe dj */
		strVar = strSection->FirstVar;
		while (strVar != (InfVar *)0)
		{
			/* Si on trouve la variable, on sort de la boucle */
			if (strcmp(strVar->VarName,var) == 0)
			{
				ok = TRUE;
				break;
			}
			strVar = strVar->NextVar;
		}
 
		/* Si variable non trouve, on la cre */
		if (! ok)
		{
			NewVar = (InfVar *)malloc (sizeof(InfVar));
			if (NewVar)
			{
				NewVar->VarName = (char *)malloc(strlen(var) + 1);
				if (! NewVar->VarName)
				{
					free (NewVar);
					return;
				}
				strcpy (NewVar->VarName, var);
				NewVar->NextVar = (InfVar *)0;
      	
				strVar = strSection->FirstVar;
				if (strVar == (InfVar *)0)
				{
					strSection->FirstVar = NewVar;
					strVar = strSection->FirstVar;
				}
				else
				{
					while (strVar->NextVar != (InfVar *)NULL)
						strVar = strVar->NextVar;

					strVar->NextVar = NewVar;
					strVar = strVar->NextVar;
				}
			} 	
		}

		/* Mise  jour de la valeur de la variable */
		strVar->VarVal = (char *)malloc (strlen(val) + 1);
		if (strVar->VarVal)
			strcpy (strVar->VarVal, val);
	}
}

/*-----------------------------------------------------------------*
 * SaveFile : Fonction permettant de sauvegarder le fichier .inf   *
 *            tout en effacant au fur er  mesure la structure.    *
 *                                                                 *
 * Entre : Le pointeur sur la structure de donnes.               *
 *                                                                 *
 * Dernire modification : 11 dcembre 1996                        *
 *-----------------------------------------------------------------*/
static void SaveFile (InitFile *strIni)
{
	FILE *fileHandle;
	InfSection *strSection, *oldSection;
	InfVar *strVar, *oldVar;

	/* Ouvrir fichier paramtre */
	fileHandle = fopen (strIni->FileName, "w");

	strSection = strIni->FirstSection;
	while (strSection != (InfSection *)0)
	{
		/* Ecriture du nom de la section */
		fprintf (fileHandle, "[%s]\n", strSection->SectionName);

		/* Premire variable */
		strVar = strSection->FirstVar;
		while (strVar != (InfVar *)0)
		{
			/* Ecriture de la variable et de sa valeur */
			fprintf (fileHandle, "%s=%s\n", strVar->VarName, strVar->VarVal);

			/* Variable suivante */
			oldVar = strVar;
			strVar = strVar->NextVar;

			/* Supprimer variable, valeur et structure */
			free (oldVar->VarName);
			free (oldVar->VarVal);
			free (oldVar);
		}
		fprintf (fileHandle, "\n");
		/* Section suivante */
		oldSection = strSection;
		strSection = strSection->NextSection;

		/* Supprimer nom et structure */
		free (oldSection->SectionName);
		free (oldSection);
	}

	/* Fermer fichier paramtre */
	fclose (fileHandle);

}

/*-----------------------------------------------------------------*
 * GetValeur : Fonction permettant de rcuprer le contenu d'une   *
 *             variable donne d'une section donne.               *
 *                                                                 *
 * Entre : Le pointeur sur la structure de donnes.               *
 *          le nom de la section                                   *
 *          le nom de la variable                                  *
 * I/O    : Le buffer qui va contenir la valeur                    *
 *                                                                 *
 * Dernire modification : 11 dcembre 1996                        *
 *-----------------------------------------------------------------*/
static void GetValeur (InitFile *strIni, char *section, char *var, char *buf)
{
	InfSection *strSection;
	InfVar *strVar;

	/* Recherche de la section */
	strSection = strIni->FirstSection;
	while ((strSection != (InfSection *)0) && strcmp(strSection->SectionName, section) != 0)
		strSection = strSection->NextSection;

	/* Si section non trouve, buffer vide */
	if (strSection == (InfSection *)0)
		strcpy (buf, "");
	else
	{
		/* Recherche de la variable */
		strVar = strSection->FirstVar;
		while ((strVar != (InfVar *)0) && strcmp(strVar->VarName, var) != 0)
			strVar = strVar->NextVar;

		/* Si variable non trouve, buffer vide */
		if (strVar == (InfVar *)0)
			strcpy (buf, "");
		else /* sinon copier la valeur dans le buffer */
			strcpy (buf, strVar->VarVal);
	}
}

/*******************************************************************/