/* faces.c - FACES.INF reader
 * FontForm, Copyright  1990,1991 Atari Corporation
 * ======================================================================
 * 910729 kbad	Added vst_error() handling in FindFonts().
 * 901219 kbad	Final sweep
 * 901202 kbad	Cleanup
 * 901127 kbad	From PageOMat
 */

#include <stdio.h>
#include <ctype.h>

#if LATTICE
#include <stdlib.h>
#include <string.h>
#include <osbind.h>
#include <vdi.h>
#include <aes.h>
#endif

#if __TURBOC__
#include <stdlib.h>
#include <string.h>
#include <tos.h>
#include <vdi.h>
#include <aes.h>
#endif

#if MWC
#include <osbind.h>
#include <vdibind.h>
#include <aesbind.h>
#include <gemdefs.h>
#include <obdefs.h>
#include <types.h>
char *bsearch();
char *calloc();
#endif

#if __GNUC__
#include <stdlib.h>
#include <string.h>
#include <osbind.h>
#include <vdibind.h>
#include <aesbind.h>
#include <gemdefs.h>
#include <obdefs.h>
#include <types.h>
#endif

#include "portab.h"
#include "gemerror.h"
#include "aesutil.h"
#include "fontform.h"
#include "fsmbind.h"
#include "faces.h"

/*
 * xqsort and xcalloc defines are required to support MWC's non-ANSI
 * definitions of qsort and calloc.
 */
#if MWC
#define xcalloc(n, size)	    calloc((unsigned)(n), (unsigned)(size))
#define xqsort(base, n, size, cmp)  qsort(base, (int)(n), (int)(size), cmp)
#else
#define xcalloc(n, size)	    calloc(n, size)
#define xqsort(base, n, size, cmp)  qsort(base, n, size, cmp)
#endif


/* External references - from fontform.c
 * ----------------------------------------------------------------------
 */

EXTERN WORD	ws[];
EXTERN WORD	wsid;
EXTERN WORD	nfonts;
EXTERN char	*alertp, alert[];


/* Global variables
 * ----------------------------------------------------------------------
 */

GLOBAL Face	*faces;
GLOBAL WORD	nfaces;
GLOBAL WORD	maxpts;


/* Local variables
 * ----------------------------------------------------------------------
 */

MLOCAL Font	*fonts;
MLOCAL WORD	nfsmfonts;


/* Local routines
 * ----------------------------------------------------------------------
 */

MLOCAL long readfile __PROTO(( const char *fname, VOIDP *pbuf, long *plen ));
MLOCAL int  cmpFont __PROTO(( const Font *a, const Font *b ));
MLOCAL int  cmpFace __PROTO(( const Face *a, const Face *b ));
MLOCAL char *nextline __PROTO(( char *bufend, char *s ));
MLOCAL Font *GetFont __PROTO(( char *s ));
MLOCAL long MakeFaces __PROTO(( char *buf, long len ));
MLOCAL long FindFonts ( VOID );


/* FACES functions
 * ======================================================================
 */

/* ----------------------------------------------------------------------
 * Malloc a buffer for a file and read it into the buffer.
 */
MLOCAL long	readfile( fname, pbuf, plen )
const char	*fname;
VOIDP		*pbuf;
long		*plen;
{
    /*
     * DTA struct declaration is local to this routine, because
     * various compilers keep it in various different include files.
     * Might I suggest a new standard include file: <tosdefs.h>?
     * Let's talk about it.
     */
    struct _dta {
	char	resv[21];
	char	attr;
	long	time;
	long	size;
	char	name[14];
    }		dta, *savedta;

    VOIDP	buf;
    long	len;
    short	fh;

    savedta = (struct _dta *)Fgetdta();
    Fsetdta( &dta ); /* suspicious pointer conversion in TC */
    Fsfirst( fname, 7 );
    Fsetdta( savedta ); /* suspicious pointer conversion in TC */

    *pbuf = NULL;
    *plen = 0L;

    if( !dta.size )
	return AEOK;

    buf = (VOIDP)Malloc( dta.size );
    if( !buf )
        return AENSMEM;


    if( (fh = (short)Fopen(fname, 0)) < 0 )
    {
	Mfree( buf );
	return (long)fh;
    }

    len = Fread( fh, dta.size, buf );
    Fclose( fh );
    if( len != dta.size )
    {
	Mfree( buf );
	return AEREADF;
    }

    *pbuf = buf;
    *plen = len;
    return AEOK;
}

/* ----------------------------------------------------------------------
 * Compare 2 fonts by font ID
 */
MLOCAL int	cmpFont( a, b )
const Font	*a;
const Font	*b;
{
    if( a->id > b->id ) return 1;
    if( a->id < b->id ) return -1;
    return 0;
}

/* ----------------------------------------------------------------------
 * Compare 2 face names which start with a leading space.
 * Toss out any leading "ITC" or "CG" before comparing.
 */
MLOCAL int	cmpFace( a, b )
const Face	*a;
const Face	*b;
{
    const char	*sa = a->name;
    const char	*sb = b->name;

    if( strncmp(sa, " ITC", 4) == 0 )
	sa += 4;
    else if( strncmp(sa, " CG", 3) == 0 )
	sa += 3;

    if( strncmp(sb, " ITC", 4) == 0 )
	sb += 4;
    else if( strncmp(sb, " CG", 3) == 0 )
	sb += 3;

    return strcmp( sa, sb );
}

/* ----------------------------------------------------------------------
 * Return pointer to the beginning of the next line, or end of buffer.
 */
MLOCAL char	*nextline( bufend, s )
char		*bufend;
char		*s;
{
    while( s < bufend && *s++ != '\n' )
    ;
    return s;
}

/* ----------------------------------------------------------------------
 * Return pointer to Font struct that matches the ID string, or NULL.
 */
MLOCAL Font	*GetFont( s )
char		*s;
{
    Font	f;

    f.id = atoi(s);

    return (Font *)bsearch( &f, fonts, (size_t)nfsmfonts, sizeof(Font),
			    (CMPFUNC)cmpFont );
}

/* ----------------------------------------------------------------------
 * Convert a text buffer of face information into a faces array.
 * Return AENSMEM if we run out of memory, or
 * AERROR if there are no faces available, otherwise return AEOK.
 */
MLOCAL long	MakeFaces( buf, len )
char		*buf;
long		len;
{
    char	*s, *bufend;
    short	i, j;
    size_t	tnfaces;
    Face	*tfaces;

    bufend = buf + len;

    /*
     * Count the number of faces in FACES.INF, then
     * allocate an array of faces to hold them all.
     */
    tnfaces = 0;
    s = buf;
    do {
	if( *s++ == '*' )
	    ++tnfaces;
	s = nextline( bufend, s );
    } while ( s < bufend );

    if( !tnfaces )
	return AERROR;

    tfaces = (Face *)xcalloc( tnfaces, sizeof(Face) );
    if( !tfaces )
	return AENSMEM;

    /*
     * Parse FACES.INF:
     *
     * *FACENAME
     * Fnnn fontname
     * Innn italname
     * Dnnn boldname
     * Onnn bolditalname
     * S,stylename,fontid,pts,set,skew,fx (unimplemented)
     * Any other character starting a line is a comment.
     */
    s = buf;
    i = -1;
    do {

	if( *s == '*' )
	{
	    ++s;
	    ++i;
	    tfaces[i].name[0] = ' ';
	    j = 0;
	    while( s < bufend && ++j < 31 && isprint(*s) )
		tfaces[i].name[j] = *s++;
	    tfaces[i].name[j] = 0;
	}

	else if( i >= 0 )
	{
	    switch( *s++ )
	    {
		case 'F':
		    if( (tfaces[i].font = GetFont(s)) != NULL )
			tfaces[i].fido |= FONTF;
		break;
		case 'I':
		    if( (tfaces[i].ital = GetFont(s)) != NULL )
			tfaces[i].fido |= FONTI;
		break;
		case 'D':
		    if( (tfaces[i].bold = GetFont(s)) != NULL )
			tfaces[i].fido |= FONTD;
		break;
		case 'O':
		    if( (tfaces[i].boldital = GetFont(s)) != NULL )
			tfaces[i].fido |= FONTO;
		break;
		default:
		; /* de nada */
	    }
	}

	s = nextline( bufend, s );
    } while( s < bufend );

    /*
     * Find out how many faces are actually available
     */
    nfaces = 0;
    for( i = 0; i < tnfaces; i++ )
	if( tfaces[i].fido )
	    ++nfaces;
    if( !nfaces )
    {
	free( tfaces );
	return AERROR;
    }

    faces = (Face *)xcalloc( (size_t)nfaces, sizeof(Face) );
    if( !faces )
    {
	free( tfaces );
	return AENSMEM;
    }

    /*
     * Fill in the faces array with available faces.
     */
    j = 0;
    for( i = 0; i < tnfaces; i++ )
	if( tfaces[i].fido )
	{
	    faces[j] = tfaces[i];
	    ++j;
	}

    free( tfaces );
    xqsort( faces, (size_t)nfaces, sizeof(Face), (CMPFUNC)cmpFace );
    return AEOK;
}

/* ----------------------------------------------------------------------
 * Create an array of all FSM fonts using a vqt_name cycle.
 * Return AENSMEM if there isn't enough memory for the font array,
 * or AERROR if there are no FSM fonts, otherwise return AEOK.
 */
MLOCAL long	FindFonts( VOID )
{
    short	i, f, pts, sysfonts, fsmerror;
    int		x;

    fonts = (Font *)xcalloc( (size_t)nfonts, sizeof(Font) );
    if( !fonts )
	return AENSMEM;

    sysfonts = ws[10]+1; /* number of fonts on screen workstation */
    f = 0;
#define VST_ERROR_CHECK 1
#if VST_ERROR_CHECK
	vst_error( wsid, 0, &fsmerror );
#endif
    for( i = sysfonts; i < sysfonts+nfonts; i++ )
    {
	fonts[f].id = vqt_name( wsid, i, fonts[f].name );
	vst_font( wsid, fonts[f].id );
	/*
	 * Keep largest point size to check against later.
	 * FSM fonts can not use bold, skew, or outline effects
	 * at point sizes larger than the biggest in the EXTEND.SYS.
	 */
	pts = vst_point( wsid, 999, &x, &x, &x, &x );
	maxpts = (pts > maxpts) ? pts : maxpts;
	if( vst_rotation(wsid, 100) == 100 )
	{
	    vst_rotation( wsid, 0 );
#if VST_ERROR_CHECK
	    /*
	     * Check to see if this is an all-uppercase font.
	     */
	    vst_point( wsid, SAMPLE_POINTSIZE, &x, &x, &x, &x );
		fsmerror = 0;
		vqt_advance( wsid, 'a', &x, &x, &x, &x );
	    fonts[f].nolower = (fsmerror) ? YES : NO;
		fsmerror = 0;
#endif
	    ++f;
	}
	/*
	 * If this program supported bitmap fonts, the loop checking
	 * for available bitmap font sizes would go here.
	 */
    }
#if VST_ERROR_CHECK
	vst_error( wsid, 1, NULL );
#endif

    if( !f )
    {
	free( fonts );
	fonts = NULL;
	return AERROR;
    }
    nfsmfonts = f;
    xqsort( fonts, (size_t)f, sizeof(Font), (CMPFUNC)cmpFont );
    return AEOK;
}

/* ----------------------------------------------------------------------
 * Read FACES.INF and create array of Faces.
 * Also find out what fonts are available.
 * Return 0 if all's well, 2 if user cancels at any point,
 * or negative GEMERROR number.
 */
GLOBAL long	GetFaces( VOID )
{
    long	ret;
    char	*title, fname[128];
    char	*fbuf = NULL;
    long	flen;

    ret = FindFonts();
    if( ret == AENSMEM )
    {
	GetAlert( MEMLOAD, &alertp );
	GetString( FONTSTR, &title );
	sprintf( alert, alertp, title );
	form_alert( 1, alert );
	return ret;
    }
    else if( ret == AERROR )
    {
	GetAlert( NOFONTS, &alertp );
	form_alert( 1, alertp );
	return ret;
    }

    strcpy( fname, "FACES.INF" );
    if( !shel_find( fname ) )
    {
	GetAlert( CANTFIND, &alertp );
	sprintf( alert, alertp, "FACES.INF" );
	ret = form_alert( 1, alert );
	if( ret == 2 )
	    return 2;

	*fname = 0;
	GetString( LOCFACES, &title );
	ret = FselName( FALSE, fname, "*.INF", title );
	if( ret == FAILURE )
	    return 2;
    }

    do {
	ret = readfile( fname, (VOIDP *)&fbuf, &flen );
	if( ret == AENSMEM )
	{
	    GetAlert( MEMLOAD, &alertp );
	    sprintf( alert, alertp, fname );
	    form_alert( 1, alert );
	    return ret;
	}
	else if( ret )
	{
	    GetAlert( ERRLOAD, &alertp );
	    sprintf( alert, alertp, fname );
    	    ret = form_alert( 1, alert );
	    if( ret == 2 )
		return 2;
	}
    } while( ret );

    ret = MakeFaces( fbuf, flen );
    if( ret == AERROR )
    {
	GetAlert( NOFONTS, &alertp );
	form_alert( 1, alertp );
    }

    Mfree( fbuf );
    return ret;
}
