/* menukeys.c - Menu/keyboard handling
 * FontForm, Copyright (c) 1990, Atari Corporation
 * ======================================================================
 * 901217 kbad	Final sweep.
 * 901206 kbad	Started
 */

#include <ctype.h>

#if LATTICE
#include <string.h>
#include <osbind.h>
#include <vdi.h>
#include <aes.h>
#endif

#if __TURBOC__
#include <string.h>
#include <tos.h>
#include <vdi.h>
#include <aes.h>
#endif

#if MWC
#include <osbind.h>
#define Kbshift(a) Getshift(a)
#include <vdibind.h>
#include <aesbind.h>
#include <gemdefs.h>
#include <obdefs.h>
#endif

#if __GNUC__
#include <string.h>
#include <osbind.h>
#include <vdibind.h>
#include <aesbind.h>
#include <gemdefs.h>
#include <obdefs.h>
#endif

#include "portab.h"
#include "aesutil.h"
#include "fontform.h"
#include "faces.h"
#include "styles.h"
#include "printout.h"
#include "menukeys.h"

/* ----------------------------------------------------------------------
 * External stuff, from FONTFORM.C
 */
EXTERN GRECT	rwork;
EXTERN WORD	wind;
EXTERN OBJECT	*menu;
EXTERN char	*version;

EXTERN VOID	Redraw __PROTO(( GRECT *clip ));

/* ----------------------------------------------------------------------
 * Local definitions
 */

/*
 * Keyboard shift state bits
 */
#define KsCAPS		0x10
#define KsALT		0x08
#define KsCONTROL	0x04
#define KsSHIFT		0x03
#define KsLSHIFT	0x02
#define KsRSHIFT	0x01

/*
 * Mapped key state bits, placed in the upper byte of the word
 * returned by MapKey.
 */
#define KbSCAN		0x8000
#define KbNUM		0x4000
#define KbALT		0x0800
#define KbCONTROL	0x0400
#define KbSHIFT		0x0300
#define KbLSHIFT	0x0200
#define KbRSHIFT	0x0100

/*
 * Key map (scan codes) for null ASCII values.
 */
/* ISO key (should only appear when using a non-US keyboard w/ US TOS.) */
#define KbISO		0x37
/* Function keys */
#define KbF1		0x3b
#define KbF2		0x3c
#define KbF3		0x3d
#define KbF4		0x3e
#define KbF5		0x3f
#define KbF6		0x40
#define KbF7		0x41
#define KbF8		0x42
#define KbF9		0x43
#define KbF10		0x44
/* Shift-function keys */
#define KbF11		0x54
#define KbF12		0x55
#define KbF13		0x56
#define KbF14		0x57
#define KbF15		0x58
#define KbF16		0x59
#define KbF17		0x5a
#define KbF18		0x5b
#define KbF19		0x5c
#define KbF20		0x5d
/* Cursor cluster */
#define KbUNDO		0x61
#define KbHELP		0x62
#define KbINSERT	0x52
#define KbHOME		0x47
#define KbUP		0x48
#define KbDOWN		0x50
#define KbLEFT		0x4b
#define KbRIGHT 	0x4d
/* Alt-numeric keys */
#define KbAlt1		0x78
#define KbAlt2		0x79
#define KbAlt3		0x7a
#define KbAlt4		0x7b
#define KbAlt5		0x7c
#define KbAlt6		0x7d
#define KbAlt7		0x7e
#define KbAlt8		0x7f
#define KbAlt9		0x80
#define KbAlt0		0x81


typedef struct
{
	char *unshift;
	char *shift;
	char *caps;
} KEYTABLE;

MLOCAL KEYTABLE	*kt;

/* ----------------------------------------------------------------------
 * Map a VDI key (returned by evnt_keybd()) to a word encoded character:
 *
 * hi byte  low byte
 * -------- --------  ACLR are keyboard shift state bits.
 * SxxxACLR CHARCODE = ASCII (S = 0) or Scan (S = 1) code.
 *
 * This function solves the sticky problem of international keyboard
 * equivalents by mapping the VDI keys, which change in different
 * countries, to known ASCII/scan codes which are the same in all countries.
 */
GLOBAL WORD	MapKey( key )
WORD		key;
{
    WORD    keystate, scancode, ret;

    /*
     * Get the keyboard tables, if we haven't yet.
     */
    if( !kt )
	kt = (KEYTABLE *)Keytbl( (VOIDP)-1L, (VOIDP)-1L, (VOIDP)-1L );

    /*
     * Extract the scancode from the input keycode, and
     * get the keyboard shift key states.
     *
     * Can't use the keyboard state bits from evnt_multi or graf_mkstate,
     * because we need the capslock state, so use Kbshift instead.
     * If I really wanted to be sleazy, I'd sniff the OS header for the
     * kbshift system variable location.  Not this time.
     */
    scancode = (key >> 8) & 0x00ff;
    keystate = (WORD)Kbshift( -1 );

    /*
     * Look up the ASCII value of the key in the appropriate table.
     */

    /*
     * Adjust Alt-numeric keys
     */
    if( (keystate & KsALT) && scancode >= 0x78 && scancode <= 0x83 )
	scancode -= 0x76;

    if( keystate & KsCAPS )
    {
	ret = kt->caps[scancode];
    }
    else if( keystate & KsSHIFT )
    {
	/*
	 * Shift function keys have adjusted scan codes, so make sure
	 * we're looking at the right spot in the table.
	 */
	if( scancode >= KbF11 && scancode <= KbF20 )
	    ret = kt->shift[scancode-0x19];
	else
	    ret = kt->shift[scancode];
    }
    else
    {
	ret = kt->unshift[scancode];
    }

    /*
     * Return scan codes for 0 ASCII values, and
     * flag numeric keypad keys.
     */
    if( ret == 0 )
	ret = scancode | KbSCAN;

    else if( scancode == 0x4a
	 ||  scancode == 0x4e
	 ||  (scancode >= 0x63 && scancode <= 0x72) )
	ret |= KbNUM;

    return ret | (keystate << 8);
}

/* ----------------------------------------------------------------------
 * Enable/Disable/Check menu items as appropriate
 */
GLOBAL VOID	EnableMenus( VOID )
{
    UWORD	fx;

    menu_ienable( menu, EDUNDO, NO );
    menu_ienable( menu, EDCUT, NO );
    menu_ienable( menu, EDCOPY, NO );
    menu_ienable( menu, EDPASTE, NO );
    menu_ienable( menu, EDCLEAR, NO );

    menu_icheck( menu, TEBOLD, (curstyle.fx & FxBOLD) == FxBOLD );
    menu_icheck( menu, TELIGHT, (curstyle.fx & FxLITE) == FxLITE );
    menu_icheck( menu, TEITALIC, (curstyle.fx & FxSKEW) == FxSKEW );
    menu_icheck( menu, TEOUTLIN, (curstyle.fx & FxOUTLINE) == FxOUTLINE );
    menu_icheck( menu, TEUNDER, (curstyle.fx & FxUL) == FxUL );
    menu_icheck( menu, TESHADOW, (curstyle.fx & FxSHADOW) == FxSHADOW );

    fx = curstyle.fx;
    curstyle.fx ^= FxOUTLINE;
    menu_ienable( menu, TEOUTLIN, ValidFx(&curstyle) == (fx ^ FxOUTLINE) );
    curstyle.fx ^= (FxOUTLINE|FxBOLD);
    menu_ienable( menu, TEBOLD, ValidFx(&curstyle) == (fx ^ FxBOLD) );
    curstyle.fx ^= FxBS;
    menu_ienable( menu, TEITALIC, ValidFx(&curstyle) == (fx ^ FxSKEW) );
    curstyle.fx ^= FxSKEW;

    /*
     * Implementing point size changes in the menu is left
     * as an exercise for the reader.
     */
    menu_ienable( menu, TEPTSIZE, NO );
    menu_ienable( menu, PTSIZE06, NO );
    menu_ienable( menu, PTSIZE09, NO );
    menu_ienable( menu, PTSIZE10, NO );
    menu_ienable( menu, PTSIZE11, NO );
    menu_ienable( menu, PTSIZE12, NO );
    menu_ienable( menu, PTSIZE14, NO );
    menu_ienable( menu, PTSIZE18, NO );
    menu_ienable( menu, PTSIZE24, NO );
    menu_ienable( menu, PTSIZE30, NO );
    menu_ienable( menu, PTSIZE36, NO );
    menu_ienable( menu, PTSIZE42, NO );
    menu_ienable( menu, PTSIZE48, NO );
    menu_ienable( menu, PTSIZE54, NO );
    menu_ienable( menu, PTSIZE60, NO );
    menu_ienable( menu, PTSIZE72, NO );
}

/* ----------------------------------------------------------------------
 * Handle a key event by mapping it to a menu equivalent.
 */
GLOBAL VOID	DoKey( mapkey, event )
WORD		mapkey;
UWORD		*event;
{
    OBJECT	*tree = menu;
    WORD	msg[8];
EXTERN WORD	apid;

    MenTitle(msg) = MenItem(msg) = NIL;

    if( !(mapkey & KbSCAN) && islower(mapkey & 0x00ff) )
	mapkey = (mapkey & 0xff00) | toupper(mapkey & 0x00ff);

    switch( mapkey )
    {
	/*
	 * Desk menu
	 */
	case KbSCAN|KbHELP:
	    MenTitle(msg) = DESKMENU;
	    MenItem(msg) = DEABOUT;
	break;

	/*
	 * File menu
	 */
	case KbCONTROL|'P':
	    MenTitle(msg) = FILEMENU;
	    MenItem(msg) = FIPRINT;
	break;
	case KbCONTROL|'Q':
	    MenTitle(msg) = FILEMENU;
	    MenItem(msg) = FIQUIT;
	break;

	/*
	 * Edit menu
	 */
	case KbSCAN|KbUNDO:
	    MenTitle(msg) = EDITMENU;
	    MenItem(msg) = EDUNDO;
	break;
	case KbCONTROL|'X':
	    MenTitle(msg) = EDITMENU;
	    MenItem(msg) = EDCUT;
	break;
	case KbCONTROL|'C':
	    MenTitle(msg) = EDITMENU;
	    MenItem(msg) = EDCOPY;
	break;
	case KbCONTROL|'V':
	    MenTitle(msg) = EDITMENU;
	    MenItem(msg) = EDPASTE;
	break;
	case '\177':
	    MenTitle(msg) = EDITMENU;
	    MenItem(msg) = EDCLEAR;
	break;

	/*
	 * Text menu
	 */
	case KbCONTROL|'F':
	    MenTitle(msg) = TEXTMENU;
	    MenItem(msg) = TEFONT;
	break;

	case KbALT|'B':
	    MenTitle(msg) = TEXTMENU;
	    MenItem(msg) = TEBOLD;
	break;
	case KbALT|'L':
	    MenTitle(msg) = TEXTMENU;
	    MenItem(msg) = TELIGHT;
	break;
	case KbALT|'I':
	    MenTitle(msg) = TEXTMENU;
	    MenItem(msg) = TEITALIC;
	break;
	case KbALT|'O':
	    MenTitle(msg) = TEXTMENU;
	    MenItem(msg) = TEOUTLIN;
	break;
	case KbALT|'U':
	    MenTitle(msg) = TEXTMENU;
	    MenItem(msg) = TEUNDER;
	break;
	case KbALT|'S':
	    MenTitle(msg) = TEXTMENU;
	    MenItem(msg) = TESHADOW;
	break;

	case KbALT|'.':
	    MenTitle(msg) = TEXTMENU;
	    MenItem(msg) = TEPTSIZE;
	break;
    }


    if( MenItem(msg) != NIL && !IsDisabled(MenItem(msg)))
    {
	menu_tnormal( tree, MenTitle(msg), 0 );
	EvntTimer( 200L );
	MsgId(msg) = MN_SELECTED;
	MsgSender(msg) = apid;
	MsgExtra(msg) = 0;
	DoMenu( msg, event );
    }
}

/* ----------------------------------------------------------------------
 * Handle a menu event.
 */
GLOBAL VOID	DoMenu( msg, event )
WORD		*msg;
UWORD		*event;
{
    OBJECT	*tree;
    Style	tmpstyle;
    GRECT	r;
    WORD	fxmask = 0, j, ignore, ignoremsg[8];

    switch( MenItem(msg) )
    {
	case DEABOUT:
	    {
		char *c, copyright[40];

		ObjcExtent( menu, DESKMENU, &r );
		GetTree( ABOUTFM, &tree );
		TedText(VERSION) = version;
		c = ObString(COPYRITE);
		strcpy( copyright, c );
		*copyright = '\275';
		ObString(COPYRITE) = copyright;
		FormInit( tree, &r, YES );
		DeselectObj( form_do( tree, 0 ) & 0x7fff );
		ObjcExtent( tree, ABTOK, &r );
		FormExit( tree, &r );
		ObString(COPYRITE) = c;
	    }
	break;

	case FIPRINT:
	    Print();
	break;
	case FIQUIT:
	    *event = 0;
	break;

	case EDUNDO:
	break;
	case EDCUT:
	break;
	case EDCOPY:
	break;
	case EDPASTE:
	break;
	case EDCLEAR:
	break;

	case TEFONT:
	    tmpstyle = curstyle;
	    StyleForm();
	    if( cmpStyle(&tmpstyle, &curstyle) != 0 )
	    {
		/*
		 * Eat up the redraw event caused by the FormExit in
		 * StyleForm, since we're redrawing the whole window
		 * anyways.
		 */
		do {
		    ignore = evnt_multi( MU_MESAG|MU_TIMER,
					0, 0, 0,
					0, 0, 0, 0, 0,
					0, 0, 0, 0, 0,
					ignoremsg, 0, 0,
					&j, &j, &j, &j, &j, &j );
		} while( ignore != MU_TIMER );

		EnableMenus();
		wind_get( wind, WF_FIRSTXYWH,
			  &r.g_x, &r.g_y, &r.g_w, &r.g_h );
		while( !RectEmpty(&r) )
		{
		    if( RectIntersect(&rwork, &r) )
			Redraw( &r );
		    wind_get( wind, WF_NEXTXYWH,
			      &r.g_x, &r.g_y, &r.g_w, &r.g_h );
		}
	    }
	break;

	case TEBOLD:
	    fxmask = FxBOLD;
	break;
	case TELIGHT:
	    fxmask = FxLITE;
	break;
	case TEITALIC:
	    fxmask = FxSKEW;
	break;
	case TEOUTLIN:
	    fxmask = FxOUTLINE;
	break;
	case TEUNDER:
	    fxmask = FxUL;
	break;
	case TESHADOW:
	    fxmask = FxSHADOW;
	break;

	/*
	 * Implementing point size changes in the menu is left
	 * as an exercise for the reader.
	 */
	case TEPTSIZE:
	break;

	case PTSIZE06:
	case PTSIZE09:
	case PTSIZE10:
	case PTSIZE11:
	case PTSIZE12:
	case PTSIZE14:
	case PTSIZE18:
	case PTSIZE24:
	case PTSIZE30:
	case PTSIZE36:
	case PTSIZE42:
	case PTSIZE48:
	case PTSIZE54:
	case PTSIZE60:
	case PTSIZE72:
	break;

    }
    if( fxmask )
    {
	curstyle.fx ^= fxmask;
	EnableMenus();
	wind_get( wind, WF_FIRSTXYWH,
		  &r.g_x, &r.g_y, &r.g_w, &r.g_h );
	while( !RectEmpty(&r) )
	{
	    if( RectIntersect(&rwork, &r) )
		Redraw( &r );
	    wind_get( wind, WF_NEXTXYWH,
		      &r.g_x, &r.g_y, &r.g_w, &r.g_h );
	}
    }

    menu_tnormal( menu, MenTitle(msg), 1 );
}
