/* sliders.c - slider handlers
 * FontForm, Copyright (C) 1990, Atari Corporation
 * ======================================================================
 * 901202 kbad	Cleanup
 * 901120 kbad	Source code cleanup
 * 901112 kbad	started, based on c.gee's slider code, which was based
 *		on my original MACCEL slider code.
 */

#if LATTICE
#include <aes.h>
#endif

#if __TURBOC__
#include <aes.h>
#endif

#if MWC
#include <aesbind.h>
#include <gemdefs.h>
#include <obdefs.h>
#endif

#if __GNUC__
#include <aesbind.h>
#include <gemdefs.h>
#include <obdefs.h>
#endif

#include "portab.h"
#include "aesutil.h"
#include "sliders.h"


extern long dcrates[]; /* table of double-click delays, in FONTFORM.C */

MLOCAL WORD	rangecheck __PROTO(( WORD value, WORD minval, WORD maxval ));



/* SlSize() - size a slider's elevator
 * ----------------------------------------------------------------------
 * IN:
 *	OBJECT	    *tree	tree containing slider
 *	WORD	    slid	slider object (contains elevator)
 *	WORD	    elev	elevator object (child of slider)
 *	WORD	    minsize	minimum pixel extent of elevator
 *	WORD	    range	total number of items represented by slider
 *	WORD	    visrange	visible number of items represented by elev.
 *
 * OUT:
 *	For horizontal sliders (width > height), elevator width scaled.
 *	For vertical sliders (height > width), elevator height scaled.
 */
GLOBAL VOID	SlSize( tree, slid, elev, minsize, range, visrange )
OBJECT		*tree;
WORD		slid, elev, minsize, range, visrange;
{
    if( ObH(slid) > ObW(slid) )
	/* Vertical */
	ObH(elev) = min( max(minsize, mul_div(ObH(slid), visrange, range)),
			 ObH(slid) );
    else
	/* Horizontal */
	ObW(elev) = min( max(minsize, mul_div(ObW(slid), visrange, range)),
			 ObW(slid) );
}


/* SlXY() - set the X or Y position of a slider's elevator
 * ----------------------------------------------------------------------
 * IN:
 *	OBJECT	    *tree	tree containing slider
 *	WORD	    slid	slider object (contains elevator)
 *	WORD	    elev	elevator object (child of slider)
 *	VOIDFUNC    draw	draw function (or NULLFUNC)
 *	WORD	    minval	left/top value
 *	WORD	    maxval	right/bottom value
 *	WORD	    value	current value
 *
 * OUT:
 *	For horizontal sliders (width > height), elev X position updated.
 *	For vertical sliders (height > width), elev Y position updated.
 */
GLOBAL VOID	SlXY( tree, slid, elev, draw, minval, maxval, value )
OBJECT		*tree;
WORD		slid, elev;
VOIDFUNC	draw;
WORD		minval, maxval, value;
{
    if( ObH(slid) > ObW(slid) )
    {
	/*
	 * Vertical
	 */
	if( value == minval )
	    ObY(elev) = ObH(slid) - ObH(elev);
	else
	    ObY(elev) = mul_div( ObH(slid)-ObH(elev)+1, maxval-value,
				 maxval-minval );
    }
    else
    {
	/*
	 * Horizontal
	 */
	if( value == maxval )
	    ObX(elev) = ObW(slid) - ObW(elev);
	else
	    ObX(elev) = mul_div( ObW(slid)-ObW(elev)+1, value-minval,
				 maxval-minval );
    }
    if( draw )
	(*draw)();
}


/* ----------------------------------------------------------------------
 * If `value' is out of range `minval'..`maxval', return the nearest
 * limit, otherwise return `value'.  Minval is not necessarily < maxval.
 */
MLOCAL WORD	rangecheck( value, minval, maxval )
WORD		value, minval, maxval;
{
    if( maxval > minval )
    {
	if( value < minval ) return minval;
	else if( value > maxval ) return maxval;
    }
    else
    {
	if( value > minval ) return minval;
	else if( value < maxval ) return maxval;
    }
    return value;
}

/* ----------------------------------------------------------------------
 * Draw a slider in its new position, only redrawing as much as needed,
 * to minimize flicker.
 * NOTE: The contents of 'old' may be modified by RectIntersect.
 */
GLOBAL VOID	SlDraw( tree, slid, elev, old, new )
OBJECT		*tree;
WORD		slid, elev;
GRECT		*old, *new;
{
    GRECT   drawrect;
    BOOLEAN vert;

    if( RectEqual(new, old) )
    {
	ObjcDraw( tree, elev, ObMAXDEPTH, new );
    }
    else
    {
	vert = (old->g_x == new->g_x);
	drawrect = *old;
	if( RectIntersect(new, old) )
	{
	    if( vert )
	    {
		/* Vertical */
		drawrect.g_h -= old->g_h - 1;
		if( old->g_y > new->g_y )
		    /* up */
		    drawrect.g_y = new->g_y + new->g_h - 1;
	    }
	    else
	    {
		/* Horizontal */
		drawrect.g_w -= old->g_w - 1;
		if( old->g_x > new->g_x )
		    /* left */
		    drawrect.g_x = new->g_x + new->g_w - 1;
	    }
	}
	/* Undraw old, then draw new */
	ObjcDraw( tree, slid, ObMAXDEPTH, &drawrect );
	ObjcDraw( tree, elev, ObMAXDEPTH, new );
    }
}


/*
 * SlDrag() - drag an elevator around in a slider.
 * ----------------------------------------------------------------------
 *
 * IN:
 *	OBJECT	    *tree	tree containing slider
 *	WORD	    slid	slider object (contains elevator)
 *	WORD	    elev	elevator object (child of slider)
 *	VOIDFUNC    draw	draw function (or NULLFUNC)
 *	WORD	    minval	left/top value
 *	WORD	    maxval	right/bottom value
 *	WORD	    *pval	points to in-range value
 *
 * OUT:
 *	*pval set to new value, elevator position updated.
 */
GLOBAL VOID	SlDrag( tree, slid, elev, draw, minval, maxval, pval )
OBJECT		*tree;
WORD		slid, elev;
VOIDFUNC	draw;
WORD		minval, maxval, *pval;
{
    WORD   newval, xoffset, yoffset, elevx, elevy;
    MoInfo  m;
    GRECT   slidr, elevr, oldr;
    BOOLEAN vert, elev_isted;
    WORD   savecolor;

    slidr = ObRect(slid);
    objc_offset( tree, slid, &slidr.g_x, &slidr.g_y );
    vert = (slidr.g_h > slidr.g_w);
    ObjcExtent( tree, elev, &elevr );

    elev_isted = IsTed(elev);
    if( elev_isted )
    {
	savecolor = (WORD)TedCBorder(elev);
	TedCBorder(elev) = ObWHITE;
    }
    else
    {
	savecolor = (WORD)ObCBorder(elev);
	ObCBorder(elev) = ObWHITE;
    }
    ObjcDraw( tree, elev, ObMAXDEPTH, &elevr );

    GrafMKState( &m );
    objc_offset( tree, elev, &xoffset, &yoffset );
    xoffset = m.x - xoffset;
    yoffset = m.y - yoffset;

    while( m.buttons )
    {
	elevx = m.x - xoffset;
	elevy = m.y - yoffset;
	if( vert )
	{
	    if( elevy <= slidr.g_y )
		newval = maxval;
	    else if( elevy >= slidr.g_y + slidr.g_h - ObH(elev) )
		newval = minval;
	    else
		newval = rangecheck( maxval +
				    mul_div(minval-maxval, elevy-slidr.g_y+1,
					   ObH(slid)-ObH(elev)+1),
				    minval, maxval );
	}
	else
	{
	    if( elevx <= slidr.g_x )
		newval = minval;
	    else if( elevx >= slidr.g_x + slidr.g_w - ObW(elev) )
		newval = maxval;
	    else
		newval = rangecheck( minval +
				    mul_div(maxval-minval, elevx-slidr.g_x+1,
					   ObW(slid)-ObW(elev)+1),
				    minval, maxval );
	}
	if( newval != *pval )
	{
	    *pval = newval;
	    SlXY( tree, slid, elev, draw, minval, maxval, newval );
	    oldr = elevr;
	    ObjcExtent( tree, elev, &elevr );
	    SlDraw( tree, slid, elev, &oldr, &elevr );
	}
	GrafMKState( &m );
    }

    if( elev_isted )
	TedCBorder(elev) = savecolor;
    else
	ObCBorder(elev) = savecolor;
    ObjcDraw( tree, elev, ObMAXDEPTH, &elevr );
}

/*
 * SlArrow() - increment/decrement elevator position while mouse
 *		button stays down.  This function can also be used
 *		for paging, with a "page" sized value increment.
 * ----------------------------------------------------------------------
 * IN:
 *	OBJECT	    *tree	tree containing slider
 *	WORD	    slid	slider object (contains elevator)
 *	WORD	    elev	elevator object (child of slider)
 *	WORD	    obj		object mouse went down on.
 *				If obj == slid, this turns into a drag when
 *				the elevator falls under the mouse.
 *	VOIDFUNC    draw	draw function (or NULLFUNC)
 *	WORD	    inc		amount to add to value
 *	WORD	    minval	left/top value
 *	WORD	    maxval	right/bottom value
 *	WORD	    *pval	points to in-range value.
 *
 * OUT:
 *	*pval set to new value, elevator position updated.
 */
GLOBAL VOID	SlArrow( tree, slid, elev, obj, draw,
			 inc, minval, maxval, pval )
OBJECT		*tree;
WORD		slid, elev, obj;
VOIDFUNC	draw;
WORD		inc, minval, maxval, *pval;
{
    MoInfo  m;
    WORD   newval, watchelev = 0;
    GRECT   elevr, oldr;
    long    delay, mindelay, quickening;

    ObjcExtent( tree, elev, &elevr );
    if( obj != ObNIL )
    {
	if( obj == slid )
	    watchelev = -1;
	else
	    ObjcSelect(obj);
    }

    delay = dcrates[evnt_dclick(0,0)];
    mindelay = delay/2;
    quickening = (delay - mindelay)/8;

    do {
	newval = rangecheck( *pval + inc, minval, maxval );
	if( newval != *pval )
	{
	    *pval = newval;
	    SlXY( tree, slid, elev, draw, minval, maxval, newval );
	    oldr = elevr;
	    ObjcExtent( tree, elev, &elevr );
	    SlDraw( tree, slid, elev, &oldr, &elevr );
	}
	EvntTimer( delay );
	if( delay > mindelay )
	    delay -= quickening;
	GrafMKState( &m );
	if( watchelev && XYinRect(m.x, m.y, &elevr) )
	    watchelev = 1;
    } while( m.buttons && watchelev < 1 );

    if( obj != ObNIL && obj != slid )
	ObjcDeselect( obj );

    if( m.buttons && watchelev == 1 )
	SlDrag( tree, slid, elev, draw, minval, maxval, pval );
}
