/* styles.c - style form handler
 * FontForm, Copyright (c) 1990, Atari Corporation
 * ======================================================================
 * 901227 kbad	Last minute cleanup
 * 901219 kbad	ValidFx() added. Final sweep
 * 901202 kbad	Cleanup
 * 901127 kbad	From PageOMat
 *
 * This module manages user selection of a given face and style.
 * It also features a utility routine to convert a face index and effects
 * word into the actual FSM font to use for that face, and the actual effects
 * that should be used with that FSM font.
 */

#include <stdio.h>

#if LATTICE
#include <stdlib.h>
#include <string.h>
#include <vdi.h>
#include <aes.h>
#endif

#if __TURBOC__
#include <stdlib.h>
#include <string.h>
#include <vdi.h>
#include <aes.h>
#endif

#if MWC
#include <vdibind.h>
#include <aesbind.h>
#include <gemdefs.h>
#include <obdefs.h>
#define vqt_fontinfo vqt_font_info
#endif

#if __GNUC__
#include <stdlib.h>
#include <string.h>
#include <vdibind.h>
#include <aesbind.h>
#include <gemdefs.h>
#include <obdefs.h>
#endif

#include "portab.h"
#include "aesutil.h"
#include "fsmbind.h"
#include "fontform.h"
#include "faces.h"
#include "sliders.h"
#include "userdef.h"
#include "styles.h"


/* Local macros
 * ----------------------------------------------------------------------
 */

#define MAXSKEW 85
#define NFACEOBS 7

/*
 * Wait for all the mouse buttons to come up.
 */
#define wait_bup() { do { GrafMKState(&m); } while( m.buttons ); }


/* External variables, from FONTFORM.C
 * ----------------------------------------------------------------------
 */

extern char	*alertp, alert[];
extern WORD	wsid, ws[];
extern WORD	hbox;
extern long	dcrates[];


/* Global variables
 * ----------------------------------------------------------------------
 */

Style	curstyle;
/* DEFAULT INITIALIZATION: styles.pts defaults to 0, indicating that
 *			   no style is currently selected.
 */



/* Module locals
 * ----------------------------------------------------------------------
 * Many of these variables are module locals so that this form handler
 * can be called repeatedly from an application without requiring total
 * reinitialization each time it is called.
 */
MLOCAL OBJECT	*tree;
MLOCAL MoInfo	m;
MLOCAL GRECT	rtree,	    /* Full extent of the style form */
		mid,	    /* Box centered in the style form */
		rpts,	    /* Extent of the text part of the point size box */
		rset,	    /* Extent of the text part of the set size box */
		rskewbig,   /* Skew slider box rectangle */
		rskew;	    /* Centered skew box rectangle */

MLOCAL Style	startstyle, tmpstyle, undostyle;
MLOCAL WORD	lastface0, face0, maxface0, maxfontobj;
MLOCAL USERBLK	fxobjub[6], sample;
MLOCAL WORD	fxobj[6] = { STYBOLDT, STYLITET, STYITALT,
			     STYUNDRT, STYOUTLT, STYSHADT };
MLOCAL WORD	fxbut[6] = { STYBOLD, STYLITE, STYITAL,
			     STYUNDR, STYOUTL, STYSHAD };

/*
 * The following 2 tables are used to determine what effects are
 * required based on what fonts are available.  q.v. ValidFx() below.
 */

MLOCAL UWORD fxtab[16][4] =
{
/* requested fx:  0,	  FxBOLD, FxSKEW, FxBS */
/* avail*/	{ 0,	  0,	  0,	  0 },
/* F    */	{ 0,	  FxBOLD, FxSKEW, FxBS },
/*  I   */	{ FxSKEW, FxBS,   FxSKEW, FxBS },
/* FI   */	{ 0,	  FxBOLD, FxSKEW, FxBS },
/*   D  */	{ FxBOLD, FxBOLD, FxBS,	  FxBS },
/* F D  */	{ 0,	  FxBOLD, FxSKEW, FxBS },
/*  ID  */	{ FxBOLD, FxBOLD, FxSKEW, FxBS },
/* FID  */	{ 0,	  FxBOLD, FxSKEW, FxBS },
/*    O */	{ FxBS,	  FxBS,	  FxBS,	  FxBS },
/* F  O */	{ 0,	  FxBOLD, FxSKEW, FxBS },
/*  I O */	{ FxSKEW, FxBS,	  FxSKEW, FxBS },
/* FI O */	{ 0,	  FxBOLD, FxSKEW, FxBS },
/*   DO */	{ FxBOLD, FxBOLD, FxBS,   FxBS },
/* F DO */	{ 0,	  FxBOLD, FxSKEW, FxBS },
/*  IDO */	{ FxBOLD, FxBOLD, FxSKEW, FxBS },
/* FIDO */	{ 0,	  FxBOLD, FxSKEW, FxBS }
};

MLOCAL UWORD bigfxtab[16][4] =
{
/* requested fx:  0,	  FxBOLD, FxSKEW, FxBS */
/* avail*/	{ 0,	  0,	  0,	  0 },
/* F    */	{ 0,	  0,	  0,	  0 },
/*  I   */	{ FxSKEW, FxSKEW, FxSKEW, FxSKEW },
/* FI   */	{ 0,	  0,	  FxSKEW, FxSKEW },
/*   D  */	{ FxBOLD, FxBOLD, FxBOLD, FxBOLD },
/* F D  */	{ 0,	  FxBOLD, 0,	  FxBOLD },
/*  ID  */	{ FxBOLD, FxBOLD, FxSKEW, FxSKEW },
/* FID  */	{ 0,	  FxBOLD, FxSKEW, FxSKEW },
/*    O */	{ FxBS,	  FxBS,	  FxBS,	  FxBS },
/* F  O */	{ 0,	  FxBS,	  FxBS,	  FxBS },
/*  I O */	{ FxSKEW, FxBS,	  FxSKEW, FxBS },
/* FI O */	{ 0,	  FxBS,   FxSKEW, FxBS },
/*   DO */	{ FxBOLD, FxBOLD, FxBS,   FxBS },
/* F DO */	{ 0,	  FxBOLD, FxBOLD, FxBS },
/*  IDO */	{ FxBOLD, FxBOLD, FxSKEW, FxBS },
/* FIDO */	{ 0,	  FxBOLD, FxSKEW, FxBS }
};

/* Local functions
 * ----------------------------------------------------------------------
 */
MLOCAL int STDARGS DrawFxobj __PROTO(( PARMBLK *parm ));
MLOCAL int STDARGS PrintSample __PROTO(( PARMBLK *parm ));

MLOCAL BOOLEAN	FlagFsmFx __PROTO(( Style *ps, BOOLEAN draw ));
MLOCAL VOID	FixFx __PROTO(( WORD obj, UWORD fx, Style *ps ));
MLOCAL VOID	FixPts __PROTO(( WORD obj, GRECT *clip, WORD inc, WORD *val ));
MLOCAL VOID	ScrollPts __PROTO(( GRECT *rpts, GRECT *rset,
				    WORD inc, Style *ps ));
MLOCAL VOID	Set2Pts __PROTO(( WORD from, WORD to, GRECT *rfrom, GRECT *rto,
				  Style *ps ));
MLOCAL VOID	FixSkew ( VOID );
MLOCAL VOID	FixFace ( VOID );
MLOCAL VOID	InitStyleForm ( VOID );
MLOCAL int	cmpStyle __PROTO(( const Style *a, const Style *b ));


/* STYLES functions
 * ======================================================================
 */


/* ----------------------------------------------------------------------
 * Convert a face index and a VDI effects word into
 * a true effects word `fsmfx', and return a pointer to the font.
 * The `fsmfx' will have the bits turned off for whichever
 * effects are satisfied by an available FSM font style.
 */
GLOBAL Font *FsmFont( face, fx, fsmfx )
WORD	    face;
UWORD	    fx;
UWORD	    *fsmfx;
{
    Face    *pface = &faces[face];
    Font    *pfont = NULL;
    UWORD   truefx = fx;

    if( (fx & FxBS) == FxBS )
    {
	/*
	 * Bold, italic style;  satisfy both if possible.
	 * If not, try skew first, then bold.
	 */
	if( (pfont = pface->boldital) != NULL )
	    truefx &= ~FxBS;

	else if( (pfont = pface->ital) != NULL )
	    truefx &= ~FxSKEW;

	else if( (pfont = pface->bold) != NULL )
	    truefx &= ~FxBOLD;
    }

    else if( (fx & FxBOLD) && ((pfont = pface->bold) != NULL) )
	/*
	 * Bold style
	 */
	truefx &= ~FxBOLD;

    else if( (fx & FxSKEW) && ((pfont = pface->ital) != NULL) )
	/*
	 * Italic style
	 */
	truefx &= ~FxSKEW;

    if( !pfont )
    {
	/*
	 * Haven't yet found a font; return whatever's available.
	 */
	if( pface->fido & FONTF )
	    pfont = pface->font;
	else if( pface->fido & FONTI )
	    pfont = pface->ital;
	else if( pface->fido & FONTD )
	    pfont = pface->bold;
	else
	    pfont = pface->boldital;
    }

    *fsmfx = truefx;
    return pfont;
}


/* ----------------------------------------------------------------------
 * Return valid effects for style *ps.
 */
GLOBAL UWORD ValidFx( ps )
Style	    *ps;
{
    Face    *pface = &faces[ps->face];
    UWORD   validfx, bsfx, fxmask;

    validfx = ps->fx;

    bsfx = ((validfx & FxSKEW) == FxSKEW) * 2 +
	    ((validfx & FxBOLD) == FxBOLD);

    if( ps->pts > maxpts || ps->set > maxpts )
    {
	validfx &= ~FxOUTLINE;
	fxmask = bigfxtab[pface->fido][bsfx];
    }
    else
    {
	fxmask = fxtab[pface->fido][bsfx];
    }

    if( fxmask != (validfx & FxBS) )
    {
	validfx &= ~FxBS;
	validfx |= fxmask;
    }
    return validfx;
}


/* ----------------------------------------------------------------------
 * Set "FSM" flags for STYBOLD and STYITAL buttons, by testing the
 * requested effects against those which can be supplied by an available
 * FSM font style, and setting the button text accordingly.
 *
 * If the current point size of the current font is too big for the
 * effects buffer, adjust the effects accordingly.
 *
 */
MLOCAL BOOLEAN FlagFsmFx( ps, draw )
Style	    *ps;
BOOLEAN	    draw;
{
static char *nofsm = "", *isfsm = "FSM";
    UWORD   x, truefx, fx = ps->fx, boldstate, italstate, outlstate;
    WORD    face = ps->face;
    char    *text, *boldtext, *italtext;

    boldstate = ObState(STYBOLD);
    italstate = ObState(STYITAL);
    outlstate = ObState(STYOUTL);
    boldtext = TedText(STYBOLD);
    italtext = TedText(STYITAL);

    FsmFont( face, fx, &truefx );
    x = truefx;
    if( (fx & FxBS) == FxBS )
    {
	switch( x & FxBS )
	{
	    /*
	     * Neither effect was satisfied by an FSM font style.
	     */
	    case FxBS:
		TedText(STYBOLD) = TedText(STYITAL) = nofsm;
	    break;

	    /*
	     * One effect was satisfied.
	     */
	    case FxBOLD:
		TedText(STYBOLD) = nofsm;
		TedText(STYITAL) = isfsm;
	    break;
	    case FxSKEW:
		TedText(STYITAL) = nofsm;
		TedText(STYBOLD) = isfsm;
	    break;

	    /*
	     * Both effects were satisfied.
	     */
	    default:
		TedText(STYBOLD) = TedText(STYITAL) = isfsm;
	    break;
	}
    }
    else if( fx & FxBOLD )
    {
	TedText(STYBOLD) = (x & FxBOLD) ? nofsm : isfsm;
	FsmFont( face, FxBS, &x );
	TedText(STYITAL) = ((x & FxBS) == 0) ? isfsm : nofsm;
    }
    else if( fx & FxSKEW )
    {
	TedText(STYITAL) = (x & FxSKEW) ? nofsm : isfsm;
	FsmFont( face, FxBS, &x );
	TedText(STYBOLD) = ((x & FxBS) == 0) ? isfsm : nofsm;
    }
    else
    {
	FsmFont( face, FxBOLD, &x );
	TedText(STYBOLD) = (x & FxBOLD) ? nofsm : isfsm;
	FsmFont( face, FxSKEW, &x );
	TedText(STYITAL) = (x & FxSKEW) ? nofsm : isfsm;
    }

    x = ps->fx;
    ps->fx = x ^ FxBOLD;
    if( ValidFx(ps) == (x ^ FxBOLD) )
	MakeTouchexit(STYBOLD), MakeTouchexit(STYBOLDT);
    else
	NoTouchexit(STYBOLD), NoTouchexit(STYBOLDT);

    ps->fx = x ^ FxSKEW;
    if( ValidFx(ps) == (x ^ FxSKEW) )
	MakeTouchexit(STYITAL), MakeTouchexit(STYITALT);
    else
	NoTouchexit(STYITAL), NoTouchexit(STYITALT);

    ps->fx = x;
    x = ValidFx( ps );

    if( x & FxBOLD )	SelectObj(STYBOLD);
    else		DeselectObj(STYBOLD);
    if( x & FxSKEW )	SelectObj(STYITAL);
    else		DeselectObj(STYITAL);
    if( x & FxOUTLINE )	SelectObj(STYOUTL);
    else		DeselectObj(STYOUTL);

    if( draw )
    {
	/*
	 * Draw buttons if they changed
	 */
	if( ObState(STYBOLD) != boldstate || TedText(STYBOLD) != boldtext )
	    ObjcDraw( tree, STYBOLD, ObMAXDEPTH, NULL );
	if( ObState(STYITAL) != italstate || TedText(STYITAL) != italtext )
	    ObjcDraw( tree, STYITAL, ObMAXDEPTH, NULL );
	if( ObState(STYOUTL) != outlstate )
	    ObjcDraw( tree, STYOUTL, ObMAXDEPTH, NULL );

	if( ps->fx != x )
	{
	    GetAlert( BADFX, &alertp );
	    GetString( CLEARFX, &text );
	    sprintf( alert, alertp, maxpts, text );
	    form_alert( 1, alert );
	    ps->fx = x;
	    ObjcDraw( tree, STYSAMP, 1, NULL );
	    return FAILURE;
	}
    }

    return SUCCESS;
}


/* ----------------------------------------------------------------------
 * Update a style effects field from a button selection, and redraw.
 */
MLOCAL VOID FixFx( obj, fx, ps )
WORD	    obj;
UWORD	    fx;
Style	    *ps;
{
    ps->fx ^= fx;
    ps->fx = ValidFx( ps );

    if( ps->fx & fx )
	ObjcSelect(obj);
    else
	ObjcDeselect(obj);

    /*
     * Find out if these effects are satisfied by an FSM font style,
     * and flag the buttons accordingly.
     */
    FlagFsmFx( ps, YES );
    ObjcDraw( tree, STYSAMP, 1, NULL );

    wait_bup();
}


/* ----------------------------------------------------------------------
 * Update a point/set size text field, and draw it if it changes.
 */
MLOCAL VOID FixPts( obj, clip, inc, val )
WORD	    obj;
GRECT	    *clip;
WORD	    inc;
WORD	    *val;
{
    char    *s, *endp, nstr[5];
    long    l;

    /*
     * Get the number out of the TEDINFO, since the user may have typed one in.
     * If it's valid, use it as a base, otherwise use the passed in value.
     */
    s = TedText(obj);
    l = strtol( s, &endp, 10 );
    if( *endp )
	*val += inc;
    else
	*val = (short)l + inc;

    /*
     * Don't allow a 0 point size.
     */
    if( !*val )
    {
	if( inc )
	    *val += inc;
	else
	    ++(*val);
    }

    /*
     * String compare/copy used here so that the box is more attractive
     * most of the time (no edit underscores).
     */
    sprintf( nstr, "%4d", *val );
    if( strcmp(s, nstr) )
    {
	strcpy(s, nstr);
	ObjcDraw( tree, obj, 1, clip );
    }
}


/* ----------------------------------------------------------------------
 * Scroll point and/or set size text field(s) with accelerating auto-repeat.
 * This is done while the user has the button down on the up/down, left/right
 * or +/- point or set size buttons.
 */
MLOCAL VOID ScrollPts( rpts, rset, inc, ps )
GRECT	    *rpts;
GRECT	    *rset;
WORD	    inc;
Style	    *ps;
{
    WORD    sampts, samset, newpts, newset;
    long    delay, mindelay, quickening;

    /*
     * Save some initial values, so we can determine if we need to
     * redraw the style sample box when done scrolling.
     */
    sampts = (ps->pts > 0) ? SAMPLE_POINTSIZE : -SAMPLE_POINTSIZE;
    samset = sampts * ps->set / ps->pts;

    delay = dcrates[evnt_dclick(0,0)];
    mindelay = delay/2;
    quickening = (delay - mindelay)/8;
    do
    {
	if( rpts )
	    FixPts( STYPTS, rpts, inc, &ps->pts );
	if( rset )
	    FixPts( STYSET, rset, inc, &ps->set );
	EvntTimer( delay );
	if( delay > mindelay )
	    delay -= quickening;
	GrafMKState( &m );
    } while( m.buttons );

    newpts = (ps->pts > 0) ? SAMPLE_POINTSIZE : -SAMPLE_POINTSIZE;
    newset = newpts * ps->set / ps->pts;
    if( sampts != newpts || samset != newset )
	ObjcDraw( tree, STYSAMP, 1, NULL );
}


/* ----------------------------------------------------------------------
 * Set the point size to the set size or vice versa.
 * This is done when the user clicks on the "Pt. Size" or "Set Size" box.
 */
MLOCAL VOID Set2Pts( from, to, rfrom, rto, ps )
WORD	    from, to;
GRECT	    *rfrom, *rto;
Style	    *ps;
{
    WORD    *pfrom, *pto;
    WORD    sampts, samset, newpts, newset;
    GRECT   rfbig, rtbig;

    /*
     * Save some initial values, so we can determine if we need to
     * redraw the style sample box when done scrolling.
     */
    sampts = (ps->pts > 0) ? SAMPLE_POINTSIZE : -SAMPLE_POINTSIZE;
    samset = sampts * ps->set / ps->pts;

    if( from == STYPTS )
    {
	ObjcExtent( tree, STYPTSEQ, &rfbig );
	ObjcExtent( tree, STYSETEQ, &rtbig );
	pfrom = &ps->pts;
	pto = &ps->set;
    }
    else
    {
	ObjcExtent( tree, STYSETEQ, &rfbig );
	ObjcExtent( tree, STYPTSEQ, &rtbig );
	pfrom = &ps->set;
	pto = &ps->pts;
    }

    FixPts( from, rfrom, 0, pfrom );
    FixPts( to, rto, 0, pto );

    GrafZoombox( &rfbig, rfrom );
    GrafMovebox( rfrom, rto->g_x, rto->g_y );
    GrafZoombox( rto, &rtbig );

    if( *pto != *pfrom )
    {
	*pto = *pfrom;
	sprintf( TedText(to), "%4d", *pto );
	ObjcDraw( tree, to, 1, rto );

	newpts = (ps->pts > 0) ? SAMPLE_POINTSIZE : -SAMPLE_POINTSIZE;
	newset = newpts * ps->set / ps->pts;
	if( sampts != newpts || samset != newset )
	    ObjcDraw( tree, STYSAMP, 1, NULL );
    }

    wait_bup();
}


/* ----------------------------------------------------------------------
 * Update the skew text field.  This function is called by the slider handler
 * during active slider manipulation.
 */
MLOCAL VOID FixSkew( VOID )
{
    sprintf( TedText(STYSKEW), "%2d", abs(tmpstyle.skew) );
}


/* ----------------------------------------------------------------------
 * Update the faces display from the slider.  This function is called by
 * the slider handler during active slider manipulation, and by FaceWipe().
 */
MLOCAL VOID FixFace( VOID )
{
    WORD    i, o, diff, newstart, newend;
    GRECT   src, dst, clip;

    /*
     * If there are less faces than slots, it's easy...
     */
    if( maxface0 == 0 )
    {
	for( o = STYFONT1; o <= maxfontobj && !IsSelected(o); o++ )
	;
	if( o <= maxfontobj )
	    ObjcDeselect(o);
	ObjcSelect(STYFONT1 + tmpstyle.face);
	return;
    }


    /*
     * Determine what the new top and bottom of the list will be, and
     * fill in the object info.
     */
    newstart = face0;
    newend = face0 + NFACEOBS;
    for( i = newstart; i < newend; i++ )
    {
	o = STYFONT1 + i - face0;
	TedText(o) = faces[i].name;
	if( tmpstyle.face == i )
	    SelectObj(o);
	else
	    DeselectObj(o);
    }


    /*
     * Blit whatever part of the list is currently displayed,
     * then redraw the rest.
     */
    diff = lastface0 - face0;
    if( abs(diff) < NFACEOBS )
    {
	ObjcExtent( tree, STYFBOX, &clip ); /* parent box */
	ObjcExtent( tree, STYFONT1, &src ); /* one element of the list */
	dst = src;
	if( diff < 0 )
	{
	    /*
	     * Move the bottom part up.
	     */
	    src.g_y -= diff * src.g_h;
	    src.g_h = dst.g_h = src.g_h * (NFACEOBS + diff);
	    newstart += NFACEOBS + diff;
	}
	else
	{
	    /*
	     * Move the top part down.
	     */
	    dst.g_y += diff * src.g_h;
	    src.g_h = dst.g_h = src.g_h * (NFACEOBS - diff);
	    newend -= NFACEOBS - diff;
	}
	RectBlit( &clip, &src, &dst );
    }
    for( i = newstart; i < newend; i++ )
	ObjcDraw( tree, STYFONT1 + i - face0, ObMAXDEPTH, NULL );
    lastface0 = face0;
}


/* ----------------------------------------------------------------------
 * Wipe-select through faces while mouse button is down.
 */
MLOCAL VOID FaceWipe( VOID )
{
    WORD    boxx, boxy, obj, curobj, lastface, last0;
    GRECT   oldr, elevr;

    objc_offset( tree, STYFBOX, &boxx, &boxy );
    boxx += ObW(STYFBOX)/2;
    ObjcExtent( tree, STYFELEV, &elevr );

    /*
     * Prime the loop so that nothing happens if this is a click
     * rather than a wipe.
     */
    GrafMKState( &m );
    while( m.buttons )
    {
	/*
	 * Get the currently selected face,
	 * and find what face is under the mouse.
	 */
	curobj = STYFONT1 + tmpstyle.face - face0;
	lastface = tmpstyle.face;
	last0 = face0;
	obj = objc_find( tree, STYFBOX, ObMAXDEPTH, boxx, m.y );

	if( obj != curobj )
	{
	    /*
	     * New object under the mouse.
	     * If it's a selectable face, deselect the old one, and
	     * select the new one.
	     */
	    if( obj >= STYFONT1 && obj <= maxfontobj )
	    {
		tmpstyle.face = face0 + obj - STYFONT1;
		ObjcDeselect(curobj);
		ObjcSelect(obj);
	    }
	    else
	    {
		/*
		 * Mouse Y is outside the range of selectable faces.
		 * Scroll the list in the up or down if possible,
		 * and select the top or bottom of the list.
		 */
		if( m.y <= boxy )
		{
		    if( face0 )
		    {
			tmpstyle.face = --face0;
			ObjcDeselect(curobj);
		    }
		    else if( tmpstyle.face )
		    {
			tmpstyle.face = 0;
			ObjcDeselect(curobj);
			ObjcSelect(STYFONT1);
		    }
		}
		else
		{
		    if( face0 < maxface0 )
		    {
			tmpstyle.face = ++face0 + NFACEOBS-1;
			ObjcDeselect(curobj);
		    }
		    else if( tmpstyle.face < nfaces-1 )
		    {
			tmpstyle.face = nfaces-1;
			ObjcDeselect(curobj);
			ObjcSelect(maxfontobj);
		    }
		}
	    }

	    if( face0 != last0 )
	    {
		/*
		 * The list was scrolled, so adjust the slider elevator.
		 */
		oldr = elevr;
		SlXY( tree, STYFSLID, STYFELEV, FixFace, maxface0, 0, face0 );
		ObjcExtent( tree, STYFELEV, &elevr );
		SlDraw( tree, STYFSLID, STYFELEV, &oldr, &elevr );
	    }

	    if( tmpstyle.face != lastface )
	    {
		/*
		 * New face is selected, so flag the effects boxes
		 * for this face, and draw the sample.
		 */
		FlagFsmFx( &tmpstyle, YES );
		ObjcDraw( tree, STYSAMP, 1, NULL );
	    }
	}
	GrafMKState( &m );
    }
}


/* ----------------------------------------------------------------------
 * Initialize style form stuff without drawing anything.
 */
MLOCAL VOID InitStyleForm( VOID )
{
    WORD    i, o;

    /*
     * Perform one-time initialization if it hasn't been done yet.
     */
    if( !tree )
    {
	GetTree( STYLEFM, &tree );

	/*
	 * Set up a bunch of extents
	 */
	form_center( tree, &rtree.g_x, &rtree.g_y, &rtree.g_w, &rtree.g_h );
	ObX(ROOT) = rtree.g_x + (rtree.g_w-ObW(ROOT))/2;
	ObY(ROOT) = rtree.g_y + (rtree.g_h-ObH(ROOT))/2;
	mid = rtree;
	mid.g_x += mid.g_w / 2;
	mid.g_y += mid.g_h / 2;
	mid.g_w = mid.g_h = 0;

	ObjcExtent( tree, STYPTS, &rpts );
	ObjcExtent( tree, STYSET, &rset );
	i = ObW(STYPSDN)+1;
	RectCenter( &rpts, i, 0, &rpts );
	RectCenter( &rset, i, 0, &rset );

	ObjcExtent( tree, STYSKBOX, &rskewbig );
	RectCenter( &rskewbig, (rskewbig.g_w-2)/2, 0, &rskew );

	/*
	 * Set up user defined fx objects and one time string settings.
	 */
	sample.ub_code = PrintSample; /* int/ptr pun in GCC */
	sample.ub_parm = (long)(&tmpstyle);
	ObType(STYSAMP) = ObUSERDEF;
	ObSpec(STYSAMP) = &sample; /* int/ptr pun in MWC & GCC */

	for( i = 0; i < 6; i++ )
	{
	    fxobjub[i].ub_code = DrawFxobj; /* int/ptr pun in GCC */
	    fxobjub[i].ub_parm = (long)(ObSpec(fxobj[i]));
	    ObType(fxobj[i]) = ObUSERDEF;
	    ObSpec(fxobj[i]) = &fxobjub[i]; /* int/ptr pun in MWC & GCC */
	}

	TedTemplate(STYSKEW)[2] = (char)'\370'; /* degrees symbol */

	maxface0 = max( 0, nfaces-NFACEOBS );
	maxfontobj = (maxface0) ? STYFONT7 : STYFONT1 + nfaces - 1;

	i = STYFONT7;
	while( i > maxfontobj )
	{
	    TedText(i) = "";
	    ObFlags(i) = ObNONE;
	    --i;
	}

	/*
	 * Style slider height must be manually set, because of the way the
	 * AES handles vertical pixel-wise alignment in lower resolutions.
	 */
	ObH(STYFSLID) = ObH(STYFBOX) - ObH(STYFUP) * 2 - 2;

	/*
	 * curstyle may be externally initialized by an application.
	 */
	if( curstyle.pts == 0 )
	{
	    curstyle.name[0] = '\0';
	    curstyle.face = curstyle.skew = curstyle.fx = 0;
	    curstyle.pts = curstyle.set = 12;
	}
    }

    /*
     * Set stuff up based on current style.
     */

    tmpstyle = curstyle;
    face0 = min( tmpstyle.face, maxface0 );
    lastface0 = face0;

    /*
     * Set up face list and face box slider.
     */
    for( o = STYFONT1; o <= maxfontobj; o++ )
    {
	i = face0 + o - STYFONT1;
	if( tmpstyle.face == i )
	    SelectObj(o);
	else
	    DeselectObj(o);
	TedText(o) = faces[i].name;
    }
    SlSize( tree, STYFSLID, STYFELEV, hbox, nfaces, NFACEOBS );
    SlXY( tree, STYFSLID, STYFELEV, NULLFUNC, maxface0, 0, face0 );

    /*
     * Set up points, set sizes, effects buttons, and skew slider.
     */
    sprintf( TedText(STYPTS), "%4d", tmpstyle.pts );
    sprintf( TedText(STYSET), "%4d", tmpstyle.set );
    FlagFsmFx( &tmpstyle, NO );
    for( i = 0; i < 6; i++ )
    {
	TedCFill(fxbut[i]) = 0; /* 4 for tri-state "unchanged" state */
	if( tmpstyle.fx & (1<<i) )
	    SelectObj(fxbut[i]);
	else
	    DeselectObj(fxbut[i]);
    }
    SlXY( tree, STYSKWSL, STYSKEW, FixSkew, MAXSKEW, -MAXSKEW, tmpstyle.skew );
}


/* ----------------------------------------------------------------------
 * Compare 2 styles.  Return 0 only if they are identical except for rotation.
 */
GLOBAL int cmpStyle( a, b )
const Style *a;
const Style *b;
{
    int	    i;

    if( (i = strcmp(a->name, b->name)) != 0 )
	return i;

    if( a->face > b->face
    ||	a->pts  > b->pts
    ||	a->set  > b->set
    ||	a->skew > b->skew
    ||	a->fx   > b->fx   )
	return 1;

    if( a->face < b->face
    ||	a->pts  < b->pts
    ||	a->set  < b->set
    ||	a->skew < b->skew
    ||	a->fx   < b->fx   )
	return -1;

    return 0;
}

/* ----------------------------------------------------------------------
 * Handle user interaction with the Style form
 */
GLOBAL VOID StyleForm( VOID )
{
    WORD    i, x, exitobj;
    BOOLEAN done;

    if( !faces )
    {
	GetAlert( NOFONTS, &alertp );
	form_alert( 1, alertp );
	return;
    }

    graf_mouse( ARROW, NULL );
    InitStyleForm();
    undostyle = startstyle = tmpstyle;

    form_dial( FMD_START, mid.g_x, mid.g_y, mid.g_w, mid.g_h,
		rtree.g_x, rtree.g_y, rtree.g_w, rtree.g_h );
    form_dial( FMD_GROW, mid.g_x, mid.g_y, mid.g_w, mid.g_h,
		rtree.g_x, rtree.g_y, rtree.g_w, rtree.g_h );
    ObjcDraw( tree, ObROOT, ObMAXDEPTH, &rtree );

    done = FALSE;
    do
    {
	exitobj = form_do( tree, 0 ) & 0x7fff;
	FixPts( STYPTS, &rpts, 0, &tmpstyle.pts );
	FixPts( STYSET, &rset, 0, &tmpstyle.set );
	switch( exitobj )
	{
	    /*
	     * Redraw sample
	     */
	    case STYSAMP:
		FlagFsmFx( &tmpstyle, YES );
		ObjcDraw( tree, STYSAMP, 1, NULL );
		wait_bup();
	    break;

	    /*
	     * Face selection
	     */
	    case STYFONT1:
	    case STYFONT2:
	    case STYFONT3:
	    case STYFONT4:
	    case STYFONT5:
	    case STYFONT6:
	    case STYFONT7:
		i =  face0 + exitobj - STYFONT1;
		if( tmpstyle.face != i )
		{
		    tmpstyle.face = i;
		    for( i = STYFONT1; i <= maxfontobj && !IsSelected(i); i++ )
		    ;
		    if( i <= maxfontobj )
			ObjcDeselect(i);
		    ObjcSelect(exitobj);
		    FlagFsmFx( &tmpstyle, YES );
		    ObjcDraw( tree, STYSAMP, 1, NULL );
		}
		FaceWipe();
	    break;

	    /*
	     * Point size selection
	     */
	    case STYPTSDN:
		ScrollPts( &rpts, NULL, -1, &tmpstyle );
	    break;
	    case STYPTSUP:
		ScrollPts( &rpts, NULL, 1, &tmpstyle );
	    break;
	    case STYSETDN:
		ScrollPts( NULL, &rset, -1, &tmpstyle );
	    break;
	    case STYSETUP:
		ScrollPts( NULL, &rset, 1, &tmpstyle );
	    break;
	    case STYPSDN:
		ScrollPts( &rpts, &rset, -1, &tmpstyle );
	    break;
	    case STYPSUP:
		ScrollPts( &rpts, &rset, 1, &tmpstyle );
	    break;
	    case STYPTSEQ:
		Set2Pts( STYSET, STYPTS, &rset, &rpts, &tmpstyle );
	    break;
	    case STYSETEQ:
		Set2Pts( STYPTS, STYSET, &rpts, &rset, &tmpstyle );
	    break;


	    /*
	     * Skew slider
	     */
	    case STYSKWLF:
	    case STYSKWRT:
		i = tmpstyle.skew;
		SlArrow( tree, STYSKWSL, STYSKEW, exitobj, FixSkew,
			 (exitobj==STYSKWLF) ? 1 : -1, MAXSKEW, -MAXSKEW,
			 &tmpstyle.skew );
		if( tmpstyle.skew != i )
		    ObjcDraw( tree, STYSAMP, 1, NULL );
	    break;
	    case STYSKWSL:
		GrafMKState( &m );
		objc_offset( tree, STYSKEW, &x, &i );
		i = tmpstyle.skew;
		SlArrow( tree, STYSKWSL, STYSKEW, exitobj, FixSkew,
			 (m.x < x) ? 10 : -10, MAXSKEW, -MAXSKEW,
			 &tmpstyle.skew );
		if( tmpstyle.skew != i )
		    ObjcDraw( tree, STYSAMP, 1, NULL );
	    break;
	    case STYSKEW:
		i = tmpstyle.skew;
		SlDrag( tree, STYSKWSL, STYSKEW, FixSkew, MAXSKEW, -MAXSKEW,
			&tmpstyle.skew );
		if( tmpstyle.skew != i )
		    ObjcDraw( tree, STYSAMP, 1, NULL );
	    break;
	    case STYSKEW0:
		GrafZoombox( &rskewbig, &rskew );
		if( tmpstyle.skew )
		{
		    tmpstyle.skew = 0;
		    SlXY( tree, STYSKWSL, STYSKEW, FixSkew, MAXSKEW,
			  -MAXSKEW, 0 );
		    ObjcDraw( tree, STYSKWSL, ObMAXDEPTH, NULL );
		    ObjcDraw( tree, STYSAMP, 1, NULL );
		}
		wait_bup();
	    break;

	    /*
	     * Font slider
	     */
	    case STYFUP:
	    case STYFDN:
		SlArrow( tree, STYFSLID, STYFELEV, exitobj, FixFace,
			 (exitobj == STYFDN) ? 1 : -1, maxface0, 0, &face0 );
	    break;
	    case STYFSLID:
		GrafMKState( &m );
		objc_offset( tree, STYFELEV, &x, &i );
		SlArrow( tree, STYFSLID, STYFELEV, exitobj, FixFace,
			 (m.y < i) ? -7 : 7, maxface0, 0, &face0 );
	    break;
	    case STYFELEV:
		SlDrag( tree, STYFSLID, STYFELEV, FixFace,
			maxface0, 0, &face0 );
	    break;

	    /*
	     * FX buttons
	     */
	    case STYBOLD:
	    case STYBOLDT:
		FixFx( STYBOLD, FxBOLD, &tmpstyle );
	    break;
	    case STYLITE:
	    case STYLITET:
		FixFx( STYLITE, FxLITE, &tmpstyle );
	    break;
	    case STYITAL:
	    case STYITALT:
		FixFx( STYITAL, FxSKEW, &tmpstyle );
	    break;
	    case STYOUTL:
	    case STYOUTLT:
		FixFx( STYOUTL, FxOUTLINE, &tmpstyle );
	    break;
	    case STYUNDR:
	    case STYUNDRT:
		FixFx( STYUNDR, FxUL, &tmpstyle );
	    break;
	    case STYSHAD:
	    case STYSHADT:
		FixFx( STYSHAD, FxSHADOW, &tmpstyle );
	    break;

	    /*
	     * Help & Undo
	     */
	    case STYHELP:
		GetAlert( NOHELP, &alertp );
		form_alert( 1, alertp );
		ObjcDeselect( exitobj );
	    break;

	    case STYUNDO:
		if( cmpStyle(&tmpstyle, &undostyle) )
		{
		    if( cmpStyle(&tmpstyle, &startstyle) )
		    {
			curstyle = startstyle;
			undostyle = tmpstyle;
		    }
		    else
		    {
			curstyle = undostyle;
			undostyle = startstyle;
		    }
		    InitStyleForm();
		    ObjcDraw( tree, STYFBOX, ObMAXDEPTH, NULL );
		    ObjcDraw( tree, STYPTS, ObMAXDEPTH, &rpts );
		    ObjcDraw( tree, STYSET, ObMAXDEPTH, &rset );
		    ObjcDraw( tree, STYSKWSL, ObMAXDEPTH, NULL );
		    ObjcDraw( tree, STYSAMP, ObMAXDEPTH, NULL );
		    for( i = 0; i < 6; i++ )
			ObjcDraw( tree, fxbut[i], ObMAXDEPTH, NULL );
		}
		ObjcDeselect( exitobj );
	    break;

	    /*
	     * Finish up
	     */
	    case STYCAN:
		curstyle = startstyle;
		done = TRUE;
		DeselectObj(exitobj);
	    break;

	    case STYOK:
		if( FlagFsmFx(&tmpstyle, YES) == SUCCESS )
		{
		    curstyle = tmpstyle;
		    done = TRUE;
		    DeselectObj(exitobj);
		}
		else
		    ObjcDeselect(exitobj);
	    break;
	}
	if( !done )
	    FlagFsmFx( &tmpstyle, YES );

    } while( !done );

    form_dial( FMD_SHRINK, mid.g_x, mid.g_y, mid.g_w, mid.g_h,
		rtree.g_x, rtree.g_y, rtree.g_w, rtree.g_h );
    form_dial( FMD_FINISH, mid.g_x, mid.g_y, mid.g_w, mid.g_h,
		rtree.g_x, rtree.g_y, rtree.g_w, rtree.g_h );
}
