/*
**	Small program to patch the 'photolab' FFT-modules for Atari TT
**	computers, so they will run on other Atari hardware with
**	680x0 (x>=2) + 68881/2 combinations.
**
**	See README-file for explanantions.
**
**	This program is public domain.
**	The entire risk as to the quality and performance of the program
**	is with the user of this program. I don't take any responsibility
**	for ANY damages that may result from this code!
**
**	Tabs set to every 4 characters.
**	Written for Turbo-C (Pure-C).
**
**	24.9.1995, Dietmar Herrendrfer
*/

#include <aes.h>
#include <tos.h>
#include <stdlib.h>
#include <string.h>

#define FALSE 0
#define TRUE 1
#define NULL ((void *) 0L)

#define MAX_MOD_SIZE 50000L				/* Max. size for the module code. */

typedef int boolean;


/* Global variables: */
int				application_id;					/* AES application id. */
unsigned char	module_buffer[MAX_MOD_SIZE];	/* Buffer for the module code. */
unsigned long	module_length;					/* Length of the module. */


/*
____	Issue a small warning for all those who haven't read the README.
____
____	Returns:  - TRUE, if modules shall be patched.
____	          - FALSE, if nothing should be done. 
*/
boolean show_warning(void)
{
	return  form_alert(2, "[1][Do you really want to modify|"
	                      "the FFT-modules?][Abbruch|Ok]") == 2;
}


/*
____	Try to find the FPU-Cookie for the system we are running on.
____
____	Returns: The FPU-cookie.
*/
long get_fpucookie(void)
{
	typedef struct {			/* Definition of cookie structure. */
		char		id[4];
		long		value;
	} COOKIE;

	COOKIE		**cookie_jar_ptr = (COOKIE**)(0x5a0L);
	COOKIE		*cookie_jar;
	long		old_ss;
	long		fpu_cookie;
	boolean		cookie_found = FALSE;


	/*++ Locate the cookie-jar: ++*/
	old_ss = Super(0L);			/* Save old supervisor stack address. */
	cookie_jar = *cookie_jar_ptr;

	if ((cookie_jar) == NULL) {
		Super((void *)old_ss);	/* End supervisor mode. */
		form_alert(1, "[3][No cookie-jar installed!][Quit]");
		appl_exit();
		exit(0);
	}
	Super((void *)old_ss);		/* End supervisor mode. */

	/*++ Search for the _FPU - cookie: ++*/
	while (!cookie_found  && ((long)(*(cookie_jar->id)) != 0L)) {
		cookie_found = strcmp(cookie_jar->id, "_FPU") == 0;
		cookie_jar++;
	}
	if (cookie_found)
		fpu_cookie = (--cookie_jar)->value;
	else {
		form_alert(1, "[3][No \"_FPU\"-cookie installed!][Quit]");
		appl_exit();
		exit(0);
	}

	return fpu_cookie;
}


/*
____	Read the (unmodified) program code.
____
____	> name				= name of the file.
*/
boolean read_code(char *name)
{
	int		file_handle;
	char	alert_message[80] = "[3][Cannot open file|\"";


	/*++ Open the file: ++*/
	if ((file_handle = Fopen(name, FO_READ)) <= 0) {
		strcat(alert_message, name);
		strcat(alert_message, "\"!][Skip]");
		form_alert(1, alert_message);
		return FALSE;
	}

	/*++ Read the file...: ++*/
	module_length = Fread(file_handle, MAX_MOD_SIZE, module_buffer);

	/*++ ... and close it: ++*/
	Fclose(file_handle);
	return TRUE;
}


/*
____	Write the (modified) program code.
____
____	> name				= name of the file.
*/
void write_code(char *name)
{
	int		file_handle;

	/*++ Open the file: ++*/
	if ((file_handle = Fcreate(name, 0)) <= 0) {
		form_alert(1, "[3][Cannot open the output file!][Quit]");
		appl_exit();
		exit(0);
	}

	/*++ Write the file...: ++*/
	Fwrite(file_handle, module_length, module_buffer);

	/*++ ... and close it: ++*/
	Fclose(file_handle);
}


/*
____	Try to patch the code in the buffer.
____
____	> cookie			= new value for the FPU-cookie (upper word).
*/
boolean patch_code(unsigned int  fpu_cookie)
{
	unsigned char			*buffer_ptr = module_buffer;		/* Pointer into the module buffer. */
	boolean					pos_found = FALSE;
	static unsigned char	old_code[] = {0x5f,0x66,0x2a,0x0c,0x28,0x00,0x4d,0x00,
							              0x01,0x66,0x22,0x0c,0x28,0x00,0x43,0x00,
							              0x02,0x66,0x1a,0x0c,0x28,0x00,0x48,0x00,
							              0x03,0x66,0x12,0x0c,0x28,0x00,0x00,0x00,
							              0x04,0x66,0x24,0x0c,0x28,0x00,0x02};
	static unsigned char	new_code[] = {0x5f,0x66,0x2a,0x0c,0x28,0x00,0x46,0x00,
							              0x01,0x66,0x22,0x0c,0x28,0x00,0x50,0x00,
							              0x02,0x66,0x1a,0x0c,0x28,0x00,0x55,0x00,
							              0x03,0x66,0x12,0x0c,0x28,0x00,0x00,0x00,
							              0x04,0x66,0x24,0x0c,0x28,0x00,0x02};
	
	/*++ Set new cookie value: ++*/
	new_code[37] = 0xff & (unsigned char)(fpu_cookie >> 3);
	new_code[38] = 0xff & (unsigned char)(fpu_cookie);

	/*++ Search the buffer for the original coding: ++*/
	do {
		if (*buffer_ptr == old_code[0])
			pos_found = memcmp(buffer_ptr, old_code, 39L) == 0;			
		if (!pos_found)
			buffer_ptr++;
	} while(!pos_found  &&  module_length > (unsigned long)(buffer_ptr - module_buffer));
	
	/*++ Replace with the new code: */
	if (pos_found)
		memcpy(buffer_ptr, new_code, 39L);

	return pos_found;
}


/*
____	Main program:
____	- The only optional command-line argument is the new FPU-cookie
____	  setting (value of upper word, i.e. if
____	  _FPU = $00020000 then "2" should be given on the command-line.
____	- Nothing is returned.
*/
void main(int argc, char *argv[])
{
	typedef struct {
		char	old[128],		/* Old name of the module. */
				new[128];		/* New name for the module. */
	} MODULE_NAMES;

	static MODULE_NAMES mod_list[] = {{".\\TT.FPU\\FFT2D064.TOS", ".\\FFT2D064.TOS"},
			                          {".\\TT.FPU\\FFT2D128.TOS", ".\\FFT2D128.TOS"},
	                                  {".\\TT.FPU\\FFT2D256.TOS", ".\\FFT2D256.TOS"},
	                                  {".\\TT.FPU\\FFT2D512.TOS", ".\\FFT2D512.TOS"},
	                                  {".\\TT.FPU\\IFT2D064.TOS", ".\\IFT2D064.TOS"},
	                                  {".\\TT.FPU\\IFT2D128.TOS", ".\\IFT2D128.TOS"},
	                                  {".\\TT.FPU\\IFT2D256.TOS", ".\\IFT2D256.TOS"},
	                                  {".\\TT.FPU\\IFT2D512.TOS", ".\\IFT2D512.TOS"},
	                                  {"END", "END"}};
	MODULE_NAMES		*names = mod_list;
	int					fpu_cookie;
	char				alert_message[80];


	/* Initialisation: */
	if ((application_id = appl_init()) == -1)		/* Get an appl. id. */
		exit(0);									/* Quit otherwise. */
	if (argc > 1)
		fpu_cookie = atoi(argv[1]);
	

	/* The few things this program does: */
	if (show_warning()) {
		if (argc == 1)
			fpu_cookie = (int)(get_fpucookie() >> 16);
			
		while (strcmp(names->old, "END") != 0) {
			if (read_code(names->old)) {
				if (patch_code(fpu_cookie))
					write_code(names->new);
				else {
					strcpy(alert_message, "[3][The module|\"");
					strcat(alert_message, names->old);
					strcat(alert_message, "\"|couldn't be patched!][Skip]");
					form_alert(1, alert_message);
				}
			}
			names++;					/* next module. */
		}
	}

	/* Clean up & exit: */
	appl_exit();
}