/* =========================================================
	Files.4th from FORTHMAC-Kernal
	modified by Rainer Saric

	GemDos interface
	Forth File System interface to the Gem operating system

	Interfaces to Gem system calls:

	f_open   ( name mode -- fhandle )
	f_close  ( fhandle -- err? )
	f_read   ( fhandle count address -- actual-count )
	f_write  ( fhandle count address -- actual-count )
	f_seek   ( offset fhandle mode -- actual-position )
	f_create ( name protection -- fhandle)
	========================================================= */

#ifndef gemdos
	bload bin\gemdos.bin >voc gemdos
#endif

mforth gemdos also definitions

/* Interfaces between the buffering code and the lower level operating
	system code.  This is the stuff that has to be reimplemented to port
	to a different operating system.

	Rounds down to a block boundary.  This causes all file accesses to the
	underlying operating system to occur on disk block boundaries.  Some
	systems (e.g. CP/M) require this; others which don't require it (e.g. Gem)
	usually run faster with alignment than without. */

: gemalign ( byteno -- aligned )   $FFFFFE00 and ;

/* An implementation factor which
	positions to byte number "byteno" in the file "fhandle" */

: gemfseek ( byteno fhandle -- )  0 ( L_SET ) f_seek drop ; 

/* Return the current position "current-position" within the file "fhandle" */
: gemftell ( fhandle -- current-position )	0 swap 1 f_seek ;

/* Return the current size "size" of the file "fhandle" */
: gemflen ( fhandle -- size )
   dup >r   /* remember the current position */
	gemftell	( current-position )
   /* seek to end of file to find out where the eof is */
	0 r@ 2 ( L_XTND ) f_seek ( current-position size )
   /* return to the original position */
   swap r> gemfseek ;	( size )
 
/* Read at most "count" bytes into the buffer at address "addr" from the
	file "fhandle" starting at position "byteno".  Returns the number of
	bytes actually read. */

: gemfread  ( addr count byteno fhandle -- #read )
   dup >r gemfseek swap r> -rot f_read ;

/* Write "count" bytes from the buffer at address "addr" to the
	file "fhandle" starting at position "byteno".  Returns the number of
	bytes actually written. */

: gemfwrite ( addr count byteno fhandle -- #written )
   dup >r gemfseek      	( addr count )
   swap r> -rot f_write ; 	( #written )

: gemfclose ( fhandle -- )  f_close drop ;

/* A "do-nothing" read routine which is used as the read routine for
	write-only files. */
: nullread ( addr count l.byteno fd -- count=0 )
   2drop 2drop 0 ;

/* &ptr is the address of a pointer.  fetch the pointed-to
	character and post-increment the pointer */

code @c@++ ( &ptr --- char )
	.l a6 )+ a0 move  a0 ) a1 move	d0 clr
  	.b a1 )+ d0 move	.l d0 a6 -) move
	a1 a0 ) move
	next end-code 
 
/* &ptr is the address of a pointer.  store the character into
	the pointed-to location and post-increment the pointer */
 
code @c!++ ( char &ptr --- )
	.l a6 )+ a0 move	   a0  ) a1 move
	   a6 )+ d0 move  .b d0 a1 )+ move 
	.l a1 a0 ) move
next end-code

-1 constant EOF |
integer file 
integer ferr 

/* The file descriptor structure describes an open file.
	There is a pool of several of these structures.	When a file is opened,
	a structure is allocated and initialized.	While performing an io
	operation, the user variable "file" contains a pointer to the file
	on which the operation is being performed. */
struct 
{	4 field bfbase		/* starting address of the buffer for this file */
	4 field bflimit	/* ending address of the buffer for this file */
	4 field bftop 	 	/* address past last valid character in the buffer */
	4 field bfend 	  	/* address past last place to write in the buffer */
	4 field bfcurrent /* address of the current character in the buffer */
	4 field bfdirty	/* contains true if the buffer has been modified */
	4 field fmode 	  	/* not-open, read, write, or modify */
	4 field fstart	  	/* Position in file corresponding to first byte in buffer */
	4 field fid		  	/* File handle for underlying operating system */
	4 field fread 	  	/* Points to routine to read blocks from underlying system */
	4 field fwrite	  	/* Points to routine to write blocks to underlying system */
	4 field fclose	  	/* Points to routine to close file on underlying system */
	4 field falign	  	/* Points to routine to align on a block boundary */
	4 field flen		/* Points to routine to return the size of the file */
} FD

/* An implementation factor which
	initializes the current descriptor to use the buffer "bufstart,buflen" */
: initbuf ( bufstart buflen fd -- )
	dup >r fstart off 
	over + r@ bflimit !
	dup r@ bfbase 	  ! 
	dup r@ bfcurrent ! 
	dup r@ bfend 	  ! 
	    r@ bftop 	  !
  	    r> bfdirty   off ;

/* If the underlying operating system requires that files be accessed
	in fixed-length records, then /fbuf must be a multiple of that length.
	Even if the system allows arbitrary length file accesses, there is probably
	a length that is particularly efficient, and /fbuf should be a multiple
	of that length for best performance.  1K works well for many systems. */

2048 constant /fbuf |

/* An implementation factor which gets a file descriptor and attaches
	a file buffer to it. */
: init-fd ( mode fd str -- mode fd str flag )
	['] gemfread  2pick fread  !
	['] gemfwrite 2pick fwrite !
	['] gemfclose 2pick fclose !
	['] gemalign  2pick falign !
	['] gemflen	  2pick flen   !
	over EOF over fmode !
	/fbuf calloc
	if		/fbuf rot initbuf true
	else	drop false
	endif ;		

/* If we are using buffered output to the output stream, we need to
	flush the buffer when the system tries to read from the input stream. */

: string-flen ( fd -- length )  2@ swap -  ;

/* These are the words that a program uses to read and write to/from a file.

	An implementation factor which ensures that the bftop is >= the bfcurrent
	variable.  bfcurrent can temporarily advance beyond bftop while a file is
	being extended. */

: sync ( -- ) ( if current > top, move up top )
	file bftop @ file bfcurrent @ < if file bfcurrent @ file bftop ! endif ;

/* If the current file's buffer is modified, write it out
	Need to better handle the case where the file can't be extended,
	for instance if the file is a memory array */
: (fflush ( -- )
	file bfdirty @
	if		sync
    		file bftop @ file bfbase @  - 					( #bytes-to-write)
    		file bfbase @ over file fstart @ file fid @  ( #bytes addr #bytes l.position fid )
    		file fwrite perform 							( #bytes-to-write #bytes-written )
    		/*  Do this before aborting to prevent an infinite loop as the
    			 quit routine tries to close all the open files */
    		file bfdirty off
    		>  to ferr 
			file bfbase @ dup file bftop ! file bfcurrent !
	endif ;

/* An implementation factor to
	fill the buffer with a block from the current file.  The block will
	be chosen so that the file address "addr" is somewhere within that
	block. */

: fillbuf ( addr -- )
	file falign perform file fstart !   /* Align the position to a buffer boundary */
  	file bfbase @ file bflimit @ over -			( addr #bytes-to-read )
  	file fstart @ file fid @ file fread perform 	( #bytes-read )
  	file bfbase @ + file bftop !
  	file bflimit @  file bfend ! ;

/* Return the current position within the current file */

: (ftell ( -- addr )  file fstart @ file bfcurrent @ file bfbase @ - + ;

/* An implementation factor to
	return the accress within the buffer which corresponds to the
	selected position "laddr" within the current file. */
: >bufaddr ( addr -- bufaddr )	file fstart @ - file bfbase @ + ;

/* An implementation factor which
	advances to the next block in the file.  This is used when accesses
	to the file are sequential (the most common case).
	Assumes the byte is not already in the buffer! */
: shortseek ( bufaddr -- )
	file bfbase @ -  file fstart @ +  ( addr ) 
  	(fflush dup fillbuf					 ( addr ) 
	>bufaddr file bftop @ min file bfcurrent !  ;

/* Close the current file. */
: (close ( -- )  
	file @ /* bfbase */
	if 	(fflush EOF file fmode ! 
			file fid @ file fclose perform ( fbase ) file @ m_free drop
			file off
	endif ;

/* Returns true if the current file has reached the end.
	XXX This may only be valid after (fseek or shortseek */
: feof? ( -- f ) 	file bfcurrent @ file bftop @ >= ;

/* Get the next byte from the current file	*/
: (fgetc ( -- byte )
	file bfcurrent @ file bftop @ <
  	if		/* desired character is in the buffer */
		 	file bfcurrent @c@++
  	else 	/* end of buffer has been reached */
		 	file bfcurrent @ shortseek
		 	feof? if EOF else file bfcurrent @c@++ endif
  	endif ;

/*	Starting here, some stuff doesn't have to be in the kernel 

	Seek to the position "laddr" within the current file. */

: (fseek ( addr -- )
	sync
	/* See if the desired byte is in the buffer */
	dup >bufaddr dup				( laddr bufaddr bufaddr )
	file bfbase @ file bfend @ within	( laddr bufaddr inbuf? )
	/* Is the desired byte already in the buffer? */
	if		swap drop				 	( bufaddr )
	else	drop (fflush dup fillbuf  ( laddr )
		 	>bufaddr					( bufaddr )
	endif
	/* Seeking past end of file actually goes to the end of the file */
	file bftop @ min file bfcurrent ! ;


/* Get the byte at the position "laddr" from the current file */
: (filec@ ( addr -- byte )
	(fseek feof? if EOF else file bfcurrent @c@++ endif ;

/* Store a byte into the current file at position "addr" */
: (filec! ( byte addr -- )
	(fseek ( byte )
	/* if daddr is >= eof address, fseek will set up bfcurrent
		at the eof address, so appending to the file will occur */
	file bfcurrent @c!++
	file bfdirty on ;

/* Store a byte into the current file at the next position */
: (fputc ( byte -- )
	file bfcurrent @ file bfend @ >=
	if		/* buffer is full */
			file bfcurrent @ shortseek
	endif
	file bfcurrent @c!++
	file bfdirty on ;

/* An implementation factor
	Copyin copies bytes starting at current into the file buffer at
	bfcurrent.  The number of bytes copied is either all the bytes from
	current to end, if the buffer has enough room, or all the bytes the
	buffer will hold, if not.
	newcurrent is left pointing to the first byte not copied. */

: copyin ( end current -- end newcurrent )
	2dup - 									( end current remaining )
	file bfend @ file bfcurrent @	-	( end current remaining bfremaining )
	min 										( end current #bytes-to-copy )
	dup if  file bfdirty on  endif 	( end current #bytes-to-copy )
	2dup file bfcurrent @ swap			( end current #bytes  current bfcurrent #bytes)
	cmove										( end current #bytes )
	dup file bfcurrent +!				( end current #bytes )
	+ ;										( end newcurrent)

/* Copyout copies bytes from the file buffer into memory starting at current.
	The number of bytes copied is either enough to fill memory up to end,
	if the buffer has enough characters, or all the bytes the
	buffer has left, if not.
	newcurrent is left pointing to the first byte not filled. */

: copyout ( end current -- end newcurrent )
	2dup - 									( end current remaining )
	file bftop @ file bfcurrent @	-	( end current remaining bfrem )
	min 										( end current #bytes-to-copy)
	2dup file bfcurrent @ rot rot 	( end current #bytes  current bfcurrent #bytes)
	cmove										( end current #bytes)
	dup file bfcurrent +!				( end current #bytes)
	+ ;										( end newcurrent )

/* Write count characters from address addr into the current file */
: (fputs ( addr count -- )
	over + swap	( endaddr startaddr )
	begin	copyin 2dup >
	while
		/* Here there should be some code to see if there are enough remaining
			bytes in the request to justify bypassing the file buffer and writing
			directly from the user's buffer.  'Enough' = more than one file buffer
		 */
			sync file bfcurrent @ shortseek ( endaddr curraddr )
	repeat
	2drop ;

/* Read up to count characters from the current file into memory starting 
	at address "addr" */

: (fgets ( addr count -- nread )
	sync
	over + over	( startaddr endaddr startaddr )
	begin	copyout 2dup >
	while
		/* Here there should be some code to see if there are enough remaining
	 		bytes in the request to justify bypassing the file buffer and reading
	 		directly to the user's buffer.  'Enough' = more than one file buffer
		 */
			file bfcurrent @ shortseek ( startaddr endaddr curraddr )
		  	feof? if nip swap - exit endif
  repeat nip swap - ;

/* ========================================================== */

: open	( flag fd str -- flag )
	init-fd
	if		2pick f_open dup 2pick fid ! 0>
			if		fmode !  true	
			else	nip @ m_free drop	false
			endif
	else	2drop drop false
	endif ;

: make	( flag fd str -- flag )
	init-fd
	if		2pick f_create dup 2pick fid ! 0>
			if		fmode !  true	
			else	nip @ m_free drop	false
			endif
	else	2drop drop false
	endif ;

: fexist	( addr -- flag )	2 swap f_sfirst 0< not ;

/* An implementation factor which returns true if the file descriptor
	fd is not currently in use */
: fdavail? ( fd -- f ) fmode @ EOF = ;
: fputs	( addr len fd -- )		 	to file (fputs  ;
: fgets	( addr len fd -- nread ) 	to file (fgets  ;
: filec@ ( addr fd -- byte ) 			to file (filec@ ;
: filec! ( byte addr fd -- ) 			to file (filec! ;
: fputc	( byte fd -- ) 				to file (fputc  ;
: fseek	( addr fd -- )					to file (fseek  ;
: ftell	( fd -- pos ) 					to file (ftell  ;
: fgetc	( fd -- byte ) 				to file (fgetc  ;
: close	( fd -- )						to file (close  ;
: fflush	( fd -- )						to file (fflush ;
: fsize 	( fd -- lsize ) 				to file
  file fstart @ file bftop @ file bfbase @ - +  ( buffered-position )
  file fid @ file flen perform max ;            ( buffered-position file-size )

hide gemalign EOF
hide file FD 
hide initbuf open

mforth 

#ifdef gem

gemdos also definitions 

create path      92 allot |  
create sel       16 allot |
create outfile  108 allot | 

: GetPath ( ext -- addr )
	path @ 0=
	if		>r [ gemdos ] 
			d_getdrv ascii A + path c!
						ascii :   path 1+ c!
			path 2+ 0 d_getpath drop
			s" \" path strcat
			   r> path strcat
	else  drop
	endif	path ;

: <. ( addr -- addr' ) strlen +  begin dup c@ ascii . <> 2pick 2pick <> and while 1- repeat ;
: <\ ( addr -- addr' ) begin dup c@ ascii \ <> 2pick 2pick <> and while 1- repeat 1+ ;

: -path  ( cstr -- cstr )	  dup <. <\ 0 swap c! ;
: -file  ( cstr -- cstr )			<. <\ ;
: +ext 	( cstr cext -- cstr )	over strcat ;
: -ext 	( cstr -- 0$ )				dup <. 0 swap c! ;
: ?ext 	( cstr -- 0$ )				dup <. nip 1+ ;

gem definitions

: SelectFile	( 'cstr' extension -- cstr true | false )
	1 wind_update 

	GetPath sel rot global w@ $140 <  
	if		drop fsel_input
	else	fsel_exinput 
	endif	dup >r
	if		path outfile strcpy
			outfile -path 
			sel over strcat
	endif r>
 	
	0 wind_update ;

hide GetPath -path

#endif

mforth

