/** GameBoy Cartridge Info and Manufacturer Codes ************/
/**                                                         **/
/**                        GBCarts.c                        **/
/**                                                         **/
/** This file contains functions to extract information     **/
/** from GameBoy cartridge images. It also include a list   **/
/** of known manufacturer codes.                            **/
/**                                                         **/
/** Copyright (C) Marat Fayzullin 1995,1996                 **/
/**     You are not allowed to distribute this software     **/
/**     commercially. Please, notify me, if you make any    **/
/**     changes to this file.                               **/
/*************************************************************/

#include "GBCarts.h"

/** Companies ************************************************/
/** This is a list of known producers and their MakerIDs.   **/
/** It is used by GB_Maker() function.                      **/
/*************************************************************/
static struct { int Code;char *Name; } Companies[] =
{
  { 0x3301,"Nintendo"  },{ 0x7901,"Accolade"  },{ 0xA400,"Konami"    },
  { 0x6701,"Ocean"     },{ 0x5601,"LJN"       },{ 0x9900,"ARC?"      },
  { 0x0101,"Nintendo"  },{ 0x0801,"Capcom"    },{ 0x0100,"Nintendo"  },
  { 0xBB01,"SunSoft"   },{ 0xA401,"Konami"    },{ 0xAF01,"Namcot?"   },
  { 0x4901,"Irem"      },{ 0x9C01,"Imagineer" },{ 0xA600,"Kawada?"   },
  { 0xB101,"Nexoft"    },{ 0x5101,"Acclaim"   },{ 0x6001,"Titus"     },
  { 0xB601,"HAL"       },{ 0x3300,"Nintendo"  },{ 0x0B00,"Coconuts?" },
  { 0x5401,"Gametek"   },{ 0x7F01,"Kemco?"    },{ 0xC001,"Taito"     },
  { 0xEB01,"Atlus"     },{ 0xE800,"Asmik?"    },{ 0xDA00,"Tomy?"     },
  { 0xB100,"ASCII?"    },{ 0xEB00,"Atlus"     },{ 0xC000,"Taito"     },
  { 0x9C00,"Imagineer" },{ 0xC201,"Kemco?"    },{ 0xD101,"Sofel?"    },
  { 0x6101,"Virgin"    },{ 0xBB00,"SunSoft"   },{ 0xCE01,"FCI?"      },
  { 0xB400,"Enix?"     },{ 0xBD01,"Imagesoft" },{ 0x0A01,"Jaleco?"   },
  { 0xDF00,"Altron?"   },{ 0xA700,"Takara?"   },{ 0xEE00,"IGS?"      },
  { 0x8300,"Lozc?"     },{ 0x5001,"Absolute?" },{ 0xDD00,"NCS?"      },
  { 0xE500,"Epoch?"    },{ 0xCB00,"VAP?"      },{ 0x8C00,"Vic Tokai" },
  { 0xC200,"Kemco?"    },{ 0xBF00,"Sammy?"    },{ 0xD200,"Quest?"    },
  { 0x1800,"Hudson Soft"    },{ 0xCA01,"Palcom/Ultra" },
  { 0xCA00,"Palcom/Ultra"   },{ 0xC500,"Data East?" },
  { 0xA900,"Technos Japan?" },{ 0xD900,"Banpresto?" },
  { 0x7201,"Broderbund?"    },{ 0x7A01,"Triffix Entertainment?" },
  { 0xE100,"Towachiki?"     },{ 0x9300,"Tsuburava?" },
  { 0xC600,"Tonkin House?"  },{ 0xCE00,"Pony Canyon" },
  { 0x7001,"Infogrames?"    },{ 0x8B01,"Bullet-Proof Software" },
  { 0x5501,"Park Place"     },{ 0xEA00,"King Records?" },
  { 0x5D01,"Tradewest?"     },{ 0x6F01,"ElectroBrain?" },
  { 0xAA01,"Broderbund?"    },{ 0xC301,"SquareSoft" },
  { 0x5201,"Activision?"    },{ 0x5A01,"Bitmap Brothers/Mindscape" },
  { 0x5301,"American Sammy" },{ 0x4701,"Spectrum Holobyte" },
  { 0x1801,"Hudson Soft"    },{ 0x8B00,"Bullet-Proof Software" },
  { 0x0000,0 }
};

/** GB_RAMSize() *********************************************/
/** Size of on-cartridge RAM in bytes.                      **/
/*************************************************************/
int GB_RAMSize(unsigned char *Header)
{
  static int Sizes[4] = { 0,2048,8192,32768 };
  return(Sizes[GB_Type(Header)&0x03]);
}

/** GB_Name() ************************************************/
/** Extract and truncate cartridge name. Returns a pointer  **/
/** to internal buffer!                                     **/
/*************************************************************/
char *GB_Name(unsigned char *Header)
{
  static char Buf[32];
  int J;

  for(J=0x134;(J<0x144)&&Header[J];J++)
    Buf[J-0x134]=Header[J]>' '? Header[J]:' ';
  Buf[J-0x134]='\0';
  return(Buf);
}

/** GB_Maker() ***********************************************/
/** Try to figure out producer's name. Returns name or NULL **/
/** if no producer is known for cartridge's GB_MakerID().   **/
/*************************************************************/
char *GB_Maker(unsigned char *Header)
{
  int Code,J;

  Code=GB_MakerID(Header);
  for(J=0;Companies[J].Name;J++)
    if(Code==Companies[J].Code) return(Companies[J].Name);
  return(0);
}

/** GB_RealCMP() *********************************************/
/** Calculate complement byte of a cartridge.               **/
/*************************************************************/
unsigned char GB_RealCMP(unsigned char *Header)
{
  unsigned char R;
  int J;

  for(J=0x134,R=25;J<0x14D;J++) R+=Header[J];
  return(0x100-R);
}

/** GB_RealCRC() *********************************************/
/** Calculate CRC of a cartridge.                           **/
/*************************************************************/
unsigned short GB_RealCRC(unsigned char *Data)
{
  unsigned short R;
  int J,Length;

  Length=GB_ROMSize(Data);R=0;
  for(J=0;J<Length;J++) R+=Data[J];
  return(R-Data[0x14E]-Data[0x14F]);
}
