/*================================================================*
 *
 *  VCSMAKER.C   by Pixelboy (Luc Miron)
 *
 *  Version 0.2a  -  Last modified on Febuary 10th 1997
 *
 *================================================================*
 *
 *  This program creates a VCS file from the data found in the
 *  "VCSMAKER.CFG" file, which is supposed to lie in the same
 *  directory as the executable file "VCSMAKER.EXE".
 *
 *  The program requires that the BIN filename (ex. "ADVNTURE.BIN"
 *  or simply "ADVNTURE" since ".BIN" extension is assumed)
 *  be passed as parameter at the DOS prompt. The program then
 *  opens the "VCSMAKER.CFG" file and looks for the configuration
 *  data associated to the BIN filename (Default values are used
 *  if no such data is found). The VCS file is then created with
 *  the found data.
 *
 *  New with version 0.2: If the "-a" switch is specified instead
 *  of a specific filename, then ALL the VCS files defined in
 *  the "VCSMAKER.CFG" file are generated in the current directory.
 *
 *================================================================*/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <conio.h>

#define  VERSION_NUM  "0.2a" /* Current version number of this program    */
#define  LINESIZE     256    /* maximum size of data line in VCSMAKER.CFG */
#define  FILENAMESIZE 128    /* maximum length of BIN filename            */

typedef struct CONFIG_DATA_STRUCT
               {
                  char Cart_Name[80];
                  char Cart_Image[80];
                  char Cart_Type[6];
                  char Dsp_FrameRate[5];
                  char Dsp_YStart[5];
                  char Dsp_Height[5];
                  char Dsp_XStart[5];
                  char Dsp_Width[5];
                  char Con_LeftDiff[2];
                  char Con_RightDiff[2];
                  char Con_TVType[6];
                  char Ctrl_Paddle0[5];
                  char Ctrl_Paddle1[5];
                  char Ctrl_Paddle2[5];
                  char Ctrl_Paddle3[5];
                  char Ctrl_Joy0[5];
                  char Ctrl_Joy1[5];
                  char Timer_Adj[2];
               } ConfigDataStruct;



/*<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<*
 *  Function pre-definitions
 *>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*/

void  Print_Help         ( );
void  Format_Filename    ( char *filename );
int   Is_Line_Of_Data    ( char *CurrentLine );
char *Get_Next_Parameter ( char *PtrLine, char *Parameter );
void  Retrieve_Data      ( char *CurrentLine, ConfigDataStruct *ConfigData,
                           FILE *vcsmaker_cfg );
void  Get_Config_Data    ( char *BinFilename, ConfigDataStruct *ConfigData );
void  Init_Default_VCS   ( char *BinFilename, ConfigDataStruct *ConfigData );
void  Make_VCS_File      ( char *Binfilename, ConfigDataStruct *ConfigData );
void  Make_All_VCS_Files ( ConfigDataStruct *ConfigData );



/*<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<*
 *  Main function
 *>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*/

void main (int argc, char *argv [])
{
   char BinFilename[FILENAMESIZE];
   ConfigDataStruct ConfigData;

   /*----------------------------*
     Displays help text if no parameter
     specified, or if '-h' switch is specified.
    *----------------------------*/
   if ( (argc != 2) ||
        (strcmp(argv[1], "-h") == 0 || strcmp(argv[1], "/h") == 0 ) )
   {
      Print_Help();
      exit(0);
   }

   /*--------------------------------------------------------*
     Generates ALL the VCS files if '-a' switch is specified.
    *--------------------------------------------------------*/
    if ( ( strcmp(argv[1], "-a") == 0 ) || (strcmp(argv[1], "/a") == 0 ) )
    {
       Make_All_VCS_Files( &ConfigData );
       exit(0);
    }

   /*------------------------------------------------------------*
     Retrieves the parameter if not '/h' or '-h' or '/a' or '-a'.
    *------------------------------------------------------------*/
   strcpy ( BinFilename, argv[1] );
   Format_Filename ( BinFilename );

   /*----------------------------------------*
     Reads the data in the VCSMAKER.CFG file.
    *----------------------------------------*/
   Get_Config_Data ( BinFilename, &ConfigData );

   /*---------------------*
     Creates the VCS file.
    *---------------------*/
   Make_VCS_File ( BinFilename, &ConfigData );

   printf("\n");
}



/*<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<*
 *  Sub-functions
 *>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*/

/*---------------------------------------------------------------*
 *
 *  Print_Help
 *
 *  Displays general instructions on how to use the program.
 *
 *---------------------------------------------------------------*/

void Print_Help ()
{
   printf("\nVCSMAKER.EXE - version %s\n", VERSION_NUM );
   printf("----------------------------\n");
   printf("Usage: VCSMAKER <filename>  (-h for help, or -a to generate all)\n");

   printf("\nThis program generates VCS files compatible with the Stella emulator.\n");
   printf("The filename required as parameter must be the name of an Atari 2600\n");
   printf("binary ROM file (\"BIN file\" for short). Since the .BIN extension is\n");
   printf("assumed, you do not have to specify it within the filename parameter.\n");

   printf("\nIf the filename specified is defined in the VCSMAKER.CFG file (which\n");
   printf("must lie in the same directory as VCSMAKER.EXE), the VCS file associated\n");
   printf("with the BIN file is created using the data found in the VCSMAKER.CFG\n");
   printf("file. If there is no such data for the filename specified, the program\n");
   printf("creates a default VCS file, which you can edit to your liking.\n");

   printf("\nPlease read the VCSMAKER.TXT file for further information.\n\n");
}


/*---------------------------------------------------------------*
 *
 *  Format_Filename
 *
 *  Removes any extension (even .BIN) and returns the striped
 *  filename. If path information is specified, the program
 *  is halted.
 *
 *---------------------------------------------------------------*/

void Format_Filename ( char *filename )
{
   char *res1, *res2;
   int i;

   /*---------------------------------------------------------------*
     Path information is not necessary, and therefore not permitted.
    *---------------------------------------------------------------*/
   res1 = strchr( filename, '/' );
   res2 = strchr( filename, '\\' );
   if (res1 != NULL || res2 != NULL)
   {
      printf("\nYou must not specify any kind of path with the filename.\n");
      exit(1);
   }

   /*------------------------------------------------------------*
     Looks for the '.' (dot) character and replaces it with '\0'.
     This removes the extension from the filename. May cause
     trouble with long filenames (was designed for MS-DOS...).
    *------------------------------------------------------------*/
   for( i=0; filename[i] != '\0'; i++ )
      if ( filename[i] == '.' )
         filename[i] = '\0';
}


/*---------------------------------------------------------------*
 *
 *  Is_Line_Of_Data
 *
 *  Returns 1 if the line given as parameter is not empty, and
 *  is not a comment line (beginning with a '#' character).
 *  Otherwise returns 0.
 *
 *---------------------------------------------------------------*/

int Is_Line_Of_Data ( char *CurrentLine )
{
   int i;

   /* Check if line is empty */
   if ( strlen(CurrentLine) == 0 )
      return 0;

   /* Pass all spaces at the beginning of the line */
   for ( i=0; CurrentLine[i] == ' '; i++ );

   /* check for comment lines or empty lines */
   if ( ( CurrentLine[i] == '#' ) || (CurrentLine[i] == '\0') ||
        ( CurrentLine[i] == '\n') )
     return 0;

   return 1;
}


/*---------------------------------------------------------------*
 *
 *  Get_Next_Parameter
 *
 *  Reads characters using the "PtrLine" pointer until the ";"
 *  character is reached. The "PtrLine" will point one character
 *  after the ";" when this function ends its execution.
 *  All characters read are put in the "Parameter" string.
 *
 *---------------------------------------------------------------*/

char *Get_Next_Parameter ( char *PtrLine,
                           char *Parameter )
{
   int i;

   /*----------------------*
     Skip blank characters.
    *----------------------*/
   for( ; *PtrLine == ' '; PtrLine++);

   /*-------------------------------------*
     Copy characters until ";" is reached.
    *-------------------------------------*/
   for( i=0; *PtrLine != ';'; PtrLine++, i++)
   {
      if ( *PtrLine == '\0')
      {
         printf("Error: Unexpected end of line!");
         exit(1);
      }
      Parameter[i] = *PtrLine;
   }

   Parameter[i] = '\0';
   PtrLine++;

   return PtrLine;
}


/*---------------------------------------------------------------*
 *
 *  Retrieve_Data
 *
 *  Retrieves the configuration data from the line given as
 *  parameter (CurrentLine), and from the following line
 *  in the VCSMAKER.CFG file.
 *
 *---------------------------------------------------------------*/

void Retrieve_Data ( char *CurrentLine,
                     ConfigDataStruct *ConfigData,
                     FILE *vcsmaker_cfg )
{
   char *PtrLine;
   char Parameter[80];

   /*-------------------------------------*
     Begin at first character of the line.
    *-------------------------------------*/
   PtrLine = CurrentLine;

   PtrLine = Get_Next_Parameter( PtrLine, Parameter );
   strcpy( ConfigData->Cart_Image, Parameter );

   PtrLine = Get_Next_Parameter( PtrLine, Parameter );
   strcpy( ConfigData->Cart_Name, Parameter );

   /*----------------------------------------*
     The rest of the configuration parameters
     should be on the next line of data.
    *----------------------------------------*/
   if ( feof(vcsmaker_cfg) )
   {
     printf("Error: End of VCSMAKER.CFG file reached prematurely!");
     exit(1);
   }

   /*--------------------------------------*
     Skip comments and empty lines, if any.
    *--------------------------------------*/
   fgets( CurrentLine, LINESIZE, vcsmaker_cfg );
   while( !Is_Line_Of_Data(CurrentLine) )
      fgets( CurrentLine, LINESIZE, vcsmaker_cfg );

   /*--------------------------------*
     Read the rest of the parameters.
    *--------------------------------*/
   PtrLine = CurrentLine;

   PtrLine = Get_Next_Parameter( PtrLine, Parameter );
   strcpy( ConfigData->Cart_Type, Parameter );

   PtrLine = Get_Next_Parameter( PtrLine, Parameter );
   strcpy( ConfigData->Dsp_FrameRate, Parameter );

   PtrLine = Get_Next_Parameter( PtrLine, Parameter );
   strcpy( ConfigData->Dsp_YStart, Parameter );

   PtrLine = Get_Next_Parameter( PtrLine, Parameter );
   strcpy( ConfigData->Dsp_Height, Parameter );

   PtrLine = Get_Next_Parameter( PtrLine, Parameter );
   strcpy( ConfigData->Dsp_XStart, Parameter );

   PtrLine = Get_Next_Parameter( PtrLine, Parameter );
   strcpy( ConfigData->Dsp_Width, Parameter );

   PtrLine = Get_Next_Parameter( PtrLine, Parameter );
   strcpy( ConfigData->Con_LeftDiff, Parameter );

   PtrLine = Get_Next_Parameter( PtrLine, Parameter );
   strcpy( ConfigData->Con_RightDiff, Parameter );

   PtrLine = Get_Next_Parameter( PtrLine, Parameter );
   strcpy( ConfigData->Con_TVType, Parameter );

   PtrLine = Get_Next_Parameter( PtrLine, Parameter );
   strcpy( ConfigData->Ctrl_Paddle0, Parameter );

   PtrLine = Get_Next_Parameter( PtrLine, Parameter );
   strcpy( ConfigData->Ctrl_Paddle1, Parameter );

   PtrLine = Get_Next_Parameter( PtrLine, Parameter );
   strcpy( ConfigData->Ctrl_Paddle2, Parameter );

   PtrLine = Get_Next_Parameter( PtrLine, Parameter );
   strcpy( ConfigData->Ctrl_Paddle3, Parameter );

   PtrLine = Get_Next_Parameter( PtrLine, Parameter );
   strcpy( ConfigData->Ctrl_Joy0, Parameter );

   PtrLine = Get_Next_Parameter( PtrLine, Parameter );
   strcpy( ConfigData->Ctrl_Joy1, Parameter );

   PtrLine = Get_Next_Parameter( PtrLine, Parameter );
   strcpy( ConfigData->Timer_Adj, Parameter );
}


/*---------------------------------------------------------------*
 *
 *  Init_Default_VCS
 *
 *   Sets the "ConfigData" structure to default values.
 *
 *---------------------------------------------------------------*/

void Init_Default_VCS ( char *BinFilename, ConfigDataStruct *ConfigData )
{
   char BinFilename2[FILENAMESIZE];

   strcpy( BinFilename2, BinFilename );
   strcat( BinFilename2, ".BIN" );

   strcpy( ConfigData->Cart_Name, BinFilename2 );
   strcpy( ConfigData->Cart_Image, BinFilename2 );
   strcpy( ConfigData->Cart_Type, "2K" );
   strcpy( ConfigData->Dsp_FrameRate, "60" );
   strcpy( ConfigData->Dsp_YStart, "38" );
   strcpy( ConfigData->Dsp_Height, "200" );
   strcpy( ConfigData->Dsp_XStart, "0" );
   strcpy( ConfigData->Dsp_Width, "160" );
   strcpy( ConfigData->Con_LeftDiff, "B" );
   strcpy( ConfigData->Con_RightDiff, "B" );
   strcpy( ConfigData->Con_TVType, "Color" );
   strcpy( ConfigData->Ctrl_Paddle0, "None" );
   strcpy( ConfigData->Ctrl_Paddle1, "None" );
   strcpy( ConfigData->Ctrl_Paddle2, "None" );
   strcpy( ConfigData->Ctrl_Paddle3, "None" );
   strcpy( ConfigData->Ctrl_Joy0, "0" );
   strcpy( ConfigData->Ctrl_Joy1, "1" );
   strcpy( ConfigData->Timer_Adj, "0" );
}


/*---------------------------------------------------------------*
 *
 *  Get_Config_Data
 *
 *  Reads the configuration data associated to the given BIN
 *  filename in the VCSMAKER.CFG file, and stores this data in
 *  the "ConfigData" record structure. If there is no data found,
 *  default values are put in "ConfigData".
 *
 *---------------------------------------------------------------*/

void Get_Config_Data ( char *BinFilename,
                       ConfigDataStruct *ConfigData )
{
  char BinFilename2[FILENAMESIZE];
  FILE *vcsmaker_cfg;
  char CurrentLine[LINESIZE];
  int bin_not_found;

  /*----------------------------*
    Opens the VCSMAKER.CFG file.
   *----------------------------*/
  vcsmaker_cfg = fopen( "VCSMAKER.CFG", "rt" );
  if (vcsmaker_cfg == NULL)
  {
     printf("File VCSMAKER.CFG was not found in current directory.");
     exit(1);
  }

  /*--------------------------------------------*
    Looks for the data in the VCSMAKER.CFG file.
   *--------------------------------------------*/
  bin_not_found = 0;
  strcpy( BinFilename2, BinFilename );
  strcat( BinFilename2, ".bin");

  while ( !feof(vcsmaker_cfg) && (bin_not_found == 0) )
  {
     fgets( CurrentLine, LINESIZE, vcsmaker_cfg );

     if ( Is_Line_Of_Data( CurrentLine ) )
     {
        if ( strnicmp(BinFilename2, CurrentLine, strlen(BinFilename2)) == 0 )
        {
           Retrieve_Data( CurrentLine, ConfigData, vcsmaker_cfg );
           bin_not_found = 1;
        }
     }
  }

  fclose(vcsmaker_cfg);

  /*-----------------------------------------------------*
    If no data was found in the VCSMAKER.CFG file for the
    specified BIN file, then default values are used.
   *-----------------------------------------------------*/
  if (bin_not_found == 0)
  {
     printf("\nWarning: No data found in VCSMAKER.CFG, so default VCS file will be generated.");
     printf("\n         Please verify that cartridge Name, Image and Type are correct.\n");
     Init_Default_VCS( BinFilename, ConfigData );
  }
}


/*---------------------------------------------------------------*
 *
 *  Make_VCS_File
 *
 *  Create the VCS file using the data stored in the
 *  ConfigData record structure. The filename of the
 *  resulting VCS file will be the BIN filename given
 *  as parameter with its ".BIN" extension replaced
 *  with a ".VCS" extension.
 *
 *---------------------------------------------------------------*/

void Make_VCS_File ( char *BinFilename,
                     ConfigDataStruct *ConfigData )
{
   char VcsFilename[FILENAMESIZE];
   FILE *vcs_file;
   char resp;

   strcpy( VcsFilename, BinFilename );
   strcat( VcsFilename, ".VCS" );

   /*------------------------------------------------------*
     If the VCS file already exists, asks for confirmation.
    *------------------------------------------------------*/
   vcs_file = fopen( VcsFilename, "r");
   if ( vcs_file != NULL )
   {
      fclose (vcs_file );
      printf("\nVCS file already exists in current directory. Overwrite? (Y-N):");
      resp = getche();
      if ( resp != 'Y' && resp != 'y' )
      {
         printf("\nVCS file not created.\n");
         exit(1);
      }
   }

   /*-------------------*
     Opens the VCS file.
    *-------------------*/
   vcs_file = fopen( VcsFilename, "w" );
   if (vcs_file == NULL )
   {
      printf( "Could not open file '%s'.", VcsFilename );
      exit(1);
   }

   /*-----------------------------------------------*
     Outputs the configuration data to the VCS file.
    *-----------------------------------------------*/
   fprintf ( vcs_file, "Stella96\n" );
   fprintf ( vcs_file, "\n" );
   fprintf ( vcs_file, "##############################################################################\n");
   fprintf ( vcs_file, "# This file was generated using VCSMAKER.EXE version %s\n", VERSION_NUM );
   fprintf ( vcs_file, "##############################################################################\n");
   fprintf ( vcs_file, "\n" );
   fprintf ( vcs_file, "##############################################################################\n");
   fprintf ( vcs_file, "# Game information\n");
   fprintf ( vcs_file, "#\n");
   fprintf ( vcs_file, "#   Cartridge type should be: 2K, 4K, E0, E7, F6, F8, FE,\n");
   fprintf ( vcs_file, "#                             F4SC, F6SC, F8SC or FASC\n");
   fprintf ( vcs_file, "#\n");
   fprintf ( vcs_file, "#   See the Bank Switching document on the Stella 96 home page if you\n");
   fprintf ( vcs_file, "#   don't know what to put for the cartridge type\n");
   fprintf ( vcs_file, "#\n");
   fprintf ( vcs_file, "##############################################################################\n");
   fprintf ( vcs_file, "Cartridge.Name = %s\n", ConfigData->Cart_Name );
   fprintf ( vcs_file, "Cartridge.Image = %s\n", ConfigData->Cart_Image );
   fprintf ( vcs_file, "Cartridge.Type = %s\n", ConfigData->Cart_Type );
   fprintf ( vcs_file, "\n");
   fprintf ( vcs_file, "##############################################################################\n");
   fprintf ( vcs_file, "# Display information\n");
   fprintf ( vcs_file, "#\n");
   fprintf ( vcs_file, "#   Frame rate should be: 10, 15, 20, 30, or 60\n");
   fprintf ( vcs_file, "#   Height should not exceed 200\n");
   fprintf ( vcs_file, "#\n");
   fprintf ( vcs_file, "##############################################################################\n");
   fprintf ( vcs_file, "Display.FrameRate = %s\n", ConfigData->Dsp_FrameRate );
   fprintf ( vcs_file, "Display.YStart = %s\n", ConfigData->Dsp_YStart );
   fprintf ( vcs_file, "Display.Height = %s\n", ConfigData->Dsp_Height );
   fprintf ( vcs_file, "Display.XStart = %s\n", ConfigData->Dsp_XStart );
   fprintf ( vcs_file, "Display.Width = %s\n", ConfigData->Dsp_Width );
   fprintf ( vcs_file, "\n");
   fprintf ( vcs_file, "##############################################################################\n");
   fprintf ( vcs_file, "# Console default settings\n");
   fprintf ( vcs_file, "#\n");
   fprintf ( vcs_file, "#   Difficulty should be: A or B\n");
   fprintf ( vcs_file, "#   Television type should be: Color or Mono\n");
   fprintf ( vcs_file, "#\n");
   fprintf ( vcs_file, "##############################################################################\n");
   fprintf ( vcs_file, "Console.LeftDifficulty = %s\n", ConfigData->Con_LeftDiff );
   fprintf ( vcs_file, "Console.RightDifficulty = %s\n", ConfigData->Con_RightDiff );
   fprintf ( vcs_file, "Console.TelevisionType = %s\n", ConfigData->Con_TVType );
   fprintf ( vcs_file, "\n");
   fprintf ( vcs_file, "##############################################################################\n");
   fprintf ( vcs_file, "# Controller information\n");
   fprintf ( vcs_file, "#\n");
   fprintf ( vcs_file, "#   Paddles should be: None or the \"physical\" paddle to use (0, 1, 2 or 3)\n");
   fprintf ( vcs_file, "#   Joysticks should be: None or the \"physical\" joystick to use (0 or 1)\n");
   fprintf ( vcs_file, "#\n");
   fprintf ( vcs_file, "##############################################################################\n");
   if ( (strcmp (ConfigData->Ctrl_Paddle0, ConfigData->Ctrl_Paddle1) == 0)
        && (strcmp (ConfigData->Ctrl_Paddle0, "None") != 0) )
      fprintf ( vcs_file, "\n# Players alternate turns so let them use the same \"physical\" controller\n");
   fprintf ( vcs_file, "Controller.Paddle0 = %s\n", ConfigData->Ctrl_Paddle0 );
   fprintf ( vcs_file, "Controller.Paddle1 = %s\n", ConfigData->Ctrl_Paddle1 );
   fprintf ( vcs_file, "Controller.Paddle2 = %s\n", ConfigData->Ctrl_Paddle2 );
   fprintf ( vcs_file, "Controller.Paddle3 = %s\n", ConfigData->Ctrl_Paddle3 );
   fprintf ( vcs_file, "\n");
   if ( (strcmp (ConfigData->Ctrl_Joy0, ConfigData->Ctrl_Joy1) == 0)
        && (strcmp (ConfigData->Ctrl_Joy0, "None") != 0) )
      fprintf ( vcs_file, "# Players alternate turns so let them use the same \"physical\" controller\n");
   fprintf ( vcs_file, "Controller.Joystick0 = %s\n", ConfigData->Ctrl_Joy0 );
   fprintf ( vcs_file, "Controller.Joystick1 = %s\n", ConfigData->Ctrl_Joy1 );
   fprintf ( vcs_file, "\n");
   fprintf ( vcs_file, "##############################################################################\n");
   fprintf ( vcs_file, "#\n");
   fprintf ( vcs_file, "# Timer.Adjustment - allows you to fix the screen jitters in some games\n");
   fprintf ( vcs_file, "#                    it should be some integer value probably less than 64 \n");
   fprintf ( vcs_file, "#\n");
   fprintf ( vcs_file, "##############################################################################\n");
   fprintf ( vcs_file, "Timer.Adjustment = %s\n", ConfigData->Timer_Adj );

   fclose( vcs_file );

   printf("\nVCS file '%s' was created.\n", VcsFilename );
}


/*---------------------------------------------------------------*
 *
 *  Make_All_VCS_Files
 *
 *  Create all the VCS files defined in the "VCSMAKER.CFG" file.
 *
 *---------------------------------------------------------------*/

void Make_All_VCS_Files ( ConfigDataStruct *ConfigData )
{
  FILE *vcsmaker_cfg;
  char CurrentLine[LINESIZE];
  char BinFileName[FILENAMESIZE];

  /*----------------------------*
    Opens the VCSMAKER.CFG file.
   *----------------------------*/
  vcsmaker_cfg = fopen( "VCSMAKER.CFG", "rt" );
  if (vcsmaker_cfg == NULL)
  {
     printf("File VCSMAKER.CFG was not found in current directory.");
     exit(1);
  }

  /*--------------------------------------------*
    Looks for the data in the VCSMAKER.CFG file.
   *--------------------------------------------*/
  while ( !feof(vcsmaker_cfg) )
  {
     fgets( CurrentLine, LINESIZE, vcsmaker_cfg );

     if ( Is_Line_Of_Data( CurrentLine ) )
     {
        Retrieve_Data( CurrentLine, ConfigData, vcsmaker_cfg );
        strcpy( BinFileName, ConfigData->Cart_Image );
        Format_Filename( BinFileName );
        Make_VCS_File( BinFileName, ConfigData );
     }
  }

  fclose(vcsmaker_cfg);

  printf("\nAll defined VCS files have been generated.");
}
