/*****************************************************************************/
/*                                                                           */
/* Module:  SOUND                                                            */
/* Purpose: Sound Blaster DAC DMA Driver V1.1                                */
/* Author:  Ron Fries                                                        */
/* Date:    September 22, 1996                                               */
/*                                                                           */
/*****************************************************************************/
/*                                                                           */
/*                 License Information and Copyright Notice                  */
/*                 ========================================                  */
/*                                                                           */
/* PokeySound is Copyright(c) 1996 by Ron Fries                              */
/*                                                                           */
/* This library is free software; you can redistribute it and/or modify it   */
/* under the terms of version 2 of the GNU Library General Public License    */
/* as published by the Free Software Foundation.                             */
/*                                                                           */
/* This library is distributed in the hope that it will be useful, but       */
/* WITHOUT ANY WARRANTY; without even the implied warranty of                */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library */
/* General Public License for more details.                                  */
/* To obtain a copy of the GNU Library General Public License, write to the  */
/* Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.   */
/*                                                                           */
/* Any permitted reproduction of these routines, in whole or in part, must   */
/* bear this legend.                                                         */
/*                                                                           */
/*****************************************************************************/

#include <dos.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <alloc.h>
#include "pokey.h"

#define MAX_NUM_SB_BUFS   8

#define DSP_RESET         0x06
#define DSP_READ          0x0a
#define DSP_WRITE         0x0c
#define DSP_ACK           0x0e

#define DMA_BASE          0x00
#define DMA_COUNT         0x01
#define DMA_MASK          0x0a
#define DMA_MODE          0x0b
#define DMA_FF            0x0c

/* declare local global variables */
static uint8 far *Sb_buffer[MAX_NUM_SB_BUFS];
static uint16  Sb_buf_size = 1000;
static uint8 Inuse[MAX_NUM_SB_BUFS];
static uint8 Offset_low[MAX_NUM_SB_BUFS];
static uint8 Offset_high[MAX_NUM_SB_BUFS];
static uint8 Page_no[MAX_NUM_SB_BUFS];
static uint8 Count_low;
static uint8 Count_high;
static uint8 Out_ptr = 0;
static uint8 In_ptr = 0;
static uint8 Audio_active = 0;
static uint8 Sb_init = 0;
static uint8 Num_sb_bufs = 4;

static uint16 IOaddr = 0x220;
static uint16 Irq = 7;
static uint16 Dma = 1;

static void interrupt (*OldIntVect)(void);

/* function prototypes */
void interrupt newIntVect (void);
void setNewIntVect (uint16 irq);
void setOldIntVect (uint16 irq);
void dsp_out (uint16 port, uint8 val);
int hextodec (char c);
void getBlasterEnv (void);
void send_audio (uint8 buf);

/* global function prototypes */
int OpenSB(uint16 playback_freq, uint16 num_bufs, uint16 buffer_size);
void CloseSB(void);
void Fillbuffer (void);


/*****************************************************************************/
/*                                                                           */
/* Module:  setNewIntVect                                                    */
/* Purpose: To set the specified interrupt vector to the sound output        */
/*          processing interrupt.                                            */
/* Author:  Ron Fries                                                        */
/* Date:    September 10, 1996                                               */
/*                                                                           */
/*****************************************************************************/

static void setNewIntVect (uint16 irq)
{
   OldIntVect = getvect (irq + 0x08);

   setvect (irq + 0x08, newIntVect);             
}


/*****************************************************************************/
/*                                                                           */
/* Module:  setOldIntVect                                                    */
/* Purpose: To restore the original vector                                   */
/* Author:  Ron Fries                                                        */
/* Date:    September 10, 1996                                               */
/*                                                                           */
/*****************************************************************************/

static void setOldIntVect (uint16 irq)
{
   setvect (irq + 0x08, OldIntVect);
}


/*****************************************************************************/
/*                                                                           */
/* Module:  dsp_out                                                          */
/* Purpose: To send a byte to the SB's DSP                                   */
/* Author:  Ron Fries                                                        */
/* Date:    September 10, 1996                                               */
/*                                                                           */
/*****************************************************************************/

static void dsp_out(uint16 port, uint8 val)
{
  /* wait for buffer to be free */
  while(inportb(IOaddr + DSP_WRITE) & 0x80)
  {                                          
     /* do nothing */
  }

  /* transmit the next byte */
  outportb(port,val);
}


/*****************************************************************************/
/*                                                                           */
/* Module:  hextodec                                                         */
/* Purpose: Convert the input character to hex                               */
/* Author:  Ron Fries                                                        */
/* Date:    September 10, 1996                                               */
/*                                                                           */
/*****************************************************************************/

int hextodec (char c)
{
   int retval = 0;

   c = toupper (c);
   
   if ((c>='0') && (c<='9'))
   {
      retval = c - '0';
   }
   else if ((c>='A') && (c<='F'))
   {
      retval = c - 'A' + 10;
   }
   
   return (retval);
}


/*****************************************************************************/
/*                                                                           */
/* Module:  getBlasterEnv                                                    */
/* Purpose: Read the BLASTER environment variable and set the local globals  */
/* Author:  Ron Fries                                                        */
/* Date:    September 10, 1996                                               */
/*                                                                           */
/*****************************************************************************/

static void getBlasterEnv (void)
{
   char *env;
   char *ptr;
   
   env = strupr(getenv("BLASTER"));
   
   /* if the environment variable exists */
   if (env)
   {   
      /* search for the address setting */
      ptr = strchr(env, 'A');
      if (ptr)
      {
         /* if valid, read and convert the IO address */
         IOaddr = (hextodec (ptr[1]) << 8) +
                  (hextodec (ptr[2]) << 4) +
                  (hextodec (ptr[3]));
      }
   
      /* search for the IRQ setting */
      ptr = strchr(env, 'I');
      if (ptr)
      {
         /* if valid, read and convert the IRQ */
         Irq = hextodec (ptr[1]);
      }
   
      /* search for the DMA setting */
      ptr = strchr(env, 'D');
      if (ptr)
      {
         /* if valid, read and convert the DMA */
         Dma = hextodec (ptr[1]);
      }
   }
}   
   
    
/*****************************************************************************/
/*                                                                           */
/* Module:  OpenSB                                                           */
/* Purpose: To reset the SB and prepare all buffers and other global         */
/*          global variables for sound output.  Allows the user to select    */
/*          the playback frequency, number of buffers, and size of each      */
/*          buffer.  Returns a value of zero if unsuccessful.                */
/* Author:  Ron Fries                                                        */
/* Date:    September 10, 1996                                               */
/*                                                                           */
/*****************************************************************************/

int OpenSB(uint16 playback_freq, 
           uint16 num_bufs, 
           uint16 buffer_size)
{
   uint8 x;
   uint16 y;
   uint32 addr;
   
   /* initialize local globals */
   /* check for valid values - if not valid, use default */
   if ((Num_sb_bufs > MAX_NUM_SB_BUFS) ||
       (Num_sb_bufs < 2))
   {        
      Num_sb_bufs = MAX_NUM_SB_BUFS;
   }
   else
   {
      Num_sb_bufs = num_bufs;
   }
    
   if (buffer_size > 0)
   { 
      Sb_buf_size = buffer_size;
   }
   
   Out_ptr = 0;            /* output queue pointer */
   In_ptr = 0;             /* input queue pointer */
   Audio_active = 0;       /* no audio currently being output */
   
   /* assume the init was successful */
   Sb_init = 1;

   /* attempt to read the Blaster Environment Variable */
   getBlasterEnv ();
   
   /* send a DSP reset to the SB */
   outportb (IOaddr + DSP_RESET, 1);

   /* wait a few microsec */
   x = inportb(IOaddr + DSP_RESET);
   x = inportb(IOaddr + DSP_RESET);
   x = inportb(IOaddr + DSP_RESET);
   x = inportb(IOaddr + DSP_RESET);

   /* clear the DSP reset */
   outportb (IOaddr + DSP_RESET,0);

   /* wait a bit until the SB indicates good status */
   y = 0;
   do
   {
      x = inportb (IOaddr + DSP_READ);
      y++;
   }while ((y < 100) && (x != 0xaa));

   /* if we were able to successfully reset the SB */
   if (x == 0xaa)
   {
      /* indicate successful initialization */
      Sb_init = 1;
      
      /* turn on speaker */
      dsp_out (IOaddr + DSP_WRITE, 0xd1);

      /* set time constant */
      dsp_out (IOaddr + DSP_WRITE, 0x40);
      dsp_out (IOaddr + DSP_WRITE, (unsigned char)(256 - 1000000L/playback_freq));

      /* setup the DSP interrupt service routine */
      setNewIntVect(Irq);

      /* Enable the interrupt used */
      outportb(0x21,inportb(0x21) & (~(1<<Irq)));

      /* make sure interrupts are enabled */
      enable();

      for (x = 0; x < Num_sb_bufs; x++)
      {
         /* if the SB init is still valid */
         if (Sb_init == 1)
         {
            /* create a buffer to hold the data */
            Sb_buffer[x] = malloc (Sb_buf_size);
        
            /* if we were able to successfully allocate the buffer */
            if (Sb_buffer[x] != 0)
            {
               /* initialize to not in use */
               Inuse[x] = 0;

               /* pre-calculate high, low and page addresses of buffer */
               /* for better performance */
               addr = ((unsigned long)FP_SEG(Sb_buffer[x]) << 4) + 
                       (unsigned long)FP_OFF(Sb_buffer[x]);
               Offset_low[x] = (unsigned char)(addr & 0x0ff);
               Offset_high[x] = (unsigned char)((addr >> 8) & 0x0ff);
               Page_no[x] = (unsigned char)(addr >> 16);
            }
            else
            {
               /* otherwise buffer allocation was unsuccessful,
                  so close the SB */
               CloseSB();
            }
         }
      }
      
      /* pre-calculate the high/low buffer size counts for better perf */
      Count_low = (Sb_buf_size-1) & 0x0ff;
      Count_high = ((Sb_buf_size-1) >> 8) & 0x0ff;
   }
   else
   {
      /* indicate SB did not initialize successfully */
      Sb_init = 0;
   }
   
   return (Sb_init);
}


/*****************************************************************************/
/*                                                                           */
/* Module:  CloseSB                                                          */
/* Purpose: Closes the SB and disables the interrupts.                       */
/* Author:  Ron Fries                                                        */
/* Date:    September 10, 1996                                               */
/*                                                                           */
/*****************************************************************************/

void CloseSB(void)
{
   uint8 x;

   /* if the SB was initialized */
   if (Sb_init)
   {
      /* indicate SB no longer active */
      Sb_init = 0;

      /* halt DMA */
      outportb (IOaddr + DSP_WRITE, 0xd0);

      /* turn the speaker off */
      outportb (IOaddr + DSP_WRITE, 0xd3);

      /* Disable the interrupt used */
      outportb(0x21,inportb(0x21) | (1<<Irq));

      /* restore the original interrupt routine */
      setOldIntVect(Irq);

      /* free any memory that had been allocated */
      for (x=0; x < Num_sb_bufs; x++)
      {
         if (Sb_buffer[x] != 0)
         { 
            free (Sb_buffer[x]);
         }
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/* Module:  send_audio                                                       */
/* Purpose: Sends an audio buffer to the SB and starts the output.           */
/* Author:  Ron Fries                                                        */
/* Date:    September 10, 1996                                               */
/*                                                                           */
/*****************************************************************************/

static void send_audio (uint8 buf)
{
   static uint8 pagePort[8] = { 0x87, 0x83, 0x81, 0x82 };

   /* if the SB initialized properly */
   if (Sb_init)
   {
      /* program the DMAC for output transfer */
      outportb (DMA_MASK              , 0x04 | Dma );
      outportb (DMA_FF                , 0 );
      outportb (DMA_MODE              , 0x48 | Dma );
      outportb (DMA_BASE + (Dma << 1) , Offset_low[buf] );
      outportb (DMA_BASE + (Dma << 1) , Offset_high[buf] );
      outportb (pagePort[Dma]         , Page_no[buf] );
      outportb (DMA_COUNT + (Dma << 1), Count_low );
      outportb (DMA_COUNT + (Dma << 1), Count_high );
      outportb (DMA_MASK              , Dma );

      /* indicate audio output is currently active */
      Audio_active = 1;

      /* start the output */
      dsp_out (IOaddr + DSP_WRITE, 0x14);
      dsp_out (IOaddr + DSP_WRITE, Count_low);
      dsp_out (IOaddr + DSP_WRITE, Count_high);
   }
}


/*****************************************************************************/
/*                                                                           */
/* Module:  Fillbuffer                                                       */
/* Purpose: Checks to see if the next buffer in the queue is empty and       */
/*          fills the buffer.  If the SB is not currently transmitting       */
/*          audio, it will send the first buffer to get the IRQ started.     */
/* Author:  Ron Fries                                                        */
/* Date:    September 10, 1996                                               */
/*                                                                           */
/*****************************************************************************/

void Fillbuffer (void)
{
   /* if the SB initialized properly */
   if (Sb_init)
   {
      /* if buffer is not inuse */
      if (Inuse[In_ptr] == 0)
      {
         /* fill the buffer with data */
         Pokey_process (Sb_buffer[In_ptr],Sb_buf_size);

         /* buffer is ready to xmit */
         Inuse[In_ptr] = 1;

         /* if we're currently not transmitting audio */
         if (!Audio_active)
         {
            /* then set output pointer to current buffer */
            Out_ptr = In_ptr;
            
            /* and force the first buffer out */
            /* (next buffer sent automatically on interrupt) */
            send_audio (Out_ptr);
         }
         
         /* shift pointer to next buffer in queue */
         In_ptr = (In_ptr + 1) % Num_sb_bufs;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/* Module:  newIntVect                                                       */
/* Purpose: The interrupt vector to handle the DAC DMAC completed interrupt  */
/*          Indicates the current buffer is complete and transmits the next  */
/*          buffer in the queue if filled.  If not, it indicates all sound   */
/*          output is stopped.                                               */
/* Author:  Ron Fries                                                        */
/* Date:    September 10, 1996                                               */
/*                                                                           */
/*****************************************************************************/

static void interrupt newIntVect (void)
{
   /* acknowledge the DSP interrupt */
   inportb (IOaddr + DSP_ACK);

   /* indicate buffer is free */
   Inuse[Out_ptr] = 0;

   /* move pointer to next buf in queue */
   Out_ptr++;
   if (Out_ptr == Num_sb_bufs)
   {
      Out_ptr = 0;
   }

   /* if the buffer is ready to xmit */
   if (Inuse[Out_ptr])
   {
      /* then send the buffer out */
      send_audio (Out_ptr);
   }
   else
   {
      /* indicate audio output currently not active */
      Audio_active = 0;
   }

   /* indicate end of interrupt  */
   outportb (0x20, 0x20);  
}
