/*
   Decoder for Scott Adams style datafiles in TRS-80 ASCII format

   Release 6 / 961228
   Release 7 / 980115

   (c) 1993,96,98 by Paul David Doherty <h0142kdd@rz.hu-berlin.de>

   Mucho thanks go to Alan Cox, Tom Phelps, Dave Rees, Robert R. Schneck,
   Terry Stewart, Andy Tepper, Adam Thornton and Jim Veneskey for their
   valuable help
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

void Fatal (char *);
void PrintVerb (int);
void PrintNoun (int);
void PrintRoom (int);
void PrintItem (int);
void PrintMessage (int);
void *MemAlloc (int);
char *ReadString (void);
void LoadDatabase (void);

typedef struct
  {
    short Magic;
    short ItemList;
    short CodeList;
    short NounVerbList;
    short RoomList;
    short MaxCarry;
    short ActorRoom;
    short TotalTreasures;
    short WordLength;
    short LightTime;
    short MessageList;
    short TreasureRoom;
  }
Header;

typedef struct
  {
    unsigned short Vocab;
    unsigned short Condition[5];
    unsigned short Action[2];
  }
Action;

typedef struct
  {
    char *Text;
    short Exits[6];
  }
Room;

typedef struct
  {
    char *Text;
    unsigned char Location;
    char *AutoGet;
  }
Item;

typedef struct
  {
    short Version;
    short AdventureNumber;
    short Magic;
  }
Tail;

Header GameHeader;
Tail GameTail;
Item *Items;
Room *Rooms;
char **Verbs;
char **Nouns;
char **Messages;
char **Comments;
Action *Actions;
int LightRefill;
char NounText[16];

static char *conditions[] =
{
  "", "HAS", "IS_in_AR", "IS_AVAIL",
  "PLAYER_IN", "IS_NOT_in_AR", "HAS_NOT", "PLAYER_NOT_IN",
  "SET_BIT", "CLEARED_BIT", "HAS_SOMETHING", "HAS_NOTHING",
  "IS_NOT_AVAIL", "IS_NOT_in_ROOM0", "IS_in_ROOM0", "COUNTER <=",
  "COUNTER >", "IS_in_OrigROOM", "IS_NOT_in_OrigROOM", "COUNTER =="
};

#define NONE 1
#define ACTION 2
#define ITEM 3
#define ROOM 4
#define NUMBER 5
#define BITFLAG 6

static char conddata[] =
{
  ACTION, ITEM, ITEM, ITEM, ROOM, ITEM, ITEM, ROOM,
  BITFLAG, BITFLAG, NONE, NONE, ITEM, ITEM, ITEM, NUMBER,
  NUMBER, ITEM, ITEM, NUMBER
};

static char *acts[] =
{
  "GET", "MOVE_INTO_AR", "GOTO", "REMOVE", "SET_NIGHT",
  "SET_DAY", "SET_BIT", "REMOVE", "CLEAR_BIT", "KILL_PLAYER",
  "MOVE_X_INTO_Y", "QUIT", "LOOK", "SCORE", "INVENTORY",
  "SET_BIT (0)", "CLEAR_BIT (0)", "FILL_LAMP", "CLS", "SAVE",
  "SWAP_ITEMS", "CONTINUE:", "GET_ALWAYS", "PUT_X_WITH_Y", "LOOK",
  "COUNTER -= 1", "PRINT_COUNTER", "COUNTER =", "SWAP_LOC_RV", "SWAP_COUNTER",
  "COUNTER +=", "COUNTER -=", "ECHO_NOUN", "ECHO_NOUN_CR", "CR",
  "SELECT_RV", "DELAY", "SHOW_PIC", "COMM90", "COMM91",
  "COMM92", "COMM93", "COMM94", "COMM95", "COMM96",
  "COMM97", "COMM98", "COMM99", "COMM100", "COMM101"
};

#define NOT 0
#define IT 1
#define RO 2
#define FL 3
#define IT_RO 4
#define IT_IT 5
#define NUM 6

static char actdata[] =
{
  IT, IT, RO, IT, NOT, NOT, FL, IT, FL, NOT,
  IT_RO, NOT, NOT, NOT, NOT, NOT, NOT, NOT, NOT, NOT,
  IT_IT, NOT, IT, IT_IT, NOT, NOT, NOT, NUM, NOT, FL,
  NUM, NUM, NOT, NOT, NOT, FL, NOT, FL, NOT, NOT,
  NOT, NOT, NOT, NOT, NOT, NOT, NOT, NOT, NOT, NOT
};

static char *direct[] =
{
  "north", "south", "east", "west", "up", "down"
};

char zero = 0;

void
Fatal (char *x)
{
  fprintf (stderr, "%s.\n", x);
  exit (1);
}

void
PrintVerb (int verb)
{
  printf ("%s", (verb == 0) ? "AUTO" : (verb > GameHeader.NounVerbList) ? "{V>MAX}" : Verbs[verb]);
}

void
PrintNoun (int noun)
{
  printf ("%s", (noun == 0) ? "ANY" : (noun > GameHeader.NounVerbList) ? "{N>MAX}" : Nouns[noun]);
}

void
PrintRoom (int room)
{
  printf ("%s", (room > GameHeader.RoomList) ? "{R>MAX}" : Rooms[room].Text);
}

void
PrintItem (int item)
{
  printf ("%s", (item > GameHeader.ItemList) ? "{I>MAX}" : Items[item].Text);
}

void
PrintMessage (int message)
{
  printf ("%s", (message > GameHeader.MessageList) ? "{M>MAX}" : Messages[message]);
}

void *
MemAlloc (int size)
{
  void *t = (void *) malloc (size);
  if (t == NULL)
    Fatal ("Out of memory");
  return (t);
}

char *
ReadString (void)
{
  char tmp[1024];
  char *t;
  int c, nc;
  int ct = 0;
  do
    {
      c = getchar ();
    }
  while (c != EOF && isspace (c));
  if (c != '"')
    Fatal ("Initial quote expected");
  do
    {
      c = getchar ();
      if (c == EOF)
	Fatal ("EOF in string");
      if (c == '"')
	{
	  nc = getchar ();
	  if (nc != '"')
	    {
	      ungetc (nc, stdin);
	      break;
	    }
	}
      if (c == '`')
	c = '"';
      if (c == '\n')
	c = '~';
      tmp[ct++] = c;
    }
  while (1);
  tmp[ct] = 0;
  t = MemAlloc (ct + 1);
  memcpy (t, tmp, ct + 1);
  return (t);
}

void
LoadDatabase (void)
{
  int nm, ni, na, nw, nr, mc, pr, tr, wl, lt, mn, trm;
  int ct;
  short lo;
  Action *ap;
  Room *rp;
  Item *ip;

  if (scanf ("%d %d %d %d %d %d %d %d %d %d %d %d",
    &nm, &ni, &na, &nw, &nr, &mc, &pr, &tr, &wl, &lt, &mn, &trm, &ct) != 12)
    Fatal ("Invalid database(bad header)");
  GameHeader.Magic = nm;
  GameHeader.ItemList = ni;
  Items = (Item *) MemAlloc (sizeof (Item) * (ni + 1));
  GameHeader.CodeList = na;
  Actions = (Action *) MemAlloc (sizeof (Action) * (na + 1));
  Comments = (char **) MemAlloc (sizeof (char *) * (na + 1));
  GameHeader.NounVerbList = nw;
  GameHeader.WordLength = wl;
  Verbs = (char **) MemAlloc (sizeof (char *) * (nw + 1));
  Nouns = (char **) MemAlloc (sizeof (char *) * (nw + 1));
  GameHeader.RoomList = nr;
  Rooms = (Room *) MemAlloc (sizeof (Room) * (nr + 1));
  GameHeader.MaxCarry = mc;
  GameHeader.ActorRoom = pr;
  GameHeader.TotalTreasures = tr;
  GameHeader.LightTime = lt;
  GameHeader.MessageList = mn;
  Messages = (char **) MemAlloc (sizeof (char *) * (mn + 1));
  GameHeader.TreasureRoom = trm;

  ct = 0;
  ap = Actions;
  while (ct < na + 1)
    {
      if (scanf ("%hd %hd %hd %hd %hd %hd %hd %hd",
		 &ap->Vocab,
		 &ap->Condition[0],
		 &ap->Condition[1],
		 &ap->Condition[2],
		 &ap->Condition[3],
		 &ap->Condition[4],
		 &ap->Action[0],
		 &ap->Action[1]) != 8)
	{
	  printf ("Bad action line (%d)\n", ct);
	  exit (1);
	}
      ap++;
      ct++;
    }

  ct = 0;
  while (ct < nw + 1)
    {
      Verbs[ct] = ReadString ();
      Nouns[ct] = ReadString ();
      ct++;
    }

  ct = 0;
  rp = Rooms;
  while (ct < nr + 1)
    {
      scanf ("%hd %hd %hd %hd %hd %hd",
	     &rp->Exits[0], &rp->Exits[1], &rp->Exits[2],
	     &rp->Exits[3], &rp->Exits[4], &rp->Exits[5]);
      rp->Text = ReadString ();
      ct++;
      rp++;
    }

  ct = 0;
  while (ct < mn + 1)
    {
      Messages[ct] = ReadString ();
      ct++;
    }

  ct = 0;
  ip = Items;
  while (ct < ni + 1)
    {
      ip->Text = ReadString ();
      ip->AutoGet = strchr (ip->Text, '/');
      if (ip->AutoGet && strcmp (ip->AutoGet, "//"))
	{
	  char *t;
	  *ip->AutoGet++ = 0;
	  t = strchr (ip->AutoGet, '/');
	  if (t != NULL)
	    *t = 0;
	}
      scanf ("%hd", &lo);
      ip->Location = (unsigned char) lo;
      ip++;
      ct++;
    }

  ct = 0;
  while (ct < na + 1)
    {
      Comments[ct] = ReadString ();
      ct++;
    }

  if (scanf ("%d %d %d", &nm, &ni, &na) != 3)
    Fatal ("Invalid database(bad tail)");
  GameTail.Version = nm;
  GameTail.AdventureNumber = ni;
  GameTail.Magic = na;
}

void
main (void)
{
  int i, j;
  int verb, noun, cond, value, act[4], par[5], pra, parcount;
  int cont = 0;

  LoadDatabase ();

/* print globals */

  printf ("ADVENTURE %d\n", GameTail.AdventureNumber);
  printf ("Version %d.%02d\n", GameTail.Version / 100, GameTail.Version % 100);
  printf ("Magic: %d, %d\n\n", GameHeader.Magic, GameTail.Magic);

  printf ("Actions: %d\n", GameHeader.CodeList);
  printf ("Words: 2 * %d (length: %d)\n", GameHeader.NounVerbList, GameHeader.WordLength);
  printf ("Rooms: %d\n", GameHeader.RoomList);
  printf ("Messages: %d\n", GameHeader.MessageList);
  printf ("Items: %d\n\n", GameHeader.ItemList);

  printf ("TGoal: store %d treasures in room %d\n", GameHeader.TotalTreasures, GameHeader.TreasureRoom);
  printf ("MaxLoad: %d items\n", GameHeader.MaxCarry);
  printf ("Current Room: %d\n", GameHeader.ActorRoom);
  if (GameHeader.LightTime != -1)
    printf ("Light runs out in %d turns\n\n", GameHeader.LightTime);
  else
    printf ("Light never runs out\n\n");

/* print action table */

  printf ("ACTIONS:\n");
  for (i = 0; i <= GameHeader.CodeList; i++)
    {
      if (Actions[i].Vocab != 0)
	cont = 0;
      if (cont == 0)
	printf ("\n");
      printf ("%3d) ", i);

      verb = Actions[i].Vocab / 150;
      noun = Actions[i].Vocab - 150 * verb;
      if (Actions[i].Vocab != 0)
	printf ("%3d %3d [", verb, noun);

      if (cont == 1)
	printf ("  [Cont'd");
      else
	{
	  if (verb == 0)
	    {
	      if (noun != 0)
		printf ("Auto %d%%", noun);
	      else
		printf ("  [");
	    }
	  else
	    {
	      PrintVerb (verb);
	      printf (" ");
	      PrintNoun (noun);
	    }
	}
      printf ("]%s%s%s\n", (strlen (Comments[i]) == 0) ? "" : "   /* ", Comments[i], (strlen (Comments[i]) == 0) ? "" : " */");

      for (j = 0; j < 4; j++)
	par[j] = 0;
      parcount = 0;

      for (j = 0; j < 5; j++)
	{
	  if (Actions[i].Condition[j] != 0)
	    {
	      value = Actions[i].Condition[j] / 20;
	      cond = Actions[i].Condition[j] - 20 * value;
	      if (cond != 0)
		{
		  printf ("     ? %s", conditions[cond]);
		  if (conddata[cond] != NONE)
		    {
		      printf (" %s%d%s", (conddata[cond] == NUMBER) ? "" : "(", value, ((conddata[cond] == ITEM) || (conddata[cond] == ROOM)) ? " = " : "");
		      if ((conddata[cond] == ITEM))
			PrintItem (value);
		      if ((conddata[cond] == ROOM))
			PrintRoom (value);
		      printf ("%s", (conddata[cond] == NUMBER) ? "" : ")");
		    }
		  printf ("\n");
		}
	      else
		par[parcount++] = value;
	    }
	  else
	    par[parcount++] = 0;
	}

      parcount = 0;
      for (j = 0; j <= 1; j++)
	{
	  act[2 * j] = Actions[i].Action[j] / 150;
	  act[2 * j + 1] = Actions[i].Action[j] - 150 * act[2 * j];
	}
      for (j = 0; j <= 3; j++)
	{
	  if (act[j] != 0)
	    {
	      printf ("    -> %d = ", act[j]);
	      if ((act[j] < 52) || (act[j] > 101))
		{
		  printf ("PRINT(");
		  PrintMessage ((act[j] < 52) ? act[j] : (act[j] - 50));
		  printf (")");
		}
	      else
		{
		  printf ("%s", acts[act[j] - 52]);

		  if (act[j] == 73)
		    cont = 1;

		  if (actdata[act[j] - 52] != NOT)
		    {
		      pra = par[parcount++];
		      printf (" %s%d", (actdata[act[j] - 52] == NUM) ? "" : "(", pra);

		      if ((actdata[act[j] - 52] == IT) || (actdata[act[j] - 52] == IT_RO) || (actdata[act[j] - 52] == IT_IT))
			{
			  printf (" = ");
			  PrintItem (pra);
			}
		      if (actdata[act[j] - 52] == RO)
			{
			  printf (" = ");
			  PrintRoom (pra);
			}
		      if ((actdata[act[j] - 52] == IT_IT) || (actdata[act[j] - 52] == IT_RO))
			{
			  pra = par[parcount++];
			  printf (") (%d = ", pra);
			  if (actdata[act[j] - 52] == IT_IT)
			    PrintItem (pra);
			  if (actdata[act[j] - 52] == IT_RO)
			    PrintRoom (pra);
			}
		      printf ("%s", (actdata[act[j] - 52] == NUM) ? "" : ")");
		    }
		}
	      printf ("\n");
	    }
	}
    }

/* print vocabulary */

  printf ("\nVERBS:\n\n");

  for (i = 0; i <= GameHeader.NounVerbList; i++)
    printf ("%3d) %6s%s", i, Verbs[i], (5 * ((i + 1) / 5) == (i + 1)) ? "\n" : "   ");
  printf ("\n");

  printf ("\nNOUNS:\n\n");

  for (i = 0; i <= GameHeader.NounVerbList; i++)
    printf ("%3d) %6s%s", i, Nouns[i], (5 * ((i + 1) / 5) == (i + 1)) ? "\n" : "   ");
  printf ("\n");

/* print rooms */

  printf ("\nROOMS:\n");

  for (i = 0; i <= GameHeader.RoomList; i++)
    {
      printf ("\n%2d)   %s\n      Items: ", i, Rooms[i].Text);
      for (j = 0; j <= GameHeader.ItemList; j++)
	if (Items[j].Location == i)
	  printf ("%d ", j);
      printf ("\n      GoTo:  ");
      for (j = 0; j < 6; j++)
	if (Rooms[i].Exits[j] != 0)
	  printf ("%s->%d  ", direct[j], Rooms[i].Exits[j]);
      printf ("\n");
    }

/* print messages */

  printf ("\nMESSAGES:\n\n");

  for (i = 0; i <= GameHeader.MessageList; i++)
    printf ("%2d)   %s\n", i, Messages[i]);

/* print items */

  printf ("\nITEMS:\n\n");

  for (i = 0; i <= GameHeader.ItemList; i++)
    printf ("%2d) %3d  %5s  %s\n", i, Items[i].Location, (&Items[i].AutoGet[0] == 0) ? "     " : Items[i].AutoGet, Items[i].Text);
}
