//******************************************************************************
// NODE NAME : EditSector.cpp
// part of the Atari Rom File Designer source files
// for the Edit and Modify Sector Window
//------------------------------------------------------------------------------
// Written by Philippe VUILLERME (c) PVBest 2001-2002
//
/*---------------------**| ARFD SOURCE REVISION HISTORY |**---------------------
Date:      | What:
-----------|--------------------------------------------------------------------
 JUNE 2002 | Clean updated source code for ARFD revision 1.9.0 (public realease)
------------------------------------------------------------------------------*/

	#define STRICT
	#include <windows.h>
	#include <windowsx.h>
	#include <commctrl.h>
   #pragma hdrstop
	#include <stdio.h>  //for sscanf
   #include "MesTypes.h"
   #include "ARFDFunctions.h"
	#include "EditSector.h"
   #include "Resource.h"
   #include "rtffuncs.h"
   #include "AmDialog.h"
   #include "monitor.h"

// variables externe
   extern HINSTANCE 	hInstance;
   extern HWND 		hChildEditWindow;
   extern HWND 		hMainWindow;
   extern HWND       hMainRText;
   extern UWORD 		SectSize;
   //extern BOOL 		bChildEditWin;
   extern UBYTE 		* lpFileBuffer;
   extern UWORD  		iCurSect;
   extern UWORD  		iCurOff;
   extern UINT   		iCurFilePos;
   extern UBYTE 		* lpRom;
   extern BOOL 		bFileModified;
   extern char 		szEnterSectTitle [32];
   extern char 		szAppPathDir[STRSIZE];
   extern char 		szFileFilter[128];
   extern char 		szChildEditWindowTitle[];
   extern char 		szEditAppName[];
   extern char       szDlgBoxMain[];
   extern HFONT      hfARFD;
   extern TMON_PARAM AmParam;
   extern HANDLE     hFile;

// variables pour les recherches ASCII/HEXA
   extern BOOL   bSearchInProgress;     // ???  recherche en cours
   extern int    nSearchDataBytes;      //selected data length
   extern int    iSearchMode;           //Ascii or Hex search mode

// fonction for Class EditSector (fonctions locales)
	#define EditSector_DefProc DefWindowProc
	BOOL EditSector_OnCreate(HWND hwnd, CREATESTRUCT FAR* lpCreateStruct);
   //void EditSector_OnDestroy(HWND hwnd);
	void EditSector_OnClose(HWND hwnd);
	void EditSector_OnPaint(HWND hwnd);
	void EditSector_OnSize(HWND hwnd, UINT state, int cx, int cy);
	void EditSector_OnSetFocus(HWND hwnd, HWND hwndOldFocus);
	void EditSector_OnKillFocus(HWND hwnd, HWND hwndNewFocus);
   void EditSector_OnCommand(HWND hwnd, int id, HWND hwndCtl, UINT codeNotify);
	void EditSector_OnChar(HWND hwnd, UINT ch, int cRepeat);
	void EditSector_OnKeyDown(HWND hwnd, UINT vk, BOOL fDown, int cRepeat, UINT flags);
	void EditSector_OnLButttonDown(HWND hwnd, BOOL fDoubleClick, int x, int y, UINT keyFlags);
   void EditSector_OnRButttonDown(HWND hwnd, BOOL fDoubleClick, int x, int y, UINT keyFlags);
   void EditSector_OnGetMinMaxInfo(HWND hwnd, LPMINMAXINFO lpMinMaxInfo);

//fonctions de ARFDesigner.CPP
   extern void Not_Yet_Available(HWND hwnd);
   extern Add_EndRichText();

// variables locales globales

   HWND hEditDisplay;            // HWND of the main EditSector window
   HWND hCommandes1, hCommandes2;// HWND of the Toolbal and bottom bar (dialogbox)
   HWND hEditStatus;             // HWND of the Status Bar
   HFONT hTheEditFont;           // The font for edition
   HFONT hTheAsciiFont;
   RECT rc;                   	// output rectangle for DrawText
   SIZE sz;                   	// string dimensions
   HDC hdc;                   	// handle of device context
   TEXTMETRIC tm;             	// structure for text metrics
   static DWORD dwCharX;      	// average width of characters in pixels
   static DWORD dwCharY;      	// height of characters in pixels
	static DWORD dwEditX;      	// X Origine of Edit area in pixels
	static DWORD dwEditY;      	// Y Origine of Edit area in pixels
   static DWORD dwMaxEditX;      // width of Edit area in pixels
	static DWORD dwMaxEditY;      // height of Edit area in pixels
   static DWORD dwZoneY;         // Height of the 2 dialog boxes + status bar
   static DWORD dwZoneX;
   static int nLineLen;    	   // line length (n char) (hexa dumping)
   static int nMaxLines;   	   // text lines in client area (n char)
   static int nZeroCaretX;       // horizontal origin of caret (n char)
   static int nZeroCaretY;       // vertical origin of caret (n char)
   static int nCaretPosX; 	      // horizontal position of caret (n char)
   static int nCaretPosY; 	      // vertical position of caret (n char)
   static DWORD dwCPX;           // horizontal position of caret in pixels
   static DWORD dwCPY;           // horizontal position of caret in pixels
   static int nAsciiGap;         // Gap between Hex and Ascii table (n char)
	static int iNumOrder;         // for Hex Modification

   UWORD EditSectorSize;  //size of the current sector in edition
   BOOL  bOKFont = FALSE; //TRUE = Une police Atari est dja en mmoire
   UINT  TheVK;           //On_KeyDown : the VK code of the input keyboard char (StatusBar)
   UINT  TheCH;           //On_Char : the Char code of the input keyboard char (StatusBar)
	UINT  iDIndex;         //The current position within the sector [0 to edit sector size]
   UBYTE uOffset;         //The current Offset position index displayed on the window
   UINT  iMemAdd;         //The current Offset position index displayed on the window
   UINT  iESPos;          //Position of the first byte of the edited sector in memory
   short nPressKey;       // Set to TRUE in OnKeydown if CONTROL is pressed (for OnChar)
   UWORD uFileNextSector;
   static char szFontFile[STRSIZE]; //String of the Atari Font File name

	UBYTE lpData[256];      //Previous Data buffer (for restore/undo) (1 sector)
   char  lpModData[256];   //Data Modified chart (1 sector) (Negative value = byte is modified)
   UBYTE lpDataCB[256];    //Local Data clipboard.
   int   nSelBytes;        //Number of bytes copied into the local data clipboard

   char szSTR[256];        //temporary string
   char * lpszEditScreen;  //buffer for the text display screen (Sector display text)

   static BOOL bEditAscii;     //Modification ASCII, sinon Hexa
   static BOOL bReplace;       //Remplacement, sinon Insertion
   static BOOL bIbmFont;       //TRUE = Police OEM IBM
   static BOOL bAtariFont;     //TRUE = Police Atari
   static BOOL bMemAddIsPos;   //TRUE = Affichage du pointeur mmoire pour iMemAdd
   BOOL bDataSelected;         //TRUE = Il y a une selection en cours
   BOOL bEditLimited;          //TRUE = Limitation de l'insertion au secteur en cours

   UINT nSelStart, nSelStop; //Plage de selection en cours (pas de selection si nSelStart == nSelStop)
   COLORREF rgbScrnColor   = RGB(255, 255, 255);  //White
   COLORREF rgbUndrlnColor = RGB(255, 191, 127);  //Light Orange
   COLORREF rgbModColor    = RGB(0, 0, 255);      //Dark Blue
   COLORREF rgbTextColor   = RGB(0, 0, 0);        //Black

   LOGFONT * lpMyLogFont;

   LONG ptMaxY,ptMaxX;

//============================================================================//
// CODE INITIALIZATION
//============================================================================//
//----------------------------------------------------------------------------//
// The EditSector Procedures
//----------------------------------------------------------------------------//

// mise en mmoire d'un nouveau secteur  editer
void Load_Sector_To_Edit(void)
{
		int i;
      iDIndex = iCurOff;
      iNumOrder = 1;
      memset(lpData,0,256);
      EditSectorSize = Size_Of_Sector(iCurSect);
      iESPos = Get_FilePosition(iCurSect,0);
      iCurFilePos = iESPos;
      if (bMemAddIsPos)
      	iMemAdd = iCurFilePos;
      //copie du buffer de donnes dans celui des donnes pour restauration (Restore/undo)
      for (i=0;i<=EditSectorSize;i++)
      	lpData[i] = lpFileBuffer[iESPos+i];
      //reset du buffer 'lpModData' (tableaux des octets modifis) pas modifi = 0:
      memset(lpModData,0,256);
}
//Restore du secteur en mmoire
// if bRTZ = TRUE -> the whole sector is fill with zero (clear sector)
// else it's just a restore to previous sector data.
void Reset_Edited_Sector(BOOL bRTZ)
{
		int i;
      if (bRTZ)
      {
      	for (i=0;i<=EditSectorSize;i++)
      		lpFileBuffer[iESPos+i] = 0;
         memset(lpModData,-1,256);
         bFileModified = TRUE;
      }
      else
      {
         //copie du buffer de donnes dans celui des donnes pour restauration (undo)
      	for (i=0;i<=EditSectorSize;i++)
      		lpFileBuffer[iESPos+i] = lpData[i];
         //reset du buffer 'lpModData' (tableaux des octets modifis) pas modifi = 0:
         memset(lpModData,0,256);
      }
}
// test du tableau 'lpModData' pour voir si une valeur a t modifie (si a < 0)
BOOL Is_Sect_Modify()
{
		int i,a;
      a = 0;
      for (i = 0; i < 256; i ++)
      {
         a += lpModData[i];
      }
      if (a == 0) return FALSE;
      else return TRUE;
}

// Affiche le caractre Atari 'cnbr'  l'cran  la bonne position index par 'ci'
void Show_Atari_Char(HDC TheDC, UBYTE cnbr, int ci, COLORREF cbkgnd)
{
   	int a,h,m,k,d,j,e;
      COLORREF cfr,cpix;
      BOOL inv = FALSE;
		if (lpModData[ci] < 0)
			cfr = rgbModColor;
      else cfr = rgbTextColor;

		if (cnbr > 127) inv = TRUE;
		cnbr &=0x7F;
		//m = 0x0c00 + (cnbr * 8);
      m = cnbr * 8;
      a = ((ci & 0x0F)+ nLineLen + nAsciiGap) * dwCharX;
		//a = (nLineLen + nAsciiGap) * dwCharX + (ci & 0x0F) * dwCharX * 2; // ZOOM
		h = (nZeroCaretY + (ci & 0xF0)/16) * dwCharY;
      for (j=0;j<8;j++)  // 1 ligne vide audessus et audessous
      {
      	SetPixelV(TheDC, (a+j), h, cbkgnd);
         SetPixelV(TheDC, (a+j), (h+1), cbkgnd);
         SetPixelV(TheDC, (a+j), (h+10), cbkgnd);
         SetPixelV(TheDC, (a+j), (h+11), cbkgnd);
      }
      h += 2;
      for (k=0;k<8;k++)  // les 8 niveaux d'un char (8 octets pour 8 lignes)
      {
      	d = lpRom[m+k];
         for (j=0;j<8;j++)   // les 8 bits
         {
         	e = (d >> (7-j)) & 0x01 ;
            if (e^inv)
            	cpix = cfr;
            else cpix = cbkgnd;
            SetPixelV(TheDC, (a+j), (h+k), cpix);
         }
      }
}

//Lecture d'une police Atari/Reading an Atari font into ARFD memory (lpRom):
//- bLoad = TRUE for reading an Atari font file
//- bLoad = FALSE to load the Atari font data from ARFD resources
//- it returns: TRUE if OK, FALSE if font not loaded into memory
BOOL Read_Rom_Font(HWND hwnd, BOOL bLoad)
{
		//HANDLE hFile;
      int fs;
   	DWORD dwTmp;
      BOOL bResult;
      char AFileName[128];
      char szDialogTitle[] = "Load an Atari Font Data File :";

      HGLOBAL hResLoad;     // handle to loaded resource
      HRSRC hRes;         // handle/ptr. to res. info. in hInstance
      LPVOID lpResLock;    // pointer to resource data

      if (!bLoad)
      {
         // Locate the Atari Font data resource in the .EXE file.
			hRes = FindResource(hInstance, "MyAtariFont", RT_RCDATA);
         if (hRes == NULL) {
    		MessageBox(hwnd, "ARFD Sys Error:\nCould not locate Atari Font resource data!", szDialogTitle, MB_OK|MB_ICONERROR);
         return FALSE;
         }
         // Load the Atari Font data into global memory.
			hResLoad = LoadResource(hInstance, hRes);
			if (hResLoad == NULL) {
    		MessageBox(hwnd, "ARFD Sys Error:\nCould not load Atari Font resource data!", szDialogTitle, MB_OK|MB_ICONERROR);
         return FALSE;
			}
         // Lock the Atari Font data into global memory.
			lpResLock = LockResource(hResLoad);
			if (lpResLock == NULL) {
    		MessageBox(hwnd, "ARFD Sys Error:\nCould not lock Atari Font data pointer!", szDialogTitle, MB_OK|MB_ICONERROR);
         return FALSE;
			}
         // Verify the length of the Atari Font data resource.
         if (SizeofResource(hInstance, hRes) != 1024) {
         MessageBox(hwnd, "ARFD Sys Error:\nWrong Atari Font data resource length!", szDialogTitle, MB_OK|MB_ICONERROR);
         return FALSE;
			}
         //Copy the Atari Font data resource in memory.
         CopyMemory((PVOID) lpRom, lpResLock, 1024);
         return TRUE;
      }
      else
      {
   		strcpy(AFileName,"");
         //strcpy(szFontFile,"");
   		//strcpy(szFontFile,szAppPathDir);
         //strcat(szFontFile,"\\*.*");
         memset(szFileFilter,0x00,128);
   		LoadString(hInstance,STRG_FILEFILTER_FON,szFileFilter,128);

         //Get the font filename to open (read)
   		bResult = Get_FileName(hwnd, (LPSTR) szFontFile,
   				szFileFilter, (LPSTR) AFileName,
   				szDialogTitle,
               TRUE,  //Open for read
               FALSE, //No need to confirm file overwritting since it's a read
               1);    //Filter index is 1 (default *.*)

         if (!bResult)
         {
            MessageBox(hwnd, "ARFD file opening Info:\n No valid Atari font file open!",
             	szDialogTitle, MB_OK|MB_ICONINFORMATION);
         	return FALSE;
         }

      	if (!Create_File(hwnd, szFontFile, szDialogTitle,'R',&hFile)) return FALSE;

      	fs = GetFileSize(hFile,NULL);
    		if (fs > 1024)
    		{
        		Close_File(&hFile);
        		MessageBox(hwnd, "ARFD file opening Error:\n The Atari font file size is more than $400 (1024) bytes!",
            	szDialogTitle, MB_OK|MB_ICONERROR);
        		return FALSE;
      	}

    		bResult = ReadFile(hFile, (LPVOID) lpRom, 1024, //16*1024 pour la ROM
       		(LPDWORD) &dwTmp, (LPOVERLAPPED) NULL);

         if (!bResult)
         {
            	Handle_System_Error(hwnd, szDialogTitle, "while reading Atari font file", GetLastError());;
               Close_File(&hFile);
               return FALSE;
         }
  			wsprintf(szSTR,"ARFD Info:\nAtari font file %s loaded!",szFontFile);
         MessageBox(hwnd, szSTR, szDialogTitle, MB_OK|MB_ICONINFORMATION);
         Close_File(&hFile);
         return TRUE;  // atari font file in memory...
	  }
}

//Lecture des paramtres de la police texte selectionne (taille en pixel des caractres)
//Reading of the selected text font parameters (size in pixels)
//- it updates: dwCharX = average width of a char in pixels
// 			    dwCharY = average heigth of a char in pixels
//              ptMaxX,Y = minimum size needed for displaying the sector (in pixels)
void Update_Font()
{
      // Get the metrics of the current font.
      hdc = GetDC(hEditDisplay);
      SelectFont(hdc,hTheEditFont);
      GetTextMetrics(hdc, &tm);
      ReleaseDC(hEditDisplay, hdc);

      // Save the average character width and height.
      dwCharX = tm.tmAveCharWidth;
      dwCharY = tm.tmHeight;

      ptMaxX = (LONG)(84 * dwCharX);
      ptMaxY = (LONG)((28 * dwCharY) + dwZoneY);
}

//Set the window placement minimum resizing size X and Y
void CalculMaxSize(HWND hwnd)
{
		WINDOWPLACEMENT wndpl;

      wndpl.length = sizeof(WINDOWPLACEMENT);
      if (!GetWindowPlacement(hwnd, &wndpl))
      	MessageBox(hwnd,"ARFD Sys Warning:\nCan not get window placement!",szDlgBoxMain,MB_OK|MB_ICONWARNING);
      wndpl.rcNormalPosition.right = ptMaxX;
      wndpl.rcNormalPosition.bottom = ptMaxY;
      if (!SetWindowPlacement(hwnd, &wndpl))
      	MessageBox(hwnd,"ARFD Sys Warning:\nCan not set window placement!",szDlgBoxMain,MB_OK|MB_ICONWARNING);
}

//Redraw the 4 rectangles on the Edit sector display screen
void Redraw_Lines(HDC TheHDC)
{
		int a,b,c,h,hh;
		HPEN NewPen, OldPen;
      NewPen = CreatePen( PS_SOLID, 1, RGB(127,127,127));
      OldPen = SelectPen(TheHDC, NewPen);

      a = (3*0 + nZeroCaretX - 1) * dwCharX + dwCharX/2;
      b = (3*8 + nZeroCaretX - 1) * dwCharX + dwCharX/2;
      c = (3*16 + nZeroCaretX - 1) * dwCharX + dwCharX/2;
      h = (nZeroCaretY-2) * dwCharY;
      hh = (EditSectorSize/16 +1 +2) * dwCharY;;
      MoveToEx(TheHDC, a, h, NULL);
      LineTo(TheHDC, c, h);
      LineTo(TheHDC, c, h+hh);
      LineTo(TheHDC, a, h+hh);
      LineTo(TheHDC, a, h);
      MoveToEx(TheHDC, b, h, NULL);
      LineTo(TheHDC, b, h+hh);
      a = (16*3 + nZeroCaretX + nAsciiGap -1) * dwCharX;
      b = (16*3 + nZeroCaretX + nAsciiGap + 7) * dwCharX;
      c = (16*3 + nZeroCaretX + nAsciiGap + 15) * dwCharX;
      MoveToEx(TheHDC, a, h, NULL);
      LineTo(TheHDC, c, h);
      LineTo(TheHDC, c, h+hh);
      LineTo(TheHDC, a, h+hh);
      LineTo(TheHDC, a, h);
      MoveToEx(TheHDC, b, h, NULL);
      LineTo(TheHDC, b, h+hh);
      SelectObject(TheHDC, OldPen);
      DeleteObject(NewPen);
}

//Ligne de titre et pointeurs pour affichage secteur
void Make_Title_String(void)
{
		int i, a, b;
   	char t[32];
   	a = iMemAdd & 0x000FFFFF;
   	strcpy(t,"");
   	strcat(lpszEditScreen,"<MmPos> Byt# "); //11 espaces
   	for (i = 0; i< 16; i++)
   	{
   		wsprintf(szSTR,"%02X ",((uOffset + i)& 0x0F));
      	strcat(lpszEditScreen,szSTR);
      	wsprintf(szSTR,"%X",((uOffset + i)& 0x0F));
      	strcat(t,szSTR);
   	}
   	wsprintf(szSTR,"  %s\n",t);
   	strcat(lpszEditScreen,szSTR);
      for (i=0;i<=78;i++)
      	strcat(lpszEditScreen,"-");
      strcat(lpszEditScreen,"\n");
      b = (EditSectorSize & 0xF0) >> 4;
      for (i=0;i<=b;i++)
      {
      		wsprintf(szSTR,"<%05X>",a);
      		a += 16;
            strcat(lpszEditScreen,szSTR);
            wsprintf(szSTR," %02X : ",((uOffset + i*16)& 0xFF));
            strcat(lpszEditScreen,szSTR);
            strcat(lpszEditScreen,"\n");
      }
}
//Calcul la position du curseur
void Calculate_Caret_At_Index_Pos(int TheIndex, BOOL bAsciiChoice)
{
		// valeur d'entre :  BOOL bAsciiChoice
   	//							 int TheIndex
   	// met  jour nCaretPosX et nCaretPosY
   	if (bAsciiChoice)
   	{
       		nCaretPosX = ((TheIndex & 0x0F)+ nLineLen + nAsciiGap);
   	}
   	else
   	{
   	 		nCaretPosX = 3 * (TheIndex & 0x0F) + nZeroCaretX;
   	}
   	nCaretPosY = (TheIndex & 0xF0)/16 + nZeroCaretY;
      dwCPX = nCaretPosX * dwCharX;
   	dwCPY = nCaretPosY * dwCharY;
}
//Place le curseur
void Set_Caret_At_Index_Pos()
{
		Calculate_Caret_At_Index_Pos(iDIndex, bEditAscii);
   	SetCaretPos(dwCPX, dwCPY);
}
//Insersion ou suppresssion d'octet
void Insert_Byte(int nByteToInsert, UBYTE cRByte)
{
      int ip,jp;
      //Marquer les octets modifis
      for (ip = iDIndex; ip <= EditSectorSize; ip++)
        lpModData[ip] = -1;
      bFileModified = TRUE;
      //Dcaller les octets modifis
      ip = iESPos + EditSectorSize;   //size: 7F or FF or other
      jp = iESPos + iDIndex;
      if (bEditLimited) //limited to the sector
        Shift_Memory(jp, ip, nByteToInsert, cRByte);
      else  // No limitation = whole Rom File
        Shift_Memory(jp, 0, nByteToInsert, cRByte);

}
//Afficher la table complete du secteur: Hexa et Ascii
void Display_Table()
{
	char s[8];
   int tmpindex;
   COLORREF CT,CB;
   UBYTE d;
   tmpindex = iDIndex;
   HideCaret(hEditDisplay);
   hdc = GetDC(hEditDisplay);
      //SetBkMode(hdc,TRANSPARENT);
   SelectFont(hdc, hTheEditFont);
   for (iDIndex = 0; iDIndex <= EditSectorSize; iDIndex++)
   {
         //d = lpData[iDIndex];
         d = lpFileBuffer[iESPos+iDIndex];
         wsprintf(s,"%02X ",d);
         Calculate_Caret_At_Index_Pos(iDIndex, FALSE);

         if (lpModData[iDIndex] < 0)
            CT = rgbModColor;
         else
            CT = rgbTextColor;

         if ((iDIndex >= nSelStart) && (iDIndex < nSelStop) && (nSelStart != nSelStop))
         {
            CB = rgbUndrlnColor; //gris
         }
         else
         	CB = rgbScrnColor;   //Blanc

         SetTextColor(hdc, CT);
         SetBkColor(hdc, CB);

         TextOut(hdc, dwCPX, dwCPY, (LPCTSTR) s, 3);
         
         s[0] = d;
         if (!bAtariFont)
         {
         	if (!bIbmFont && (d<0x20))
         		s[0] = '.';
         	dwCPX = ((iDIndex & 0x0F)+ nLineLen + nAsciiGap) * dwCharX;
      		TextOut(hdc, dwCPX, dwCPY, (LPCTSTR) s, 1);
         }
         else Show_Atari_Char(hdc, d, iDIndex, CB);
 	}
   dwCPY += (2*dwCharY);
   strcpy(szSTR,"");
   SetTextColor(hdc, rgbTextColor);
   SetBkColor(hdc, rgbScrnColor);
   uFileNextSector = (UWORD) ((Get_Sector_Param(szSTR, iCurSect) >> 8) & 0x0000FFFF);
   TextOut (hdc, 2*dwCharX, dwCPY, (LPCTSTR) szSTR, strlen(szSTR));
   Redraw_Lines(hdc);
   ReleaseDC(hEditDisplay, hdc);
   iDIndex = tmpindex;
   Set_Caret_At_Index_Pos();
   ShowCaret(hEditDisplay);
}
//Afficher l'octet  la position iDIndex: valeur Hexa  + caractre ASCII
void Display_Byte(UBYTE ubTheByte)
{
  		int tmpCPX,tmpCPY;
      char tmpChar[4];
      DWORD dwTheParam;
      HideCaret(hEditDisplay);
      hdc = GetDC(hEditDisplay);
      SetBkColor(hdc, rgbScrnColor);
      SelectFont(hdc, hTheEditFont);
      SetTextColor(hdc, rgbModColor);
      // Hexa value display
      wsprintf(tmpChar,"%02X",ubTheByte);
      tmpCPX = (3 * (iDIndex & 0x0F) + nZeroCaretX) * dwCharX;
      tmpCPY = ((iDIndex & 0xF0)/16 + nZeroCaretY) * dwCharY;
      TextOut(hdc, tmpCPX, tmpCPY, (LPCTSTR) tmpChar, 2);
      //ASCII char display
      tmpChar[0] = (char) ubTheByte;
      if (!bAtariFont)
      {
         if (!bIbmFont && (ubTheByte < 0x20)) tmpChar[0] = '.';
      	tmpCPX = ((iDIndex & 0x0F)+ nLineLen + nAsciiGap) * dwCharX;
      	TextOut(hdc, tmpCPX, tmpCPY, (LPCTSTR) tmpChar, 1);
      }
      else
         Show_Atari_Char(hdc, ubTheByte, iDIndex, rgbScrnColor);
      //The sector parameters line
      SetTextColor(hdc, rgbTextColor);
      tmpCPY = ((EditSectorSize / 16) + nZeroCaretY + 2) * dwCharY;
      tmpCPX = nZeroCaretX*dwCharX;
      // Get the sector param. (checksum, next sector, byte per sector)
      dwTheParam = Get_Sector_Param(szSTR, iCurSect);
      // Display the checksum
      wsprintf(szSTR,"$%02X",(dwTheParam & 0x000000FF));
      TextOut (hdc, tmpCPX, tmpCPY, (LPCTSTR) szSTR, strlen(szSTR));
      //The next sector
      uFileNextSector = (UWORD) ((dwTheParam >> 8) & 0x0000FFFF);
      //If nextsector = 0xFFFF, this is not a standard sector, so no display
      if (uFileNextSector != 0xFFFF)
      {
      	wsprintf(szSTR,"$%03X", uFileNextSector);
      	tmpCPX += (19*dwCharX);
      	TextOut (hdc, tmpCPX, tmpCPY, (LPCTSTR) szSTR, strlen(szSTR));
      	wsprintf(szSTR,"$%02X",((dwTheParam >> 24) & 0x000000FF));
      	tmpCPX += (25*dwCharX);
      	TextOut (hdc, tmpCPX, tmpCPY, (LPCTSTR) szSTR, strlen(szSTR));
      }
      Redraw_Lines(hdc);
      ReleaseDC(hEditDisplay, hdc);
      ShowCaret(hEditDisplay);
}
//Dselection des octets du secteur
void UnSelect_Data()
{
	bDataSelected = FALSE;
   nSelStart = iDIndex;
   nSelStop = iDIndex;
}
//Activate the data selection on display
void Selection_Data(int inc)
{
		if (!bDataSelected) // pas en cours de selection ?
      {
      	bDataSelected = TRUE;  // alors en cours de selection
         nSelStart = iDIndex;     // sauvegarde du premier octet selectionn
      }
      // inc = nombre d'octet  selectionner
      // Si inc = 0, alors selectionner jusqu' la fin du secteur
      if (inc > 0)
      	nSelStop = iDIndex + inc;
      else
         nSelStop = EditSectorSize + 1;
}
//Put the string of the selection into the ClipBoard
void Copy_Into_CB(HWND hwnd, char * szTheString)
{
   int s = strlen(szTheString)+1;
	OpenClipboard(hwnd);
   HANDLE hData = GlobalAlloc(GMEM_SHARE, s);
   HANDLE pData = GlobalLock(hData);
   memcpy(pData,szTheString,s);
   GlobalUnlock(hData);
   SetClipboardData(CF_TEXT, hData);
   CloseClipboard();
   GlobalFree(hData);
}
//Mise  jour affichage StatusBar
void Update_Status(void)
{
      wsprintf(szSTR,"Sector $%03X; Index $%02X; Data $%02X)",iCurSect,iDIndex,lpFileBuffer[iESPos+iDIndex]);
      SendMessage(hEditStatus,SB_SETTEXT,0,(LPARAM)szSTR);

      wsprintf(szSTR,"Caret %d:%d",nCaretPosX,nCaretPosY);
      SendMessage(hEditStatus,SB_SETTEXT,1,(LPARAM)szSTR);

      if (bDataSelected)
         wsprintf(szSTR,"Selection $%02X-$%02X ($%03X)",nSelStart,(nSelStop-1),(nSelStop-nSelStart));
      else if (nSelBytes)
           wsprintf(szSTR,"$%03X bytes in selection",nSelBytes);
      else
      	wsprintf(szSTR,"No Selection");
      SendMessage(hEditStatus,SB_SETTEXT,2,(LPARAM)szSTR);

      wsprintf(szSTR,"Keydown code $%02X",TheVK);
      SendMessage(hEditStatus,SB_SETTEXT,3,(LPARAM)szSTR);

      wsprintf(szSTR,"Char Virtual-Key code $%02X",TheCH);
      SendMessage(hEditStatus,SB_SETTEXT,4,(LPARAM)szSTR);
}
//Copy/Cut and Paste procedures
void EditSector_Edit_Command(HWND hwnd, int iTheCmnd)
{
   UBYTE ubNewByte;
   int i, n, p;
   UINT f, r;
   char sztmp[8];
   char szTheCmnd[3][64] = {"Insert some empty bytes from the selection position?",
                            "Suppress the data bytes from the selection position?",
                            "Copy the selected ASCII text displayed into the clipboard ? "};

   // Si pas Paste et pas de selection alors afficher erreur:
   iNumOrder = 1;
	if (!bDataSelected && (iTheCmnd != 3))
   {
               MessageBox(hwnd,"ARFD Info:\nAction not possible: no data byte selected!",
            	szChildEditWindowTitle,MB_OK|MB_ICONINFORMATION);
               return;
   }
   //Caractre de remplacement
   ubNewByte = (UBYTE)((bEditAscii && !bAtariFont) ? 0x20 : 0x00);
   n = (int) (nSelStop - nSelStart);
   p = iDIndex;
   if (iTheCmnd < 3)
   	if (MessageBox(hwnd,szTheCmnd[(iTheCmnd-1)],
            	szChildEditWindowTitle,MB_OKCANCEL|MB_ICONQUESTION) != IDOK)
      	return;
   switch (iTheCmnd)
   {
   	case 1: //Insertion of bytes (no matter Insertion/Replacement mode
         iDIndex = nSelStart;
      	Insert_Byte(n, ubNewByte);
         break;
      case 2: //Suppression of bytes (no matter Insertion/Replacement mode
         iDIndex = nSelStart;
         Insert_Byte((-n), ubNewByte);
         UnSelect_Data();
         break;
      case 3: //PASTE (regarding Insertion/Replacement mode)
         if (!nSelBytes) //nSelBytes from previous selection
         {
         	MessageBox(hwnd,"ARFD Info:\nAction not possible: nothing to paste!",
            	szChildEditWindowTitle,MB_OK|MB_ICONINFORMATION);
         	return;
         }
         if (bEditLimited)
         	f = iESPos + EditSectorSize;   //size: 7F or FF or other
         else
            f = 0; //limited to the sector
         if (bDataSelected && !bReplace)
         {
            if (MessageBox(hwnd,"Click OK to replace the current selection?",
            	szChildEditWindowTitle,MB_OKCANCEL|MB_ICONINFORMATION) == IDCANCEL)
                  break;
            iDIndex = nSelStart;
            Insert_Byte((-n), ubNewByte); //Suppression de la slection
            r = Copy_In_Memory(lpDataCB,(UINT)(iESPos+iDIndex),f,(UINT)nSelBytes,TRUE);
         }
         else
         {
            if (MessageBox(hwnd,"Click OK to paste data at the current position?",
            	szChildEditWindowTitle,MB_OKCANCEL|MB_ICONINFORMATION) == IDCANCEL)
                  break;
            r = Copy_In_Memory(lpDataCB,(UINT)(iESPos+iDIndex),f,(UINT)nSelBytes,!bReplace);
         }
         //Marquer les octets modifis.
         if ((iDIndex+r) > EditSectorSize) r = EditSectorSize - iDIndex + 1;
         for (i=0;i<r;i++) //Marquer les octets modifis.
            lpModData[iDIndex+i] = -1;
         bFileModified = TRUE;
         UnSelect_Data();
         break;
      case 4: //Copy
      case 5: //Cut
         iDIndex = nSelStart;
         //Nombre d'octets selectionns
         nSelBytes = n; //For the next time: there is something in Data clipboard
         //Copy the data into the Data clipboard:
         for (i=0;i<n;i++)
         	lpDataCB[i] = lpFileBuffer[iESPos+iDIndex+i];
         if (iTheCmnd == 5) //Cut
         {
            Insert_Byte((0-n), ubNewByte);
            UnSelect_Data();
         }
         else //Copy
            iDIndex = p;  //Stay at the end of selection
         break;
      // Paste as text into Clipboard:
      case 6:
      	if (MessageBox(hwnd,szTheCmnd[2],
            	szChildEditWindowTitle,MB_OKCANCEL|MB_ICONQUESTION) != IDOK)
                 return;
         iDIndex = nSelStart;
         strcpy(lpszEditScreen,"");
         for (i=0;i<n;i++)
         {
            ubNewByte = lpFileBuffer[iESPos+iDIndex+i];
            if (bEditAscii)
            {
            	if (ubNewByte < 0x20) ubNewByte  = '.';
               wsprintf(sztmp,"%c", ubNewByte);
            }
            else
         		wsprintf(sztmp,"%02X ", ubNewByte);
            strcat(lpszEditScreen,sztmp);
         }
         Copy_Into_CB(hwnd, lpszEditScreen);
         iDIndex = p;
         break;
   }
   //Raffichage donnes
   Display_Table();
   Update_Status();
}
//====================================================================================//
//====================================================================================//
// MAIN WINDOW PROCEDURES
//====================================================================================//
//====================================================================================//
//----------------------------------------------------------------------------//
// Register the window
//----------------------------------------------------------------------------//
BOOL Register_EditSector(HINSTANCE hInst)
{
		WNDCLASS WndClass;

  		WndClass.style          = CS_HREDRAW | CS_VREDRAW;
  		WndClass.lpfnWndProc    = ChildEditWindowProc;
  		WndClass.cbClsExtra     = 0;
  		WndClass.cbWndExtra     = 0;
  		WndClass.hInstance      = hInst;
      WndClass.hIcon          = LoadIcon(hInst, "MonIcone");
  		WndClass.hCursor        = LoadCursor(NULL, IDC_ARROW);
  		WndClass.hbrBackground  = (HBRUSH)(WHITE_BRUSH);
  		WndClass.lpszMenuName   = "EditMenu";
  		WndClass.lpszClassName  = szEditAppName;

  		return (RegisterClass (&WndClass) != 0);
}

//----------------------------------------------------------------------------//
// Create the window
//----------------------------------------------------------------------------//
HWND Create_EditSectorChild(HINSTANCE hInst, int nCmdShow)
{
  		HWND hwnd = CreateWindow(szEditAppName, szChildEditWindowTitle,
      							//with a Resizing border:
                           WS_OVERLAPPEDWINDOW,
                           CW_USEDEFAULT, CW_USEDEFAULT,
                           CW_USEDEFAULT, CW_USEDEFAULT,
                           hMainWindow, //NULL,
                           NULL, hInst, NULL);

  		if (hwnd == NULL)
     			return hwnd;
  		ShowWindow(hwnd, nCmdShow);
  		UpdateWindow(hwnd);
  		return hwnd;
}

//----------------------------------------------------------------------------//
// The Window Procedure
//----------------------------------------------------------------------------//
LRESULT CALLBACK _export ChildEditWindowProc(HWND hwnd, UINT Message,
                             WPARAM wParam, LPARAM lParam)
{
  	switch(Message)
  	{
    	HANDLE_MSG(hwnd, WM_CREATE, EditSector_OnCreate);
      HANDLE_MSG(hwnd, WM_CLOSE, EditSector_OnClose);
      HANDLE_MSG(hwnd, WM_GETMINMAXINFO, EditSector_OnGetMinMaxInfo);
    	HANDLE_MSG(hwnd, WM_SIZE, EditSector_OnSize);
    	HANDLE_MSG(hwnd, WM_SETFOCUS, EditSector_OnSetFocus);
    	HANDLE_MSG(hwnd, WM_KILLFOCUS, EditSector_OnKillFocus);
    	HANDLE_MSG(hwnd, WM_KEYDOWN, EditSector_OnKeyDown);
      HANDLE_MSG(hwnd, WM_COMMAND, EditSector_OnCommand);
    	HANDLE_MSG(hwnd, WM_CHAR, EditSector_OnChar);
    	HANDLE_MSG(hwnd, WM_PAINT, EditSector_OnPaint);
    	HANDLE_MSG(hwnd, WM_LBUTTONDOWN, EditSector_OnLButttonDown);
      HANDLE_MSG(hwnd, WM_RBUTTONDOWN, EditSector_OnRButttonDown);
    	default:
      	return EditSector_DefProc(hwnd, Message, wParam, lParam);
  	}
}

//----------------------------------------------------------------------------//
// Creation procedure handles the WM_CREATE request
//----------------------------------------------------------------------------//
#pragma argsused
BOOL EditSector_OnCreate(HWND hwnd, CREATESTRUCT FAR* lpCreateStruct)
{
    	int sbParts[5];
    	LPINT aWidth = sbParts;

      lpszEditScreen =(char *) GlobalAlloc(GMEM_FIXED,65536);

      Load_Sector_To_Edit();
      bMemAddIsPos = TRUE;
      iMemAdd = iCurFilePos;
      bReplace = TRUE;
      bDataSelected = FALSE;
      bSearchInProgress = FALSE;
      bEditLimited = TRUE;
      nSelStart = 0x00;
      nSelStop = 0x00;
      nSelBytes = 0;

    	hEditStatus = CreateStatusWindow(WS_CHILD | WS_VISIBLE | SBS_SIZEGRIP,
                              "Status Bar", hwnd, 200);
    	sbParts[0] = 180; //pixels from left
    	sbParts[1] = 245; //pixels from left
    	sbParts[2] = 375;
    	sbParts[3] = 475;
    	sbParts[4] =  -1; //pixels from left
    	SendMessage(hEditStatus,SB_SETPARTS,5,(LPARAM)aWidth);

    	hCommandes1 = CreateDialog(hInstance,"MyDlg1",hwnd,(DLGPROC)MyDlgProc1);
    	hCommandes2 = CreateDialog(hInstance,"MyDlg2",hwnd,(DLGPROC)MyDlgProc2);

      hEditDisplay = CreateWindow("static",
                        "",
                        //WS_VSCROLL|WS_HSCROLL|//WS_AUTOVSCROLL|WS_AUTOHSCROLL|
                        WS_CHILDWINDOW|WS_VISIBLE|WS_BORDER,
                        //CW_USEDEFAULT,  CW_USEDEFAULT,  CW_USEDEFAULT,  CW_USEDEFAULT,
                        0,0,0,0,
                        hwnd, (HMENU)0,
                        hInstance,NULL);

      if (!hCommandes1 || !hCommandes2 || !hEditDisplay)
      {
          MessageBox(hwnd,"ARFD Sys Error:\nSome dialog boxes are not available, exiting!",
          	"Edit sector Window Creation:",MB_OK | MB_ICONSTOP);
          return FALSE;
      }

     	hTheAsciiFont = hfARFD;

      if (bIbmFont)
      	hTheEditFont = GetStockFont(OEM_FIXED_FONT);
      else
      	hTheEditFont = hTheAsciiFont;

      //Mise  jour de la police avec choix  OEM Font + Redimensionnement fentre
      bAtariFont = FALSE;
      bIbmFont = TRUE;
      Update_Font();
      PostMessage(hCommandes2,WM_COMMAND,IDC_RADIOBUTTON107,0);

      nZeroCaretX = 6 + 7; //7 pour les iMemAdd
      nZeroCaretY = 4;
      nLineLen = nZeroCaretX + 47;  //47 = 3*16 -1
      nMaxLines = nZeroCaretY + (EditSectorSize/16) -1 ; // + 14 ou + 6
      nAsciiGap = 3;
      nCaretPosX = nZeroCaretX;
      nCaretPosY = nZeroCaretY;

      iNumOrder = 1;
      //iDIndex = 0;
      iDIndex = iCurOff;
      uOffset = 0;

      ptMaxX = 0;
      ptMaxY = 0;

      bEditAscii = FALSE;
      Calculate_Caret_At_Index_Pos(iDIndex,bEditAscii);

      if (!bOKFont)
      	bOKFont = Read_Rom_Font(hwnd,FALSE);    // charger la font par dfaut...

      return TRUE;
}

//----------------------------------------------------------------------------//
// Close procedure handles the WM_CLOSE request
//----------------------------------------------------------------------------//
void EditSector_OnClose(HWND hwnd)
 {
     hChildEditWindow = NULL;
     GlobalFree (lpszEditScreen);
     DestroyWindow(hwnd);
 }

//----------------------------------------------------------------------------//
// Close procedure handles the WM_CLOSE request
//----------------------------------------------------------------------------//
void EditSector_OnGetMinMaxInfo(HWND hwnd, LPMINMAXINFO lpMinMaxInfo)
{
   if (hChildEditWindow)
   {
   	lpMinMaxInfo->ptMinTrackSize.y = ptMaxY;
   	lpMinMaxInfo->ptMinTrackSize.x = ptMaxX;
   }
}

//----------------------------------------------------------------------------//
// Size procedure handles the WM_SIZE messages
//----------------------------------------------------------------------------//
void EditSector_OnSize(HWND hwnd, UINT state, int cx, int cy)
{
      RECT Rct;
      int tcx,tcy;
      int h;
      int hcmd1;
      int hcmd2;
      int sbh = 0;
      tcx = cx;
      tcy = cy;

      if (hEditStatus)
      {
    				MoveWindow(hEditStatus,0,0,0,0,TRUE);
      			SendMessage(hEditStatus, WM_SIZE, state, MAKELONG(tcx,tcy));
      			GetClientRect(hEditStatus,&Rct);
    				sbh  = Rct.bottom +1; // status bar height
      }
      GetClientRect(hCommandes1,&Rct);
      hcmd1 = Rct.bottom;
      MoveWindow(hCommandes1,0,1,tcx,hcmd1,TRUE);
      hcmd1 ++;

      GetClientRect(hCommandes2,&Rct);
      hcmd2 = Rct.bottom;
      dwZoneX = Rct.right;  //this is the minimal window size
      h = tcy-sbh-hcmd2;
      MoveWindow(hCommandes2,0,h,tcx,hcmd2,TRUE);

      SetWindowPos(hEditDisplay, NULL,0,hcmd1,tcx,(tcy-sbh-hcmd1-hcmd2), SWP_NOZORDER);

      GetClientRect(hEditDisplay,&Rct);
      dwEditX = 0;
      dwEditY = hcmd1;
      dwZoneY = sbh+hcmd1+hcmd2;
      dwMaxEditX = tcx;
      dwMaxEditY = hcmd1 + Rct.bottom;
}

//----------------------------------------------------------------------------//
// Paint procedure handles WM_PAINT messages
//----------------------------------------------------------------------------//
void EditSector_OnPaint(HWND hwnd)
{
  		PAINTSTRUCT PaintStruct;
  		RECT Rect;
  		HDC PaintDC;
      HBRUSH NewBrush, OldBrush;
      int h;

      HideCaret(hEditDisplay);

      wsprintf(lpszEditScreen," Editing Sector $%03X (%d) (Rom File memory pointer : $%05X)\n\n",
      	iCurSect,iCurSect,iCurFilePos);

      Make_Title_String();

  		PaintDC = BeginPaint(hEditDisplay, &PaintStruct);

  		SetBkMode(PaintDC, TRANSPARENT);
      GetClientRect(hEditDisplay, &Rect);

      NewBrush = CreateSolidBrush(rgbScrnColor);
      OldBrush = SelectBrush(PaintDC,NewBrush);
      Rectangle (PaintDC, 0, 0, Rect.right, Rect.bottom);

      SelectBrush(PaintDC, OldBrush);
      DeleteBrush(NewBrush);

      SelectFont(PaintDC,hTheEditFont);
      SetTextColor(PaintDC,rgbTextColor);
      DrawText(PaintDC, lpszEditScreen, -1, &Rect, DT_LEFT);

      Display_Table();

      EndPaint(hEditDisplay, &PaintStruct);

	   Update_Status();

      ShowCaret(hEditDisplay);

      PaintDC = BeginPaint(hwnd, &PaintStruct);
      GetClientRect(hwnd, &Rect);
      h = Rect.bottom;
      Rectangle (PaintDC, 0, 0, Rect.right, 1);
      GetClientRect(hEditStatus, &Rect);
      h = h - Rect.bottom -1;
      Rectangle (PaintDC, 0, h, Rect.right, h+1);
      EndPaint(hwnd, &PaintStruct);
}

//----------------------------------------------------------------------------//
// SetFocus procedure handles the WM_SETFOCUS messages
//----------------------------------------------------------------------------//
void EditSector_OnSetFocus(HWND hwnd, HWND hwndOldFocus)
{
		// Create, position, and display the caret when the
      // window receives the keyboard focus.
      SetFocus(hwnd);
      if (bReplace) CreateCaret(hEditDisplay, NULL, dwCharX, dwCharY);
      else CreateCaret(hEditDisplay, NULL, 2, dwCharY);
      Set_Caret_At_Index_Pos();
      ShowCaret(hEditDisplay);
}

//----------------------------------------------------------------------------//
// KillFocus procedure handles the WM_KILLFOCUS messages
//----------------------------------------------------------------------------//
void EditSector_OnKillFocus(HWND hwnd, HWND hwndNewFocus)
{
		// Hide and destroy the caret when the window loses the
      // keyboard focus.
      HideCaret(hEditDisplay);
      DestroyCaret();
      SetFocus(hwndNewFocus);
}

//----------------------------------------------------------------------------//
// OnKeyDown procedure handles the WM_KEYDOWN messages
//----------------------------------------------------------------------------//
void EditSector_OnKeyDown(HWND hwnd, UINT vk, BOOL fDown, int cRepeat, UINT flags)
{
      UBYTE ubNewByte;
      BOOL bSel;
      int i,j;

      TheVK = vk;
      nPressKey = GetKeyState(VK_SHIFT); //is SHIFT pressed ?
      bSel = nPressKey & 0x8000; //bSel = TRUE if YES the shift key is pressed
      nPressKey = GetKeyState(VK_CONTROL); //is CONTROL pressed ?
      nPressKey &= 0x8000;

      switch (TheVK) //(wParam)
      {
      	case VK_LEFT:   // LEFT ARROW
            bSel = FALSE;
            if (iNumOrder == 1)
            {
					iDIndex--;
            	if (iDIndex > EditSectorSize) iDIndex = 0 ;
            }
            break;
         case VK_RIGHT:  // RIGHT ARROW
         	if (bSel)  // processus de slection
            {
                Selection_Data(1);
                Display_Table();
            }
            // dans tous les cas : augmenter iDIndex et sortir du switch et du traitement :
            iDIndex++;
            if (iDIndex > EditSectorSize) iDIndex = EditSectorSize;
            break;
         case VK_UP:     // UP ARROW
            bSel = FALSE;
         	iDIndex -=16;
            if (iDIndex > EditSectorSize) iDIndex +=16;
            break;
         case VK_DOWN:   // DOWN ARROW
            if ((iDIndex+16) > EditSectorSize) break;
            if (bSel)  // processus de slection
            {
               Selection_Data(16);
               Display_Table();
            }
         	iDIndex +=16;
            break;
         case VK_HOME:   // HOME
            bSel = FALSE;
         	iDIndex = 0;
            break;
         case VK_END:   // END
         	if (bSel)  // processus de slection
            {
                Selection_Data(-1); //Tout selectionner jusqu' la fin
                Display_Table();
            }
            iDIndex = EditSectorSize;
            break;
         case VK_DELETE:  // DELETE
            ubNewByte = (UBYTE)((bEditAscii && !bAtariFont) ? 0x20 : 0x00);
            if (bDataSelected)
            {
               j = iDIndex;
               iDIndex = nSelStart;
            	if (!bReplace) //Mode Insertion: Effacer la selection par shifting
             	    Insert_Byte((nSelStart-nSelStop), ubNewByte);
               else //Mode Replacement: Effacer la selection par le caractre nul
               {
         			for (i=0;i<(nSelStop-nSelStart);i++)
         			{
         				lpFileBuffer[iESPos+iDIndex+i] = ubNewByte;
            			lpModData[iDIndex+i] = -1;
         			}
                  bFileModified = TRUE;
                  iDIndex = j;
               }
            }
            else
            {
            	lpFileBuffer[iESPos+iDIndex] = ubNewByte;
            	if (!bReplace) //Insertion
            	{
             		Insert_Byte(-1,ubNewByte);
            	}
            	else //Replacement
            	{
               	lpModData[iDIndex] = -1;
                  bFileModified = TRUE;
               	if (iNumOrder == 1)
               		iDIndex ++;
               	if (iDIndex > EditSectorSize) iDIndex = EditSectorSize ;
            	}
            }
            Display_Table();
            break;
         case VK_TAB:
         	bEditAscii = !bEditAscii;
            SendMessage(hCommandes2, WM_COMMAND, IDLG2_RESET, 0);
            return;
         case VK_INSERT:
            bReplace = !bReplace;
            SetFocus(hCommandes2);
            SendMessage(hCommandes2, WM_COMMAND, IDLG2_RESET, 0);
            break;
         case VK_PRIOR:
            SendMessage(hCommandes1, WM_COMMAND, IDC_B103_PRIORSECT, 0);
            break;
         case VK_NEXT:
            SendMessage(hCommandes1, WM_COMMAND, IDC_B104_NEXTSECT, 0);
            break;
         case VK_F3:
            // Relancer la recherche et sortir
         	PostMessage(hwnd, WM_COMMAND, CM_SEARCHAGAIN, 0);
            return;
         case VK_ESCAPE:
            // Annuler et sortir
         	SendMessage(hCommandes1, WM_COMMAND, IDC_B106_CANCELEXIT, 0);
            return;
         case 'T':
            if (nPressKey) PostMessage(hwnd, WM_COMMAND, CM_FILENEXTSECTOR, 0);
            return;
         case 'F':
            if (nPressKey) PostMessage(hwnd, WM_COMMAND, CM_SEARCHHEX, 0);
            return;
         case 'H':
            if (nPressKey) PostMessage(hwnd, WM_COMMAND, CM_REPLACEHEX, 0);
            return;
         case 'Z':
            if (nPressKey) SendMessage(hCommandes1, WM_COMMAND, IDC_B102_RESTOREBYTE, 0);
            return;
         case 'C':
            if (nPressKey) PostMessage(hwnd, WM_COMMAND, CM_EDITCOPY, 0);
            return;
         case 'V':
            if (nPressKey) PostMessage(hwnd, WM_COMMAND, CM_EDITPASTE, 0);
            return;
         case 'X':
            if (nPressKey) PostMessage(hwnd, WM_COMMAND, CM_EDITCUT, 0);
            //MessageBox(hwnd,"C, V, X avec Control presses","Clavier",MB_OK);
            return;
         //case VK_BACK: // Trait dans On_Char.
         //case VK_RETURN: // Pas de traitement
         default:   //autres touches clavier:
            // Mettre  jour la StatusBar
            Update_Status();
            // Il faut traiter dans On_Char, donc on sort directement.
         	return;
      }

      //si plus d'appui sur SHIFT, plus de selection: on annule toute selection
      if (bDataSelected && !bSel)
		{
      		UnSelect_Data();
            Display_Table();
      }
      //Sortie normale de la fonction:
      iNumOrder = 1;
      Set_Caret_At_Index_Pos();
      Update_Status();
}
//----------------------------------------------------------------------------//
// OnChar procedure handles the WM_CHAR messages
//----------------------------------------------------------------------------//
void EditSector_OnChar(HWND hwnd, UINT ch, int cRepeat)
{
            static UBYTE ubNewByte;
            static int iByteValue;

            TheCH = ch & 0xFF;

            //If CONTROL key is pressed or the Ch is TAB: no action, exit:
            if ((nPressKey) || (TheCH == 0x09)) return;

            // dans WM_CHAR : pas de selection possible: on annule toute selection
            if (bDataSelected)
            {
					UnSelect_Data();
               Display_Table();
            }

            // Le test du TheCH est dplac en tte avant le switch...
            // Si le caractre est affich alors aller en fin de traitement
            // Si ce n'est pas un caractre affichable, alors c'est un caractre de contrle  passer dans le switch
            // Si il n'est pas passer dans le switch (arrive en dfault) faire un bip !!!

            // #################################################################
            // Les caractres clavier affichables / Displayable keyboard char
            if (!bEditAscii)  // HEXA modification
            {
                     //Test : est-ce un caratre de contrle?  / Is it a control char?
                  	if ( !((TheCH >= 0x30) && (TheCH <= 0x46) && (TheCH != 0x40)) &&
                   		  !((TheCH >= 0x61) && (TheCH <= 0x66)) ) goto Control;
                     //Minuscule / Lower case
                   	if ((TheCH >= 0x61) && (TheCH <= 0x66)) TheCH &= 0xDF;

                     //Insertion et nouveau caractre = dcaller avant d'afficher le nouveau:
                     //Insert and New char = memory data shifting before display the new one:
                     if (!bReplace && (iNumOrder == 1))
                     {
                        Insert_Byte(1,0x00);
                        Display_Table();
                     }
                     //Traitement du caractre clavier entr / Entered kbcode treatment:
                     lpModData[iDIndex] = -1;
                     bFileModified = TRUE;
                     ubNewByte = lpFileBuffer[iESPos+iDIndex];
                     iByteValue = TheCH - 0x30;
                    	if (iByteValue > 9)
                     	iByteValue = TheCH - 0x37;
                    	if (iNumOrder == 1)
                    	{
                       	iByteValue *= 16;
                       	if (bReplace)
                        	ubNewByte &= 0x0F;
                       	else
                        	ubNewByte = 0;
                    	}
                    	else //iNumOrder = 2;
                    		ubNewByte &= 0xF0;
                     //
                    	ubNewByte += (UBYTE) iByteValue;
                     //Curseur au Caractre suivant / Caret at the Next digit:
                     nCaretPosX ++;
                     dwCPX += dwCharX;
                     iNumOrder ++;
            }
            else // ASCII modification
            {
                     //Test : est-ce un caratre de contrle?  / Is it a control char?
                    	if ( !((TheCH >= 0x20) && (TheCH <= 0xFF)) ) goto Control;
                     //Conversion pour Atari char
                     if (bAtariFont && (TheCH >= 0x20) && (TheCH <= 0x5A))
                     	TheCH -= 0x20;
                     //Insertion et nouveau caractre = dcaller avant d'afficher le nouveau
                     if (!bReplace)
                     {
                     	Insert_Byte(1,0x20);
                     	Display_Table();  //on decalle d'un octet
                     }
                     lpModData[iDIndex] = -1;
                     bFileModified = TRUE;
                     ubNewByte = (UBYTE) TheCH;
                     lpFileBuffer[iESPos+iDIndex] = ubNewByte;
                     //Toujours le dernier digit en ASCII
                     iNumOrder = 3;
            }
            //mise en memoire/Store in memory:
            lpFileBuffer[iESPos+iDIndex] = ubNewByte;
            //Afficher le nouvel octet/Display new byte:
            Display_Byte(ubNewByte);
            
            // end tests et traitement des caractres affichable
            // #################################################################
            goto End;

            // #################################################################
            // Les caractres clavier de contrle / Control keyboard char
Control:
            switch (TheCH)
            {
                // BACKSPACE :
                case 0x08:
                   if ((iDIndex == 0) && (iNumOrder == 1)) break;
                   ubNewByte = (UBYTE)((bEditAscii && !bAtariFont) ? 0x20 : 0x00);
                   iByteValue = ubNewByte;
                   if (!bEditAscii) // HEXA Modification
                   {
                        if (iNumOrder == 1)
                           iDIndex--;
                        else
                           iNumOrder = 1;
                   }
                   else  //ASCII Modification
                   {
                        iDIndex--;
                        iNumOrder = 1;
                   }
                   if (iDIndex > EditSectorSize) // > means a value lower than '0'
                       	iDIndex = 0;
                   Set_Caret_At_Index_Pos();
                   if (!bReplace) Insert_Byte(-1,ubNewByte);
                   else
                   {
                     	lpFileBuffer[iESPos+iDIndex] = ubNewByte;
                     	lpModData[iDIndex] = -1;
                        bFileModified = TRUE;
                   }
                   Display_Table();
                   break;
                case 0x1B:  // escape
                case 0x0A:  // linefeed
                case 0x0D:  // carriage return
                    return; // -> ne rien faire, ni afficher
                case 0x20: //Space
                    // Avancer d'un caractre (digit)
                    // si modification hexa et si remplacement uniquement
                	  if (!bEditAscii && bReplace)
                    {
                        nCaretPosX ++;
                        dwCPX += dwCharX;
                        iNumOrder ++;
                        break;
                    }
                default:    // Caractre inconnu/Unknown kbchar !!
                   MessageBeep(0xFFFFFFFF);
                   return;
         	}
            // end switch() des touches de contrle
            // #################################################################
End:
            if (iNumOrder == 3)
            {
                    	  iNumOrder = 1;
                       iDIndex ++;
                       if (iDIndex > EditSectorSize)
                       {
                         if (MessageBox(hwnd,"Load the Next sector?",
                         	szChildEditWindowTitle,MB_OKCANCEL|MB_ICONQUESTION) == IDOK)
                         {
                               iDIndex = 0 ;
                               PostMessage(hCommandes1,WM_COMMAND,IDC_B104_NEXTSECT,0);
                         }
                         else
                               iDIndex = EditSectorSize;

                       }
                       Set_Caret_At_Index_Pos();
            }
            SetCaretPos(dwCPX,dwCPY);
            Update_Status();
}

//----------------------------------------------------------------------------//
// OnLeftButtonDown procedure handles the WM_LBUTTONDOWN messages
//----------------------------------------------------------------------------//
void EditSector_OnLButttonDown(HWND hwnd, BOOL fDoubleClick, int x, int y, UINT keyFlags)
{
      short nVirtKey;
      BOOL bSel;
      int iSelIndex;
      UINT iNewIndex;
   	int a,h,la,lh;

      iSelIndex = (int) iDIndex;
      x -= dwEditX;
      y -= dwEditY;

      // lh = nombre de lignes n - 1 (1   n)
      // lh = 16 pour EditSectorSize = 255  ou lh = 8 pour EditSectorSize = 127
      lh = ((EditSectorSize + 1) / 16) -1;
      // d = nombre de colonnes pour la dernire ligne  (sinon c'est 16 colonnes)
      la = (EditSectorSize % 16);
      if (la == 0)
         la = 15;
      else
      {
      	lh++;
         la--;
      }

      a = x/dwCharX - nZeroCaretX;
      if (a < 0) // return;  // gauche de la zone Hex = fin
      a = 0;

      h = y/dwCharY - nZeroCaretY;
   	if (h < 0) h = 0;
   	if (h > lh) h = lh;

      if (a < 48) //zone hexa  3*16 caractres = 48 caractres
      {
         bEditAscii = FALSE;
         a /= 3;
      }
      else //zone Ascii ?
      {
			a = (x/dwCharX) - nAsciiGap - nLineLen;
         if (a < 0) return; // zone intermdiaire = fin
         if (a > 15)
			  	a = 15;
         bEditAscii = TRUE;
      }

      iNewIndex = a + h*16;
      if (iNewIndex > EditSectorSize) iNewIndex = EditSectorSize;

      iSelIndex = (int)iNewIndex - iSelIndex + 1;

      nVirtKey = GetKeyState(VK_SHIFT); //is SHIFT pressed ?
      bSel = nVirtKey & 0x8000; //bSel = TRUE if YES (0x8000 = bit si touche prsse)
      if (!bSel && bDataSelected)
        UnSelect_Data();
      else if (bSel)
      {
         if (iSelIndex > 0)
            Selection_Data(iSelIndex);
         else UnSelect_Data();
      }
      iDIndex = iNewIndex;
      iNumOrder = 1;
      Display_Table();
      SendMessage(hCommandes2, WM_COMMAND, IDLG2_RESET, 0);
}
//----------------------------------------------------------------------------//
// OnRightButtonDown procedure handles the WM_RBUTTONDOWN messages
//----------------------------------------------------------------------------//
void EditSector_OnRButttonDown(HWND hwnd, BOOL fDoubleClick, int x, int y, UINT keyFlags)
{
    POINT pt;
    HMENU hmenu;            // menu template
    HMENU hmenuTrackPopup;  // shortcut menu

    //  Load the menu template containing the shortcut menu from the
    // application's resources.
    hmenu = LoadMenu(hInstance, "EditMenu");
    if (hmenu == NULL) 
        return; 
 
    // Get the first shortcut menu in the menu template. This is the
    // menu that TrackPopupMenu displays.
    hmenuTrackPopup = GetSubMenu(hmenu, 1);
 
    // TrackPopup uses screen coordinates, so convert the 
    // coordinates of the mouse click to screen coordinates.
    pt.x = x;
    pt.y = y;
    ClientToScreen(hwnd, (LPPOINT) &pt);

    // Draw and track the shortcut menu.
    TrackPopupMenu(hmenuTrackPopup, TPM_LEFTALIGN | TPM_LEFTBUTTON, 
        pt.x, pt.y, 0, hwnd, NULL); 
 
    // Destroy the menu.
    DestroyMenu(hmenu);

}
//----------------------------------------------------------------------------//
// OnCommand procedure handles the WM_COMMAND messages
//----------------------------------------------------------------------------//
void EditSector_OnCommand(HWND hwnd, int id, HWND hwndCtl, UINT codeNotify)
{
    HMENU hM;
    int Result;
    switch(id)
    {
        //reset, reload and redisplay sector #1 (command coming from ARFD
        //when EditSector window is already open and new file is put in memory)
        case CM_RESETSECTORONE:
        		Load_Sector_To_Edit();
            UnSelect_Data();
            InvalidateRect(hChildEditWindow, NULL, TRUE);
            SetFocus(hChildEditWindow);
            break;
        // message venant de Search pour affichage selection octet  trouver :
        case CM_SHOWSELECTION:
           bDataSelected = TRUE;
           nSelStart = iCurOff;
           nSelStop = iCurOff + nSearchDataBytes;
           if (nSelStop > (SectSize + 1)) nSelStop = SectSize + 1;
           break;
        case CM_FILENEXTSECTOR:
           if ((uFileNextSector <= AmParam.MAXSect) && (uFileNextSector >= 1))
           {
            	iCurSect = uFileNextSector;
           }
           else break;
        //reset, reload and redisplay sector (command coming from ARFD
        //when EditSector window is already open)
        case CM_EDITSECTORRESET:
            Load_Sector_To_Edit();
        		InvalidateRect(hChildEditWindow, NULL, TRUE);
            SetFocus(hChildEditWindow);
        		break;
        // commandes du menu: duplicant les boutons du dialogue suprieur :
	     case IDC_B101_RESTORESECTOR:
        case IDC_B102_RESTOREBYTE:
        case IDC_B103_PRIORSECT:
        case IDC_B104_NEXTSECT:
        case IDC_B105_ENTERSECT:
        case IDC_B106_CANCELEXIT:
        case IDC_B107_QUIT:
        		SendMessage(hCommandes1,WM_COMMAND,id,0);
            break;
        // commandes du menu: duplicant les boutons du dialogue infrieur :
        case IDC_RADIOBUTTON101:
        case IDC_RADIOBUTTON102:
        case IDC_RADIOBUTTON103:
        case IDC_RADIOBUTTON104:
        case IDC_RADIOBUTTON105:
        case IDC_RADIOBUTTON106:
        case IDC_RADIOBUTTON107:
            SendMessage(hCommandes2,WM_COMMAND,id,0);
            break;
        //autres commandes du menu :
        //MENU: Clear to zero the whole sector
        case CM_EDITSECTORCLEAR:
            if (MessageBox(hwnd,"Click OK to clear the current sector?",
            	szChildEditWindowTitle,MB_OKCANCEL) == IDCANCEL) break;
				Reset_Edited_Sector(TRUE); //TRUE = mise  zro
            InvalidateRect(hChildEditWindow, NULL, TRUE);
            SetFocus(hChildEditWindow);
            break;
        //MENU: About
        case CM_EDITABOUT:
        		DialogBox(hInstance,"AboutDlg", hwnd, (DLGPROC)AboutDlgProc);
        		//MessageBox(hwnd,"Not Available Yet","",MB_OK);
            break;
        //MENU: Load an atari font
        case CM_LOADATARIFONT:
        		if (Read_Rom_Font(hwnd,TRUE))
            	bOKFont = TRUE;
            InvalidateRect(hChildEditWindow, NULL, TRUE);
            SetFocus(hChildEditWindow);
            break;
        //MENU: Reset Atari font
        case CM_RESETATARIFONT:
            if (Read_Rom_Font(hwnd,FALSE))
            	bOKFont = TRUE;
            InvalidateRect(hChildEditWindow, NULL, TRUE);
            SetFocus(hChildEditWindow);
            break;
        //MENU: Search Hexadecimal
        case CM_SEARCHHEX:
            iSearchMode = 1;
            bSearchInProgress = FALSE;
            Data_Searching(hwnd);
            break;
        //MENU: Search Ascii
        case CM_SEARCHASCII:
            iSearchMode = 2;
            bSearchInProgress = FALSE;
            Data_Searching(hwnd);
            break;
        //MENU: Search Again and VK_F3 message (On_KeyDown)
        case CM_SEARCHAGAIN:
            if (bSearchInProgress)
             	Cont_Data_Searching(hwnd,iCurSect,(UWORD)iDIndex);
            else MessageBox(hwnd,"ARFD Info:\nNo search in progress: please start a new search!",
            	"Search Data :",MB_OK|MB_ICONINFORMATION);
            break;
        //MENU: Replace Hex
        case CM_REPLACEHEX:
            iSearchMode = 3;
            Data_Searching(hwnd);
            break;
        //MENU: Replace Ascii
        case CM_REPLACEASCII:
            iSearchMode = 4;
            Data_Searching(hwnd);
            break;
        //MENU: Suppress the current sector
        case CM_SUPSECTOR:
            if (MessageBox(hwnd,"Click OK to suppress the current sector?\n(the next sectors will be shifted backward)",
            	szChildEditWindowTitle,MB_OKCANCEL|MB_ICONWARNING|MB_DEFBUTTON2) == IDCANCEL) break;
            Shift_Memory(iESPos,0,-(EditSectorSize+1),0);
            Load_Sector_To_Edit();
            wsprintf(szSTR,"Sector $%03X (d%d) has been suppressed and replaced \r\n ",iCurSect,iCurSect);
      	   Add_EndRichText (hMainRText,szSTR);
            InvalidateRect(hChildEditWindow, NULL, TRUE);
            SetFocus(hChildEditWindow);
            break;
        //MENU: Insert a whole sector
        case CM_INSSECTOR:
            if (MessageBox(hwnd,"Click OK to insert a blank sector at the current sector position?\n(the next sectors will be shifted forward)",
            	szChildEditWindowTitle,MB_OKCANCEL|MB_ICONWARNING|MB_DEFBUTTON2) == IDCANCEL) break;
            Shift_Memory(iESPos,0,+(EditSectorSize+1),0);
            memset(lpData,0,256);
            memset(lpModData,0,256);
            wsprintf(szSTR,"Sector $%03X (d%d) has been inserted (empty) \r\n ",iCurSect,iCurSect);
      	   Add_EndRichText (hMainRText,szSTR);
            InvalidateRect(hChildEditWindow, NULL, TRUE);
            SetFocus(hChildEditWindow);
            break;
        //MENU: Insersion Limitation to current sector
        case CM_EDITLIMIT:
        		if ((bEditLimited) && (MessageBox(hwnd,"Are you sure to want to activate the whole file Insertion mode?\n(The whole file data bytes will be shifted)",
            		szChildEditWindowTitle,MB_OKCANCEL|MB_ICONWARNING|MB_DEFBUTTON2) == IDCANCEL)) break;
            bEditLimited = !bEditLimited;
            hM = GetMenu(hwnd);
            Result = bEditLimited ? (MF_BYCOMMAND|MF_CHECKED) : (MF_BYCOMMAND|MF_UNCHECKED);
            CheckMenuItem(hM,CM_EDITLIMIT, Result);
            strcpy(szSTR, bEditLimited ? "Insertion (only in current sector)" : "INSERTION on the WHOLE file !");
            SetDlgItemText(hCommandes2,IDC_RADIOBUTTON104,szSTR);
            break;
        //MENU:
        case CM_SUPBYTE:
            EditSector_Edit_Command(hwnd,2);
            break;
        case CM_INSBYTE:
            EditSector_Edit_Command(hwnd,1);
            break;
        case CM_EDITCOPY:
            EditSector_Edit_Command(hwnd,4);
            break;
        case CM_EDITCUT:
            EditSector_Edit_Command(hwnd,5);
            break;
        case CM_EDITPASTE:
            EditSector_Edit_Command(hwnd,3);
            break;
        case CM_EDITCOPYCB:
            EditSector_Edit_Command(hwnd,6);
            break;
        //MENU: Change Ascii Font:
        case CM_EDITASCIIFONT:
            if (Change_EditSector_Ascii_Font(hwnd))
            {
            	hTheAsciiFont = CreateFontIndirect(lpMyLogFont);
               SetFocus(hCommandes2);
            	SendMessage(hCommandes2,WM_COMMAND,IDC_RADIOBUTTON106,0);
            }
            break;
    }
}

//============================================================================//
// DIALOGBOXES PROCEDURES
//============================================================================//
// Load new/next/Previous sector procedure for toolbar dialogbox
void New_Sector_To_Load(HWND hDlg, char cNS)
{
   int iResult;
   char szSectMod[] = "This sector has been modified:\nClick YES to accept modifications,\nClick NO to abort modifications?";
	if (Is_Sect_Modify())
   {
   	iResult = MessageBox(hDlg, szSectMod, szChildEditWindowTitle,
   		MB_YESNOCANCEL | MB_ICONQUESTION);
   	if (iResult == IDNO)
   		Reset_Edited_Sector(FALSE);
   	else if (iResult == IDCANCEL)
      	return;
      else
      {
         wsprintf(szSTR,"Sector $%03X (d%d) has been manually modified \r\n ",iCurSect,iCurSect);
      	Add_EndRichText (hMainRText,szSTR);
      }
   }
   if (cNS == 'E') // load a new sector
   {
   	strcpy(szEnterSectTitle,"to modify :");
      DialogBox(hInstance,"EnterSectNumber", hDlg,
      	(DLGPROC)EnterSectDlgProc);
   }
   else    	// load next or previous sector
   {
   	if (!bMemAddIsPos)
   	{
   		if (cNS == '+') iMemAdd += (SectSize + 1);
      	else iMemAdd -= (SectSize + 1); //(cNS == '-')
   		iMemAdd &= 0x000FFFFF;
   	}
   	if (cNS == '+') INC_Sect(&iCurSect);
   	else DEC_Sect(&iCurSect); //(cNS == '-')
   }
   Load_Sector_To_Edit();
}
// TOOLBAR dialogbox procedure
BOOL CALLBACK MyDlgProc1(HWND hDlg, UINT Message, WPARAM wParam, LPARAM lParam)
{
    int iResult, i;

    switch(Message)
    {
        case WM_INITDIALOG:
            return FALSE; // No keyboard focus on this dialog box.
        case WM_COMMAND:
        {
            switch(wParam)
            {
                //BUTTON: Restore whole sector
                case IDC_B101_RESTORESECTOR:
                		if (MessageBox(hDlg, "Restore the whole sector (undo all sector modifications)?", szChildEditWindowTitle,
                     	MB_OKCANCEL | MB_ICONQUESTION) != IDOK) break;
                     Reset_Edited_Sector(FALSE);  //FALSE = restore to previous values
                     break;
                //BUTTON: Restore previous byte
                case IDC_B102_RESTOREBYTE:
                    if (bDataSelected)
                    {
                       iResult = MessageBox(hDlg,"Restore all the bytes of the selection?\n(NO will restore the curent caret byte only)",
                       		szChildEditWindowTitle,MB_YESNOCANCEL|MB_ICONQUESTION);
                       if (iResult == IDCANCEL) break;
                       if (iResult == IDYES)
                       {
                       		for (i=nSelStart;i<nSelStop;i++)
                    			{
                       			lpFileBuffer[iESPos+i] = lpData[i];
                       			lpModData[i] = 0;
                    			}
                       		break;
                       }
                    }
                    lpFileBuffer[iESPos+iDIndex] = lpData[iDIndex];
                    lpModData[iDIndex] = 0;
                    break;
                //BUTTON: Previous Sector and VK_PRIOR message (On_KeyDown)
                case IDC_B103_PRIORSECT:
                     New_Sector_To_Load(hDlg,'-');
                     break;
                //BUTTON: Next Sector and VK_NEXT message (On_KeyDown)
                case IDC_B104_NEXTSECT:
                     New_Sector_To_Load(hDlg,'+');
                     break;
                //BUTTON: Enter a new sector number
                case IDC_B105_ENTERSECT:
                     New_Sector_To_Load(hDlg,'E');
                     break;
                //BUTTON: Cancel changes and exit
                case IDC_B106_CANCELEXIT:
                		if (Is_Sect_Modify())
                     	if (MessageBox(hDlg,
                           "This sector has been modified:\nClick OK to abort current sector modifications and exit?",
                        	szChildEditWindowTitle,	MB_OKCANCEL | MB_ICONQUESTION) != IDOK) break;
                     Reset_Edited_Sector(FALSE);
                     // send exit command
                     PostMessage(hChildEditWindow,WM_CLOSE,0,0);
                     break;
                //BUTTON: Quit
                case IDC_B107_QUIT:
                     if (MessageBox(hDlg, "Click OK to Exit?", szChildEditWindowTitle,
                     	MB_OKCANCEL | MB_ICONQUESTION) != IDOK) break;
                     //if (bResult) bFileModified = TRUE; //useless since
                     //bFileModified is set to TRUE during keyboard editing.
                     if (Is_Sect_Modify())
                     {
                         wsprintf(szSTR,"Sector $%03X (d%d) has been manually modified \r\n ",iCurSect,iCurSect);
      	                Add_EndRichText (hMainRText,szSTR);
                     }
                     // send exit command
                     PostMessage(hChildEditWindow,WM_CLOSE,0,0);
                     break;
                //No return nor enddialog :
                case IDOK:
                     //EndDialog(hDlg, IDOK);
                     //return TRUE;
                case IDCANCEL:
                     //EndDialog(hDlg, IDCANCEL);
                     //return TRUE;
                default:
                	   return FALSE;
            }
            //deselection des donnes
            UnSelect_Data();
            //On redessine toute la fentre EditSector
            InvalidateRect(hChildEditWindow, NULL, TRUE);
            //The focus (keyboard input) returns to main edit window
            SetFocus(hChildEditWindow);
        }
    }
    return FALSE;
 }

//----------------------------------------------------------------------------//
// BOTTOM dialogbox procedures
//----------------------------------------------------------------------------//
// Initialisation Radiobuttons of bottom dialogbox:
void hDlg2_InitRadioButton(HWND hDlg)
{
      // RadioBouton Hexa-Ascii:
		Button_SetCheck(GetDlgItem(hDlg,IDC_RADIOBUTTON101),!bEditAscii);
      Button_SetCheck(GetDlgItem(hDlg,IDC_RADIOBUTTON102),bEditAscii);
      // RadioBouton Replacement-Insertion:
      Button_SetCheck(GetDlgItem(hDlg,IDC_RADIOBUTTON103),bReplace);
      Button_SetCheck(GetDlgItem(hDlg,IDC_RADIOBUTTON104),!bReplace);
      // RadioButton Font :
      if (bAtariFont) //ATARI-ASCII-PCOEM
      	CheckRadioButton(hDlg,IDC_RADIOBUTTON105,IDC_RADIOBUTTON107,IDC_RADIOBUTTON105);
      else if (bIbmFont)
         CheckRadioButton(hDlg,IDC_RADIOBUTTON105,IDC_RADIOBUTTON107,IDC_RADIOBUTTON107);
      else
         CheckRadioButton(hDlg,IDC_RADIOBUTTON105,IDC_RADIOBUTTON107,IDC_RADIOBUTTON106);
}
//Reset of the Bottom dialogbox
void hDlg2_Reset(HWND hDlg)
{
     //Radiobutton of the bottom dialog updating
     hDlg2_InitRadioButton(hDlg);
     //Put caret a the iDindex position
     Set_Caret_At_Index_Pos();
     //Update the status bar
     Update_Status();
     //The focus (keyboard input) returns to main edit window
     SetFocus(hChildEditWindow);
}
//The Bottom dialogbox procedure
BOOL CALLBACK MyDlgProc2(HWND hDlg,UINT Message, WPARAM wParam, LPARAM lParam)
{
    int a,b,c;
    int i,Result;
    switch(Message)
    {
        case WM_INITDIALOG:
            // Initialisation Radiobuttons:
            hDlg2_InitRadioButton(hDlg);
            // les valeurs par dfaut dans les champs d'dition:
            SetDlgItemText(hDlg,IDC_EDIT101,"00");     //OFFSET
            wsprintf(szSTR,"%05X",iMemAdd);
            SetDlgItemText(hDlg,IDC_EDIT102,szSTR);  //MEMADD
            SetDlgItemText(hDlg,IDC_EDIT103,"00");     //XOR MASK
            return FALSE; //No keyboard Focus on the first tabstop of this dialog
        case WM_COMMAND:
        {
            switch(wParam)
            {
                //Send by other Procedures to update the fields
                // and the buttons position of the bottom dialogbox
                case IDLG2_RESET:
                    hDlg2_Reset(hDlg);
                    break;
                //Apply button:
                //Sector Display Offset modification:
                case IDC_BUTTON110:
                    GetDlgItemText(hDlg,IDC_EDIT101,szSTR,5);
                    sscanf(szSTR,"%X",&a);
                    wsprintf(szSTR,"%X",a);
                    SetDlgItemText(hDlg,IDC_EDIT101,szSTR);
                    a &= 0x000000FF;
                    uOffset = (UBYTE) a;
                    InvalidateRect(hChildEditWindow, NULL, TRUE);
                    SetFocus(hChildEditWindow);
                    break;
                //Apply button:
                //Memory Display position index modification:
                case IDC_BUTTON111:
                    GetDlgItemText(hDlg,IDC_EDIT102,szSTR,6);
                    sscanf(szSTR,"%X",&a);
                    if (a<0) bMemAddIsPos = TRUE;
                    else bMemAddIsPos = FALSE;
                    if (!bMemAddIsPos)
                    {
                    		a &= 0x000FFFFF;
                    		iMemAdd = a;
                    }
                    else iMemAdd = iCurFilePos;
                    wsprintf(szSTR,"%X",iMemAdd);
                    SetDlgItemText(hDlg,IDC_EDIT102,szSTR);
                    InvalidateRect(hChildEditWindow, NULL, TRUE);
                    SetFocus(hChildEditWindow);
                    break;
                //Apply button:
                //XOR Mask Button:
                case IDC_BUTTON112:
                    GetDlgItemText(hDlg,IDC_EDIT103,szSTR,5);
                    sscanf(szSTR,"%X",&a);
                    wsprintf(szSTR,"%X",a);
                    //b = strlen(szSTR);
                    SetDlgItemText(hDlg,IDC_EDIT103,szSTR);
                    a &= 0x000000FF;
                    if (a == 0)
                    {
                        MessageBox(hDlg,"ARFD Info:\nThe $00 XOR mask does not make any changes!",
                       		szChildEditWindowTitle,MB_OK|MB_ICONINFORMATION);
                        break;
                    }
                    b = 0;
                    c = EditSectorSize + 1;
                    if (bDataSelected)
                       Result = MessageBox(hDlg,"Click YES to Apply XOR mask to the selection?\n(NO will apply to the whole sector)",
                       		szChildEditWindowTitle,MB_YESNOCANCEL|MB_ICONQUESTION);
                    else
                       Result = MessageBox(hDlg,"Click OK to apply the XOR mask to the sector?",
                       		szChildEditWindowTitle,MB_OKCANCEL|MB_ICONQUESTION);
                    if (Result == IDYES)
                    {
                    		b = nSelStart;
                        c = nSelStop;
                    }
                    else if (Result == IDCANCEL) break;

                    for (i=b;i<c;i++)
                    {
                       lpFileBuffer[iESPos+i] ^= (UBYTE) a;
                       lpModData[i] = -1;
                    }
                    bFileModified = TRUE;
                    Display_Table();
                    SetFocus(hChildEditWindow);
                    break;
                //Radiobutton: HEX modification
                case IDC_RADIOBUTTON101:
                    bEditAscii = FALSE;
                    hDlg2_Reset(hDlg);
                    break;
                //Radiobutton: TEXT modification
                case IDC_RADIOBUTTON102:
                    bEditAscii = TRUE;
                    hDlg2_Reset(hDlg);
                    break;
                //Radiobutton: Replacement caret
                case IDC_RADIOBUTTON103:
                    bReplace = TRUE;
                    hDlg2_Reset(hDlg);
                    break;
                //Radiobutton: Insersion caret
                case IDC_RADIOBUTTON104:
                    bReplace = FALSE;
                    hDlg2_Reset(hDlg);
                    break;
                //Radiobutton: Atari Font Char Text:
                case IDC_RADIOBUTTON105:
                    bAtariFont = TRUE;
                    hTheEditFont = GetStockFont(OEM_FIXED_FONT);
                    goto font_exit;
                //Radiobutton: PC OEM Font Char Text:
                case IDC_RADIOBUTTON107:
                    bAtariFont = FALSE;
                    bIbmFont = TRUE;
                    hTheEditFont = GetStockFont(OEM_FIXED_FONT);
                    goto font_exit;
                //Radiobutton: PC ASCII Font Char Text:
                case IDC_RADIOBUTTON106:
                    bAtariFont = FALSE;
                    bIbmFont = FALSE;
                    hTheEditFont = hTheAsciiFont;
font_exit:
                    Update_Font();
                    CalculMaxSize(hChildEditWindow);
                    ShowWindow(hChildEditWindow,SW_SHOW);
                    InvalidateRect(hChildEditWindow, NULL, TRUE);
                    hDlg2_Reset(hDlg);
                    break;
                //No return nor enddialog :
                case IDOK:
                     //EndDialog(hDlg, IDOK);
                     //return TRUE;
                case IDCANCEL:
                     //EndDialog(hDlg, IDCANCEL);
                     //return TRUE
                default :
                    return FALSE;
            }
        }
    }
    return FALSE;
 }

