//******************************************************************************
// NODE NAME : NewCom.cpp
// part of the Atari Rom File Designer source files
// Contient les prcdures et fonctions relatives  la boite de dialogue Communication
// contains all the Communication dialog box functions and procedures
//------------------------------------------------------------------------------
// Written by Philippe VUILLERME (c) PVBest 2001-2002
//
/*---------------------**| ARFD SOURCE REVISION HISTORY |**---------------------
 Date:     | What:
-----------|--------------------------------------------------------------------
 JUNE 2002 | Clean updated source code for ARFD revision 1.9.0 (public realease)
------------------------------------------------------------------------------*/

#define STRICT
#include <windows.h>
#include <windowsx.h>
#include <commctrl.h>
#include <commdlg.h>
#pragma hdrstop
#include <stdio.h>
#include "MesTypes.h"
#include "SerialLib.h"
#include "NewCom.h"
#include "AmDialog.h"
#include "Monitor.h"
#include "ARFDFunctions.h"

//External data

extern UBYTE AtrCOM;
extern UBYTE AtrDDEVIC;
extern int   AtrDTIMEOUT1;
extern int   AtrDTIMEOUT2;

extern 	HWND 		hMainWindow;
extern   HINSTANCE hInstance;
extern 	char 		sTmpSTR[256];
extern 	char * 	lpInt_To_Char(int a, char* lpszTmpStr);
extern 	UBYTE * 	lpFileBuffer;
extern 	char * 	lpsTmpCB;
extern   UWORD  	SectSize;
extern   UWORD  	iCurSect;
extern   UWORD  	iNewSect;
extern   BOOL     bFileMem;
extern   BOOL     bFileModified;
extern   BOOL     bFileResult;
extern   int 	   iFileLength;
extern   TMON_PARAM AmParam;

//External functions

extern 	int  iGet_Hex(char * lpszTmpStr);
extern 	int  iGet_Dec(char * lpszTmpStr);
extern   void Not_Yet_Available(HWND hwnd);
extern   void Set_Com_Default_Config();

//Local data:

static BOOL  bCloseDialog;
static BOOL  bStopCom = FALSE;
static BOOL  bCommand = FALSE;
static BOOL  bWantNewFile = TRUE;
static BOOL  bWantCompare = FALSE;
static BOOL  bWantEnhanced = TRUE;
static BOOL  bWantPut = FALSE;
static BOOL  bFileSaved = FALSE;
static BOOL  bFileDifferent = FALSE;
static TAtariSerialSIOV * acom;
static char  szComError[] = "Communication Error :";
static char  szComStatus[] = "Communication Status :";
static char  szComReading[] = "Read Disk Drive Sectors :";
static char  szComWriting[] = "Write Disk Drive Sectors  :";
static char  szComCompare[] = "Read and Compare Sectors :";
static char  szComFormat[] = "Atari Disk Drive Format :";
static char  szComTitle[] = "Communication with Atari Disk Drive";
static char  szComDialogTitle[256];
static char  szCOM[8];

BOOL 		bSerialConnected;
int  		iSerialCommand;
HWND 		hComDialog;
HWND 		hErrorLog;
HWND 		hReadProgress;
HWND 		hStatusLog;
UWORD 	AtrStartSector;
UWORD 	AtrEndSector;
UWORD 	AtrCurrentSector;
UBYTE 	AtrCKS;
UWORD 	AtrDSIZE;
UBYTE 	AtrErrorNbr;
int		iPourcent;

//Local functions

BOOL     Get_ComDialog_Param(HWND hDialog);
void 		OnSerialError(UBYTE TheError);
void 		OnSerialProgress(UBYTE TheProgress);
void 		Init_Read_Dialog(HWND hDialog);
void 		Set_On_DisConnected_Dialog(HWND hDialog);
void 		Set_On_Connected_Dialog(HWND hDialog);
void 		AddText(HWND hwnd, char * lpszAddString);
BOOL 		Open_Com(HWND hDialog);
void 		Get_Status(void);
void 		Update_Progress(HWND hDialog);
void     Init_Progress(HWND hDialog);
void 		Retreive_Status(void);
void 		Read_Sector();
void 		Load_Read_Sector();
void 		Atari_Message(UBYTE TheError);
BOOL 		Get_Read(HWND hDialog);
void     Command_In_Progress(HWND hDialog);
BOOL     Get_Write(HWND hDialog);
void     Load_Write_Sector();
void     Write_Sector();
void     Verify_Sector();
void     Format_Sector();
BOOL     Get_ReadAndCompare(HWND hDialog);
BOOL     Read_Compare();
void     Set_Com_Title(int pc);
void     Com_Progress_Error(HWND hDlg);
void     Com_Progress_Complete(HWND hDlg);

BOOL CALLBACK _export ComErrorDlgProc(HWND hDlg, UINT Message,
                             WPARAM wParam, LPARAM lParam);

//============================================================================//
//  FUNCTIONS CODE
//============================================================================//
//----------------------------------------------------------------------------//
//Communication DialogBox main dialog procedure :
//----------------------------------------------------------------------------//
BOOL CALLBACK _export ComDlgProc(HWND hDlg, UINT Message,
                             WPARAM wParam, LPARAM lParam)
 {
    BOOL result;
    switch(Message)
    {
        case WM_INITDIALOG:
            bCloseDialog = FALSE;
        		hComDialog = hDlg;
            Init_Read_Dialog(hDlg);
            if (!Open_Com(hDlg)) // Problemes a l'ouverture ?
            	PostMessage(hDlg,WM_COMMAND,IDCANCEL,0); // oui = fermer le dialogue
           	return TRUE;
        case WM_COMMAND:
            switch (LOWORD(wParam))
            {
               case IDCR_CHECKBOX3:
               	bWantPut = !bWantPut;
                  break;
               case IDCR_CHECKBOX4:
               	bWantEnhanced = !bWantEnhanced;
                  break;
               case IDCR_BUTTONSTATUS:   //Send status command to the Atari 1050 disk drive
                  if (!bCommand)
                  {
                  	Init_Progress(hDlg);
                  	bCommand = TRUE;
                  	AddText(hErrorLog,"Disk Drive Status request sending ...\r\n> ");
               		Get_Status();
                  }
                  else
               		Command_In_Progress(hDlg);
                  break;
               case IDCR_BUTTONREAD:   //Send read command to Atari 1050 Disk Drive
                  if (!bCommand)
                  {
                  	if (Get_ComDialog_Param(hDlg)) // Parametres secteurs debut/fin OK ?
                  	{
                        if (!bWantCompare)
                        	result = Get_Read(hDlg);  // initialisation memoire OK ?
                        else
                           result = Get_ReadAndCompare(hDlg);

                     	if (result)
                     	{
                           if (!bWantCompare)
                           {
                              sprintf(sTmpSTR,"Reading sectors $%03X-$%03X and saving into Rom file in memory ",AtrStartSector,AtrEndSector);
                              // Ultime confirmation pour lecture en dernier, c'est  dire ici !
                              strcpy(lpsTmpCB,sTmpSTR);
                              strcat(sTmpSTR,"...\r\n> ");
                              strcat(lpsTmpCB,":\nClick OK to Read Atari Disk Drive sectors?");
      								if (MessageBox(hDlg,lpsTmpCB,szComReading,MB_OKCANCEL | MB_ICONQUESTION) != IDOK)
            							break; // break = on annule la procdure de lecture
                           }
                           else
                           {
                              sprintf(sTmpSTR,"Reading and Comparing sectors $%03X-$%03X to Rom file in memory ",AtrStartSector,AtrEndSector);
                              // Ultime confirmation !
                              strcpy(lpsTmpCB,sTmpSTR);
                              strcat(sTmpSTR,"...\r\n> ");
                              strcat(lpsTmpCB,":\nClick OK to launch the Comparison of the Atari Disk Drive sectors?");
      								if (MessageBox(hDlg,lpsTmpCB,szComCompare,MB_OKCANCEL | MB_ICONQUESTION) != IDOK)
                              	break; // break = on annule la procdure de comparaison
                              bFileDifferent = FALSE;
                           }
      							AddText(hErrorLog,sTmpSTR);
                           bCommand = TRUE;
                  			Read_Sector();
                     	}
                  	}
                  }
                  else
               		Command_In_Progress(hDlg);
                  break;
               case IDCR_BUTTONWRITE:   //Send Write command to Atari 1050 Disk Drive
                  if (!bCommand)
                  {
                  	if (Get_ComDialog_Param(hDlg)) // Parametres secteurs debut/fin OK ?
                  	{
                     	if (Get_Write(hDlg))  // initialisation memoire OK ?
                     	{
                        	sprintf(sTmpSTR,"Writing sectors $%03X-$%03X with data from Rom file in memory ",AtrStartSector,AtrEndSector);
                           // Ultime confirmation !
                           strcpy(lpsTmpCB,sTmpSTR);
                           strcat(sTmpSTR,"...\r\n> ");
                           strcat(lpsTmpCB,":\nARFD WARNING : All the data on the Atari disk will be erased and changed!\nClick OK to Write Atari Disk Drive sectors?");
      							if (MessageBox(hDlg,lpsTmpCB,szComWriting,MB_OKCANCEL | MB_ICONWARNING | MB_DEFBUTTON2) != IDOK)
         							break; // break = on annule la procdure d'criture
      							AddText(hErrorLog,sTmpSTR);
                           bCommand = TRUE;
                           Load_Write_Sector();
                  			Write_Sector();
                     	}
                  	}
                  }
                  else
               		Command_In_Progress(hDlg);
                  break;
               case IDCR_BUTTONVERIFY:   //verify sector command to Atari 1050 Disk Drive
               	Not_Yet_Available(hDlg);
                  /*if (!bCommand)
                  {
                  	if (Get_ComDialog_Param(hDlg)) // Parametres secteurs debut/fin OK ?
                  	{
                        	bCommand = TRUE;
                        	sprintf(sTmpSTR,"Verifying Sectors ($%03X-$%03X) ...\r\n> ",AtrStartSector,AtrEndSector);
      							AddText(hErrorLog,sTmpSTR);
                  			Verify_Sector();
                  	}
                  }
                  else
               		Command_In_Progress(hDlg);*/
                  break;
               case IDCR_BUTTONFORMAT:   //format disk command to Atari 1050 Disk Drive
                  if (!bCommand)
                  {
                     sprintf(sTmpSTR,"Formating an Atari Disk on Drive - ");
                     if (bWantEnhanced)
                     	strcat(sTmpSTR,"Enhanced (130Kb, $410 sectors) ");
                     else
                     	strcat(sTmpSTR,"Normal (90Kb, $2D0 sectors) ");
                     // confirmation ultime
                     strcpy(lpsTmpCB,sTmpSTR);
                     strcat(sTmpSTR,"...\r\n> ");
                     strcat(lpsTmpCB,":\nARFD WARNING : All the data on the Atari disk will be destroyed!\nClick OK to Format the Atari floppy disk in Drive?");
                  	if (MessageBox(hDlg, lpsTmpCB, szComFormat,
                     	MB_OKCANCEL | MB_ICONWARNING | MB_DEFBUTTON2) != IDOK)
                        	break;
                     AddText(hErrorLog,sTmpSTR);
                     Init_Progress(hDlg);
                     bCommand = TRUE;
                     Format_Sector();
                  }
                  else
               		Command_In_Progress(hDlg);
                  break;
               case IDCR_STOPCOM:  //Stop  and close COM Port Button
                  // AtrCurrentSector = (UWORD) (AtrEndSector - 1);
                  bStopCom = TRUE;
                  sprintf(sTmpSTR,"%s Stop/Reset user's Request ...\r\n> ",szCOM);
                  AddText(hErrorLog,sTmpSTR);
               	acom->AtrSerial_Disconnect();
                  break;
               case IDCR_REOPEN:   //Reconnect (reopen) COM port Button
                  sprintf(sTmpSTR,"%s Re-Open user's Request ...\r\n> ",szCOM);
                  AddText(hErrorLog,sTmpSTR);
               	Open_Com(hDlg);
                  bStopCom = FALSE;
                  break;
               case IDCR_SAVEFILE:  // Save File Button
                  SendMessage(hMainWindow, WM_COMMAND, CMR_SAVEFILE, 0);
                  if (bFileResult) // La commande n'a pas chou !
                  {
                  	bFileSaved = TRUE;
                 		sprintf(sTmpSTR,"File saved into '%s' at %d Kb length!\r\n> ",AmParam.FName,iFileLength/1024);
                  	AddText(hErrorLog,sTmpSTR);
                  	Set_Com_Title(-1);
                  }
                  SetFocus(hDlg);
                  break;
               case IDCR_OPENFILE:  // Open File Button
                  SendMessage(hMainWindow, WM_COMMAND, CMR_OPENFILE, 0);
                  if (bFileResult) // La commande n'a pas chou !
                  {
                  	bFileSaved = TRUE;
                  	sprintf(sTmpSTR,"File '%s' open at %d Kb length!\r\n> ",AmParam.FName,iFileLength/1024);
                  	AddText(hErrorLog,sTmpSTR);
                  	Set_Com_Title(-1);
                  }
                  SetFocus(hDlg);
                  break;
               case IDCR_VIEWFRAME:
                  Not_Yet_Available(hDlg);
                  break;
               case CM_ONDISCONNECTED:  // Connected  result message
                  Set_On_DisConnected_Dialog(hDlg);
                  bCommand = FALSE;
         			sprintf(sTmpSTR,"ARFD Info: Port %s Disconnected!",szCOM);
         			MessageBox(hDlg,sTmpSTR,szComStatus,MB_OK);
         			strcat(sTmpSTR,"\r\n> ");
      				AddText(hErrorLog,sTmpSTR);
         			if (bCloseDialog) // fermeture dialogue demande ?
                  	PostMessage(hDlg,WM_COMMAND,CM_CLOSEDIALOG,0);
                  break;
               case CM_ONCONNECTED:    // Connected  result message
                  Set_On_Connected_Dialog(hDlg);
         			sprintf(sTmpSTR,"ARFD Info: Port %s Connected!",szCOM);
         			MessageBox(hComDialog,sTmpSTR,szComStatus,MB_OK);
         			strcat(sTmpSTR,"\r\n> ");
      				AddText(hErrorLog,sTmpSTR);
               	break;
               case CM_ONPROGRESSERROR:
               	Com_Progress_Error(hDlg);
                  break;
					case CM_ONPROGRESSCOMPLETE:
               	Com_Progress_Complete(hDlg);
               	break;
               case IDOK:   			// Close BUTTON
               	if (bSerialConnected)
                  {
               		bCloseDialog = TRUE; // fermeture dialogue demande
                  	acom->AtrSerial_Disconnect();
                     break;
                  }
                  //else IDCANCEL :
               case IDCANCEL:      		// CANCEL CLOSE WINDOW
               case CM_CLOSEDIALOG:    //
                  delete acom;
                  acom = NULL;
                	EndDialog(hDlg, IDOK);
                  hComDialog = NULL;
                  bCloseDialog = FALSE;
                  iCurSect = 1;
                	return TRUE;
            }
    }
    return FALSE;
}

//----------------------------------------------------------------------------//
// Autres Procedures Principales
//----------------------------------------------------------------------------//
//Adding text into the display edit window of the dialogbox
void AddText(HWND hwnd, char * lpszAddString)
 {
    int cpMin,cpMax;
    cpMin = SendMessage(hwnd,WM_GETTEXTLENGTH,0,0);
    cpMax = cpMin;
    SendMessage(hwnd, EM_SETSEL, cpMin, cpMax);
    SendMessage(hwnd, EM_REPLACESEL, 0, (LPARAM)lpszAddString);
 }

//Read_Complete function when current sector Reading is complete
void Read_Complete(HWND hDlg)
{
	// Read Complete command result message :
   	Update_Progress(hDlg);
      if (!bWantCompare)
      	Load_Read_Sector();
      else
      {
      	if (!Read_Compare())
      	{
            bFileDifferent = TRUE;
      		sprintf(sTmpSTR,"Sectors $%03X (%d) are different!",AtrCurrentSector,AtrCurrentSector);
      		strcpy(lpsTmpCB,sTmpSTR);
      		strcat(sTmpSTR,"\r\n> ");
      		AddText(hErrorLog,sTmpSTR);
      		strcat(lpsTmpCB,"\nClick OK to continue comparison?");
      		if (MessageBox(hDlg,lpsTmpCB,szComCompare,MB_OKCANCEL|MB_ICONINFORMATION) != IDOK)
      		{
      			sprintf(sTmpSTR,"Read and Compare sectors aborted by the user\r\n> ");
      			AddText(hErrorLog,sTmpSTR);
               bFileDifferent = FALSE;
      			bCommand = FALSE;
      			return;
      		}
      	}
      }
      if (!bSerialConnected || bStopCom) return;
      // si 'deconnection' ne pas redemmander un secteur
      AtrCurrentSector ++;
      if (AtrCurrentSector > AtrEndSector)
      {
      	SetForegroundWindow(hMainWindow);
      	if (!bWantCompare)
      	{
      		sprintf(sTmpSTR,"Read Atari Disk Drive sectors completed!");
      		MessageBox(hDlg,sTmpSTR,szComReading,MB_OK);
      	}
      	else
      	{
      		sprintf(sTmpSTR,"Read and Compare sectors completed: ");
            if (bFileDifferent)
            {
            	strcat(sTmpSTR,"some differences have been found!");
               MessageBox(hDlg,sTmpSTR,szComCompare,MB_OK|MB_ICONINFORMATION);
            }
            else
            {
               strcat(sTmpSTR,"all the data are identical");
               MessageBox(hDlg,sTmpSTR,szComCompare,MB_OK);
            }
            bFileDifferent = FALSE;
      	}
      	strcat(sTmpSTR,"\r\n> ");
      	AddText(hErrorLog,sTmpSTR);
      	bCommand = FALSE;
      }
      else
      	Read_Sector();
}

//Write_Complete function when current sector Writting is complete
void Write_Complete(HWND hDlg)
{
	// Write Complete command result message :
   	Update_Progress(hDlg);
      if (!bSerialConnected || bStopCom) return; // si 'deconnection' ne pas redemmander un secteur
      AtrCurrentSector ++;
      if (AtrCurrentSector > AtrEndSector)
      {
      	sprintf(sTmpSTR,"Write Atari Disk Drive sectors completed!");
      	MessageBox(hDlg,sTmpSTR,szComWriting,MB_OK);
      	strcat(sTmpSTR,"\r\n> ");
      	AddText(hErrorLog,sTmpSTR);
      	bCommand = FALSE;
      }
      else
      {
      	Load_Write_Sector();
      	Write_Sector();
      }
}

//----------------------------------------------------------------------------//

//----------------------------------------------------------------------------//

void OnSerialError(UBYTE TheError)
{
    AtrErrorNbr = TheError;
    if (TheError == 1)
    {
    	sprintf(sTmpSTR,
      	" ARFD internal COM PORT Error %i: The COM port will be closed",
      	TheError);
      Handle_System_Error(hComDialog,szComError,sTmpSTR,acom->dwGetLastError);
    	//MessageBox(hComDialog,sTmpSTR,szComError,MB_OK|MB_ICONERROR);
    	strcat(sTmpSTR,"\r\n> ");
    	AddText(hErrorLog,sTmpSTR);
    }
    // Others Errors are Atari Protocol Error and are displayed  in next ''OnSerialProgress(4) '' function
    //else
    //{
     // sprintf(sTmpSTR," ATARI Protocol ERROR: %d with DERROR = %02X",TheError,acom->DERROR);
     // if (iSerialCommand != 0x52) // sera trait dans progress...
     //		MessageBox(hComDialog,sTmpSTR,szComError,MB_OK|MB_ICONWARNING);
     //	strcat(sTmpSTR,"\r\n");
     //	AddText(hErrorLog,sTmpSTR);
    //}
    //return;
}
void Com_Progress_Error(HWND hDlg)
{
	int result;
      	sprintf(lpsTmpCB,"Atari protocol error number %d :\r\n  ",AtrErrorNbr);
      	Atari_Message(acom->DERROR); //ajoute via sTmpSTR le libell de l'erreur
      	strcat(lpsTmpCB,sTmpSTR);    // c'est dans lpsTmpCB
         SetForegroundWindow(hMainWindow);
        	if (iSerialCommand == 0x52)
        	{
            sprintf(sTmpSTR,"  during the reading of sector $%03X (%d)",AtrCurrentSector,AtrCurrentSector);
            strcat(lpsTmpCB,sTmpSTR);

            result = DialogBox(hInstance,"ComErrorDlg", hDlg, (DLGPROC)ComErrorDlgProc);

            strcat(lpsTmpCB,"\r\n> ");
            AddText(hErrorLog,lpsTmpCB);

         	switch (result)
            {
            	case IDRETRY:
                  sprintf(sTmpSTR,"Sector $%03X (%d) reading retry ...\r\n> ",AtrCurrentSector,AtrCurrentSector);
                  AddText(hErrorLog,sTmpSTR);
               	//PostMessage(hDlg,WM_COMMAND,CM_RETRYREAD, 0);
                  //case CM_RETRYREAD:   // Read Retry command result message :
                  Read_Sector();
                  ////////////////////
                  break;
               case IDCR_BUTTONBAD:
               	//Make Bad in Sector (fill acom->inbuffer with bad)
         			acom->AtrSerial_WriteBadSector();
            		sprintf(sTmpSTR,"ARFD Info: Bad Sector written at sector $%03X (%d)",AtrCurrentSector,AtrCurrentSector);
            		MessageBox(hDlg,sTmpSTR,szComReading,MB_OK|MB_ICONINFORMATION);
            		strcat(sTmpSTR,"\r\n> ");
      	   		AddText(hErrorLog,sTmpSTR);
                  //PostMessage(hDlg,WM_COMMAND,CM_READCOMPLETE, 0);
                  //////////////////////
                  Read_Complete(hDlg);
                  //////////////////////
                  break;
               case IDIGNORE:
                  sprintf(sTmpSTR,"OK : Skiping the reading of sector $%03X (%d) ...\r\n> ",AtrCurrentSector,AtrCurrentSector);
      	   		AddText(hErrorLog,sTmpSTR);
         			//PostMessage(hDlg,WM_COMMAND,CM_READCOMPLETE, 0);
                  //////////////////////
                  Read_Complete(hDlg);
                  //////////////////////
                  break;
               case IDCANCEL:
               case IDABORT:
               	sprintf(sTmpSTR,"Read Atari Disk Drive sectors aborted on sector $%03X (%d)!",AtrCurrentSector,AtrCurrentSector);
            		MessageBox(hDlg,sTmpSTR,szComReading,MB_OK|MB_ICONINFORMATION);
               	strcat(sTmpSTR,"\r\n> ");
      	   		AddText(hErrorLog,sTmpSTR);
                  bCommand = FALSE;
            }
        }
        else if (iSerialCommand == 0x57)
        {
            sprintf(sTmpSTR,"  during the writing of sector $%03X (%d)",AtrCurrentSector,AtrCurrentSector);
            strcat(lpsTmpCB,sTmpSTR);
            result = MessageBox(hDlg,lpsTmpCB,szComWriting,MB_ABORTRETRYIGNORE | MB_ICONWARNING | MB_DEFBUTTON2);
            strcat(lpsTmpCB,"\r\n> ");
            AddText(hErrorLog,lpsTmpCB);
            if (result == IDIGNORE)
               //PostMessage(hDlg,WM_COMMAND,CM_WRITECOMPLETE, 0);
               //////////////////////////
               Write_Complete(hDlg);
               /////////////////////////
            else if (result == IDRETRY)
            {
            	sprintf(sTmpSTR,"Sector $%03X (%d) writing retry ...\r\n> ",AtrCurrentSector,AtrCurrentSector);
            	AddText(hErrorLog,sTmpSTR);
               //PostMessage(hDlg,WM_COMMAND,CM_RETRYWRITE, 0);
               //case CM_RETRYWRITE:   // Read Retry command result message :
               /////////////////////////
               Load_Write_Sector();
               Write_Sector();
               /////////////////////////
            }
            else
            {
            	sprintf(sTmpSTR,"Write Atari Disk Drive sectors aborted on sector $%03X (%d)!",AtrCurrentSector,AtrCurrentSector);
               MessageBox(hDlg,sTmpSTR,szComWriting,MB_OK|MB_ICONINFORMATION);
               strcat(sTmpSTR,"\r\n> ");
               AddText(hErrorLog,sTmpSTR);
               bCommand = FALSE;
            }
        }
        else
        {
         	MessageBox(hDlg,lpsTmpCB,szComError, MB_OK | MB_ICONWARNING);
            strcat(lpsTmpCB,"> ");
            AddText(hErrorLog,lpsTmpCB);
            bCommand = FALSE;
        }
}
void Com_Progress_Complete(HWND hDlg)
{
         switch (iSerialCommand)
         {
         	case 0x53:
               //PostMessage(hDlg,WM_COMMAND,CM_RSTATUS,0);
               //case CM_RSTATUS:   // Status command result message :
               ///////////////////////////////////
                  sprintf(sTmpSTR,"Atari Disk Drive Status received!");
                  MessageBox(hDlg,sTmpSTR,szComStatus,MB_OK);
                  strcat(sTmpSTR,"\r\n> ");
                  AddText(hErrorLog,sTmpSTR);
                  Retreive_Status();
    					SetDlgItemText(hDlg,IDCR_EDIT3,"1");
    					SetDlgItemText(hDlg,IDCR_EDIT4,lpInt_To_Char(AtrDSIZE,sTmpSTR));
                  bCommand = FALSE;
               break;
            case 0x52:
               //sprintf(sTmpSTR,"OK Sector $%02X (%d) read !\r\n",AtrCurrentSector,AtrCurrentSector);
               //AddText(hErrorLog,sTmpSTR);
               AtrCKS = acom->abuff.RCHKSUM;
               //PostMessage(hDlg,WM_COMMAND,CM_READCOMPLETE,0);
               //////////////////////////////////
               Read_Complete(hDlg);
               break;
            case 0x50:
            case 0x57:
               AtrCKS = acom->abuff.outbuffer[128];
               //PostMessage(hDlg,WM_COMMAND,CM_WRITECOMPLETE,0);
               /////////////////////////////////
               Write_Complete(hDlg);
               break;
            case 0x21:
            case 0x56:
            	sprintf(sTmpSTR,"Atari Disk Format Completed!");
               MessageBox(hDlg,sTmpSTR,szComFormat,MB_OK);
               strcat(sTmpSTR,"\r\n> ");
               AddText(hErrorLog,sTmpSTR);
               bCommand = FALSE;
               break;
         }
}
void OnSerialProgress(UBYTE TheStatus)
{
    switch (TheStatus)
    {
      case 0: // DisConnected
         bSerialConnected = FALSE;
         PostMessage(hComDialog,WM_COMMAND,CM_ONDISCONNECTED,0);
         break;
    	case 1: // Connected !
    		bSerialConnected = TRUE;
         PostMessage(hComDialog,WM_COMMAND,CM_ONCONNECTED,0);
         break;
      case 2: // Communication in progress... (start command)
         // PostMessage(hComDialog,WM_COMMAND,CM_PROGRESSSTARTED,0);
        	// AddText(hErrorLog,"Communication in progress...\r\n");
         return;
      case 4:  // Atari Protocol Error
         PostMessage(hComDialog,WM_COMMAND,CM_ONPROGRESSERROR,0);
         break;
      case 8: // Command Completed !!
         PostMessage(hComDialog,WM_COMMAND,CM_ONPROGRESSCOMPLETE,0);
         break;
      default :
         sprintf(sTmpSTR,"ARFD internal Error: Unknown Progress Code %d :\r\n  ",TheStatus);
         MessageBox(hComDialog,sTmpSTR,szComError,MB_OK|MB_ICONSTOP);
         break;
    }
}

//----------------------------------------------------------------------------//
//****************************************************************************
//   Procedures de communications avec Atari Disk Drive
//****************************************************************************
//----------------------------------------------------------------------------//

void Init_Read_Dialog(HWND hDialog)
{
   //SetClassLong(hDlg, GCL_HICON, (LONG) LoadIcon(hInstance, "MonIcone"));
   HMENU SysMenu = GetSystemMenu (hDialog, FALSE);
   EnableMenuItem(SysMenu,2,MF_BYPOSITION|MF_GRAYED);
	EnableMenuItem(SysMenu,4,MF_BYPOSITION|MF_GRAYED);
   Set_Com_Title(-1);
   acom = new TAtariSerialSIOV;
   acom->SetFunctionAtariOnProgress (OnSerialProgress);
   acom->SetFunctionAtariOnError    (OnSerialError);
   bSerialConnected  = FALSE;
   AtrDSIZE = (UWORD) 0x2D0; // 720 secteurs par dfaut
   hErrorLog = GetDlgItem(hDialog,IDCR_EDIT2);
   hStatusLog = GetDlgItem(hDialog,IDCR_EDIT1);
   hReadProgress = GetDlgItem(hDialog,IDCR_PROGRESS1);
   sprintf(szCOM,"COM%d",AtrCOM);
   sprintf(sTmpSTR,"> Opening %s Port [19200 Bauds, 8 bit data, 1 bit stop, no parity]...\r\n> ",szCOM);
   AddText(hErrorLog,sTmpSTR);
	SetDlgItemText(hDialog,IDCR_EDIT2,sTmpSTR);
   SetDlgItemText(hDialog,IDCR_EDIT1,"Please press 'Drive Status' Button to Get Drive Status Information...");
   SetDlgItemText(hDialog,IDCR_EDIT3,"1");
   SetDlgItemText(hDialog,IDCR_EDIT4,"2D0");
   SetDlgItemText(hDialog,IDCR_STATIC1,"");
   SetDlgItemText(hDialog,IDCR_STATIC2,"");
   CheckDlgButton(hDialog, IDCR_CHECKBOX1, ((bWantNewFile) ? BST_CHECKED : BST_UNCHECKED));
   CheckDlgButton(hDialog, IDCR_CHECKBOX2, ((bWantCompare) ? BST_CHECKED : BST_UNCHECKED));
   CheckDlgButton(hDialog, IDCR_CHECKBOX3, ((bWantPut) ? BST_CHECKED : BST_UNCHECKED));
   CheckDlgButton(hDialog, IDCR_CHECKBOX4, ((bWantEnhanced) ? BST_CHECKED : BST_UNCHECKED));
   Set_On_DisConnected_Dialog(hDialog);
}
void Set_On_DisConnected_Dialog(HWND hDialog)
{
	EnableWindow(GetDlgItem(hDialog,IDCR_STOPCOM),FALSE);
	EnableWindow(GetDlgItem(hDialog,IDCR_REOPEN),TRUE);
	EnableWindow(GetDlgItem(hDialog,IDCR_BUTTONSTATUS),FALSE);
	EnableWindow(GetDlgItem(hDialog,IDCR_BUTTONREAD),FALSE);
	EnableWindow(GetDlgItem(hDialog,IDCR_BUTTONWRITE),FALSE);
	EnableWindow(GetDlgItem(hDialog,IDCR_BUTTONVERIFY),FALSE);
	EnableWindow(GetDlgItem(hDialog,IDCR_BUTTONFORMAT),FALSE);
}
void Set_On_Connected_Dialog(HWND hDialog)
{
   EnableWindow(GetDlgItem(hDialog,IDCR_STOPCOM),TRUE);
   EnableWindow(GetDlgItem(hDialog,IDCR_REOPEN),FALSE);
   EnableWindow(GetDlgItem(hDialog,IDCR_BUTTONSTATUS),TRUE);
   EnableWindow(GetDlgItem(hDialog,IDCR_BUTTONREAD),TRUE);
   EnableWindow(GetDlgItem(hDialog,IDCR_BUTTONWRITE),TRUE);
   EnableWindow(GetDlgItem(hDialog,IDCR_BUTTONVERIFY),TRUE);
   EnableWindow(GetDlgItem(hDialog,IDCR_BUTTONFORMAT),TRUE);
}
BOOL Open_Com(HWND hDialog)
{
	int r;
   acom->DTIMEOUT1 = AtrDTIMEOUT1;
   acom->DTIMEOUT2 = AtrDTIMEOUT2 * 1000;
   acom->DDEVIC = (UBYTE) (0x30 + AtrDDEVIC);
   sprintf(sTmpSTR,"Atari Serial Com to Disk Drive #%d ($%02X)\r\n> ",AtrDDEVIC,acom->DDEVIC);
   AddText(hErrorLog,sTmpSTR);
   sprintf(sTmpSTR,"Atari Serial Com DTIMEOUT1 : %d ms, DTIMEOUT2 : %d ms\r\n> ",acom->DTIMEOUT1,acom->DTIMEOUT2);
   AddText(hErrorLog,sTmpSTR);
   r = acom->AtrSerial_ConnectInit(szCOM);
   switch (r)
   {
      case 0:
      	return TRUE;
   	case 1:
      	sprintf(sTmpSTR," ARFD internal Error %d : %s COMMMASK SETTING FAILED !",r,szCOM);
         break;
      case 2:
         sprintf(sTmpSTR," ARFD internal Error %d : %s COMMTIMEOUT SETTING FAILED !",r,szCOM);
         break;
      case 4:
         sprintf(sTmpSTR," ARFD internal Error %d : %s DCB SETTING FAILED !",r,szCOM);
         break;
      case 8:
         sprintf(sTmpSTR," ARFD internal Error %d : %s INVALID COM HANDLE NAME !",r,szCOM);
         if (acom->dwGetLastError == 5) strcat(sTmpSTR,"\n(This Serial Com Port is already open by an other process)");
         if (acom->dwGetLastError == 2) strcat(sTmpSTR,"\n(This Serial Com Port Connector does not exist on computer)");
         break;
      case 16:
      	sprintf(sTmpSTR," ARFD internal Error %d : COM PORT NAME STRING NULL !",r);
         break;
      default:
         sprintf(sTmpSTR," ARFD internal unknown Error %d on %s",r,szCOM);
         break;
   }
   Handle_System_Error(hDialog, "Communication Port Opening :", sTmpSTR, acom->dwGetLastError);
   //MessageBox(hDialog,sTmpSTR,"Communication Port Opening :",MB_OK|MB_ICONSTOP);
   return FALSE;
}
void Get_Status(void)
{
   iSerialCommand = 0x53;
	acom->DCOMND = 0x53;
   acom->DAUX1 = 0;
   acom->DAUX2 = 0;
   acom->AtrSerial_SendSIOVCommand();
}
void Read_Sector()
{
   iSerialCommand = 0x52;
	acom->DCOMND = 0x52;
   acom->DAUX1 = (UBYTE) (AtrCurrentSector & 0x00FF);
   acom->DAUX2 = (UBYTE) (AtrCurrentSector >> 8);
   acom->AtrSerial_SendSIOVCommand();
}
void Write_Sector()
{
   iSerialCommand = 0x57;
      if (bWantPut)
		acom->DCOMND = 0x50;
   else
   	acom->DCOMND = 0x57;
   acom->DAUX1 = (UBYTE) (AtrCurrentSector & 0x00FF);
   acom->DAUX2 = (UBYTE) (AtrCurrentSector >> 8);
   acom->AtrSerial_SendSIOVCommand();
}
void Verify_Sector()
{
   iSerialCommand = 0x56;
	acom->DCOMND = 0x56;
   acom->DAUX1 = (UBYTE) (AtrCurrentSector & 0x00FF);
   acom->DAUX2 = (UBYTE) (AtrCurrentSector >> 8);
   acom->AtrSerial_SendSIOVCommand();
}
void Format_Sector()
{
   iSerialCommand = 0x21;
   if (bWantEnhanced)
		acom->DCOMND = 0x22;
   else
   	acom->DCOMND = 0x21;
   acom->DAUX1 = 0;
   acom->DAUX2 = 0;
   acom->AtrSerial_SendSIOVCommand();
}
void Retreive_Status()
{
    UBYTE DVSTATS = acom->abuff.inbuffer[0];
    UBYTE DVSTATH = acom->abuff.inbuffer[1];
    UBYTE DVSTAT1 = acom->abuff.inbuffer[2];
    UBYTE DVSTAT2 = acom->abuff.inbuffer[3];

    SetWindowText(hStatusLog,"ATARI DISK DRIVE STATUS :\r\n");
    sprintf(sTmpSTR,"- STATUS FRAME : $%02X $%02X $%02X $%02X\r\n",DVSTATS,DVSTATH,DVSTAT1,DVSTAT2);
    AddText(hStatusLog,sTmpSTR);

    sprintf(sTmpSTR,"- ATARI DRIVE ");
    AddText(hStatusLog,sTmpSTR);
    if (!(DVSTATS | DVSTATH))
      {  //les 2 octets de status sont a 0
         sprintf(sTmpSTR,"NOT READY OR NOT CONNECTED");
         AddText(hStatusLog,sTmpSTR);
      }
    else
      {
         if (!(DVSTATH & 0x80))
         { // if the bit #7 of second byte is 0
           sprintf(sTmpSTR,"STATUS = NO DISK IN DRIVE");
           AddText(hStatusLog,sTmpSTR);
         }
         else 
         {
           sprintf(sTmpSTR,"STATUS = DISK READY\r\n");
           AddText(hStatusLog,sTmpSTR);

           if (DVSTATS & 0x08) sprintf(sTmpSTR,"- DISK is WRITE PROTECTED\r\n");
           else sprintf(sTmpSTR,"- DISK is WRITABLE\r\n");
           AddText(hStatusLog,sTmpSTR);

           if (!(DVSTATS & 0x80)) 
           {
               AtrDSIZE = 720;
               sprintf(sTmpSTR,"- DISK is SINGLE DENSITY\r\n   $2D0 (720) sectors FORMATED\r\n");
           }
           else
           {
               AtrDSIZE = 1040;
               sprintf(sTmpSTR,"- DISK is ENHANCED DENSITY\r\n   $410 (1040) sectors FORMATED\r\n");
           }
           AddText(hStatusLog,sTmpSTR);
          }
       }
}
void Init_Progress(HWND hDialog)
{
    SendMessage(hReadProgress,PBM_SETRANGE,0,MAKELPARAM(AtrStartSector,AtrEndSector+1));
    SendMessage(hReadProgress,PBM_SETSTEP, 1, 0);
    SendMessage(hReadProgress,PBM_SETPOS, 0, 0);
    iPourcent = -1;
    SetDlgItemText(hDialog,IDCR_STATIC3,"0%");
    Set_Com_Title(-1);

}
BOOL Get_ComDialog_Param(HWND hDialog)
{
    if (GetDlgItemText(hDialog,IDCR_EDIT3,sTmpSTR,5))
           AtrStartSector = (UWORD) iGet_Hex(sTmpSTR);
    else AtrStartSector = 0;
    if ((AtrStartSector > AtrDSIZE) || (AtrStartSector < 1))
    {
        		MessageBox(hDialog, "ARFD input Error:\nInvalid start sector!",
     				szComReading,MB_OK | MB_ICONWARNING);
        		return FALSE;
    }
    //if (IsDlgButtonChecked(hDialog,IDCR_RADIOBUTTON62) & BST_CHECKED)
    if (GetDlgItemText(hDialog,IDCR_EDIT4,sTmpSTR,5))
           AtrEndSector = (UWORD) iGet_Hex(sTmpSTR);
    else AtrEndSector = 0;
    if ((AtrEndSector > AtrDSIZE) || (AtrEndSector < AtrStartSector))
    {
        		MessageBox(hDialog, "ARFD input Error:\nInvalid end sector!",
     				szComReading,MB_OK | MB_ICONWARNING);
        		return FALSE;
    }
    bWantNewFile = (IsDlgButtonChecked(hDialog,IDCR_CHECKBOX1) & BST_CHECKED) ? TRUE : FALSE;
    bWantCompare = (IsDlgButtonChecked(hDialog,IDCR_CHECKBOX2) & BST_CHECKED) ? TRUE : FALSE;
    bWantPut = (IsDlgButtonChecked(hDialog,IDCR_CHECKBOX3) & BST_CHECKED) ? TRUE : FALSE;
    bWantEnhanced = (IsDlgButtonChecked(hDialog,IDCR_CHECKBOX4) & BST_CHECKED) ? TRUE : FALSE;
    SetDlgItemText(hDialog,IDCR_EDIT3,lpInt_To_Char(AtrStartSector,sTmpSTR));
    SetDlgItemText(hDialog,IDCR_EDIT4,lpInt_To_Char(AtrEndSector,sTmpSTR));
    Init_Progress(hDialog);
    return TRUE;
}
void Update_Progress(HWND hDialog)
{
   int p;
   sprintf(sTmpSTR,"$%03X (%d)",AtrCurrentSector,AtrCurrentSector);
   SetDlgItemText(hDialog,IDCR_STATIC1,sTmpSTR);
   sprintf(sTmpSTR,"$%02X",AtrCKS);
   SetDlgItemText(hDialog,IDCR_STATIC2,sTmpSTR);
   SendMessage(hReadProgress, PBM_STEPIT, 0, 0);
   p = 100 * (AtrCurrentSector +1 - AtrStartSector)/(AtrEndSector + 1 - AtrStartSector);
   if (p != iPourcent)
   {
   	sprintf(sTmpSTR,"%d%%",p);
   	SetDlgItemText(hDialog,IDCR_STATIC3,sTmpSTR);
   	Set_Com_Title(p);
      iPourcent = p;
   }
}
void Load_Read_Sector()
{
    UINT pos;
    int c;
    pos = Get_FilePosition(AtrCurrentSector,0);
    for (c = 0 ; c < 128; c ++)
    {
        lpFileBuffer[pos+c] = acom->abuff.inbuffer[c];
    }
}
void Load_Write_Sector()
{
    UINT pos;
    int c;
    pos = Get_FilePosition(AtrCurrentSector,0);
    for (c = 0 ; c < 128; c ++)
    {
        acom->abuff.outbuffer[c] = lpFileBuffer[pos+c];
    }
}
BOOL Read_Compare()
{
    UINT pos;
    int c;
    pos = Get_FilePosition(AtrCurrentSector,0);
    for (c = 0 ; c < 128; c ++)
    {
        if (acom->abuff.inbuffer[c] != lpFileBuffer[pos+c])
        	return FALSE;
    }
    return TRUE;
}
void Atari_Message(UBYTE TheError)
{
    switch (TheError) //code d'erreur pour echec communication
    {
        case 138:
            sprintf(sTmpSTR,"TIME OUT REACHED : ATARI ERROR $8A (138) !\r\n  Verify your hardware connection cable or Atari Disk Drive is Off...\r\n");
            break;
        case 143:
            sprintf(sTmpSTR,"NO VALID CHECK SUM : ATARI ERROR $8F (143) !\r\n  Error in serial communication with Atari Disk Drive...\r\n");
            break;
        case 144:
         	sprintf(sTmpSTR,"ATARI DRIVE ERROR : $90 (144) [ACK 'E' $45] !\r\n  Atari Disk Drive encountered an error during command execution...\r\n");
            break;
        case 139:
        		sprintf(sTmpSTR,"ATARI DRIVE NOT ACK : $8B (139) [ACK 'N' $4E] !\r\n  Command cannot be executed : check for errors in the command request...\r\n");
        		break;
        default:
        		sprintf(sTmpSTR,"Unknown Atari Error Code = $%02X\r\n", TheError);
    }
}
BOOL Get_ReadAndCompare(HWND hDialog)
{

      SetDlgItemText(hDialog, IDCR_STATIC4, "Compared Sector :");
		AtrCurrentSector = AtrStartSector;
      // si pas de fichier en mmoire : ca va pas !
      if (!bFileMem)
      {
         MessageBox(hDialog,"ARFD Error:\nNo file is in memory to compare to!\nPlease Open first a file in memory",
         	szComCompare,MB_OK | MB_ICONWARNING);
         return FALSE;
      }
      // si fichier en mmoire mais si il n'a pas la bonne taille : ca va pas !
      if ((AtrEndSector > AmParam.MAXSect) || (AtrStartSector > AmParam.MAXSect))
      {
         MessageBox(hDialog,"ARFD Error:\nInvalid sector number to compare!\nSome requested sectors do not exist in the Rom file in memory!",
         	szComCompare,MB_OK | MB_ICONWARNING);
         return FALSE;
      }
      return TRUE;
}
BOOL Get_Read(HWND hDialog)
{
      int result;
      SetDlgItemText(hDialog, IDCR_STATIC4, "Read Sector :");
		AtrCurrentSector = AtrStartSector;
      SectSize = 0x7F; //taille des secteurs = 0x7f
      iNewSect = AtrDSIZE; // nombre de secteurs pour le openfile de Main Window ARFDesigner
      if (!bWantNewFile)
      // (bWantNewFile == FALSE) utilisation du fichier ROM en mmoire (pas de cration d'un nouveau fichier demande) :
      {
      	if (!bFileMem)
      	{
         // si pas de fichier en mmoire et pas de cration d'un nouveau demande: ca va pas !
         	if (MessageBox(hDialog,"ARFD Error:\nNo Rom file in memory!\nClick OK to continue, creating a new Rom file in memory?",
         		szComReading,MB_OKCANCEL | MB_ICONWARNING) != IDOK)
               	return FALSE; // CANCEL = on annule la procdure de lecture
      	}
      	else // il y a un fichier en mmoire, vrifions si il est OK utilisable ...
         {
				if ( (AtrDSIZE != AmParam.MAXSect) || (AmParam.NBytSect != 0x80) )
				{
         	// si le fichier en mmoire n'a pas la bonne size et pas de cration d'un nouveau demande: ca va pas !
         		if (MessageBox(hDialog,"ARFD Error:\nActual Rom file size in memory does not match the read request :\nClick OK to continue, creating a new file in memory?",
         			szComReading,MB_OKCANCEL | MB_ICONWARNING) != IDOK)
               		return FALSE; // CANCEL = on annule la procdure de lecture
      		}
         	else // Test sur le fichier  utiliser en mmoire est OK,
            // pas de cration d'un nouveau demande: demand la confirmation pour utiliser celui en mmoire !
            {
      			if (MessageBox(hDialog,"The current data on Rom file in memory will be updated :\nClick OK to use current Rom file in memory?",
         			szComReading,MB_OKCANCEL | MB_ICONQUESTION) != IDOK)
                  	return FALSE; // CANCEL = on annule la procdure de lecture

            	return TRUE; // OK = C'est fini, on se sert du fichier en mmoire !
            }
         }
      }
      else // Creation d'un nouveau fichier en mmoire demande (bWantNewFile == TRUE)
      {
      	if (!bFileSaved && bFileMem)
      	{
         	// le dernier fichier ouvert n'a pas t sauvegard : doit on alors vraiment ouvrir un nouveau fichier ?
         	result = MessageBox(hDialog,
            	"The Rom file in memory has not been saved yet.\nThe Rom file in memory will be lost if OK is selected!\nClick OK to close the current Rom file and to load a new one?",
         		szComReading,MB_OKCANCEL | MB_ICONQUESTION);
         	if (result == IDNO) return TRUE; // NON = C'est fini on garde le fichier dja en mmoire malgr tout
        		if (result == IDCANCEL) return FALSE; // CANCEL = on annule la procdure de lecture
      	}
      }
      // on peut crer un nouveau fichier ...
      // c'est nous qui dcidons de charger New File: pas de confirmation dans Main Window ARFDesigner si le fichier a t modifi
      // et on demande l'ouverture d'un autre :
      bFileModified = FALSE;
      // La creation d'un nouveau fichier :
      SendMessage(hMainWindow, WM_COMMAND, CMR_NEWFILE, 0);
      bFileSaved = FALSE; // Nouveau Fichier en mmoire pas encore sauvegard !!
      sprintf(sTmpSTR,"File '%s' open at %d Kb length...\r\n> ",AmParam.FName,AmParam.MAXFL/1024);
      AddText(hErrorLog,sTmpSTR);
      Set_Com_Title(-1);
      SetFocus(hDialog);
      return TRUE;
}

BOOL Get_Write(HWND hDialog)
{
      SetDlgItemText(hDialog, IDCR_STATIC4, "Written sector :");
		AtrCurrentSector = AtrStartSector;
      // si pas de fichier en mmoire : ca va pas !
      if (!bFileMem)
      {
         MessageBox(hDialog,"ARFD Error:\nNo Rom file is in memory for writing source!\nPlease Open first a source Rom file in memory!",
         	szComWriting,MB_OK | MB_ICONWARNING);
         return FALSE;
      }
      if (AmParam.NBytSect != 0x80)
      {
         sprintf(sTmpSTR,"ARFD Error:\nThe sector size of %d bytes is not supported!\nTry to Convert the Rom file in memory before!",
         	AmParam.NBytSect);
         MessageBox(hDialog,sTmpSTR, szComWriting, MB_OK | MB_ICONWARNING);
         return FALSE;
      }
      // si fichier en mmoire mais si il n'a pas la bonne taille : ca va pas !
      if ((AtrEndSector > AmParam.MAXSect) || (AtrStartSector > AmParam.MAXSect))
      {
         MessageBox(hDialog,"ARFD Error:\nInvalid sector number to write!\nSome requested sectors do not exist in the source Rom file in memory",
         	szComWriting,MB_OK | MB_ICONWARNING);
         return FALSE;
      }
      return TRUE;
}

BOOL CALLBACK _export ComErrorDlgProc(HWND hDlg, UINT Message,
                             WPARAM wParam, LPARAM lParam)
 {
    HICON hi;
    int iResult;
    switch(Message)
    {
        case WM_INITDIALOG:
        		SetDlgItemText(hDlg,IDCR_STATIC90,lpsTmpCB);
            strcpy(sTmpSTR,"<ABORT> will Stop an Abort the Reading");
            strcat(sTmpSTR,"\n<RETRY> will Retry the Sector Reading");
            strcat(sTmpSTR,"\n<IGNORE> will Skip the Sector (Data are corrupted)");
            strcat(sTmpSTR,"\n<BAD SECTOR> will Write a Bad Sector and Skip");
            SetDlgItemText(hDlg,IDCR_STATIC91,sTmpSTR);
        		hi = LoadIcon(NULL,IDI_EXCLAMATION	);
            if (hi)
            	SendMessage(GetDlgItem(hDlg, IDIR_WARNING), STM_SETIMAGE,
                		IMAGE_ICON, LPARAM(hi));
            return TRUE;
        case WM_COMMAND:
        		iResult = LOWORD(wParam);
            switch (iResult)
            {
               case IDRETRY:
               case IDIGNORE:
            	case IDCANCEL:
               case IDABORT:
               case IDCR_BUTTONBAD:
                	//if (hi) DeleteObject(hi); //No need for Windows ressources
                	EndDialog(hDlg, iResult);
                	return TRUE;
            }
            break;
    }
    return FALSE;
 }

void Command_In_Progress(HWND hDialog)
{
	MessageBox(hDialog,"ARFD Info:\nAnother command is already in progress!\nPlease Wait!",
   	szComStatus,MB_OK | MB_ICONINFORMATION);
}

void Set_Com_Title(int pc)
{
   if (pc >= 0)
   	sprintf(szComDialogTitle,"%d%% - ",pc);
   else
      strcpy(szComDialogTitle,"");

   strcat(szComDialogTitle,szComTitle);

   if (bFileMem)
   {
   	sprintf(sTmpSTR," - %s",AmParam.FName);
      strcat(szComDialogTitle,sTmpSTR);
   }
   SetWindowText(hMainWindow,szComDialogTitle);
   SetWindowText(hComDialog,szComDialogTitle);
}

void Init_Com_Param_Dlg(HWND hDialog)
{
	CheckRadioButton(hDialog, IDCR_RADIOBUTTON41, IDCR_RADIOBUTTON44, IDCR_RADIOBUTTON41 + (AtrCOM - 1));
   CheckRadioButton(hDialog, IDCR_RADIOBUTTON45, IDCR_RADIOBUTTON48, IDCR_RADIOBUTTON45 + (AtrDDEVIC - 1));
   sprintf(sTmpSTR,"%d",AtrDTIMEOUT1);
   SetDlgItemText(hDialog, IDCR_EDIT41,sTmpSTR);
   sprintf(sTmpSTR,"%d",AtrDTIMEOUT2);
   SetDlgItemText(hDialog, IDCR_EDIT42,sTmpSTR);
}

BOOL Get_Com_Param_Dlg(HWND hDialog)
{
   char szDlgTitle[] = "Communication Configuration :";
   int i;
   for (i = 0; i < 4; i++)
   {
		if (IsDlgButtonChecked(hDialog, (IDCR_RADIOBUTTON41 + i)) & BST_CHECKED)
      	AtrCOM = (UBYTE) (i + 1);
      if (IsDlgButtonChecked(hDialog, (IDCR_RADIOBUTTON45 + i)) & BST_CHECKED)
      	AtrDDEVIC = (UBYTE) (i + 1);
   }

   if (GetDlgItemText(hDialog,IDCR_EDIT41,sTmpSTR,5))
           AtrDTIMEOUT1 = iGet_Dec(sTmpSTR);
   else AtrDTIMEOUT1 = 0;
   if ((AtrDTIMEOUT1 > 200) || (AtrDTIMEOUT1 < 10))
   {
        		MessageBox(hDialog, "ARFD Error:\nInvalid DTIMEOUT1!\nDTIMEOUT1 must be a value\nbetween 10 - 200 millisecond",
     				szDlgTitle, MB_OK | MB_ICONWARNING);
        		return FALSE;
   }
   if (GetDlgItemText(hDialog,IDCR_EDIT42,sTmpSTR,5))
           AtrDTIMEOUT2 = iGet_Dec(sTmpSTR);
   else AtrDTIMEOUT2 = 0;
   if ((AtrDTIMEOUT2 > 1000) || (AtrDTIMEOUT2 < 10))
   {
        		MessageBox(hDialog, "ARFD Error:\nInvalid DTIMEOUT2!\nDTIMEOUT2 must be a value\nbetween 10 - 1000 second",
     				szDlgTitle, MB_OK | MB_ICONWARNING);
        		return FALSE;
   }
   return TRUE;
}

BOOL CALLBACK _export ComParamDlgProc(HWND hDlg, UINT Message,
                             WPARAM wParam, LPARAM lParam)
{
    switch(Message)
    {
        case WM_INITDIALOG:
            Init_Com_Param_Dlg(hDlg);
        		return TRUE;
        case WM_COMMAND:
            switch (LOWORD(wParam))
            {
               case IDCR_BUTTON41:
                  Set_Com_Default_Config();
                  Init_Com_Param_Dlg(hDlg);
                  break;
            	case IDOK:
               	if (!Get_Com_Param_Dlg(hDlg))
                  	break;
               case IDCANCEL:
               	EndDialog(hDlg, IDOK);
                	return TRUE;
            }
            break;
    }
    return FALSE;
}

