//******************************************************************************
// NODE NAME : SerialLib.cpp
// part of the Atari Rom File Designer source files
// Contient la classe de Communication PC<->Atari
// Contains the PC<->Atari Communication Class for the NewCom.cpp
//------------------------------------------------------------------------------
// Written by Philippe VUILLERME (c) PVBest 2001-2002
//
// Part of my code is *based* on an article from Thierry Schneider, entitled:
// ''PC serial port connection object for event-driven programs''
// (c) 2001 Thierry Schneider
// (more info: visit his Web page ''Serial Communication for WIN32'' at
//    http://www.tetraedre.com/advanced/index.php3)
//
// I have written all the Atari protocol functions and recoded the 'Run' thread
//
/*---------------------**| ARFD SOURCE REVISION HISTORY |**---------------------
 Date:     | What:
-----------|--------------------------------------------------------------------
 JUNE 2002 | Clean updated source code for ARFD revision 1.9.0 (public realease)
------------------------------------------------------------------------------*/


#define STRICT
#include <windows.h>
#pragma hdrstop
#include "SerialLib.h"

DWORD WINAPI TAtari_Serial_thread_start(void *arg);


// ---------------------------------------------------------------------------
// ---------------------  TAtari_Serial_thread_start  ------------------------
// ---------------------------------------------------------------------------

//    This function is not part of the TAtari_Serial object. It is simply used
//    to start the thread from an external point of the object.

DWORD WINAPI TAtari_Serial_thread_start(void * arg)
{
    class TAtariSerialSIOV *Tserial_class;

    Tserial_class = (TAtariSerialSIOV *) arg;

    if (Tserial_class!=0)
        Tserial_class->AtrSerial_Run();
    return 1;
}

// -------------------------------------------------------------------------
// CONSTRUCTOR TAtari_Serial
// -------------------------------------------------------------------------
TAtariSerialSIOV::TAtariSerialSIOV()
{
    int i;
    bConnected       = FALSE;

    // Internal use ...
    bSerialReady     = FALSE;
    bRTS 				= FALSE;
    bAckCInReadDataFrame = FALSE;

    dwThreadId       = 0;
    hCOMHandle		   = INVALID_HANDLE_VALUE;

    fWrite_in_progress   = FALSE;
    fRead_in_progress    = FALSE;

    AtariOnProgressFunction = NULL;
    AtariOnErrorFunction	 = NULL;

    // creating Events for the different sources
    for (i=0; i<TSERIAL_SIGNAL_NBR; i++)
    {
        if ((i==SIG_COM_READER) || (i==SIG_COM_WRITER) )
            hSerialEvent[i] = CreateEvent(NULL, TRUE, FALSE, NULL);  // Manual Reset
        else
            hSerialEvent[i] = CreateEvent(NULL, FALSE, FALSE, NULL); // Auto reset
    }
}

// --------------------------------------------------------------------
// DESTRUCTOR   ~TAtariSerialSIOV
// --------------------------------------------------------------------
TAtariSerialSIOV::~TAtariSerialSIOV()
{
    int i;

    for (i=0; i<TSERIAL_SIGNAL_NBR; i++)         // deleting the events
    {
        if (hSerialEvent[i]!=INVALID_HANDLE_VALUE)
            CloseHandle(hSerialEvent[i]);
        hSerialEvent[i] = INVALID_HANDLE_VALUE;
    }

    if (hCOMHandle!=INVALID_HANDLE_VALUE)
        CloseHandle(hCOMHandle);
    hCOMHandle = INVALID_HANDLE_VALUE;
}

// --------------------------------------------------------------------
// --------------------------    SetSignal    -------------------------
// --------------------------------------------------------------------
void TAtariSerialSIOV::fnSetSignal(long mask)
{
    if ((mask<TSERIAL_SIGNAL_NBR) && (mask>=0))
    {
        if (hSerialEvent[mask]!=INVALID_HANDLE_VALUE)
            SetEvent(hSerialEvent[mask]);
    }
}

// --------------------------------------------------------------------
// --------------------------    PowerDown    -------------------------
// --------------------------------------------------------------------
void TAtariSerialSIOV::fnPowerDown(void)
{
    bSerialReady = FALSE;
    bConnected   = FALSE;
    if (hCOMHandle!=INVALID_HANDLE_VALUE)
        CloseHandle(hCOMHandle);
    hCOMHandle = INVALID_HANDLE_VALUE;

    if(dwGetLastError != 0)
    	  fnOnError(1); // il y eu des erreurs !!! (sinon arret normal)

    fnOnProgress(0); //DISCONNECTED //changement d'tat de la comm...

}

// ----------------------------------------------------------------------
// ------------------------      StartThread       ----------------------
// ----------------------------------------------------------------------
void TAtariSerialSIOV::fnStartThread(void)
{
    fnSetSignal(SIG_POWER_ON);   // sending a power on signal to myself

    /*HANDLE CreateThread(
    LPSECURITY_ATTRIBUTES lpThreadAttributes,	// pointer to thread security attributes
    DWORD dwStackSize,	// initial thread stack size, in bytes
    LPTHREAD_START_ROUTINE lpStartAddress,	// pointer to thread function
    LPVOID lpParameter,	// argument for new thread
    DWORD dwCreationFlags,	// creation flags
    LPDWORD lpThreadId 	// pointer to returned thread identifier 
    );*/
    HANDLE hSerialThread = CreateThread(NULL,0, TAtari_Serial_thread_start,
     (void *) this, 0, &dwThreadId);
    SetThreadPriority(hSerialThread,THREAD_PRIORITY_HIGHEST); // normal+2
    // ou voir THREAD_PRIORITY_ABOVE_NORMAL  (normal+1)
}

// ----------------------------------------------------------------------
// ------------------------     GetChecksum      ------------------------
// ----------------------------------------------------------------------
UBYTE TAtariSerialSIOV::fnGetCheckSum(UBYTE * lpBuffer, int length)
{
   int i;
   UWORD ck,cd;
   ck = 0;
   for (i = 0; i <length; i++)
   {
      cd = lpBuffer[i];
   	ck += (UWORD) (cd & 0x00FF);  // on ne prend que l'octet
      if (ck > 255) ck = (UWORD) (ck - 256 + 1);
   }
   return (UBYTE) ck;
}

// --------------------------------------------------------------------
// --------------------------   OnProgress    -------------------------
// --------------------------------------------------------------------
void TAtariSerialSIOV::fnOnProgress(UBYTE TheProgress)
{
    if (AtariOnProgressFunction != NULL)
        AtariOnProgressFunction(TheProgress);
}

// ---------------------------------------------------------------------
// --------------------------   OnError    -----------------------------
// ---------------------------------------------------------------------
void TAtariSerialSIOV::fnOnError(UBYTE TheError)
{
    if (AtariOnErrorFunction != NULL)
        AtariOnErrorFunction(TheError);
}

// ---------------------------------------------------------------------
// ---------------------  SetFunctionAtariOnProgress -------------------
// ---------------------------------------------------------------------
void TAtariSerialSIOV::SetFunctionAtariOnProgress (TUBYTEFunction fnct)
{
    if ((AtariOnProgressFunction == NULL) || (fnct == NULL))
        AtariOnProgressFunction = fnct;
}

// ---------------------------------------------------------------------
// ---------------------  SetFunctionAtariOnError ----------------------
// ---------------------------------------------------------------------
void TAtariSerialSIOV::SetFunctionAtariOnError(TUBYTEFunction fnct)
{
    if ((AtariOnErrorFunction == NULL) || (fnct == NULL))
        AtariOnErrorFunction = fnct;
}

// ---------------------------------------------------------------------
// -----------------------   AtrSerial_GetNbrOfBytes  ------------------
// ---------------------------------------------------------------------
// (not used )
int TAtariSerialSIOV::AtrSerial_GetNbrOfBytes    (void)
{
    struct _COMSTAT status;
    int        n;
    DWORD		etat;

    n = 0;

    if (hCOMHandle!=INVALID_HANDLE_VALUE)
    {
        ClearCommError(hCOMHandle, &etat, &status);
        n = status.cbInQue;
    }
    return(n);
}

// ---------------------------------------------------------------------
// --------------------------    Serial_AtrSendChar --------------------
// ---------------------------------------------------------------------
// (not used)
void TAtariSerialSIOV::AtrSerial_SendChar(char data)
{
    //data_to_send = data;
    fnSetSignal(SIG_DATA_OUT);
    return;
}

// ----------------------------------------------------------------------
// MAIN LOOP AtrSerial_Run
// ----------------------------------------------------------------------
//
// This function is the main thread of the TAtariSerialSIOV object.
// This is a loop waiting for some event objects.
// This is not a busy wait since we use the WaitForMultipleObject function
//
void TAtariSerialSIOV::AtrSerial_Run(void)
{
    static LONG   lStatus;
    static DWORD  dwFuncRTS;
    static BOOL   bSuccess;
    static BOOL 	bDone;
    static BOOL 	bErrorInReadData ;

    bSerialReady      	= TRUE;
    bDone              	= FALSE;
    fWrite_in_progress  = FALSE;
    fRead_in_progress   = FALSE;
    bErrorInReadData    = FALSE;

    //fnOnConnected();
    GetLastError();     // just to clear any pending error

    // -----------------------------------------------------------
    while(!bDone)
    {
    		// on peut placer la fonction de lecture ici,
         // en position perpetuelle d'attente...

        // ----------------------------------------------------
        //            Waiting  for some signal Events
        // ----------------------------------------------------
        //if (!bDone)
        //{
            // Main wait function. Waiting for something to happen.
            // This may be either the completion of a Read or a Write or
            // the reception of Power On, Power Down, new Tx

            lStatus = WaitForMultipleObjects(TSERIAL_SIGNAL_NBR, hSerialEvent,
                                            FALSE, INFINITE);

            // processing answer to filter other failures
            lStatus = lStatus - WAIT_OBJECT_0;
            if ((lStatus < 0) || (lStatus >= TSERIAL_SIGNAL_NBR))
            {
                // Error occured !
                dwGetLastError = GetLastError();
                bDone = 1;
            }
            else
            {
                // ++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                // ++++++++++++++++++++ EVENT DISPATCHER ++++++++++++++++++
                // ++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                switch(lStatus)
                {
                    case SIG_POWER_ON :
                        // OnConnected Here !!
                        bConnected = TRUE;
                        fnOnProgress(1); //CONNECTED
                        break;
                    // ---------------------------------------------------------
                    case SIG_POWER_DOWN:
                        // disconnection...
                        fWrite_in_progress  = FALSE;
    							fRead_in_progress   = FALSE;
                        dwGetLastError = 0; // pas d'erreur sortie volontaire !
                        bDone = 1;
                        break;
                    // ---------------------------------------------------------
                    case SIG_INVERT_RTS:
                       if (!bRTS) //LINE RTS OFF (CLEAR) if bRTS = FALSE
                       {
                           dwFuncRTS = CLRRTS;
                       }
                       else      //LINE RTS ON (SET) if bRTS = TRUE
                       {
                           dwFuncRTS = SETRTS;
                       }
                       // DWORD dwFunc : extended function to perform
                       bSuccess = EscapeCommFunction(hCOMHandle,
    		 							 			dwFuncRTS);
                       if (bSuccess)
                       		fnOnRTSSet(bRTS);
                       else
                       {
                            // Error occured !
                            dwGetLastError = GetLastError();
                            bDone = 1;
                       }
                       //Values	Meaning :
							  //CLRDTR	Clears the DTR (data-terminal-ready) signal.
							  //CLRRTS	Clears the RTS (request-to-send) signal.
							  //SETDTR	Sends the DTR (data-terminal-ready) signal.
							  //SETRTS	Sends the RTS (request-to-send) signal.
                       break;
                    // ---------------------------------------------------------
                    case SIG_SET_TIMEOUT:
                       if(SetCommTimeouts(hCOMHandle,&cto))
                       		fnOnTimeOutSet();
                       else
                       {
                            // Error occured !
                            dwGetLastError = GetLastError();
                            bDone = 1;
                       }
                       break;
                    // ########################################################
                    // ----------------------------------------------------
                    //            WRITE Process
                    // ----------------------------------------------------
                    case SIG_DATA_OUT:
                    		if (!fRead_in_progress && !fWrite_in_progress)
                    		{
                        	bSuccess = WriteFile(hCOMHandle, lpOUTBufferAddress, dwWriteNbr,
                                            &dwTotalWrite, &oWriter);
                        	// sending data on the COM port
                        	if (bSuccess)
                        	{
                            	// Write returns immdiately
                            	ResetEvent(hSerialEvent[SIG_COM_WRITER]);
                            	fWrite_in_progress = FALSE;
                            	fnOnDataSent();
                            	// transmission over !
                              break;
                        	}
                        	else
                        	{
                            	dwGetLastError = GetLastError();
                            	if(dwGetLastError == ERROR_IO_PENDING)
                                	// write is pending :
                                	// will wait for Termination event (SIG_COM_WRITER)
                                	fWrite_in_progress = TRUE;
                            	else
                                	// Error occured !
                                	bDone = 1;
                        	}
                     	}
                        else
                        {
             					// An other Operation is still in progress !!
                           if (fWrite_in_progress)
                           	fnOnError(8);
                           else
                           	fnOnError(9);
                        }
                     	break;

                    // ########################################################
                    case SIG_COM_WRITER:
                        // Checking the result of the terminated write
                        if (GetOverlappedResult(hCOMHandle, &oWriter,
                            &dwTotalWrite, FALSE))
                        {
                            // Write operation completed successfully
                            ResetEvent(hSerialEvent[SIG_COM_WRITER]);
                            fWrite_in_progress = FALSE;
                            fnOnDataSent();
                            // transmission over !
                        }
                        else
                        {
                            // GetOverlapped didn't succeed !
                            // What's the reason ?
                            dwGetLastError = GetLastError();
                            if(dwGetLastError == ERROR_IO_INCOMPLETE )
                                // Not Possible !!!
                                // This is a failure as wa have waited for this event !!!
                                // Shall we wait completion ???
                                // yes :
                                // fRead_in_progress = TRUE; // read not ended
                                // no :
                                bDone = 1;
                            else
                                // Error Occured
                                bDone = 1;
                        }
                        break;
                    // ########################################################
                    // ----------------------------------------------------
                    //            READ Process
                    // ----------------------------------------------------
                    case SIG_COM_READER:
                        // Checking the result of the terminated read
                        if (GetOverlappedResult(hCOMHandle, &oReader,
                            &dwTotalRead, FALSE))
                        {
                        	// Read operation completed successfully
                           ResetEvent(hSerialEvent[SIG_COM_READER]);
                           fRead_in_progress = FALSE;
                        	// Check for Timeout :
                    			if (dwTotalRead != dwReadNbr)
                           {
            						// -> timeout !
            						fnOnTimeOut();
                              break;
                           }
                           // %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
                           if (bAckCInReadDataFrame) // ACK #2 Receiving !
                           {
                                    // Check of ACK #2 which should be 'C' in the begining of readen data frame
                                    bAckCInReadDataFrame = FALSE;
    											// ACK 'C' step for read frame (after ACK 'A')
            								iProgress = 9;
            								DACKCODE2 = abuff.ackcode;
                                    dwReadNbr = (DWORD) (DFRLEN);
          									lpINBufferAddress = abuff.inbuffer;
                                    // check the received ACK code != 'C' ???
                                    // if it is ACK 0x45 'E' it is an Error Data Frame...
            								if (DACKCODE2 == 0x45)
            								{
                                       // if it is not ACK 'C' read the wrong data frame (error)
                                       bErrorInReadData = TRUE;
                                    }
                                    else // DACKCODE2 should be 0x43 'C' complete with no error
                                    	bErrorInReadData = FALSE;
            								// fnOnProgress(ACK #2 received : Waiting for DFRLEN data frame...)
                        			   // Start new reading Process for the data frame
                                    //without exiting the main loop
                                    SetEvent(hSerialEvent[SIG_DATA_IN]);
                                    break;
                           }
                           else //only for lonely ACK CODE #1 or for the read Data Frame
                           {
                           			if (bErrorInReadData == TRUE)
                                    {
                                       // we have received a bad data frame because ACK #2 was 'E' 0x4E
                                       bErrorInReadData = FALSE;
               								DERROR = 144;
                                    	fnOnError(5); //(Error : Wrong ACK !)
               								fnOnProgress(4); //(END : Complete  with ACK 'C' error !)
                                    }
                                    else
                                       // no problem : ACK #1 or Data Frame Received !
                                 		fnOnDataReceived();
                        				// Reading over !!
                                    break;
                           }
                           // %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
                        }
                        else  // GetOverlapped returns False !
                        {
                            // Check why...
                            dwGetLastError = GetLastError();
                            if(dwGetLastError == ERROR_IO_INCOMPLETE )
                                // Not Possible !!!
                                // This is a failure as wa have waited for this event !!!
                                // Shall we wait completion ???
                                // yes :
                                // fRead_in_progress = TRUE; // read not ended
                                // no :
                                bDone = 1;
                            else
                                // Error Occured
                                bDone = 1;
                        }
                        break;
                    // ########################################################
                    case SIG_DATA_IN:
                    		if (!fRead_in_progress && !fWrite_in_progress)
        						{
begin:
                				bSuccess = ReadFile(hCOMHandle,lpINBufferAddress,dwReadNbr,&dwTotalRead,&oReader);
                				// receiving data on the COM port
               				 if (bSuccess)
                            {
										// immediate return => data processing
										ResetEvent(hSerialEvent[SIG_COM_READER]);
                    				fRead_in_progress = FALSE;

                    				// Check for Timeout :
                    				if (dwTotalRead != dwReadNbr)
                    				{
            							// -> timeout !
            							fnOnTimeOut();
                                 break;
            		  				}
                              // %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
                              if (bAckCInReadDataFrame) // ACK #2 Receiving !
                              {
                                    // Check of ACK #2 which should be 'C' in the begining of readen data frame
                                    bAckCInReadDataFrame = FALSE;
    											// ACK 'C' step for read frame (after ACK 'A')
            								iProgress = 9;
            								DACKCODE2 = abuff.ackcode;
                                    dwReadNbr = (DWORD) (DFRLEN);
          									lpINBufferAddress = abuff.inbuffer;
                                    // check the received ACK code != 0x43 'C' ???
                                    // if it is ACK 0x45 'E' it is an Error Data Frame...
            								if (DACKCODE2 == 0x45)
            								{
                                       // if it is not ACK 'C' read the wrong data frame (error)
                                       bErrorInReadData = TRUE;
                                    }
                                    else // DACKCODE2 should be 0x43 'C' complete with no error
                                    	bErrorInReadData = FALSE;
            								// fnOnProgress(ACK #2 received : Waiting for DFRLEN data frame...)
                        			   // Start new reading Process for the data frame
                                    //without exiting the main loop
                                    SetEvent(hSerialEvent[SIG_DATA_IN]);
                                    break;
                              }
                              else //only for lonely ACK CODE #1 or for the read Data Frame
                              {
                           			if (bErrorInReadData == TRUE)
                                    {
                                       // we have received a bad data frame because ACK #2 was 'E' 0x4E
                                       bErrorInReadData = FALSE;
               								DERROR = 144;
                                    	fnOnError(5); //(Error : Wrong ACK !)
               								fnOnProgress(4); //(END : Complete  with ACK 'C' error !)
                                    }
                                    else
                                    	// no problem : ACK #1 or Data Frame Received !
                                 		fnOnDataReceived();
                        				// Reading over !!
                                    break;
                              }
                              // %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
                            }
                            else  //ReadFile returned FALSE
                            {
                    				dwGetLastError = GetLastError();
                    				if(dwGetLastError != ERROR_IO_PENDING )
                        			// Error occured !
                        			bDone = 1;
                    				else
                                 // read is pending :
                                	// will wait for Termination event (SIG_COM_READER)
                        			fRead_in_progress = TRUE; // read is pending
                				}
        						}
        						else //fWrite_in_progress or fRead_in_progress
        						{
             					// An other Operation is still in progress !!
             					if (fWrite_in_progress)
             						fnOnError(8);
             					else
             						fnOnError(9);
        						}
                    		break;
                    // ########################################################
                }
                // +++++++++++++++++ END EVENT DISPATCHER +++++++++++++++++
            }
        //}
    };

    fnPowerDown();
}

// ----------------------------------------------------------------------
// --------------------------   Serial_Wait   ---------------------------
// ----------------------------------------------------------------------
void TAtariSerialSIOV::fnSerialWait()
{
    // attente 2 ms minimum (valeur Atari: 1275 microsecond)
    DWORD dwTime;
    dwTime = GetTickCount() + 2;
    while ( GetTickCount() <= dwTime) {;}
    return;
}

// ----------------------------------------------------------------------
// --------------------------    Serial_ClearBuffer  --------------------
// ----------------------------------------------------------------------
void TAtariSerialSIOV::fnClearBuffers()
{
    memset(&(abuff.cbuffer),0,7);     //Commande frame buffer
    abuff.ackcode = 0;
    abuff.RCHKSUM;
    memset(&(abuff.inbuffer),0,136); //Data In Buffer
    return;
}

// ----------------------------------------------------------------------
// -----------------------   AtrSerial_WriteBadSector  ------------------
// ----------------------------------------------------------------------
void TAtariSerialSIOV::AtrSerial_WriteBadSector(void)
{
    int i;
    for (i=0;i<128;i=i+4)
    {
        //Mise en place de 'BAD!' dans le buffer d'entre
        abuff.inbuffer[i] = 0x42; // 'B' comme BAD!
        abuff.inbuffer[i+1] = 0x41; // 'A' comme BAD!
        abuff.inbuffer[i+2] = 0x44; // 'D' comme BAD!
        abuff.inbuffer[i+3] = 0x21; // '!' comme BAD!
    }
    abuff.inbuffer[128] = 0xFF; // dans le cas d'un bad sector
    abuff.RCHKSUM = 0xFF;
    //La checksum 0xFF n'est pas juste - 0xFF Wrong checksum calculation
    // arbitraire mais reconnaissable ! [il faudrait le bon]
    // this bad checksum value makes the bad sector identification
}

// ----------------------------------------------------------------------
// --------------------------    SetRTS    ------------------------------
// ----------------------------------------------------------------------
void TAtariSerialSIOV::fnSetRTS(BOOL bRTSFlag)
{

    bRTS = bRTSFlag;
    fnSetSignal(SIG_INVERT_RTS);
    return;
}

// -----------------------------------------------------------------------
// ------------------------     SetTimeOut     ---------------------------
// -----------------------------------------------------------------------
void TAtariSerialSIOV::fnSetTimeOut(int iTimeOut)
{
     cto.ReadTotalTimeoutConstant = (DWORD) iTimeOut;
     fnSetSignal(SIG_SET_TIMEOUT);
     return;
}

// ---------------------------------------------------------------------
//   ***********       PUBLIC FUNCTIONS                *****************
// ---------------------------------------------------------------------

// --------------------------------------------------------------------
// -----------------------    AtrSerial_ConnectInit -------------------
// --------------------------------------------------------------------
//     Serial port, file and overlapped structures initialization
//
int  TAtariSerialSIOV::AtrSerial_ConnectInit (char * lpszThePortName)
{
    int erreur;
    DCB  dcb;

    ZeroMemory(&cto,sizeof(COMMTIMEOUTS));
    //cto = { 0, 0, 0, 0, 0 };

    // ----------------------------------------------
    if (hCOMHandle!=INVALID_HANDLE_VALUE)
        CloseHandle(hCOMHandle);
    hCOMHandle = INVALID_HANDLE_VALUE;

    if (lpszThePortName != NULL)
    {
        erreur    = 0;
        ZeroMemory(&oReader,sizeof(oReader));      // clearing the overlapped
        ZeroMemory(&oWriter,sizeof(oWriter));
        memset(&dcb,0,sizeof(dcb));

        // set DCB to configure the serial port
        dcb.DCBlength      	= sizeof(dcb);
        // BaudRate
        dcb.BaudRate       	= CBR_19200;
        // No Parity :
        dcb.Parity      		= NOPARITY;
        dcb.fParity     		= 0;
        // 1 Stop Bit
        dcb.StopBits       	= ONESTOPBIT;
        // 8 bits for byte size
        dcb.ByteSize       	= 8;
        // other non-use value :
        dcb.fOutxCtsFlow  	 	= 0;
        dcb.fOutxDsrFlow   	= 0;
        dcb.fDtrControl    	= DTR_CONTROL_DISABLE;
        dcb.fDsrSensitivity	= 0;
        dcb.fRtsControl    	= RTS_CONTROL_DISABLE;
        dcb.fOutX          	= 0;
        dcb.fInX           	= 0;
        // misc parameters
        dcb.fErrorChar      	= 0;
        dcb.fBinary         	= 1;
        dcb.fNull           	= 0;
        dcb.fAbortOnError   	= 0;
        dcb.wReserved       	= 0;
        dcb.XonLim          	= 0;
        dcb.XoffLim         	= 0;
        dcb.XonChar         	= 0;
        dcb.XoffChar        	= 0;
        dcb.EvtChar         	= 0;

        // COM HANDLE OPENING
        hCOMHandle    = CreateFile(lpszThePortName, GENERIC_READ | GENERIC_WRITE,
                   0, NULL, OPEN_EXISTING, FILE_FLAG_OVERLAPPED, NULL);
                   // opening the serial COM port

        // Overlapped events
        oReader.hEvent = hSerialEvent[SIG_COM_READER];
        oWriter.hEvent = hSerialEvent[SIG_COM_WRITER];

        if (hCOMHandle    != INVALID_HANDLE_VALUE)
        {
            if(!SetCommMask(hCOMHandle, 0))
                erreur = 1;   //error in CommMask setting

            // set timeouts
            if(!SetCommTimeouts(hCOMHandle,&cto))
                erreur = 2;   //error in TIMEOUT setting

            // set DCB
            if(!SetCommState(hCOMHandle,&dcb))
                erreur = 4;  //error in DCB setting
        }
        else
            erreur = 8;  // INVALID_HANDLE_VALUE
    }
    else
        erreur = 16;  // Port Name Null String


    // ----------------------------------------------
    if (erreur!=0)
    {
        dwGetLastError = GetLastError();
        CloseHandle(hCOMHandle);
        hCOMHandle = INVALID_HANDLE_VALUE;
    }
    else
    {
        dwGetLastError = 0;

        lpINBufferAddress = &(abuff.ackcode);
        lpOUTBufferAddress = &(abuff.cbuffer[0]);
        dwReadNbr = 0;
        dwWriteNbr = 0;
        bAckCInReadDataFrame = FALSE;

        memset(&abuff,0,sizeof(TABUFFER));

        fnStartThread();
    }

    // ----------------------------------------------
    return(erreur);
}

// ---------------------------------------------------------------------
// -------------------------- AtrSerial_Disconnect ---------------------
// ---------------------------------------------------------------------
void TAtariSerialSIOV::AtrSerial_Disconnect(void)
{
    fnSetSignal(SIG_POWER_DOWN);
    return;
}

// ----------------------------------------------------------------------
// -------------------------- AtrSerial_SIOV ----------------------------
// ----------------------------------------------------------------------
UBYTE TAtariSerialSIOV::AtrSerial_SendSIOVCommand(void)
{
      UBYTE cks;

      DERROR = 0;
      dwGetLastError = 0;
      iProgress = 0;
   	fnClearBuffers();
      DACKCODE1 = 0xFF;
      DACKCODE2 = 0xFF;
      DACKCODE3 = 0xFF;

      fWrite_in_progress = FALSE;
      fRead_in_progress = FALSE;

  		// Sending of the command frame -
  		// Out com. values initialization -
  		abuff.cbuffer[0] = DDEVIC;
  		abuff.cbuffer[1] = DCOMND;
  		abuff.cbuffer[2] = DAUX1;
  		abuff.cbuffer[3] = DAUX2;
      cks = fnGetCheckSum(&(abuff.cbuffer[0]),4);
      abuff.cbuffer[4] = cks;

  		// In com. values initialization -
  		switch (DCOMND)
    	{
      	case 0x21:
      	case 0x22:
       		DFRLEN = 129;   // 128 octets + 1 cks  en reception/receiving
            DSTATS = 0x40;  //Read Operation
       		break;
      	case 0x53:
       		DFRLEN = 5;     //4 octets + 1 cks  en reception/receiving
            DSTATS = 0x40;  //Read Operation
       		break;
      	case 0x52:
       		DFRLEN = 129;   //128 octets + 1 cks en reception/receiving
            DSTATS = 0x40;  //Read Operation
       		break;
         case 0x50:
      	case 0x57:
       		DFRLEN = 129;   // 128 octets + 1 cks en emission/sending
            DSTATS = 0x80;  //Write Operation
            cks = fnGetCheckSum(&(abuff.outbuffer[0]),128);
            abuff.outbuffer[128] = cks;
       		break;
      	default:
            //DERROR = ;
       		//sprintf(sTmpSTR,"Command $%02X not supported ?!?!\n\r",DCOMMAND);
            //strcat(lpsTmpCB,sTmpSTR);
       		return 0;
    	}

      iProgress = 1;
      fnOnProgress(2);	//(Ready START ! : On envoi RTS actif/RTS activated)
      fnSetRTS(TRUE);

      return 1;
}

// ----------------------------------------------------------------------
// --------------------------    OnRTSSet     ---------------------------
// ----------------------------------------------------------------------
void TAtariSerialSIOV::fnOnRTSSet(BOOL bRTSOn)
{
      if (bRTSOn) //RTS active
      {
      	iProgress = 2;
         // fnOnProgress(RTS active : Wait 1.275 milisecond (as a mini) ...)
      	fnSerialWait();
         iProgress = 3;
         // fnOnProgress(Wait OK : On envoi la command frame (sending)...)
      	fnSendCommandFrame();
      }
      else // RTS desactiv  (iProgress == 4)
      {
         iProgress = 5;
         // fnOnProgress(RTS desactive : Setting the TimeOut for ACK 'A' ...)
      	fnSetTimeOut(DTIMEOUT1);
      }
      return;

}

// ----------------------------------------------------------------------
// --------------------------    OnTimeOutSet  --------------------------
// ----------------------------------------------------------------------
void TAtariSerialSIOV::fnOnTimeOutSet(void)
{
		if (iProgress == 5)    // Read and write operations
      {
      	iProgress = 6;
      	// fnOnProgress(TimeOut at 40 ms Ok : Attente reception ACK 'A'...)
      	fnReceiveAckCode(1);  // Waiting for ACK 'A'
      }
      else if (iProgress == 7)  // Read operation
      {
         iProgress = 8;
         //fnOnProgress(OK : TimeOut Set at 204000 ms, Waiting for ACK' C' code ...)
      	fnReceiveAckCode(2);
      }
      else if (iProgress == 28)  // Write operation
      {
         iProgress = 29;
         //fnOnProgress(OK : TimeOut Set at 40 ms Ok , Waiting for second ACK' A' code ...)
      	fnReceiveAckCode(1);
      }
      else //iProgress = 30  // Write operation
      {
         iProgress = 31;
         //fnOnProgress(OK : TimeOut Set at 204000 ms Ok , Waiting for last  ACK' C' code ...)
      	fnReceiveAckCode(1);
      }
      return;

}

// ----------------------------------------------------------------------
// --------------------------    SendCommandFrame -----------------------
// ----------------------------------------------------------------------
void TAtariSerialSIOV::fnSendCommandFrame()
{
      dwWriteNbr = (DWORD) 5;
      lpOUTBufferAddress = &(abuff.cbuffer[0]);
    	fnSetSignal(SIG_DATA_OUT);
      return;
}

// ----------------------------------------------------------------------
// --------------------------    SendDataFrame --------------------------
// ----------------------------------------------------------------------
void TAtariSerialSIOV::fnSendDataFrame(void)
{
      dwWriteNbr = (DWORD) DFRLEN;
      lpOUTBufferAddress = &(abuff.outbuffer[0]);
    	fnSetSignal(SIG_DATA_OUT);
      return;
}

// ----------------------------------------------------------------------
// --------------------------    OnDataSent  ----------------------------
// ----------------------------------------------------------------------
void TAtariSerialSIOV::fnOnDataSent(void)
{
      if (iProgress == 3) // Command frame
      {
      	iProgress = 4;
      	// fnOnProgress(Set RTS to inactive level ...)
      	fnSetRTS(FALSE);
      }
      else // Data Frame sent ; waiting for an ACK 'A'
      {
         iProgress = 28;
         // fnOnProgress(RTS desactive : mise en place TimeOut pour ACK 'A' ...)
      	fnSetTimeOut(DTIMEOUT1);
      }
      return;

}

// ----------------------------------------------------------------------
// --------------------------    OnAckReceived --------------------------
// ----------------------------------------------------------------------
void TAtariSerialSIOV::fnOnDataReceived(void)
{
    UBYTE tmp;
    switch (iProgress)
    {
        case 6:  // Receiving ACK 'A' step
            iProgress = 7;
            DACKCODE1 = abuff.ackcode;
            if (DACKCODE1 != 0x41) // check the received ACK code != 'A' ???
            {
            // if Yes :
               DERROR = 139;
               fnOnError(3); //(Error : Wrong ACK !)
               fnOnProgress(4); //(END : Complete  with ACK 'A' error on command frame!)
            	return;
            }
            else // if NO: OK !!!
            {
            	if (DSTATS == 0x40) // lecture READ
         		{
            		// fnOnProgress(ACK 'A' received : Setting new timeout for ACK 'C' ...)
            		fnSetTimeOut(DTIMEOUT2); //204000
         		}
         		else // (DSTATS == 0x80) ecriture WRITE
               {
         			// fnOnProgress(ACK 'A' received : Waiting 1.275 milisecond (mini) ...)
      				fnSerialWait();
         			iProgress = 20+7;
         			// fnOnProgress(Wait OK : On envoi la data frame ...)
      				fnSendDataFrame();
               }
        		}
            break;
        case 29: // Is data frame well received ??? (ACK 'A' ???)
        		iProgress = 30;
            DACKCODE1 = abuff.ackcode;
            if (DACKCODE1 != 0x41) // check the received ACK code != 'A' ???
            {
            // if Yes :
               DERROR = 139;
               fnOnError(11); //(Error : Wrong ACK !)
               fnOnProgress(4); //(END : Complete  with ACK 'A' error on dataframe !)
            	return;
            }
            else
            {
            	// fnOnProgress(ACK 'A' received : Set new timeout for ACK 'C' ...)
               fnSetTimeOut(DTIMEOUT2); //204000
            }
            break;
        case 8:
            // already done in Serial thread function...!
        		break;
        case 31:
            iProgress = 32;
            DACKCODE3 = abuff.ackcode;
            if (DACKCODE3 != 0x43)  // check the received ACK code != 'C' ???
            {
               DERROR = 144;
               fnOnError(13); //(Error : Wrong ACK !)
               fnOnProgress(4); //(END : Complete  with ACK 'C' error !)
            	return;
            }
            else // ACK code 'C' OK
      		{
      			iProgress = 33;
      			fnOnProgress(8); //(END : Complete with no error !)
      		}
            break;
        case 9:
        		iProgress = 10;
      		// Check the Checksum
      		abuff.RCHKSUM = abuff.inbuffer[DFRLEN-1];
            tmp = fnGetCheckSum(&(abuff.inbuffer[0]),(DFRLEN-1));
      		if (abuff.RCHKSUM != tmp)
      		{
      			DERROR = 143;
      			fnOnError(7); // (Error : Checksum is Wrong !);
      			fnOnProgress(4); //(END : Complete with CheckSum Error!);
      		}
      		else //checksum OK
      		{
      			iProgress = 11;
      			fnOnProgress(8); //(END : Complete with no error !)
      		}
      		break;
        default:
        		break;
            //Error ???
    }
}

// ----------------------------------------------------------------------
// --------------------------     ReceiveAckCode     --------------------
// ----------------------------------------------------------------------
void TAtariSerialSIOV::fnReceiveAckCode(UBYTE SetpNbr)
{
  		if (SetpNbr == 2)
      {
      	 bAckCInReadDataFrame = TRUE;
      }
      else
      {
          bAckCInReadDataFrame = FALSE;
      }
      dwReadNbr = (DWORD) 1;
      lpINBufferAddress = &(abuff.ackcode);
      fnSetSignal(SIG_DATA_IN);
      return;
}

// ----------------------------------------------------------------------
// --------------------------    OnTimeOut     --------------------------
// ----------------------------------------------------------------------
void TAtariSerialSIOV::fnOnTimeOut(void)
{
   if (iProgress == 6)  // TimeOut on Receiving ACK 'A' step
   {
   	DERROR = 138;
      fnOnError(2); //(Error : Timeout on ACK 'A' !)
   }
   else if (iProgress == 8)  // ACK 'C' step  READ OPERATION
   {
  		DERROR = 138;
      fnOnError(4); // (Error : Timeout on ACK 'C' !)
   }
   else if (iProgress == 9)  // DFRLEN bytes step  READ OPERATION
   {
  		DERROR = 138;
      fnOnError(6); // (Error : Timeout on Data Frame receiving !)
   }
   else if (iProgress == 28) // WRITE OPERATION
   {
  		DERROR = 138;
      fnOnError(10); // (Error : Timeout on ACK 'A' n2 receiving !)
   }
   else // iProgress == 29  // WRITE OPERATION
   {
      DERROR = 138;
      fnOnError(12); // (Error : Timeout on ACK 'C' receiving !)
   }
   fnOnProgress(4); //(END : PROTOCOL ATARI ERROR)
}

