/*
Copyright (c) 1998 Richard Lawrence

This program is free software; you can redistribute it and/or modify it under the terms 
of the GNU General Public License as published by the Free Software Foundation; either 
version 2 of the License, or (at your option) any later version. This program is 
distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even
the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details. You should have received a copy of the GNU
General Public License along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include <windows.h>
#include <stdlib.h>
#include <crtdbg.h>
#include <stdio.h>
#include <io.h>
#include <fcntl.h>
#include <mmsystem.h>
#include "WinConfig.h"
#include "Resource.h"
#include "atari800.h"
#include "globals.h"
#include "crc.h"
#include "zlib.h"
#include "registry.h"
#include "display_win.h"
#include "sound_win.h"
#include "input_win.h"
#include "misc_win.h"


/* Public objects */

char g_szSystemText[ 64 ];

FARPROC	pgzread  = NULL;
FARPROC	pgzopen  = NULL;
FARPROC	pgzclose = NULL;
FARPROC	pgzwrite = NULL;
FARPROC	pgzerror = NULL;

/* Private objects */

static BOOL   s_bCheckZlib = TRUE;
static HANDLE s_hZlib      = NULL;

#ifdef WIN_USE_PERF_COUNTER
static LARGE_INTEGER s_lnTimeFreq;
#endif /*WIN_USE_PERF_COUNTER*/


/* Simple routine to return TRUE if the given filename is any one of the various compressed
   types. Currently the only test for this is by extension - also the only way you can actually
   read a compressed file in, so that seems valid enough. */
BOOL
IsCompressedFile( char *filename )
{
	int i;
	char szTempFile[ MAX_PATH ];

	for( i = 0; i < (int)strlen( filename ); i++ )
		szTempFile[ i ] = toupper( filename[ i ] );
	szTempFile[ i ] = 0;

	if( strstr( szTempFile, ".GZ"  ) ||
		strstr( szTempFile, ".ATZ" ) ||
		strstr( szTempFile, ".XFZ" ) ||
		strstr( szTempFile, ".DCM") )
		return TRUE;

	return FALSE;
}

int
ReadDisabledROMs( void )
{
	int	romfile;

	romfile = _open( atari_basic_filename, O_BINARY | O_RDONLY, 0777 );
	if( romfile == -1 )
	{
		Aprint( "Could not open %s for reading", atari_basic_filename );
		return FALSE;
	}

	if( _read( romfile, atari_basic, 8192 ) < 8192 )
	{
		Aprint( "Could not read all of atari basic from %s", atari_basic_filename );
		return FALSE;
	}
	_close( romfile );

	romfile = _open( atari_xlxe_filename, O_BINARY | O_RDONLY, 0777 );
	if( romfile == -1 )
	{
		Aprint( "Could not open %s for reading", atari_xlxe_filename );
		return FALSE;
	}

	if( _read( romfile, atari_xlxe_filename, 16384 ) < 16384 )
	{
		Aprint( "Could not read entire atari ROM from %s", atari_xlxe_filename );
		return FALSE;
	}
	_close( romfile );
	return TRUE;
}

int
prepend_tmpfile_path( char *buffer )
{
	int	nOffset = 0;

	nOffset += GetTempPath( MAX_PATH, buffer ) - 1;
	if( nOffset == -1 || buffer[ nOffset ] != '\\' )
		nOffset++;

	return nOffset;
}

int
zlib_capable( void )
{
	if( !s_hZlib )
	{
		Aprint( "Cannot gzip/ungzip without gzip.dll loaded properly" );
		return -1;
	}
	return 1;
}

int
Atari_PEN( int vertical )
{
	return vertical ? 0xff : 0;
}

BOOL
RestartEmulation( BOOL bForceWinReInit )
{
	if( bForceWinReInit ) /* Initialise Windows stuff */
	{
		InitialiseInput();

		if( InitialiseSound() && InitialiseScreen( TRUE ) )
		{
			g_ulAtariState &= ~(ATARI_LOAD_FAILED | ATARI_PAUSED);
			g_ulAtariState |= ATARI_RUNNING;
		}
		else
		{
			g_ulAtariState |= ATARI_LOAD_FAILED | ATARI_PAUSED;
			return FALSE;
		}
	}
	Atari_ReInit();
	return TRUE;
}

void
StartAtariTimer( void )
{
#ifdef WIN_USE_PERF_COUNTER
	LARGE_INTEGER	lnCurrentTime;

	QueryPerformanceCounter( &lnCurrentTime );
	g_ulAtariHWNexttime = lnCurrentTime.LowPart + g_ulDeltaT;
#else /*WIN_USE_PERF_COUNTER*/
	g_ulAtariHWNexttime = timeGetTime() + g_ulDeltaT;
#endif /*WIN_USE_PERF_COUNTER*/
}

void
Atari_Initialise( int *argc, char *argv[] )
{
	memset( g_szSystemText, 0, 64 );
	if( s_bCheckZlib )
	{
		s_bCheckZlib = FALSE;
		if( !s_hZlib )
		{
			s_hZlib = LoadLibrary( "ZLIB.DLL" );
			if( !s_hZlib )
				MessageBox( g_hMainWnd, "Could not load ZLIB.DLL. No decompression of gzip files will be possible.", "Atari800Win", MB_ICONEXCLAMATION | MB_OK );
			else
			{
				pgzread  = GetProcAddress( s_hZlib, "gzread" );
				pgzopen  = GetProcAddress( s_hZlib, "gzopen" );
				pgzclose = GetProcAddress( s_hZlib, "gzclose" );
				pgzwrite = GetProcAddress( s_hZlib, "gzwrite" );
				pgzerror = GetProcAddress( s_hZlib, "gzerror" );
				if( !pgzread || !pgzopen || !pgzclose || !pgzwrite || !pgzerror)
				{
					FreeLibrary( s_hZlib );
					s_hZlib = NULL;
					MessageBox( g_hMainWnd, "Could not obtain necessary functions from ZLIB.DLL. No decompression of gzip files will be possible.", "Atari800Win", MB_ICONEXCLAMATION | MB_OK );
				}
			}
		}
	}

#ifdef WIN_USE_PERF_COUNTER
	memset( &s_lnTimeFreq, 0, sizeof(s_lnTimeFreq) );
	memset( &g_ulAtariHWNexttime, 0, sizeof(g_ulAtariHWNexttime) );
#endif /*WIN_USE_PERF_COUNTER*/

	if( tv_mode == TV_PAL )
	{
#ifdef WIN_USE_PERF_COUNTER
		QueryPerformanceFrequency( &s_lnTimeFreq );
		g_ulDeltaT = s_lnTimeFreq.LowPart / 50;
		g_nSleepThreshold  = s_lnTimeFreq.LowPart / 1000L;
		g_nSleepThreshold *= (SLEEP_TIME_IN_MS + 1);
#else /*WIN_USE_PERF_COUNTER*/
		g_ulDeltaT = 20;
		g_nSleepThreshold  = SLEEP_TIME_IN_MS + 1;
#endif /*WIN_USE_PERF_COUNTER*/
	}
	else
	{
#ifdef WIN_USE_PERF_COUNTER
		QueryPerformanceFrequency( &s_lnTimeFreq );
		g_ulDeltaT = s_lnTimeFreq.LowPart / 60;
		g_nSleepThreshold  = s_lnTimeFreq.LowPart / 1000L;
		g_nSleepThreshold *= (SLEEP_TIME_IN_MS + 1);
#else /*WIN_USE_PERF_COUNTER*/
		g_ulDeltaT = 17;
		g_nSleepThreshold  = SLEEP_TIME_IN_MS + 1;
#endif /*WIN_USE_PERF_COUNTER*/
	}

	if( !g_ulModesAvail )
		CheckDDrawModes();

	if( !g_lpbmi )
	{
		/* Only initialize this the first time it's allocated, since we'll come through here
		   on changing hardware types and it will already be set up correctly at that point */
		g_lpbmi = (LPBITMAPINFO)calloc( 1, (sizeof( BITMAPINFOHEADER ) + sizeof( RGBQUAD ) * 256) );

		if( !g_lpbmi )
			Atari_Exit( 1 );

		g_lpbmi->bmiHeader.biSize        = sizeof(BITMAPINFOHEADER);
		g_lpbmi->bmiHeader.biWidth       = ATARI_VIS_WIDTH;
		g_lpbmi->bmiHeader.biHeight      = -ATARI_HEIGHT;	/* Negative because we are a top-down bitmap */
		g_lpbmi->bmiHeader.biPlanes      = 1;
		g_lpbmi->bmiHeader.biBitCount    = 8;				/* Each byte stands for a color value */
		g_lpbmi->bmiHeader.biCompression = BI_RGB;			/* Uncompressed format */

		UseAtariPalette();
	}
	ResetInput();
}

void
UseAtariPalette( void )
{
	int i;

	_ASSERT(g_lpbmi);

	for( i = 0; i < 256; i++ )
	{
		int	rgb = colortable[ i ];
		
		g_lpbmi->bmiColors[ i ].rgbRed      = g_pe[ i ].peRed   = (rgb & 0x00ff0000) >> 16;
		g_lpbmi->bmiColors[ i ].rgbGreen    = g_pe[ i ].peGreen = (rgb & 0x0000ff00) >> 8;
		g_lpbmi->bmiColors[ i ].rgbBlue     = g_pe[ i ].peBlue  = rgb & 0x000000ff;
		g_lpbmi->bmiColors[ i ].rgbReserved = g_pe[ i ].peFlags = 0;
	}
}

void
Toggle_Pause( void )
{
	if( g_ulAtariState & ATARI_RUNNING )
	{
		if( g_ulAtariState & ATARI_PAUSED )
		{
			g_ulAtariState &= ~ATARI_PAUSED;
			RestartSound();
 		}
		else
		{
			g_ulAtariState |= ATARI_PAUSED;
			ClearSound( FALSE );
			if( g_hMainWnd )
				SetTimer( g_hMainWnd, TIMER_READ_JOYSTICK, 100, NULL ); 
			DrawPausedScreen( (UBYTE *)atari_screen );
		}
		DescribeAtariSystem();
	}
}

void
Toggle_FullSpeed( void )
{
	if( g_ulAtariState & ATARI_RUNNING )
	{
		if( g_ulMiscStates & MS_FULL_SPEED )
		{
			g_ulMiscStates &= ~MS_FULL_SPEED;
			RestartSound();
		}
		else
		{
			g_ulMiscStates |= MS_FULL_SPEED;
			ClearSound( FALSE );
		}
		DescribeAtariSystem();
	}
}

void
Toggle_SIOPatch( void )
{
	if( enable_rom_patches )
	{
		if( enable_sio_patch )
		{
			enable_sio_patch = 0;
			RestoreSIO();
		}
		else
		{
			enable_sio_patch = 1;
			SetSIOEsc();
		}
		DescribeAtariSystem();
	}
}

int
LaunchMonitor( void )
{
	/* Get us back to last used windowed display and stop making noises */
	SetSafeDisplay( FALSE );
	/* CAtari800WinView::OnKillFocus will invoke that */
//	ClearSound( FALSE );

	if( monitor() )	/* Launch monitor */
	{
//		RestartSound();
		return 1;	/* Go back to emulation */
	}
	else
	{
		/* Start a quit (this will end up in Atari_Exit(0)) */
		PostMessage( g_hMainWnd, WM_CLOSE, 0, 0L );
	}
	return 0;
}

void
Atari_ReInit( void )
{
	g_ulAtariState = ATARI_UNINITIALIZED | ATARI_PAUSED;
	Remove_ROM();
	Clear_Temp_Files();
	main( 1, NULL );
	g_hWnd = NULL;
}

int
Atari_Exit( int nPanic )
{
	if( nPanic && g_ulAtariState & ATARI_RUNNING )
		g_ulAtariState |= ATARI_CRASHED;
	
	if( ATARI_CRASHED == g_ulAtariState )
	{
		wsync_halt = 1;	/* turn off CPU */
		return 0;
	}

	if( nPanic && g_ulAtariState & ATARI_RUNNING )
	{
		char szMessage[ LOADSTRING_STRING_SIZE ];
		int  nResult;

		LoadString( NULL, IDS_WARNING_ATARI_PANIC, szMessage, LOADSTRING_STRING_SIZE );
		nResult = MessageBox( g_hMainWnd, szMessage, "Atari800Win", MB_ICONSTOP | MB_YESNO );

		InvalidateRect( g_hMainWnd, NULL, FALSE );

		if( IDYES == nResult && LaunchMonitor() /* Launch monitor */ )
		{
			g_ulAtariState &= ~ATARI_CRASHED; /* For CIM service */
			return 1;
		}
		g_ulAtariState = ATARI_CRASHED;
		wsync_halt = 1;	/* turn off CPU */
		test_val   = 32767;
	}
	else
	{
		/* Reset everything DirectDraw */
		ClearScreen( TRUE );

		/* Reset everything MultiMedia/DirectSound */
		ClearSound( TRUE );

		/* Reset everything DirectInput */
		ClearInput();

		g_ulAtariState = ATARI_UNINITIALIZED | ATARI_CLOSING;

		Remove_ROM();

		if( g_lpbmi )
			free( g_lpbmi );
		g_lpbmi = NULL;

		Clear_Temp_Files();

		if( s_hZlib )
		{
			FreeLibrary( s_hZlib );
   			pgzopen = pgzclose = pgzread = pgzwrite = pgzerror = NULL;
		}
	}
	return 0;
}

void
DescribeAtariSystem( void )
{
	if( !g_hMainWnd )
		return;

	if( g_ulScreenMode & SM_MODE_FULL )
		strcpy( g_szSystemText, "  " );
	else
		strcpy( g_szSystemText, "Atari800Win: " );

	if( g_ulAtariState & ATARI_LOAD_FAILED )
	{
		strcat( g_szSystemText, "Initialization failure" );
		SetWindowText( g_hMainWnd, g_szSystemText );
		return;
	}

	if( g_ulAtariState & ATARI_RUNNING )
	{
		if( g_ulAtariState & (ATARI_PAUSED | ATARI_NOFOCUS) )
		{
			if( g_ulAtariState & ATARI_NOFOCUS && !(g_ulAtariState & ATARI_PAUSED) )
				strcat( g_szSystemText, "Stopped" );
			if( g_ulAtariState & ATARI_PAUSED )
				strcat( g_szSystemText, "Paused (F9 to continue)" );
		}
		else
		{
			switch( default_system )
			{
				case 1:
					strcat( g_szSystemText, "800 OS-A " );
					break;
				case 2:
					strcat( g_szSystemText, "800 OS-B " );
					break;
				case 3:
					strcat( g_szSystemText, "800XL " );
					break;
				case 4:
					strcat( g_szSystemText, "130XE " );
					break;
				case 5:
					strcat( g_szSystemText, "320XE " );
					if( 1 == Ram256 )
						strcat( g_szSystemText, "(Rambo) " );
					else
						strcat( g_szSystemText, "(Compy) " );
					break;
				case 6:
					strcat( g_szSystemText, "5200 " );
					break;
				default:
					strcat( g_szSystemText, "Unknown system! " );
					break;
			}
			
			switch( default_tv_mode )
			{
			case 1:
				if( g_ulMiscStates & MS_FULL_SPEED )
					strcat( g_szSystemText, "PAL FULL speed" );
				else
					strcat( g_szSystemText, "PAL 1.77Mhz" );
				break;
				
			case 2:
				if( g_ulMiscStates & MS_FULL_SPEED )
					strcat( g_szSystemText, "NTSC FULL speed" );
				else
					strcat( g_szSystemText, "NTSC 1.79Mhz" );
				break;
				
			default:
				strcat( g_szSystemText, "Unknown TV type!" );
				break;
			}

			if( enable_sio_patch )
				strcat( g_szSystemText, " SIO" );
		}
	}
	else
		strcat( g_szSystemText, "not running" );

	if( g_ulScreenMode & SM_MODE_WIND )
		SetWindowText( g_hMainWnd, g_szSystemText );
}

BOOL
TestAndSetPath( char *rompath, char *filename, unsigned long ulExpectedCRC )
{
	char	szFullPath[ MAX_PATH ];
	unsigned long ulCRC;
	int		nResult;

	ulCRC = CheckFile( filename, &nResult );
	if( ulCRC != 0 && ulCRC == ulExpectedCRC )
	{
		GetCurrentDirectory( MAX_PATH, rompath );
		strcat( rompath, "\\" );
		strcat( rompath, filename );
		return TRUE;
	}
	else
	{
		strcpy( szFullPath, "ROMS\\" );
		strcat( szFullPath, filename );

		ulCRC = CheckFile( szFullPath, &nResult );
		if( ulCRC != 0 && ulCRC == ulExpectedCRC )
		{
			GetCurrentDirectory( MAX_PATH, rompath );
			strcat( rompath, "\\ROMS\\" );
			strcat( rompath, filename );
			return TRUE;
		}
	}
	return FALSE;
}

/* Routine tries to locate the system ROMs in either the current directory or
   a subdirectory "ROMS" under the current directory. Returns true if at least
   one ROM file was found */

BOOL
TestRomPaths( void )
{
	BOOL	bFoundRom = FALSE;

	if( TestAndSetPath( atari_osa_filename, "atariosa.rom", 0L ) )
		bFoundRom = TRUE;
	if( TestAndSetPath( atari_osb_filename, "atariosb.rom", 3252116993L ) )
		bFoundRom = TRUE;
	if( TestAndSetPath( atari_xlxe_filename, "atarixl.rom", 3764596111L ) )
		bFoundRom = TRUE;
	if( TestAndSetPath( atari_xlxe_filename, "atarixe.rom", 3764596111L ) )
		bFoundRom = TRUE;
	if( TestAndSetPath( atari_5200_filename, "atari5200.rom", 3182898204L ) )
		bFoundRom = TRUE;
	if( TestAndSetPath( atari_5200_filename, "5200.rom", 3182898204L ) )
		bFoundRom = TRUE;
	if( TestAndSetPath( atari_5200_filename, "atari52.rom", 3182898204L ) )
		bFoundRom = TRUE;
	if( TestAndSetPath( atari_basic_filename, "ataribas.rom", 2190982779L ) )
		bFoundRom = TRUE;
	if( TestAndSetPath( atari_basic_filename, "basic.rom", 2190982779L ) )
		bFoundRom = TRUE;

	return bFoundRom;
}
