	;;
	;; Sample image viewer, shows how to interface to jpeg decoder
	;; and displays image in a few different modes, also has output
	;; of data to file option
	;;
	;; Raphael Espino
	;; last updated 17-Nov-00
	;; 

	
;; ---------- renderer zero page addresses, 192 and up are available
	
rendpt = 192			; pointer to image data - 2 bytes
filtpt = 194			; filter pointer - 2 bytes
drawtemp = 196			; 2 temporary storage bytes
	
;; ---------- end of renderer zero page addresses

;; ------------- decoder information addresses ------------------
	
DISPCOLS = $600 	    ; (# of columns to display)/8 must be <= 40
DISPROWS = $601 	    ; (# of rows to display)/8
coloff   = $602             ; Coloumn offset to display image from
rowoff   = $603             ; Row offset to display image from
numcols  = $604		    ; image width, 1 column = 8 pixels (pixels/8)
numrows  = $605	  	    ; image height 1 row = 8 pixels (pixels/8)

IOCBNUM  = $606		    ; IOCB to read jpeg data from
STACKPT  = $607		    ; stack pointer at program start, use to return
			    ; to DOS at any point
ERROR    = $608		    ; non 0 if error ocurred decoding jpeg
			    ; error codes are as defined in decoder
RERUN    = $609		    ; 2 bytes - restart address, use to rerun decoder
VERSION  = $60B	   	    ; decoder version number
	
	;; page 6 addresses from $630 up are available to renderer
	
;; ------------- end of decoder info addresses ----------------------

LODCHN = 2		; IOCB to use for loading file
SAVECHNUM = 3		; IOCB to use for saving to file


SAVMSC = 88
VKEYBD = 520			; keyboard IRQ vector
SDLSTL = 560
GPRIOR = 623			; priority register, enable GR.9
SDMCTL = 559			; shadow DMA control address
COLOR0 = 708			; colour shadow addresses
COLOR1 = 709
COLOR2 = 710
COLOR3 = 711
COLOR4 = 712

CH     = 764			; last keypress shadow address
	
	;; IOCB addresses
ICCOM  = 834
ICSTA  = 835
ICBAL  = 836
ICBAH  = 837
ICBLL  = 840
ICBLH  = 841
ICAX1  = 842
ICAX2  = 843

	
MAXLEN = 60                 ; max file name length		
RDBUF  = $630		    ; use page 6 for file name buffer


KBCODE = 53769		    ; hardware keyboard code address
PORTB  = 54017		    ; OS RAM control on XL/XE
DMACTL = 54272		    ; DMA control
NMIEN  = 54286		    ; NMI enable
CIOV   = 58454		    ; CIO vector
SETVBV = 58460		    ; Set VBI vector
EXITIM = 58463		    ; Exit Immediate VBI vector


	;; ICCOM values
OPEN   = 3   ; open channel
GETLNE = 5   ; get line
PUTBUF = 11  ;  put buffer
CLOSE  = 12  ; close channel

	;; IOCB open modes
READ   = 4   ; open for read
WRITE  = 8   ; open for write


		;; set up viewer jmp vectors
	.ADDR segvector
	.ADDR segvectorend-1
	
 	.ORG $0620
	
segvector:
	
	;; next 12 bytes should be JMP's to renderer's init, start, draw
	;; and end code
	
	JMP RafRendInit		; init renderer
	JMP RafRendStart	; image data about to arrive
	JMP RafRendDraw		; 8 lines of image data available
	JMP RafRendEnd		; image completed
	JMP UnusedVec		; unused for now, point at an RTS
	
segvectorend:	

		;; header for viewer
	.ADDR segcode
	.ADDR segcodeend-1

		;; renderer has area from $8E00 upwards for itself and screen
	
	 	.ORG $8E00
	;;
	;; init code, this will be called when decoder starts or
	;; when it is re-run.  Renderer should display start up information,
	;; open an IOCB for the decoder to read the jpeg data from
	;; and store the (IOCB number * 16) in IOCBNUM ($606)
	;; The rowoff and coloff values can optionally be set here
	;; or in RendDraw.  If they are not set they will default to 0
	;; DISPCOLS and DISPROWS can also optionally be set here or in
	;; RendDraw.  If not set they default to DISPCOLS = 40 (320 pixels) 
	;; DISPROWS = 24 (192 pixels)
	;; *** DISPCOLS should NEVER be set to > 40 ***
	;;
	
segcode:
	
RafRendInit:

	 ;;  display rendered information
	 jsr strout
	 .BYTE "RafRender v.0.3 (17Nov00)",155,155
         .BYTE "file name:",0
	
	 JSR RDLINE		;  get filename from user
	 LDY #0

	 ;;  make sure IOCB is available
	 LDX #(LODCHN*16)
	 JSR CloseFile
         BMI @error
	
	 ;;  now open the file
  	 STX IOCBNUM		; tell decoder what IOCB to use
	 LDA #READ
	 JSR OpenFile
	 BMI @error

	 ;; get column and row offsets from user
	 jsr strout
	 .BYTE "offset col:0",30, 0
         jsr getnum		; get number from user
         sta coloff
         jsr strout
         .BYTE "offset row:0",30,0
         jsr getnum		; get number from user
         sta rowoff
	 RTS			; all done, return to decoder

	;; an error occured, display error code and restart
@error:
	 LDX #(LODCHN*16)
  	 LDA ICSTA,X		; read status value
	 pha			; remember error code
	 jsr CloseFile		; close file
         jsr strout
         .BYTE "load error $",0
	 pla
         jsr HexOut		; display error code

	 jsr WAITKEYP		; tell user to press a key to continue
	 jmp (RERUN)		; and restart decoder

	;; to return to environment here, you can do:
	;; LDX STACKPT
	;; TXS
	;; RTS

	;;
	;; renderer start code, will be called immediately before
	;; the image data is about to start arriving.  Renderer
	;; should open graphics mode, open output file, etc. here.
	;; This is the first point that numcols and numrows information
	;; is valid.  Renderer can optionally setup DISPCOLS, DISPROWS,
	;; coloff, rowoff information here.
	;; *** DISPCOLS should NEVER be set to > 40 ***
	;;
	
RafRendStart:
	lda #25
	sta DISPROWS

@askagain:
	;; display menu
	jsr strout
	.BYTE 155,"S KEY TOGGLES SCREEN WHILE DECODING",155
        .BYTE 155,"1) 320x200x2 (GR.8)",155
	.BYTE "2) 160x200x4 (GR.15)",155
	.BYTE "3) 160x200x4 (GR.15) 1/1",155
	.BYTE "4) 80x200x16 (GR.9)",155
	.BYTE "5) 80x200x16 (GR.9) 2/1", 155
	.BYTE "6) 80x96x16 (GR.9) 1/1", 155
	.BYTE "7) Save 320x200x256 to disk",155
	.BYTE 0


	jsr CKOSRAM		; is there any RAM under the OS?
	lda OSRAM
	bne @doram		; if so then display 64K options too
	jmp @noram
@doram:
	jsr strout
	.BYTE "8) 320x200x2x2 (GR.8) Needs 64K", 155
	.BYTE "9) 160x200x4x2 (GR.15) Needs 64K", 155
	.BYTE "10)160x200x4x2 (GR.15) 1/1, 64K", 155
	.BYTE "11)80x200x16x2 (GR.9) Needs 64K", 155
	.BYTE "12)80x200x16x2 (GR.9) 2/1, 64K", 155
	.BYTE "13)80x96x16x2 (GR.9) 1/1, 64K", 155
	.BYTE 0
@noram:
	jsr strout
	.BYTE "Choice?"
	.BYTE 0
	jsr getnum		; get user selection
	sta RENDMODE		; remember user selection
	tax
	and #%00001000		; options >= 8 use 2 screens
	sta USE2SCR
	txa
	cmp #7			; option 7 - save to disk
	bne @chk1
	jmp @savetofile
@chk1:
	cmp #1
	bne @skgr8
@jmgr8:
	jmp @gr8
@skgr8:
	cmp #2
	bne @skgr15
@jmgr15:
	jmp @gr15
@skgr15:
	cmp #3
	bne @skgr15a
@jmgr15a:
	jmp @gr15a
@skgr15a:
	cmp #4
	beq @gr9
	cmp #5
	beq @gr9a
	cmp #6
	beq @gr9a
	ldy OSRAM		; modes >= 8 only for >=64K machines
	beq @jmask
	cmp #8
	beq @jmgr8
	cmp #11
	beq @gr9
	cmp #12
	beq @gr9a
	cmp #13
	beq @gr9a
	cmp #9
	beq @jmgr15
	cmp #10
	beq @jmgr15a
@jmask:	
	jmp @askagain		; unknown option, prompt user again

@gr9:
	lda #10			; 80/8 = 10 columns
	sta DISPCOLS

@gr9a:
	lda #15			; Graphics 9
	jsr OpenGr

	lda #<SCR2TMP		; save 2nd screen of gr.9 data temp area
	sta gr9scr22+1
	lda #>SCR2TMP
	sta gr9scr22+2

	ldx SAVMSC		; set up gr.9 screen address
	stx gr9scr+1
	stx gr9scr2+1
	ldy SAVMSC+1
	sty gr9scr+2
	sty gr9scr2+2

	lda RENDMODE		; check if image is to be doubled vertically
	cmp #6			; modes 6 and 13 are
	beq @dogrd
	cmp #13
	bne @skgrd
@dogrd:	
	txa
	clc
	adc #40
	sta gr9scrd+1		; set up addresses for doubled data
	sta gr9scr2d+1
	bcc @noiny
	iny
@noiny:	
	sty gr9scrd+2
	sty gr9scr2d+2
	ldy gr9scr22+2
	lda gr9scr22+1
	clc
	adc #40
	sta gr9scr22d+1
	bcc @no92d
	iny
@no92d:
	sty gr9scr22d+2
	
	lda #$9D		; enable doubling of data
	sta gr9scrd
	sta gr9scr2d
	sta gr9scr22d
	lda #80
	sta LINEBYTES
	lda #12
	sta DISPROWS
	bne @sknop
@skgrd:
        lda #$EA		; NOP operation
	sta gr9scrd		; disable doubling of data
	sta gr9scrd+1
	sta gr9scrd+2
	sta gr9scr2d
	sta gr9scr2d+1
	sta gr9scr2d+2
	sta gr9scr22d
	sta gr9scr22d+1
	sta gr9scr22d+2
	lda #40
	sta LINEBYTES

@sknop:	
	lda #64
	sta GPRIOR		; enable gr.9
	bne @exit	

@gr8:
	lda #15			; Graphics 8 -> DL mode 15
	jsr OpenGr		; open graphics mode
	lda SAVMSC
	sta gr8scr+1
	lda SAVMSC+1
	sta gr8scr+2
	lda #<SCR2TMP		; save 2nd screen of gr.8 data temp area
	sta gr8scr2+1
	lda #>SCR2TMP
	sta gr8scr2+2

	;; graphics 8 colours are 0 and 1
	lda #0
	sta COLOR2
	lda #8
	sta COLOR1
	bne @exit
@gr15:
	lda #20			; 160/8 = 20 columns
	sta DISPCOLS

@gr15a:				; leave DISPCOLS at default
	lda #14			; Graphics 15 -> DL mode 14
	jsr OpenGr		; open graphics mode
	lda SAVMSC
	sta gr15scr+1
	lda SAVMSC+1
	sta gr15scr+2
	lda #<SCR2TMP		; save 2nd screen of gr.15 data temp area
	sta gr15scr2+1
	lda #>SCR2TMP
	sta gr15scr2+2

		;; graphics 15 colours are 0, 1, and 2 + background (4)
	lda #4
	sta COLOR0
	lda #8
	sta COLOR1
	lda #12
	sta COLOR2

@exit:
	;; set up our own keyboard IRQ to toggle DMA if key pressed
	lda #0
	sta SAVEDM
	lda VKEYBD		; save current keyboard IRQ for later
	sta OSKEYBIRQ+1
	lda VKEYBD+1
	sta OSKEYBIRQ+2

	sei			; inhibit interrupts
	lda #<dmatoggle		; set up keyboard IRQ to toggle DMA on/off
	sta VKEYBD
	lda #>dmatoggle
	sta VKEYBD+1
	cli			; renable interrupts
	rts


	;; save the data to a file
@savetofile:	
	jsr strout
	.BYTE 155,"Save file name:",0
        JSR RDLINE		;  get filename from user
	LDY #0

	;;  make sure IOCB is available
	ldx #(SAVECHNUM*16)
	JSR CloseFile
        BMI @error
	
	;;  now open the file
	LDA #WRITE
	jsr OpenFile
	BMI @error
	
	lda #40			; 320/8 = 40 columns
	sta DISPCOLS
	bne @exit

@error:
	ldx #(SAVECHNUM*16)
  	LDA ICSTA,X		; read status value
	pha
	jsr CloseFile		; close channel
        jsr strout		; display error message
        .BYTE "error $",0
	pla
        jsr HexOut		; display error code
        jsr strout
        .BYTE " opening file",155,0
	jmp @askagain		; display menu again
	
	;; toggle DMA to speed processing up
dmatoggle:
	lda KBCODE
	and #%00111111		; ignore Shift and Control keys
	cmp #62			; toggle DMA if s key is pressed
	bne OSKEYBIRQ		; pass key press to OS
	txa
	pha
	lda SDMCTL		; toggle DMA value
	ldx SAVEDM
	stx SDMCTL
	stx DMACTL
	sta SAVEDM
	pla
	tax
OSKEYBIRQ:
	jmp 30000		; gets changed - jump to previous keyb IRQ
		

SAVEDM:	.BYTE 0			; save previous DMA value
RENDMODE: .BYTE 0		; graphics mode menu selection
USE2SCR: .BYTE 0		; use 2 screens?
LINEBYTES: .BYTE 0		; number of bytes to skip for next GR.9 line

	;;
	;; draw image data.  This gets called when 8 lines of image
	;; data have been read and decoded from jpeg image.  Renderer
	;; should display/save/etc the 8 lines as it sees fit.  Address
	;; of data is in Acc (lo) and Y (hi).  Data is 256 levels of
	;; greyscale (8 bit luminance), 1 byte per pixel.  Data is arranged
	;; as 8 lines of 320 pixels ($A00 consecutive bytes).
	;; If DISPCOLS < 40 then remaining data in line will be empty, i.e.
	;; if DISPCOLS = 10 then each line will be 80 (10*8) bytes of image
	;; data followed by 240 (30*8) bytes that should be ignored
	;; *** DISPCOLS should NEVER be set to > 40 ***
	;; 

RafRendDraw:
  	sta rendpt		; save data buffer address
  	sty rendpt+1
        ldy #8			; draw 8 lines
	sty rendline
	lda RENDMODE
	cmp #4
	beq rendgr9		; drawing in gr.9 mode
	cmp #5
	beq rendgr9
	cmp #6
	beq rendgr9
	cmp #7
	beq savefile		; save data to file
	cmp #2
	beq @jrgr15		; drawing in gr.15 mode
	cmp #3
	beq @jrgr15		; drawing in gr.15 keeping aspect
	cmp #9
	beq @jrgr15
	cmp #10
	beq @jrgr15
	cmp #11
	beq @jmgr92
	cmp #12
	beq @jmgr92
	cmp #13
	beq @jmgr92

	jmp rendgr8		; drawing in gr. 8
@jrgr15:
	jmp rendgr15		; drawing in gr. 15
@jmgr92:
	jmp rendgr92		; drawing in gr.9 with 2 screens
	
	;; save data to file
savefile:
 	ldx #(SAVECHNUM*16)	; IOCB should already be open
	lda #PUTBUF		; so just dump all data to file
	sta ICCOM,X
	lda rendpt
	sta ICBAL,X
	lda rendpt+1
	sta ICBAH,X
 	lda #<(320*8)		; save 8 lines worth of data
	sta ICBLL,X
	lda #>(320*8)
	sta ICBLH,X
 	JSR CIOV
	bpl @rts
	jmp saveerr		; error saving data
@rts:
UnusedVec:
	rts


	;; draw data in gr.9 mode
rendgr9:

	lda RENDMODE
	cmp #5
	beq @dofilt
	cmp #6
	bne @nofilt
@dofilt:
 	jsr filter		; reduce pixels horizontally
@nofilt:

	ldy #0
	ldx #0
gr9loop:
	lda (rendpt),y		; get 1st pixel
	and #%11110000		; use top 4 bits for grey level
	sta drawtemp
	iny
	lda (rendpt),y		; get 2nd pixel
	lsr			; shift top 4 bits to the low 4 bits
	lsr			; of pixel data
	lsr
	lsr
	ora drawtemp
gr9scr:	
	sta 30000,x		; put pixel on screen
gr9scrd:
	sta 30000,x		; double pixel vertically
	iny
	inx
	cpx #40			; do 40 bytes worth of data at a time
	bcc gr9loop		; 40 bytes = 80 pixels
	
	lda gr9scr+1		; move screen pointer onto next line
	clc
	adc LINEBYTES
	sta gr9scr+1
	bcc @nogr9hi
	inc gr9scr+2
@nogr9hi:
	lda RENDMODE
	cmp #6
	bne @nogr9hi2

	lda gr9scrd+1		; move screen pointer onto next line
	clc
	adc #80
	sta gr9scrd+1
	bcc @nogr9hi2
	inc gr9scrd+2
@nogr9hi2:

	lda rendpt		; move data pointer onto next line
	clc
	adc #<320		; skip the other 240 bytes as they
	sta rendpt		; are empty
	lda rendpt+1
	adc #>320
	sta rendpt+1
	
	dec rendline		; have all 8 lines been drawn?
	bne rendgr9		; no, go back and do the rest

        rts

	;; 
	;; draw data in gr.15 mode
	;; 
rendgr15:
	lda USE2SCR		; are we using 2 screens?
	beq dogr15
	sei
	LDA NMIEN
	PHA
	LDA #0
	STA NMIEN		; disable interrupts
	lda PORTB
	pha
	and #%11111110		; enable OS RAM
	sta PORTB

dogr15:
	lda RENDMODE
	cmp #3			; should we reduce image width with filter?
	beq @filt
	cmp #10
	bne @nofilt
@filt:
 	jsr filter
@nofilt:
	
	ldy #0
	ldx #0
gr15loop:
	lda (rendpt),y		; get 1st pixel
	and #%11000000		; use top 2 bits for grey level
	sta drawtemp
	iny
	lda (rendpt),y		; get 2nd pixel
	and #%11000000		; use top 2 bits and shift into position
	lsr
	lsr
	ora drawtemp
	sta drawtemp
	iny
	lda (rendpt),y		; get 3rd pixel
	and #%11000000
	lsr
	lsr
	lsr
	lsr
	ora drawtemp
	sta drawtemp
	iny
	lda (rendpt),y		; get 4th pixel
	rol
	rol
	rol
	and #%00000011
        ora drawtemp
gr15scr:	
	sta 30000,x		; display on screen
	iny
	inx
	cpx #40			; do 40 bytes worth of data
	bcc gr15loop
	lda USE2SCR
	beq @nogr152
	jsr rendgr152		; save data for 2nd GR.15 screen
@nogr152:	
	lda gr15scr+1		; move screen pointer onto next line
	clc
	adc #40
	sta gr15scr+1
	bcc @nogr15hi
	inc gr15scr+2
@nogr15hi:
	lda rendpt		; move data pointer onto next line
	clc
	adc #<320
	sta rendpt
	lda rendpt+1
	adc #>320
	sta rendpt+1
	
	dec rendline		; have we done 8 lines yet?
	bne dogr15		; if not then go do the rest

	lda USE2SCR		; do we need to reenable interrupts?
	beq @rts
	PLA
	STA PORTB
        PLA
	STA NMIEN
	cli			; renable them
@rts:	
        rts

	;; draw data in gr.8 mode
rendgr8:
	lda USE2SCR		; are we using 2 screens?
	beq dogr8
	sei			; yes so disable interrupts
	LDA NMIEN
	PHA
	LDA #0
	STA NMIEN
	lda PORTB
	pha
	and #%11111110		; enable OS RAM
	sta PORTB
dogr8:	
	ldy #0
	ldx #0
gr8loop:
	lda #0
	sta drawtemp
	sta drawtemp+1
	lda (rendpt),y		; get 1st pixel
	rol			; set GR.8 pixel if value >= 128
	rol drawtemp
	rol
	rol drawtemp+1
	iny	
	lda (rendpt),y		; get 2nd pixel
	rol
	rol drawtemp
	rol
	rol drawtemp+1
	iny	
	lda (rendpt),y		; get 3rd pixel
	rol
	rol drawtemp
	rol
	rol drawtemp+1
	iny	
	lda (rendpt),y		; get 4th pixel
	rol
	rol drawtemp
	rol
	rol drawtemp+1
	iny	
	lda (rendpt),y		; get 5th pixel
	rol
	rol drawtemp
	rol
	rol drawtemp+1
	iny	
	lda (rendpt),y		; get 6th pixel
	rol
	rol drawtemp
	rol
	rol drawtemp+1
	iny	
	lda (rendpt),y		; get 7th pixel
	rol
	rol drawtemp
	rol
	rol drawtemp+1
	iny	
	lda (rendpt),y		; get 8th pixel
	rol
	rol drawtemp
	rol
	rol drawtemp+1
        lda drawtemp
	
gr8scr:	
	sta 30000,x
	iny
	bne @noinchi
	inc rendpt+1
@noinchi:
	lda USE2SCR
	beq gr8noram
	lda drawtemp+1
gr8scr2:
	sta 30000,x
gr8noram:	
	inx
	cpx #40			; do 40 bytes worth of data
	bcc gr8loop
	lda gr8scr+1		; move screen pointer onto next line
	clc
	adc #40
	sta gr8scr+1
	bcc @nogr8hi
	inc gr8scr+2
@nogr8hi:
	lda gr8scr2+1		; move screen pointer onto next line
	clc
	adc #40
	sta gr8scr2+1
	bcc @nogr8hi2
	inc gr8scr2+2
@nogr8hi2:

 	lda rendpt		; move data pointer onto next line
 	clc
 	adc #64
 	sta rendpt
	bcc @noincp1
	inc rendpt+1
@noincp1:
	
	dec rendline
	beq @exit
	jmp dogr8
@exit:
	lda USE2SCR		; are we using 2 screens?
	beq @rts
	PLA			; yes, so reenable interrupts
	STA PORTB
        PLA
	STA NMIEN
	cli
@rts:
        rts

saveerr:
	jsr reset		; restore DMA + IRQ
	jsr strout
	.BYTE "error $",0
	ldx #(SAVECHNUM*16)
	lda ICSTA,x
	jsr HexOut
	jsr strout
	.BYTE " writing to file",155,0
	ldx #(SAVECHNUM*16)
	jsr CloseFile		; close output file
	ldx #(LODCHN*16)
	jsr CloseFile		; close input file
	jsr WAITKEYP		; wait for key press
	jmp (RERUN)             ; and return to environment

	
	;; do a 1-2-1 filter on pixels to keep aspect ration in gr.15 
filter:
	lda rendpt
	sta filtpt
	sta @destadr+1
	lda rendpt+1
	sta filtpt+1
	sta @destadr+2
	ldx #0
@filtlp:
	ldy #0
	lda (filtpt),y		; get 1st pixel
	lsr			; divide it by 4
	lsr
	sta @drawtemp2
	iny
	lda (filtpt),y		; get 2nd pixel
	lsr			; divide it by 2
	clc
	adc @drawtemp2		; and add to pixel 2
	sta @drawtemp2
	iny
	lda (filtpt),y		; get 3rd pixel
	lsr			; divide it by 4
	lsr
	clc
	adc @drawtemp2		; and add to pixel 1 + pixel 2
@destadr:
	sta 30000,x		; store result back at original pos
	clc
	lda filtpt		; and skip a pixel
	adc #2
	sta filtpt
	bcc @noinchi
	inc filtpt+1
@noinchi:
	inx
	cpx #160
	bcc @filtlp
	rts
		
@drawtemp2:	 .BYTE 0

	;; 
	;; copy the second gr.15 screen to under the OS
	;; 
rendgr152:	
	ldy #0
	ldx #0
gr152loop:
	lda (rendpt),y		; get 1st pixel
	and #%00110000		; use top 2 bits for grey level
	asl
	asl
	sta drawtemp
	iny
	lda (rendpt),y		; get 2nd pixel
	and #%00110000		; use top 2 bits and shift into position
	ora drawtemp
	sta drawtemp
	iny
	lda (rendpt),y		; get 3rd pixel
	and #%00110000
	lsr
	lsr
	ora drawtemp
	sta drawtemp
	iny
	lda (rendpt),y		; get 4th pixel
	and #%00110000
	lsr
	lsr
	lsr
	lsr
        ora drawtemp
gr15scr2:	
	sta 30000,x		; display on screen
	iny
	inx
	cpx #40			; do 40 bytes worth of data
	bcc gr152loop
	lda gr15scr2+1		; move screen pointer onto next line
	clc
	adc #40
	sta gr15scr2+1
	bcc @nogr152hi
	inc gr15scr2+2
@nogr152hi:
        rts	

	;; 
	;; copy the second gr.9 screen to under the OS
	;; 
rendgr92:

	sei			; disable interrupts
	LDA NMIEN
	PHA
	LDA #0
	STA NMIEN
	lda PORTB
	pha
	and #%11111110		; enable OS RAM
	sta PORTB
dogr92:
	lda RENDMODE
	and #%11111110
	cmp #12			; option 12 or 13?
	bne @nofilt
 	jsr filter
@nofilt:

	ldy #0
	ldx #0
gr9loop2:
	lda #0
	sta drawtemp
	sta drawtemp+1
	lda (rendpt),y		; get 1st pixel
	asl			; shift low 4 bits to the top 4 bits
	rol drawtemp
	asl			; of pixel data
	rol drawtemp+1
	asl
	rol drawtemp
	asl
	rol drawtemp+1
	asl
	rol drawtemp
	asl
	rol drawtemp+1
	asl
	rol drawtemp
	asl
	rol drawtemp+1

	iny
	lda (rendpt),y		; get 2nd pixel
	asl			; shift low 4 bits to the top 4 bits
	rol drawtemp
	asl			; of pixel data
	rol drawtemp+1
	asl
	rol drawtemp
	asl
	rol drawtemp+1
	asl
	rol drawtemp
	asl
	rol drawtemp+1
	asl
	rol drawtemp
	asl
	rol drawtemp+1

        lda drawtemp
gr9scr2:	
	sta 30000,x		; put pixel on screen
gr9scr2d:
	sta 30000,x		; double pixel vertically
	
	lda drawtemp+1
gr9scr22:	
	sta 30000,x		; put second pixel into temp memory
gr9scr22d:	
	sta 30000,x		; double pixel vertically
	
	iny
	inx
	cpx #40			; do 40 bytes worth of data at a time
	bcc gr9loop2		; 40 bytes = 80 pixels

	lda gr9scr2+1		; move screen pointer onto next line
	clc
	adc LINEBYTES
	sta gr9scr2+1
	bcc @nogr92hi
	inc gr9scr2+2
@nogr92hi:
	lda gr9scr22+1		; move screen pointer onto next line
	clc
	adc LINEBYTES
	sta gr9scr22+1
	bcc @nogr92hi2
	inc gr9scr22+2
@nogr92hi2:

	lda RENDMODE
	cmp #13
	bne @nogr92hi4
@doscr2d:
	lda gr9scr2d+1		; move screen pointer onto next line
	clc
	adc #80
	sta gr9scr2d+1
	bcc @nogr92hi3
	inc gr9scr2d+2
@nogr92hi3:
	lda gr9scr22d+1		; move screen pointer onto next line
	clc
	adc #80
	sta gr9scr22d+1
	bcc @nogr92hi4
	inc gr9scr22d+2
@nogr92hi4:

	lda rendpt		; move data pointer onto next line
	clc
	adc #<320		; skip the other 240 bytes as they
	sta rendpt		; are empty
	lda rendpt+1
	adc #>320
	sta rendpt+1
	
	dec rendline		; have all 8 lines been drawn?
	beq @exit
	jmp dogr92		; no, go back and do the rest
@exit:
	
	PLA			; renable interrupts
	STA PORTB
        PLA
	STA NMIEN
	cli
	rts
	
rendline: .BYTE 0

	
	;;
	;; end of data.  Gets called when image has finished.  Renderer
	;; should close files, restore system back to original state,
	;; wait for user to press a key and then return.  After returning
	;; decoder will exit back to environment (DOS).  To restart decoder
	;; instead do JMP (RERUN) instead of RTS.  If decoder failed to
	;; decode image then this will be called with ERROR > 0
	;;
	
RafRendEnd:
        jsr reset		; reset IRQ and DMA

	;; close input file
  	ldx #(LODCHN*16)
	jsr CloseFile

	ldx #255		; clear last key press
	stx CH
	inx
	stx COLOR4		; set background colour to black
	
	lda RENDMODE
	cmp #7
	bne @notsave		; we're not saving to disk
	
	;; we are saving to disk, so close file
	ldx #(SAVECHNUM*16)
	jsr CloseFile

	lda ERROR
	bne @waitmsg
	jsr strout
	.BYTE 155,"Save complete",0
@waitmsg:
	jsr WAITKEYP		; wait for key press and return
	jmp (RERUN)

@notsave:
	ldx ERROR		; did decoder complete successfully?
	bne @waitmsg

	lda USE2SCR
	beq @waitkp		; if we copied data to OS RAM
	jsr copyscr		; then copy it back down again

	LDY #<IMVBI		; set up immediate VBI to switch screens
	LDX #>IMVBI
	LDA #6
 	JSR SETVBV

	;; get key press
@waitkp:
	lda #0
	lda CH
	and #%00111111		; ignore Shift and Control keys
	cmp #28			; ESC key - return to environment
	beq @exit
	cmp #12			; RETURN key - rerun decoder
	beq @exit
	cmp #33			; SPACE BAR - rerun decode
	beq @exit
	cmp #6			; + key
	beq @incclr
	cmp #14			; - key
	bne @waitkp
				; decrease colour number used
				; leave luminance unchanged
	lda COLOR4
	and #%11110000
	sec
	sbc #16
	jmp @dorest
@incclr:			; increase colour number used
				; leave luminance unchanged
	lda COLOR4
	and #%11110000
	clc
	adc #16
@dorest:	
	sta COLOR4
	ldx #2
	;; update other colour registers based on colour in COLOR4
	;; leave luminance values unchanged
@clrloop:	
	lda COLOR0,x
	and #%00001111
	ora COLOR4
	sta COLOR0,x
	dex
	bpl @clrloop
	lda #255
	sta CH
        bne @waitkp
@exit:
	pha
	LDY #<EXITIM		; disable our VBI
	LDX #>EXITIM
	LDA #6
 	JSR SETVBV

	;; clear key press and return
	lda #255
	sta CH
	pla
	cmp #28			; if this was ESC key, then go do to DOS
	beq @rts
	jmp (RERUN)
	
@rts:
	rts
	
	;; reset IRQ and DMA
reset:
	lda VKEYBD+1
	cmp #>dmatoggle
	bne @rts		; IRQ was not put in place, so Init not called
	lda VKEYBD
	cmp #<dmatoggle		; just wait for keypress and exit then
	bne @rts
	
	;;  undo keyboard IRQ
	sei
	lda OSKEYBIRQ+1
	sta VKEYBD
	lda OSKEYBIRQ+2
	sta VKEYBD+1
	cli
	
	;; is DMA disabled, then re-enable it
	lda SAVEDM
	beq @rts		; DMA already enabled
	sta SDMCTL		; enable DMA
	sta DMACTL
@rts:	
	rts

 ;
 ; immediate VBI code to switch between 2 screens
 ;

IMVBI:
	lda #>SCRADR		; if currently on screen 1, switch to 2
	cmp DLADR+5
	beq @swscr2
	sta DLADR+5			; otherwise switch to 1
	lda #(>SCRADR)+16
	sta DLADR+109
	lda RENDMODE
	cmp #8
	bne @ck9
	lda #8			; its a GR.8 screen
	sta COLOR1
@exit:	
	jmp EXITIM
@ck9:
	cmp #11
	bcs @exit
	lda COLOR0
	and #%11110000
	ora #8			; its a GR.15 screen
	sta COLOR0
	lda COLOR1
	and #%11110000
	ora #10
	sta COLOR1
	lda COLOR2
	and #%11110000
	ora #12
	sta COLOR2
	jmp EXITIM
	
@swscr2:			; switch to screen 2
	lda #>SCR2ADR
	sta DLADR+5
	lda #(>SCR2ADR)+16
	sta DLADR+109
	lda RENDMODE
	cmp #8			; is this a GR.8 mode screen?
	bne @ck92
	lda #4
	sta COLOR1
	jmp EXITIM
@ck92:
	cmp #11
	bcs @exit
	lda COLOR0
	and #%11110000
	ora #2			; its a GR.15 screen
	sta COLOR0
	lda COLOR1
	and #%11110000
	ora #4
	sta COLOR1
	lda COLOR2
	and #%11110000
	ora #6
	sta COLOR2
	jmp EXITIM

vbinum:	.BYTE 0

 
	;; **********************************************************
	;; library routines, these may be useful in your own renderer
	;; or they may be too lame for you to use =)
	;; **********************************************************

	; display embeded string on screen
	; string data should follow strout call and be terminated
	; with a 0
strout:
	 PLA			; get string address from stack
	 TAY
	 PLA
	 TAX
	 INY
	 BNE @NIN2
	 INX
@NIN2:
	 STY ICBAL		; point at start of string
	 STX @loop+2
	 STX ICBAH
	 LDX #0			; no data yet
	 STX ICBLL	
	 STX ICBLH
@loop:   lda $c000,y		; - this address gets modified
         beq @exit		; continue until we find terminating 0
	 INC ICBLL		; increase length count
	 bne @skplh
	 inc ICBLH
@skplh:
	 iny
         bne @loop
         inc @loop+2		; bump up to next page
         bne @loop
@exit:
	 LDA #PUTBUF		; found end of string
	 STA ICCOM
	 TYA
	 PHA

	 JSR CIOV		; let CIO display it
         PLA
	 TAY			; modify return address to 
	 lda @loop+2		; return immediately after
         pha			; terminating 0
         tya
         pha
         rts

	;; 
	;; use CIO to read a line of input from user
	;; 
RDLINE: 
	 LDX #0
	 LDA #GETLNE
	 STA ICCOM,X
	 LDA #<RDBUF
	 STA ICBAL,X
	 LDA #>RDBUF
	 STA ICBAH,X
	 LDA #<MAXLEN  ; MAX BYTES PER LINE (INCLUDING RETURN)
	 STA ICBLL,X
	 LDA #>MAXLEN  ; MAX BYTES PER LINE (INCLUDING RETURN)
	 STA ICBLH,X
	 JSR CIOV
	 BPL @EXTOK

@EOL:
	;; make sure there is a return char at end of line
	 LDA #<RDBUF
	 CLC
	 ADC ICBLL,X
	 STA @STA01+1
	 LDA #>RDBUF
	 ADC ICBLH,X
	 STA @STA01+2
	 LDA #155
@STA01:
	 STA 30000 ; GETS CHANGED
@EXTOK:
	 LDY #0
	 RTS
	
	;; 
	;; close IOCB
	;; 
CloseFile:	
	 LDA #CLOSE
	 STA ICCOM,X
	 JMP CIOV

	;;
	;; wait for key press
	;; 
WAITKEYP:
	jsr strout
	.BYTE 155,"press a key to continue",0
@waitkpd:
	lda CH
	cmp #255
	beq @waitkpd
	lda #255
	sta CH
	rts

	
DLADR = $9F00			; display list address
SCRADR = $A010			; 1st screen address
SCR2ADR = $5010 		; 2nd screen address
SCR2TMP = $E000			; temporary storage for screen 2 until
				; decoder finishes
	;; 
	;; open screen in appropriate graphics mode
	;; graphics mode in y register
	;;

OpenGr:
	STA @mode
	ldy #2
	lda #96
@blnk3:	
	sta DLADR,y		; do 21 blank scan lines
	dey
	bpl @blnk3
	ldy #3
	lda #64			; LMS
	clc
	adc @mode		; add in LMS screen mode
	sta DLADR,y
	iny
	lda #<SCRADR		; set up screen start address
	sta DLADR,y
	iny
	lda #>SCRADR
	sta DLADR,y
	ldy #100		; do 101 lines
	lda @mode
@filgr1:
	sta DLADR+6,y
	dey
	bpl @filgr1
	ldy #107
	lda #64			; LMS
	clc
	adc @mode		; add in LMS screen mode
	sta DLADR,y
	iny
	lda #0
	sta DLADR,y		; set up data address for LMS
	iny
	lda #(>SCRADR)+16
	sta DLADR,y
	ldy #96			; do 97 lines
	lda @mode
@filgr2:
	sta DLADR+110,y
	dey
	bpl @filgr2

	ldy #207
	lda #65			; end the DL
	sta DLADR,y
	iny
	lda #<DLADR		; point back at start of DL
	sta DLADR,y
	iny
	lda #>DLADR
	sta DLADR,y
	 
	lda #<SCRADR		; clear screen memory area
	sta SAVMSC
	
	lda #>SCRADR
	sta SAVMSC+1
	jsr clrscr		; clear 8k of ram

	lda USE2SCR		; check if temp scr area needs cleaning too
	beq @notmp
	
	sei
	LDA NMIEN		; disable all interrupt
	PHA
	LDA #0
	STA NMIEN		; disable interrupts
	lda PORTB
	pha
        and #%11111110
	sta PORTB		; enable OS RAM
	lda #>SCR2TMP		; clear temp area too
	jsr clrscr
	PLA
	STA PORTB		; enable OS ROM
        PLA
	STA NMIEN		; enable interrupts
	cli
	
@notmp:
	lda #<DLADR		; set up display list
	sta SDLSTL
	lda #>DLADR
	sta SDLSTL+1

	RTS
	
@mode:	.BYTE 0

	;; clear 8K worth of RAM for screen
clrscr:
	sta filtpt+1		; screen starting page
	ldy #0
	tya
	sta filtpt
	ldx #32			; 8K worth of screen data
@clrlp1:
	dey
  	sta (filtpt),y
	bne @clrlp1
	inc filtpt+1
	dex
	bne @clrlp1
	rts

	;;
	;;  Print hex number on screen
	;; 
HexOut:
         PHA			;  save original value
         LSR			;  move hi hex digit to bottom nibble
         LSR
         LSR
         LSR
         JSR @print		;  display hi hex digit
         PLA			;  restore original value
         AND #$0F		;  and get lo hex digit
@print:  cmp #10		;  if digit is > 9, then convert it to A
         BCC @c1
         ADC #6
@c1:     ADC #48
	 LDX #PUTBUF
	 STX ICCOM
	 LDX #0
	 STX ICBLL
	 STX ICBLH
	 JMP CIOV

	;; 
	;; get number from from user
	;; 
getnum:	   
	 JSR RDLINE		; read input data from user
         lda #00
         TAY
@loop2:  sta temp
@loop:   LDA RDBUF,Y
	 cmp #155		; finished if return key press found
         beq @done
         INY
         sec
         sbc #'0'		; convert from ASCII to number
         sta temp+1
         ldx #10
         lda #00
         clc
@l2:	 adc temp		; multiply by 10
         dex
         bne @l2
         adc temp+1
         jmp @loop2
@done:  
         lda temp
         rts

temp:	 .WORD 0


	;;
	;; open an IOCB
	;; Acc = 4 -> open file for read
	;; Acc = 8 -> open file for write
	;; 
OpenFile:
	 STA ICAX1,X
	 LDA #OPEN
	 STA ICCOM,X
	 LDA #<RDBUF
	 STA ICBAL,X
	 LDA #>RDBUF
	 STA ICBAH,X
	 LDA #0
	 STA ICAX2,X
	 JMP CIOV

	
OSRTST = 52224 ; check this address to see if it is RAM or not

	;; check if there is any RAM under the OS
CKOSRAM:
	 PHP
	 SEI
	 LDA NMIEN
	 PHA
	 LDA #0
	 STA NMIEN		; disable interrupts

	 LDY OSRTST		; PICK LOC. IN OS TO TEST
	 LDA PORTB		; save current PORTB state
         PHA
	 AND #%11111110		; try enabling OS RAM
	 STA PORTB
	 LDX #255
	 STX OSRTST		; store $FF to test if it is RAM
	 CPX OSRTST
	 BNE @NTRAM		; not RAM
	 INX
	 STX OSRTST		; store 0 to test if it is RAM
	 CPX OSRTST
	 BNE @NTRAM		; not RAM
	 STY OSRTST		; restore original value
	 LDY #1			; 1 FLAGS IT IS RAM 
	 BNE @ISRAM

@NTRAM:
	 STY OSRTST		; RESTORE JUST IN CASE
	 LDY #0

@ISRAM:
	 STY OSRAM
	 PLA
	 STA PORTB
	 PLA
	 STA NMIEN
	 PLP			; restore status register including irq bit
 
	 RTS

OSRAM:	 .BYTE 0
	
	;; copy screen data back down again from OS RAM
copyscr:	
	sei
	LDA NMIEN
	PHA
	LDA #0
	STA NMIEN		; disable interrupts
	lda PORTB
	pha
	and #%11111110		; enable OS RAM
	sta PORTB

	lda #<SCR2TMP
	sta rendpt
	lda #>SCR2TMP
	sta rendpt+1

	lda #<SCR2ADR
	sta filtpt
	lda #>SCR2ADR
	sta filtpt+1

	LDY #0
@CPCD1:
	LDA (rendpt),Y
	STA (filtpt),Y
	INY
	BNE @CPCD1
	INC filtpt+1
	INC rendpt+1		; copy data up to $FFFF
	BNE @CPCD1
	
	PLA
	STA PORTB
        PLA
	STA NMIEN
	cli

	rts
	
segcodeend:
	;; end of renderer code

