/****************************************************************************
File    : registry.c
/*
@(#) #SY# Atari800Win
@(#) #IS# Registry service implementation for Win32 platforms
@(#) #BY# Richard Lawrence, Tomasz Szymankowski
@(#) #LM# 01.10.2000
*/

/*
Copyright (c) 1998 Richard Lawrence

This program is free software; you can redistribute it and/or modify it under the terms 
of the GNU General Public License as published by the Free Software Foundation; either 
version 2 of the License, or (at your option) any later version. This program is 
distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even
the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details. You should have received a copy of the GNU
General Public License along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include <stdio.h>
#include <crtdbg.h>
#include <stdlib.h>
#include <io.h>
#include <fcntl.h>
#include <windows.h>
#include "WinConfig.h"
#include "Resource.h"
#include "atari800.h"
#include "globals.h"
#include "misc_win.h"
#include "input_win.h"
#include "sound_win.h"
#include "registry.h"

/* Constants definition */

#define FILE_TYPES_NO	5

/* Public objects */

struct FileExtInfo_t g_aFileExtInfo[] =
{
	{ FEI_DSK, FILE_ATR, ".atr", IDC_FILEASSOCIATIONS_ATR },
	{ FEI_DSK, FILE_XFD, ".xfd", IDC_FILEASSOCIATIONS_XFD },
	{ FEI_DSK, FILE_DCM, ".dcm", IDC_FILEASSOCIATIONS_DCM },
	{ FEI_DSK, FILE_ATZ, ".atz", IDC_FILEASSOCIATIONS_ATZ },
	{ FEI_DSK, FILE_XFZ, ".xfz", IDC_FILEASSOCIATIONS_XFZ },
	{ FEI_BIN, FILE_XEX, ".xex", IDC_FILEASSOCIATIONS_XEX },
	{ FEI_ROM, FILE_ROM, ".rom", IDC_FILEASSOCIATIONS_ROM },
	{ FEI_ROM, FILE_BIN, ".bin", IDC_FILEASSOCIATIONS_BIN },
	{ FEI_CRT, FILE_CRT, ".crt", IDC_FILEASSOCIATIONS_CRT },
	{ FEI_A8S, FILE_A8S, ".a8s", IDC_FILEASSOCIATIONS_A8S }
};

const int g_nFileExtInfoNo = sizeof(g_aFileExtInfo)/sizeof(g_aFileExtInfo[0]);

/* Private objects */

static char *s_apszRootType[ FILE_TYPES_NO ] =
{
	"Atari8.dsk", "Atari8.bin", "Atari8.rom", "Atari8.crt", "Atari8.a8s"
};

static int s_nVersion = CURRENT_REV;


/*========================================================
Function : DeleteAllRegKeys
=========================================================*/
/* #FN#
   DeleteAllRegKeys will recursively delete everything from a supplied initial
   Key. All subkeys are enumerated and deleted as found. Note that ALL values
   underneath a key are deleted when that key is deleted. */
void
/* #AS#
   Nothing */
DeleteAllRegKeys( HKEY  hkInput,
                  char *pszName )
{
	HKEY     hkKey;
	DWORD    dwIndex   = 0;
	DWORD    dwBufSize = 256;
	FILETIME dummy;
	char     szSubKeyName[ 256 ];

	if( hkInput == NULL )
	{
		if( RegOpenKeyEx(
			HKEY_CURRENT_USER,	/* handle of open key                */
			pszName,			/* address of name of subkey to open */
			0,					/* reserved                          */
			KEY_ALL_ACCESS,		/* security access mask              */
			&hkKey 				/* address of handle of open key     */
			) != ERROR_SUCCESS )
		{
			return;
		}
	}
	else
		hkKey = hkInput;

	while( RegEnumKeyEx( 
				hkKey,			/* handle of key to enumerate           */
				dwIndex++,		/* index of subkey to enumerate         */
				szSubKeyName,	/* address of buffer for subkey name    */
				&dwBufSize,		/* address for size of subkey buffer    */
				NULL,			/* reserved                             */
				NULL,			/* address of buffer for class string   */
				NULL,			/* address for size of class buffer     */
				&dummy			/* address for time key last written to */
		 ) == ERROR_SUCCESS )
	{
		DeleteAllRegKeys( hkKey, szSubKeyName );
	}
	RegDeleteKey( HKEY_CURRENT_USER, pszName );
} /* #OF# DeleteAllRegKeys */

/*========================================================
Function : WriteRegDWORD
=========================================================*/
/* #FN#
   WriteRegDWORD writes out an int to the preset Registry key HKEY_CURRENT_USER\REGNAME.
   If the HKEY passed in is valid it is used, otherwise the key is grabbed and released
   within the function. Note that RegOpenKey is used here instead of RegCreateKey, which
   is only used at init time. No calls should be made to this prior to HandleRegistry().
   Any write to the Registry that doesn't work is skipped with user notification. */
void
/* #AS#
   Nothing */
WriteRegDWORD( HKEY  hkInput,
			   char *item,
			   DWORD value)
{
	HKEY hkKey;

	if( hkInput == NULL )
	{
		if( RegOpenKeyEx(
			HKEY_CURRENT_USER,		// handle of open key 
			REGNAME,				// address of name of subkey to open 
			0,						// reserved 
			KEY_WRITE,				// security access mask 
			&hkKey 					// address of handle of open key 
			) != ERROR_SUCCESS )
		{
			DisplayMessage( IDS_REG_OPEN_ERROR, 0, MB_ICONSTOP | MB_OK );
			return;
		}
	}
	else
		hkKey = hkInput;
	
	if( RegSetValueEx(
		hkKey,						// handle of key to set value for  
		item,						// address of value to set 
		0,							// reserved 
		REG_DWORD,					// flag for value type 
		(unsigned char *)&value,	// address of value data 
		sizeof( DWORD )				// data buffer size
		) != ERROR_SUCCESS )
	{
		DisplayMessage( IDS_REG_WRITE_ERROR, 0, MB_ICONSTOP | MB_OK );
	}	
	if( hkInput == NULL )
		RegCloseKey( hkKey );
} /* #OF# WriteRegDWORD */

/*========================================================
Function : ReadRegDWORD
=========================================================*/
/* #FN#
   ReadRegDWORD retrieves an existing value. To make it bulletproof the
   calling routine can request to display errors or not, depending on how
   fatal they are considered. */
int
/* #AS#
   The value was read */
ReadRegDWORD( HKEY   hkInput,
			  char  *item,
			  DWORD *data,
			  BOOL   bShowError )
{
	DWORD type, size, value;
	HKEY  hkKey;
	
	if( hkInput == NULL )
	{
		if( RegOpenKeyEx(
			HKEY_CURRENT_USER,		// handle of open key 
			REGNAME,				// address of name of subkey to open 
			0,						// reserved 
			KEY_READ,				// security access mask 
			&hkKey 					// address of handle of open key 
			) != ERROR_SUCCESS )
		{
			if( bShowError )
			{
				DisplayMessage( IDS_REG_OPEN_ERROR, 0, MB_ICONSTOP | MB_OK );
				return READ_REG_FAIL;
			}
		}
	}
	else
		hkKey = hkInput;
	
	// Win95 is really picky about having this size set; WinNT doesn't care. Go figure.
	size = sizeof( DWORD );
	if( RegQueryValueEx(
		hkKey,						// handle of key to query 
		item,						// address of name of value to query 
		0,							// reserved 
		&type,						// address of buffer for value type 
		(unsigned char *)&value,	// address of data buffer 
		&size			 			// address of data buffer size 
		) != ERROR_SUCCESS )
	{
		if( bShowError )
			DisplayMessage( IDS_REG_LOAD_ERROR, 0, MB_ICONEXCLAMATION | MB_OK, item );
		else
		{
			if( hkInput == NULL )
				RegCloseKey( hkKey );
		}
		if( hkInput == NULL )
			RegCloseKey( hkKey );
		return READ_REG_FAIL;
	}
	
	if( type != REG_DWORD || size != sizeof( DWORD ) )
	{
		DisplayMessage( IDS_REG_WRONG_SIZE, 0, MB_ICONSTOP | MB_OK );
		if( hkInput == NULL )
			RegCloseKey( hkKey );
		return READ_REG_FAIL;
	}

	if( hkInput == NULL )
		RegCloseKey( hkKey );
	
	*data = value;
	return TRUE;
} /* #OF# ReadRegDWORD */

/*========================================================
Function : ReadRegBinary
=========================================================*/
/* #FN#
   Retrieves an existing value */
DWORD
/* #AS#
   The value was read */
ReadRegBinary( HKEY  hkInput,
			   char *item,
			   char *buffer,
			   ULONG maxsize,
			   BOOL  bShowError )
{
	HKEY  hkKey;
	DWORD type;
	unsigned int iFullSize = maxsize;
	
	if( hkInput == NULL )
	{
		if( RegOpenKeyEx(
			HKEY_CURRENT_USER,		// handle of open key 
			REGNAME,				// address of name of subkey to open 
			0,						// reserved 
			KEY_READ,				// security access mask 
			&hkKey 					// address of handle of open key 
			) != ERROR_SUCCESS )
		{
			if( bShowError )
			{
				DisplayMessage( IDS_REG_OPEN_ERROR, 0, MB_ICONSTOP | MB_OK );
				return READ_REG_FAIL;
			}
		}
	}
	else
		hkKey = hkInput;
	
	if( RegQueryValueEx(
		hkKey,						// handle of key to query 
		item,						// address of name of value to query 
		0,							// reserved 
		&type,						// address of buffer for value type 
		(unsigned char *)buffer,	// address of data buffer 
		&maxsize			 		// address of data buffer size 
		) != ERROR_SUCCESS )
	{
		if( bShowError )
			DisplayMessage( IDS_REG_KEY_ERROR, 0, MB_ICONSTOP | MB_OK, item );
		else
		{
			if( hkInput == NULL )
				RegCloseKey( hkKey );
			*buffer = 0;
		}
		if( hkInput == NULL )
			RegCloseKey( hkKey );
		return READ_REG_FAIL;
	}

	if( iFullSize > maxsize + 1 )
		buffer[ maxsize + 1 ] = 0;
	
	if( hkInput == NULL )
		RegCloseKey( hkKey );
	
	return type;
} /* #OF# ReadRegBinary */

/*========================================================
Function : WriteRegBinary
=========================================================*/
/* #FN#
   WriteRegBinary is similar to WriteRegDWORD except it dumps an arbitrary
   binary section of data */
void
/* #AS#
   Nothing */
WriteRegBinary( HKEY hkInput,
			    char *item,
				unsigned char *data,
				int size )
{
	HKEY hkKey;
	
	if( hkInput == NULL )
	{
		if( RegOpenKeyEx(
			HKEY_CURRENT_USER,	// handle of open key 
			REGNAME,			// address of name of subkey to open 
			0,					// reserved 
			KEY_WRITE,			// security access mask 
			&hkKey 				// address of handle of open key 
			) != ERROR_SUCCESS )
		{
			DisplayMessage( IDS_REG_OPEN_ERROR, 0, MB_ICONSTOP | MB_OK );
			return;
		}
	}
	else
		hkKey = hkInput;
	
	if( RegSetValueEx(
		hkKey,					// handle of key to set value for  
		item,					// address of value to set 
		0,						// reserved 
		REG_BINARY,				// flag for value type 
		data,					// address of value data 
		size					// data buffer size
		) != ERROR_SUCCESS )
	{
		DisplayMessage( IDS_REG_WRITE_ERROR, 0, MB_ICONSTOP | MB_OK );
	}	
	if( hkInput == NULL )
		RegCloseKey( hkKey );
} /* #OF# WriteRegBinary */

/*========================================================
Function : WriteRegString
=========================================================*/
/* #FN#
   WriteRegString is similar to WriteRegBinary except it writes a null-terminated
   string */
void
/* #AS#
   Nothing */
WriteRegString( HKEY hkInput,
			    char *item,
				char *data )
{
	HKEY	hkKey;
	
	if( hkInput == NULL )
	{
		if( RegOpenKeyEx(
			HKEY_CURRENT_USER,		// handle of open key 
			REGNAME,				// address of name of subkey to open 
			0,						// reserved 
			KEY_WRITE,				// security access mask 
			&hkKey 					// address of handle of open key 
			) != ERROR_SUCCESS )
		{
			DisplayMessage( IDS_REG_OPEN_ERROR, 0, MB_ICONSTOP | MB_OK );
			return;
		}
	}
	else
		hkKey = hkInput;
	
	if( RegSetValueEx(
		hkKey,						// handle of key to set value for  
		item,						// address of value to set 
		0,							// reserved 
		REG_SZ,						// flag for value type 
		(const unsigned char *)data,// address of value data 
		strlen( data )				// data buffer size
		) != ERROR_SUCCESS )
	{
		DisplayMessage( IDS_REG_WRITE_ERROR, 0, MB_ICONSTOP | MB_OK );
	}	
	if( hkInput == NULL )
		RegCloseKey( hkKey );
} /* #OF# WriteRegString */

int
ReadRegKeyset( HKEY hkInput,
			   int  nKeyset )
{
	ULONG ulKeysetReg1 = 0L;
	ULONG ulKeysetReg2 = 0L;
	ULONG ulKeysetReg3 = 0L;
	BOOL  bRegFail     = FALSE;
	HKEY  hkKey        = NULL;
	int   i;

	if( !hkInput )
	{
		if( RegOpenKeyEx(
			HKEY_CURRENT_USER,		// handle of open key 
			REGNAME,				// address of name of subkey to open 
			0,						// reserved 
			KEY_READ,				// security access mask 
			&hkKey 					// address of handle of open key 
			) != ERROR_SUCCESS )
		{
			DisplayMessage( IDS_REG_OPEN_ERROR, 0, MB_ICONSTOP | MB_OK );
			return FALSE;
		}
	}
	else
		hkKey = hkInput;

	if( KEYS_A_JOYSTICK == nKeyset )
	{
		if( ReadRegDWORD( hkKey, REG_KEYSET_A1, (DWORD *)&ulKeysetReg1, TRUE ) == READ_REG_FAIL )
		{
			ulKeysetReg1 = 0L;
			bRegFail = TRUE;
		}
		if( ReadRegDWORD( hkKey, REG_KEYSET_A2, (DWORD *)&ulKeysetReg2, TRUE ) == READ_REG_FAIL )
		{
			ulKeysetReg2 = 0L;
			bRegFail = TRUE;
		}
		if( ReadRegDWORD( hkKey, REG_KEYSET_A3, (DWORD *)&ulKeysetReg3, TRUE ) == READ_REG_FAIL )
		{
			ulKeysetReg3 = 0L;
			bRegFail = TRUE;
		}
	}
	else
	{
		if( ReadRegDWORD( hkKey, REG_KEYSET_B1, (DWORD *)&ulKeysetReg1, TRUE ) == READ_REG_FAIL )
		{
			ulKeysetReg1 = 0L;
			bRegFail = TRUE;
		}
		if( ReadRegDWORD( hkKey, REG_KEYSET_B2, (DWORD *)&ulKeysetReg2, TRUE ) == READ_REG_FAIL )
		{
			ulKeysetReg2 = 0L;
			bRegFail = TRUE;
		}
		if( ReadRegDWORD( hkKey, REG_KEYSET_B3, (DWORD *)&ulKeysetReg3, TRUE ) == READ_REG_FAIL )
		{
			ulKeysetReg3 = 0L;
			bRegFail = TRUE;
		}
	}
	_ASSERT(NUM_KBJOY_KEYS - 2 <= sizeof(ULONG) * 2);

	i = NUM_KBJOY_KEYS - 1;

	g_anKeyset[ nKeyset + NUM_KBJOY_DEVICES ][ i-- ] = (WORD)(ulKeysetReg3 & 0x000000ffL);
	ulKeysetReg3 = ulKeysetReg3 >> 8;
	g_anKeyset[ nKeyset + NUM_KBJOY_DEVICES ][ i-- ] = (WORD)(ulKeysetReg3 & 0x000000ffL);

	for( ; i > -1; )
	{
		g_anKeyset[ nKeyset + NUM_KBJOY_DEVICES ][ i-- ] = (WORD)(ulKeysetReg2 & 0x000000ffL);
		ulKeysetReg2 = ulKeysetReg2 >> 8;
		g_anKeyset[ nKeyset + NUM_KBJOY_DEVICES ][ i-- ] = (WORD)(ulKeysetReg1 & 0x000000ffL);
		ulKeysetReg1 = ulKeysetReg1 >> 8;
	}

	if( hkInput == NULL )
		RegCloseKey( hkKey );

	return (bRegFail ? FALSE : TRUE);
}

void
WriteRegKeyset( HKEY hkInput,
			    int  nKeyset )
{
	ULONG ulKeysetReg1 = 0L;
	ULONG ulKeysetReg2 = 0L;
	ULONG ulKeysetReg3 = 0L;
	HKEY  hkKey        = NULL;
	int   i;

	if( hkInput == NULL )
	{
		if( RegOpenKeyEx(
			HKEY_CURRENT_USER,		// handle of open key 
			REGNAME,				// address of name of subkey to open 
			0,						// reserved 
			KEY_ALL_ACCESS,			// security access mask 
			&hkKey 					// address of handle of open key 
			) != ERROR_SUCCESS )
		{
			DisplayMessage( IDS_REG_OPEN_ERROR, 0, MB_ICONSTOP | MB_OK );
			return;
		}
	}
	else
		hkKey = hkInput;

	/* The key values will be packed into DWORDs */
	_ASSERT(NUM_KBJOY_KEYS - 2 <= sizeof(ULONG) * 2);

	for( i = 0; i < NUM_KBJOY_KEYS - 2; )
	{
		ulKeysetReg1 = ulKeysetReg1 << 8;
		ulKeysetReg1 |= g_anKeyset[ nKeyset + NUM_KBJOY_DEVICES ][ i++ ] & 0x000000ffL;
		ulKeysetReg2 = ulKeysetReg2 << 8;
		ulKeysetReg2 |= g_anKeyset[ nKeyset + NUM_KBJOY_DEVICES ][ i++ ] & 0x000000ffL;
	}
	ulKeysetReg3 |= g_anKeyset[ nKeyset + NUM_KBJOY_DEVICES ][ i++ ] & 0x000000ffL;
	ulKeysetReg3 = ulKeysetReg3 << 8;
	ulKeysetReg3 |= g_anKeyset[ nKeyset + NUM_KBJOY_DEVICES ][ i ] & 0x000000ffL;

	if( KEYS_A_JOYSTICK == nKeyset )
	{
		WriteRegDWORD( hkKey, REG_KEYSET_A1, ulKeysetReg1 );
		WriteRegDWORD( hkKey, REG_KEYSET_A2, ulKeysetReg2 );
		WriteRegDWORD( hkKey, REG_KEYSET_A3, ulKeysetReg3 );
	}
	else
	{
		WriteRegDWORD( hkKey, REG_KEYSET_B1, ulKeysetReg1 );
		WriteRegDWORD( hkKey, REG_KEYSET_B2, ulKeysetReg2 );
		WriteRegDWORD( hkKey, REG_KEYSET_B3, ulKeysetReg3 );
	}
}

void
WriteRegDrives( HKEY hkInput )
{
	HKEY hkKey = NULL;

	if( hkInput == NULL )
	{
		if( RegOpenKeyEx(
			HKEY_CURRENT_USER,		// handle of open key 
			REGNAME,				// address of name of subkey to open 
			0,						// reserved 
			KEY_ALL_ACCESS,			// security access mask 
			&hkKey 					// address of handle of open key 
			) != ERROR_SUCCESS )
		{
			DisplayMessage( IDS_REG_OPEN_ERROR, 0, MB_ICONSTOP | MB_OK );
			return;
		}
	}
	else
		hkKey = hkInput;

	WriteRegString( hkKey, REG_HD1, atari_h1_dir );
	WriteRegString( hkKey, REG_HD2, atari_h2_dir );
	WriteRegString( hkKey, REG_HD3, atari_h3_dir );
	WriteRegString( hkKey, REG_HD4, atari_h4_dir );

	WriteRegString( hkKey, g_apszDriveRegID[ 0 ], sio_filename[ 0 ] );
	WriteRegString( hkKey, g_apszDriveRegID[ 1 ], sio_filename[ 1 ] );
	WriteRegString( hkKey, g_apszDriveRegID[ 2 ], sio_filename[ 2 ] );
	WriteRegString( hkKey, g_apszDriveRegID[ 3 ], sio_filename[ 3 ] );
	WriteRegString( hkKey, g_apszDriveRegID[ 4 ], sio_filename[ 4 ] );
	WriteRegString( hkKey, g_apszDriveRegID[ 5 ], sio_filename[ 5 ] );
	WriteRegString( hkKey, g_apszDriveRegID[ 6 ], sio_filename[ 6 ] );
	WriteRegString( hkKey, g_apszDriveRegID[ 7 ], sio_filename[ 7 ] );

	WriteRegString( hkKey, REG_EXE_PATH, atari_exe_dir );
	WriteRegString( hkKey, REG_STATE_PATH, atari_state_dir );

	if( hkInput == NULL )
		RegCloseKey( hkKey );
}

void
WriteAtari800Registry( HKEY hkInput )
{
	HKEY hkKey = NULL;

	if( hkInput == NULL )
	{
		if( RegOpenKeyEx(
			HKEY_CURRENT_USER,	// handle of open key 
			REGNAME,			// address of name of subkey to open 
			0,					// reserved 
			KEY_ALL_ACCESS,		// security access mask 
			&hkKey 				// address of handle of open key 
			) != ERROR_SUCCESS )
		{
			DisplayMessage( IDS_REG_OPEN_ERROR, 0, MB_ICONSTOP | MB_OK );
			return;
		}
	}
	else
		hkKey = hkInput;

	WriteRegDWORD ( hkKey, REG_REFRESH_RATE,      refresh_rate         );
	WriteRegDWORD ( hkKey, REG_DEFAULT_SYSTEM,    default_system       );
	WriteRegDWORD ( hkKey, REG_DEFAULT_TV_MODE,   default_tv_mode      );
	WriteRegDWORD ( hkKey, REG_HD_READ_ONLY,      hd_read_only         );
	WriteRegDWORD ( hkKey, REG_HOLD_OPTION,       hold_option          );
	WriteRegDWORD ( hkKey, REG_ENABLE_SIO_PATCH,  enable_sio_patch     );
	WriteRegDWORD ( hkKey, REG_ENABLE_C000_RAM,   enable_c000_ram      );
	WriteRegDWORD ( hkKey, REG_ENABLE_ROM_PATCH,  enable_rom_patches   );
	WriteRegDWORD ( hkKey, REG_ENABLE_STEREO,     stereo_enabled       );
	WriteRegDWORD ( hkKey, REG_CART_TYPE,         cart_type            );
	WriteRegDWORD ( hkKey, REG_ARTIF_MODE,        global_artif_mode    );
	WriteRegString( hkKey, REG_OSA_ROM,           atari_osa_filename   );
	WriteRegString( hkKey, REG_OSB_ROM,           atari_osb_filename   );
	WriteRegString( hkKey, REG_XLXE_ROM,          atari_xlxe_filename  );
	WriteRegString( hkKey, REG_5200_ROM,          atari_5200_filename  );
	WriteRegString( hkKey, REG_BASIC_ROM,         atari_basic_filename );
	WriteRegDWORD ( hkKey, REG_USE_VOLUME_ONLY,   g_nUseVolumeOnly     );
	WriteRegString( hkKey, REG_CURRENT_ROM,       g_szCurrentRom       );
	WriteRegString( hkKey, REG_OTHER_ROM,         g_szOtherRom         );
	WriteRegString( hkKey, REG_KEY_TEMPLATE,      g_szTemplateFile     );
	WriteRegString( hkKey, REG_EXT_PALETTE,       g_szPaletteFile      );
	WriteRegDWORD ( hkKey, REG_MISC_STATES,       g_ulMiscStates       );
	WriteRegDWORD ( hkKey, REG_DDRAW_MODE,        g_ulScreenMode       );
	WriteRegDWORD ( hkKey, REG_START_XPOS,        g_nStartX            );
	WriteRegDWORD ( hkKey, REG_START_YPOS,        g_nStartY            );
	WriteRegDWORD ( hkKey, REG_SOUND_STATE,       g_ulSoundState       );
	WriteRegDWORD ( hkKey, REG_SOUND_RATE,        g_nSoundRate         );
	WriteRegDWORD ( hkKey, REG_SOUND_VOLUME,      g_nSoundVol          );
	WriteRegDWORD ( hkKey, REG_SOUND_UPDATE,      g_nSkipUpdateDefault );
	WriteRegDWORD ( hkKey, REG_DONT_SHOW,         g_ulDontShowFlags    );
	WriteRegDWORD ( hkKey, REG_JOYSTICKS,         g_ulJoystickSelects  );
	WriteRegDWORD ( hkKey, REG_STICK_REPEAT,      g_nJoyRepeat         );
	WriteRegDWORD ( hkKey, REG_STICK_SKIP,        g_nJoySkip           );
	WriteRegDWORD ( hkKey, REG_MOUSE_MODE,        g_nMouseMode         );
	WriteRegDWORD ( hkKey, REG_MOUSE_PORT,        g_nMousePort         );
	WriteRegDWORD ( hkKey, REG_MOUSE_SPEED,       g_nMouseSpeed        );
	WriteRegDWORD ( hkKey, REG_PADDLE_RANGE,      g_nPaddleRange       );
	WriteRegDWORD ( hkKey, REG_JOYSTICK_INERTIA,  g_nJoystickInertia   );
	WriteRegDWORD ( hkKey, REG_PEN_XOFFSET,       g_nPenOffsetX        );
	WriteRegDWORD ( hkKey, REG_PEN_YOFFSET,       g_nPenOffsetY        );
	WriteRegDWORD ( hkKey, REG_FILE_ASSOCIATIONS, g_ulFileAssociations );
	WriteRegDWORD ( hkKey, REG_STRETCHED_RATE,    g_nStretchedRate     );
	WriteRegDWORD ( hkKey, REG_SPEED_PERCENT,     g_nSpeedPercent      );
	WriteRegDWORD ( hkKey, REG_CURRENT_REV,       s_nVersion           );

	WriteRegDrives( hkKey );

	WriteRegKeyset( hkKey, KEYS_A_JOYSTICK );
	WriteRegKeyset( hkKey, KEYS_B_JOYSTICK );

	if( hkInput == NULL )
		RegCloseKey( hkKey );
}

void
InitialiseRegistry( HKEY hkInput,
					BOOL bErasePaths )
{
	char szDirPath[ MAX_PATH ];
	HKEY hkKey;
	int  i;

	if( !hkInput )
	{
		DWORD disposition = REG_OPENED_EXISTING_KEY;
		
		if( RegOpenKeyEx(
			HKEY_CURRENT_USER,		// handle of open key 
			REGNAME,				// address of name of subkey to open 
			0,						// reserved 
			KEY_READ,				// security access mask 
			&hkKey 					// address of handle of open key 
			) != ERROR_SUCCESS )
		{
			if( RegCreateKeyEx(
				HKEY_CURRENT_USER,		// handle of an open key 
				REGNAME,				// address of subkey name 
				0,						// reserved 
				"Atari800Win",			// address of class string 
				REG_OPTION_NON_VOLATILE,// special options flag 
				KEY_ALL_ACCESS,			// desired security access 
				NULL,					// address of key security structure 
				&hkKey,					// address of buffer for opened handle  
				&disposition 			// address of disposition value buffer 
				) != ERROR_SUCCESS )
			{
				DisplayMessage( IDS_REG_OPEN_ERROR, 0, MB_ICONSTOP | MB_OK );
				/* Probably should make this fault in a more elegant manner */
				exit( 1 );
			}
		}
	}
	else
		hkKey = hkInput;

	refresh_rate         = 1;
	default_system       = 3;
	default_tv_mode      = 1;
	hd_read_only         = 1;
	hold_option          = 1;
	enable_c000_ram      = 0;
	enable_sio_patch     = 1;
	enable_rom_patches   = 1;
	stereo_enabled       = 0;
	cart_type            = NO_CART;
	global_artif_mode    = 0;
	g_nUseVolumeOnly     = 0;
	g_nSoundRate         = 44100;
	g_nSoundVol          = 0;
	g_nSkipUpdateDefault = POKEY_UPDATE_DIVISOR_DEFAULT;
	g_ulScreenMode       = DISPLAY_MODE_DEFAULT;
	g_ulSoundState       = SOUND_MMSOUND | SOUND_CUSTOM_RATE;
	g_ulMiscStates       = MS_STICK_RELEASE;
	g_ulDontShowFlags    = 0L;
	g_ulJoystickSelects  = 0x7f7f7ffc;
	g_nJoyRepeat         = 0;
	g_nJoySkip           = 0;
	g_nMouseMode         = MOUSE_OFF;
	g_nMousePort         = 0;
	g_nMouseSpeed        = 5;
	g_nPaddleRange       = 228;
	g_nJoystickInertia   = 5;
	g_nPenOffsetX        = 0;
	g_nPenOffsetY        = 0;
	g_nStretchedRate     = 2;
	g_nSpeedPercent      = 100;

	/* Clear keysets A & B */
	for( i = 0; i < NUM_KBJOY_KEYS; i++ )
	{
		g_anKeyset[ KEYS_A_JOYSTICK + NUM_KBJOY_DEVICES ][ i ] = 0;
		g_anKeyset[ KEYS_B_JOYSTICK + NUM_KBJOY_DEVICES ][ i ] = 0;
	}
	
	GetHomeDirectory( szDirPath );
	
	if( bErasePaths || *atari_osa_filename == '\0' ) /* WinNT wants this */
	{
		strcpy( atari_osa_filename, szDirPath );
		strcat( atari_osa_filename, DEFAULT_OSA );
	}
	if( bErasePaths || *atari_osb_filename == '\0' )
	{
		strcpy( atari_osb_filename, szDirPath );
		strcat( atari_osb_filename, DEFAULT_OSB );
	}
	if( bErasePaths || *atari_xlxe_filename == '\0' )
	{
		strcpy( atari_xlxe_filename, szDirPath );
		strcat( atari_xlxe_filename, DEFAULT_OXL );
	}
	if( bErasePaths || *atari_5200_filename == '\0' )
	{
		strcpy( atari_5200_filename, szDirPath );
		strcat( atari_5200_filename, DEFAULT_O52 );
	}
	if( bErasePaths || *atari_basic_filename == '\0' )
	{
		strcpy( atari_basic_filename, szDirPath );
		strcat( atari_basic_filename, DEFAULT_BAS );
	}
	if( bErasePaths || *g_szTemplateFile == '\0' )
		strncpy( g_szTemplateFile, DEFAULT_A8K, MAX_PATH );
	if( bErasePaths || *g_szPaletteFile == '\0' )
		strncpy( g_szPaletteFile, DEFAULT_ACT, MAX_PATH );
	if( bErasePaths || *atari_state_dir == '\0' )
		strncpy( atari_state_dir, DEFAULT_A8S, MAX_PATH );

	if( bErasePaths || *atari_h1_dir == '\0' )
		strcpy( atari_h1_dir, "." );
	if( bErasePaths || *atari_h2_dir == '\0' )
		strcpy( atari_h2_dir, "." );
	if( bErasePaths || *atari_h3_dir == '\0' )
		strcpy( atari_h3_dir, "." );
	if( bErasePaths || *atari_h4_dir == '\0' )
		strcpy( atari_h4_dir, "." );

	if( bErasePaths || *atari_exe_dir == '\0' )
		strcpy( atari_exe_dir, "." );

	strcpy( g_szCurrentRom, "None" );
	strcpy( g_szOtherRom,   "None" );

	for( i = 0; i < MAX_DRIVES; i++ )
	{
		strcpy( sio_filename[ i ], "Off" ) ;
		drive_status[ i ] = Off;
	}

	WriteAtari800Registry( hkKey );

	if( hkInput == NULL )
		RegCloseKey( hkKey );
}

BOOL
ReadRegPaths( HKEY hkInput )
{
	HKEY	hkKey    = NULL;
	BOOL	bRegFail = FALSE;
	char	szDirPath[ MAX_PATH ];

	GetCurrentDirectory( MAX_PATH, szDirPath );
	if( !hkInput )
	{
		if( RegOpenKeyEx(
			HKEY_CURRENT_USER,		// handle of open key 
			REGNAME,				// address of name of subkey to open 
			0,						// reserved 
			KEY_READ,				// security access mask 
			&hkKey 					// address of handle of open key 
			) != ERROR_SUCCESS )
		{
			DisplayMessage( IDS_REG_OPEN_ERROR, 0, MB_ICONSTOP | MB_OK );
			return FALSE;
		}
	}
	else
		hkKey = hkInput;

	if( ReadRegBinary( hkInput, REG_OSA_ROM, atari_osa_filename, MAX_PATH, FALSE ) == READ_REG_FAIL )
	{
		strcpy( atari_osa_filename, szDirPath );
		strcat( atari_osa_filename, DEFAULT_OSA );
		bRegFail = TRUE;
	}
	if( ReadRegBinary( hkInput, REG_OSB_ROM, atari_osb_filename, MAX_PATH, FALSE ) == READ_REG_FAIL )
	{
		strcpy( atari_osb_filename, szDirPath );
		strcat( atari_osb_filename, DEFAULT_OSB );
		bRegFail = TRUE;
	}
	if( ReadRegBinary( hkInput, REG_XLXE_ROM, atari_xlxe_filename, MAX_PATH, FALSE ) == READ_REG_FAIL )
	{
		strcpy( atari_xlxe_filename, szDirPath );
		strcat( atari_xlxe_filename, DEFAULT_OXL );
		bRegFail = TRUE;
	}
	if( ReadRegBinary( hkInput, REG_5200_ROM, atari_5200_filename, MAX_PATH, FALSE ) == READ_REG_FAIL )
	{
		strcpy( atari_5200_filename, szDirPath );
		strcat( atari_5200_filename, DEFAULT_O52 );
		bRegFail = TRUE;
	}
	if( ReadRegBinary( hkInput, REG_BASIC_ROM, atari_basic_filename, MAX_PATH, FALSE ) == READ_REG_FAIL )
	{
		strcpy( atari_basic_filename, szDirPath );
		strcat( atari_basic_filename, DEFAULT_BAS );
		bRegFail = TRUE;
	}
	if( ReadRegBinary( hkInput, REG_KEY_TEMPLATE, g_szTemplateFile, MAX_PATH, FALSE ) == READ_REG_FAIL )
	{
		strncpy( g_szTemplateFile, DEFAULT_A8K, MAX_PATH );
		bRegFail = TRUE;
	}
	if( ReadRegBinary( hkInput, REG_EXT_PALETTE, g_szPaletteFile, MAX_PATH, FALSE ) == READ_REG_FAIL )
	{
		strncpy( g_szPaletteFile, DEFAULT_ACT, MAX_PATH );
		bRegFail = TRUE;
	}
	if( ReadRegBinary( hkInput, REG_STATE_PATH, atari_state_dir, MAX_PATH, FALSE ) == READ_REG_FAIL )
	{
		strncpy( atari_state_dir, DEFAULT_A8S, MAX_PATH );
		bRegFail = TRUE;
	}
	if( ReadRegBinary( hkInput, REG_CURRENT_ROM, g_szCurrentRom, MAX_PATH, FALSE ) == READ_REG_FAIL )
	{
		strcpy( g_szCurrentRom, "None" );
		bRegFail = TRUE;
	}
	if( ReadRegBinary( hkInput, REG_OTHER_ROM, g_szOtherRom, MAX_PATH, FALSE ) == READ_REG_FAIL )
	{
		strcpy( g_szOtherRom, "None" );
		bRegFail = TRUE;
	}
	if( ReadRegBinary( hkInput, REG_HD1, atari_h1_dir, MAX_PATH, FALSE ) == READ_REG_FAIL )
	{
		strcpy( atari_h1_dir, "." );
		bRegFail = TRUE;
	}
	if( ReadRegBinary( hkInput, REG_HD2, atari_h2_dir, MAX_PATH, FALSE ) == READ_REG_FAIL )
	{
		strcpy( atari_h2_dir, "." );
		bRegFail = TRUE;
	}
	if( ReadRegBinary( hkInput, REG_HD3, atari_h3_dir, MAX_PATH, FALSE ) == READ_REG_FAIL )
	{
		strcpy( atari_h3_dir, "." );
		bRegFail = TRUE;
	}
	if( ReadRegBinary( hkInput, REG_HD4, atari_h4_dir, MAX_PATH, FALSE ) == READ_REG_FAIL )
	{
		strcpy( atari_h4_dir, "." );
		bRegFail = TRUE;
	}
	if( ReadRegBinary( hkInput, REG_EXE_PATH, atari_exe_dir, MAX_PATH, FALSE ) == READ_REG_FAIL )
	{
		strcpy( atari_exe_dir, "." );
		bRegFail = TRUE;
	}
	if( hkInput == NULL )
		RegCloseKey( hkKey );

	return (bRegFail ? FALSE : TRUE);
}

void
ReadRegDrives( HKEY hkInput )
{
	HKEY hkKey    = NULL;
	BOOL bRegFail = FALSE;
	int  i;

	if( !hkInput )
	{
		if( RegOpenKeyEx(
			HKEY_CURRENT_USER,		// handle of open key 
			REGNAME,				// address of name of subkey to open 
			0,						// reserved 
			KEY_READ,				// security access mask 
			&hkKey 					// address of handle of open key 
			) != ERROR_SUCCESS )
		{
			DisplayMessage( IDS_REG_OPEN_ERROR, 0, MB_ICONSTOP | MB_OK );
			return;
		}
	}
	else
		hkKey = hkInput;

	if( ReadRegBinary( hkKey, g_apszDriveRegID[ 0 ], sio_filename[ 0 ], MAX_PATH, FALSE ) == READ_REG_FAIL )
	{
		strcpy( sio_filename[ 0 ], "Off" );
		bRegFail = TRUE;
	}
	if( ReadRegBinary( hkKey, g_apszDriveRegID[ 1 ], sio_filename[ 1 ], MAX_PATH, FALSE ) == READ_REG_FAIL )
	{
		strcpy( sio_filename[ 1 ], "Off" );
		bRegFail = TRUE;
	}
	if( ReadRegBinary( hkKey, g_apszDriveRegID[ 2 ], sio_filename[ 2 ], MAX_PATH, FALSE ) == READ_REG_FAIL )
	{
		strcpy( sio_filename[ 2 ], "Off" );
		bRegFail = TRUE;
	}
	if( ReadRegBinary( hkKey, g_apszDriveRegID[ 3 ], sio_filename[ 3 ], MAX_PATH, FALSE ) == READ_REG_FAIL )
	{
		strcpy( sio_filename[ 3 ], "Off" );
		bRegFail = TRUE;
	}
	if( ReadRegBinary( hkKey, g_apszDriveRegID[ 4 ], sio_filename[ 4 ], MAX_PATH, FALSE ) == READ_REG_FAIL )
	{
		strcpy( sio_filename[ 4 ], "Off" );
		bRegFail = TRUE;
	}
	if( ReadRegBinary( hkKey, g_apszDriveRegID[ 5 ], sio_filename[ 5 ], MAX_PATH, FALSE ) == READ_REG_FAIL )
	{
		strcpy( sio_filename[ 5 ], "Off" );
		bRegFail = TRUE;
	}
	if( ReadRegBinary( hkKey, g_apszDriveRegID[ 6 ], sio_filename[ 6 ], MAX_PATH, FALSE ) == READ_REG_FAIL )
	{
		strcpy( sio_filename[ 6 ], "Off" );
		bRegFail = TRUE;
	}
	if( ReadRegBinary( hkKey, g_apszDriveRegID[ 7 ], sio_filename[ 7 ], MAX_PATH, FALSE ) == READ_REG_FAIL )
	{
		strcpy( sio_filename[ 7 ], "Off" );
		bRegFail = TRUE;
	}

	for( i = 0; i < MAX_DRIVES; i++ )
	{
		if( *sio_filename[ i ] == '\0' )
			strcpy( sio_filename[ i ], "Off" );

		if( strcmp( sio_filename[ i ], "Off" ) == 0 )
			drive_status[ i ] = Off;
		else
		if( strcmp( sio_filename[ i ], "Empty" ) == 0 )
			drive_status[ i ] = NoDisk;
	}

	if( hkInput == NULL )
		RegCloseKey( hkKey );

	if( bRegFail )
		WriteAtari800Registry( NULL );
}

/*========================================================
Function : HandleRegistry
=========================================================*/
/* #FN#
   Creates the Registry entries if they don't exist and read all
   the defaults in at runtime (this is called from MainFrame) */
BOOL
/* #AS#
   TRUE if the Registry has been initialized, otherwise FALSE */
HandleRegistry( void )
{
	DWORD disposition = REG_OPENED_EXISTING_KEY;
	BOOL  bRegFail    = FALSE;
	BOOL  bInitialReg = FALSE;
	HKEY  hkInitKey;

	if( RegOpenKeyEx(
		HKEY_CURRENT_USER,		// handle of open key 
		REGNAME,				// address of name of subkey to open 
		0,						// reserved 
		KEY_ALL_ACCESS,			// security access mask 
		&hkInitKey 				// address of handle of open key 
		) != ERROR_SUCCESS )
	{
		if( RegCreateKeyEx(
			HKEY_CURRENT_USER,		// handle of an open key 
			REGNAME,				// address of subkey name 
			0,						// reserved 
			"Atari800Win",			// address of class string 
			REG_OPTION_NON_VOLATILE,// special options flag 
			KEY_ALL_ACCESS,			// desired security access 
			NULL,					// address of key security structure 
			&hkInitKey,				// address of buffer for opened handle  
			&disposition 			// address of disposition value buffer 
			) != ERROR_SUCCESS )
		{
			DisplayMessage( IDS_REG_OPEN_ERROR, 0, MB_ICONSTOP | MB_OK );
			// Probably should make this fault in a more elegant manner
			exit( 1 );
		}
	}
	
	// If the key doesn't exist, fill in defaults. This is the only time these
	// will be written all at once. From here on out we trust the Registry to hold them
	// (yeah, right) and update when key dialogs are used to configure them
	if( disposition == REG_CREATED_NEW_KEY )
	{
		bInitialReg = TRUE;
		InitialiseRegistry( hkInitKey, !TestRomPaths( NULL, NULL ) );
	}
	else
	{
		// Read in the values from the Registry. Only fail and return error when it
		// is REALLY fatal (you never know what somebody might do with their registry)
		// For most of these an error will result in the value being the default run-time
		if( ReadRegDWORD( hkInitKey, REG_CURRENT_REV, (DWORD *)&s_nVersion, FALSE) == READ_REG_FAIL )
			s_nVersion = CURRENT_REV - 1;
		
		if( s_nVersion != CURRENT_REV )
		{
			DisplayMessage( IDS_WARNING_OUTDATED, 0, MB_ICONINFORMATION | MB_OK );
			s_nVersion = CURRENT_REV;

			ReadRegPaths( hkInitKey );	/* Since we already have a registry, read the paths + filenames at least */
										/* Note that this will have to change if I ever invalidate the path system (unlikely) */
			DeleteAllRegKeys( hkInitKey, REGNAME );

			RegCloseKey( hkInitKey );
			if( RegCreateKeyEx(
				HKEY_CURRENT_USER,		// handle of an open key 
				REGNAME,				// address of subkey name 
				0,						// reserved 
				"Atari800Win",			// address of class string 
				REG_OPTION_NON_VOLATILE,// special options flag 
				KEY_ALL_ACCESS,			// desired security access 
				NULL,					// address of key security structure 
				&hkInitKey,				// address of buffer for opened handle  
				&disposition 			// address of disposition value buffer 
				) != ERROR_SUCCESS )
			{
				DisplayMessage( IDS_REG_OPEN_ERROR, 0, MB_ICONSTOP | MB_OK );
				// Probably should make this fault in a more elegant manner
				exit( 1 );
			}
			InitialiseRegistry( hkInitKey, FALSE );
			bInitialReg = TRUE;
		}
		else
		{
			if( ReadRegDWORD( hkInitKey, REG_REFRESH_RATE, (DWORD *)&refresh_rate, TRUE ) == READ_REG_FAIL )
			{
				refresh_rate = 1;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_STRETCHED_RATE, (DWORD *)&g_nStretchedRate, TRUE ) == READ_REG_FAIL )
			{
				g_nStretchedRate = 2;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_HD_READ_ONLY, (DWORD *)&hd_read_only, TRUE ) == READ_REG_FAIL )
			{
				hd_read_only = 1;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_SPEED_PERCENT, (DWORD *)&g_nSpeedPercent, TRUE ) == READ_REG_FAIL )
			{
				g_nSpeedPercent = 100;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_MOUSE_PORT, (DWORD *)&g_nMousePort, TRUE ) == READ_REG_FAIL )
			{
				g_nMousePort = 0;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_MOUSE_SPEED, (DWORD *)&g_nMouseSpeed, TRUE ) == READ_REG_FAIL )
			{
				g_nMouseSpeed = 5;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_PADDLE_RANGE, (DWORD *)&g_nPaddleRange, TRUE ) == READ_REG_FAIL )
			{
				g_nPaddleRange = 228;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_JOYSTICK_INERTIA, (DWORD *)&g_nJoystickInertia, TRUE ) == READ_REG_FAIL )
			{
				g_nJoystickInertia = 5;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_PEN_XOFFSET, (DWORD *)&g_nPenOffsetX, TRUE ) == READ_REG_FAIL )
			{
				g_nPenOffsetX = 0;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_PEN_YOFFSET, (DWORD *)&g_nPenOffsetY, TRUE ) == READ_REG_FAIL )
			{
				g_nPenOffsetY = 0;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_DEFAULT_SYSTEM, (DWORD *)&default_system, TRUE ) == READ_REG_FAIL )
			{
				default_system = 3;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_DEFAULT_TV_MODE, (DWORD *)&default_tv_mode, TRUE ) == READ_REG_FAIL )
			{
				default_tv_mode = 1;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_HOLD_OPTION, (DWORD *)&hold_option, TRUE ) == READ_REG_FAIL )
			{
				hold_option = 1;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_ENABLE_C000_RAM, (DWORD *)&enable_c000_ram, TRUE ) == READ_REG_FAIL )
			{
				enable_c000_ram = 0;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_ENABLE_SIO_PATCH, (DWORD *)&enable_sio_patch, TRUE ) == READ_REG_FAIL )
			{
				enable_sio_patch = 1;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_ENABLE_ROM_PATCH, (DWORD *)&enable_rom_patches, TRUE ) == READ_REG_FAIL )
			{
				enable_rom_patches = 1;
				bRegFail = TRUE;
			}
#ifndef NO_VOL_ONLY
			if( ReadRegDWORD( hkInitKey, REG_USE_VOLUME_ONLY, (DWORD *)&g_nUseVolumeOnly, TRUE ) == READ_REG_FAIL )
			{
#endif /*NO_VOL_ONLY*/
				g_nUseVolumeOnly = 0;
#ifndef NO_VOL_ONLY
				bRegFail = TRUE;
			}
#endif /*NO_VOL_ONLY*/
#ifdef STEREO
			if( ReadRegDWORD( hkInitKey, REG_ENABLE_STEREO, (DWORD *)&stereo_enabled, TRUE ) == READ_REG_FAIL )
			{
#endif /*STEREO*/
				stereo_enabled = 0;
#ifdef STEREO
				bRegFail = TRUE;
			}
#endif /*STEREO*/
			if( ReadRegDWORD( hkInitKey, REG_ARTIF_MODE, (DWORD *)&global_artif_mode, TRUE ) == READ_REG_FAIL )
			{
				global_artif_mode = 0;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_DDRAW_MODE, (DWORD *)&g_ulScreenMode, FALSE ) == READ_REG_FAIL )
			{
				g_ulScreenMode = DISPLAY_MODE_DEFAULT;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_FILE_ASSOCIATIONS, (DWORD *)&g_ulFileAssociations, TRUE ) == READ_REG_FAIL )
			{
				g_ulFileAssociations = 0L;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_START_XPOS, (DWORD *)&g_nStartX, FALSE ) == READ_REG_FAIL )
			{
				g_nStartX = 0;
				bRegFail = TRUE;
			}

			if( g_nStartX < 0 )
				g_nStartX = 0;
			if( g_nStartX > GetSystemMetrics( SM_CXFULLSCREEN ) )
				g_nStartX = GetSystemMetrics( SM_CXFULLSCREEN ) - (ATARI_WIDTH);
			
			if( ReadRegDWORD( hkInitKey, REG_START_YPOS, (DWORD *)&g_nStartY, FALSE ) == READ_REG_FAIL )
			{
				g_nStartY = 0;
				bRegFail = TRUE;
			}

			if( g_nStartY < 0 )
				g_nStartY = 0;
			if( g_nStartY > GetSystemMetrics( SM_CYFULLSCREEN ) )
				g_nStartY = GetSystemMetrics( SM_CYFULLSCREEN ) - ATARI_HEIGHT;
			
			if( ReadRegDWORD( hkInitKey, REG_SOUND_RATE, (DWORD *)&g_nSoundRate, FALSE) == READ_REG_FAIL )
			{
				g_nSoundRate = 44100;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_SOUND_STATE, (DWORD *)&g_ulSoundState, FALSE) == READ_REG_FAIL )
			{
				g_ulSoundState = SOUND_MMSOUND;
				if( g_nSoundRate != 22050 )
					g_ulSoundState |= SOUND_CUSTOM_RATE;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_SOUND_VOLUME, (DWORD *)&g_nSoundVol, FALSE) == READ_REG_FAIL )
			{
				g_nSoundVol = 0;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_SOUND_UPDATE, (DWORD *)&g_nSkipUpdateDefault, FALSE) == READ_REG_FAIL )
			{
				g_nSkipUpdateDefault = POKEY_UPDATE_DIVISOR_DEFAULT;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_MISC_STATES, (DWORD *)&g_ulMiscStates, FALSE) == READ_REG_FAIL )
			{
				g_ulMiscStates = MS_STICK_RELEASE;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_DONT_SHOW, (DWORD *)&g_ulDontShowFlags, FALSE) == READ_REG_FAIL )
			{
				g_ulDontShowFlags = 0L;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_STICK_REPEAT, (DWORD *)&g_nJoyRepeat, FALSE) == READ_REG_FAIL )
			{
				g_nJoyRepeat = 0;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_STICK_SKIP, (DWORD *)&g_nJoySkip, FALSE) == READ_REG_FAIL )
			{
				g_nJoySkip = 0;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_JOYSTICKS, (DWORD *)&g_ulJoystickSelects, FALSE) == READ_REG_FAIL )
			{
				g_ulJoystickSelects = 0x7f7f7ffc;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_MOUSE_MODE, (DWORD *)&g_nMouseMode, FALSE) == READ_REG_FAIL )
			{
				g_nMouseMode = MOUSE_PAD;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_CART_TYPE, (DWORD *)&cart_type, FALSE) == READ_REG_FAIL )
			{
				cart_type = NO_CART;
				bRegFail = TRUE;
			}
			if( !ReadRegPaths( NULL ) )
				bRegFail = TRUE;

			if( !ReadRegKeyset( hkInitKey, KEYS_A_JOYSTICK ) )
				bRegFail = TRUE;
			if( !ReadRegKeyset( hkInitKey, KEYS_B_JOYSTICK ) )
				bRegFail = TRUE;
		}
	}
	RegCloseKey( hkInitKey );

	if( bRegFail )
		WriteAtari800Registry( NULL );

	return bInitialReg;
} /* #OF# HandleRegistry */

/*========================================================
Function : GetRegKeyHandle
=========================================================*/
/* #FN#
   Opens/Creates registry key pszKeyName in HKEY_CLASSES_ROOT section */
static
HKEY
/* #AS#
   Opened/Created registry key handle */
GetRegKeyHandle(
	LPCSTR pszKeyName, /* #IN# Name of registry key */
	BOOL   bCreateKey  /* #IN# Create the specified key if the key does not exist in the registry */ )
{
	HKEY  hkKey       = NULL;
	DWORD disposition = REG_OPENED_EXISTING_KEY;
	BOOL  bRegFail    = TRUE;

	if( RegOpenKeyEx(
			HKEY_CLASSES_ROOT,		// handle of open key 
			pszKeyName,				// address of name of subkey to open 
			0,						// reserved 
			KEY_ALL_ACCESS,			// security access mask 
			&hkKey					// address of handle of open key 
		) != ERROR_SUCCESS )
	{
		if( bCreateKey )
			if( RegCreateKeyEx(
					HKEY_CLASSES_ROOT,			// handle of an open key 
					pszKeyName,					// address of subkey name 
					0,							// reserved 
					"Atari800Win",				// address of class string 
					REG_OPTION_NON_VOLATILE,	// special options flag 
					KEY_ALL_ACCESS,				// desired security access 
					NULL,						// address of key security structure 
					&hkKey,						// address of buffer for opened handle  
					&disposition 				// address of disposition value buffer 
				) != ERROR_SUCCESS )
			{
				DisplayMessage( IDS_REG_OPEN_ERROR, 0, MB_ICONSTOP | MB_OK );
			}
			else
				bRegFail = FALSE;
	}
	else
		bRegFail = FALSE;

	return (bRegFail ? NULL : hkKey);
} /* #OF# GetRegKeyHandle */

/*========================================================
Function : WriteRegFileExt
=========================================================*/
/* #FN#
   Sets up the file associations for shell integration */
void
/* #AS#
   Nothing */
WriteRegFileExt(
	ULONG ulFileAssociations,
	char *pszAppDir, /* #IN# Current application directory */
	char *pszAppName /* #IN# Name of application           */ )
{
	char  szKeyValue[ MAX_PATH ];
	char  szKeyEntry[ 32 ];
	char *apszOptns [ FILE_TYPES_NO ] = { "", "-run", "-rom", "-cart", "-state" };
	char *apszDescs [ FILE_TYPES_NO ] = { "Disk Image", "Executable", "ROM Image", "Cartridge Image", "State File" };
	int   nAppDirLen = strlen( pszAppDir );
	HKEY  hkKey      = NULL;
	int   i;

	/* Ending backslash is forbidden */
	if( nAppDirLen > 0 && pszAppDir[ nAppDirLen - 1 ] == '\\' )
		pszAppDir[ nAppDirLen - 1 ] = '\0';

	for( i = 0; i < FILE_TYPES_NO; i++ )
	{
		/* Name the main file type entries */
		if( hkKey = GetRegKeyHandle( s_apszRootType[ i ], TRUE ) )
		{
			sprintf( szKeyValue, "Atari 8-bit %s", apszDescs[ i ] );
			WriteRegString( hkKey, "", szKeyValue );
			RegCloseKey( hkKey );
		}
		/* Write open command for this file type */
		sprintf( szKeyEntry, "%s\\shell\\open\\command", s_apszRootType[ i ] );
		if( hkKey = GetRegKeyHandle( szKeyEntry, TRUE ) )
		{
			sprintf( szKeyValue, "%s\\%s.exe %s \"%%1\"", pszAppDir, pszAppName, apszOptns[ i ] );
			WriteRegString( hkKey, "", szKeyValue );
			RegCloseKey( hkKey );
		}
		/* There is the icon number 2 for shell integration */
		sprintf( szKeyEntry, "%s\\DefaultIcon", s_apszRootType[ i ] );
		if( hkKey = GetRegKeyHandle( szKeyEntry, TRUE ) )
		{
			sprintf( szKeyValue, "%s\\%s.exe,1", pszAppDir, pszAppName );
			WriteRegString( hkKey, "", szKeyValue );
			RegCloseKey( hkKey );
		}
	}

	for( i = 0; i < g_nFileExtInfoNo; i++ )
	{
		if( hkKey = GetRegKeyHandle( g_aFileExtInfo[ i ].pszFileExt, TRUE ) )
		{
			if( ulFileAssociations & g_aFileExtInfo[ i ].dwRegFlag )
			{
				/* See enum FileType values defined in globals.h file */
				WriteRegString( hkKey, "", s_apszRootType[ g_aFileExtInfo[ i ].ftType ] );
			}
			else /* delete this association */
				RegDeleteKey( hkKey, "" );/*pFileExtInfo[ i ].pszFileExt );*/
			RegCloseKey( hkKey );
		}
	}
} /* #OF# WriteRegFileExt */

/*========================================================
Function : ReadRegFileExt
=========================================================*/
/* #FN#
   Reads the current file associations state */
ULONG
/* #AS#
   Current file associations state */
ReadRegFileExt(
	char *pszAppDir, /* #IN# Current application directory */
	char *pszAppName /* #IN# Name of application           */ )
{
	char  szKeyValue   [ MAX_PATH ];
	char  szKeyBuffer  [ MAX_PATH ];
	char  szKeyEntry   [ 32 ];
	BOOL  abEntryExists[ FILE_TYPES_NO ];
	ULONG ulFileAssociations = 0L;
	int   nAppDirLen         = strlen( pszAppDir );
	HKEY  hkKey              = NULL;
	int   i;

	/* Ending backslash is forbidden */
	if( nAppDirLen > 0 && pszAppDir[ nAppDirLen - 1 ] == '\\' )
		pszAppDir[ nAppDirLen - 1 ] = '\0';

	for( i = 0; i < FILE_TYPES_NO; i++ )
	{
		abEntryExists[ i ] = FALSE;

		if( hkKey = GetRegKeyHandle( s_apszRootType[ i ], FALSE ) )
		{
			RegCloseKey( hkKey );
			/* Read open command for this file type */
			sprintf( szKeyEntry, "%s\\shell\\open\\command", s_apszRootType[ i ] );
			if( hkKey = GetRegKeyHandle( szKeyEntry, FALSE ) )
			{
				DWORD dwType;
				UINT  uiMaxSize = MAX_PATH;

				if( RegQueryValueEx(
					hkKey,					// handle of key to query 
					NULL,					// address of name of value to query 
					0,						// reserved 
					&dwType,				// address of buffer for value type 
					(UCHAR *)szKeyBuffer,	// address of data buffer 
					&uiMaxSize			 	// address of data buffer size 
					) == ERROR_SUCCESS )
				{
					sprintf( szKeyValue, "%s\\%s.exe", pszAppDir, pszAppName );
					strupr( szKeyValue ); strupr( szKeyBuffer );

					if( strstr( szKeyBuffer, szKeyValue ) )
						abEntryExists[ i ] = TRUE;
				}
				RegCloseKey( hkKey );
			}
		}
	}
	for( i = 0; i < g_nFileExtInfoNo; i++ )
	{
		if( hkKey = GetRegKeyHandle( g_aFileExtInfo[ i ].pszFileExt, FALSE ) )
		{
			if( abEntryExists[ g_aFileExtInfo[ i ].ftType ] )
				/* Read this extension flag into variable */
				ulFileAssociations |= g_aFileExtInfo[ i ].dwRegFlag;
			RegCloseKey( hkKey );
		}
	}
	return ulFileAssociations;
} /* #OF# ReadRegFileExt */
