/****************************************************************************
File    : PaletteDlg.cpp
/*
@(#) #SY# Atari800Win
@(#) #IS# CPaletteDlg implementation file
@(#) #BY# Tomasz Szymankowski
@(#) #LM# 17.09.2000
*/

/*
Copyright (c) 2000 Tomasz Szymankowski

This program is free software; you can redistribute it and/or modify it under the terms 
of the GNU General Public License as published by the Free Software Foundation; either 
version 2 of the License, or (at your option) any later version. This program is 
distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even
the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details. You should have received a copy of the GNU
General Public License along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include "StdAfx.h"
#include "Atari800Win.h"
#include "Helpers.h"
#include "PaletteDlg.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

#define IDC_PALETTE_FIRST		IDC_PALETTE_LOAD
#define IDC_PALETTE_LAST		IDC_PALETTE_CANCEL

#define PAL_ENTRIES_NO	256
#define BAR_LINES_NO	8
#define BAR_ENTRIES_NO	(PAL_ENTRIES_NO / BAR_LINES_NO)

/* Note: afxData.cxBorder and afxData.cyBorder aren't used anymore */
#define CX_BORDER   2
#define CY_BORDER   2


/////////////////////////////////////////////////////////////////////////////
// CPaletteDlg dialog

BEGIN_MESSAGE_MAP(CPaletteDlg, CDialog)
	//{{AFX_MSG_MAP(CPaletteDlg)
	ON_BN_CLICKED(IDC_PALETTE_BROWSE, OnBrowse)
	ON_EN_KILLFOCUS(IDC_PALETTE_EDIT, OnKillfocusEdit)
	ON_BN_CLICKED(IDC_PALETTE_LOAD, OnLoad)
	ON_WM_PAINT()
	ON_WM_HELPINFO()
	//}}AFX_MSG_MAP
	ON_BN_CLICKED(IDC_PALETTE_OK, OnOK)
	ON_BN_CLICKED(IDC_PALETTE_CANCEL, OnCancel)
END_MESSAGE_MAP()

/*========================================================
Method   : CPaletteDlg::CPaletteDlg
=========================================================*/
/* #FN#
   Standard constructor */
CPaletteDlg::
CPaletteDlg( CWnd *pParent /*=NULL*/ /* #IN# Pointer to parent window */ )
	: CDialog( (g_ulScreenMode & SM_MODE_FULL ? IDD_PALETTE_SMALL : CPaletteDlg::IDD), pParent)
{
	//{{AFX_DATA_INIT(CPaletteDlg)
		// NOTE: the ClassWizard will add member initialization here
	//}}AFX_DATA_INIT
} /* #OF# CPaletteDlg::CPaletteDlg */

/*========================================================
Method   : CPaletteDlg::DoDataExchange
=========================================================*/
/* #FN#
   Dynamic Data Exchange (not using) */
void
/* #AS#
   Nothing */
CPaletteDlg::
DoDataExchange( CDataExchange *pDX /* #IN# Pointer to CDataExchange object */ )
{
	CDialog::DoDataExchange( pDX );
	//{{AFX_DATA_MAP(CPaletteDlg)
		// NOTE: the ClassWizard will add DDX and DDV calls here
	//}}AFX_DATA_MAP
} /* #OF# CPaletteDlg::DoDataExchange */


/////////////////////////////////////////////////////////////////////////////
// CPaletteDlg implementation

/*========================================================
Method   : CPaletteDlg::SetDlgState
=========================================================*/
/* #FN#
   Sets up the state of the dialog controls */
void
/* #AS#
   Nothing */
CPaletteDlg::
SetDlgState()
{
	CButton *pButton = (CButton *)GetDlgItem( IDC_PALETTE_LOAD );
	ASSERT(pButton);
	pButton->SetCheck( (m_ulMiscStates & MS_USE_EXT_PALETTE) != 0 );

	SetDlgItemText( IDC_PALETTE_EDIT, m_szPaletteFile );

	if( !m_bSmallMode )
	{
		CWnd *pStatic = NULL;

		if( !(m_ulMiscStates & MS_USE_EXT_PALETTE) &&
			!read_palette( m_szPaletteFile ) )
		{
			DisplayMessage( IDS_ERROR_ACT_READ, 0, MB_ICONEXCLAMATION | MB_OK );
		}
		pStatic = GetDlgItem( IDC_PALETTE_BAR );
		ASSERT(pStatic);
		pStatic->GetWindowRect( m_rcPalBar );
		ScreenToClient( m_rcPalBar );
		m_rcPalBar.DeflateRect( CX_BORDER, CY_BORDER );
	}
} /* #OF# CPaletteDlg::SetDlgState */

/*========================================================
Method   : CPaletteDlg::PaintPalette
=========================================================*/
/* #FN#
   Fills the palette bar */
BOOL
/* #AS#
   TRUE if succeded, otherwise FALSE */
CPaletteDlg::
PaintPalette( CDC *pDC )
{
	CPalette     palPalette;
	CPalette    *pOldPalette;
	CBrush      *pBrush;
	CBrush      *pOldBrush;
	int	         nRGB;
	HANDLE       hLogPal;    /* Handle to a logical palette */
	LPLOGPALETTE lpPal;      /* Pointer to a logical palette */
	BOOL         bResult;

	/* Allocate memory block for logical palette */
	hLogPal = ::GlobalAlloc( GHND/*LPTR*/, sizeof(LOGPALETTE) + sizeof(PALETTEENTRY) * PAL_ENTRIES_NO );
	/* If not enough memory, clean up and return NULL */
	if( !hLogPal )
		return FALSE;

	lpPal = (LPLOGPALETTE)::GlobalLock( (HGLOBAL)hLogPal );
	/* Init palette header info */
	lpPal->palVersion    = 0x300;
	lpPal->palNumEntries = (WORD)PAL_ENTRIES_NO;

	/* Copy entries */
	for( int i = 0; i < PAL_ENTRIES_NO; i++ )
	{
		nRGB = colortable[ i ];
		
		lpPal->palPalEntry[ i ].peRed   = (nRGB & 0x00ff0000) >> 16;
		lpPal->palPalEntry[ i ].peGreen = (nRGB & 0x0000ff00) >> 8;
		lpPal->palPalEntry[ i ].peBlue  = nRGB & 0x000000ff;
		lpPal->palPalEntry[ i ].peFlags = 0;
	}
	/* Create palette */
	bResult = (BOOL)(palPalette.CreatePalette( lpPal ));

	::GlobalUnlock( (HGLOBAL) hLogPal );
	::GlobalFree( (HGLOBAL) hLogPal );

	if( bResult )
	{
		int nWidth   = (m_rcPalBar.Width() - BAR_ENTRIES_NO) / BAR_ENTRIES_NO;
		int nHeight  = (m_rcPalBar.Height() - BAR_LINES_NO) / BAR_LINES_NO;

		int nOffsetX = (m_rcPalBar.Width() - (nWidth + 1) * BAR_ENTRIES_NO) / 2 + 1;
		int nOffsetY = (m_rcPalBar.Height() - (nHeight + 1) * BAR_LINES_NO) / 2 + 1;

		/* Select and realize palette */
		pOldPalette = pDC->SelectPalette( &palPalette, TRUE );
		pDC->RealizePalette();

		for( int i = 0; i < BAR_LINES_NO; i++ ) /* Eight lines x 32 color picks */
		{
			/* Draw each stripe */
			for( int j = 0; j < BAR_ENTRIES_NO; j++ )
			{
				pBrush = new CBrush;
				/* A palette-relative pBrush is used for drawing */
				if( TRUE == pBrush->CreateSolidBrush( PALETTEINDEX( j + (i * BAR_ENTRIES_NO) ) ) )
				{
					/* Setup pBrush */
					pOldBrush = pDC->SelectObject( pBrush );
					pBrush->UnrealizeObject();
					/* Draw stripe */
					pDC->PatBlt( m_rcPalBar.left + nOffsetX + j * nWidth + j,
								 m_rcPalBar.top + nOffsetY + i * nHeight + i,
								 nWidth, nHeight, PATCOPY );
					/* Restore old brush */
					pDC->SelectObject( pOldBrush );
				}
				delete pBrush;
			}
		}
		/* Back to old palette */
		pDC->SelectPalette( pOldPalette, TRUE );
		pDC->RealizePalette();
	}
	return bResult;
} /* #OF# CPaletteDlg::PaintPalette */


/////////////////////////////////////////////////////////////////////////////
// CPaletteDlg message handlers

/*========================================================
Method   : CPaletteDlg::OnInitDialog
=========================================================*/
/* #FN#
   Performs special processing when the dialog box is initialized */
BOOL
/* #AS#
   TRUE unless you set the focus to a control */
CPaletteDlg::
OnInitDialog()
{
	CDialog::OnInitDialog();
	
	m_bSmallMode   = g_ulScreenMode & SM_MODE_FULL;
	m_ulMiscStates = g_ulMiscStates;

	strncpy( m_szPaletteFile, g_szPaletteFile, MAX_PATH );

	/* Unfortunately, the context help is available only in windowed modes */
	if( g_ulScreenMode & SM_MODE_WIND )
		SetWindowLong( GetSafeHwnd(), GWL_EXSTYLE, WS_EX_CONTEXTHELP | GetWindowLong( GetSafeHwnd(), GWL_EXSTYLE ) );

	SetDlgState();

	return TRUE;  // return TRUE unless you set the focus to a control
	              // EXCEPTION: OCX Property Pages should return FALSE
} /* #OF# CPaletteDlg::OnInitDialog */

/*========================================================
Method   : CPaletteDlg::OnLoad
=========================================================*/
/* #FN#
   The "Use this palette" check-box notification handler */
void
/* #AS#
   Nothing */
CPaletteDlg::
OnLoad()
{
	CButton *pButton = (CButton *)GetDlgItem( IDC_PALETTE_LOAD );
	ASSERT(pButton);

	if( pButton->GetCheck() )
		m_ulMiscStates |= MS_USE_EXT_PALETTE;
	else
		m_ulMiscStates &= ~MS_USE_EXT_PALETTE;
} /* #OF# CPaletteDlg::OnLoad */

/*========================================================
Method   : CPaletteDlg::OnBrowse
=========================================================*/
/* #FN#
   Allows to select palette file using FileDialog window */
void
/* #AS#
   Nothing */
CPaletteDlg::
OnBrowse()
{
	char szPaletteOld[ MAX_PATH ];

	strcpy( szPaletteOld, m_szPaletteFile );

	if( PickFileName( TRUE, m_szPaletteFile, "Load Atari palette file",
					  PF_ACT_FILTER, "act", PF_LOAD_FLAGS ) &&
		*m_szPaletteFile != '\0' &&
		_stricmp( szPaletteOld, m_szPaletteFile ) != 0 )
	{
		SetDlgItemText( IDC_PALETTE_EDIT, m_szPaletteFile );
		ReadPaletteFile( m_szPaletteFile, FALSE ); /* Read palette from file */
	}
} /* #OF# CPaletteDlg::OnBrowse */

/*========================================================
Method   : CPaletteDlg::ReadPaletteFile
=========================================================*/
/* #FN#
   Reads an external palette from file */
BOOL
/* #AS#
   Nothing */
CPaletteDlg::
ReadPaletteFile( LPSTR pszPaletteFile,
				 BOOL  bCheckIfExists /*=TRUE*/ )
{
	BOOL bResult = FALSE;

	if( *pszPaletteFile != '\0' )
	{
		CFileStatus fsStatus;
		CFile       fDisk;

		if( !bCheckIfExists || fDisk.GetStatus( pszPaletteFile, fsStatus ) )
		{
			if( !read_palette( pszPaletteFile ) )
			{
				DisplayMessage( IDS_ERROR_ACT_READ, 0, MB_ICONEXCLAMATION | MB_OK );
			}
			else
				bResult = TRUE;

			/* Palette bar is drawn in windowed modes only */
			if( !m_bSmallMode )
				/* This will make the changes visible */
				Invalidate();
		}
	}
	return bResult;
} /* #OF# CPaletteDlg::ReadPaletteFile */

/*========================================================
Method   : CPaletteDlg::OnKillfocusEdit
=========================================================*/
/* #FN#
   The framework calls this function before edit losing the input focus */
void
/* #AS#
   Nothing */
CPaletteDlg::
OnKillfocusEdit()
{
	char szPaletteOld[ MAX_PATH ];

	strcpy( szPaletteOld, m_szPaletteFile );
	GetDlgItemText( IDC_PALETTE_EDIT, m_szPaletteFile, MAX_PATH );

	if( _stricmp( szPaletteOld, m_szPaletteFile ) != 0 )
		ReadPaletteFile( m_szPaletteFile ); /* Read palette from file */
} /* #OF# CPaletteDlg::OnKillfocusEdit */

/*========================================================
Method   : CPaletteDlg::OnPaint
=========================================================*/
/* #FN#
   The framework calls this member function when Windows or an application
   makes a request to repaint a portion of an applications window */
void
/* #AS#
   Nothing */
CPaletteDlg::
OnPaint()
{
	CPaintDC dc( this ); /* Device context for painting */

	if( !m_bSmallMode )
		PaintPalette( &dc );
} /* #OF# CPaletteDlg::OnPaint */

/*========================================================
Method   : CPaletteDlg::OnOK
=========================================================*/
/* #FN#
   Called when the user clicks the OK button */
void
/* #AS#
   Nothing */
CPaletteDlg::
OnOK()
{
	if( m_ulMiscStates != g_ulMiscStates )
	{
		g_ulMiscStates = m_ulMiscStates;
		WriteRegDWORD( NULL, REG_MISC_STATES, g_ulMiscStates );
	}	
	/* Get the external palette name from edit (OnKillfocusEdit()
	   didn't work when user had pressed 'Enter' or 'Alt-O' */
	GetDlgItemText( IDC_PALETTE_EDIT, m_szPaletteFile, MAX_PATH );
	/* Windows NT doesn't like empty strings in Registry */
	if( *m_szPaletteFile == '\0' )
		strcpy( m_szPaletteFile, DEFAULT_ACT );

	if( _stricmp( g_szPaletteFile, m_szPaletteFile ) != 0 )
	{
		strncpy( g_szPaletteFile, m_szPaletteFile, MAX_PATH );
		WriteRegString( NULL, REG_EXT_PALETTE, g_szPaletteFile );
	}
	/* Always reload the selected palette */
	if( ReadPaletteFile( g_szPaletteFile, FALSE ) ) /* Read palette from file */
	{
		/* Apply the changes to the main window */
		UseAtariPalette();
		if( m_bSmallMode )
		{
			/* Make the changes visible for DirectDraw full-screen modes */
			SetupScreenPalette();
		}
	}
	CDialog::OnOK();
} /* #OF# CPaletteDlg::OnOK */

/*========================================================
Method   : CPaletteDlg::OnCancel
=========================================================*/
/* #FN#
   Called when the user clicks the CANCEL button */
void
/* #AS#
   Nothing */
CPaletteDlg::
OnCancel()
{
	if( _stricmp( g_szPaletteFile, m_szPaletteFile ) != 0 )
	{
		int	nRGB;

		strncpy( m_szPaletteFile, g_szPaletteFile, MAX_PATH );
		SetDlgItemText( IDC_PALETTE_EDIT, g_szPaletteFile );
		/* Restore old palette */
		for( int i = 0; i < PAL_ENTRIES_NO; i++ )
		{
			nRGB  = (g_pe[ i ].peRed   & 0xff) << 16;
			nRGB |= (g_pe[ i ].peGreen & 0xff) <<  8;
			nRGB |= (g_pe[ i ].peBlue  & 0xff);

			colortable[ i ] = nRGB;
		}
	}
	CDialog::OnCancel();
} /* #OF# CPaletteDlg::OnCancel */

BOOL
CPaletteDlg::
OnHelpInfo( HELPINFO *pHelpInfo )
{
	if( g_ulScreenMode & SM_MODE_WIND )
		HelpInfo( pHelpInfo,
				  GetSafeHwnd(),
				  IDD_PALETTE,
				  IDC_PALETTE_FIRST, IDC_PALETTE_LAST );

	return TRUE; //CDialog::OnHelpInfo(pHelpInfo);
}
