/****************************************************************************
File    : MainFrame.cpp
/*
@(#) #SY# Atari800Win
@(#) #IS# CMainFrame implementation file
@(#) #BY# Richard Lawrence, Tomasz Szymankowski
@(#) #LM# 13.05.2000
*/

/*
Copyright (c) 1998 Richard Lawrence

This program is free software; you can redistribute it and/or modify it under the terms 
of the GNU General Public License as published by the Free Software Foundation; either 
version 2 of the License, or (at your option) any later version. This program is 
distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even
the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details. You should have received a copy of the GNU
General Public License along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include "StdAfx.h"
#include "Atari800Win.h"
#include "Helpers.h"
#include "DriveDlg.h"
#include "SettingsDlg.h"
#include "FileAssociationsDlg.h"
#include "RomImagesDlg.h"
#include "GraphicsDlg.h"
#include "DriveSmallDlg.h"
#include "GraphicsSmallDlg.h"
#include "SoundDlg.h"
#include "FileSmallDlg.h"
#include "HarddiskDlg.h"
#include "JoystickDlg.h"
#include "ErrorLogDlg.h"
#include "KeyboardDlg.h"
#include "PaletteDlg.h"
#include "WarningDlg.h"
#include "MainFrame.h"

//#include "core.h"			// AtariWin core

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


#define ID_ATARI_MACHINETYPE_BASE		ID_ATARI_MACHINETYPE_800OSA
#define ID_ATARI_VIDEOSYSTEM_BASE		ID_ATARI_VIDEOSYSTEM_PAL
#define ID_GRAPHICS_ARTIFACTING_BASE	ID_GRAPHICS_ARTIFACTING_NONE

/////////////////////////////////////////////////////////////////////////////
// CMainFrame window

IMPLEMENT_DYNCREATE(CMainFrame, CFrameWnd)

BEGIN_MESSAGE_MAP(CMainFrame, CFrameWnd)
	//{{AFX_MSG_MAP(CMainFrame)
	ON_WM_CREATE()
	ON_COMMAND(ID_FILE_LOADEXECUTABLE, OnFileLoadExecutable)
	ON_COMMAND(ID_FILE_ATTACHDISKIMAGE_DRIVE1, OnFileAttachDiskImageDrive1)
	ON_COMMAND(ID_FILE_ATTACHDISKIMAGE_DRIVE2, OnFileAttachDiskImageDrive2)
	ON_COMMAND(ID_FILE_ATTACHDISKIMAGE_DRIVE3, OnFileAttachDiskImageDrive3)
	ON_COMMAND(ID_FILE_ATTACHDISKIMAGE_DRIVE4, OnFileAttachDiskImageDrive4)
	ON_COMMAND(ID_FILE_ATTACHDISKIMAGE_DRIVE5, OnFileAttachDiskImageDrive5)
	ON_COMMAND(ID_FILE_ATTACHDISKIMAGE_DRIVE6, OnFileAttachDiskImageDrive6)
	ON_COMMAND(ID_FILE_ATTACHDISKIMAGE_DRIVE7, OnFileAttachDiskImageDrive7)
	ON_COMMAND(ID_FILE_ATTACHDISKIMAGE_DRIVE8, OnFileAttachDiskImageDrive8)
	ON_COMMAND(ID_FILE_ATTACHCARTRIDGEIMAGE, OnFileAttachCartridgeImage)
	ON_COMMAND(ID_FILE_TURNDISK, OnFileTurnDisk)
	ON_COMMAND(ID_FILE_DISKDRIVES, OnFileDiskDrives)
	ON_COMMAND(ID_FILE_ROMIMAGES, OnFileRomImages)
	ON_COMMAND(ID_FILE_HARDDISKS, OnFileHardDisks)
	ON_COMMAND(ID_FILE_READSTATE, OnFileReadState)
	ON_COMMAND(ID_FILE_SAVESTATE_NORMAL, OnFileSaveStateNormal)
	ON_COMMAND(ID_FILE_SAVESTATE_VERBOSE, OnFileSaveStateVerbose)
	ON_COMMAND(ID_ATARI_MACHINETYPE, OnAtariMachineType)
	ON_COMMAND(ID_ATARI_MACHINETYPE_800OSA, OnAtariMachineType800osa)
	ON_COMMAND(ID_ATARI_MACHINETYPE_800OSB, OnAtariMachineType800osb)
	ON_COMMAND(ID_ATARI_MACHINETYPE_800XL, OnAtariMachineType800xl)
	ON_COMMAND(ID_ATARI_MACHINETYPE_130XE, OnAtariMachineType130xe)
	ON_COMMAND(ID_ATARI_MACHINETYPE_320XECOMPY, OnAtariMachineType320xeCompy)
	ON_COMMAND(ID_ATARI_MACHINETYPE_320XERAMBO, OnAtariMachineType320xeRambo)
	ON_COMMAND(ID_ATARI_MACHINETYPE_5200, OnAtariMachineType5200)
	ON_COMMAND(ID_ATARI_VIDEOSYSTEM, OnAtariVideoSystem)
	ON_COMMAND(ID_ATARI_VIDEOSYSTEM_NTSC, OnAtariVideoSystemNtsc)
	ON_COMMAND(ID_ATARI_VIDEOSYSTEM_PAL, OnAtariVideoSystemPal)
	ON_COMMAND(ID_ATARI_SETTINGS, OnAtariSettings)
	ON_COMMAND(ID_ATARI_SIOPATCH, OnAtariSioPatch)
	ON_COMMAND(ID_ATARI_FULLSPEED, OnAtariFullSpeed)
	ON_COMMAND(ID_ATARI_PAUSE, OnAtariPause)
	ON_COMMAND(ID_ATARI_KEYBOARD, OnAtariKeyboard)
	ON_COMMAND(ID_ATARI_JOYSTICKS, OnAtariJoysticks)
	ON_COMMAND(ID_ATARI_WARMSTART, OnAtariWarmstart)
	ON_COMMAND(ID_ATARI_COLDSTART, OnAtariColdstart)
	ON_COMMAND(ID_GRAPHICS_GRAPHICSOPTIONS, OnGraphicsOptions)
	ON_COMMAND(ID_GRAPHICS_GDIFORWINDOWS, OnGraphicsGdiForWindows)
	ON_COMMAND(ID_GRAPHICS_SCANLINES, OnGraphicsScanlines)
	ON_COMMAND(ID_GRAPHICS_ARTIFACTING, OnGraphicsArtifacting)
	ON_COMMAND(ID_GRAPHICS_ARTIFACTING_NONE, OnGraphicsArtifactingNone)
	ON_COMMAND(ID_GRAPHICS_ARTIFACTING_BLUEBROWN1, OnGraphicsArtifactingBluebrown1)
	ON_COMMAND(ID_GRAPHICS_ARTIFACTING_BLUEBROWN2, OnGraphicsArtifactingBluebrown2)
	ON_COMMAND(ID_GRAPHICS_ARTIFACTING_GTIA, OnGraphicsArtifactingGtia)
	ON_COMMAND(ID_GRAPHICS_ARTIFACTING_CTIA, OnGraphicsArtifactingCtia)
	ON_COMMAND(ID_GRAPHICS_PALETTE, OnGraphicsPalette)
	ON_COMMAND(ID_GRAPHICS_SAVESCREENSHOT_NORMAL, OnGraphicsSaveScreenshotNormal)
	ON_COMMAND(ID_GRAPHICS_SAVESCREENSHOT_INTERLACED, OnGraphicsSaveScreenshotInterlaced)
	ON_COMMAND(ID_GRAPHICS_PERFORMANCETEST, OnGraphicsPerformanceTest)
	ON_COMMAND(ID_SOUND_SOUNDOPTIONS, OnSoundOptions)
	ON_COMMAND(ID_SOUND_DIGITIZEDEFFECTS, OnSoundDigitizedEffects)
	ON_COMMAND(ID_SOUND_STEREO, OnSoundStereo)
	ON_COMMAND(ID_SOUND_MUTE, OnSoundMute)
	ON_COMMAND(ID_SOUND_SAVESOUND, OnSoundSaveSound)
	ON_COMMAND(ID_SOUND_PERFORMANCETEST, OnSoundPerformanceTest)
	ON_COMMAND(ID_MISC_FILEASSOCIATIONS, OnMiscFileAssociations)
	ON_COMMAND(ID_MISC_CONVERT_DCMTOATR, OnMiscConvertDcmToAtr)
	ON_COMMAND(ID_MISC_CLEARALLSETTINGS, OnMiscClearAllSettings)
	ON_COMMAND(ID_MISC_RESTARTEMULATION, OnMiscRestartEmulation)
	ON_COMMAND(ID_MISC_VIEWLOGFILE, OnMiscViewLogFile)
	ON_COMMAND(ID_MISC_MONITOR, OnMiscMonitor)
	ON_WM_INITMENU()
	ON_WM_DROPFILES()
	ON_WM_MOVE()
	ON_WM_TIMER()
	ON_WM_CLOSE()
	ON_UPDATE_COMMAND_UI(ID_FILE_TURNDISK, OnUpdateFileTurnDisk)
	ON_UPDATE_COMMAND_UI(ID_ATARI_MACHINETYPE_800OSA, OnUpdateAtariMachineType)
	ON_UPDATE_COMMAND_UI(ID_ATARI_MACHINETYPE_800OSB, OnUpdateAtariMachineType)
	ON_UPDATE_COMMAND_UI(ID_ATARI_MACHINETYPE_800XL, OnUpdateAtariMachineType)
	ON_UPDATE_COMMAND_UI(ID_ATARI_MACHINETYPE_130XE, OnUpdateAtariMachineType)
	ON_UPDATE_COMMAND_UI(ID_ATARI_MACHINETYPE_320XECOMPY, OnUpdateAtariMachineType)
	ON_UPDATE_COMMAND_UI(ID_ATARI_MACHINETYPE_320XERAMBO, OnUpdateAtariMachineType)
	ON_UPDATE_COMMAND_UI(ID_ATARI_MACHINETYPE_5200, OnUpdateAtariMachineType)
	ON_UPDATE_COMMAND_UI(ID_ATARI_VIDEOSYSTEM_NTSC, OnUpdateAtariVideoSystem)
	ON_UPDATE_COMMAND_UI(ID_ATARI_VIDEOSYSTEM_PAL, OnUpdateAtariVideoSystem)
	ON_UPDATE_COMMAND_UI(ID_ATARI_SIOPATCH, OnUpdateAtariSioPatch)
	ON_UPDATE_COMMAND_UI(ID_ATARI_FULLSPEED, OnUpdateAtariFullSpeed)
	ON_UPDATE_COMMAND_UI(ID_ATARI_PAUSE, OnUpdateAtariPause)
	ON_UPDATE_COMMAND_UI(ID_GRAPHICS_GDIFORWINDOWS, OnUpdateGraphicsGdiForWindows)
	ON_UPDATE_COMMAND_UI(ID_GRAPHICS_SCANLINES, OnUpdateGraphicsScanlines)
	ON_UPDATE_COMMAND_UI(ID_GRAPHICS_ARTIFACTING_NONE, OnUpdateGraphicsArtifacting)
	ON_UPDATE_COMMAND_UI(ID_GRAPHICS_ARTIFACTING_BLUEBROWN1, OnUpdateGraphicsArtifacting)
	ON_UPDATE_COMMAND_UI(ID_GRAPHICS_ARTIFACTING_BLUEBROWN2, OnUpdateGraphicsArtifacting)
	ON_UPDATE_COMMAND_UI(ID_GRAPHICS_ARTIFACTING_GTIA, OnUpdateGraphicsArtifacting)
	ON_UPDATE_COMMAND_UI(ID_GRAPHICS_ARTIFACTING_CTIA, OnUpdateGraphicsArtifacting)
	ON_UPDATE_COMMAND_UI(ID_SOUND_DIGITIZEDEFFECTS, OnUpdateSoundDigitizedEffects)
	ON_UPDATE_COMMAND_UI(ID_SOUND_STEREO, OnUpdateSoundStereo)
	ON_UPDATE_COMMAND_UI(ID_SOUND_MUTE, OnUpdateSoundMute)
	ON_UPDATE_COMMAND_UI(ID_SOUND_SAVESOUND, OnUpdateSoundSaveSound)
	//}}AFX_MSG_MAP
	// Global help commands
	ON_COMMAND(ID_HELP_FINDER, CFrameWnd::OnHelpFinder)
	ON_COMMAND(ID_HELP, CFrameWnd::OnHelp)
	ON_COMMAND(ID_CONTEXT_HELP, CFrameWnd::OnContextHelp)
	ON_COMMAND(ID_DEFAULT_HELP, CFrameWnd::OnHelpFinder)
END_MESSAGE_MAP()

static UINT indicators[] =
{
	ID_SEPARATOR // status line indicator
};

/*========================================================
Method   : CMainFrame::CMainFrame
=========================================================*/
/* #FN#
   Standard constructor */
CMainFrame::
CMainFrame()
{
	char szMessage[ MAX_PATH + 64 ] = { '\0' };

	BuildCRCTable();

	HandleRegistry();
	/* These variables were not written in the Registry */
	m_strStateName    = "atarisav.a8s";
	m_strSoundName    = "atarisnd.wav";
	m_strSnapshotName = "atari000.pcx";

	if( g_ulMiscStates & MS_LAST_BOOT_FAILED )
	{
		LoadString( NULL, IDS_LOCKUP_WARNING, g_tszErrorString, LOADSTRING_STRING_SIZE );
		if( IDYES == MessageBox( g_tszErrorString, "Atari800Win", MB_ICONQUESTION | MB_YESNO ) )
			InitialiseRegistry( NULL, TRUE );
	}
	/* Load keys template and external palette */
	if( g_ulMiscStates & MS_USE_KEYTEMPLATE && 
		!ReadKeyTemplate( g_szTemplateFile ) )
	{
		sprintf( szMessage, "Could not load keyboard template file %s, turning off keyboard template.", g_szTemplateFile );
		g_ulMiscStates &= ~MS_USE_KEYTEMPLATE;
	}
	if( g_ulMiscStates & MS_USE_PALETTE && 
		!read_palette( g_szPaletteFile ) )
	{
		sprintf( szMessage, "Could not load external palette file %s, turning off external palette.", g_szPaletteFile );
		g_ulMiscStates &= ~MS_USE_PALETTE;
	}
	if( *szMessage != '\0' )
	{
		MessageBox( szMessage, "Atari800Win", MB_ICONEXCLAMATION | MB_OK );
		WriteRegDWORD( NULL, REG_MISC_STATES, g_ulMiscStates );
	}
	/* Call the main of Atari emulator */
	main( g_argc, g_argv );
} /* #OF# CMainFrame::CMainFrame */

/*========================================================
Method   : CMainFrame::~CMainFrame
=========================================================*/
/* #FN#
   Destructor */
CMainFrame::
~CMainFrame()
{
	/* In case desktop isn't getting refreshed when exiting directly from
	   DirectDraw full screen */
	::InvalidateRect( NULL, NULL, FALSE );
} /* #OF# CMainFrame::~CMainFrame */


/////////////////////////////////////////////////////////////////////////////
// CMainFrame implementation

/*========================================================
Method   : CMainFrame::PreCreateWindow
=========================================================*/
/* #FN#
   Called by the framework before the creation of the Windows window */
BOOL
/* #AS#
   Nonzero if the window creation should continue, 0 to indicate creation
   failure */
CMainFrame::
PreCreateWindow( CREATESTRUCT& cs )
{
	cs.x = g_nStartX;
	cs.y = g_nStartY;

//	cs.style |= FWS_ADDTOTITLE;
//	cs.style |= CS_CLASSDC;
//	cs.style |= CS_PARENTDC;
//	cs.style |= CS_BYTEALIGNCLIENT;
//	cs.style |= CS_BYTEALIGNWINDOW;
	cs.style |= CS_OWNDC;
	cs.style &= ~(WS_THICKFRAME | WS_MAXIMIZEBOX);

	return CFrameWnd::PreCreateWindow( cs );
} /* #OF# CMainFrame::PreCreateWindow */

/*========================================================
Method   : CMainFrame::ActivateFrame
=========================================================*/
/* #FN#
   Activates and restores the frame window */
void
/* #AS#
   Nothing */
CMainFrame::
ActivateFrame( int nCmdShow /* #IN# Parameter to pass to CWnd::ShowWindow */ )
{
	/* Get the real size of status bar */
	CRect rect;
	pStatusBar->GetWindowRect( rect );
	g_nStatusSize = rect.Height();

	CFrameWnd::ActivateFrame( nCmdShow );
	// Causes some problems with window mode switching
	//	(g_ulScreenMode & SM_MODE_FULL) ?
	//	SW_SHOWMAXIMIZED : nCmdShow );
} /* #OF# CMainFrame::ActivateFrame */

/*========================================================
Method   : CMainFrame::DriveInsert
=========================================================*/
/* #FN#
   Inserts disk image to pointed virtual drive */
void
/* #AS#
   Nothing */
CMainFrame::
DriveInsert( int nDriveNum )
{
	CString strDiskName = sio_filename[ nDriveNum - 1 ];
	char    szPrompt[ 64 ];

	sprintf( szPrompt, "Select disk to insert in drive %d", nDriveNum );

	if( PickFileName( TRUE, strDiskName, szPrompt,
					  PF_ATR_FILTER, NULL, PF_LOAD_FLAGS ) &&
		!strDiskName.IsEmpty() )
	{
		SIO_Dismount( nDriveNum );
		SIO_Mount( nDriveNum, (LPSTR)(LPCSTR)strDiskName );
		WriteRegDrives( NULL );
	}
	ClearScreen( FALSE );
} /* #OF# CMainFrame::DriveInsert */

/*========================================================
Method   : CMainFrame::SaveState
=========================================================*/
/* #FN#
   Saves state of the Atari to a file */
void
/* #AS#
   Nothing */
CMainFrame::
SaveState( int nSaveRom )
{
	if( zlib_capable() == -1 )
	{
		MessageBox( "Cannot save/restore state files without a working ZLIB.DLL.", "Atari800Win", MB_ICONEXCLAMATION | MB_OK );
		return;
	}
	if( nSaveRom &&
		(IDNO == MessageBox( "It is not normally necessary to do this. It will save the Atari ROMS and BASIC in your state file, making it larger. The only reason this would be necessary is if you are running with patched ROMS/BASIC. Do you want to continue anyway?", "Atari800Win", MB_ICONQUESTION | MB_YESNO )) )
		return;

	if( PickFileName( FALSE, m_strStateName, "Save Atari saved-state file",
					  PF_A8S_FILTER, "a8s", PF_SAVE_FLAGS, FALSE ) &&
		!m_strStateName.IsEmpty() )
	{
		if( !SaveAtariState( (LPSTR)(LPCSTR)m_strStateName, "wb", nSaveRom ) )
			MessageBox( "The Atari save state operation failed. You can check the 'File/View Log File' option to see a description of the problem.", "Atari800Win", MB_ICONEXCLAMATION | MB_OK );
	}
	DescribeAtariSystem();

	ClearScreen( FALSE );
} /* #OF# CMainFrame::SaveState */

/*========================================================
Method   : CMainFrame::SaveSnapshot
=========================================================*/
/* #FN#
   Saves snapshot of the Atari screen to .PCX file */
void
/* #AS#
   Nothing */
CMainFrame::
SaveSnapshot( int nInterlace )
{
	static BOOL bChanged = FALSE;
	CString strSnapshotName, strPrompt;

	if( !bChanged )
		m_strSnapshotName = Find_PCX_name();

	strSnapshotName = m_strSnapshotName;
	strPrompt.Format( "Save Atari %s screenshot file", nInterlace ? "interlaced" : "normal" );

	if( PickFileName( FALSE, m_strSnapshotName, strPrompt,
					  PF_PCX_FILTER, "pcx", PF_SAVE_FLAGS, FALSE ) &&
		!m_strSnapshotName.IsEmpty() )
	{
		if( !Save_PCX_file( nInterlace, (LPSTR)(LPCSTR)m_strSnapshotName ) )
			MessageBox( "The Atari save screenshot operation failed.", "Atari800Win", MB_ICONEXCLAMATION | MB_OK );

		if( !bChanged &&
			strSnapshotName.CompareNoCase( m_strSnapshotName.Right( m_strSnapshotName.GetLength() - m_strSnapshotName.ReverseFind( '\\' ) - 1 ) ) != 0 )
			bChanged = TRUE;
	}
	ClearScreen( FALSE );
} /* #OF# CMainFrame::SaveSnapshot */

/*========================================================
Method   : CMainFrame::PreTranslateMessage
=========================================================*/
/* #FN#
   Translates window messages before they are dispatched to the
   TranslateMessage and DispatchMessage Windows functions */
BOOL
/* #AS#
   Nonzero if the message was translated and should not be dispatched;
   0 if the message was not translated and should be dispatched */
CMainFrame::
PreTranslateMessage( MSG *pMsg )
{
	WPARAM wp = pMsg->wParam;

	switch( pMsg->message )
	{
		/* This needs to be handled here instead of a OnXXX message because
		   we want to intercept control keys before they are parsed as
		   accelerators */
		case WM_KEYDOWN:
		{
			if( wp == (WPARAM)g_nCurrentVK )
				return TRUE;

			if( g_bKBJoystick || g_iToggleKeys[ wp ] )
			{
				if( ToggleKeyDown( wp, pMsg->lParam ) )
					return TRUE;
			}

			if( g_ulMiscStates & MS_USE_KEYTEMPLATE &&
				g_aucKeysCovered[ wp ] )
			{
				g_nNewKey = g_nKBTable[ wp ];
				
				if( g_nNewKey != AKEY_NONE )
				{
					if( SHIFT_KEY )
						g_nNewKey |= AKEY_SHFT;
					if( CTRL_KEY )
						g_nNewKey |= AKEY_CTRL;

					g_nNewVK = wp;
				}
				else
					g_nNewVK = 0;
				return TRUE;
			}

			if( SHIFT_KEY && CTRL_KEY )
			{
				g_nNewKey = g_iShftCtrlKeys[ wp ];
				g_nNewVK  = (g_nNewKey != AKEY_NONE ? wp : 0);
				return TRUE;
			}

			if( CTRL_KEY )
			{
//				if( wp >= VK_F1 && wp <= VK_F8 )
//				{
//					DriveInsert( wp - VK_F1 + 1 );
//				}
//				else
//				{
					g_nNewKey = g_iCtrlKeys[ wp ];
					g_nNewVK  = (g_nNewKey != AKEY_NONE ? wp : 0);
//				}
				return TRUE;
			}

			if( SHIFT_KEY )
			{
				g_nNewKey = g_iShftKeys[ wp ];
				g_nNewVK  = (g_nNewKey != AKEY_NONE ? wp : 0);
				/* Do standard translate for function keys */
				if( wp < VK_F1 || wp > VK_F12 )
					return TRUE;
			}
			
			if( g_iNormKeys[ wp ] != AKEY_NONE )
			{
				g_nNewKey = g_iNormKeys[ wp ];
				g_nNewVK  = (g_nNewKey != AKEY_NONE ? wp : 0);
				return TRUE;
			}
			break;
		}

		case WM_SYSKEYDOWN:
		{
			/* Switch between windowed and full screen modes */
			if( wp == VK_RETURN )
			{
				if( g_ulScreenMode & SM_MODE_FULL )
				{
					ShowMousePointer();

					g_ulScreenMode &= ~SM_MODE_FULL;
					g_ulScreenMode |= SM_MODE_WIND;
				}
				else
				{
					_ASSERT(g_ulScreenMode & SM_MODE_WIND);

					g_ulScreenMode &= ~SM_MODE_WIND;
					g_ulScreenMode |= SM_MODE_FULL;
				}
				WriteRegDWORD( NULL, REG_DDRAW_MODE, g_ulScreenMode );
				InitialiseScreen( FALSE );
				return TRUE;
			}
			break;
		}

		case WM_KEYUP:
		{
			if( g_bKBJoystick || g_iToggleKeys[ wp ] )
			{
				if( ToggleKeyUp( wp, pMsg->lParam ) )
					return TRUE;
			}

			if( wp == (WPARAM)g_nCurrentVK )
			{
				g_nCurrentKey = AKEY_NONE;
				g_nCurrentVK  = 0;
				KEYPRESSED    = FALSE;
			}

			if( wp == (WPARAM)g_nNewVK )
			{
				if( !(g_nNewKey & SPECIAL_HANDLER_MASK) )
				{
					g_nNewKey = AKEY_NONE;
					g_nNewVK  = 0;
				}
			}
			break;
		}

		default:
			break;
	}
	return CFrameWnd::PreTranslateMessage( pMsg );
}


/////////////////////////////////////////////////////////////////////////////
// CMainFrame diagnostics

#ifdef _DEBUG
void CMainFrame::AssertValid() const
{
	CFrameWnd::AssertValid();
}

void CMainFrame::Dump(CDumpContext& dc) const
{
	CFrameWnd::Dump(dc);
}

#endif //_DEBUG


/////////////////////////////////////////////////////////////////////////////
// CMainFrame message handlers

/*========================================================
Method   : CMainFrame::OnCreate
=========================================================*/
/* #FN#
   Called by the framework when an application requests that the Windows
   window be created */
int
/* #AS#
   0 to continue the creation, 1 to destroy the Windows window */
CMainFrame::
OnCreate( LPCREATESTRUCT lpCreateStruct /* #IN# Pointer to a CREATESTRUCT
										        structure that contains
												information about the CWnd
												object being created */ )
{
#ifdef WIN_USE_PERF_COUNTER
	LARGE_INTEGER lnTimerRes;

	QueryPerformanceFrequency( &lnTimerRes ); 
	ASSERT(lnTimerRes.HighPart == 0);
	if( lnTimerRes.LowPart == 0 )
	{
		LoadString( NULL, IDS_NO_HIRES_TIMER, g_tszErrorString, LOADSTRING_STRING_SIZE );
		MessageBox( g_tszErrorString, NULL, MB_ICONSTOP | MB_OK );
		return FALSE;
	}
#endif /*WIN_USE_PERF_COUNTER*/

	if( CFrameWnd::OnCreate( lpCreateStruct ) == -1 )
		return -1;
	
	if( !m_wndStatusBar.Create( this ) ||
		!m_wndStatusBar.SetIndicators( indicators,
		  sizeof(indicators)/sizeof(UINT) ) )
	{
		TRACE0("Failed to create status bar\n");
		return -1; // fail to create
	}
	pStatusBar = (CWnd*)&m_wndStatusBar;

	DragAcceptFiles( TRUE );

	return 0;
} /* #OF# CMainFrame::OnCreate */

/*========================================================
Method   : CMainFrame::OnInitMenu
=========================================================*/
/* #FN#
   The framework calls this member function when a pop-up menu
   is about to become active (or when Alt key is pressed) */
void
/* #AS#
   Nothing */
CMainFrame::
OnInitMenu( CMenu *pMenu )
{
	CFrameWnd::OnInitMenu( pMenu );
	
#ifdef WIN_USE_FLIP_BUFFER
	/* Get surface for GDI if necessary */
	FlipToGDI();
#endif
	ShowMousePointer();
} /* #OF# CMainFrame::OnInitMenu */

/*========================================================
Method   : CMainFrame::OnFileLoadExecutable
=========================================================*/
/* #FN#
   Loads an Atari executable file directly */
void
/* #AS#
   Nothing */
CMainFrame::
OnFileLoadExecutable()
{
//	char	szCurDir[ MAX_PATH ];
	UINT    unFileType  = IAF_EXECUTABLE;
	CString	strFileName = atari_exe_dir;

//	GetCurrentDirectory( MAX_PATH, szCurDir );
//	SetCurrentDirectory( atari_exe_dir );

	if( PickFileName( TRUE, strFileName, "Select Atari executable file",
					  PF_XEX_FILTER, "xex", PF_LOAD_FLAGS ) &&
		!strFileName.IsEmpty() &&
		IsAtariFile( strFileName, &unFileType ) )
	{
		if( !RunAtariExe( strFileName ) )
		{
			LoadString( NULL, IDS_RUNEXE_ERROR, g_tszErrorString, LOADSTRING_STRING_SIZE );
			MessageBox( g_tszErrorString, "Atari800Win", MB_ICONEXCLAMATION | MB_OK );
		}
	}
//	SetCurrentDirectory( szCurDir );
	DescribeAtariSystem();

	ClearScreen( FALSE );
} /* #OF# CMainFrame::OnFileLoadExecutable */

void
CMainFrame::
OnFileAttachDiskImageDrive1() 
{
	DriveInsert( 1 );
}

void
CMainFrame::
OnFileAttachDiskImageDrive2() 
{
	DriveInsert( 2 );
}

void
CMainFrame::
OnFileAttachDiskImageDrive3() 
{
	DriveInsert( 3 );
}

void
CMainFrame::
OnFileAttachDiskImageDrive4() 
{
	DriveInsert( 4 );
}

void
CMainFrame::
OnFileAttachDiskImageDrive5() 
{
	DriveInsert( 5 );
}

void
CMainFrame::
OnFileAttachDiskImageDrive6() 
{
	DriveInsert( 6 );
}

void
CMainFrame::
OnFileAttachDiskImageDrive7() 
{
	DriveInsert( 7 );
}

void
CMainFrame::
OnFileAttachDiskImageDrive8() 
{
	DriveInsert( 8 );
}

void
CMainFrame::
OnFileTurnDisk()
{
	char szDiskPath[ MAX_PATH ];

//	ReadRegBinary( NULL, g_apszDriveRegID[ 0 ], szDiskPath, MAX_PATH, FALSE );
	strncpy( szDiskPath, sio_filename[ 0 ], MAX_PATH );
	if( strcmp( szDiskPath, "Empty" ) != 0 &&
		strcmp( szDiskPath, "Off" ) != 0 )
	{
		char  szDrive [ _MAX_DRIVE ];
		char  szDir   [ _MAX_DIR ];
		char  szFile  [ _MAX_FNAME ];
		char  szExt   [ _MAX_EXT ];
		char  szCurDir[ MAX_PATH ];
		char  szNewDir[ MAX_PATH ];

		const char acSides[ 3 ][ 2 ] = { {'1', '9'}, {'A', 'Z'}, {'a', 'z'} };
		char  cDiskSide, cNewDiskSide;
		BOOL  bTurnDisk = FALSE;
		int   i; /* Loops index */

		CFile cfFile;

		GetCurrentDirectory( MAX_PATH, szCurDir );
		strncpy( szNewDir, szDiskPath, MAX_PATH );

		for( i = strlen( szNewDir ); i > 0 && szNewDir[ i ] != '\\'; i-- );
		if( i > 0 )
			szNewDir[ i ] = '\0';

		SetCurrentDirectory( szNewDir );

		/* Split disk full path */
		_tsplitpath( szDiskPath, szDrive, szDir, szFile, szExt );

		cNewDiskSide = cDiskSide = szFile[ strlen( szFile ) - 1 ];

		for( i = 0; i < 3 && !bTurnDisk; i++ )
		{
			if( cNewDiskSide >= acSides[ i ][ 0 ] && cNewDiskSide <= acSides[ i ][ 1 ] )
			{
				do
				{
					if( ++cNewDiskSide > acSides[ i ][ 1 ] )
						cNewDiskSide = acSides[ i ][ 0 ];

					szFile[ strlen( szFile ) - 1 ] = cNewDiskSide;
					_makepath( szDiskPath, szDrive, szDir, szFile, szExt );
					if( cfFile.Open( szDiskPath, CFile::modeRead | CFile::typeBinary, NULL ) )
					{
						bTurnDisk = TRUE;
						cfFile.Close();
						break;
					}
				}
				while( cNewDiskSide != cDiskSide );
			}
		}
		if( bTurnDisk )
		{
			SIO_Dismount( 1 );
			SIO_Mount( 1, szDiskPath );
			WriteRegDrives( NULL );
		}
		/* Restore directory */
		SetCurrentDirectory( szCurDir );
	}
}

/*========================================================
Method   : CMainFrame::OnFileDiskDrives
=========================================================*/
/* #FN#
   Displays Drive Selections dialog box */
void
/* #AS#
   Nothing */
CMainFrame::
OnFileDiskDrives()
{
	if( g_ulScreenMode & SM_ATTR_SMALL_DLG )
	{
		CDriveSmallDlg dlgDriveSmall( this );
		dlgDriveSmall.DoModal();
	}
	else
	{
		CDriveDlg dlgDrive( this );
		dlgDrive.DoModal();
	}
	ClearScreen( FALSE );
} /* #OF# CMainFrame::OnFileDiskDrives */

void
CMainFrame::
OnFileAttachCartridgeImage() 
{
	char szCurCart[ MAX_PATH ];
	char szNewCart[ MAX_PATH ];

	ReadRegBinary( NULL, REG_OTHER_ROM, szCurCart, MAX_PATH, FALSE );
	strncpy( szNewCart, szCurCart, MAX_PATH );

	if( PickCartridge( szNewCart ) &&
		(_stricmp( szNewCart, szCurCart ) != 0 || !rom_inserted) )
	{
		AttachCartridge( szNewCart );

		WriteRegString( NULL, REG_CURRENT_ROM, current_rom );
		WriteRegString( NULL, REG_OTHER_ROM,   current_rom );
		WriteRegDWORD ( NULL, REG_CART_TYPE,   cart_type );

		RestartEmulation();
	}
	ClearScreen( FALSE );
}

/*========================================================
Method   : CMainFrame::OnFileRomImages
=========================================================*/
/* #FN#
   Displays ROM Images dialog box */
void
/* #AS#
   Nothing */
CMainFrame::
OnFileRomImages() 
{
	CRomImagesDlg dlgRomImages;

	dlgRomImages.DoModal();
	ClearScreen( FALSE );
} /* #OF# CMainFrame::OnFileRomImages */

/*========================================================
Method   : CMainFrame::OnFileHardDisks
=========================================================*/
/* #FN#
   Displays Pick Virtual Harddisk Directories dialog box */
void
/* #AS#
   Nothing */
CMainFrame::
OnFileHardDisks() 
{
	CHarddiskDlg dlgHardDisk;

	dlgHardDisk.DoModal();
	ClearScreen( FALSE );
} /* #OF# CMainFrame::OnFileHardDisks */

/*========================================================
Method   : CMainFrame::OnFileReadState
=========================================================*/
/* #FN#
   Restores saved state file */
void
/* #AS#
   Nothing */
CMainFrame::
OnFileReadState()
{
	if( zlib_capable() == -1 )
	{
		MessageBox( "Cannot save/restore state files without a working ZLIB.DLL.", "Atari800Win", MB_ICONEXCLAMATION | MB_OK );
		return;
	}

	if( PickFileName( TRUE, m_strStateName, "Load Atari saved-state file",
					  PF_A8S_FILTER, "a8s", PF_LOAD_FLAGS, FALSE ) &&
		!m_strStateName.IsEmpty() )
	{
		if( !ReadAtariState( (LPSTR)(LPCSTR)m_strStateName, "rb" ) )
			MessageBox( "The Atari read state operation failed. You can check the 'File/View Log File' option to see a description of the problem.", "Atari800Win", MB_ICONEXCLAMATION | MB_OK );

		if( !(g_ulAtariState & ATARI_PAUSED) )
		{
			ClearScreen( FALSE );
			Toggle_Pause();
		}
		DescribeAtariSystem();
		return;
	}
	g_ulAtariState &= ~ATARI_PAUSED;

	ClearScreen( FALSE );
} /* #OF# CMainFrame::OnFileReadState */

/*========================================================
Method   : CMainFrame::OnFileSaveStateNormal
=========================================================*/
/* #FN#
   Dumps the exact state of the Atari to a file */
void
/* #AS#
   Nothing */
CMainFrame::
OnFileSaveStateNormal() 
{
	SaveState( 0 );
} /* #OF# CMainFrame::OnFileSaveStateNormal */

/*========================================================
Method   : CMainFrame::OnFileSaveStateVerbose
=========================================================*/
/* #FN#
   Saves a memory image with the ROMs */
void
/* #AS#
   Nothing */
CMainFrame::
OnFileSaveStateVerbose() 
{
	SaveState( 1 );
} /* #OF# CMainFrame::OnFileSaveStateVerbose */

/*========================================================
Method   : CMainFrame::OnAtariMachineType
=========================================================*/
/* #FN#
   Flips emulated machines */
void
/* #AS#
   Nothing */
CMainFrame::
OnAtariMachineType()
{
	if( 5 == default_system && 2 == Ram256 )
	{
		Ram256 = 1; /* Rambo */
		g_ulMiscStates |= MS_RAMBO_MODE;
		WriteRegDWORD( NULL, REG_MISC_STATES, g_ulMiscStates );
	}
	else if( ++default_system > 6 )
	{
		default_system = 1;
	}
	else if( 5 == default_system )
	{
		Ram256 = 2; /* Compy Shop */
		g_ulMiscStates &= ~MS_RAMBO_MODE;
		WriteRegDWORD( NULL, REG_MISC_STATES, g_ulMiscStates );
	}
	DescribeAtariSystem();

	WriteRegDWORD( NULL, REG_DEFAULT_SYSTEM, default_system );
	RestartEmulation();
} /* #OF# CMainFrame::OnAtariMachineType */

void
CMainFrame::
OnAtariMachineType800osa() 
{
	default_system = 1;		// OS-A
	DescribeAtariSystem();

	WriteRegDWORD( NULL, REG_DEFAULT_SYSTEM, default_system );
	RestartEmulation();
}

void
CMainFrame::
OnAtariMachineType800osb() 
{
	default_system = 2;		// OS-B
	DescribeAtariSystem();

	WriteRegDWORD( NULL, REG_DEFAULT_SYSTEM, default_system );
	RestartEmulation();
}

void
CMainFrame::
OnAtariMachineType800xl() 
{
	default_system = 3;		// 800 XL
	DescribeAtariSystem();

	WriteRegDWORD( NULL, REG_DEFAULT_SYSTEM, default_system );
	RestartEmulation();
}

void
CMainFrame::
OnAtariMachineType130xe() 
{
	default_system = 4;		// 130 XE
	DescribeAtariSystem();

	WriteRegDWORD( NULL, REG_DEFAULT_SYSTEM, default_system );
	RestartEmulation();
}

void
CMainFrame::
OnAtariMachineType320xeCompy() 
{
	default_system = 5;
	Ram256 = 2;
	g_ulMiscStates &= ~MS_RAMBO_MODE;
	DescribeAtariSystem();

	WriteRegDWORD( NULL, REG_DEFAULT_SYSTEM, default_system );
	WriteRegDWORD( NULL, REG_MISC_STATES,    g_ulMiscStates );
	RestartEmulation();
}

void
CMainFrame::
OnAtariMachineType320xeRambo() 
{
	default_system = 5;
	Ram256 = 1;
	g_ulMiscStates |= MS_RAMBO_MODE;
	DescribeAtariSystem();

	WriteRegDWORD( NULL, REG_DEFAULT_SYSTEM, default_system );
	WriteRegDWORD( NULL, REG_MISC_STATES,    g_ulMiscStates );
	RestartEmulation();
}

void
CMainFrame::
OnAtariMachineType5200() 
{
	default_system = 6;
	DescribeAtariSystem();

	WriteRegDWORD( NULL, REG_DEFAULT_SYSTEM, default_system );
	RestartEmulation();
}

/*========================================================
Method   : CMainFrame::OnAtariVideoSystem
=========================================================*/
/* #FN#
   Flips emulated video standards */
void
/* #AS#
   Nothing */
CMainFrame::
OnAtariVideoSystem()
{
	if( ++default_tv_mode > 2 )
		default_tv_mode = 1;

	WriteRegDWORD( NULL, REG_DEFAULT_TV_MODE, default_tv_mode );
	RestartEmulation();
} /* #OF# CMainFrame::OnAtariVideoSystem */

/*========================================================
Method   : CMainFrame::OnAtariVideoSystemNtsc
=========================================================*/
/* #FN#
	Sets the state of object regarding to NTSC menu */
void
/* #AS#
   Nothing */
CMainFrame::
OnAtariVideoSystemNtsc() 
{
	default_tv_mode = 2;

	WriteRegDWORD( NULL, REG_DEFAULT_TV_MODE, default_tv_mode );
	RestartEmulation();
}

/*========================================================
Method   : CMainFrame::OnAtariVideoSystemPal
=========================================================*/
/* #FN#
	Sets the state of object regarding to PAL menu */
void
/* #AS#
   Nothing */
CMainFrame::
OnAtariVideoSystemPal() 
{
	default_tv_mode = 1;

	WriteRegDWORD( NULL, REG_DEFAULT_TV_MODE, default_tv_mode );
	RestartEmulation();
}
 
/*========================================================
Method   : CMainFrame::OnAtariSettings
=========================================================*/
/* #FN#
   Displays Atari Settings dialog box */
void
/* #AS#
   Nothing */
CMainFrame::
OnAtariSettings()
{
	CSettingsDlg dlgSettings( this );

	dlgSettings.DoModal();
	ClearScreen( FALSE );
} /* #OF# CMainFrame::OnAtariSettings */

void
CMainFrame::
OnAtariSioPatch()
{
	Toggle_SIOPatch();
}

void
CMainFrame::
OnAtariFullSpeed()
{
	Toggle_FullSpeed();
}

void
CMainFrame::
OnAtariPause()
{
	Toggle_Pause();
}

/*========================================================
Method   : CMainFrame::OnAtariKeyboard
=========================================================*/
/* #FN#
   Displays Keyboard Templates dialog box */
void
/* #AS#
   Nothing */
CMainFrame::
OnAtariKeyboard()
{
	CKeyboardDlg dlgKeyboard;

	dlgKeyboard.DoModal();
	ClearScreen( FALSE );
} /* #OF# CMainFrame::OnAtariKeyboard */

/*========================================================
Method   : CMainFrame::OnAtariJoysticks
=========================================================*/
/* #FN#
   Displays Joystick Options dialog box */
void
/* #AS#
   Nothing */
CMainFrame::
OnAtariJoysticks() 
{
	CJoystickDlg dlgJoystick;

	dlgJoystick.DoModal();
	ClearScreen( FALSE );
} /* #OF# CMainFrame::OnAtariJoysticks */

void
CMainFrame::
OnAtariWarmstart() 
{
	if( g_ulAtariState & ATARI_LOAD_FAILED )
	{
		RestartSound(); /* CAtari800WinView::OnSetFocus won't invoke that */
		/* Non Hercules contra plures... */
		RestartEmulation();
	}
	else
	{
		if( g_ulAtariState & ATARI_CRASHED )
		{
			g_ulAtariState = ATARI_RUNNING;
			wsync_halt = 0;	/* Turn on CPU */
			test_val   = 0;
			RestartSound(); /* Vide supra */
		}
		/* Do warm reset */
		Warmstart();
	}
	DescribeAtariSystem();
}

void
CMainFrame::
OnAtariColdstart() 
{
	if( g_ulAtariState & ATARI_LOAD_FAILED )
	{
		RestartSound(); /* CAtari800WinView::OnSetFocus won't invoke that */
		/* Non Hercules contra plures... */
		RestartEmulation();
	}
	else
	{
		if( g_ulAtariState & ATARI_CRASHED )
		{
			g_ulAtariState = ATARI_RUNNING;
			wsync_halt = 0;	/* Turn on CPU */
			test_val   = 0;
			RestartSound(); /* Vide supra */
		}
		/* Do cold reset */
		Coldstart();
	}
	DescribeAtariSystem();
}

/*========================================================
Method   : CMainFrame::OnGraphicsOptions
=========================================================*/
/* #FN#
   Displays Graphics Options dialog box */
void
/* #AS#
   Nothing */
CMainFrame::
OnGraphicsOptions() 
{
	int	nResult;
	if( g_ulScreenMode & SM_ATTR_SMALL_DLG )
	{
		CGraphicsSmallDlg dlgGraphicsSmall( this );
		nResult = dlgGraphicsSmall.DoModal();
	}
	else
	{
		CGraphicsDlg dlgGraphics( this );
		nResult = dlgGraphics.DoModal();
	}
	/* Show the window because if we changed DDraw modes the frame will
	   be missing until forced to redraw here */
	if( IDOK == nResult )
		ShowWindow( SW_SHOWNORMAL );

	ClearScreen( FALSE );
} /* #OF# CMainFrame::OnGraphicsOptions */

/*========================================================
Method   : CMainFrame::OnGraphicsGdiForWindows
=========================================================*/
/* #FN#
   Switches between GDI and DirectDraw for Window modes */
void
/* #AS#
   Nothing */
CMainFrame::
OnGraphicsGdiForWindows()
{
	ULONG ulSavedMode = g_ulScreenMode;

	if( g_ulScreenMode & SM_OPTN_USE_GDI )
		g_ulScreenMode &= ~SM_OPTN_USE_GDI;
	else
		g_ulScreenMode |= SM_OPTN_USE_GDI;

	if( !InitialiseScreen( FALSE ) )
	{
		g_ulScreenMode = ulSavedMode;
		InitialiseScreen( FALSE );
	}
	WriteRegDWORD( NULL, REG_DDRAW_MODE, g_ulScreenMode );
} /* #OF# CMainFrame::OnGraphicsGdiForWindows */

void
CMainFrame::
OnGraphicsScanlines()
{
	ULONG ulSavedMode = g_ulScreenMode;

	if( g_ulScreenMode & SM_OPTN_SHOW_SCANLINES )
		g_ulScreenMode &= ~SM_OPTN_SHOW_SCANLINES;
	else
		g_ulScreenMode |= SM_OPTN_SHOW_SCANLINES;

	if( !InitialiseScreen( FALSE ) )
	{
		g_ulScreenMode = ulSavedMode;
		InitialiseScreen( FALSE );
	}
	WriteRegDWORD( NULL, REG_DDRAW_MODE, g_ulScreenMode );
}

/*========================================================
Method   : CMainFrame::OnGraphicsArtifacting
=========================================================*/
/* #FN#
   Flips artifacting modes */
void
/* #AS#
   Nothing */
CMainFrame::
OnGraphicsArtifacting()
{
	if( ++global_artif_mode > 4 )
		global_artif_mode = 0;
	artif_init();

	WriteRegDWORD( NULL, REG_ARTIF_MODE, global_artif_mode );
} /* #OF# CMainFrame::OnGraphicsArtifacting */

void
CMainFrame::
OnGraphicsArtifactingNone()
{
	global_artif_mode = 0;

	artif_init();
	WriteRegDWORD( NULL, REG_ARTIF_MODE, global_artif_mode );
}

void
CMainFrame::
OnGraphicsArtifactingBluebrown1()
{
	global_artif_mode = 1;

	artif_init();
	WriteRegDWORD( NULL, REG_ARTIF_MODE, global_artif_mode );
}

void
CMainFrame::
OnGraphicsArtifactingBluebrown2()
{
	global_artif_mode = 2;

	artif_init();
	WriteRegDWORD( NULL, REG_ARTIF_MODE, global_artif_mode );
}

void
CMainFrame::
OnGraphicsArtifactingGtia()
{
	global_artif_mode = 3;

	artif_init();
	WriteRegDWORD( NULL, REG_ARTIF_MODE, global_artif_mode );
}

void
CMainFrame::
OnGraphicsArtifactingCtia()
{
	global_artif_mode = 4;

	artif_init();
	WriteRegDWORD( NULL, REG_ARTIF_MODE, global_artif_mode );
}

void
CMainFrame::
OnGraphicsPalette()
{
	CPaletteDlg dlgPalette( this );

	dlgPalette.DoModal();
	ClearScreen( FALSE );
}

/*========================================================
Method   : CMainFrame::OnGraphicsSaveScreenshotNormal
=========================================================*/
/* #FN#
   Saves normal screen snapshot */
void
/* #AS#
   Nothing */
CMainFrame::
OnGraphicsSaveScreenshotNormal()
{
	SaveSnapshot( 0 );
} /* #OF# CMainFrame::OnGraphicsSaveScreenshotNormal */

/*========================================================
Method   : CMainFrame::OnGraphicsSaveScreenshotInterlaced
=========================================================*/
/* #FN#
   Saves interlaced screen snapshot */
void
/* #AS#
   Nothing */
CMainFrame::
OnGraphicsSaveScreenshotInterlaced()
{
	SaveSnapshot( 1 );
} /* #OF# CMainFrame::OnGraphicsSaveScreenshotInterlaced */

/*========================================================
Method   : CMainFrame::OnGraphicsPerformanceTest
=========================================================*/
/* #FN#
   Renders speed of the current graphics mode */
void
/* #AS#
   Nothing */
CMainFrame::
OnGraphicsPerformanceTest()
{
	ScreenBenchmark();
	ClearScreen( FALSE );
} /* #OF# CMainFrame::OnGraphicsPerformanceTest */

/*========================================================
Method   : CMainFrame::OnSoundOptions
=========================================================*/
/* #FN#
   Displays Sound Options dialog box */
void
/* #AS#
   Nothing */
CMainFrame::
OnSoundOptions() 
{
	CSoundDlg dlgSound;

	dlgSound.DoModal();
	ClearScreen( FALSE );
} /* #OF# CMainFrame::OnSoundOptions */

void
CMainFrame::
OnSoundDigitizedEffects()
{
	use_volume_only = use_volume_only ? 0 : 1;

	WriteRegDWORD( NULL, REG_USE_VOLUME_ONLY, use_volume_only );
	ClearSound( TRUE );
	InitialiseSound();
}

void
CMainFrame::
OnSoundStereo() 
{
#ifndef STEREO
	stereo_enabled = 0;
#else
	stereo_enabled = stereo_enabled ? 0 : 1;
#endif
	WriteRegDWORD( NULL, REG_ENABLE_STEREO, stereo_enabled );
	ClearSound( TRUE );
	InitialiseSound();
}

void
CMainFrame::
OnSoundMute() 
{
	if( g_ulSoundState & SOUND_NOSOUND )
	{
		g_ulSoundState &= ~SOUND_NOSOUND;
		/* Registry is corrupted? */
		if( !(g_ulSoundState & (SOUND_MMSOUND | SOUND_DIRECTSOUND)) )
			g_ulSoundState |= SOUND_MMSOUND;
	}
	else
		g_ulSoundState |= SOUND_NOSOUND;

	WriteRegDWORD( NULL, REG_SOUND_STATE, g_ulSoundState );
	ClearSound( TRUE );
	InitialiseSound();
}

/*========================================================
Method   : CMainFrame::OnSoundSaveSound
=========================================================*/
/* #FN#
   Saves the output of the Pokey processor to a file */
void
/* #AS#
   Nothing */
CMainFrame::
OnSoundSaveSound()
{
	int nResult = IDOK;

	if( g_pfSndOutput )
	{
		/* Close Sound Output file */
		CloseSndOutput();
		return; /* The sound is turned off now */
	}

	if( g_ulSoundState & SOUND_NOSOUND )
	{
		LoadString( NULL, IDS_SOUND_RECORD, g_tszErrorString, LOADSTRING_STRING_SIZE );
		MessageBox( g_tszErrorString, "Atari800Win", MB_ICONINFORMATION | MB_OK );
	}

	if( !(g_ulDontShowFlags & DONT_SHOW_SOUNDFILE_WARN) )
	{
		CWarningDlg	dlgWarning;
		char szMessage[ LOADSTRING_STRING_SIZE ];

		LoadString( NULL, IDS_WARNING_SOUNDFILE, szMessage, LOADSTRING_STRING_SIZE );
		dlgWarning.m_strWarnText = szMessage;
		dlgWarning.m_ulWarnBit   = DONT_SHOW_SOUNDFILE_WARN;
		nResult = dlgWarning.DoModal();
	}
	if( nResult == IDOK )
	{
		if( PickFileName( FALSE, m_strSoundName, "Name of WAV file to save",
						  PF_WAV_FILTER, "wav", PF_SAVE_FLAGS, FALSE ) &&
			!m_strSoundName.IsEmpty() )
		{
			/* Open Sound Output file */
			OpenSndOutput( (LPSTR)(LPCSTR)m_strSoundName );
		}
	}
	ClearScreen( FALSE );
} /* #OF# CMainFrame::OnSoundSaveSound */

/*========================================================
Method   : CMainFrame::OnSoundPerformanceTest
=========================================================*/
/* #FN#
   Tests the Pokey processor overhead with current settings */
void
/* #AS#
   Nothing */
CMainFrame::
OnSoundPerformanceTest()
{
	ULONG ulStartTime, ulTotalTime;
	int   nSampleSize;
	char *pszBuffer = NULL;

	if( !(g_ulAtariState & ATARI_RUNNING) || g_ulSoundState & SOUND_NOSOUND )
	{
		LoadString( NULL, IDS_SOUND_TEST_WARNING, g_tszErrorString, LOADSTRING_STRING_SIZE );
		MessageBox( g_tszErrorString, "Atari800Win", MB_ICONINFORMATION | MB_OK );
		return;
	}
	if( default_tv_mode == 1 )
		nSampleSize = g_nSoundRate / 50;
	else
		nSampleSize = g_nSoundRate / 60;

	if( !(pszBuffer = (char*)calloc( 1, nSampleSize + 1 )) )
		return;

	LoadString( NULL, IDS_SOUND_TEST_BEGIN, g_tszErrorString, LOADSTRING_STRING_SIZE );
	MessageBox( g_tszErrorString, "Atari800Win", MB_ICONINFORMATION | MB_OK );

	ulStartTime = timeGetTime();
	/* Begin performance test */
	for( int i = 0; i < 60; i++ )
		Pokey_process( (UCHAR*)pszBuffer, nSampleSize );

	ulTotalTime = timeGetTime() - ulStartTime;

	LoadString( NULL, IDS_SOUND_TEST_RESULT, g_tszErrorString, LOADSTRING_STRING_SIZE );
	sprintf( pszBuffer, g_tszErrorString, nSampleSize, ulTotalTime, (double)ulTotalTime / 60.0 );

	LoadString( NULL, IDS_BENCHMARK_RESULTS, g_tszErrorString, LOADSTRING_STRING_SIZE );
	MessageBox( pszBuffer, g_tszErrorString, MB_ICONINFORMATION | MB_OK );

	free( pszBuffer );

	ClearScreen( FALSE );
} /* #OF# CMainFrame::OnSoundPerformanceTest */

/*========================================================
Method   : CMainFrame::OnMiscFileAssociations
=========================================================*/
/* #FN#
   Displays File Associations dialog box */
void
/* #AS#
   Nothing */
CMainFrame::
OnMiscFileAssociations()
{
	CFileAssociationsDlg dlgFileAssociations;

	dlgFileAssociations.DoModal();
	ClearScreen( FALSE );
} /* #OF# CMainFrame::OnMiscFileAssociations */

/*========================================================
Method   : CMainFrame::OnMiscConvertDcmToAtr
=========================================================*/
/* #FN#
   Converts .DCM files to .ATR files */
void
/* #AS#
   Nothing */
CMainFrame::
OnMiscConvertDcmToAtr()
{
	char szLocalFileBuffer[ 8192 ];
	CFileDialog	dlgDiskImage( TRUE, NULL, NULL, OFN_PATHMUSTEXIST | OFN_EXPLORER | OFN_FILEMUSTEXIST | OFN_ALLOWMULTISELECT | OFN_HIDEREADONLY, PF_DCM_FILTER, this );

	*szLocalFileBuffer = '\0';
	dlgDiskImage.m_ofn.lpstrFile  = szLocalFileBuffer;
	dlgDiskImage.m_ofn.nMaxFile   = 8192;
	dlgDiskImage.m_ofn.lpstrTitle = "Select DCM file(s) to convert";

	if( IDOK == dlgDiskImage.DoModal() )
	{
		char     szDestDir[ MAX_PATH ];
		CString  strInput;
		FILE    *fpInput, *fpOutput;
		int      i;

		strcpy( szDestDir, dlgDiskImage.GetPathName() );
		for( i = strlen( szDestDir ); i > 0 && szDestDir[ i ] != '\\' && szDestDir[i] != '.'; i-- );
		if( i == 0 || szDestDir[ i ] != '.' )
			i = strlen( szDestDir );
		strcpy( &szDestDir[ i ], ".atr" );

		CFileDialog	dlgDestDir( FALSE, NULL, szDestDir, OFN_PATHMUSTEXIST | OFN_EXPLORER | OFN_HIDEREADONLY, NULL, this );
		dlgDestDir.m_ofn.lpstrTitle = "Select destination directory to write file(s)";
		if( dlgDestDir.DoModal() != IDOK )
			return;
		strcpy( szDestDir, dlgDestDir.GetPathName() );

		for( i = strlen( szDestDir ); i > 0 && szDestDir[ i ] != '\\'; i-- );
		szDestDir[ i ] = '\0';

		if( *szDestDir != '\0' )
		{
			POSITION pos = dlgDiskImage.GetStartPosition();

			ASSERT(pos);
			if( !pos )
				return;

			strInput = dlgDiskImage.GetNextPathName( pos );
			while( strInput != "" )
			{
				char szFullDestName[ MAX_PATH ];
				char szThisFileName[ MAX_PATH ];

				strcpy( szThisFileName, strInput );
				fpInput = fopen( szThisFileName, "rb" );
				if( !fpInput )
				{
					char szError[ MAX_PATH + 32 ];
					sprintf( szError, "Can't open input file %s.", szThisFileName );
					MessageBox( szError, "Atari800Win", MB_ICONEXCLAMATION | MB_OK );
					return;
				}
				for( i = strlen( szThisFileName ); i > 0 && szThisFileName[ i ]!= '\\'; i-- );
				if( szThisFileName[i] == '\\' )
					i++;

				strcpy( szFullDestName, szDestDir );
				strcat( szFullDestName, "\\" );
				strcat( szFullDestName, &szThisFileName[ i ] );

				for( i = strlen( szFullDestName ); i > 0 && szFullDestName[ i ] != '\\' && szFullDestName[ i ] != '.'; i-- );
				if( i == 0 || szFullDestName[ i ] != '.' )
					i = strlen( szFullDestName );
				strcpy( &szFullDestName[ i ], ".atr" );
				
				fpOutput = fopen( szFullDestName, "rb" );
				if( fpOutput )
				{
					char szWarning[ 256 + MAX_PATH ];
					int  nResult;

					sprintf( szWarning, "The file %s already exists, are you sure you want to overwrite it?", szFullDestName );
					nResult = MessageBox( szWarning, "Atari800Win", MB_ICONQUESTION | MB_YESNO );
					if( nResult == IDNO )
					{
						fclose( fpInput );
						return;
					}
					fclose( fpOutput );
				}
				fpOutput = fopen( szFullDestName, "wb" );
				if( !fpOutput )
				{
					char szWarning[ 256 + MAX_PATH ];
					sprintf( szWarning, "The file %s could not be opened for output, stopping operation.", szFullDestName );
					MessageBox( szWarning, "Atari800Win", MB_ICONEXCLAMATION | MB_OK );
					return;
				}
				if( !dcmtoatr( fpInput, fpOutput, szThisFileName, szFullDestName ) )
				{
					int nResult;
					nResult = MessageBox( "There was an error during the operation and it was aborted. Would you like to check the error log for details?", "Atari800Win", MB_ICONQUESTION | MB_YESNO );
					fclose( fpInput );
					fclose( fpOutput );
					if( nResult == IDYES )
					{
						CErrorLogDlg dlgErrorLog;
						dlgErrorLog.DoModal();
					}
					return;
				}
				fclose( fpOutput );
				fclose( fpInput );
				if( pos )
					strInput = dlgDiskImage.GetNextPathName( pos );
				else
					strInput = "";
			}
		}
		MessageBox( "Conversion successfull.", "Atari800Win", MB_ICONINFORMATION | MB_OK );
	}
	ClearScreen( FALSE );
} /* #OF# CMainFrame::OnMiscConvertDcmToAtr */

/*========================================================
Method   : CMainFrame::OnMiscClearAllSettings
=========================================================*/
/* #FN#
   Clears all saved settings/drives/ROMs */
void
/* #AS#
   Nothing */
CMainFrame::
OnMiscClearAllSettings()
{
	char szMessage[ MAX_PATH + 64 ] = { '\0' };
	int  nResult;

	LoadString( NULL, IDS_CLEAR_AFFIRM, g_tszErrorString, LOADSTRING_STRING_SIZE );
	nResult = MessageBox( g_tszErrorString, "Atari800Win", MB_ICONQUESTION | MB_YESNO );

	if( nResult == IDYES )
	{
		DeleteAllRegKeys( NULL, REGNAME );
		HandleRegistry();

		if( g_ulMiscStates & MS_USE_KEYTEMPLATE && 
			!ReadKeyTemplate( g_szTemplateFile ) )
		{
			sprintf( szMessage, "Could not load keyboard template file %s, turning off keyboard template.", g_szTemplateFile );
			g_ulMiscStates &= ~MS_USE_KEYTEMPLATE;
		}
		if( g_ulMiscStates & MS_USE_PALETTE && 
			!read_palette( g_szPaletteFile ) )
		{
			sprintf( szMessage, "Could not load external palette file %s, turning off external palette.", g_szPaletteFile );
			g_ulMiscStates &= ~MS_USE_PALETTE;
		}
		if( *szMessage != '\0' )
		{
			MessageBox( szMessage, "Atari800Win", MB_ICONEXCLAMATION | MB_OK );
			WriteRegDWORD( NULL, REG_MISC_STATES, g_ulMiscStates );
		}
	}
} /* #OF# CMainFrame::OnMiscClearAllSettings */

/*========================================================
Method   : CMainFrame::OnMiscRestartEmulation
=========================================================*/
/* #FN#
   Reinitializes the Windows stuff and emulated Atari */
void
/* #AS#
   Nothing */
CMainFrame::
OnMiscRestartEmulation()
{
	/* Restart sound if we need that */
	RestartSound();
	/* Force Windows stuff initialization */
	RestartEmulation();
} /* #OF# CMainFrame::OnMiscRestartEmulation */

/*========================================================
Method   : CMainFrame::OnMiscViewLogFile
=========================================================*/
/* #FN#
   Displays Error Log dialog box */
void
/* #AS#
   Nothing */
CMainFrame::
OnMiscViewLogFile() 
{
	CErrorLogDlg dlgErrorLog;

	dlgErrorLog.DoModal();
	ClearScreen( FALSE );
} /* #OF# CMainFrame::OnMiscViewLogFile */

/*========================================================
Method   : CMainFrame::OnMiscMonitor
=========================================================*/
/* #FN#
   Launches monitor console */
void
/* #AS#
   Nothing */
CMainFrame::
OnMiscMonitor() 
{
	LaunchMonitor();
	ClearScreen( FALSE );
} /* #OF# CMainFrame::OnMiscMonitor */

/*========================================================
Method   : CMainFrame::OnTimer
=========================================================*/
/* #FN#
   The framework calls this member function after each interval specified
   in the SetTimer member function used to install a timer */
void
/* #AS#
   Nothing */
CMainFrame::
OnTimer( UINT nIDEvent /* #IN# Specifies the identifier of the timer */ )
{
	if( nIDEvent == TIMER_READ_JOYSTICK )
	{
		ReadJoystickInput( 0, 0 );
		ReadJoystickInput( 1, 0 );
		ReadJoystickInput( 2, 0 );
		ReadJoystickInput( 3, 0 );
		if( !(g_ulAtariState & ATARI_PAUSED) )
			::KillTimer( g_hMainWnd, TIMER_READ_JOYSTICK );
	}
	CFrameWnd::OnTimer( nIDEvent );
} /* #OF# CMainFrame::OnTimer */

void
CMainFrame::
OnDropFiles( HDROP hDropInfo )
{
	UINT    unFileType  = IAF_ATARI_FILE;
	CString strFileName = "";

	if( ::DragQueryFile( hDropInfo, (UINT)-1, NULL, 0 ) )
	{
		if( ::DragQueryFile( hDropInfo, 0, NULL, 0 ) < MAX_PATH )
		{
			::DragQueryFile( hDropInfo, 0, strFileName.GetBuffer( MAX_PATH ), MAX_PATH );
			strFileName.ReleaseBuffer();
		}
	}
	::DragFinish( hDropInfo );

	if( IsAtariFile( strFileName, &unFileType ) )
	{
		/* Is it an Atari executable? */
		if( IAF_EXECUTABLE == unFileType )
		{
			if( !RunAtariExe( strFileName ) )
			{
				LoadString( NULL, IDS_RUNEXE_ERROR, g_tszErrorString, LOADSTRING_STRING_SIZE );
				MessageBox( g_tszErrorString, "Atari800Win", MB_ICONEXCLAMATION | MB_OK );
			}
		}
		else
		/* Is it an Atari disk image? */
		if( IAF_DISK_IMAGE == unFileType )
		{
			SIO_Dismount( 1 );
			SIO_Mount( 1, (LPSTR)(LPCSTR)strFileName );
			WriteRegDrives( NULL );

			Coldstart();
		}
		else
		/* Is it an Atari cartridge image? */
		if( IAF_CART_IMAGE == unFileType )
		{
			AttachCartridge( (LPSTR)(LPCSTR)strFileName );

			WriteRegString( NULL, REG_CURRENT_ROM, current_rom );
			WriteRegString( NULL, REG_OTHER_ROM,   current_rom );
			WriteRegDWORD ( NULL, REG_CART_TYPE,   cart_type );

			RestartEmulation( FALSE ); /* Don't reinitialise Windows stuff */
		}
	}
	/* The main window has lost the focus, we have to take it back */
//	SetActiveWindow(); /* Doesn't work properly */

	DescribeAtariSystem();
//	CFrameWnd::OnDropFiles( hDropInfo );
}

/*========================================================
Method   : CMainFrame::OnMove
=========================================================*/
/* #FN#
   The framework calls this function after the CWnd object has been moved */
void
/* #AS#
   Nothing */
CMainFrame::
OnMove( int x, /* #IN# New x-coordinate location of the upper-left corner */
	    int y  /* #IN# New y-coordinate location of the upper-left corner */ )
{
	CFrameWnd::OnMove( x, y );

	if( g_hWnd )
		ComputeClipArea();
} /* #OF# CMainFrame::OnMove */

/*========================================================
Method   : CMainFrame::OnClose
=========================================================*/
/* #FN#
   The framework calls this function as a signal that the CWnd or an
   application is to terminate */
void
/* #AS#
   Nothing */
CMainFrame::
OnClose() 
{
	RECT rc;

	if( g_pfSndOutput )
		/* Close Sound Output file */
		CloseSndOutput();

	GetWindowRect( &rc );

	if( (rc.left != 0) && (g_nStartX != rc.left) )
		WriteRegDWORD( NULL, REG_START_XPOS, (DWORD)rc.left );

	if( (rc.top != 0) && (g_nStartY != rc.top) )
		WriteRegDWORD( NULL, REG_START_YPOS, (DWORD)rc.top );
	
	CFrameWnd::OnClose();
} /* #OF# CMainFrame::OnClose */

void
CMainFrame::
OnUpdateFileTurnDisk( CCmdUI *pCmdUI )
{
	pCmdUI->Enable( 
		strcmp( sio_filename[ 0 ], "Off" )   != 0 &&
		strcmp( sio_filename[ 0 ], "Empty" ) != 0 );
}

void
CMainFrame::
OnUpdateAtariMachineType( CCmdUI *pCmdUI )
{
	int nDefaultSystem = default_system;
	if( (5 == nDefaultSystem && 1 == Ram256) || nDefaultSystem > 5 )
		nDefaultSystem++;

	pCmdUI->SetRadio( (pCmdUI->m_nID - ID_ATARI_MACHINETYPE_BASE) == DWORD(nDefaultSystem - 1) );
}

void
CMainFrame::
OnUpdateAtariVideoSystem( CCmdUI *pCmdUI )
{
	pCmdUI->SetRadio( (pCmdUI->m_nID - ID_ATARI_VIDEOSYSTEM_BASE) == DWORD(default_tv_mode - 1) );
}

void
CMainFrame::
OnUpdateAtariSioPatch( CCmdUI *pCmdUI )
{
	pCmdUI->SetCheck( enable_sio_patch );
	pCmdUI->Enable( enable_rom_patches );
}

void
CMainFrame::
OnUpdateAtariFullSpeed( CCmdUI *pCmdUI )
{
	pCmdUI->SetCheck( (g_ulMiscStates & MS_FULL_SPEED) != 0 );
}

void
CMainFrame::
OnUpdateAtariPause( CCmdUI *pCmdUI )
{
	pCmdUI->SetCheck( (g_ulAtariState & ATARI_PAUSED) != 0 );
}

void
CMainFrame::
OnUpdateGraphicsGdiForWindows( CCmdUI *pCmdUI )
{
	pCmdUI->SetCheck( (g_ulScreenMode & SM_OPTN_USE_GDI) != 0 );
}

void
CMainFrame::
OnUpdateGraphicsScanlines( CCmdUI *pCmdUI )
{
	pCmdUI->SetCheck( (g_ulScreenMode & SM_OPTN_SHOW_SCANLINES) != 0 );
}

void
CMainFrame::
OnUpdateGraphicsArtifacting( CCmdUI *pCmdUI )
{
	pCmdUI->SetRadio( (pCmdUI->m_nID - ID_GRAPHICS_ARTIFACTING_BASE) == DWORD(global_artif_mode) );
}

void
CMainFrame::
OnUpdateSoundDigitizedEffects( CCmdUI *pCmdUI )
{
	pCmdUI->SetCheck( use_volume_only );
}

void
CMainFrame::
OnUpdateSoundStereo( CCmdUI *pCmdUI )
{
	pCmdUI->SetCheck( stereo_enabled );
}

void
CMainFrame::
OnUpdateSoundMute( CCmdUI *pCmdUI )
{
	pCmdUI->SetCheck( (g_ulSoundState & SOUND_NOSOUND) != 0 );
}

void
CMainFrame::
OnUpdateSoundSaveSound( CCmdUI *pCmdUI )
{
	pCmdUI->SetCheck( (BOOL)(g_pfSndOutput != NULL) );
}
