/****************************************************************************
File    : input_win.c
/*
@(#) #SY# Atari800Win
@(#) #IS# Input implementation for Win32 platforms
@(#) #BY# Richard Lawrence, Tomasz Szymankowski
@(#) #LM# 13.04.2000
*/

/*
Copyright (c) 1998 Richard Lawrence

This program is free software; you can redistribute it and/or modify it under the terms 
of the GNU General Public License as published by the Free Software Foundation; either 
version 2 of the License, or (at your option) any later version. This program is 
distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even
the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details. You should have received a copy of the GNU
General Public License along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include <stdio.h>
#include <dinput.h>
#include <crtdbg.h>
#include "WinConfig.h"
#include "Resource.h"
#include "atari800.h"
#include "globals.h"
#include "display_win.h"
#include "misc_win.h"
#include "sound_win.h"
#include "registry.h"
#include "input_win.h"


/* Public objects */

HINSTANCE g_hDInput     = NULL;  /* DirectInput 5.0 interface object */
int       g_nCurrentKey = AKEY_NONE;
int       g_nCurrentVK  = 0;
int       g_nNewKey     = AKEY_NONE;
int       g_nNewVK      = 0;
int       g_nNumDIDevs  = 0;
BOOL      g_bKBJoystick = FALSE;
int       CTRL_KEY;

char g_acInputDevNames      [ MAX_INPUT_DEVICES  ][ INPUT_DEV_NAMELEN ];
int  g_anSelectedInputDevice[ MAX_ATARI_JOYPORTS ] =
{
	-4, 127, 127, 127
};

#define VK_NUMPADENTER	95 /* For temporary use */

BYTE g_anExtendedKeyTab[ 256 ] =
{
/*000*/ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, VK_NUMPADENTER, 0, 0,
/*016*/ 0, VK_RCONTROL, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*032*/ 0, VK_PRIOR, VK_NEXT, VK_END, VK_HOME, VK_LEFT, VK_UP, VK_RIGHT, VK_DOWN, 0, 0, 0, 0, VK_INSERT, VK_DELETE, 0,
/*048*/ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*064*/ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*080*/ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, VK_LWIN, VK_RWIN, 0, 0, 0,
/*096*/ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, VK_DIVIDE,
/*112*/ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*128*/ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*144*/ VK_NUMLOCK, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*160*/ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*176*/ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*192*/ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*208*/ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*224*/ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*240*/ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
};

WORD g_anKeyset[ NUM_KBJOY_DEVICES ][ NUM_KBJOY_KEYS ] =
{
	/* Default is numpad "joystick" */
	{ VK_NUMPAD7, VK_NUMPAD8, VK_NUMPAD9, VK_NUMPAD6, VK_NUMPAD3, VK_NUMPAD2, VK_NUMPAD1, VK_NUMPAD4, VK_NUMPAD5, VK_NUMPAD0 },
	/* Cursor Keys + RIGHT CTRL */
	{ 0, VK_UP, 0, VK_RIGHT, 0, VK_DOWN, 0, VK_LEFT, 0, VK_RCONTROL },
	/* User defined A & B Keysets */
	{ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
	{ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 }
};

/* Private objects */

static LPDIRECTINPUT        s_lpDInput = NULL;
static LPDIRECTINPUTDEVICE2 s_alpDevFound[ MAX_INPUT_DEVICES ];
static LPDIRECTINPUTDEVICE2 s_alpJoyPorts[ MAX_ATARI_JOYPORTS ] =
{
	NULL, NULL, NULL, NULL
};

static int	s_nJoyCount = 0;
static int  s_nConsol;
static BOOL	s_bBtn2Flip;
static BOOL	s_bBtn3Flip;
static BOOL	s_bBtn4Flip;

static int  s_anStick    [ MAX_ATARI_JOYPORTS ];
static int  s_anTrig     [ MAX_ATARI_JOYPORTS ];

static UINT s_aunKeystick[ NUM_KBJOY_DEVICES ] = { STICK_CENTRE, STICK_CENTRE, STICK_CENTRE, STICK_CENTRE };
static UINT s_aunKeytrig [ NUM_KBJOY_DEVICES ] = { 1, 1, 1, 1 };

static const int s_anStickTable[ NUM_KBJOY_KEYS - 1 ] =
{
	STICK_UL,
	STICK_FORWARD,
	STICK_UR,
	STICK_RIGHT,
	STICK_LR,
	STICK_BACK,
	STICK_LL,
	STICK_LEFT,
	STICK_CENTRE
};

static BOOL GetDIErrorString( HRESULT hResult, LPSTR lpszErrorBuff, DWORD cchError );
		
#ifdef _DEBUG
#define ServeDInputError( nUID, hResult, bQuit ) \
		ShowDInputError( nUID, hResult, bQuit, __FILE__, __LINE__ )
#else /*_DEBUG*/
#define ServeDInputError( nUID, hResult, bQuit ) \
		ShowDInputError( nUID, hResult, bQuit )
#endif /*_DEBUG*/


/*========================================================
Function : ShowDInputError
=========================================================*/
/* #FN#
   Fatal DirectInput error function. Since we cannot continue, bring us back
   to a known working state before displaying the error, then start an exit.
   Make sure to shut down the Atari layer on the way. */
static
void
/* #AS#
   Nothing */
ShowDInputError( UINT    nUID,    /* #IN# */
				 HRESULT hResult, /* #IN# */
				 BOOL    bQuit    /* #IN# */
#ifdef _DEBUG
			   , char   *pszFile,
				 DWORD   dwLine
#endif /*_DEBUG*/
)
{
	TCHAR tszError    [ LOADSTRING_STRING_SIZE ];
	TCHAR szFullString[ LOADSTRING_STRING_SIZE ];
	TCHAR szAction    [ LOADSTRING_STRING_SIZE ];
	
#ifdef _DEBUG
	Aprint( "DirectInput error: %s@%ld", pszFile, dwLine );
#endif /*_DEBUG*/

	/* Get us back to a GDI display and stop making noises */
	SetSafeDisplay( FALSE );
	ClearSound( FALSE );

	/* Get the error string and present it to the user */
	GetDIErrorString( hResult, tszError, LOADSTRING_STRING_SIZE );
	LoadString( NULL, IDS_DINPUT_ERROR_PROMPT, g_tszErrorString, LOADSTRING_STRING_SIZE );
	LoadString( NULL, nUID, szAction, LOADSTRING_STRING_SIZE );
	wsprintf( szFullString, g_tszErrorString, szAction, tszError );
	LoadString( NULL, IDS_DINPUT_ERROR_HDR, g_tszErrorString, LOADSTRING_STRING_SIZE );
	MessageBox( g_hMainWnd, szFullString, g_tszErrorString, MB_ICONSTOP | MB_OK );

	/* Start a quit (this will end up in Atari_Exit()) */
	if( bQuit )
	{
		/* Make sure the atari is turned off */
		g_ulAtariState = ATARI_UNINITIALIZED | ATARI_PAUSED;
		PostMessage( g_hMainWnd, WM_CLOSE, 0, 0L );
	}
	else
		RestartSound();
} /* #OF# ShowDInputError */

/*========================================================
Function : ResetKeys
=========================================================*/
/* #FN#
   Resets the keys state to default */
void
/* #AS#
   Nothing */
ResetKeys( void )
{
	int i;
	for( i = 0; i < NUM_KBJOY_DEVICES; i++ )
	{
		s_aunKeystick[ i ] = STICK_CENTRE;
		s_aunKeytrig [ i ] = 1;
	}
	g_nCurrentKey = g_nNewKey = AKEY_NONE;
	g_nCurrentVK  = g_nNewVK  = 0;
	SHIFT_KEY     = CTRL_KEY  = FALSE;
	KEYPRESSED    = FALSE;
	s_nConsol     = 0x07;
} /* #OF# ResetKeys */

/*========================================================
Function : ResetInput
=========================================================*/
/* #FN#
   Resets the input (keys and joys) state to default */
void
/* #AS#
   Nothing */
ResetInput( void )
{
	int i;

	s_bBtn2Flip = FALSE;
	s_bBtn3Flip = FALSE;
	s_bBtn4Flip = FALSE;

	ResetKeys();
	for( i = 0; i < MAX_ATARI_JOYPORTS; i++ )
	{
		s_anTrig [ i ] = 1;
		s_anStick[ i ] = STICK_CENTRE;
	}
} /* #OF# ResetInput */

/*========================================================
Function : SetDIDwordProperty
=========================================================*/
/* #FN#
   Helper function just makes sure the Property struct is set correctly */
static
HRESULT
/* #AS#
   DI_OK, DI_PROPNOEFFECT, if the methods succeeds. DIERR_OBJECTNOTFOUND,
   DIERR_UNSUPPORTED, DIERR_NOTINITIALIZED, DIERR_INVALIDPARAM, if the
   method fails */
SetDIDwordProperty( LPDIRECTINPUTDEVICE lpInputDev,   /* #IN# */
					REFGUID             guidProperty, /* #IN# */
					DWORD               dwObject,     /* #IN# */
					DWORD               dwHow,        /* #IN# */
					DWORD               dwValue       /* #IN# */ )
{
	DIPROPDWORD dipdw;
	
	dipdw.diph.dwSize       = sizeof(dipdw);
	dipdw.diph.dwHeaderSize = sizeof(dipdw.diph);
	dipdw.diph.dwObj        = dwObject;
	dipdw.diph.dwHow        = dwHow;
	dipdw.dwData            = dwValue;
	
	return IDirectInputDevice_SetProperty( lpInputDev, guidProperty, &dipdw.diph );
} /* #OF# SetDIDwordProperty */

/*========================================================
Function : InitJoystickInput
=========================================================*/
/* #FN#
   Callback function that will be called with a description of each
   DirectInput device */
static
BOOL CALLBACK
/* #AS#
   Returns DIENUM_CONTINUE to continue the enumeration or DIENUM_STOP
   to stop the enumeration */
InitJoystickInput( LPCDIDEVICEINSTANCE lpDevInstance,
				   LPVOID pvRef )
{
	LPDIRECTINPUT       lpDirectInput = pvRef;
	LPDIRECTINPUTDEVICE pdev;
	DIPROPRANGE         diprg;
	HRESULT             hResult;
	
	/* Create the DirectInput joystick device */
	hResult = IDirectInput_CreateDevice( lpDirectInput, &lpDevInstance->guidInstance, &pdev, NULL );
	if( FAILED(hResult) )
	{
		ServeDInputError( IDS_DIERR_CREATING_DEVICE, hResult, FALSE );
		return DIENUM_STOP;/*DIENUM_CONTINUE;*/
	}
	/* Set joystick data format */
	hResult = IDirectInputDevice_SetDataFormat( pdev, &c_dfDIJoystick );
	if( FAILED(hResult) )
	{
		ServeDInputError( IDS_DIERR_SETTING_JOYSTICK_FORMAT, hResult, FALSE );
		IDirectInputDevice_Release( pdev );
		return DIENUM_STOP;
	}
	/* Set the cooperative level */
	hResult = IDirectInputDevice_SetCooperativeLevel( pdev, g_hMainWnd, DISCL_NONEXCLUSIVE | DISCL_BACKGROUND );
	if( FAILED(hResult) )
	{
		ServeDInputError( IDS_DIERR_SETTING_COOP, hResult, FALSE );
		IDirectInputDevice_Release( pdev );
		return DIENUM_STOP;
	}
	/* Set X-axis range to (-1000 ... +1000)
	   This lets us test against 0 to see which way the s_anStick is pointed. */
	diprg.diph.dwSize       = sizeof(diprg);
	diprg.diph.dwHeaderSize = sizeof(diprg.diph);
	diprg.diph.dwObj        = DIJOFS_X;
	diprg.diph.dwHow        = DIPH_BYOFFSET;
	diprg.lMin              = -1000;
	diprg.lMax              = +1000;
	
	hResult = IDirectInputDevice_SetProperty( pdev, DIPROP_RANGE, &diprg.diph );
	if( FAILED(hResult) )
	{
		ServeDInputError( IDS_DIERR_SETTING_PROPS, hResult, FALSE );
		IDirectInputDevice_Release( pdev );
		return FALSE;
	}
	
	/* And again for Y-axis range */
	diprg.diph.dwObj = DIJOFS_Y;
	
	hResult = IDirectInputDevice_SetProperty( pdev, DIPROP_RANGE, &diprg.diph );
	if( FAILED(hResult) )
	{
		ServeDInputError( IDS_DIERR_SETTING_PROPS, hResult, FALSE );
		IDirectInputDevice_Release( pdev );
		return FALSE;
	}
	/* Set X axis dead zone to 50% (to avoid accidental turning)
	   Units are ten thousandths, so 50% = 5000/10000. */
	hResult = SetDIDwordProperty(pdev, DIPROP_DEADZONE, DIJOFS_X, DIPH_BYOFFSET, 5000);
	if( FAILED(hResult) )
	{
		ServeDInputError( IDS_DIERR_SETTING_DEADZONE, hResult, FALSE );
		IDirectInputDevice_Release( pdev );
		return FALSE;
	}
	/* Set Y axis dead zone to 50% (to avoid accidental thrust)
	   Units are ten thousandths, so 50% = 5000/10000. */
	hResult = SetDIDwordProperty( pdev, DIPROP_DEADZONE, DIJOFS_Y, DIPH_BYOFFSET, 5000 );
	if( FAILED(hResult) )
	{
		ServeDInputError( IDS_DIERR_SETTING_DEADZONE, hResult, FALSE );
		IDirectInputDevice_Release( pdev );
		return FALSE;
	}
	/* Add it to our list of devices */ 
	hResult = IDirectInputDevice_QueryInterface( pdev, &IID_IDirectInputDevice2, (LPVOID *)&s_alpDevFound[ g_nNumDIDevs ] );
	if( FAILED(hResult) )
	{
		ServeDInputError( IDS_DIERR_ADDING_DEVICE, hResult, FALSE );
		return FALSE;
	}
	strncpy( g_acInputDevNames[ g_nNumDIDevs++ ], lpDevInstance->tszInstanceName, 32 );
	IDirectInputDevice_Release( pdev );

	return DIENUM_CONTINUE;
} /* #OF# InitJoystickInput */

/*========================================================
Function : ReacquireInput
=========================================================*/
/* #FN#
   */
static
BOOL
/* #AS#
   TRUE if succeeded, FALSE if failed */
ReacquireInput( int nStickNum )
{
    /* If we have a current device */
    if( s_alpJoyPorts[ nStickNum ] )
    {
		/* Acquire the device */
		if( FAILED(
			IDirectInputDevice_Acquire( s_alpJoyPorts[ nStickNum ] )) )
			return FALSE;	/* acquisition failed */
    }
    else	/* We don't have a current device */
		return FALSE;

	return TRUE;
} /* #OF# ReacquireInput */

/*========================================================
Function : PickJoystick
=========================================================*/
/* #FN#
   */
static
void
/* #AS#
   Nothing */
PickJoystick( int nStickNum,
			  int nJoystick )
{
	int i;

	for( i = 0; i < MAX_ATARI_JOYPORTS; i++ )
	{
		if( s_alpJoyPorts[ i ] != NULL &&
			(nJoystick >= 0 && nJoystick < MAX_INPUT_DEVICES) &&
			s_alpJoyPorts[ i ] == s_alpDevFound[ nJoystick ] )
		{
			IDirectInputDevice_Unacquire( s_alpJoyPorts[ i ] );
			s_alpJoyPorts[ i ] = NULL;
		}
	}

	g_anSelectedInputDevice[ nStickNum ] = nJoystick;

	if( s_alpJoyPorts[ nStickNum ] )
		IDirectInputDevice_Unacquire( s_alpJoyPorts[ nStickNum ] );
	s_alpJoyPorts[ nStickNum ] = NULL;

	g_bKBJoystick = FALSE;
	for( i = 0; i < MAX_ATARI_JOYPORTS; i++ )
	{
		if( g_anSelectedInputDevice[ i ] < 0 )
			g_bKBJoystick = TRUE;
	}

	/* (nJoystick < 0) means to use the keyboard instead */
	if( nJoystick < 0 )
		return;

	if( nJoystick < MAX_INPUT_DEVICES && s_alpDevFound[ nJoystick ] )
	{
		s_alpJoyPorts[ nStickNum ] = s_alpDevFound[ nJoystick ];
		if( !ReacquireInput( nStickNum ) )
			ServeDInputError( IDS_DIERR_ACQUIRE_INIT, DIERR_NOTACQUIRED, FALSE );
	}
} /* #OF# PickJoystick */

/*========================================================
Function : InitialiseInput
=========================================================*/
/* #FN#
   */
int
/* #AS#
   */
InitialiseInput( void )
{
	char    cValue;
	int     nStick;
	HRESULT	hResult;

	g_nCurrentKey = g_nNewKey = AKEY_NONE;
	g_nCurrentVK  = g_nNewVK  = 0;
	KEYPRESSED    = FALSE;

	ResetInput();

	if( !s_lpDInput )
	{
		ZeroMemory( s_alpDevFound, sizeof(LPDIRECTINPUTDEVICE) * MAX_INPUT_DEVICES );

		/* Create the DirectInput 5.0 interface object */
		hResult = DirectInputCreate( g_hDInput, DIRECTINPUT_VERSION, &s_lpDInput, NULL );
		if( FAILED(hResult) )
		{
			if( hResult == DIERR_OLDDIRECTINPUTVERSION )
			{
				s_lpDInput = NULL;
			}
			else
			{
				ServeDInputError( IDS_DIERR_CREATING_OBJ, hResult, FALSE );
				return 0;
			}
		}
		else
		{
			/* Enumerate the joystick devices */
			hResult = IDirectInput_EnumDevices( s_lpDInput, DIDEVTYPE_JOYSTICK, InitJoystickInput, s_lpDInput, DIEDFL_ATTACHEDONLY );
			if( FAILED(hResult) )
			{
				ServeDInputError( IDS_DIERR_ENUMERATING, hResult, FALSE );
				return 0;
			}
			IDirectInput_Release( s_lpDInput );
		}
	}
	cValue = (char)(g_ulJoystickSelects & 0x000000ff);
	nStick = cValue;
	PickJoystick( 0, nStick );

	nStick = g_ulJoystickSelects & 0x0000ff00;
	nStick >>= 8;
	cValue = (char)nStick;
	nStick = cValue;
	if( nStick == g_anSelectedInputDevice[ 0 ] )
		nStick = NO_JOYSTICK;
	PickJoystick( 1, nStick );

	nStick = g_ulJoystickSelects & 0x00ff0000;
	nStick >>= 16;
	cValue = (char)nStick;
	nStick = cValue;
	if( nStick == g_anSelectedInputDevice[ 0 ] || nStick == g_anSelectedInputDevice[ 1 ] )
		nStick = NO_JOYSTICK;
	PickJoystick( 2, nStick );

	nStick = g_ulJoystickSelects & 0xff000000;
	nStick >>= 24;
	cValue = (char)nStick;
	nStick = cValue;
	if( nStick == g_anSelectedInputDevice[ 0 ] || nStick == g_anSelectedInputDevice[ 1 ] || nStick == g_anSelectedInputDevice[ 2 ] )
		nStick = NO_JOYSTICK;
	PickJoystick( 3, nStick );

	s_nJoyCount = g_nJoyRepeat;

	return 1;
} /* #OF# InitialiseInput */

/*========================================================
Function : ClearInput
=========================================================*/
/* #FN#
   */
void
/* #AS#
   Nothing */
ClearInput( void )
{
	int i;
	/* Make sure the device is unacquired it doesn't harm
	   to unacquire a device that isn't acquired */
	for( i = 0; i < MAX_ATARI_JOYPORTS; i++ )
	{
		if( s_alpJoyPorts[ i ] )
		{
			IDirectInputDevice_Unacquire( s_alpJoyPorts[ i ] );
			s_alpJoyPorts[ i ] = NULL;
		}
	}
	/* Release all the devices we created */
	for( i = 0; i < g_nNumDIDevs; i++ )
	{
		if( s_alpDevFound[ i ] )
		{
			IDirectInputDevice_Release( s_alpDevFound[ i ] );
			s_alpDevFound[ i ] = NULL;
		}
	}
} /* #OF# ClearInput */

/*========================================================
Function : ReadJoystickInput
=========================================================*/
/* #FN#
   */
int
/* #AS#
   */
ReadJoystickInput( int  nStickNum,
				   BOOL bTestButton )
{
	DIJOYSTATE js;
	int        nStickVal = 0;
	HRESULT    hResult;

	if( nStickNum >= MAX_INPUT_DEVICES || !s_alpJoyPorts[ nStickNum ] )
		return STICK_CENTRE;

	/* Poll the joystick to read the current state */
	hResult = IDirectInputDevice2_Poll( s_alpJoyPorts[ nStickNum ] );

	/* Get data from the joystick */
	hResult = IDirectInputDevice_GetDeviceState( s_alpJoyPorts[ nStickNum ], sizeof(DIJOYSTATE), &js );
	if( FAILED(hResult) )
	{
		/* did the read fail because we lost input for some reason?
		   if so, then attempt to reacquire.  If the second acquire
		   fails, then the error from GetDeviceData will be
		   DIERR_NOTACQUIRED, so we won't get stuck an infinite loop. */
		if( hResult == DIERR_INPUTLOST )
		{
			ReacquireInput( nStickNum );
			/* return the fact that we did not read any data */
			return 0;
		}
		ServeDInputError( IDS_DIERR_ACQUIRE, DIERR_NOTACQUIRED, FALSE );
	}

	if( machine == Atari5200 )
	{
		int i;
		for( i = 0; i < 4; i++ )
		{
			if( js.rgbButtons[ i ] & 0x80 )
				s_anTrig[ i ] = 0;
			else
				s_anTrig[ i ] = 1;
		}
	}
	else
	{
		if( !s_bBtn2Flip )
		{
			if( js.rgbButtons[ 1 ] & 0x80 )
			{
				Toggle_FullSpeed();
				s_bBtn2Flip = TRUE;
			}
		}
		else
		{
			if( !(js.rgbButtons[ 1 ] & 0x80) )
				s_bBtn2Flip = FALSE;
		}

		if( !s_bBtn3Flip )
		{
			if( js.rgbButtons[ 2 ] & 0x80 )
			{
				Toggle_Pause();
				s_bBtn3Flip = TRUE;
			}
		}
		else
		{
			if( !(js.rgbButtons[ 2 ] & 0x80) )
				s_bBtn3Flip = FALSE;
		}

		if( !s_bBtn4Flip )
		{
			if( js.rgbButtons[ 3 ] & 0x80 )
			{
				Toggle_SIOPatch();
				s_bBtn4Flip = TRUE;
			}
		}
		else
		{
			if( !(js.rgbButtons[ 3 ] & 0x80) )
				s_bBtn4Flip = FALSE;
		}

		if( bTestButton )
		{
			/* See if the repeat/skip options are being used for holding button down */
			if( g_nJoyRepeat )
			{
				if( js.rgbButtons[ 0 ] & 0x80 )
				{
					/* Count down the number of repeat cycles and return button up for those */
					if( --s_nJoyCount > 0 )
						s_anTrig[ nStickNum ] = 1;

					/* If we've counted the number of skip cycles, start over and send button down */
					if( s_nJoyCount < -g_nJoySkip )
					{
						s_nJoyCount = g_nJoyRepeat;
						s_anTrig[ nStickNum ] = 0;
					}
				}
				else
				{
					/* Otherwise continue skipping, sending button up */
					s_nJoyCount = g_nJoyRepeat;
					s_anTrig[ nStickNum ] = 1;
				}
			}
			else
			{
				/* No special handling, just return whether it is held down */
				if( js.rgbButtons[ 0 ] & 0x80)
					s_anTrig[ nStickNum ] = 0;
				else
					s_anTrig[ nStickNum ] = 1;
			}
		}
	}

	if( js.lX < 0 )
	{
		if( js.lY < 0 )
			nStickVal |= STICK_UL;
		if( js.lY > 0 )
			nStickVal |= STICK_LL;
		if( js.lY == 0 )
			nStickVal |= STICK_LEFT;
	}

	if( js.lX > 0 )
	{
		if( js.lY < 0 )
			nStickVal |= STICK_UR;
		if( js.lY > 0 )
			nStickVal |= STICK_LR;
		if( js.lY == 0 )
			nStickVal |= STICK_RIGHT;
	}

	if( js.lX == 0 )
	{
		if( js.lY < 0 )
			nStickVal |= STICK_FORWARD;

		if( js.lY > 0 )
			nStickVal |= STICK_BACK;

		if( js.lY == 0 )
			nStickVal |= STICK_CENTRE;
	}
	s_anStick[ nStickNum ] = nStickVal;
	return 1;
} /* #OF# ReadJoystickInput */

/*========================================================
Function : Atari_Keyboard
=========================================================*/
/* #FN#
   In other Atari800 versions, keyboard handling is here. In Atari800Win
   I handle the actual key processing in the mainframe of the app and by
   the time it gets here we already have the appropriate Atari key code
   (except for keycodes that require multiple logic operations, like the
   s_anStick) */
int
/* #AS#
   Current Atari key value */
Atari_Keyboard( void )
{
	if( g_nCurrentKey != AKEY_NONE )
	{
		if( GetAsyncKeyState( g_nCurrentVK ) >= 0 )
		{
			KEYPRESSED = FALSE;
			g_nCurrentKey = AKEY_NONE;
		}
	}
	if( g_nNewKey == AKEY_NONE )
		return g_nCurrentKey;

	if( machine == Atari5200 )
	{
		switch( g_nNewKey )
		{
		case AKEY_COLDSTART:
			g_nNewKey = AKEY_COLDSTART;
			break;
			
		case AKEY_WARMSTART:
			g_nNewKey = 0x29;	/* 5200 has no warmstart */	/* reset */
			break;
			
		case AKEY_STARTDWN:
			g_nNewKey = 0x39;	/* start */
			break;
			
		case AKEY_OPTIONDWN:		
			g_nNewKey = 0x31;	/* pause */
			break;
			
		case AKEY_0:
			g_nNewKey = 0x2b;
			break;
			
		case AKEY_1:
			g_nNewKey = 0x0f;
			break;
			
		case AKEY_2:
			g_nNewKey = 0x3f;
			break;
			
		case AKEY_3:
			g_nNewKey = 0x3d;
			break;
			
		case AKEY_4:
			g_nNewKey = 0x3b;
			break;
			
		case AKEY_5:
			g_nNewKey = 0x37;
			break;
			
		case AKEY_6:
			g_nNewKey = 0x35;
			break;
			
		case AKEY_7:
			g_nNewKey = 0x33;
			break;
			
		case AKEY_8:
			g_nNewKey = 0x2f;
			break;
			
		case AKEY_9:
			g_nNewKey = 0x2d;
			break;
			
		case AKEY_MINUS:
			g_nNewKey = 0x23;	/* # key on 5200 controller */
			break;
			
		case AKEY_ASTERISK:		/* * key on 5200 controller */
			g_nNewKey = 0x27;
			break;
			
		default:
			g_nNewKey = AKEY_NONE;
			break;
		}
	}

	if( g_nNewKey & SPECIAL_HANDLER_MASK )
	{
		int	nKeyTemp = g_nNewKey;
		g_nNewKey = AKEY_NONE;

//		if( g_nNewVK >= VK_NUMPAD0 && g_nNewVK <= VK_NUMPAD9 && !g_bKBJoystick )
//			g_nNewKey = nKeyTemp & ~SPECIAL_HANDLER_MASK;

		switch( nKeyTemp )
		{
			case (AKEY_HELP | SPECIAL_HANDLER_MASK):
				if( mach_xlxe )
					g_nNewKey = nKeyTemp & ~SPECIAL_HANDLER_MASK;
				break;
				
			case (AKEY_F1 | SPECIAL_HANDLER_MASK):
			case (AKEY_F2 | SPECIAL_HANDLER_MASK):
			case (AKEY_F3 | SPECIAL_HANDLER_MASK):
			case (AKEY_F4 | SPECIAL_HANDLER_MASK):
//				if( SHIFT_KEY )
//					g_nNewKey |= AKEY_SHFT;
				if( machine == AtariXL )
					g_nNewKey = nKeyTemp & ~SPECIAL_HANDLER_MASK;
				break;
				
			case AKEY_BREAK:
				IRQST &= ~0x80;
				if( IRQEN & 0x80 )
					GenerateIRQ();
				break;

/* We have used accelerators instead of that */

//			case AKEY_MONITOR:
//				LaunchMonitor();
//				break;

//			case AKEY_COLDSTART:
//				Coldstart();
//				break;
				
//			case AKEY_WARMSTART:
//				Warmstart();
//				break;
				
//			case AKEY_SCREENSHOT:
//				Save_PCX((UBYTE *)atari_screen);
//				break;

//			case AKEY_SCREENSHOT_INTERLACE:
//				Save_PCX_interlaced();
//				break;
/**/

/* Unfortunately these are not used in this version */

//			case AKEY_PIL:
//				if( pil_on )
//					DisablePILL();
//				else
//					EnablePILL();
//				break;

//			case AKEY_EXIT:
//				PostMessage( g_hMainWnd, WM_CLOSE, 0, 0L );
//				break;
/**/
			default:
				break;
		}
		if( g_nNewKey == AKEY_NONE )
			g_nNewVK = 0;
	}

	if( g_nNewKey != AKEY_NONE && g_nCurrentKey == AKEY_NONE )
	{
		KEYPRESSED   = TRUE;
		KBCODE       = g_nCurrentKey = g_nNewKey;
		g_nCurrentVK = g_nNewVK;
		g_nNewKey    = AKEY_NONE;
		g_nNewVK     = 0;

		IRQST &= ~0x40;
		if( IRQEN & 0x40 )
			GenerateIRQ();
	}
	return g_nCurrentKey;
} /* #OF# Atari_Keyboard */

/*========================================================
Function : GetJoystickInput
=========================================================*/
/* #FN#
   */
void
/* #AS#
   Nothing */
GetJoystickInput( void )
{
	int i;

	for( i = 0; i < MAX_ATARI_JOYPORTS; i++ )
	{
		if( s_alpJoyPorts[ i ] )
			ReadJoystickInput( i, TRUE );
		else
		{
			if( g_anSelectedInputDevice[ i ] < 0 )
			{
				s_anStick[ i ] = s_aunKeystick[ g_anSelectedInputDevice[ i ] + NUM_KBJOY_DEVICES ];
				s_anTrig [ i ] = s_aunKeytrig [ g_anSelectedInputDevice[ i ] + NUM_KBJOY_DEVICES ];
			}
			else
			{
				s_anStick[ i ] = 0x0f;
				s_anTrig [ i ] = 1;
			}
		}
	}
} /* #OF# GetJoystickInput */

/*========================================================
Function : Atari_PORT
=========================================================*/
/* #FN#
   */
int
/* #AS#
   */
Atari_PORT( int nNum )
{
	if( !nNum )
		return (s_anStick[ 1 ] << 4) | s_anStick[ 0 ];
//		return( (s_anStick[ 1 ] & 0xF) << 4 | (s_anStick[ 0 ] & 0xF) );
	else
		return (s_anStick[ 3 ] << 4) | s_anStick[ 2 ];
} /* #OF# Atari_PORT */

/*========================================================
Function : Atari_TRIG
=========================================================*/
/* #FN#
   */
int
/* #AS#
   */
Atari_TRIG( int nNum )
{
	return s_anTrig[ nNum ];
} /* #OF# Atari_TRIG */

/*========================================================
Function : Atari_POT
=========================================================*/
/* #FN#
   */
int
/* #AS#
   */
Atari_POT( int nNum )
{
	if( machine == Atari5200 )
	{
		if( nNum == 0 )
		{
			unsigned char ucPot0 = 120;
			if( !(s_anStick[ 0 ] & 0x04) )
				ucPot0 = 15;
			if( !(s_anStick[ 0 ] & 0x08) )
				ucPot0 = 220;
			return ucPot0;
		}
		if( nNum == 1)
		{
			unsigned char ucPot1 = 120;
			if( !(s_anStick[ 0 ] & 0x01) )
				ucPot1 = 15;
			if( !(s_anStick[ 0 ] & 0x02) )
				ucPot1 = 220;
			return ucPot1;
		}
	}
 	return 228;
} /* #OF# Atari_POT */

/*========================================================
Function : Atari_CONSOL
=========================================================*/
/* #FN#
   */
int
/* #AS#
   */
Atari_CONSOL( void )
{
  return s_nConsol;
} /* #OF# Atari_CONSOL */

/*========================================================
Function : Atari_Set_CONSOL
=========================================================*/
/* #FN#
   */
static
void
/* #AS#
   Nothing */
Atari_Set_CONSOL( int nNewConsol )
{
	switch( nNewConsol )
	{
		case AKEY_OPTIONDWN:	/* OPTION */
			s_nConsol &= 0x03;
			break;

		case AKEY_SELECTDWN:	/* SELECT */
			s_nConsol &= 0x05;
			break;

		case AKEY_STARTDWN:		/* START */
			s_nConsol &= 0x06;
			break;

		case AKEY_OPTIONUP:
			s_nConsol |= 0x04;
			break;

		case AKEY_SELECTUP:
			s_nConsol |= 0x02;
			break;

		case AKEY_STARTUP:
			s_nConsol |= 0x01;
			break;
	}
} /* #OF# Atari_Set_CONSOL */

/*========================================================
Function : ServeKBJoystickDown
=========================================================*/
/* #FN#
   */
static
BOOL
/* #AS#
   TRUE for ending of key down event handling, FALSE for continue */
ServeKBJoystickDown( WPARAM wp, LPARAM lp )
{
	int nIDevice;
	register int i, j;

	/* Translate extended codes */
	if( lp & 0x1000000/*(1 << 24)*/ )
	{
		wp = g_anExtendedKeyTab[ wp ];
	}
	else if( wp == VK_SHIFT && lp & 0x100000 )
		wp = VK_RSHIFT;

	for( i = 0; i < MAX_ATARI_JOYPORTS; i++ )
	{
		/* Check the "keyboard joystick" */
		if( g_anSelectedInputDevice[ i ] < 0 )
		{
			nIDevice = g_anSelectedInputDevice[ i ] + NUM_KBJOY_DEVICES;
			if( wp == g_anKeyset[ nIDevice ][ KEYSET_FIRE ] )
			{
				s_aunKeytrig[ nIDevice ] = 0;
				return TRUE;
			}
			else
			{
				for( j = 0; j < NUM_KBJOY_KEYS - 1/* The last is KEYSET_FIRE */; j++ ) 
					if( wp == g_anKeyset[ nIDevice ][ j ] )
					{
						s_aunKeystick[ nIDevice ] &= s_anStickTable[ j ];
						return TRUE;
					}
			}
		}
	}
	return FALSE;
} /* #OF# ServeKBJoystickDown */

/*========================================================
Function : ServeKBJoystickUp
=========================================================*/
/* #FN#
   */
static
BOOL
/* #AS#
   TRUE for ending of key up event handling, FALSE for continue */
ServeKBJoystickUp( WPARAM wp, LPARAM lp )
{
	int nIDevice;
	register int i, j;

	/* Translate extended codes */
	if( lp & 0x1000000/*(1 << 24)*/ )
	{
		wp = g_anExtendedKeyTab[ wp ];
	}
	else if( wp == VK_SHIFT && lp & 0x100000 )
		wp = VK_RSHIFT;

	for( i = 0; i < MAX_ATARI_JOYPORTS; i++ )
	{
		/* Check the "keyboard joystick" */
		if( g_anSelectedInputDevice[ i ] < 0 )
		{
			nIDevice = g_anSelectedInputDevice[ i ] + NUM_KBJOY_DEVICES;
			if( wp == g_anKeyset[ nIDevice ][ KEYSET_FIRE ] )
			{
				s_aunKeytrig[ nIDevice ] = 1;
				return TRUE;
			}
			else if( g_ulMiscStates & MS_STICK_RELEASE )
			{
				for( j = 0; j < NUM_KBJOY_KEYS - 1/* The last is KEYSET_FIRE */; j++ ) 
					if( wp == g_anKeyset[ nIDevice ][ j ] )
					{
						/* Turn bits for this key off */
						s_aunKeystick[ nIDevice ] |= (~s_anStickTable[ j ]) & 0x0f;
						return TRUE;
					}
			}
		}
	}
	return FALSE;
} /* #OF# ServeKBJoystickUp */

/*========================================================
Function : ToggleKeyDown
=========================================================*/
/* #FN#
   */
BOOL
/* #AS#
   TRUE for ending of key down event handling, FALSE for continue */
ToggleKeyDown( WPARAM wp, LPARAM lp )
{
	if( g_bKBJoystick && ServeKBJoystickDown( wp, lp ) )
		return TRUE; /* End of key down event handling */

	switch( wp )
	{
		case VK_SHIFT:
			SHIFT_KEY = TRUE;
			return TRUE;
			break;

		case VK_CONTROL:
			CTRL_KEY = TRUE;
			return TRUE;
			break;

		case VK_F2:
			if( !SHIFT_KEY )	/* Console keys only work in non-shift mode */
				Atari_Set_CONSOL( AKEY_OPTIONDWN );
			break;

		case VK_F3:
			if( !SHIFT_KEY )	/* Console keys only work in non-shift mode */
				Atari_Set_CONSOL( AKEY_SELECTDWN );
			break;

		case VK_F4:
			if( !SHIFT_KEY )	/* Console keys only work in non-shift mode */
				Atari_Set_CONSOL( AKEY_STARTDWN );
			break;			

/* We have used accelerators instead of that */

//		case VK_F7:
//			Toggle_FullSpeed();
//			return TRUE;
//			break;
			
//		case VK_F9:
//			Toggle_Pause();
//			return TRUE;
//			break;
			
//		case VK_F11:
//			Toggle_SIOPatch();
//			return TRUE;
//			break;
/**/
	}
	return FALSE;
} /* #OF# ToggleKeyDown */

/*========================================================
Function : ToggleKeyUp
=========================================================*/
/* #FN#
   */
BOOL
/* #AS#
   TRUE for ending of key up event handling, FALSE for continue */
ToggleKeyUp( WPARAM wp, LPARAM lp )
{
	if( g_bKBJoystick && ServeKBJoystickUp( wp, lp ) )
		return TRUE; /* End of key up event handling */

	switch( wp )
	{
		case VK_SHIFT:
			SHIFT_KEY = FALSE;
			return TRUE;
			break;

		case VK_CONTROL:
			CTRL_KEY = FALSE;
			return TRUE;
			break;

		case VK_F2:
			if( !SHIFT_KEY )	/* Console keys only work in non-shift mode */
				Atari_Set_CONSOL( AKEY_OPTIONUP );
			break;

		case VK_F3:
			if( !SHIFT_KEY )	/* Console keys only work in non-shift mode */
				Atari_Set_CONSOL( AKEY_SELECTUP );
			break;

		case VK_F4:
			if( !SHIFT_KEY )	/* Console keys only work in non-shift mode */
				Atari_Set_CONSOL( AKEY_STARTUP );
			break;
	}
	return FALSE;
} /* #OF# ToggleKeyUp */

/*========================================================
Function : GetDIErrorString
=========================================================*/
/* #FN#
   Outputs a debug string to debugger */
static
BOOL
/* #AS#
   TRUE if succeeded, FALSE if failed */
GetDIErrorString( HRESULT hResult, LPSTR lpszErrorBuff, DWORD cchError )
{
    DWORD cLen;
    LPSTR lpszError;
    TCHAR szMsg[ 256 ];

    /* Check parameters */
    if( !lpszErrorBuff || !cchError )
    {
        /* Error, invalid parameters */
        return FALSE;
    }

    switch( hResult )
    {
		case DI_OK:
			/* The operation completed successfully. This value is equal to the S_OK standard COM return value. */
			lpszError = TEXT("DI_OK");
			break;

#ifdef WIN_TRANSLATE_ERRORS

		case DI_BUFFEROVERFLOW:
			//The device buffer overflowed and some input was lost. This value is equal to the S_FALSE standard COM return value. 
			lpszError = TEXT("DI_BUFFEROVERFLOW");
			break;

		case DI_DOWNLOADSKIPPED:
			//The parameters of the effect were successfully updated, but the effect could not be downloaded because the associated device was not acquired in exclusive mode. 
			lpszError = TEXT("DI_DOWNLOADSKIPPED");
			break;

		case DI_EFFECTRESTARTED:
			//The effect was stopped, the parameters were updated, and the effect was restarted. 
			lpszError = TEXT("DI_EFFECTRESTARTED");
			break;

//		case DI_NOEFFECT:
			//The operation had no effect. This value is equal to the S_FALSE standard COM return value. 
//			lpszError = TEXT("DI_NOEFFECT");
//			break;

//		case DI_NOTATTACHED:
			//The device exists but is not currently attached. This value is equal to the S_FALSE standard COM return value. 
//			lpszError = TEXT("DI_NOTATTACHED");
//			break;
	
		case DI_POLLEDDEVICE:
			//The device is a polled device. As a result, device buffering will not collect any data and event notifications will not be signaled until the IDirectInputDevice2::Poll method is called. 
			lpszError = TEXT("DI_POLLEDDEVICE");
			break;

//		case DI_PROPNOEFFECT:
			//The change in device properties had no effect. This value is equal to the S_FALSE standard COM return value. 
//			lpszError = TEXT("DI_PROPNOEFFECT");
//			break;

		case DI_TRUNCATED:
			//The parameters of the effect were successfully updated, but some of them were beyond the capabilities of the device and were truncated to the nearest supported value. 
			lpszError = TEXT("DI_TRUNCATED");
			break;

		case DI_TRUNCATEDANDRESTARTED:
			//Equal to DI_EFFECTRESTARTED | DI_TRUNCATED.
			lpszError = TEXT("DI_TRUNCATEDANDRESTARTED");
			break;

		case DIERR_ACQUIRED:
			//The operation cannot be performed while the device is acquired. 
			lpszError = TEXT("DIERR_ACQUIRED");
			break;

		case DIERR_ALREADYINITIALIZED:
			//This object is already initialized 
			lpszError = TEXT("DIERR_ALREADYINITIALIZED");
			break;

		case DIERR_BADDRIVERVER:
			//The object could not be created due to an incompatible driver version or mismatched or incomplete driver components. 
			lpszError = TEXT("DIERR_BADDRIVERVER");
			break;

		case DIERR_BETADIRECTINPUTVERSION:
			//The application was written for an unsupported prerelease version of DirectInput. 
			lpszError = TEXT("DIERR_BETADIRECTINPUTVERSION");
			break;

		case DIERR_DEVICEFULL:
			//The device is full. 
			lpszError = TEXT("DIERR_DEVICEFULL");
			break;

		case DIERR_DEVICENOTREG:
			//The device or device instance is not registered with DirectInput. This value is equal to the REGDB_E_CLASSNOTREG standard COM return value. 
			lpszError = TEXT("DIERR_DEVICENOTREG");
			break;

		case DIERR_EFFECTPLAYING:
			//The parameters were updated in memory but were not downloaded to the device because the device does not support updating an effect while it is still playing. 
			lpszError = TEXT("DIERR_EFFECTPLAYING");
			break;

		case DIERR_HASEFFECTS:
			//The device cannot be reinitialized because there are still effects attached to it.
			lpszError = TEXT("DIERR_HASEFFECTS");
			break;

		case DIERR_GENERIC:
			//An undetermined error occurred inside the DirectInput subsystem. This value is equal to the E_FAIL standard COM return value. 
			lpszError = TEXT("DIERR_GENERIC");
			break;

		case DIERR_HANDLEEXISTS:
			//The device already has an event notification associated with it. This value is equal to the E_ACCESSDENIED standard COM return value. 
			lpszError = TEXT("DIERR_HANDLEEXISTS");
			break;

		case DIERR_INCOMPLETEEFFECT:
			//The effect could not be downloaded because essential information is missing. For example, no axes have been associated with the effect, or no type-specific information has been supplied.
			lpszError = TEXT("DIERR_INCOMPLETEEFFECT");
			break;

		case DIERR_INPUTLOST:
			//Access to the input device has been lost. It must be reacquired. 
			lpszError = TEXT("DIERR_INPUTLOST");
			break;

		case DIERR_INVALIDPARAM:
			//An invalid parameter was passed to the returning function, or the object was not in a state that permitted the function to be called. This value is equal to the E_INVALIDARG standard COM return value. 
			lpszError = TEXT("DIERR_INVALIDPARAM");
			break;

		case DIERR_MOREDATA:
			//Not all the requested information fitted into the buffer.
			lpszError = TEXT("DIERR_MOREDATA");
			break;

		case DIERR_NOAGGREGATION:
			//This object does not support aggregation. 
			lpszError = TEXT("DIERR_NOAGGREGATION");
			break;

		case DIERR_NOINTERFACE:
			//The specified interface is not supported by the object. This value is equal to the E_NOINTERFACE standard COM return value. 
			lpszError = TEXT("DIERR_NOINTERFACE");
			break;

		case DIERR_NOTACQUIRED:
			//The operation cannot be performed unless the device is acquired. 
			lpszError = TEXT("DIERR_NOTACQUIRED");
			break;

		case DIERR_NOTBUFFERED:
			//The device is not buffered. Set the DIPROP_BUFFERSIZE property to enable buffering.
			lpszError = TEXT("DIERR_NOTBUFFERED");
			break;

		case DIERR_NOTDOWNLOADED:
			//The effect is not downloaded.
			lpszError = TEXT("DIERR_NOTDOWNLOADED");
			break;

		case DIERR_NOTEXCLUSIVEACQUIRED:
			//The operation cannot be performed unless the device is acquired in DISCL_EXCLUSIVE mode. 
			lpszError = TEXT("DIERR_NOTEXCLUSIVEACQUIRED");
			break;

		case DIERR_NOTFOUND:
			//The requested object does not exist. 
			lpszError = TEXT("DIERR_NOTFOUND");
			break;

		case DIERR_NOTINITIALIZED:
			//This object has not been initialized. 
			lpszError = TEXT("DIERR_NOTINITIALIZED");
			break;

//		case DIERR_OBJECTNOTFOUND:
			//The requested object does not exist. 
//			lpszError = TEXT("DIERR_OBJECTNOTFOUND");
//			break;

		case DIERR_OLDDIRECTINPUTVERSION:
			//The application requires a newer version of DirectInput. 
			lpszError = TEXT("DIERR_OLDDIRECTINPUTVERSION");
			break;

//		case DIERR_OTHERAPPHASPRIO:
			//Another application has a higher priority level, preventing this call from succeeding. This value is equal to the E_ACCESSDENIED standard COM return value. This error can be returned when an application has only foreground access to a device but is attempting to acquire the device while in the background.
//			lpszError = TEXT("DIERR_OTHERAPPHASPRIO");
//			break;

		case DIERR_OUTOFMEMORY:
			//The DirectInput subsystem couldn't allocate sufficient memory to complete the call. This value is equal to the E_OUTOFMEMORY standard COM return value. 
			lpszError = TEXT("DIERR_OUTOFMEMORY");
			break;

//		case DIERR_READONLY:
//			//The specified property cannot be changed. This value is equal to the E_ACCESSDENIED standard COM return value. 
//			lpszError = TEXT("DIERR_READONLY");
//			break;

		case DIERR_UNSUPPORTED:
			//The function called is not supported at this time. This value is equal to the E_NOTIMPL standard COM return value. 
			lpszError = TEXT("DIERR_UNSUPPORTED");
			break;

		case E_PENDING:
			//Data is not yet available.
			lpszError = TEXT("E_PENDING");
			break;

#endif /*WIN_TRANSLATE_ERRORS*/

		default:
			// Unknown DI Error
			wsprintf (szMsg, "Error #%ld", (DWORD)(hResult & 0x0000FFFFL) );
			//wsprintf (szMsg, "Unknown Error #%ld", (DWORD)(hResult & 0x0000FFFFL));
			lpszError = szMsg;
			break;
	}
    // Copy DS Error string to buff
    cLen = strlen( lpszError );
    if( cLen >= cchError )
    {
        cLen = cchError - 1;
    }
	if( cLen )
    {
        strncpy( lpszErrorBuff, lpszError, cLen );
        lpszErrorBuff[ cLen ] = 0;
    }
    return TRUE;
} /* #OF# GetDIErrorString */
