/****************************************************************************
File    : misc_win.c
/*
@(#) #SY# Atari800Win
@(#) #IS# Miscellanous stuff implementation for Win32 platforms
@(#) #BY# Richard Lawrence, Tomasz Szymankowski
@(#) #LM# 05.07.2000
*/

/*
Copyright (c) 1998 Richard Lawrence

This program is free software; you can redistribute it and/or modify it under the terms 
of the GNU General Public License as published by the Free Software Foundation; either 
version 2 of the License, or (at your option) any later version. This program is 
distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even
the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details. You should have received a copy of the GNU
General Public License along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include <windows.h>
#include <stdlib.h>
#include <crtdbg.h>
#include <stdio.h>
#include <io.h>
#include <fcntl.h>
#include <mmsystem.h>
#include "WinConfig.h"
#include "Resource.h"
#include "atari800.h"
#include "globals.h"
#include "crc.h"
#include "zlib.h"
#include "registry.h"
#include "display_win.h"
#include "sound_win.h"
#include "input_win.h"
#include "misc_win.h"

/* Constants definition */

#define ROM_TYPES_NO		4

#define FCT_CHECK_16STD		0x01
#define FCT_CHECK_16OSS		0x02
#define FCT_CHECK_16ROM		(FCT_CHECK_16STD | FCT_CHECK_16OSS)
#define FCT_CHECK_32AGS		0x04
#define FCT_CHECK_32DB		0x08
#define FCT_CHECK_32ROM		(FCT_CHECK_32AGS | FCT_CHECK_32DB)

/* Public objects */

char g_szSystemText[ 64 ];

FARPROC	pgzread  = NULL;
FARPROC	pgzopen  = NULL;
FARPROC	pgzclose = NULL;
FARPROC	pgzwrite = NULL;
FARPROC	pgzerror = NULL;

/* Private objects */

static BOOL   s_bCheckZlib  = TRUE;
static HANDLE s_hZlib       = NULL;
static UINT   s_unCartCheck = FCT_CHECK_16ROM;

#ifdef WIN_USE_PERF_COUNTER
static LARGE_INTEGER s_lnTimeFreq;
#endif /*WIN_USE_PERF_COUNTER*/


int
ReadDisabledROMs( void )
{
	int	nRomFile;

	nRomFile = _open( atari_basic_filename, O_BINARY | O_RDONLY, 0777 );
	if( nRomFile == -1 )
	{
		Aprint( "Could not open %s for reading.", atari_basic_filename );
		return FALSE;
	}

	if( _read( nRomFile, atari_basic, 8192 ) < 8192 )
	{
		Aprint( "Could not read all of atari basic from %s.", atari_basic_filename );
		return FALSE;
	}
	_close( nRomFile );

	nRomFile = _open( atari_xlxe_filename, O_BINARY | O_RDONLY, 0777 );
	if( nRomFile == -1 )
	{
		Aprint( "Could not open %s for reading.", atari_xlxe_filename );
		return FALSE;
	}

	if( _read( nRomFile, atari_xlxe_filename, 16384 ) < 16384 )
	{
		Aprint( "Could not read entire atari ROM from %s.", atari_xlxe_filename );
		return FALSE;
	}
	_close( nRomFile );
	return TRUE;
}

int
prepend_tmpfile_path( char *pszBuffer )
{
	int	nOffset = 0;

	nOffset += GetTempPath( MAX_PATH, pszBuffer ) - 1;
	if( nOffset == -1 || pszBuffer[ nOffset ] != '\\' )
		nOffset++;

	return nOffset;
}

int
zlib_capable( void )
{
	if( !s_hZlib )
	{
		Aprint( "Cannot gzip/ungzip without gzip.dll loaded properly." );
		return -1;
	}
	return 1;
}

int
Atari_PEN( int vertical )
{
	return vertical ? 0xff : 0;
}

void
StartAtariTimer( void )
{
#ifdef WIN_USE_PERF_COUNTER
	LARGE_INTEGER	lnCurrentTime;

	QueryPerformanceCounter( &lnCurrentTime );
	g_ulAtariHWNexttime = lnCurrentTime.LowPart + g_ulDeltaT;
#else /*WIN_USE_PERF_COUNTER*/
	g_ulAtariHWNexttime = timeGetTime() + g_ulDeltaT;
#endif /*WIN_USE_PERF_COUNTER*/
}

void
Atari_Initialise( int *argc, char *argv[] )
{
	memset( g_szSystemText, 0, 64 );
	if( s_bCheckZlib )
	{
		s_bCheckZlib = FALSE;
		if( !s_hZlib )
		{
			s_hZlib = LoadLibrary( "ZLIB.DLL" );
			if( !s_hZlib )
				MessageBox( g_hMainWnd, "Could not load ZLIB.DLL. No decompression of gzip files will be possible.", "Atari800Win", MB_ICONEXCLAMATION | MB_OK );
			else
			{
				pgzread  = GetProcAddress( s_hZlib, "gzread" );
				pgzopen  = GetProcAddress( s_hZlib, "gzopen" );
				pgzclose = GetProcAddress( s_hZlib, "gzclose" );
				pgzwrite = GetProcAddress( s_hZlib, "gzwrite" );
				pgzerror = GetProcAddress( s_hZlib, "gzerror" );
				if( !pgzread || !pgzopen || !pgzclose || !pgzwrite || !pgzerror)
				{
					FreeLibrary( s_hZlib );
					s_hZlib = NULL;
					MessageBox( g_hMainWnd, "Could not obtain necessary functions from ZLIB.DLL. No decompression of gzip files will be possible.", "Atari800Win", MB_ICONEXCLAMATION | MB_OK );
				}
			}
		}
	}

#ifdef WIN_USE_PERF_COUNTER
	memset( &s_lnTimeFreq, 0, sizeof(s_lnTimeFreq) );
	memset( &g_ulAtariHWNexttime, 0, sizeof(g_ulAtariHWNexttime) );
#endif /*WIN_USE_PERF_COUNTER*/

	if( tv_mode == TV_PAL )
	{
#ifdef WIN_USE_PERF_COUNTER
		QueryPerformanceFrequency( &s_lnTimeFreq );
		g_ulDeltaT = s_lnTimeFreq.LowPart / 50;
		g_nSleepThreshold  = s_lnTimeFreq.LowPart / 1000L;
		g_nSleepThreshold *= (SLEEP_TIME_IN_MS + 1);
#else /*WIN_USE_PERF_COUNTER*/
		g_ulDeltaT = 20;
		g_nSleepThreshold  = SLEEP_TIME_IN_MS + 1;
#endif /*WIN_USE_PERF_COUNTER*/
	}
	else
	{
#ifdef WIN_USE_PERF_COUNTER
		QueryPerformanceFrequency( &s_lnTimeFreq );
		g_ulDeltaT = s_lnTimeFreq.LowPart / 60;
		g_nSleepThreshold  = s_lnTimeFreq.LowPart / 1000L;
		g_nSleepThreshold *= (SLEEP_TIME_IN_MS + 1);
#else /*WIN_USE_PERF_COUNTER*/
		g_ulDeltaT = 17;
		g_nSleepThreshold  = SLEEP_TIME_IN_MS + 1;
#endif /*WIN_USE_PERF_COUNTER*/
	}

	if( !g_ulModesAvail )
		CheckDDrawModes();

	if( !g_lpbmi )
	{
		/* Only initialize this the first time it's allocated, since we'll come through here
		   on changing hardware types and it will already be set up correctly at that point */
		g_lpbmi = (LPBITMAPINFO)calloc( 1, (sizeof( BITMAPINFOHEADER ) + sizeof( RGBQUAD ) * 256) );

		if( !g_lpbmi )
			Atari_Exit( 1 );

		g_lpbmi->bmiHeader.biSize        = sizeof(BITMAPINFOHEADER);
		g_lpbmi->bmiHeader.biWidth       = ATARI_VIS_WIDTH;
		g_lpbmi->bmiHeader.biHeight      = -ATARI_HEIGHT;	/* Negative because we are a top-down bitmap */
		g_lpbmi->bmiHeader.biPlanes      = 1;
		g_lpbmi->bmiHeader.biBitCount    = 8;				/* Each byte stands for a color value */
		g_lpbmi->bmiHeader.biCompression = BI_RGB;			/* Uncompressed format */

		UseAtariPalette();
	}
	ResetInput();
}

void
UseAtariPalette( void )
{
	int i;

	_ASSERT(g_lpbmi);

	for( i = 0; i < 256; i++ )
	{
		int	rgb = colortable[ i ];
		
		g_lpbmi->bmiColors[ i ].rgbRed      = g_pe[ i ].peRed   = (rgb & 0x00ff0000) >> 16;
		g_lpbmi->bmiColors[ i ].rgbGreen    = g_pe[ i ].peGreen = (rgb & 0x0000ff00) >> 8;
		g_lpbmi->bmiColors[ i ].rgbBlue     = g_pe[ i ].peBlue  = rgb & 0x000000ff;
		g_lpbmi->bmiColors[ i ].rgbReserved = g_pe[ i ].peFlags = 0;
	}
}

void
Toggle_Pause( void )
{
	if( g_ulAtariState & ATARI_RUNNING )
	{
		if( g_ulAtariState & ATARI_PAUSED )
		{
			g_ulAtariState &= ~ATARI_PAUSED;
			RestartSound();
 		}
		else
		{
			g_ulAtariState |= ATARI_PAUSED;
			ClearSound( FALSE );
			if( g_hMainWnd )
				SetTimer( g_hMainWnd, TIMER_READ_JOYSTICK, 100, NULL ); 
			DrawPausedScreen( (UBYTE *)atari_screen );
		}
		DescribeAtariSystem();
	}
}

void
Toggle_FullSpeed( void )
{
	if( g_ulAtariState & ATARI_RUNNING )
	{
		if( g_ulMiscStates & MS_FULL_SPEED )
		{
			g_ulMiscStates &= ~MS_FULL_SPEED;
			RestartSound();
		}
		else
		{
			g_ulMiscStates |= MS_FULL_SPEED;
			ClearSound( FALSE );
		}
		/* This will be written with OnDestroy method */
//		WriteRegDWORD( NULL, REG_MISC_STATES, g_ulMiscStates );
		DescribeAtariSystem();
	}
}

void
Toggle_SIOPatch( void )
{
	if( enable_rom_patches )
	{
		if( enable_sio_patch )
		{
			enable_sio_patch = 0;
			RestoreSIO();
		}
		else
		{
			enable_sio_patch = 1;
			SetSIOEsc();
		}
		/* This will be written with OnDestroy method */
//		WriteRegDWORD( NULL, REG_ENABLE_SIO_PATCH, enable_sio_patch );
		DescribeAtariSystem();
	}
}

int
LaunchMonitor( void )
{
	/* Get us back to last used windowed display and stop making noises */
	SetSafeDisplay( FALSE );
	/* CAtari800WinView::OnKillFocus will invoke that */
//	ClearSound( FALSE );

	if( monitor() )	/* Launch monitor */
	{
//		RestartSound();
		return 1;	/* Go back to emulation */
	}
	else
	{
		/* Start a quit (this will end up in Atari_Exit(0)) */
		PostMessage( g_hMainWnd, WM_CLOSE, 0, 0L );
	}
	return 0;
}

int
Atari_Exit( int nPanic )
{
	if( nPanic && g_ulAtariState & ATARI_RUNNING )
		g_ulAtariState |= ATARI_CRASHED;
	
	if( ATARI_CRASHED == g_ulAtariState )
	{
		wsync_halt = 1;	/* turn off CPU */
		return 0;
	}

#ifdef WIN_USE_FLIP_BUFFER
	/* Get surface for GDI if necessary */
	FlipToGDI();
#endif
	ShowMousePointer();

	if( nPanic && g_ulAtariState & ATARI_RUNNING )
	{
		char szMessage[ LOADSTRING_STRING_SIZE ];
		int  nResult;

		LoadString( NULL, IDS_WARNING_ATARI_PANIC, szMessage, LOADSTRING_STRING_SIZE );
		nResult = MessageBox( g_hMainWnd, szMessage, "Atari800Win", MB_ICONSTOP | MB_YESNO );

		InvalidateRect( g_hMainWnd, NULL, FALSE );

		if( IDYES == nResult && LaunchMonitor() /* Launch the monitor */ )
		{
			g_ulAtariState &= ~ATARI_CRASHED; /* For the CIM service */
			return 1;
		}
		g_ulAtariState = ATARI_CRASHED;
		wsync_halt = 1;	/* turn off CPU */
		g_nTestVal = 32767;
	}
	else
	{
		/* Reset everything DirectDraw */
		ClearScreen( TRUE );

		/* Reset everything MultiMedia/DirectSound */
		ClearSound( TRUE );

		/* Reset everything DirectInput */
		ClearInput();

		g_ulAtariState = ATARI_UNINITIALIZED | ATARI_CLOSING;

		Remove_ROM();

		if( g_lpbmi )
			free( g_lpbmi );
		g_lpbmi = NULL;

		Clear_Temp_Files();

		if( s_hZlib )
		{
			FreeLibrary( s_hZlib );
   			pgzopen = pgzclose = pgzread = pgzwrite = pgzerror = NULL;
		}
	}
	return 0;
}

/*========================================================
Function : ShowMousePointer
=========================================================*/
/* #FN#
   Makes mouse pointer visible */
void
/* #AS#
   Nothing */
ShowMousePointer( void )
{
	if( g_ulScreenMode & SM_MODE_FULL && g_ulScreenMode & SM_OPTN_HIDE_CURSOR )
	{
		if( g_nShowCursor <= 0 )
			ShowCursor( TRUE );

		g_nShowCursor = 150;
	}
} /* #OF# ShowMousePointer */

/*========================================================
Function : ShowMousePointer
=========================================================*/
/* #FN#
   Prints info about emulated system in main window */
void
/* #AS#
   Nothing */
DescribeAtariSystem( void )
{
	if( !g_hMainWnd )
		return;

	if( g_ulScreenMode & SM_MODE_FULL )
		strcpy( g_szSystemText, "  " );
	else
		strcpy( g_szSystemText, "Atari800Win: " );

	if( g_ulAtariState & ATARI_LOAD_FAILED )
	{
		strcat( g_szSystemText, "Initialization failure" );
		SetWindowText( g_hMainWnd, g_szSystemText );
		return;
	}
	if( g_ulAtariState & ATARI_RUNNING )
	{
		if( g_ulAtariState & (ATARI_PAUSED | ATARI_NOFOCUS) )
		{
			if( g_ulAtariState & ATARI_NOFOCUS && !(g_ulAtariState & ATARI_PAUSED) )
				strcat( g_szSystemText, "Stopped" );
			if( g_ulAtariState & ATARI_PAUSED )
				strcat( g_szSystemText, "Paused (F9 to continue)" );
		}
		else
		{
			/* What about the emulated system type? */
			switch( default_system )
			{
				case 1:
					strcat( g_szSystemText, "800 OS-A " );
					break;
				case 2:
					strcat( g_szSystemText, "800 OS-B " );
					break;
				case 3:
					strcat( g_szSystemText, "800XL " );
					break;
				case 4:
					strcat( g_szSystemText, "130XE " );
					break;
				case 5:
					strcat( g_szSystemText, "320XE " );
					if( 1 == Ram256 )
						strcat( g_szSystemText, "(Rambo) " );
					else
						strcat( g_szSystemText, "(Compy) " );
					break;
				case 6:
					strcat( g_szSystemText, "5200 " );
					break;
				default:
					strcat( g_szSystemText, "Unknown system! " );
					break;
			}
			/* What about the emulated TV mode? */
			switch( default_tv_mode )
			{
				case 1:
					if( g_ulMiscStates & MS_FULL_SPEED )
						strcat( g_szSystemText, "PAL FULL speed" );
					else
						strcat( g_szSystemText, "PAL 1.77MHz" );
					break;
					
				case 2:
					if( g_ulMiscStates & MS_FULL_SPEED )
						strcat( g_szSystemText, "NTSC FULL speed" );
					else
						strcat( g_szSystemText, "NTSC 1.79MHz" );
					break;
					
				default:
					strcat( g_szSystemText, "Unknown TV type!" );
					break;
			}
			/* What about the SIO patch? */
			if( enable_sio_patch )
				strcat( g_szSystemText, " SIO" );
		}
	}
	else
		strcat( g_szSystemText, "not running" );

	if( g_ulScreenMode & SM_MODE_WIND )
		SetWindowText( g_hMainWnd, g_szSystemText );
} /* #OF# DescribeAtariSystem */


/*========================================================
Function : GetHomeDirectory
=========================================================*/
/* #FN#
   Retrieves the path for the Atari800Win home directory */
BOOL
/* #AS#
   TRUE if the path was retrieved, otherwise FALSE */
GetHomeDirectory( LPSTR pszHomeDir )
{
	BOOL bResult = FALSE;

	if( GetModuleFileName( NULL, pszHomeDir, MAX_PATH ) )
	{
		size_t i;
		if( i = strlen( pszHomeDir ) )
		{
			while( i && pszHomeDir[ i ] != '\\' )
				i--;
			if( i )
				pszHomeDir[ i ] = '\0';
			bResult = TRUE;
		}
	}
	return bResult;
} /* #OF# GetHomeDirectory */

/*========================================================
Function : TestAndSetPath
=========================================================*/
/* #FN#
   Searches the Atari system ROMs in home folder and its subfolders */
static
BOOL
/* #AS#
   TRUE if the ROM files has been found, otherwise FALSE */
TestAndSetPath( LPSTR pszFileName,
                ULONG ulExpectedCRC,
                LPSTR pszPath )
{
	char   szStartPath[ MAX_PATH ];
	char   szImagePath[ MAX_PATH ];
	char   szPattern  [ MAX_PATH ];
	ULONG  ulCRC;
	int    nResult;

	WIN32_FIND_DATA fileData;
	HANDLE hFile;

	if( NULL == pszPath )
	{
		/* We begin searching at Atari800Win home directory */
		GetHomeDirectory( szStartPath );
	}
	else
		strcpy( szStartPath, pszPath );

	strcpy( szPattern, szStartPath );
	strcat( szPattern, "\\*.rom" );

	hFile = FindFirstFile( szPattern, &fileData );
	/* If any file was found... */
	while( INVALID_HANDLE_VALUE != hFile )
	{
		if( !(fileData.dwFileAttributes & (FILE_ATTRIBUTE_HIDDEN | FILE_ATTRIBUTE_SYSTEM | FILE_ATTRIBUTE_DIRECTORY)) )
		{
			strcpy( szImagePath, szStartPath );
			strcat( szImagePath, "\\" );
			strcat( szImagePath, fileData.cFileName );

			ulCRC = CheckFile( szImagePath, &nResult );
			if( ulCRC != 0 && ulCRC == ulExpectedCRC )
			{
				/* Again, strncpy gives the strange effects here */
				_ASSERT(strlen( szImagePath ) < MAX_PATH);
				strcpy( pszFileName, szImagePath );
				return TRUE;
			}
		}
		/* Try to find a next file */
		if( !FindNextFile( hFile, &fileData ) )//&& (GetLastError() == ERROR_NO_MORE_FILES) )
		{
			FindClose( hFile );
			hFile = INVALID_HANDLE_VALUE; /* Ending the loop */
		}
	}

	strcpy( szPattern, szStartPath );
	strcat( szPattern, "\\*.*" );

	hFile = FindFirstFile( szPattern, &fileData );
	/* If any folder was found... */
	while( INVALID_HANDLE_VALUE != hFile )
	{
		if( !(fileData.dwFileAttributes & (FILE_ATTRIBUTE_HIDDEN | FILE_ATTRIBUTE_SYSTEM)) &&
			 (fileData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) && (*fileData.cFileName != '.') )
		{
			strcpy( szImagePath, szStartPath );
			strcat( szImagePath, "\\" );
			strcat( szImagePath, fileData.cFileName );

			/* Search the ROM file in this subfolder */
			if( TestAndSetPath( pszFileName, ulExpectedCRC, szImagePath ) )
				return TRUE;
		}
		/* Try to find a next folder */
		if( !FindNextFile( hFile, &fileData ) )//&& (GetLastError() == ERROR_NO_MORE_FILES) )
		{
			FindClose( hFile );
			hFile = INVALID_HANDLE_VALUE; /* Ending the loop */
		}
	}
	return FALSE;
} /* #OF# TestAndSetPath */

/*========================================================
Function : TestRomPaths
=========================================================*/
/* #FN#
   Routine tries to locate the system ROMs in either the home directory
   or a subdirectory "ROMS" under the home directory */
BOOL
/* #AS#
   TRUE if at least one ROM file was found, otherwise FALSE */
TestRomPaths( LPSTR pszStartPath )
{
	BOOL bFoundRom = FALSE;

	if( TestAndSetPath( atari_osa_filename,   2370568491L, pszStartPath ) )
		bFoundRom = TRUE;
	if( TestAndSetPath( atari_osb_filename,   3252116993L, pszStartPath ) )
		bFoundRom = TRUE;
	if( TestAndSetPath( atari_xlxe_filename,  3764596111L, pszStartPath ) )
		bFoundRom = TRUE;
	if( TestAndSetPath( atari_5200_filename,  3182898204L, pszStartPath ) )
		bFoundRom = TRUE;
	if( TestAndSetPath( atari_basic_filename, 2190982779L, pszStartPath ) )
		bFoundRom = TRUE;

	return bFoundRom;
} /* #OF# TestRomPaths */

static
int
CALLBACK
RomTypeDlgProc( HWND hDialog,
				UINT uiMsg,
				UINT wParam,
				LONG lParam )
{
	UINT aunRadioIDs [ ROM_TYPES_NO ] = { IDC_ROMTYPE_16STD, IDC_ROMTYPE_16OSS, IDC_ROMTYPE_32AGS, IDC_ROMTYPE_32DB };
	UINT aunCartTypes[ ROM_TYPES_NO ] = { NORMAL16_CART, OSS_SUPERCART, AGS32_CART, DB_SUPERCART };
	char szCartDesc  [ LOADSTRING_STRING_SIZE ];
	int  i;

	switch( uiMsg )
	{
		case WM_INITDIALOG:
			for( i = 0; i < ROM_TYPES_NO; i++ )
				EnableWindow( GetDlgItem( hDialog, aunRadioIDs[ i ] ), (BOOL)(s_unCartCheck & (1 << i)) );

			CheckRadioButton( hDialog, IDC_ROMTYPE_16STD, IDC_ROMTYPE_32DB,
							  (s_unCartCheck & FCT_CHECK_16ROM ? IDC_ROMTYPE_16STD : IDC_ROMTYPE_32AGS) );
			/* Set description text */
			LoadString( NULL, (s_unCartCheck & FCT_CHECK_16ROM ? IDS_ROM_16STD_DESC : IDS_ROM_32AGS_DESC), szCartDesc, LOADSTRING_STRING_SIZE );
			SetWindowText( GetDlgItem( hDialog, IDC_ROMTYPE_DESCRIPTION ), szCartDesc );
			return TRUE;

		case WM_CLOSE:
			wParam = IDCANCEL;

		case WM_COMMAND:
			switch( wParam )
			{
				case IDC_ROMTYPE_16STD:
					LoadString( NULL, IDS_ROM_16STD_DESC, szCartDesc, LOADSTRING_STRING_SIZE );
					SetWindowText( GetDlgItem( hDialog, IDC_ROMTYPE_DESCRIPTION ), szCartDesc );
					return TRUE;

				case IDC_ROMTYPE_16OSS:
					LoadString( NULL, IDS_ROM_16OSS_DESC, szCartDesc, LOADSTRING_STRING_SIZE );
					SetWindowText( GetDlgItem( hDialog, IDC_ROMTYPE_DESCRIPTION ), szCartDesc );
					return TRUE;

				case IDC_ROMTYPE_32AGS:
					LoadString( NULL, IDS_ROM_32AGS_DESC, szCartDesc, LOADSTRING_STRING_SIZE );
					SetWindowText( GetDlgItem( hDialog, IDC_ROMTYPE_DESCRIPTION ), szCartDesc );
					return TRUE;

				case IDC_ROMTYPE_32DB:
					LoadString( NULL, IDS_ROM_32DB_DESC, szCartDesc, LOADSTRING_STRING_SIZE );
					SetWindowText( GetDlgItem( hDialog, IDC_ROMTYPE_DESCRIPTION ), szCartDesc );
					return TRUE;

				case IDOK:
					for( i = 0; i < ROM_TYPES_NO; i++ )
						if( BST_CHECKED == IsDlgButtonChecked( hDialog, aunRadioIDs[ i ] ) )
							break;
					EndDialog( hDialog, aunCartTypes[ i ] );
					return TRUE;

				case IDCANCEL:
					EndDialog( hDialog, NO_CART );
					return TRUE;
			}
			break;
	}
	return FALSE;
}

/*========================================================
Function : FindCartType
=========================================================*/
/* #FN#
   Checks the pointed cartridge type */
int
/* #AS#
   Cartridge type */
FindCartType( LPCSTR pszFileName, /* #IN# Name of the file to test     */
			  BOOL   bAutoDetect  /* #IN# Don't ask user for cart type */ )
{
	UINT unFileType = IAF_CRT_IMAGE | IAF_ROM_IMAGE;
	int  nCartType  = NO_CART;

	if( !IsAtariFile( pszFileName, &unFileType ) )
		return NO_CART;

	if( IAF_ROM_IMAGE == unFileType )
	{
		int	nFileLen = 0;
		int fd = _open( pszFileName, _O_RDONLY | _O_BINARY, 0 );
		if( -1 == fd )
			return NO_CART;

		nFileLen = _filelength( fd );
		_close( fd );

		if( nFileLen < 8193 )
		{
			nCartType = NORMAL8_CART;
		}
		else
		if( nFileLen < 16385 )
		{
			if( bAutoDetect )
				nCartType = NORMAL16_CART;
			else
			{
				/* Ask the user for 16K cart type */
				s_unCartCheck = FCT_CHECK_16ROM;
				nCartType = DialogBox( g_hDInput, MAKEINTRESOURCE(IDD_ROMTYPE),
									   g_hMainWnd, (DLGPROC)RomTypeDlgProc );
			}
		}
		else
		/* BountyBob cart has got the size greater than 40K */
		/*if( nFileLen < 32769 )*/
		{
			if( bAutoDetect )
				nCartType = AGS32_CART;
			else
			{
				/* Ask the user for 32K cart type */
				s_unCartCheck = FCT_CHECK_32ROM;
				nCartType = DialogBox( g_hDInput, MAKEINTRESOURCE(IDD_ROMTYPE),
									   g_hMainWnd, (DLGPROC)RomTypeDlgProc );
			}
		}
	}
	else
	if( IAF_CRT_IMAGE == unFileType )
		nCartType = CARTRIDGE;

	return nCartType;
} /* #OF# FindCartType */

/*========================================================
Function : IsAtariFile
=========================================================*/
/* #FN#
   Checks if the pointed image is known Atari file */
BOOL
/* #AS#
   TRUE if the image might be an Atari file, otherwise FALSE */
IsAtariFile( LPCSTR  pszFileName, /* #IN# Name of the file to test */
			 UINT   *pFileType    /* #IN/OUT# The file type */ )
{
	char szTempFile[ MAX_PATH ];
	int  fd;

	strcpy( szTempFile, pszFileName );
	strupr(	szTempFile );

	if( IAF_DSK_IMAGE & *pFileType )
	{
		if( strstr( szTempFile, ".GZ"  ) ||
			strstr( szTempFile, ".ATZ" ) ||
			strstr( szTempFile, ".XFZ" ) ||
			strstr( szTempFile, ".DCM" ) ||
			strstr( szTempFile, ".ATR" ) ||
			strstr( szTempFile, ".XFD" ) )
		{
			*pFileType = IAF_DSK_IMAGE;
			return TRUE;
		}
	}
	if( IAF_ROM_IMAGE & *pFileType )
	{
		if( strstr( szTempFile, ".ROM" ) ||
			strstr( szTempFile, ".BIN" ) )
		{
			*pFileType = IAF_ROM_IMAGE;
			return TRUE;
		}
	}
	if( IAF_CRT_IMAGE & *pFileType &&
		(fd = _open( pszFileName, _O_RDONLY | _O_BINARY, 0 )) != -1 )
	{
		char cBuffer[ 4 ];
		int  nBytesRead = 0;

		_lseek( fd, 0L, SEEK_SET );
		nBytesRead = _read( fd, cBuffer, 4 );
		_close( fd );

		if( 4 == nBytesRead &&
			'C' == cBuffer[ 0 ] &&
			'A' == cBuffer[ 1 ] &&
			'R' == cBuffer[ 2 ] &&
			'T' == cBuffer[ 3 ] )
		{
			*pFileType = IAF_CRT_IMAGE;
			return TRUE;
		}
	}
	if( IAF_A8S_IMAGE & *pFileType )
	{
		if( strstr( szTempFile, ".A8S" ) )
		{
			*pFileType = IAF_A8S_IMAGE;
			return TRUE;
		}
	}
	/* That's the last check because some carts have $FFFF header */
	if( IAF_BIN_IMAGE & *pFileType &&
		(fd = _open( pszFileName, _O_RDONLY | _O_BINARY, 0 )) != -1 )
	{
		char cBuffer[ 2 ];
		int  nBytesRead = 0;

		_lseek( fd, 0L, SEEK_SET );
		nBytesRead = _read( fd, cBuffer, 2 );
		_close( fd );

		if( 2 == nBytesRead &&
			0xff == (BYTE)cBuffer[ 0 ] &&
			0xff == (BYTE)cBuffer[ 1 ] )
		{
			*pFileType = IAF_BIN_IMAGE;
			return TRUE;
		}
	}
	*pFileType = 0;

	return FALSE;
} /* #OF# IsAtariFile */

/*========================================================
Function : IsCompressedFile
=========================================================*/
/* #FN#
   Simple routine to check if the given filename is any one of the various
   compressed types. Currently the only test for this is by extension, also
   the only way you can actually read a compressed file in, so that seems
   valid enough. */
BOOL
/* #AS#
   TRUE if the file is compressed, otherwise FALSE */
IsCompressedFile( LPCSTR pszFileName /* #IN# Name of the file to test */ )
{
	char szTempFile[ MAX_PATH ];

	strcpy( szTempFile, pszFileName );
	strupr(	szTempFile );

	if( strstr( szTempFile, ".GZ"  ) ||
		strstr( szTempFile, ".ATZ" ) ||
		strstr( szTempFile, ".XFZ" ) ||
		strstr( szTempFile, ".DCM" ) )
		return TRUE;

	return FALSE;
} /* #OF# IsCompressedFile */

/*========================================================
Function : RunAtariExe
=========================================================*/
/* #FN#
   Loads and executes the pointed Atari 8-bit executable file */
BOOL
/* #AS#
   TRUE if succeeded, otherwise FALSE */
RunAtariExe( LPSTR pszFileName /* #IN# Name of the executable file to run */ )
{
	char szNewDir[ MAX_PATH ];
	BOOL bResult;

	strncpy( szNewDir, pszFileName, MAX_PATH );
	GetFolderPath( szNewDir );
	if( _stricmp( szNewDir, atari_exe_dir ) != 0 )
	{
		strcpy( atari_exe_dir, szNewDir );
		WriteRegString( NULL, REG_EXE_PATH, atari_exe_dir );
	}
#ifdef USE_NEW_BINLOAD
	if( use_new_binload )
	{
		bResult = BIN_loader( pszFileName );
	}
	else
#endif /*USE_NEW_BINLOAD*/
		bResult = ReadAtariExe( pszFileName );

	return bResult;
} /* #OF# RunAtariExe */

/*========================================================
Function : GetFolderPath
=========================================================*/
/* #FN#
   Extracts the folder from full file path */
void
/* #AS#
   Nothing */
GetFolderPath( LPSTR pszFileName /* #IN/OUT# Full file path */ )
{
	int  i;
	for( i = strlen( pszFileName ) - 1; i > 0 && pszFileName[ i ] != '\\'; i-- );
	if( i > 0 )
		pszFileName[ i ] = '\0';

	/* Add ending backslash to drive name */
	if( strlen( pszFileName ) == 2 && pszFileName[ 1 ] == ':' )
		strcat( pszFileName, "\\" );
	
} /* #OF# GetFolderPath */
