/*-----------------------------------------------------------------------------
 * AMY Music Box						    version 1.1
 * (Plays Atari Music Composer files on AMY via the parallel printer port)
 * John Howard Palevich, MRS, ASR		      (Started) August 15, 1983
 *-----------------------------------------------------------------------------
 * This program displays a menu of music files, then plays
 * the user's selections on Amy.
 *-----------------------------------------------------------------------------
 * Source Files:
 * mbox.c	main file -- has menu selection and music performance code
 * gfname.c	has directory searching code
 * parse.c	converts an Atari Music Composer file into an event string
 * jcode.asm	plays an event string, using a 72hz interupt handler
 *-----------------------------------------------------------------------------
 * Batch files:
 * cmbox.bat	compile mbox.c
 * cgfname.bat	compile gfname.c
 * cparse.bat	compile parse.c
 * lmbox.bat	link all the object files together into mbox.exe, and run mbox
 *-----------------------------------------------------------------------------
 * How to create this file from scratch:
 * Have lc1, lc2, link, masm, etc. on A:, have all the source and batch files on
 * B:, then type:
 * B> cmbox
 * B> cgfname
 * B> cparse
 * B> a:masm jcode;
 * B> lmbox
 *----------------------------------------------------------------------------
 *		An explanation of the jcode.asm routines
 *
 * char score[] is the list of 4-byte note events, each of which represents a
 *		note being attacked, held, released and an inter-record gap.
 *
 * int music_s is a semaphore -- when not zero, the interrupt routine doesn't
 *	       play music.  The idea is to set up score[] and music_p[], set
 *	       music_s to -1, set herror to 0, call init_clo(), and set music_s
 *	       to 0.  At this
 *	       point, the interrupt handler will begin playing the music.
 *	       When the music is done, the interrupt handler will set music_s
 *	       to -1.  Then you call quit_clo() and are done.
 *
 * int music_p[] is an array of indexes into the score[] tabel.  These indexes
 *		 are set to the beginning of each voice's notes by the parser,
 *		 and are incremented by the interrupt handler as it plays the
 *		 notes.  A value of -1 indicates that a voice has finished
 *		 playing.  (This also how fewer than N_VOICES voices are
 *		 played.)
 *
 * char herror is a status register which will become 1 or 2 if-and-when the
 *	       8051 does not respond to the handshake signals. You should set
 *	       it to 0, then check it periodically.
 *
 * init_clo() is the interrupt handler initialization routine.	DO NOT call
 *	      this procedure twice in a row -- call quit_clo() in-between!
 *	      init_clo() returns the port address of the lpt: port.  If there
 *	      is no lpt: port, init_clo() will return 0.
 *
 * quit_clo() is the interrupt handler de-initialization routine.  DO NOT call
 *	      it before calling init_clo().  DO call it before attempting to
 *	      return to DOS!
 *
 * sendab()  sends a byte to amy.  Should only be done while the music isn't
 *	     playing. . . but AFTER at least one call to init_clo()
 *
 *----------------------------------------------------------------------------
 */

#include <a:stdio.h>
#define DEBUG if(testing)
#define VOICES 4
extern	int gfname(), parse();
long	clock_t;

int	music_p[VOICES];	/* -1 for not-playing, otherwise points to score */
char	note, herror, testing, music_s;

#define MBUFLEN 10000
char mbuf[MBUFLEN];	/* input buffer for the music composer file */
int msize;	/* # of characters in existance */

#define MAXNBUF 500
char	name_buf[MAXNBUF];	 /* directory of *.MUS files buffer */
/* character codes used to draw a pretty box around the file names */

#define TOPLEFT  201
#define TOPRIGHT 187
#define MIDLEFT  204
#define MIDRIGHT 185
#define BOTLEFT  200
#define BOTRIGHT 188
#define VERTICAL 186
#define HORIZON  205

main(){
	int i;
	testing = 0;
     /* init_clo(); */	/* initialize printer addresses */
	quit_clo();
	demo();
}

cleartop(){
	int i;
	printf("\033[H");
	for(i = 0; i < 23; i += 1)
		printf("\033[K\n");
	fflush(stdout);
}

/*
 * demo
 */

#define MAXENT 26

demo(){
	char *flist[MAXENT];
	int nomf,i,j,jrem;	 /* number of music files */
	char c;

	nomf = gfname("????????MUS", flist, MAXENT, name_buf, MAXNBUF);
	if(nomf <= 0){
		printf("There are no music files in the default directory.\n");
		fflush(stdout);
		return;
	}
	j = nomf / 2;
	jrem = nomf % 2;

      for(;;){	/* menu loop */
	printf("\033[2J\n\t");    /* clear screen */
	putchar(TOPLEFT);
	for(i = 0; i < 49; i += 1)putchar(HORIZON);
	putchar(TOPRIGHT);
	putchar('\n');

	printf("\t%c               AMY Music Box v1.2                %c\n",
		VERTICAL, VERTICAL);

	putchar('\t');
	putchar(MIDLEFT);
	for(i = 0; i < 49; i += 1)putchar(HORIZON);
	putchar(MIDRIGHT);
	putchar('\n');

	for(i = 0; i < j; i += 1)
		printf("\t%c %c) %-20s %c) %-20s %c\n", VERTICAL, i + 'a',  flist [i],
			i + j + jrem + 'a',flist [i + j + jrem], VERTICAL);

	if(jrem)printf("\t%c %c) %-20s                         %c\n", VERTICAL,
		j + 'a', flist [j], VERTICAL);


	putchar('\t');
	putchar(MIDLEFT);
	for(i = 0; i < 49; i += 1)putchar(HORIZON);
	putchar(MIDRIGHT);
	putchar('\n');

	printf("\t%c  Type the letter of the piece you want (a-%c),   %c\n",
		VERTICAL, nomf + 'a' - 1, VERTICAL);

	printf("\t%c  or '*' to toggle debugging, or ESC to quit.    %c\n",
		VERTICAL, VERTICAL);

	putchar('\t');
	putchar(BOTLEFT);
	for(i = 0; i < 49; i += 1)putchar(HORIZON);
	putchar(BOTRIGHT);
	putchar('\n');

	for(;;){

		c = getch();	/* unbuffered input */

		if(c == 27){
			printf("Good bye!\n");
			fflush(stdout);
			return;      /* escape out of program */
		}
		if (c == ' '){
			monitor();
			continue;
		}

		if(c == '*'){
			testing = 1-testing;
			printf("Testing %s\n", testing ? "on" : "off");
			fflush(stdout);
			continue;
		}

		c = c - 'a';

		if (c < 0 || c >= nomf)
		{
			printf("Don't type that!\n");
			fflush(stdout);
			continue;
		}
		else break;
	}

	printf("\nPlaying %c) %-20s\n", c + 'a', flist[c]);
	fflush(stdout);

	play(flist [c]);

	if (testing)
	{
		printf("Type space bar to go on.\n");
		fflush(stdout);
		getch();
	}
      }
}

/* Allow the user to send out individual amy codes by hand */

monitor(){
	char c1, c2;
	printf("<");
	fflush(stdout);
	c1 = getch();
	printf("%c", c1);
	fflush(stdout);
	c2 = getch();
	printf("%c", c2);
	fflush(stdout);
	herror = 0;
	send_amy((c2d(c1) << 4) | c2d(c2));
	if (herror != 0)
	{
		printf( "Handshake error %d", herror);
	}
	printf(">");
	fflush(stdout);
}

/* Convert a hexidecimal character to 0..15 */

c2d(c)
	char c;
{
	if (c >= '0' && c <= '9')return(c - '0');
	if (c >= 'a' && c <= 'f')return(c - 'a' + 10);
	if (c >= 'A' && c <= 'F')return(c - 'A' + 10);
	return(-1);
}

/* play a piece of music
 */

play(fname)
char *fname;
{
	int fp, i;
	fp = open(fname, 0x8000);

	if(fp <= 0){
		printf("Couldn't open '%s'.\n", fname);
		fflush(stdout);
		return;
	}

	DEBUG{
		printf("I'm playing %s\n",fname);
		fflush(stdout);
	}

	msize = read(fp, mbuf, MBUFLEN);
	close(fp);
	if(msize < 0){
		printf("Couldn't read the music file.\n");
		fflush(stdout);
		return;
	}
	if(msize == MBUFLEN){
		printf("Music file too large.\n");
		fflush(stdout);
		return;
	}
	DEBUG printf("There are %d bytes in the file\n", msize);

	if(parse(mbuf, msize, music_p, testing) != 0){
		fflush(stdout);
		return;
	}

	if(testing){
		printf("Type space to see the score table:\n");
		fflush(stdout);
		if (getch() == ' ')
			print_score(music_p);
	}

	printf("Playing the music now. (any key will stop it)\n");
	fflush(stdout);
	music_s = -1;
	herror = 0;	/* handshake error */
	i = init_clock();	/* returns 0 if no printer port found */
	DEBUG{
		printf("Using the printer at %xh.\n", i);
		fflush(stdout);
	}

	music_s = 0;
	while(music_s == 0){
		if (herror)
		{
			printf("Handshake error %d\n", herror);
			fflush(stdout);
			if (! testing)
			{
				printf("Press any key to go on\n");
				fflush(stdout);
				getch();
			}
			break;
		}
		if (kbhit() != 0)
		{
			getch();
			break;
		}
		if (testing)
		{
			printf("music_p: ");
			for(i = 0; i < VOICES; i += 1)
				printf("%4d ", music_p[i]);
			printf("\n");
			fflush(stdout);
		}
	}
	quit_clock();
	send_amy(0x13); /* silence command */

}
/* === end of mbox === */

